/*
Copyright (C) 2001 Lkb

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "Interface/MainWindow.h"
#include "Core/R4300_Regs.h"
#include "Core/Memory.h"
#include "Core/CPU.h"
#include "Core/ROM.h"
#include "DynaRec/DynaRec.h"
#include "OSHLE/Patch.h"
#include "OSHLE/ultra_R4300.h"
extern DWORD MemoryRegionSizes[NUM_MEM_BUFFERS];

// I could have used the Standard C++ Library but it seems to grow memory streams by a constant 0x200 (resulting in extremely slow behavior)
// Additionally, the Microsoft implementation is probably deliberately obfuscated (or just written in a truly horrible way)

struct SaveState_ostream
{
	/*
	ostream& _stream;

	SaveState_ostream(ostream& stream)
		: _stream(stream)
	{}
	*/

	BYTE* _ptr;
	BYTE* _startptr;

	SaveState_ostream(void* ptr)
		: _ptr((BYTE*)ptr), _startptr((BYTE*)ptr)
	{}

	template<typename T>
	inline SaveState_ostream& operator << (const T& data)
	{
		write(&data, sizeof(T));
		return *this;
	}

	inline void skip(size_t size, char c = 0)
	{
		/*
		char buffer[256];
		memset(buffer, c, 256);
		for(int i = 0; i < size; i += 256)
		{
			_stream.write(buffer, 256);
		}
		_stream.write(buffer, size % 256);
		*/
		memset(_ptr, c, size);
		_ptr += size;
	}

	inline void write(const void* data, size_t size)
	{
		//if(size)
		//	_stream.write((const char*)data, size);
		memcpy(_ptr, data, size);
		_ptr += size;
	}

	inline void write_memory_buffer(int buffernum, int size = 0)
	{
		write(g_pMemoryBuffers[buffernum], size ? min(MemoryRegionSizes[buffernum], size) : MemoryRegionSizes[buffernum]);
		if(size > MemoryRegionSizes[buffernum])
			skip(size - MemoryRegionSizes[buffernum]);
	}

	// I had to add this as the PIF ram now lies in the same chunk as the PIF rom.
	inline void write_memory_buffer_offset(int buffernum, int offset, int size )
	{
		write((u8*)g_pMemoryBuffers[buffernum] + offset, size ? min(MemoryRegionSizes[buffernum], size) : MemoryRegionSizes[buffernum]);
		if(size > MemoryRegionSizes[buffernum])
			skip(size - MemoryRegionSizes[buffernum]);
	}
};

struct SaveState_istream
{
	/*
	istream& _stream;

	SaveState_istream(istream& stream)
		: _stream(stream)
	{}
	*/

	BYTE* _ptr;
	BYTE* _startptr;

	SaveState_istream(void* ptr)
		: _ptr((BYTE*)ptr), _startptr((BYTE*)ptr)
	{}

	template<typename T>
	inline SaveState_istream& operator >> (T& data)
	{
		read(&data, sizeof(T));
		return *this;
	}

	inline void read(void* data, size_t size)
	{
		//if(size)
		//	_stream.read((char*)data, size);
		memcpy(data, _ptr, size);
		_ptr += size;
	}

	inline void skip(size_t size)
	{
		//_stream.seekg(size, ios_base::cur);
		_ptr += size;
	}

	inline void read_memory_buffer(int buffernum, int size = 0)
	{
		read(g_pMemoryBuffers[buffernum], size ? min(MemoryRegionSizes[buffernum], size) : MemoryRegionSizes[buffernum]);
		if(size > MemoryRegionSizes[buffernum])
			skip(size - MemoryRegionSizes[buffernum]);
	}

	inline void read_memory_buffer_offset(int buffernum, int offset, int size )
	{
		read((u8*)g_pMemoryBuffers[buffernum] + offset, size ? min(MemoryRegionSizes[buffernum], size) : MemoryRegionSizes[buffernum]);
		if(size > MemoryRegionSizes[buffernum])
			skip(size - MemoryRegionSizes[buffernum]);
	}
	
	
	inline void read_memory_buffer_write_value(int buffernum, int address)
	{
		int i;
		for(i = 0; i < MemoryRegionSizes[buffernum]; i += 4)
		{
			DWORD value;
			*this >> value;
			Write32Bits(address + i, value);
		}
	}
};

void* SaveState_ReadAddress(DWORD address)
{
	void* pTranslated;
	BOOL old_g_bWarnMemoryErrors = g_DaedalusConfig.bWarnMemoryErrors;
	g_DaedalusConfig.bWarnMemoryErrors = FALSE;
	InternalReadAddress(address, &pTranslated);
	g_DaedalusConfig.bWarnMemoryErrors = old_g_bWarnMemoryErrors;
	return pTranslated;
}

// No idea why this value was chosen. It's not a readable text string.
const DWORD SAVESTATE_PROJECT64_MAGIC_NUMBER = 0x23D8A6C8;

void SaveState_Save(void* pSaveState)
{
	SaveState_ostream stream(pSaveState);
	BOOL cpuWasRunning = g_bCPURunning;
	RunCPUUntilStateIsSimple();

	//SaveState_ostream stream(p_stream);
	stream << (DWORD)SAVESTATE_PROJECT64_MAGIC_NUMBER;
	stream << (DWORD)g_dwRamSize;
	ROMHeader romHeader;
	memcpy(&romHeader, &g_ROM.rh, 64);
	ROM_ByteSwap_3210(&romHeader, 64);
	stream << romHeader;
	//stream.skip(4);
	stream << (DWORD)max(CPU_GetVideoInterruptEvent()->dwCount, 1);

	stream << (DWORD)g_dwPC;
	stream.write(g_qwGPR, 256);
	int i;
	for(i = 0; i < 32; i++)
	{
		stream << (DWORD)g_qwCPR[1][i];
	}
	stream.skip(0x80);
	for(i = 0; i < 32; i++)
	{
		stream << (DWORD)g_qwCPR[0][i];
	}
	for(i = 0; i < 32; i++)
	{
		stream << (DWORD)g_qwCCR[1][i];
	}
	stream << (ULONGLONG)g_qwMultHI;
	stream << (ULONGLONG)g_qwMultLO;
	
	stream.write_memory_buffer(MEM_RD_REG0, 0x28);
	stream.write_memory_buffer(MEM_SP_REG,  0x28);
	stream.write_memory_buffer(MEM_DPC_REG, 0x28);

	stream.write_memory_buffer(MEM_MI_REG);
	stream.write_memory_buffer(MEM_VI_REG);
	stream.write_memory_buffer(MEM_AI_REG);
	stream.write_memory_buffer(MEM_PI_REG);
	stream.write_memory_buffer(MEM_RI_REG);
	stream << *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_DRAM_ADDR_REG & 0xFF));
	stream << *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_PIF_ADDR_RD64B_REG & 0xFF));
	stream << *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_PIF_ADDR_WR64B_REG & 0xFF));
	stream << *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_STATUS_REG & 0xFF));
	for(i = 0; i < 32; i++)
	{
		// boolean that tells whether the entry is defined
		stream << (DWORD)(((g_TLBs[i].pfne & TLBLO_V) || (g_TLBs[i].pfno & TLBLO_V)) ? 1 : 0);
		stream << (DWORD)g_TLBs[i].pagemask;
		stream << (DWORD)g_TLBs[i].hi;
		stream << (DWORD)g_TLBs[i].pfne;
		stream << (DWORD)g_TLBs[i].pfno;
	}
	
	stream.write_memory_buffer_offset(MEM_PIF_RAM, 0x7C0, 0x40);
	
	BYTE* pRamToSave = new BYTE[g_dwRamSize];
	for(i = 0; i < g_dwRamSize; i += 4)
	{
		DWORD* pOp = (DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_RD_RAM] + i);
		DWORD dwOp = *pOp;
		if (R4300_OP(dwOp) == OP_SRHACK_UNOPT ||
			R4300_OP(dwOp) == OP_SRHACK_OPT ||
			R4300_OP(dwOp) == OP_SRHACK_NOOPT)
		{
			DWORD dwEntry = dwOp & 0x03ffffff;
			if((dwEntry < g_dwNumStaticEntries) && (SaveState_ReadAddress(g_pDynarecCodeTable[dwEntry]->dwStartPC) == pOp)) // ReadAddress is used because dwStartPC could be in a TLB mapped area
				dwOp = g_pDynarecCodeTable[dwEntry]->dwOriginalOp;
		}
		// FIXME: Add a StartPC field to the breakpoint struct
		// In the meantime, don't use breakpoints if you want to use savestates!
		/*
		else if (R4300_OP(dwOp) == OP_DBG_BKPT)
		{
			DWORD dwBreakPoint = dwOp & 0x03ffffff;

			if((dwBreakPoint < g_BreakPoints.size()) && g_BreakPoints[dwBreakPoint]
				dwOp = g_BreakPoints[dwBreakPoint].dwOriginalOp;
		}
		*/

		// Hacks may have replaced patch, so drop through to below:
		if (R4300_OP(dwOp) == OP_PATCH)
		{
			DWORD dwPatch = dwOp & 0x00FFFFFF;

			
			if( (dwPatch < ARRAYSIZE(g_PatchSymbols)) && (i == g_PatchSymbols[dwPatch]->dwLocation) )
				dwOp = g_PatchSymbols[dwPatch]->dwReplacedOp;
		}
		*(DWORD*)(pRamToSave + i) = dwOp;
	}
	stream.write(pRamToSave, g_dwRamSize);
	delete [] pRamToSave;
	
	stream.write_memory_buffer(MEM_SP_MEM);

	TCHAR dummyReasonBuffer[MAX_PATH];
	dummyReasonBuffer[0] = 0;
	if(cpuWasRunning)
		StartCPUThread(dummyReasonBuffer, MAX_PATH);
}

void R4300_TLB_TLBWI(DWORD dwOp);

size_t SaveState_GetSize()
{
	return g_dwRamSize + 0x275c;
}

void SaveState_Load(void* pSaveState)
{
	SaveState_istream stream(pSaveState);

	BOOL cpuWasRunning = g_bCPURunning;
	StopCPUThread();
	ROM_ReBoot();

	//SaveState_istream stream(p_stream);
	DWORD value;
	stream >> value;
	if(value != SAVESTATE_PROJECT64_MAGIC_NUMBER)
	{
		CMainWindow::Get()->MessageBox("Wrong magic number - savestate could be damaged or not in Daedalus/Project64 format", "Warning", MB_ICONWARNING);
	}
	stream >> (DWORD&)g_dwRamSize;
	ROMHeader romHeader;
	stream >> romHeader;
	ROM_ByteSwap_3210(&romHeader, 64);
	if(memcmp(g_ROM.rh.szName, romHeader.szName, 20) != 0)
	{
		CMainWindow::Get()->MessageBox("ROM name in savestate is different from the name of the currently loaded ROM", "Warning", MB_ICONWARNING);
	}

	stream >> (DWORD&)CPU_GetVideoInterruptEvent()->dwCount;
	//stream.skip(4);
	stream >> value;
	CPU_SetPC(value);
	stream.read(g_qwGPR, 256);
	int i;
	for(i = 0; i < 32; i++)
	{
		stream >> value;
		g_qwCPR[1][i] = value;
	}
	stream.skip(0x80); // used when FPU is in 64-bit mode
	DWORD g_dwNewCPR0[32];
	for(i = 0; i < 32; i++)
	{
		stream >> g_dwNewCPR0[i];
	}
	for(i = 0; i < 32; i++)
	{
		stream >> value;
		g_qwCCR[1][i] = value;
	}
	stream >> (ULONGLONG&)g_qwMultHI;
	stream >> (ULONGLONG&)g_qwMultLO;
	stream.read_memory_buffer(MEM_RD_REG0, 0x28); //, 0x84040000);
	stream.read_memory_buffer(MEM_SP_REG, 0x28); //, 0x84040000);

	BYTE* dpcRegData = new BYTE[MemoryRegionSizes[MEM_DPC_REG]];
	stream.read(dpcRegData, MemoryRegionSizes[MEM_DPC_REG]);
	memcpy(g_pMemoryBuffers[MEM_DPC_REG], dpcRegData, MemoryRegionSizes[MEM_DPC_REG]);

	stream.skip(8); // PJ64 stores 10 MEM_DP_COMMAND_REGs
	
	BYTE* miRegData = new BYTE[MemoryRegionSizes[MEM_MI_REG]];
	stream.read(miRegData, MemoryRegionSizes[MEM_MI_REG]);
	memcpy(g_pMemoryBuffers[MEM_MI_REG], miRegData, MemoryRegionSizes[MEM_MI_REG]);

	stream.read_memory_buffer_write_value(MEM_VI_REG, 0x84400000); // call WriteValue to update global and GFX plugin data
	stream.read_memory_buffer_write_value(MEM_AI_REG, 0x84500000); // call WriteValue to update audio plugin data
	
	// here to undo any modifications done by plugins
	memcpy(g_pMemoryBuffers[MEM_DPC_REG], dpcRegData, MemoryRegionSizes[MEM_DPC_REG]);
	delete [] dpcRegData;
	memcpy(g_pMemoryBuffers[MEM_MI_REG], miRegData, MemoryRegionSizes[MEM_MI_REG]);
	delete [] miRegData;
	
	stream.read_memory_buffer(MEM_PI_REG); //, 0x84600000);
	stream.read_memory_buffer(MEM_RI_REG); //, 0x84700000);
	stream >> value; *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_DRAM_ADDR_REG & 0xff)) = value;
	stream >> value; *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_PIF_ADDR_RD64B_REG & 0xff)) = value;
	stream >> value; *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_PIF_ADDR_WR64B_REG & 0xff)) = value;
	stream >> value; *(DWORD*)((INT_PTR)g_pMemoryBuffers[MEM_SI_REG] + (SI_STATUS_REG & 0xff)) =  value;
	for(i = 0; i < 32; i++)
	{
		stream.skip(4); // boolean that tells whether the entry is defined - seems redundant
		g_qwCPR[0][C0_INX] = i;
		stream >> value;
		g_qwCPR[0][C0_PAGEMASK] = value;
		stream >> value;
		g_qwCPR[0][C0_ENTRYHI] = value;
		stream >> value;
		g_qwCPR[0][C0_ENTRYLO0] = value;
		stream >> value;
		g_qwCPR[0][C0_ENTRYLO1] = value;

		R4300_TLB_TLBWI(0);
	}
	for(i = 0; i < 32; i++)
	{
		if(i == C0_SR)
		{
			R4300_SetSR(g_dwNewCPR0[i]);
		}
		else if(i == C0_COMPARE)
		{
			CPU_SetCompare(g_dwNewCPR0[i]);
		}
		else
		{
			g_qwCPR[0][i] = g_dwNewCPR0[i];
		}
	}
	//stream.skip(0x40);

	stream.read_memory_buffer_offset(MEM_PIF_RAM, 0x7C0, 0x40);

	stream.read(g_pMemoryBuffers[MEM_RD_RAM], g_dwRamSize);
	stream.read_memory_buffer(MEM_SP_MEM); //, 0x84000000);

	Patch_ApplyPatches();

	TCHAR dummyReasonBuffer[MAX_PATH];
	dummyReasonBuffer[0] = 0;
	if(cpuWasRunning)
		StartCPUThread(dummyReasonBuffer, MAX_PATH);
}

void SaveState_LoadFromFile(LPCTSTR pszFileName)
{
	HANDLE hFile = CreateFile(pszFileName, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
	size_t size = SaveState_GetSize();
	HANDLE hFileMapping = CreateFileMapping(hFile, NULL, PAGE_READONLY, 0, size, NULL);
	void* pFile = MapViewOfFile(hFileMapping, FILE_MAP_READ, 0, 0, 0);
	SaveState_Load(pFile);
	UnmapViewOfFile(pFile);
	CloseHandle(hFileMapping);
	CloseHandle(hFile);
}

void SaveState_SaveToFile(LPCTSTR pszFileName)
{
	HANDLE hFile = CreateFile(pszFileName, GENERIC_WRITE | GENERIC_READ, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
	size_t size = SaveState_GetSize();
	HANDLE hFileMapping = CreateFileMapping(hFile, NULL, PAGE_READWRITE, 0, size, NULL);
	void* pFile = MapViewOfFile(hFileMapping, FILE_MAP_ALL_ACCESS, 0, 0, 0);
	SaveState_Save(pFile);
	UnmapViewOfFile(pFile);
	CloseHandle(hFileMapping);
	CloseHandle(hFile);
}
