/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "ROM.h"
#include "Interface/MainWindow.h"

#include "Interface/Inifile.h"

#include "Utility/UnZipROM.h"

#include "Debug/DBGConsole.h"
#include "Debug/Debug.h"
#include "Core/CPU.h"
#include "Core/RSP.h"
#include "Core/RSP_HLE.h"
#include "Core/PIF.h"		// Controller_Initialise
#include "OSHLE/Patch.h"			// Patch_ApplyPatches
#include "OSHLE/ultra_r4300.h"
#include "OSHLE/ultra_os.h"		// System type
#include "Utility/Daedalus_CRC.h"


static void ROM_GetRomNameFromHeader(TCHAR * szName, ROMHeader * pHdr);
static u32 ROM_CrcBootCode(u8 * pRomBase);
static void ROM_CheckSumMario();
static void ROM_CheckSumZelda();

static BOOL ROM_ByteSwap(u8 * pBytes, DWORD dwLen);

RomInfo g_ROM;

static void ROM_CheckIniFile(LPROMINFO pRomInfo);
static BOOL ROM_LoadCartToMemory(LPCTSTR szFileName);
static HRESULT ROM_LoadDataFromFile(LPCTSTR szFileName, BYTE ** ppBuffer, DWORD * pdwLenToRead, bool useAllocRegion = false);
static void DumpROMInfo();

inline DWORD SwapEndian(DWORD x)
{
	return ((x >> 24)&0x000000FF) |
		   ((x >> 8 )&0x0000FF00) |
		   ((x << 8 )&0x00FF0000) |
		   ((x << 24)&0xFF000000);
}


#define MARIO_BOOT_CRC 0xb9c47dc8		// CIC-6102 or CIC-7101
#define STARF_BOOT_CRC 0xb4086651		// CIC-6101
#define LYLAT_BOOT_CRC 0xb3d6a525		// CIC-7102				// Lylat - Entyrpoint = 0x80000480
#define BANJO_BOOT_CRC 0xedce5ad9		// CIC-6103 or CIC-7103 // Diddy - Entrypoint + 0x00100000 
#define YOSHI_BOOT_CRC 0x06d8ed9c		// CIC-6106 or CIC-7106 // Yoshi - Entrypoint + 0x00200000 
#define ZELDA_BOOT_CRC 0xb53c588a		// CIC-6105 or CIC-7105

// Rom uses Unknown boot (0x51ae9f98)
// Rom uses Unknown boot (0x8dbba989)

#define MARIO_BOOT_CIC 0x3f
#define STARF_BOOT_CIC 0x3f // Incorrect
#define LYLAT_BOOT_CIC 0x3f // Incorrect
#define BANJO_BOOT_CIC 0x78
#define YOSHI_BOOT_CIC 0x85				// Same as FZero???
#define ZELDA_BOOT_CIC 0x91


const CountryIDInfo g_CountryCodeInfo[] = 
{
	{ 0, TEXT("0"), OS_TV_NTSC },
	{ '7', TEXT("Beta"), OS_TV_NTSC },
	{ 'A', TEXT("NTSC"), OS_TV_NTSC },
	{ 'D', TEXT("Germany"), OS_TV_PAL },
	{ 'E', TEXT("USA"), OS_TV_NTSC },
	{ 'F', TEXT("France"), OS_TV_PAL },
	{ 'I', TEXT("Italy"), OS_TV_PAL },
	{ 'J', TEXT("Japan"), OS_TV_NTSC },
	{ 'P', TEXT("Europe"), OS_TV_PAL },
	{ 'S', TEXT("Spain"), OS_TV_PAL },
	{ 'U', TEXT("Australia"), OS_TV_PAL },
	{ 'X', TEXT("PAL"), OS_TV_PAL },
	{ 'Y', TEXT("PAL"), OS_TV_PAL },
	{ -1, NULL, OS_TV_NTSC }
};

//*****************************************************************************
//
//*****************************************************************************
void DumpROMInfo()
{	
	DBGConsole_Msg(0, "First DWORD:     0x%02x%02x%02x%02x", g_ROM.rh.x1, g_ROM.rh.x2, g_ROM.rh.x3, g_ROM.rh.x4);
	DBGConsole_Msg(0, "Clockrate:       0x%08x", g_ROM.rh.dwClockRate);
	DBGConsole_Msg(0, "Program Counter: 0x%08x", SwapEndian(g_ROM.rh.dwBootAddressOffset));
	DBGConsole_Msg(0, "Release:         0x%08x", g_ROM.rh.dwRelease);
	DBGConsole_Msg(0, "CRC1:            0x%08x", g_ROM.rh.dwCRC1);
	DBGConsole_Msg(0, "CRC2:            0x%08x", g_ROM.rh.dwCRC2);
	DBGConsole_Msg(0, "Unknown64:       0x%08x%08x", *((DWORD *)&g_ROM.rh.qwUnknown1 + 0),  *((DWORD *)&g_ROM.rh.qwUnknown1 + 1));
	DBGConsole_Msg(0, "ImageName:       '%s'", g_ROM.rh.szName);
	DBGConsole_Msg(0, "Unknown32:       0x%08x", g_ROM.rh.dwUnknown2);
	DBGConsole_Msg(0, "Manufacturer:    0x%02x", g_ROM.rh.nManufacturer);
	DBGConsole_Msg(0, "CartID:          0x%04x", g_ROM.rh.wCartID);
	DBGConsole_Msg(0, "CountryID:       0x%02x - '%c'", g_ROM.rh.nCountryID, (char)g_ROM.rh.nCountryID);
}

//*****************************************************************************
//
//*****************************************************************************
static void ROM_RunPIFBoot(u32 cic)
{
	CPU_SetPC(0xBFC00000);

	// Set up CIC value in 0xbfc007e4
	*(u32*)((u8*)g_pMemoryBuffers[MEM_PIF_RAM] + 0x7C0 + 0x24) = (cic << 8) | (0xff);
}

//*****************************************************************************
//
//*****************************************************************************
static void ROM_SimulatePIFBoot(u32 cic, u32 nSystemType)
{
   // g_qwCPR[0][C0_SR]		= 0x34000000;	//*SR_FR |*/ SR_ERL | SR_CU2|SR_CU1|SR_CU0;
	R4300_SetSR(0x34000000);
	g_qwCPR[0][C0_CONFIG]	= 0x0006E463;	// 0x00066463;	

	// These are the effects of the PIF Boot Rom:

	g_qwGPR[REG_v0] = 0xFFFFFFFFD1731BE9;	// Required by Zelda, ignored by other boot codes
	g_qwGPR[REG_v1]	= 0xFFFFFFFFD1731BE9;
	g_qwGPR[REG_a0]	= 0x0000000000001BE9;
	g_qwGPR[REG_a1]	= 0xFFFFFFFFF45231E5;
	g_qwGPR[REG_a2]	= 0xFFFFFFFFA4001F0C;
	g_qwGPR[REG_a3]	= 0xFFFFFFFFA4001F08;
	g_qwGPR[REG_t0]	= 0x00000000000000C0;	// 0x70?
	g_qwGPR[REG_t1]	= 0x0000000000000000;
	g_qwGPR[REG_t2]	= 0x0000000000000040;
	g_qwGPR[REG_t3] = 0xFFFFFFFFA4000040;	// Required by Zelda, ignored by other boot codes
	g_qwGPR[REG_t4]	= 0xFFFFFFFFD1330BC3;
	g_qwGPR[REG_t5]	= 0xFFFFFFFFD1330BC3;
	g_qwGPR[REG_t6]	= 0x0000000025613A26;
	g_qwGPR[REG_t7]	= 0x000000002EA04317;
	g_qwGPR[REG_t8]	= 0x0000000000000000;
	g_qwGPR[REG_t9]	= 0xFFFFFFFFD73F2993;

	g_qwGPR[REG_s0]	= 0x0000000000000000;
	g_qwGPR[REG_s1]	= 0x0000000000000000;
	g_qwGPR[REG_s2]	= 0x0000000000000000;
	g_qwGPR[REG_s3]	= 0x0000000000000000;
	g_qwGPR[REG_s4] = nSystemType;
	g_qwGPR[REG_s5] = 0x0000000000000000;
    g_qwGPR[REG_s6] = cic;
    g_qwGPR[REG_s7] = 0x0000000000000006;
	g_qwGPR[REG_s8] = 0x0000000000000000;

	g_qwGPR[REG_sp] = 0xFFFFFFFFA4001FF0;
	//g_qwGPR[REG_ra]	= 0xFFFFFFFFA4000040;????
	g_qwGPR[REG_ra]	= 0xFFFFFFFFA4001554;
	// Copy low 1000 bytes to MEM_SP_MEM
	memcpy((u8*)g_pMemoryBuffers[MEM_SP_MEM], 
		   (u8*)g_pMemoryBuffers[MEM_CARTROM],
		   0x1000);

	// Need to copy crap to SP_IMEM for Zelda boot.
	u8 * pIMemBase = (u8*)g_pMemoryBuffers[ MEM_SP_MEM ] + 0x1000;
	
	u32 data[] = 
	{
		0x3c0dbfc0,
		0x8da807fc,
		0x25ad07c0,
		0x31080080,
		0x5500fffc,
		0x3c0dbfc0,
		0x8da80024,
		0x3c0bb000
	};

	memcpy( pIMemBase, data, sizeof(data) );

	// Also need to set up PI_BSD_DOM1 regs etc!
	CPU_SetPC(0xA4000040);
}

/* A test running the pif rom with different values for the cic
cic: 0x00000000
r0:*00000000 *t0:000000f0 s0:00000000  t8:00000000
at: 00000000 *t1:00000000 s1:00000000  t9:d73f2993
v0: d1731be9 *t2:00000040 s2:00000000 *k0:00000000
v1: d1731be9 *t3:a4000000 s3:00000000 *k1:00000000
a0: 00001be9 t4: d1330bc3 s4:00000001 *gp:00000000
a1: f45231e5 t5: d1330bc3 s5:00000000 *sp:a4001ff0
a2:*a4001f0c t6: 25613a26 s6:00000000 *s8:00000000
a3:*a4001f08 t7: 2ea04317 s7:00000000 *ra:a4001550
  
0xcdcdcdcd
r0:*00000000 *t0:000000f0 s0:00000000  t8:00000002
at: 00000001 *t1:00000000 s1:00000000  t9:688076ac
v0: 57ab3f3c *t2:00000040 s2:00000000 *k0:00000000
v1: 57ab3f3c *t3:a4000000 s3:00000001 *k1:00000000
a0: 00003f3c  t4:1c9fac27 s4:00000001 *gp:00000000
a1: 05825895  t5:112fe29d s5:00000000 *sp:a4001ff0
a2:*a4001f0c  t6:191df4b2 s6:000000cd *s8:00000000
a3:*a4001f08  t7:eafcd1fc s7:00000001 *ra:a4001550

0xffffffff
r0:*00000000 *t0:000000f0 s0:00000000  t8:00000000
at: 00000000 *t1:00000000 s1:00000000  t9:a4e54e8c
v0: 4a882c3f *t2:00000040 s2:00000000 *k0:00000000
v1: 4a882c3f *t3:a4000000 s3:00000001 *k1:00000000
a0: 00002c3f  t4:4ceb6550 s4:00000001 *gp:00000000
a1: 6848e133  t5:4ceb6550 s5:00000001 *sp:a4001ff0
a2:*a4001f0c  t6:24a38463 s6:000000ff *s8:00000000
a3:*a4001f08  t7:3257a25f s7:00000001 *ra:a4001550

*/

//*****************************************************************************
//
//*****************************************************************************
void ROM_ReBoot()
{
	u32 crc;
	u32 cic;
	u8 * pBase;
	DWORD dwBase;
	LONG i;

	// Do this here - probably not necessary as plugin loaded/unloaded with audio thread
	if (g_pAiPlugin != NULL)
		g_pAiPlugin->RomClosed();
	if ( CGraphicsPlugin::IsAvailable() )
		CGraphicsPlugin::Get()->RomClosed();

	// Reset here
	CPU_Reset(0xbfc00000);
	SR_Reset();
	RSP_Reset();
	RDP_Reset();
	if (g_ROM.bExpansionPak)
		Memory_Reset(MEMORY_8_MEG);		// Later set this to rom specific size
	else
		Memory_Reset(MEMORY_4_MEG);
	Patch_Reset();
	/*hr = */Controller_Initialise();

	// CRC from Cart + 0x40 for 0x1000 - 0x40 bytes
	dwBase = InternalReadAddress(0xb0000000, (void**)&pBase);
	if (dwBase != MEM_UNUSED)
	{
		crc = ROM_CrcBootCode(pBase);
		switch (crc)
		{
		case MARIO_BOOT_CRC: cic = MARIO_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Mario boot]"); break;
		case STARF_BOOT_CRC: cic = LYLAT_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Starfox boot]"); break;
		case LYLAT_BOOT_CRC: cic = LYLAT_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Lylat boot]"); break;
		case YOSHI_BOOT_CRC: cic = YOSHI_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Yoshi boot]"); break;
		case ZELDA_BOOT_CRC: cic = ZELDA_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Zelda boot]"); break;
		case BANJO_BOOT_CRC: cic = BANJO_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Banjo boot]"); break;
		default:			 cic = MARIO_BOOT_CIC;  DBGConsole_Msg(0, "[YRom uses Unknown boot (0x%08x)]", crc); break;
		}
	}
	else
	{
		crc = MARIO_BOOT_CRC;
		cic = MARIO_BOOT_CIC;
	}

	// Switch again, doing boot specific stuff
	switch (crc)
	{
	case MARIO_BOOT_CRC:
		ROM_CheckSumMario();
		break;
	case ZELDA_BOOT_CRC:
		ROM_CheckSumZelda();
		break;
	}

	g_ROM.nTvType = OS_TV_NTSC;
	for (i = 0; g_CountryCodeInfo[i].szName != NULL; i++)
	{
		if (g_CountryCodeInfo[i].nCountryID == g_ROM.rh.nCountryID)
		{
			g_ROM.nTvType = g_CountryCodeInfo[i].nTvType;
			break;
		}
	}


	//ROM_RunPIFBoot(cic);
	ROM_SimulatePIFBoot(cic, g_ROM.nTvType);
	
	//CPU_AddBreakPoint( 0xbfc00000 );
	//CPU_AddBreakPoint( 0xA4000040 );


	if ( CGraphicsPlugin::IsAvailable() )
		CGraphicsPlugin::Get()->RomOpen();


	DBGConsole_UpdateDisplay();
}

//*****************************************************************************
//
//*****************************************************************************
void ROM_Unload()
{
	Memory_Cleanup();
	Controller_Finalise();
}



//*****************************************************************************
// Copy across text, null terminate, and strip spaces
//*****************************************************************************
void ROM_GetRomNameFromHeader(TCHAR * szName, ROMHeader * pHdr)
{
	TCHAR * p;

	memcpy(szName, pHdr->szName, 20);
	szName[20] = '\0';

	p = szName + (lstrlen(szName) -1);		// -1 to skip null
	while (p >= szName && *p == ' ')
	{
		*p = 0;
		p--;
	}
}

//*****************************************************************************
//
//*****************************************************************************
BOOL ROM_LoadFile(LPCTSTR szFileName)
{
	ROM_Unload();

	DBGConsole_Msg(0, "Reading rom image: [C%s]", szFileName);
	
	if (!ROM_LoadCartToMemory(szFileName))
	{
		DBGConsole_Msg(0, "ROM_LoadCartToMemory(%s) Failed", szFileName);
		return FALSE;
	}

	lstrcpyn(g_ROM.szFileName, szFileName, MAX_PATH);
	
	// Get information about the rom header
	memcpy(&g_ROM.rh, g_pMemoryBuffers[MEM_CARTROM], sizeof(ROMHeader));
	ROM_ByteSwap_3210( &g_ROM.rh, sizeof(ROMHeader) );

	ROM_GetRomNameFromHeader(g_ROM.szGameName, &g_ROM.rh);
	DumpROMInfo();

	// Copy BootCode from ROM to DMEM
	g_ROM.u32BootAddres = SwapEndian(g_ROM.rh.dwBootAddressOffset);


	ROM_CheckIniFile(&g_ROM);


	if (g_ROM.bDisablePatches)
		g_bApplyPatches = FALSE;
	else
		g_bApplyPatches = TRUE;

	if (g_ROM.bDisableTextureCRC)
		g_bCRCCheck = FALSE;
	else
		g_bCRCCheck = TRUE;

	if (g_ROM.bDisableEeprom)
		g_bEepromPresent = FALSE;
	else
		g_bEepromPresent = TRUE;

	if (g_ROM.bDisableSpeedSync)
		g_bSpeedSync = FALSE;
	else
		g_bSpeedSync = TRUE;

	if (g_ROM.bDisableDynarec)
		g_bUseDynarec = FALSE;
	else
		g_bUseDynarec = TRUE;


	DBGConsole_Msg(0, "[G%s]", g_ROM.szGameName);
	DBGConsole_Msg(0,"This game has been certified as [G%s] (%s)",
		g_ROM.szComment, g_ROM.szInfo);

	DBGConsole_Msg(0, "ApplyPatches: [G%s]", g_bApplyPatches ? "on" : "off");
	DBGConsole_Msg(0, "Texture CRC Check: [G%s]", g_bCRCCheck ? "on" : "off");
	DBGConsole_Msg(0, "Eeprom: [G%s]", g_bEepromPresent ? "on" : "off");
	DBGConsole_Msg(0, "SpeedSync: [G%s]", g_bSpeedSync ? "on" : "off");
	DBGConsole_Msg(0, "DynaRec: [G%s]", g_bUseDynarec ? "on" : "off");

	//Patch_ApplyPatches();

	ROM_ReBoot();

	if (g_DaedalusConfig.bRunAutomatically)
	{
		CHAR szReason[300+1];
		// Ignore failure
		StartCPUThread(szReason, 300);
	}

	return TRUE;
}


//*****************************************************************************
// Initialise the RomInfo structure
//*****************************************************************************
BOOL ROM_InitRomInfo(LPROMINFO pRomInfo)
{
	ROMHeader * prh;
	BYTE * pBytes;
	HRESULT hr;
	DWORD dwRomSize;

	// Set rom size to rom headers size to indicate just this bit of file
	// should be read
	dwRomSize = sizeof(ROMHeader);
		
	//
	// If this is a zipfile, we need to unzip. Otherwise just read
	// directly from disk
	//
	if (lstrcmpi(PathFindExtension(pRomInfo->szFileName), TEXT(".zip")) == 0)
	{
#ifdef ZLIB_SUPPORT

		std::ostrstream messages;

		hr = UnzipRom_Unzip(pRomInfo->szFileName, 
							(LPBYTE*)&pBytes,
							&dwRomSize, false, messages);
		if (FAILED(hr))
		{	
			messages << '\0';
			DBGConsole_Msg(0, "[CUnable to get rom info from %s]", pRomInfo->szFileName);
			return FALSE;
		}
#else
		return FALSE;
#endif
	}
	else
	{
		hr = ROM_LoadDataFromFile(pRomInfo->szFileName,
								  (LPBYTE*)&pBytes,
								  &dwRomSize);
		if (FAILED(hr))
		{	
			DBGConsole_Msg(0, "[CUnable to get rom info from %s]", pRomInfo->szFileName);
			return FALSE;
		}
	}

	//
	// Get the address of the rom header
	//
	prh = (ROMHeader *)pBytes;

	if (!ROM_ByteSwap(pBytes, sizeof(ROMHeader)))
	{
		DBGConsole_Msg(0, "[CUnknown ROM format for %s: 0x%02x %02x %02x %02x",
							pRomInfo->szFileName,
							pBytes[0], pBytes[1], pBytes[2], pBytes[3]);
		return FALSE;
	}

	ROM_ByteSwap_3210( prh, sizeof(ROMHeader) );

	//
	// Copy the rom header
	//
	pRomInfo->rh = *prh;
	pRomInfo->u32BootAddres = SwapEndian(pRomInfo->rh.dwBootAddressOffset);

	lstrcpyn(pRomInfo->szComment, "No Comment", 100);

	pRomInfo->dwRomSize = dwRomSize;
	pRomInfo->nCountryID = prh->nCountryID;

	ROM_GetRomNameFromHeader(pRomInfo->szGameName, prh);

	// Initialise comments, other stuff
	ROM_CheckIniFile(pRomInfo);

	delete [] pBytes;

	return TRUE;
}


//*****************************************************************************
// Read a cartridge into memory. this routine byteswaps if necessary
//*****************************************************************************
BOOL ROM_LoadCartToMemory(LPCTSTR szFileName)
{
	HRESULT hr;
	DWORD dwROMSize;
	BYTE *pBytes;

	Memory_FreeCart();

	// If this is a zipfile, we need to unzip. Otherwise just read
	// directly from disk
	if (lstrcmpi(PathFindExtension(szFileName), TEXT(".zip")) == 0)
	{
#ifdef ZLIB_SUPPORT

		std::ostrstream messages;

		// Set rom size to 0 to indicate all of file should be read
		dwROMSize = 0;
		hr = UnzipRom_Unzip(szFileName, 
							(LPBYTE*)&pBytes,
							&dwROMSize, true, messages);
		if (FAILED(hr))
		{	
			messages << '\0';
			CMainWindow::Get()->MessageBox( messages.str(), g_szDaedalusName, MB_ICONERROR | MB_OK );
			return FALSE;
		}
#else
		DBGConsole_Msg(0, "[CZipfiles not supported in this build!]");
		return FALSE;
#endif
	}
	else
	{
		dwROMSize = 0;
		hr = ROM_LoadDataFromFile(szFileName,
								  (LPBYTE*)&pBytes,
								  &dwROMSize, true);
		if (FAILED(hr))
		{	
			DBGConsole_Msg(0, "[CUnable to get rom info from %s]", szFileName);
			return FALSE;
		}
	}

	if (!ROM_ByteSwap(pBytes, dwROMSize))
	{
		DBGConsole_Msg(0, "[CUnknown ROM format for %s: 0x%02x %02x %02x %02x",
			szFileName, pBytes[0], pBytes[1], pBytes[2], pBytes[3]);

		delete []pBytes;
		return FALSE;
	}

	// The cart is already loaded to memory
	//Memory_SetCart(pBytes, dwROMSize);

	return TRUE;
}




//*****************************************************************************
// Load specified number of bytes from the rom
// Allocates memory and returns pointer
// Byteswapping is handled elsewhere
// Pass in length to read in pdwLenToRead. (0 for entire rom)
// Actual length read returned in pdwLenToRead
//*****************************************************************************
HRESULT ROM_LoadDataFromFile(LPCTSTR szFileName, BYTE ** ppBuffer, DWORD * pdwLenToRead, bool useAllocRegion)
{
	HANDLE hFile;
	BOOL bSuccess;
	DWORD dwNumRead;
	DWORD dwLenToRead;
	BYTE * pBytes;
	DWORD dwFileSize;

	dwLenToRead = * pdwLenToRead;

	*pdwLenToRead = 0;
	*ppBuffer = NULL;

	// Open the file so that we can read in the data
	hFile = CreateFile(szFileName,	  // open the file 
		GENERIC_READ,                 // open for writing 
		0,                            // do not share 
		NULL,                         // no security 
		OPEN_EXISTING,                // replace any existing file
		FILE_ATTRIBUTE_NORMAL,        // normal file 
		NULL);                        // no attr. template 
	if (hFile == INVALID_HANDLE_VALUE)
	{ 
		return E_FAIL;
	}

	// If length to read is unspecified, use entire filesize
	dwFileSize = GetFileSize(hFile, NULL);
	if (dwLenToRead == 0)
		dwLenToRead = dwFileSize;

	if (dwLenToRead == ~0)
	{
		CloseHandle(hFile);
		return E_FAIL;
	}

	// Now, allocate memory for rom. 
	if(useAllocRegion)
	{
		pBytes = (u8*)Memory_AllocCart(dwLenToRead);
	}
	else
	{
		pBytes = new BYTE[dwLenToRead];
	}
	
	if (pBytes == NULL)
	{
		CloseHandle(hFile);
		return E_OUTOFMEMORY;
	}

	// Try and read in data
	bSuccess = ReadFile(hFile, (void *)pBytes, dwLenToRead, &dwNumRead, NULL);
	if (!bSuccess || dwNumRead != dwLenToRead)
	{
		CloseHandle(hFile);
		if(useAllocRegion)
		{
			Memory_FreeCart();
		}
		else
		{
			delete [] pBytes;
		}
		return E_FAIL;
	}

	// We're done with it now
	CloseHandle(hFile);

	*ppBuffer = pBytes;
	*pdwLenToRead = dwFileSize;

	return S_OK;
}


//*****************************************************************************
//
//*****************************************************************************
BOOL ROM_ByteSwap(u8 * pBytes, DWORD dwLen)
{
	u32 nSig;

	nSig = *(u32*)pBytes;

	switch (nSig)
	{
	case 0x80371240:
		// Pre byteswapped - no need to do anything
		break;
	case 0x40123780:
		ROM_ByteSwap_3210(pBytes, dwLen);
		break;
	case 0x12408037:
		ROM_ByteSwap_2301(pBytes, dwLen);
		break;
	default:
		return FALSE;
	}

	return TRUE;
}


#ifndef swap
#define swap( a, b ) a^=b^=a^=b
#endif 

//*****************************************************************************
// Swap bytes from 37 80 40 12
// to              40 12 37 80
//*****************************************************************************
void ROM_ByteSwap_2301(void *v, DWORD dwLen)
{
	// BEGIN MODIFIED BY Lkb - 9/jun/2001 - GCC support
/*
	__asm
	{
		mov		esi, v
		mov		edi, v
		mov		ecx, dwLen

		add		edi, ecx

	top:
		mov		al, byte ptr [esi + 0]
		mov		bl, byte ptr [esi + 1]
		mov		cl, byte ptr [esi + 2]
		mov		dl, byte ptr [esi + 3]

		mov		byte ptr [esi + 0], cl		//2
		mov		byte ptr [esi + 1], dl		//3
		mov		byte ptr [esi + 2], al		//0
		mov		byte ptr [esi + 3], bl		//1

		add		esi, 4
		cmp		esi, edi
		jne		top

	}
*/
	// There is no need to use assembly - the compiler generates better code in a portable way

	DWORD* p = (DWORD*)v;
	DWORD* maxp = (DWORD*)((UINT_PTR)v + dwLen);
	for(; p < maxp; p++)
	{
		swap(((WORD*)p)[0], ((WORD*)p)[1]);
	}
	// END MODIFIED BY Lkb - 9/jun/2001 - GCC support
}


//*****************************************************************************
// Swap bytes from 80 37 12 40
// to              40 12 37 80
//*****************************************************************************
void ROM_ByteSwap_3210(void *v, DWORD dwLen)
{
	// BEGIN MODIFIED BY Lkb - 9/jun/2001 - GCC support
	/*
	__asm
	{
		mov		esi, v
		mov		edi, v
		mov		ecx, dwLen

		add		edi, ecx

	top:
		mov		al, byte ptr [esi + 0]
		mov		bl, byte ptr [esi + 1]
		mov		cl, byte ptr [esi + 2]
		mov		dl, byte ptr [esi + 3]

		mov		byte ptr [esi + 0], dl		//3
		mov		byte ptr [esi + 1], cl		//2
		mov		byte ptr [esi + 2], bl		//1
		mov		byte ptr [esi + 3], al		//0

		add		esi, 4
		cmp		esi, edi
		jne		top

	}
	*/
	// There is no need to use assembly - the compiler generates better code in a portable way

	DWORD* p = (DWORD*)v;
	DWORD* maxp = (DWORD*)((UINT_PTR)v + dwLen);
	for(; p < maxp; p++)
	{
		swap(((BYTE*)p)[0], ((BYTE*)p)[3]);
		swap(((BYTE*)p)[1], ((BYTE*)p)[2]);
	}
	// END MODIFIED BY Lkb - 9/jun/2001 - GCC support
}



//*****************************************************************************
//
//*****************************************************************************
u32 ROM_CrcBootCode(u8 * pRomBase)
{
	return daedalus_crc32(0, pRomBase + 0x40, 0x1000 - 0x40);
}

//*****************************************************************************
//
//*****************************************************************************
void ROM_CheckSumMario()
{

	DWORD * pdwBase;
	DWORD dwAddress;
	DWORD a1;
	DWORD t7;
	DWORD v1 = 0;
	DWORD t0 = 0;
	DWORD v0 = 0xF8CA4DDC; //(MARIO_BOOT_CIC * 0x5d588b65) + 1;
	DWORD a3 = 0xF8CA4DDC;
	DWORD t2 = 0xF8CA4DDC;
	DWORD t3 = 0xF8CA4DDC;
	DWORD s0 = 0xF8CA4DDC;
	DWORD a2 = 0xF8CA4DDC;
	DWORD t4 = 0xF8CA4DDC;
	DWORD t8, t6, a0;


	pdwBase = (DWORD *)g_pMemoryBuffers[MEM_CARTROM];


	DBGConsole_Msg(0, "");		// Blank line

	for (dwAddress = 0; dwAddress < 0x00100000; dwAddress+=4)
	{
		if ((dwAddress % 0x2000) == 0)
		{
			DBGConsole_MsgOverwrite(0, "Generating CRC [M%d / %d]",
				dwAddress, 0x00100000);
		}


		v0 = pdwBase[(dwAddress + 0x1000)>>2];
		v1 = a3 + v0;
		a1 = v1;
		if (v1 < a3) 
			t2++;
	
		v1 = v0 & 0x001f;
		t7 = 0x20 - v1;
		t8 = (v0 >> (t7&0x1f));
		t6 = (v0 << (v1&0x1f));
		a0 = t6 | t8;

		a3 = a1;
		t3 ^= v0;
		s0 += a0;
		if (a2 < v0)
			a2 ^= a3 ^ v0;
		else
			a2 ^= a0;

		t0 += 4;
		t7 = v0 ^ s0;
		t4 += t7;
	}
	DBGConsole_MsgOverwrite(0, "Generating CRC [M%d / %d]",
		0x00100000, 0x00100000);


	a3 ^= t2 ^ t3;	// CRC1
	s0 ^= a2 ^ t4;	// CRC2

	if (a3 != pdwBase[0x10>>2] || s0 != pdwBase[0x14>>2])
	{
		DBGConsole_Msg(0, "[MWarning, CRC values don't match, fixing]");

		pdwBase[0x10>>2] = a3;
		pdwBase[0x14>>2] = s0;
	}


}

//*****************************************************************************
// Thanks Lemmy!
//*****************************************************************************
void ROM_CheckSumZelda()
{
	DWORD * pdwBase;
	DWORD dwAddress;
	DWORD dwAddress2;
	DWORD t5 = 0x00000020;
	DWORD a3 = 0xDF26F436;
	DWORD t2 = 0xDF26F436;
	DWORD t3 = 0xDF26F436;
	DWORD s0 = 0xDF26F436;
	DWORD a2 = 0xDF26F436;
	DWORD t4 = 0xDF26F436;
	DWORD v0, v1, a1, a0;


	pdwBase = (DWORD *)g_pMemoryBuffers[MEM_CARTROM];

	dwAddress2 = 0;

	DBGConsole_Msg(0, "");		// Blank line

	for (dwAddress = 0; dwAddress < 0x00100000; dwAddress += 4)
	{
		if ((dwAddress % 0x2000) == 0)
			DBGConsole_MsgOverwrite(0, "Generating CRC [M%d / %d]",
				dwAddress, 0x00100000);


		v0 = pdwBase[(dwAddress + 0x1000)>>2];
		v1 = a3 + v0;
		a1 = v1;
		
		if (v1 < a3)
			t2++;

		v1 = v0 & 0x1f;
		a0 = (v0 >> (t5-v1)) | (v0 << v1);
		a3 = a1;
		t3 = t3 ^ v0;
		s0 += a0;
		if (a2 < v0)
			a2 ^= a3 ^ v0;
		else
			a2 ^= a0;

		t4 += pdwBase[(dwAddress2 + 0x750)>>2] ^ v0;
		dwAddress2 = (dwAddress2 + 4) & 0xFF;

	}

	a3 ^= t2 ^ t3;
	s0 ^= a2 ^ t4;

	if (a3 != pdwBase[0x10>>2] || s0 != pdwBase[0x14>>2])
	{
		DBGConsole_Msg(0, "[MWarning, CRC values don't match, fixing]");

		pdwBase[0x10>>2] = a3;
		pdwBase[0x14>>2] = s0;
	}
	DBGConsole_MsgOverwrite(0, "Generating CRC [M%d / %d]",
		0x00100000, 0x00100000);

}

//*****************************************************************************
//
//*****************************************************************************
void ROM_CheckIniFile(LPROMINFO pRomInfo)
{
	LONG i;

	i = IniFile::Get()->FindEntry(pRomInfo->rh.dwCRC1,
							  pRomInfo->rh.dwCRC2,
							  pRomInfo->rh.nCountryID,
							  pRomInfo->szGameName);

	pRomInfo->ucode = IniFile::Get()->sections[i].ucode;

	lstrcpyn(pRomInfo->szGameName, IniFile::Get()->sections[i].name, 50);
	lstrcpyn(pRomInfo->szComment, IniFile::Get()->sections[i].comment, 100);
	lstrcpyn(pRomInfo->szInfo, IniFile::Get()->sections[i].info, 100);

	pRomInfo->bDisablePatches		= IniFile::Get()->sections[i].bDisablePatches;
	pRomInfo->bDisableTextureCRC	= IniFile::Get()->sections[i].bDisableTextureCRC;
	pRomInfo->bDisableEeprom		= IniFile::Get()->sections[i].bDisableEeprom;
	pRomInfo->bIncTexRectEdge		= IniFile::Get()->sections[i].bIncTexRectEdge;
	pRomInfo->bDisableSpeedSync		= IniFile::Get()->sections[i].bDisableSpeedSync;
	pRomInfo->bDisableDynarec		= IniFile::Get()->sections[i].bDisableDynarec;
	pRomInfo->bExpansionPak			= IniFile::Get()->sections[i].bExpansionPak;

	pRomInfo->dwEepromSize			= IniFile::Get()->sections[i].dwEepromSize;
	pRomInfo->dwRescanCount			= IniFile::Get()->sections[i].dwRescanCount;
}
