// Profiler.h: interface for the CProfiler class.
//
//////////////////////////////////////////////////////////////////////

#ifndef __PROFILER_H__
#define __PROFILER_H__

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "Unique.h"
#include "Debug/DBGConsole.h"

#ifndef DAEDALUS_RELEASE_BUILD
#define START_PROFILE( item ) CProfiler::Get()->StartTiming( item )
#define STOP_PROFILE( item ) CProfiler::Get()->StopTiming( item )
#else
#define START_PROFILE( item )
#define STOP_PROFILE( item )
#endif

class CProfiler : public CUnique< CProfiler >  
{
	protected:
		friend class CUnique< CProfiler >;
		CProfiler()
		{
			m_nCurrent = 0;
			QueryPerformanceFrequency( &m_liFrequency );

		}


		enum { NUM_HISTORY_ITEMS = 5 };

	public:
		virtual ~CProfiler() {}

		// Update s_ItemNames when adding to this
		enum ProfileItem
		{
			PROFILE_FRAME = 0,		// Time for a single frame
			PROFILE_GFX,			// Time for all graphics processing
			PROFILE_AUD,			// Time for audio processing (in plugin)

			PROFILE_DYNAREC,		// Time for dynamic recompilation

			// Components of Graphics processing
			PROFILE_GFX_TCACHE,		// Time for texture cache
			PROFILE_GFX_CONVERT,	// Time for decompression
			PROFILE_GFX_COMBINER,	// Time for combiner

			NUM_PROFILE_ITEMS
		};

		// Reset all times. Probably do this once per frame
		void Reset()
		{
			m_nCurrent = (m_nCurrent+1) % NUM_HISTORY_ITEMS;

			for ( u32 i = 0; i < NUM_PROFILE_ITEMS; i++ )
			{
				m_fTimes[ m_nCurrent ][ i ] = 0;
			}

			QueryPerformanceFrequency( &m_liFrequency );
		}

		// Start stop profiling for an item
		void StartTiming( ProfileItem item )
		{
			QueryPerformanceCounter( &m_liStartTimes[ (u32)item ] );
		}

		void StopTiming( ProfileItem item )
		{
			LARGE_INTEGER liEnd;

			QueryPerformanceCounter( &liEnd );

			m_fTimes[ m_nCurrent ][ (u32)item ] += (float)(liEnd.QuadPart - m_liStartTimes[ (u32)item ].QuadPart) / m_liFrequency.QuadPart;
		}

		// Return the time spent on this item since resetting
		float GetTime( u32 history, ProfileItem item )
		{
			// Look back history number of items in the buffer.
			// I've avoided doing (curr - hist) % NUM here as I think mod
			// works odly with signed numbers...
			s32 actual_history = m_nCurrent - (history % NUM_HISTORY_ITEMS);
			
			while ( actual_history < 0 )
				actual_history += NUM_HISTORY_ITEMS;

			return m_fTimes[ actual_history ][ (u32)item ];
		}

		void Display()
		{
#ifndef DAEDALUS_RELEASE_BUILD
			DBGConsole_DisplayProfile();
			/*for ( u32 i = 0; i < NUM_PROFILE_ITEMS; i++ )
			{
				DBGConsole_Msg(0, "%12s: %f", s_ItemNames[ i ], m_fTimes[ i ] );
			}*/
#endif
		}

		u32 GetNumHistoryItems() const { return NUM_HISTORY_ITEMS; }
		u32 GetNumProfileItems() const { return NUM_PROFILE_ITEMS; }

		const CHAR * GetProfileItemName( ProfileItem item )
		{
			return s_ItemInfo[ (u32) item ].name;
		}

	protected:

		typedef struct 
		{
			ProfileItem item;
			ProfileItem parent;
			const CHAR * name;

		} ProfileInfo;

		u32				m_nCurrent;


		LARGE_INTEGER	m_liStartTimes[ NUM_PROFILE_ITEMS ];

		float			m_fTimes[NUM_HISTORY_ITEMS ][ NUM_PROFILE_ITEMS ];

		LARGE_INTEGER	m_liFrequency;

		static const ProfileInfo s_ItemInfo[ NUM_PROFILE_ITEMS ];



};

#endif // __PROFILER_H__
