/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"

#include "MainWindow.h"

#include "Input/InputManager.h"
#include "Debug/Debug.h"
#include "Interface/ConfigDialog.h"
#include "Interface/RomSettingsDialog.h"
#include "Core/Interrupt.h"
#include "Core/CPU.h"
#include "DynaRec/DynaRec.h"
#include "Core/RSP_HLE.h"				// g_dwNumFrames
#include "OSHLE/ultra_rcp.h"
#include "OSHLE/ultra_r4300.h"
#include "OSHLE/ultra_os.h"			// Controller buttons defs
#include "Core/PIF.h"
#include "Debug/DBGConsole.h"
#include "Core/ROM.h"
#include "Utility/FileNameHandler.h"
#include "Utility/ResourceString.h"		// Useful everywhere

#include "Resources/resource.h"

enum 
{
	COLUMN_NAME = 0,
	COLUMN_COUNTRY = 1,
	COLUMN_SIZE = 2,
	COLUMN_COMMENT = 3,
	COLUMN_INFO = 4
};

class CListViewItemInfo
{
	public:
		CListViewItemInfo( const CHAR * szFileName )
		{
			ZeroMemory( &info, sizeof(info) );
			lstrcpyn( info.szFileName, szFileName, MAX_PATH );
			bInitialised = FALSE;
			bInvalid = FALSE;
		}

		LPROMINFO GetRomInfo() 
		{
			if ( bInvalid )
				return NULL;

			if ( !bInitialised )
			{
				// Init
				if (!ROM_InitRomInfo(&info))
				{
					bInvalid = TRUE;
					return NULL;
				}

				bInitialised = TRUE;
			}
			return &info;
		}

	protected:

		RomInfo		info;
		BOOL		bInvalid;
		BOOL		bInitialised;		// If false, most of pRomInfo has not been set up
};


//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
//                       Eternals
//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
extern DWORD g_dwTLBRHit;
extern DWORD g_dwTLBWHit;

//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
//                        Locals
//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////

static const char c_szRomFileStr[] = "ROM Files\0*.v64;*.z64;*.n64;*.rom;*.jap;*.pal;*.usa;*.zip\0All Files (*.*)\0";
static const char c_szSaveStateFileStr[] = "Savestate Files\0*.n64state;*.n6s;*.pj;*.pj0;*.pj1;*.pj2;*.pj3;*.pj4;*.pj5;*.pj6;*.pj7;*.pj8;*.pj9\0All Files (*.*)\0";
static const TCHAR	sc_szMainWindowClassname[256] = "DaedalusWindowClass";


class IMainWindow : public CMainWindow
{
	protected:
		friend class CUnique< CMainWindow >;
		IMainWindow()
		{
			m_hWnd = NULL;
			m_hWndStatus = NULL;
			m_hWndList = NULL;
			m_bActive = TRUE;		
		}

		HRESULT Initialise(void);

		BOOL CreateStatus(HWND hWndParent);

		BOOL CreateList(HWND hWndParent);
		BOOL InitList();
		BOOL FillList();

		static int CALLBACK ListView_CompareItems(LPARAM lParam1, LPARAM lParam2, LPARAM lpData);


		BOOL AddDirectory(LPCTSTR szDirectory);
		BOOL AddItem(LPCTSTR szFileName);

		void SaveWindowPos(HWND hWnd);

		void PauseDrawing();
		void RestartDrawing();

		void AskForRomDirectory(HWND hWndParent);

		static LRESULT CALLBACK WndProcStatic(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

		BOOL OnCreate(HWND hWnd, LPCREATESTRUCT lpCreateStruct);
		void OnDestroy(HWND hWnd);

		void OnInitMenu(HWND hwnd, HMENU hMenu);
		void OnTimer(HWND hWnd, UINT id);
		void OnMove(HWND hWnd, int x, int y);
		void OnSize(HWND hWnd, UINT state, int cx, int cy);

		void OnCommand(HWND hWnd, int id, HWND hwndCtl, UINT codeNotify);
		void OnKeyDown(HWND hWnd, UINT vk, BOOL fDown, int cRepeat, UINT flags);

		void OnStartEmu(HWND hWnd);
		void OnEndEmu(HWND hWnd);


		LRESULT OnNotify(int idCtrl, LPNMHDR pnmh);
		LRESULT OnListView_DeleteItem(int idCtrl, LPNMHDR pnmh);
		LRESULT OnListView_GetDispInfo(int idCtrl, LPNMHDR pnmh);
		LRESULT OnListView_ColumnClick(int idCtrl, LPNMHDR pnmh);
		LRESULT OnListView_DblClick(int idCtrl, LPNMHDR pnmh);
		LRESULT OnListView_RClick(int idCtrl, LPNMHDR pnmh);
		LRESULT OnListView_Return(int idCtrl, LPNMHDR pnmh);


		//
		// Member variables
		//
		HWND m_hWndList;
		BOOL m_bActive;





	public:
		HRESULT CreateWnd(INT nWinMode);

		void SelectSaveDir(HWND hWndParent);


		void __cdecl SetStatus(INT nPiece, LPCTSTR szFormat, ...);

		LRESULT WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
};


HRESULT CUnique< CMainWindow >::Create()
{
	HRESULT hr;

	_ASSERT(m_pInstance == NULL);

	IMainWindow * pInstance = new IMainWindow();

	hr = pInstance->Initialise();
	if (FAILED(hr))
	{
		delete pInstance;
		return hr;
	}

	m_pInstance = pInstance;
	return S_OK;
}


///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
HRESULT IMainWindow::Initialise()
{ 
    WNDCLASS wc; 
 
    // Register the frame window class. 
    wc.style         = 0; 
    wc.hInstance     = g_hInstance; 
    wc.cbClsExtra    = 0; 
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW); 
    wc.lpfnWndProc   = (WNDPROC)WndProcStatic; 
    wc.cbWndExtra    = 0; 
    wc.hIcon         = LoadIcon(g_hInstance, MAKEINTRESOURCE(IDI_DAEDALUS_ICON)); 
    wc.hbrBackground = (HBRUSH)NULL;// (COLOR_APPWORKSPACE + 1); //(HBRUSH)GetStockObject(BLACK_BRUSH);
    wc.lpszMenuName  = MAKEINTRESOURCE(IDR_APP_MENU); 
    wc.lpszClassName = sc_szMainWindowClassname; 
 
    if (!RegisterClass (&wc) ) 
        return E_FAIL; 

    return S_OK; 
} 



HRESULT IMainWindow::CreateWnd(INT nWinMode) 
{
	m_hWnd = CreateWindow(sc_szMainWindowClassname,
								g_szDaedalusName,
								WS_OVERLAPPEDWINDOW,
								g_DaedalusConfig.rcMainWindow.left,
								g_DaedalusConfig.rcMainWindow.top,
								g_DaedalusConfig.rcMainWindow.right - g_DaedalusConfig.rcMainWindow.left,
								g_DaedalusConfig.rcMainWindow.bottom - g_DaedalusConfig.rcMainWindow.top,
								HWND_DESKTOP,
								LoadMenu(g_hInstance, MAKEINTRESOURCE(IDR_APP_MENU)),
								g_hInstance,
								NULL);

	// Get the window pos, if CW_USEDEFAULT has been used for left/top
	if (IsWindow(m_hWnd))
	{
		ShowWindow(m_hWnd, nWinMode);
		UpdateWindow(m_hWnd);

		SaveWindowPos(m_hWnd);
		FillList();

		return S_OK;
	}

	return E_FAIL;
}


void IMainWindow::SaveWindowPos(HWND hWnd)
{
	/*bResult = */GetWindowRect(hWnd, &g_DaedalusConfig.rcMainWindow);
}

void __cdecl IMainWindow::SetStatus(INT nPiece, LPCTSTR szFormat, ...)
{
    va_list va;
	TCHAR szBuffer[2048+1];

	// Format the output
	va_start(va, szFormat);
	// Don't use wvsprintf as it doesn't handle floats!
	vsprintf(szBuffer, szFormat, va);
	va_end(va);

	// I changed this to PostMessage as the SendMessage would 
	// stall if the user tried to stop the CPU thread while 
	// the game was searching for OS functions to patch.
	::PostMessage(m_hWndStatus, SB_SETTEXT, nPiece | 0, (LPARAM)szBuffer);
}


void IMainWindow::PauseDrawing()
{
	//LPDIRECTDRAW8 pDD;

    m_bActive = FALSE;

//	CGraphicsContext::Get()->Lock();
/*D3D8FIX		pDD = CGraphicsContext::Get()->GetDD();
		if( pDD )
			pDD->FlipToGDISurface();*/
//	CGraphicsContext::Get()->Unlock();


    DrawMenuBar( m_hWnd );
    RedrawWindow( m_hWnd, NULL, NULL, RDW_FRAME);
    ShowCursor(TRUE);
}

void IMainWindow::RestartDrawing()
{
    m_bActive = TRUE;
    ShowCursor(FALSE);
}


static BOOL __stdcall AboutDlgProc(HWND hWndDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{

	switch (msg)
	{
	case WM_INITDIALOG:
		SendMessage(GetDlgItem(hWndDlg, IDC_DAEDALUS_LOGO), STM_SETIMAGE, IMAGE_BITMAP, (LPARAM)LoadBitmap(g_hInstance, MAKEINTRESOURCE(IDB_DAEDALUS)));
		return TRUE;
	case WM_COMMAND:
		switch (LOWORD(wParam))
		{
		case IDOK:
			EndDialog(hWndDlg, 0);
			break;
		}
		break;	
	}

	return 0;

}

// BEGIN ADDED BY Lkb - 21/jul/2001 - savestate support
void SaveState_LoadFromFile(LPCTSTR pszFileName);
void SaveState_SaveToFile(LPCTSTR pszFileName);
// END ADDED BY Lkb - 21/jul/2001 - savestate support

///////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////
LRESULT CALLBACK IMainWindow::WndProcStatic(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	return Get()->WndProc(hWnd, message, wParam, lParam);
}

LRESULT IMainWindow::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message)
	{
		case WM_ENTERSIZEMOVE: CInputManager::Get()->Unaquire(); break;
		case WM_ENTERMENULOOP: CInputManager::Get()->Unaquire(); PauseDrawing(); break;
	    case WM_EXITMENULOOP: RestartDrawing(); break;

		HANDLE_MSG( hWnd, WM_CREATE, OnCreate );
		HANDLE_MSG( hWnd, WM_DESTROY, OnDestroy );
		HANDLE_MSG( hWnd, WM_TIMER, OnTimer );
		HANDLE_MSG( hWnd, WM_INITMENU, OnInitMenu );
		HANDLE_MSG( hWnd, WM_MOVE, OnMove );
		HANDLE_MSG( hWnd, WM_SIZE, OnSize );
		HANDLE_MSG( hWnd, WM_COMMAND, OnCommand );
		HANDLE_MSG( hWnd, WM_KEYDOWN, OnKeyDown );

		case WM_CLOSE:		break;

		case WM_PAINT:		if ( g_bCPURunning && CGraphicsPlugin::IsAvailable() ) CGraphicsPlugin::Get()->DrawScreen(); break;	//return 0;

		//case MW_PAINT:	CGraphicsPlugin::Get()->DrawScreen(); break;
		case WM_NOTIFY: return OnNotify((int)wParam, (LPNMHDR)lParam);
		case MWM_STARTEMU: return (OnStartEmu(hWnd), 0);
		case MWM_ENDEMU: return (OnEndEmu(hWnd), 0);
			
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}



BOOL IMainWindow::OnCreate(HWND hWnd, LPCREATESTRUCT lpCreateStruct)
{
	HRESULT hr;

	if (CreateList(hWnd))
	{
		if(InitList())
		{
			
		}
	}

	CreateStatus(hWnd);

	SetTimer(hWnd, 0, 2000, NULL);

	hr = CInputManager::Get()->Initialise( hWnd );
	if (FAILED(hr))
	{
		MessageBox(CResourceString(IDS_UNABLETOINITINPUT));
	}

	// Created ok
	return TRUE;
}

void IMainWindow::OnDestroy(HWND hWnd)
{
	SaveWindowPos(hWnd);

	StopCPUThread();
	m_hWndList = NULL;
	m_hWndStatus = NULL;

	CInputManager::Get()->Finalise();

	PostQuitMessage(0);
}


LRESULT IMainWindow::OnNotify(int idCtrl, LPNMHDR pnmh)
{
	LRESULT lResult = 0;
	BOOL bHandled;

	bHandled = FALSE;

	if (pnmh->idFrom == IDC_ROMLISTVIEW)
	{
		bHandled = TRUE;
		switch (pnmh->code)
		{
		case LVN_DELETEITEM:	lResult = OnListView_DeleteItem(idCtrl, pnmh); break;
		case LVN_GETDISPINFO:	lResult = OnListView_GetDispInfo(idCtrl, pnmh); break;
		case LVN_COLUMNCLICK:	lResult = OnListView_ColumnClick(idCtrl, pnmh); break;
		case NM_DBLCLK:			lResult = OnListView_DblClick(idCtrl, pnmh); break;
		case NM_RCLICK:			lResult = OnListView_RClick(idCtrl, pnmh); break;
		case NM_RETURN:			lResult = OnListView_Return(idCtrl, pnmh); break;
		default:
			bHandled = FALSE;
			break;
		}
	}

	return lResult;
}

void IMainWindow::OnInitMenu(HWND hWnd, HMENU/* hMenu*/)
{
	// Only need this for Display menu
	HMENU hMenu;

	hMenu = GetMenu(hWnd);

	// BEGIN ADDED BY Lkb - 8/jun/2001 - New combiner
	if (g_bDisplayNewCombiner)
		CheckMenuItem(hMenu, IDM_NEWCOMBINER, MF_BYCOMMAND | MF_CHECKED);
	else
		CheckMenuItem(hMenu, IDM_NEWCOMBINER, MF_BYCOMMAND | MF_UNCHECKED);
	// END ADDED BY Lkb

}

void IMainWindow::OnTimer(HWND hWnd, UINT id)
{
	static u64 qwLastCount = 0;
	// BEGIN MODIFIED BY Lkb - 12/jul/2001 - GCC support
#ifdef __GNUC__
	static LARGE_INTEGER liLastCount = {0};
#else
	static LARGE_INTEGER liLastCount = {0,0};
#endif
	// END MODIFIED BY Lkb - 12/jul/2001 - GCC support
	s64 qwNum = g_qwCPR[0][C0_COUNT] - qwLastCount;
	static DWORD dwLastNumFrames = 0;
	static DWORD dwLastNumVIs = 0;
	static DWORD dwLastTLBRHit = 0;
	static DWORD dwLastTLBWHit = 0;
	static DWORD dwLastNumEx = 0;

	static DWORD dwNumSRCompiled = 0;
	static DWORD dwNumSROptimised = 0;
	static DWORD dwNumSRFailed = 0;

	if (g_bCPURunning)
	{
		LARGE_INTEGER li;
		LARGE_INTEGER liDiff;
		LARGE_INTEGER liFreq;
		float fElapsedTime;
		float fMIPS;
		float fFPS;
		float fVIs;
		float fTLBRS;
		float fTLBWS;
		float fExS;
		float fSRCS;
		float fSROS;
		float fSRFS;

		if (QueryPerformanceCounter(&li) && QueryPerformanceFrequency(&liFreq))
		{
			if (liLastCount.QuadPart == 0)
				liLastCount = li;
			if (dwLastNumFrames == 0)
				dwLastNumFrames = g_dwNumFrames;
			if (dwLastNumVIs == 0)
				dwLastNumVIs = g_dwVIs;

			liDiff.QuadPart = li.QuadPart - liLastCount.QuadPart;
			liLastCount = li;

			fElapsedTime = (float)liDiff.QuadPart / (float)liFreq.QuadPart;
			fMIPS = ((float)qwNum / 1000000.0f)/fElapsedTime;
			fFPS = ((float)(g_dwNumFrames-dwLastNumFrames)/fElapsedTime);
			fVIs = ((float)(g_dwVIs - dwLastNumVIs)/fElapsedTime);
			fTLBRS = ((float)(g_dwTLBRHit - dwLastTLBRHit)/fElapsedTime);
			fTLBWS = ((float)(g_dwTLBWHit - dwLastTLBWHit)/fElapsedTime);
			fExS = ((float)(g_dwNumExceptions - dwLastNumEx)/fElapsedTime);

			fSRCS = ((float)(g_dwNumSRCompiled - dwNumSRCompiled)/fElapsedTime);
			fSROS = ((float)(g_dwNumSROptimised - dwNumSROptimised)/fElapsedTime);
			fSRFS = ((float)(g_dwNumSRFailed - dwNumSRFailed)/fElapsedTime);

			DBGConsole_Stats( STAT_MIPS, "MIPS: %#.2f  FPS: %#.2f", fMIPS, fFPS);
			DBGConsole_Stats( STAT_VIS, "VIs/s: %#.2f",fVIs);
			DBGConsole_Stats( STAT_SYNC, "Sync: %#.2f%%",	100 * DESIRED_VI_INTERVAL / g_fCurrentRate);
			DBGConsole_Stats( STAT_TLB, "TLB: R %#.2f W: %#.2f", fTLBRS, fTLBWS);
			DBGConsole_Stats( STAT_EXCEP, "Ex/S %#.2f", fExS);
			DBGConsole_Stats( STAT_DYNAREC, "DR:  C%d O%d F%d", (DWORD)fSRCS, (DWORD)fSROS, (DWORD)fSRFS);

			CMainWindow::Get()->SetStatus(1, "VIs/s: %#.2f FPS: %#.2f",fVIs, fFPS);

		}

	}
	qwLastCount = g_qwCPR[0][C0_COUNT];
	dwLastNumFrames = g_dwNumFrames;
	dwLastNumVIs = g_dwVIs;
	dwLastTLBRHit = g_dwTLBRHit;
	dwLastTLBWHit = g_dwTLBWHit;
	dwLastNumEx = g_dwNumExceptions;
	dwNumSRCompiled = g_dwNumSRCompiled;
	dwNumSROptimised = g_dwNumSROptimised;
	dwNumSRFailed = g_dwNumSRFailed;
}

void IMainWindow::OnMove(HWND hWnd, int x, int y) 
{
	if ( CGraphicsPlugin::IsAvailable() )
	{
		CGraphicsPlugin::Get()->MoveScreen(x, y);
	}
}

void IMainWindow::OnSize(HWND hWnd, UINT state, int cx, int cy)
{

	WORD wTop = 0;
	LONG nStatusHeight;

	if (IsWindow(m_hWndStatus))
	{
		RECT rc;

		::SendMessage(m_hWndStatus, WM_SIZE, (WPARAM)state, (LPARAM)MAKELPARAM(cx,cy));

		GetWindowRect(m_hWndStatus, &rc);
		nStatusHeight = rc.bottom - rc.top;
	}
	else
	{
		nStatusHeight = 0;
	}

	cy -= nStatusHeight;

	//resize the ListView to fit our window
	if (IsWindow(m_hWndList))
	{
		MoveWindow(m_hWndList, 0, wTop, cx, cy, TRUE);
	}

//	CGraphicsContext::Get()->OnSize( state );
}


void IMainWindow::OnCommand(HWND hWnd, int id, HWND hwndCtl, UINT codeNotify)
{
	switch ( id )
	{
		// File menu commands
		case IDM_OPEN:
			{
				TCHAR szFileName[MAX_PATH+1];

				FileNameHandler * pfn;

				pfn = new FileNameHandler("Daedalus", c_szRomFileStr, 1, "v64");
				if (pfn != NULL)
				{
					BOOL bSelected;
					
					bSelected = pfn->GetOpenName(hWnd, szFileName);
					if (bSelected)
					{
						/*bSuccess = */ROM_LoadFile(szFileName);

						// Add this directory to the rom directory list?
						//pfn->GetCurrentDirectory(g_DaedalusConfig.szRomsDir);
					}
					SAFE_DELETE(pfn);
				}
			}
			break;

// BEGIN ADDED BY Lkb - 26/jul/2001 - savestate support
		case IDM_LOAD:
			{
				BOOL cpuWasRunning = g_bCPURunning;
				StopCPUThread();

				TCHAR szFileName[MAX_PATH+1];

				FileNameHandler * pfn;

				pfn = new FileNameHandler("Savestates", c_szSaveStateFileStr, 1, "n64state", g_DaedalusConfig.szSaveDir);
				if (pfn != NULL)
				{
					BOOL bSelected;
					
					bSelected = pfn->GetOpenName(hWnd, szFileName);
					if (bSelected)
					{
						SaveState_LoadFromFile(szFileName);
						pfn->GetCurrentDirectory(g_DaedalusConfig.szSaveDir);
					}
					SAFE_DELETE(pfn);
				}

				TCHAR dummyReasonBuffer[MAX_PATH];
				dummyReasonBuffer[0] = 0;
				if(cpuWasRunning)
					StartCPUThread(dummyReasonBuffer, MAX_PATH);
			}
			break;

		case IDM_SAVE:
			{
				BOOL cpuWasRunning = g_bCPURunning;
				StopCPUThread();

				TCHAR szFileName[MAX_PATH+1];

				FileNameHandler * pfn;

				pfn = new FileNameHandler("Savestates", c_szSaveStateFileStr, 1, "n64state", g_DaedalusConfig.szSaveDir);
				if (pfn != NULL)
				{
					BOOL bSelected;
					
					bSelected = pfn->GetSaveName(hWnd, szFileName);
					if (bSelected)
					{
						SaveState_SaveToFile(szFileName);
						pfn->GetCurrentDirectory(g_DaedalusConfig.szSaveDir);
					}
					SAFE_DELETE(pfn);
				}

				TCHAR dummyReasonBuffer[MAX_PATH];
				dummyReasonBuffer[0] = 0;
				if(cpuWasRunning)
					StartCPUThread(dummyReasonBuffer, MAX_PATH);
			}
			break;
// END ADDED BY Lkb - 21/jul/2001 - savestate support



		case IDM_EXIT:
			::SendMessage( hWnd, WM_CLOSE, 0, 0 );
			break;

		case IDM_CPUSTART:
			{
				CHAR szReason[300+1];

				if (!StartCPUThread(szReason, 300))
					MessageBox(szReason);
			}
			break;

		case IDM_CPUSTOP:
			StopCPUThread();
			break;

		case IDM_REFRESH:
			// Refresh
			FillList();
			break;
		case IDM_SELECTDIR:
			{
				//AskForRomDirectory(hWnd);

				// Refresh
				FillList();
			}
			break;

		case IDM_INPUTCONFIG:
			CInputManager::Get()->Configure(hWnd);
			break;

		case IDM_CONFIGURATION:
			{
				CConfigDialog::Create();
				CConfigDialog::Get()->DoModal(hWnd);
				CConfigDialog::Destroy();
			}
			break;

		case IDM_CHANGEDEVICE:
			if ( CGraphicsPlugin::IsAvailable() )
			{
				CGraphicsPlugin::Get()->DllConfig( hWnd );
			}
			break;

		case IDM_NEWCOMBINER:
			g_bDisplayNewCombiner = !g_bDisplayNewCombiner;
			if ( CGraphicsPlugin::IsAvailable() )
			{
				CGraphicsPlugin::Get()->ExecuteCommand( "UseNewCombiner", (LPVOID*)&g_bDisplayNewCombiner );
			}
			break;

		case IDM_ABOUT:
			DialogBox(g_hInstance, MAKEINTRESOURCE(IDD_ABOUT), hWnd, AboutDlgProc);
			break;
		case IDM_SCREENSHOT:
			if ( CGraphicsPlugin::IsAvailable() )
			{
				CGraphicsPlugin::Get()->ExecuteCommand( "CaptureScreen", NULL );
			}
			break;

		case IDM_TOGGLEFULLSCREEN:
			{
				if ( CGraphicsPlugin::IsAvailable() )
				{
					CGraphicsPlugin::Get()->ChangeWindow();
				}
			}
			
			break;

		case IDM_DAEDALUS_HOME:
			// Show the daedalus home page
			ShellExecute( hWnd, "open", DAEDALUS_SITE, NULL, "", SW_SHOWNORMAL );
			break;

	}
}

void IMainWindow::OnKeyDown(HWND hWnd, UINT vk, BOOL fDown, int cRepeat, UINT flags)
{
	switch ( vk )
	{
		// None of this will work, because we're using exclusive dinput:
		case VK_F1:
			if ( CGraphicsPlugin::IsAvailable() )
			{
				CGraphicsPlugin::Get()->ExecuteCommand( "DumpDisplayList", NULL );
			}
			break;	

		case VK_F2:
			#ifdef DAEDALUS_LOG
			g_bLog = !g_bLog;
			if (g_bLog)
				DBGConsole_Msg(0, "Debugging Output On");
			else
				DBGConsole_Msg(0, "Debugging Output Off");
			#endif
			break;	

		// BEGIN ADDED BY Lkb - 8/jun/2001 - New combiner
		case VK_F10:
			g_bDisplayNewCombiner = !g_bDisplayNewCombiner;
			break;
		// END ADDED BY Lkb - 8/jun/2001

		// NB Leave VK_SNAPSHOT (printscreen) so that Windows can do its own screenshots
		//case VK_SNAPSHOT:
			//if ( CGraphicsPlugin::IsAvailable() )
			//{
			//	CGraphicsPlugin::Get()->ExecuteCommand( "CaptureScreen", NULL );
			//}
			//break;
	}
}




void IMainWindow::OnStartEmu(HWND hWnd)
{

	TCHAR szName[256+1];
	// Save the current window size
	SaveWindowPos(hWnd);

	// Set the title text to include the rom name
	wsprintf(szName, "Daedalus - %s", g_ROM.szGameName);
	SetWindowText(hWnd, szName);

	EnableWindow(m_hWndList, FALSE);
	ShowWindow(m_hWndList, SW_HIDE);

	// Remove keyboard focus from the list (if is stil has it)
	SetFocus( hWnd );

	RECT rc;
	GetWindowRect(hWnd, &rc);
	if ( CGraphicsPlugin::IsAvailable() )
	{
		CGraphicsPlugin::Get()->MoveScreen(rc.left,rc.top);
	}
}


void IMainWindow::OnEndEmu(HWND hWnd)
{
	// Release the keyboard!
	CInputManager::Get()->Unaquire();

	ShowWindow(m_hWndList, SW_SHOW);
	EnableWindow(m_hWndList, TRUE);

	// Restore the title text
	SetWindowText(hWnd, "Daedalus");

	// TODO: Restore window size/pos?
	//SizeForDisplay();
}


BOOL IMainWindow::CreateStatus(HWND hWndParent)
{
	DWORD dwStyle;
	INT sizes[2] = { 340, -1 };

	dwStyle  = WS_CHILD|WS_VISIBLE;
	dwStyle |= SBARS_SIZEGRIP;

	m_hWndStatus = CreateStatusWindow(dwStyle, TEXT("Daedalus"), hWndParent, IDC_STATUSBAR);
	if (!IsWindow(m_hWndStatus))
		return FALSE;

	::SendMessage(m_hWndStatus, SB_SETPARTS, 2, (LPARAM)sizes);

	return TRUE;
}

BOOL IMainWindow::CreateList(HWND hWndParent)
{
	DWORD dwStyle;
	DWORD dwExStyle;
	
	dwStyle =   WS_TABSTOP | 
        WS_VISIBLE |
        WS_CHILD | 
        WS_BORDER | 
        LVS_SINGLESEL |
        LVS_REPORT | 
       // LVS_NOSORTHEADER |
        LVS_SHAREIMAGELISTS;

	dwExStyle = WS_EX_CLIENTEDGE;

	m_hWndList = ::CreateWindowEx( dwExStyle,
								WC_LISTVIEW,
								NULL,
								dwStyle,
								0,
								0,
								0,
								0,
								hWndParent,
								(HMENU)IDC_ROMLISTVIEW, // Child window id
								g_hInstance,
								NULL);
	
	return IsWindow(m_hWndList);

}

/*enum
{
	COLUMN_NAME = 0,
	COLUMN_COUNTRY,
	COLUMN_SIZE,
	COLUMN_BOOT,
	COLUMN_COMMENTS,
	COLUMN_INFO
};*/
//BOOL g_bColumnEnabled[] = { TRUE, TRUE, TRUE, TRUE, TRUE, TRUE };
//LONG g_nColumnOrder[] = { 0, 1, 2, 3, 4, 5 };


BOOL IMainWindow::InitList()
{
	LV_COLUMN   lvColumn;
	TCHAR       szString[MAX_PATH];

	// Allow the columns to be rearranged
	//ListView_SetExtendedListViewStyle(m_hWndList, LVS_EX_HEADERDRAGDROP );
	
	//initialize the columns
	lvColumn.mask = LVCF_FMT | LVCF_WIDTH | LVCF_TEXT | LVCF_SUBITEM;
	lvColumn.fmt = LVCFMT_LEFT;
	lvColumn.pszText = szString;
	
	lvColumn.cx = 180;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_NAME), sizeof(szString));
	ListView_InsertColumn(m_hWndList, COLUMN_NAME, &lvColumn);
		
	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_COUNTRY), sizeof(szString));
	ListView_InsertColumn(m_hWndList, COLUMN_COUNTRY, &lvColumn);
	
	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_SIZE), sizeof(szString));
	ListView_InsertColumn(m_hWndList, COLUMN_SIZE, &lvColumn);

	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_COMMENTS), sizeof(szString));
	ListView_InsertColumn(m_hWndList, COLUMN_COMMENT, &lvColumn);

	lvColumn.cx = 100;
	lstrcpyn(szString, CResourceString(IDS_COLUMN_INFO), sizeof(szString));
	ListView_InsertColumn(m_hWndList, COLUMN_INFO, &lvColumn);
/*

	ListView_SetColumnWidth(m_hWndList, COLUMN_NAME, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(m_hWndList, COLUMN_COUNTRY, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(m_hWndList, COLUMN_SIZE, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(m_hWndList, COLUMN_COMMENT, LVSCW_AUTOSIZE_USEHEADER);
	ListView_SetColumnWidth(m_hWndList, COLUMN_INFO, LVSCW_AUTOSIZE_USEHEADER);
*/
	return TRUE;
}

int CALLBACK IMainWindow::ListView_CompareItems(LPARAM lParam1, LPARAM lParam2, LPARAM lpData)
{
	// lpData is column to sort on:

	CListViewItemInfo * plvii1 = (CListViewItemInfo *)lParam1;
	CListViewItemInfo * plvii2 = (CListViewItemInfo *)lParam2;

	LPROMINFO pRomInfo1 = plvii1->GetRomInfo();
	LPROMINFO pRomInfo2 = plvii2->GetRomInfo();

	if ( !pRomInfo1 || !pRomInfo2 )
		return 0;

	switch ((LONG)lpData)
	{
	case COLUMN_NAME:    return (int)lstrcmpi(pRomInfo1->szGameName, pRomInfo2->szGameName);
	case COLUMN_COUNTRY: return (int)pRomInfo1->nCountryID - (int)pRomInfo2->nCountryID;
	case COLUMN_SIZE:    return (int)pRomInfo1->dwRomSize - (int)pRomInfo2->dwRomSize;
	case COLUMN_COMMENT: return (int)lstrcmpi(pRomInfo1->szComment, pRomInfo2->szComment);
	case COLUMN_INFO:    return (int)lstrcmpi(pRomInfo1->szInfo, pRomInfo2->szInfo);
	}

	return 0;
}


void IMainWindow::SelectSaveDir(HWND hWndParent)
{
	BROWSEINFO bi;
	LPITEMIDLIST lpidl;
	TCHAR szDisplayName[MAX_PATH+1];

	bi.hwndOwner = hWndParent;
	bi.pidlRoot = NULL;
	bi.pszDisplayName = szDisplayName;
	bi.lpszTitle = CResourceString(IDS_SELECTSAVEFOLDER);
	bi.ulFlags = BIF_RETURNONLYFSDIRS;
	bi.lpfn = NULL;
	bi.lParam = NULL;
	bi.iImage = 0;

	lpidl = SHBrowseForFolder(&bi);
	if (lpidl)
	{
		/*bResult = */SHGetPathFromIDList(lpidl, g_DaedalusConfig.szSaveDir);
	}
	else
	{

	}
}

//*****************************************************************************
// Prompt for a default rom directory if none are currently defined
//*****************************************************************************
void IMainWindow::AskForRomDirectory(HWND hWndParent)
{
	BROWSEINFO bi;
	LPITEMIDLIST lpidl;
	TCHAR szDisplayName[MAX_PATH+1];

	bi.hwndOwner = hWndParent;
	bi.pidlRoot = NULL;
	bi.pszDisplayName = szDisplayName;
	bi.lpszTitle = CResourceString(IDS_SELECTFOLDER);
	bi.ulFlags = BIF_RETURNONLYFSDIRS;
	bi.lpfn = NULL;
	bi.lParam = NULL;
	bi.iImage = 0;

	lpidl = SHBrowseForFolder(&bi);
	if (lpidl)
	{
		if ( SHGetPathFromIDList(lpidl, g_DaedalusConfig.szRomsDirs[ 0 ] ) )
		{
			g_DaedalusConfig.nNumRomsDirs = 1;
		}
	}
	else
	{

	}
}


BOOL IMainWindow::FillList()
{
	if (g_DaedalusConfig.nNumRomsDirs == 0 || !PathIsDirectory(g_DaedalusConfig.szRomsDirs[ 0 ]))
	{
		// Don't pass in m_hWnd - as it may not have been set up yet!
		AskForRomDirectory(m_hWndList);
		if (!PathIsDirectory(g_DaedalusConfig.szRomsDirs[ 0 ]))
		{
			return FALSE;
		}
	}

	//empty the list
	ListView_DeleteAllItems(m_hWndList);

	for ( u32 rd = 0; rd < g_DaedalusConfig.nNumRomsDirs; rd++ )
	{
		/* bResult = */AddDirectory(g_DaedalusConfig.szRomsDirs[ rd ]);
	}

	//sort the items by name
	ListView_SortItems(m_hWndList, ListView_CompareItems, (LPARAM)0);

	return TRUE;
}

// Return FALSE on critical error (stop recursing)
BOOL IMainWindow::AddDirectory(LPCTSTR szDirectory)
{
	HANDLE hFind;
	WIN32_FIND_DATA wfd;
	TCHAR szFullPath[MAX_PATH+1];
	TCHAR szSearchSpec[MAX_PATH+1];
	LPTSTR szExt;

	if (!PathIsDirectory(szDirectory))
		return FALSE;

	// Recurse, adding all roms that we find...
	PathCombine(szSearchSpec, szDirectory, TEXT("*.*"));
	
	hFind = FindFirstFile(szSearchSpec, &wfd);
	if (hFind == INVALID_HANDLE_VALUE)
		return FALSE;

	do 
	{
		// Ignore current/parent directories
		if (lstrcmp(wfd.cFileName, TEXT(".")) == 0 ||
			lstrcmp(wfd.cFileName, TEXT("..")) == 0)
			continue;

		PathCombine(szFullPath, szDirectory, wfd.cFileName);

		if (PathIsDirectory(szFullPath))
		{
			// Recurse - add the contents of this directory
			if (!AddDirectory(szFullPath))
			{
				// Break/Exit??
			}
		}
		else
		{
			szExt = PathFindExtension(wfd.cFileName);

			if (lstrcmpi(szExt, TEXT(".v64")) == 0 ||
				lstrcmpi(szExt, TEXT(".z64")) == 0 ||
				lstrcmpi(szExt, TEXT(".n64")) == 0 ||
				lstrcmpi(szExt, TEXT(".rom")) == 0 ||
				lstrcmpi(szExt, TEXT(".jap")) == 0 ||
				lstrcmpi(szExt, TEXT(".pal")) == 0 ||
				lstrcmpi(szExt, TEXT(".usa")) == 0 ||
				lstrcmpi(szExt, TEXT(".zip")) == 0)
			{
				// Add to the list
				AddItem(szFullPath);
			}
		}
		
	} while (FindNextFile(hFind, &wfd));

	FindClose(hFind);

	return TRUE;
}


BOOL IMainWindow::AddItem(LPCTSTR szFileName)
{
	LV_ITEM  lvItem;
	CListViewItemInfo * plvii;

	plvii = new CListViewItemInfo( szFileName );
	if (plvii == NULL)
		return FALSE;

	ZeroMemory(&lvItem, sizeof(lvItem));
	
	lvItem.mask = LVIF_TEXT | /*LVIF_IMAGE |*/ LVIF_PARAM;		
	lvItem.iItem = ListView_GetItemCount(m_hWndList);	//add the item to the end of the list
	lvItem.lParam = (LPARAM)plvii;
	lvItem.pszText = LPSTR_TEXTCALLBACK;
	//lvItem.iImage = I_IMAGECALLBACK;
	
	ListView_InsertItem(m_hWndList, &lvItem);

	return TRUE;
}

LRESULT IMainWindow::OnListView_DeleteItem(int idCtrl, LPNMHDR pnmh)
{
	NM_LISTVIEW *lpnmlv = (NM_LISTVIEW*)pnmh;
	CListViewItemInfo * plvii = (CListViewItemInfo *)lpnmlv->lParam;
	
	//delete the pidl because we made a copy of it
	if (plvii)
	{
		delete (plvii);
	}

	return 0;
}


LRESULT IMainWindow::OnListView_GetDispInfo(int idCtrl, LPNMHDR pnmh)
{	
	LV_DISPINFO * lpdi = (LV_DISPINFO *)pnmh;
	CListViewItemInfo * plvii = (CListViewItemInfo *)lpdi->item.lParam;

	if ( plvii == NULL )
		return 0;

	// This will initialise the rom info if it has not been done already
	LPROMINFO pRomInfo = plvii->GetRomInfo();

	if ( pRomInfo == NULL )
		return 0;

	// Which column is being requested?
	switch ( lpdi->item.iSubItem )
	{
		case COLUMN_NAME:
			//is the text being requested?
			if(lpdi->item.mask & LVIF_TEXT)
			{
				lstrcpyn(lpdi->item.pszText, pRomInfo->szGameName, lpdi->item.cchTextMax);			
			}
			
			//is the image being requested?
			if(lpdi->item.mask & LVIF_IMAGE)
			{
				lpdi->item.iImage = -1;
			}
			break;

		// Country
		case COLUMN_COUNTRY:
			//is the text being requested?
			if (lpdi->item.mask & LVIF_TEXT)
			{
				LONG i;
				//if the data text is NULL, get the no value string
				//if(lstrlen(lpdi->item.pszText) == 0)
				//	LoadString(_Module.GetModuleInstance(), IDS_UNKNOWN, lpdi->item.pszText, lpdi->item.cchTextMax);

				// Default (if not found) is character string
				wsprintf(lpdi->item.pszText, "%c", pRomInfo->nCountryID);
				for (i = 0; g_CountryCodeInfo[i].szName != NULL; i++)
				{
					if (g_CountryCodeInfo[i].nCountryID == pRomInfo->nCountryID)
					{
						lstrcpyn(lpdi->item.pszText, g_CountryCodeInfo[i].szName, lpdi->item.cchTextMax);
						break;
					}
				}

			}
			break;

		// Size
		case COLUMN_SIZE:
			//is the text being requested?
			if (lpdi->item.mask & LVIF_TEXT)
			{
				StrFormatByteSize(pRomInfo->dwRomSize, lpdi->item.pszText, lpdi->item.cchTextMax);
				//lstrcpyn(lpdi->item.pszText, TEXT(""), lpdi->item.cchTextMax);
			}
			break;

		// Comments
		case COLUMN_COMMENT:
			//is the text being requested?
			if (lpdi->item.mask & LVIF_TEXT)
			{
				// Don't give sizes for directories:
				lstrcpyn(lpdi->item.pszText, pRomInfo->szComment, lpdi->item.cchTextMax);
			}
			break;

		// Info
		case COLUMN_INFO:
			//is the text being requested?
			if (lpdi->item.mask & LVIF_TEXT)
			{
				// Don't give sizes for directories:
				lstrcpyn(lpdi->item.pszText, pRomInfo->szInfo, lpdi->item.cchTextMax);
			}
			break;

		default:
			// Huh?
			break;
	}
	return 0;
}

LRESULT IMainWindow::OnListView_ColumnClick(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	// Process LVN_COLUMNCLICK to sort items by column. 
	ListView_SortItems(lpnmlv->hdr.hwndFrom, ListView_CompareItems, (LPARAM)(lpnmlv->iSubItem));

	return 0;

}

LRESULT IMainWindow::OnListView_DblClick(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	if (lpnmlv->iItem != -1)
	{
		LV_ITEM lvItem;

		ZeroMemory(&lvItem, sizeof(LV_ITEM));
		lvItem.mask = LVIF_PARAM;
		lvItem.iItem = lpnmlv->iItem;

		if (ListView_GetItem(m_hWndList, &lvItem))
		{
			CListViewItemInfo * plvii;
			plvii = (CListViewItemInfo *)lvItem.lParam;

			if (plvii != NULL)
			{
				/*bSuccess = */ROM_LoadFile(plvii->GetRomInfo()->szFileName);
			}
		}
	}

	return 0;

}


LRESULT IMainWindow::OnListView_RClick(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	if (lpnmlv->iItem != -1)
	{
		LV_ITEM lvItem;

		ZeroMemory(&lvItem, sizeof(LV_ITEM));
		lvItem.mask = LVIF_PARAM;
		lvItem.iItem = lpnmlv->iItem;

		if (ListView_GetItem(m_hWndList, &lvItem))
		{
			CListViewItemInfo * plvii;
			plvii = (CListViewItemInfo *)lvItem.lParam;

			if (plvii != NULL)
			{
				RomSettingsDialog_DoModal(m_hWnd, plvii->GetRomInfo());
				// Refresh
				ListView_Update(m_hWndList, lpnmlv->iItem);
			}

		}
	}

	return 0;

}

LRESULT IMainWindow::OnListView_Return(int idCtrl, LPNMHDR pnmh)
{
	LPNMLISTVIEW   lpnmlv;
	LONG iItem;

	lpnmlv = (LPNMLISTVIEW)pnmh;

	// Get the currently highlighted item
	iItem = ListView_GetNextItem(m_hWndList, -1, LVNI_SELECTED);

	if (iItem != -1)
	{
		LV_ITEM lvItem;

		ZeroMemory(&lvItem, sizeof(LV_ITEM));
		lvItem.mask = LVIF_PARAM;
		lvItem.iItem = iItem;

		if (ListView_GetItem(m_hWndList, &lvItem))
		{
			CListViewItemInfo * plvii;
			plvii = (CListViewItemInfo *)lvItem.lParam;

			if (plvii != NULL)
			{
				/*bSuccess = */ROM_LoadFile(plvii->GetRomInfo()->szFileName);
			}

		}
	}

	return 0;
}

