/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Display stuff like registers, instructions, memory usage and so on
#include "stdafx.h"


#include "Core/Memory.h"
#include "Core/CPU.h"
#include "Core/RSP.h"
#include "Core/Interrupt.h"
#include "Dump.h"
#include "Utility/PrintOpCode.h"
#include "debug.h"
#include "OSHLE/Patch.h"		// For GetCorrectOp
#include "OSHLE/ultra_r4300.h"
#include "Interface/MainWindow.h"	// Main_SelectSaveDir()


static TCHAR g_szDumpDir[MAX_PATH+1] = TEXT("");

//*****************************************************************************
//
//*****************************************************************************
static void Dump_EnsureExists(LPCTSTR szPath)
{
	TCHAR szPathParent[MAX_PATH+1];

	if (PathIsDirectory(szPath))
		return;

	// Make sure parent exists, 
	lstrcpyn(szPathParent, szPath, MAX_PATH);
	PathRemoveBackslash(szPathParent);
	PathRemoveFileSpec(szPathParent);
	Dump_EnsureExists(szPathParent);

	CreateDirectory(szPath, NULL);
}

//*****************************************************************************
// Initialise the directory where files are dumped
//*****************************************************************************
void Dump_GetDumpDirectory(LPTSTR szFilePath, LPCTSTR szSubDir)
{
	// Appends szBase to the global dump base. Stores in szFilePath


	if (lstrlen(g_szDumpDir) == 0)
	{
		// Initialise
		PathCombine(g_szDumpDir, g_szDaedalusExeDir, TEXT("Dumps"));
	}

	// If a subdirectory was specified, append 
	if (lstrlen(szSubDir) > 0)
		PathCombine(szFilePath, g_szDumpDir, szSubDir);
	else
		lstrcpyn(szFilePath, g_szDumpDir, MAX_PATH);

	Dump_EnsureExists(szFilePath);

}


//*****************************************************************************
// E.g. Dump_GetSaveDirectory([out], "c:\roms\test.rom", ".sra")
// would first try to find the save in g_DaedalusConfig.szSaveDir. If this is not
// found, g_DaedalusConfig.szRomsDir is checked.
//*****************************************************************************
void Dump_GetSaveDirectory(LPTSTR szFilePath, LPCTSTR szRomName, LPCTSTR szExt)
{
	TCHAR szFileName[MAX_PATH+1];

	// If the Save path has not yet been set up, prompt user
	if (lstrlen(g_DaedalusConfig.szSaveDir) == 0)
	{
		CMainWindow::Get()->SelectSaveDir( CMainWindow::Get()->GetWindow() );

		// User may have cancelled
		if (lstrlen(g_DaedalusConfig.szSaveDir) == 0)
		{
			// Default to rom path
			lstrcpyn(g_DaedalusConfig.szSaveDir, szRomName, MAX_PATH);
			PathRemoveFileSpec(g_DaedalusConfig.szSaveDir);

			DBGConsole_Msg(0, "SaveDir is still empty - defaulting to [C%s]", g_DaedalusConfig.szSaveDir);
		}
	}

	// Form the filename from the file spec (i.e. strip path)
	lstrcpyn(szFileName, szRomName, MAX_PATH);

	PathStripPath(szFileName);
	PathRemoveExtension(szFileName);
	PathAddExtension(szFileName, szExt);

	PathCombine(szFilePath, g_DaedalusConfig.szSaveDir, szFileName);
	if (PathFileExists(szFilePath))
		return;

	PathCombine(szFilePath, g_DaedalusConfig.szRomsDirs[ 0 ], szFileName);
	if (PathFileExists(szFilePath))
		return;

	// Not in either dir - default to SaveDir
	PathCombine(szFilePath, g_DaedalusConfig.szSaveDir, szFileName);

}



//*****************************************************************************
// pdwBase[0] = instruction at dwStart
//*****************************************************************************
static void DumpRange(LPCTSTR szFileName,
					  LPCTSTR szMode,
					  DWORD * pdwBase,
					  DWORD dwStart,
					  DWORD dwEnd,
					  BOOL bTranslatePatches)
{
	char opinfo[400];
	FILE * fp;	

	fp = fopen(szFileName, szMode);
	if (fp == NULL)
		return;

	for (DWORD dwPC = dwStart; dwPC < dwEnd; dwPC+=4)
	{
		DWORD dwOriginalOp = pdwBase[(dwPC-dwStart)/4];
		DWORD dwOpCode;

		if (bTranslatePatches)
		{
			dwOpCode = GetCorrectOp(dwOriginalOp);
			if (IsJumpTarget(dwOriginalOp))
			{
				fprintf(fp, "\n");
				fprintf(fp, "\n");
				fprintf(fp, "// %s():\n", Patch_GetJumpAddressName(dwPC));
			}
		}			
		else
		{
			dwOpCode = GetCorrectOp(dwOriginalOp);
		}

		SprintOpCodeInfo(opinfo, dwPC, dwOpCode);
		fprintf(fp, "0x%08x: <0x%08x> %s\n", dwPC, dwOpCode, opinfo);
	}

	fclose(fp);
}


//*****************************************************************************
//
//*****************************************************************************
void Dump_Disassemble(DWORD dwStart, DWORD dwEnd, LPCTSTR szFileName)
{
	TCHAR szFilePath[MAX_PATH+1];

	// Cute hack - if the end of the range is less than the start,
	// assume it is a length to disassemble
	if (dwEnd < dwStart)
		dwEnd = dwStart + dwEnd;

	if (szFileName == NULL || lstrlen(szFileName) == 0)
	{
		Dump_GetDumpDirectory(szFilePath, TEXT(""));
		PathAppend(szFilePath, TEXT("dis.txt"));
	}
	else
	{
		lstrcpyn(szFilePath, szFileName, MAX_PATH);
	}

		
	DWORD * pdwBase;
	if (InternalReadAddress(dwStart, (void**)&pdwBase) == MEM_UNUSED)
	{
		DBGConsole_Msg(0, "[Ydis: Invalid base 0x%08x]", dwStart);
	}
	else
	{
		DBGConsole_Msg(0, "Disassembling from 0x%08x to 0x%08x ([C%s])",
			dwStart, dwEnd, szFilePath);

		DumpRange(szFilePath, "w", pdwBase, dwStart, dwEnd, TRUE);
	}
}




//*****************************************************************************
// pdwBase[0] = instruction at dwStart
//*****************************************************************************
static void DumpRSPRange(LPCTSTR szFileName,
						 LPCTSTR szMode,
						 DWORD * pdwBase,
						 DWORD dwStart,
						 DWORD dwEnd)
{
	char opinfo[400];
	FILE * fp;	

	fp = fopen(szFileName, szMode);
	if (fp == NULL)
		return;

	for (DWORD dwPC = dwStart; dwPC < dwEnd; dwPC+=4)
	{
		DWORD dwOpCode = pdwBase[(dwPC-dwStart)/4];

		SprintRSPOpCodeInfo(opinfo, dwPC, dwOpCode);
		fprintf(fp, "0x%08x: <0x%08x> %s\n", dwPC, dwOpCode, opinfo);
	}

	fclose(fp);
}

//*****************************************************************************
//
//*****************************************************************************
void Dump_RSPDisassemble(LPCTSTR szFileName)
{
	TCHAR szFilePath[MAX_PATH+1];
	FILE * fp;
	DWORD dwStart = 0xa4000000;
	DWORD dwEnd = 0xa4002000;

	if (szFileName == NULL || lstrlen(szFileName) == 0)
	{
		Dump_GetDumpDirectory(szFilePath, TEXT(""));
		PathAppend(szFilePath, TEXT("rdis.txt"));
	}
	else
	{
		lstrcpyn(szFilePath, szFileName, MAX_PATH);
	}


	DWORD * pdwBase;
	if (InternalReadAddress(dwStart, (void**)&pdwBase) == MEM_UNUSED)
	{
		DBGConsole_Msg(0, "[Yrdis: Invalid base 0x%08x]", dwStart);
		return;
	}

	DBGConsole_Msg(0, "Disassembling from 0x%08x to 0x%08x ([C%s])",
		dwStart, dwEnd, szFilePath);


	// Overwrite here
	fp = fopen(szFilePath, "w");
	if (fp == NULL)
		return;
	
	// Memory dump
	DWORD dwPC;
	for (dwPC = 0xa4000000; dwPC < 0xa4001000; dwPC += 16)
	{
		fprintf(fp, "0x%08x: %08x %08x %08x %08x ", dwPC, 
			pdwBase[(dwPC-dwStart)/4 + 0],
			pdwBase[(dwPC-dwStart)/4 + 1],
			pdwBase[(dwPC-dwStart)/4 + 2],
			pdwBase[(dwPC-dwStart)/4 + 3]);

			// Concat characters
			for (int i = 0; i < 16; i++)
			{
				char c = ((BYTE*)pdwBase)[((dwPC-dwStart)+i) ^ 0x3];
				// The cast to unsigned int avoids widening char to a signed int.
				// This cast avoids a failed assertion popup in MSVC 7.0
				if (isprint((unsigned int)(unsigned char)c))
					fprintf(fp, "%c", c);
				else 
					fprintf(fp, ".");

				if ((i%4)==3)
					fprintf(fp, " ");
			}
			fprintf(fp, "\n");



	}
	fclose(fp);

	// Append here
	DumpRSPRange(szFilePath, "a", pdwBase+(0x1000/4), dwStart+0x1000, dwEnd);

}


//*****************************************************************************
//
//*****************************************************************************
void Dump_Strings( LPCTSTR szFileName )
{
	TCHAR szFilePath[ MAX_PATH+1 ];
	FILE * fp;

	static const u32 min_length = 5;

	if (szFileName == NULL || lstrlen(szFileName) == 0)
	{
		Dump_GetDumpDirectory(szFilePath, TEXT(""));
		PathAppend(szFilePath, TEXT("strings.txt"));
	}
	else
	{
		lstrcpyn(szFilePath, szFileName, MAX_PATH);
	}


	extern DWORD MemoryRegionSizes[];	// Ugh!
	u8 * p_base = (u8*)g_pMemoryBuffers[ MEM_CARTROM ];
	u32 length = MemoryRegionSizes[ MEM_CARTROM ];


	DBGConsole_Msg(0, "Dumping strings in rom ([C%s])", szFilePath);
	
	// Overwrite here
	fp = fopen(szFilePath, "w");
	if (fp == NULL)
		return;
	
	// Memory dump
	u32 ascii_start;
	u32 ascii_count = 0;
	for ( u32 i = 0; i < length; i ++)
	{
		if ( p_base[ i ^ 0x3 ] >= ' ' &&
			 p_base[ i ^ 0x3 ] < 200 )
		{
			if ( ascii_count == 0 )
			{
				ascii_start = i;
			}
			ascii_count++;
		}
		else
		{
			if ( ascii_count >= min_length )
			{
				fprintf( fp, "0x%08x: ", ascii_start );

				for ( u32 j = 0; j < ascii_count; j++ )
				{
					fprintf( fp, "%c", p_base[ (ascii_start + j ) ^ 0x3 ] );
				}

				fprintf( fp, "\n");
			}

			ascii_count = 0;
		}
	}
	fclose(fp);

}


