////////////////////////////////////////////////////////////////////////////////
//
// Glide 3.0 header adaption for Delphi/Pascal by Erik Unger
// (Based on the Glide 2.43 adaption from Erik Unger and Eric Nowinski)
//
// This Glide3x.pas unit contains the follwing converted C-headers:
// 3Dfx.h, Sst1vid.h, GlideSys.h, Glide.h and GlideUtl.h
//
// Download: http://www.bigfoot.com/~ungerik/
// E-Mail: Erik Unger <ungerik@bigfoot.com>
//
//
// This unit has 3 managment-functions:
//
// function InitGlide : Boolean;
// Loads the proc-addresses from Glide3x.DLL at runtime.
// It's called automaticaliy from the Initialization section,
// so you don't have to worry about it.
//
// InitGlideFromLibrary(PathDLL: string) : Boolean;
// Loads the proc-addresses from PathDLL at runtime.
// Use it, if you want to link a different DLL
// than the Glide3x.DLL in the current or System directory
//
// procedure CloseGlide;
// Releases the runtime-link to a Glide DLL and sets all proc-addresses to nil.
// You don't have to call this procedure, because InitGlide and
// the Finalization section do it for you.
//
// function CheckProcAddresses : integer;
// Returns the number of ProcAddresses which are nil
//
// Modyfied: 9. Oct. 1998
//
// 1. Added the parameter pixelPipeline: TFxBool to grLfbWriteRegion
// 2. Added CheckProcAddresses to test every ProcAddress for nil
// 3. Fixed GetProcAddress for grGlideSetVertexLayout
// 4. Fixed GR_QUERY_ANY
// 5. Fixed grDrawVertexArray and some functions with CONST parameters
//
////////////////////////////////////////////////////////////////////////////////


unit Glide3x;

{$MINENUMSIZE 4}
{$ALIGN ON}
{$DEFINE GLIDE3}

interface

uses
  Windows,
  SysUtils;

{$IFNDEF VER110}
type
  LongWord = LongInt;
{$ENDIF}


////////////////////////////////////////////////////////////////////////////////
// 3Dfx.h
////////////////////////////////////////////////////////////////////////////////

(*
** Copyright (c) 1995, 3Dfx Interactive, Inc.
** All Rights Reserved.
**
** This is UNPUBLISHED PROPRIETARY SOURCE CODE of 3Dfx Interactive, Inc.;
** the contents of this file may not be disclosed to third parties, copied or
** duplicated in any form, in whole or in part, without the prior written
** permission of 3Dfx Interactive, Inc.
**
** RESTRICTED RIGHTS LEGEND:
** Use, duplication or disclosure by the Government is subject to restrictions
** as set forth in subdivision (c)(1)(ii) of the Rights in Technical Data
** and Computer Software clause at DFARS 252.227-7013, and/or in similar or
** successor clauses in the FAR, DOD or NASA FAR Supplement. Unpublished  -
** rights reserved under the Copyright Laws of the United States.
**
** $Revision: 9 $
** $Date: 12/23/97 9:57a $
*)

(*
** basic data types
*)
type
  TFxU8      = Byte;
  TFxI8      = Shortint;
  TFxU16     = Word;
  TFxI16     = Smallint;
  TFxU32     = LongWord;
  TFxI32     = LongInt;
  TFxBool    = LongBool;
  TFxFloat   = Single;
  TFxDouble  = Double;

(*
** color types
*)
  TFxColor = LongWord;
  TFxColor4  = packed record
    r,g,b,a : TFxFloat;
  end;

(*
** fundamental types
*)
const
  FXTRUE = True;
  FXFALSE = False;

(*
** helper macros
*)

function FXUNUSED(a: variant) : boolean;
function FXBIT(i: variant) : variant;

////////////////////////////////////////////////////////////////////////////////
// Sst1vid.h
////////////////////////////////////////////////////////////////////////////////

(*
** Copyright (c) 1995, 3Dfx Interactive, Inc.
** All Rights Reserved.
**
** This is UNPUBLISHED PROPRIETARY SOURCE CODE of 3Dfx Interactive, Inc.;
** the contents of this file may not be disclosed to third parties, copied or
** duplicated in any form, in whole or in part, without the prior written
** permission of 3Dfx Interactive, Inc.
**
** RESTRICTED RIGHTS LEGEND:
** Use, duplication or disclosure by the Government is subject to restrictions
** as set forth in subdivision (c)(1)(ii) of the Rights in Technical Data
** and Computer Software clause at DFARS 252.227-7013, and/or in similar or
** successor clauses in the FAR, DOD or NASA FAR Supplement. Unpublished -
** rights reserved under the Copyright Laws of the United States.
**
** $Header: /devel/cvg/incsrc/SST1VID.H 6     7/24/98 1:41p Hohn $
** $Log: /devel/cvg/incsrc/SST1VID.H $
** 
** 6     7/24/98 1:41p Hohn
** 
** 5     3/10/98 2:33p Psmith
** separating cvg tree from h3/h4 trees
 * 
 * 4     9/09/97 7:35p Sellers
 * Added 400x300 resolution
 * 
 * 3     8/24/97 9:31a Sellers
 * moved new video timing to sst1vid.h
 * redefined 1600x1280 to be 1600x1200
 * 
 * 2     6/05/97 11:14p Pgj
 * 
 * 5     7/24/96 3:43p Sellers
 * added 512x384 @ 60 Hz for arcade monitors
 * added 512x256 @ 60 Hz for arcade monitors
 * 
 * 4     7/18/96 10:58a Sellers
 * fixed FT and TF clock delay values for lower frequencies with
 * .5/.5 combos
 * 
 * 3     6/18/96 6:54p Sellers
 * added sst1InitShutdownSli() to fix Glide Splash screen problems with
 * SLI
 * 
 * 2     6/13/96 7:45p Sellers
 * added "voodoo.ini" support
 * added DirectX support
 * misc cleanup
 * 
 * 2     6/11/96 1:43p Sellers
 * added support for 60, 75, 85, and 120 Hz refresh rates for "most"
 * resolutions
 * 
 * 1     5/08/96 5:43p Paik
 * Video definitions
*)

type
  TGrProc = pointer;
  
(* Video defines *)
  TGrScreenRefresh = TFxU32;

const
  GR_REFRESH_60Hz = 0;
  GR_REFRESH_70Hz = 1;
  GR_REFRESH_72Hz = 2;
  GR_REFRESH_75Hz = 3;
  GR_REFRESH_80Hz = 4;
  GR_REFRESH_90Hz = 5;
  GR_REFRESH_100Hz = 6;
  GR_REFRESH_85Hz = 7;
  GR_REFRESH_120Hz = 8;
  GR_REFRESH_NONE = $ff;

type
  TGrScreenResolution = TFxU32;

const
  GR_RESOLUTION_320x200 = 0;
  GR_RESOLUTION_320x240 = 1;
  GR_RESOLUTION_400x256 = 2;
  GR_RESOLUTION_512x384 = 3;
  GR_RESOLUTION_640x200 = 4;
  GR_RESOLUTION_640x350 = 5;
  GR_RESOLUTION_640x400 = 6;
  GR_RESOLUTION_640x480 = 7;
  GR_RESOLUTION_800x600 = 8;
  GR_RESOLUTION_960x720 = 9;
  GR_RESOLUTION_856x480 = 10;
  GR_RESOLUTION_512x256 = 11;
  GR_RESOLUTION_1024x768 = 12;
  GR_RESOLUTION_1280x1024 = 13;
  GR_RESOLUTION_1600x1200 = 14;
  GR_RESOLUTION_400x300 = 15;
  GR_RESOLUTION_NONE = $ff;

  GR_RESOLUTION_MIN       = GR_RESOLUTION_320x200;
  GR_RESOLUTION_MAX	  = GR_RESOLUTION_1600x1200;

////////////////////////////////////////////////////////////////////////////////
// GlideSys.h
////////////////////////////////////////////////////////////////////////////////

(*
** Copyright (c) 1995, 3Dfx Interactive, Inc.
** All Rights Reserved.
**
** This is UNPUBLISHED PROPRIETARY SOURCE CODE of 3Dfx Interactive, Inc.;
** the contents of this file may not be disclosed to third parties, copied or
** duplicated in any form, in whole or in part, without the prior written
** permission of 3Dfx Interactive, Inc.
**
** RESTRICTED RIGHTS LEGEND:
** Use, duplication or disclosure by the Government is subject to restrictions
n** as set forth in subdivision (c)(1)(ii) of the Rights in Technical Data
** and Computer Software clause at DFARS 252.227-7013, and/or in similar or
** successor clauses in the FAR, DOD or NASA FAR Supplement. Unpublished -
** rights reserved under the Copyright Laws of the United States.
**
** $Header: /devel/cvg/glide3/src/glidesys.h 3     7/24/98 1:41p Hohn $
** $Log: /devel/cvg/glide3/src/glidesys.h $
** 
** 3     7/24/98 1:41p Hohn
**
** 2     6/15/98 10:50a Peter
** made csim compile time option
 * 
 * 1     1/16/98 4:29p Atai
 * create glide 3 src
 * 
 * 10    12/09/97 12:20p Peter
 * mac glide port
 * 
 * 9     11/04/97 4:00p Dow
 * Banshee Mods
 * 
 * 8     8/18/97 3:52p Peter
 * pre-hw arrival fixes/cleanup
 * 
 * 7     6/02/97 4:09p Peter
 * Compile w/ gcc for Dural
 * 
 * 6     5/27/97 1:16p Peter
 * Basic cvg, w/o cmd fifo stuff. 
 * 
 * 5     5/21/97 6:05a Peter
*)

(* Endianness is stored in bits [30:31] *)
const
  GLIDE_ENDIAN_SHIFT  = 30;
  GLIDE_ENDIAN_LITTLE = $1 shl GLIDE_ENDIAN_SHIFT;
  GLIDE_ENDIAN_BIG    = $2 shl GLIDE_ENDIAN_SHIFT;

(* OS is stored in bits [0:6] *)
  GLIDE_OS_SHIFT   = $0;
  GLIDE_OS_UNIX    = $1;
  GLIDE_OS_DOS32   = $2;
  GLIDE_OS_WIN32   = $4;
  GLIDE_OS_SYSTEM7 = $8;
  GLIDE_OS_OS2     = $10;
  GLIDE_OS_OTHER   = $40; (* For Proprietary Arcade HW *)

(* Sim vs. Hardware is stored in bits [7:8] *)
  GLIDE_SST_SHIFT  = $7;
  GLIDE_SST_SIM    = $1 shl GLIDE_SST_SHIFT;
  GLIDE_SST_HW     = $2 shl GLIDE_SST_SHIFT;

(* Hardware Type is stored in bits [9:13] *)
  GLIDE_HW_SHIFT   = $9;
  GLIDE_HW_SST1    = $1 shl GLIDE_HW_SHIFT;
  GLIDE_HW_SST96   = $2 shl GLIDE_HW_SHIFT;
  GLIDE_HW_H3      = $4 shl GLIDE_HW_SHIFT;
  GLIDE_HW_SST2    = $8 shl GLIDE_HW_SHIFT;
  GLIDE_HW_CVG     = $10 shl GLIDE_HW_SHIFT;

  GLIDE_ENDIAN = GLIDE_ENDIAN_LITTLE;
  GLIDE_OS = GLIDE_OS_WIN32;
  GLIDE_SST = GLIDE_SST_HW;
  GLIDE_HW = GLIDE_HW_SST1;

  GLIDE_PLATFORM = GLIDE_ENDIAN or GLIDE_OS or GLIDE_SST or GLIDE_HW;
  GLIDE_NUM_TMU  = 2;

////////////////////////////////////////////////////////////////////////////////
// Glide.h
////////////////////////////////////////////////////////////////////////////////

(*
** Copyright (c) 1995, 3Dfx Interactive, Inc.
** All Rights Reserved.
**
** This is UNPUBLISHED PROPRIETARY SOURCE CODE of 3Dfx Interactive, Inc.;
** the contents of this file may not be disclosed to third parties, copied or
** duplicated in any form, in whole or in part, without the prior written
** permission of 3Dfx Interactive, Inc.
**
** RESTRICTED RIGHTS LEGEND:
** Use, duplication or disclosure by the Government is subject to restrictions
** as set forth in subdivision (c)(1)(ii) of the Rights in Technical Data
** and Computer Software clause at DFARS 252.227-7013, and/or in similar or
** successor clauses in the FAR, DOD or NASA FAR Supplement. Unpublished -
** rights reserved under the Copyright Laws of the United States.
*/

/*
** GLIDE.H
**
** The following #defines are relevant when using Glide:
**
** One of the following "platform constants" must be defined during
** compilation:
**
**            __DOS__           Defined for 32-bit DOS applications
**            __WIN32__         Defined for 32-bit Windows applications
**            __sparc__         Defined for Sun Solaris/SunOS
**            __linux__         Defined for Linux applications
**            __IRIX__          Defined for SGI Irix applications
**
*)

(*
** -----------------------------------------------------------------------
** TYPE DEFINITIONS
** -----------------------------------------------------------------------
*)
type
  TGrColor = TFxU32;
  TGrAlpha = TFxU8;
  TGrMipMapId = TFxU32;
  PGrFog = ^TGrFog;
  TGrFog = TFxU8;
  TGrContext = TFxU32;

(*
** -----------------------------------------------------------------------
** CONSTANTS AND TYPES
** -----------------------------------------------------------------------
*)
const
  GR_NULL_MIPMAP_HANDLE  = -1;

  GR_MIPMAPLEVELMASK_EVEN = (1 shl 0);
  GR_MIPMAPLEVELMASK_ODD  = (1 shl 1);
  GR_MIPMAPLEVELMASK_BOTH = GR_MIPMAPLEVELMASK_EVEN or GR_MIPMAPLEVELMASK_ODD;

  GR_LODBIAS_BILINEAR  = 0.5;
  GR_LODBIAS_TRILINEAR = 0.0;

type
  TGrChipID = TFxI32;
const
  GR_TMU0 = $0;
  GR_TMU1 = $1;
  GR_TMU2 = $2;

  GR_FBI  = $3;

type
  TGrCombineFunction = (
    GR_COMBINE_FUNCTION_ZERO,
    GR_COMBINE_FUNCTION_LOCAL,
    GR_COMBINE_FUNCTION_LOCAL_ALPHA,
    GR_COMBINE_FUNCTION_SCALE_OTHER,
    GR_COMBINE_FUNCTION_SCALE_OTHER_ADD_LOCAL,
    GR_COMBINE_FUNCTION_SCALE_OTHER_ADD_LOCAL_ALPHA,
    GR_COMBINE_FUNCTION_SCALE_OTHER_MINUS_LOCAL,
    GR_COMBINE_FUNCTION_SCALE_OTHER_MINUS_LOCAL_ADD_LOCAL,
    GR_COMBINE_FUNCTION_SCALE_OTHER_MINUS_LOCAL_ADD_LOCAL_ALPHA,
    GR_COMBINE_FUNCTION_SCALE_MINUS_LOCAL_ADD_LOCAL,
    GR_COMBINE_FUNCTION_SCALE_MINUS_LOCAL_ADD_LOCAL_ALPHA
  );

const
  GR_COMBINE_FUNCTION_NONE = GR_COMBINE_FUNCTION_ZERO;
  GR_COMBINE_FUNCTION_BLEND_OTHER = GR_COMBINE_FUNCTION_SCALE_OTHER;
  GR_COMBINE_FUNCTION_BLEND
      = GR_COMBINE_FUNCTION_SCALE_OTHER_MINUS_LOCAL_ADD_LOCAL;
  GR_COMBINE_FUNCTION_BLEND_LOCAL
      = GR_COMBINE_FUNCTION_SCALE_MINUS_LOCAL_ADD_LOCAL;

type
  TGrCombineFactor = (
    GR_COMBINE_FACTOR_ZERO,
    GR_COMBINE_FACTOR_LOCAL,
    GR_COMBINE_FACTOR_OTHER_ALPHA,
    GR_COMBINE_FACTOR_LOCAL_ALPHA,
    GR_COMBINE_FACTOR_TEXTURE_ALPHA,
    GR_COMBINE_FACTOR_LOD_FRACTION,
    GR_COMBINE_INVALID_6,
    GR_COMBINE_INVALID_7,
    GR_COMBINE_FACTOR_ONE,
    GR_COMBINE_FACTOR_ONE_MINUS_LOCAL,
    GR_COMBINE_FACTOR_ONE_MINUS_OTHER_ALPHA,
    GR_COMBINE_FACTOR_ONE_MINUS_LOCAL_ALPHA,
    GR_COMBINE_FACTOR_ONE_MINUS_TEXTURE_ALPHA,
    GR_COMBINE_FACTOR_ONE_MINUS_LOD_FRACTION
  );
const
  GR_COMBINE_FACTOR_NONE = GR_COMBINE_FACTOR_ZERO;
  GR_COMBINE_FACTOR_TEXTURE_RGB = GR_COMBINE_FACTOR_LOD_FRACTION;   //?
  GR_COMBINE_FACTOR_DETAIL_FACTOR = GR_COMBINE_FACTOR_TEXTURE_ALPHA;
  GR_COMBINE_FACTOR_ONE_MINUS_DETAIL_FACTOR =
      GR_COMBINE_FACTOR_ONE_MINUS_TEXTURE_ALPHA;

type
  TGrCombineLocal = (
    GR_COMBINE_LOCAL_ITERATED,
    GR_COMBINE_LOCAL_CONSTANT,
    GR_COMBINE_LOCAL_DEPTH
  );
  
const
  GR_COMBINE_LOCAL_NONE = GR_COMBINE_LOCAL_CONSTANT;

type
  TGrCombineOther = (
    GR_COMBINE_OTHER_ITERATED,
    GR_COMBINE_OTHER_TEXTURE,
    GR_COMBINE_OTHER_CONSTANT
  );
const
  GR_COMBINE_OTHER_NONE = GR_COMBINE_OTHER_CONSTANT;

type
  TGrAlphaSource = (
    GR_ALPHASOURCE_CC_ALPHA,
    GR_ALPHASOURCE_ITERATED_ALPHA,
    GR_ALPHASOURCE_TEXTURE_ALPHA,
    GR_ALPHASOURCE_TEXTURE_ALPHA_TIMES_ITERATED_ALPHA
  );

  TGrColorCombineFnc = (
    GR_COLORCOMBINE_ZERO,
    GR_COLORCOMBINE_CCRGB,
    GR_COLORCOMBINE_ITRGB,
    GR_COLORCOMBINE_ITRGB_DELTA0,
    GR_COLORCOMBINE_DECAL_TEXTURE,
    GR_COLORCOMBINE_TEXTURE_TIMES_CCRGB,
    GR_COLORCOMBINE_TEXTURE_TIMES_ITRGB,
    GR_COLORCOMBINE_TEXTURE_TIMES_ITRGB_DELTA0,
    GR_COLORCOMBINE_TEXTURE_TIMES_ITRGB_ADD_ALPHA,
    GR_COLORCOMBINE_TEXTURE_TIMES_ALPHA,
    GR_COLORCOMBINE_TEXTURE_TIMES_ALPHA_ADD_ITRGB,
    GR_COLORCOMBINE_TEXTURE_ADD_ITRGB,
    GR_COLORCOMBINE_TEXTURE_SUB_ITRGB,
    GR_COLORCOMBINE_CCRGB_BLEND_ITRGB_ON_TEXALPHA,
    GR_COLORCOMBINE_DIFF_SPEC_A,
    GR_COLORCOMBINE_DIFF_SPEC_B,
    GR_COLORCOMBINE_ONE
  );

type
  TGrAlphaBlendFnc = (
    GR_BLEND_ZERO,
    GR_BLEND_SRC_ALPHA,
    GR_BLEND_SRC_COLOR,
    GR_BLEND_DST_ALPHA,
    GR_BLEND_ONE,
    GR_BLEND_ONE_MINUS_SRC_ALPHA,
    GR_BLEND_ONE_MINUS_SRC_COLOR,
    GR_BLEND_ONE_MINUS_DST_ALPHA,
    GR_BLEND_RESERVED_8,
    GR_BLEND_RESERVED_9,
    GR_BLEND_RESERVED_A,
    GR_BLEND_RESERVED_B,
    GR_BLEND_RESERVED_C,
    GR_BLEND_RESERVED_D,
    GR_BLEND_RESERVED_E,
    GR_BLEND_ALPHA_SATURATE
  );

const
  GR_BLEND_DST_COLOR = GR_BLEND_SRC_COLOR;
  GR_BLEND_ONE_MINUS_DST_COLOR = GR_BLEND_ONE_MINUS_SRC_COLOR;
  GR_BLEND_PREFOG_COLOR = GR_BLEND_ALPHA_SATURATE;

type
  TGrAspectRatio = TFxI32;

const
  GR_ASPECT_LOG2_8x1      =  3;       (* 8W x 1H *)
  GR_ASPECT_LOG2_4x1      =  2;       (* 4W x 1H *)
  GR_ASPECT_LOG2_2x1      =  1;       (* 2W x 1H *)
  GR_ASPECT_LOG2_1x1      =  0;       (* 1W x 1H *)
  GR_ASPECT_LOG2_1x2      = -1;       (* 1W x 2H *)
  GR_ASPECT_LOG2_1x4      = -2;       (* 1W x 4H *)
  GR_ASPECT_LOG2_1x8      = -3;       (* 1W x 8H *)

type
  TGrBuffer = (
    GR_BUFFER_FRONTBUFFER,
    GR_BUFFER_BACKBUFFER,
    GR_BUFFER_AUXBUFFER,
    GR_BUFFER_DEPTHBUFFER,
    GR_BUFFER_ALPHABUFFER,
    GR_BUFFER_TRIPLEBUFFER
  );

  TGrChromakeyMode = (
    GR_CHROMAKEY_DISABLE,
    GR_CHROMAKEY_ENABLE
  );

  TGrChromaRangeMode = (
    GR_CHROMARANGE_DISABLE_EXT,
    GR_CHROMARANGE_ENABLE_EXT
  );

const
  GR_CHROMARANGE_RGB_ALL_EXT = TGrChromaRangeMode(0);

type
  TGrTexChromakeyMode = (
    GR_TEXCHROMA_DISABLE_EXT,
    GR_TEXCHROMA_ENABLE_EXT
  );

const
  GR_TEXCHROMARANGE_RGB_ALL_EXT = TGrTexChromakeyMode(0);

type
  TGrCmpFnc = (
    GR_CMP_NEVER,
    GR_CMP_LESS,
    GR_CMP_EQUAL,
    GR_CMP_LEQUAL,
    GR_CMP_GREATER,
    GR_CMP_NOTEQUAL,
    GR_CMP_GEQUAL,
    GR_CMP_ALWAYS
  );

  TGrColorFormat = (
    GR_COLORFORMAT_ARGB,
    GR_COLORFORMAT_ABGR,
    GR_COLORFORMAT_RGBA,
    GR_COLORFORMAT_BGRA
  );

  TGrCullMode = (
    GR_CULL_DISABLE,
    GR_CULL_NEGATIVE,
    GR_CULL_POSITIVE
  );

  TGrDepthBufferMode = (
    GR_DEPTHBUFFER_DISABLE,
    GR_DEPTHBUFFER_ZBUFFER,
    GR_DEPTHBUFFER_WBUFFER,
    GR_DEPTHBUFFER_ZBUFFER_COMPARE_TO_BIAS,
    GR_DEPTHBUFFER_WBUFFER_COMPARE_TO_BIAS
  );

  TGrDitherMode = (
    GR_DITHER_DISABLE,
    GR_DITHER_2x2,
    GR_DITHER_4x4
  );

type
  TGrFogMode = TFxI32;
const
  GR_FOG_DISABLE         = $0;
  GR_FOG_WITH_TABLE_ON_FOGCOORD_EXT = $1;
  GR_FOG_WITH_TABLE_ON_Q = $2;
  GR_FOG_WITH_TABLE_ON_W = GR_FOG_WITH_TABLE_ON_Q;
  GR_FOG_WITH_ITERATED_Z = $3;
  GR_FOG_MULT2           = $100;
  GR_FOG_ADD2            = $200;

type
  TGrLock = TFxU32;
const
  GR_LFB_READ_ONLY  = $00;
  GR_LFB_WRITE_ONLY = $01;
  GR_LFB_IDLE       = $00;
  GR_LFB_NOIDLE     = $10;

type
  TGrLfbBypassMode = (
    GR_LFBBYPASS_DISABLE,
    GR_LFBBYPASS_ENABLE
  );

  TGrLfbWriteMode = (
    GR_LFBWRITEMODE_565, // RGB:RGB
    GR_LFBWRITEMODE_555, // RGB:RGB
    GR_LFBWRITEMODE_1555, // ARGB:ARGB
    GR_LFBWRITEMODE_RESERVED1,
    GR_LFBWRITEMODE_888, // RGB
    GR_LFBWRITEMODE_8888, // ARGB
    GR_LFBWRITEMODE_RESERVED2,
    GR_LFBWRITEMODE_RESERVED3,
    GR_LFBWRITEMODE_RESERVED4,
    GR_LFBWRITEMODE_RESERVED5,
    GR_LFBWRITEMODE_RESERVED6,
    GR_LFBWRITEMODE_RESERVED7,
    GR_LFBWRITEMODE_565_DEPTH, // RGB:DEPTH
    GR_LFBWRITEMODE_555_DEPTH, // RGB:DEPTH
    GR_LFBWRITEMODE_1555_DEPTH,// ARGB:DEPTH
    GR_LFBWRITEMODE_ZA16       // DEPTH:DEPTH
  );

const
  GR_LFBWRITEMODE_ANY       = TGrLfbWriteMode($FF);

type
  TGrOriginLocation = (
    GR_ORIGIN_UPPER_LEFT,
    GR_ORIGIN_LOWER_LEFT
  );
const
  GR_ORIGIN_ANY           = TGrOriginLocation($FF);

type
  TGrLfbInfo = packed record
    size : Integer;
    lfbPtr : Pointer;
    strideInBytes : TFxU32;
    writeMode : TGrLfbWriteMode;
    origin : TGrOriginLocation;
  end;

  TGrLOD = (
    GR_LOD_LOG2_1,
    GR_LOD_LOG2_2,
    GR_LOD_LOG2_4,
    GR_LOD_LOG2_8,
    GR_LOD_LOG2_16,
    GR_LOD_LOG2_32,
    GR_LOD_LOG2_64,
    GR_LOD_LOG2_128,
    GR_LOD_LOG2_256
  );

  TGrMipMapMode = (
    GR_MIPMAP_DISABLE, // no mip mapping
    GR_MIPMAP_NEAREST, // use nearest mipmap
    GR_MIPMAP_NEAREST_DITHER // GR_MIPMAP_NEAREST + LOD dith
  );

  TGrSmoothingMode = (
    GR_SMOOTHING_DISABLE,
    GR_SMOOTHING_ENABLE
  );

  TGrTextureClampMode = (
    GR_TEXTURECLAMP_WRAP,
    GR_TEXTURECLAMP_CLAMP,
    GR_TEXTURECLAMP_MIRROR_EXT
  );

  TGrTextureCombineFnc = (
    GR_TEXTURECOMBINE_ZERO, (* texout = 0 *)
    GR_TEXTURECOMBINE_DECAL, (* texout = texthis *)
    GR_TEXTURECOMBINE_OTHER, (* this TMU in passthru mode *)
    GR_TEXTURECOMBINE_ADD, (* tout = tthis + t(this+1) *)
    GR_TEXTURECOMBINE_MULTIPLY, (* texout = tthis * t(this+1) *)
    GR_TEXTURECOMBINE_SUBTRACT, (* Sutract from upstream TMU *)
    GR_TEXTURECOMBINE_DETAIL, (* detail--detail on tthis *)
    GR_TEXTURECOMBINE_DETAIL_OTHER, (* detail--detail on tthis+1 *)
    GR_TEXTURECOMBINE_TRILINEAR_ODD, (* trilinear--odd levels tthis*)
    GR_TEXTURECOMBINE_TRILINEAR_EVEN, (*trilinear--even levels tthis*)
    GR_TEXTURECOMBINE_ONE (* texout = $FFFFFFFF *)
  );

  TGrTextureFilterMode = (
    GR_TEXTUREFILTER_POINT_SAMPLED,
    GR_TEXTUREFILTER_BILINEAR
  );

  TGrTextureFormat = (
    GR_TEXFMT_8BIT,
    GR_TEXFMT_YIQ_422,
    GR_TEXFMT_ALPHA_8, // (0..0xFF) alpha     */
    GR_TEXFMT_INTENSITY_8, // (0..0xFF) intensity */
    GR_TEXFMT_ALPHA_INTENSITY_44,
    GR_TEXFMT_P_8, // 8-bit palette */
    GR_TEXFMT_RSVD0,
    GR_TEXFMT_RSVD1,
    GR_TEXFMT_16BIT,
    GR_TEXFMT_AYIQ_8422,
    GR_TEXFMT_RGB_565,
    GR_TEXFMT_ARGB_1555,
    GR_TEXFMT_ARGB_4444,
    GR_TEXFMT_ALPHA_INTENSITY_88,
    GR_TEXFMT_AP_88, // 8-bit alpha 8-bit palette */
    GR_TEXFMT_RSVD2
  );

const
  GR_TEXFMT_RGB_332               = GR_TEXFMT_8BIT;
  GR_TEXFMT_ARGB_8332             = GR_TEXFMT_16BIT;

type
  TGrTexTable = (
    GR_TEXTABLE_NCC0,
    GR_TEXTABLE_NCC1,
    GR_TEXTABLE_PALETTE,
    GR_TEXTABLE_PALETTE_6666_EXT
  );

  TGrNCCTable = (
    GR_NCCTABLE_NCC0,
    GR_NCCTABLE_NCC1
  );

  TGrTexBaseRange = (
    GR_TEXBASE_32_TO_1,
    GR_TEXBASE_64,
    GR_TEXBASE_128,
    GR_TEXBASE_256
  );

  TGrEnableMode = (
    GR_MODE_DISABLE,
    GR_AA_ORDERED,
    GR_ALLOW_MIPMAP_DITHER,
    GR_PASSTHRU,
    GR_SHAMELESS_PLUG,
    GR_VIDEO_SMOOTHING
  );

const
  GR_MODE_ENABLE = TGrEnableMode(1);

type
  TGrCoordinateSpaceMode = (
    GR_WINDOW_COORDS,
    GR_CLIP_COORDS
  );

const
(* Types of data in strips *)
  GR_FLOAT       = 0;
  GR_U8          = 1;

(* Parameters for strips *)
  GR_PARAM_XY       = $01;
  GR_PARAM_Z        = $02;
  GR_PARAM_W        = $03;
  GR_PARAM_Q        = $04;
  GR_PARAM_FOG_EXT  = $05;

  GR_PARAM_A        = $10;

  GR_PARAM_RGB      = $20;

  GR_PARAM_PARGB    = $30;

  GR_PARAM_ST0      = $40;
  GR_PARAM_ST1      = GR_PARAM_ST0+1;
  GR_PARAM_ST2      = GR_PARAM_ST0+2;

  GR_PARAM_Q0       = $50;
  GR_PARAM_Q1       = GR_PARAM_Q0+1;
  GR_PARAM_Q2       = GR_PARAM_Q0+2;

  GR_PARAM_DISABLE  = $00;
  GR_PARAM_ENABLE   = $01;


(*
** grDrawVertexArray/grDrawVertexArrayContiguous primitive type
*)
  GR_POINTS                       = 0;
  GR_LINE_STRIP                   = 1;
  GR_LINES                        = 2;
  GR_POLYGON                      = 3;
  GR_TRIANGLE_STRIP               = 4;
  GR_TRIANGLE_FAN                 = 5;
  GR_TRIANGLES                    = 6;
  GR_TRIANGLE_STRIP_CONTINUE      = 7;
  GR_TRIANGLE_FAN_CONTINUE        = 8;

(* 
** grGet/grReset types
*)
  GR_BITS_DEPTH                   = $01;
  GR_BITS_RGBA                    = $02;
  GR_FIFO_FULLNESS                = $03;
  GR_FOG_TABLE_ENTRIES            = $04;
  GR_GAMMA_TABLE_ENTRIES          = $05;
  GR_GLIDE_STATE_SIZE             = $06;
  GR_GLIDE_VERTEXLAYOUT_SIZE      = $07;
  GR_IS_BUSY                      = $08;
  GR_LFB_PIXEL_PIPE               = $09;
  GR_MAX_TEXTURE_SIZE             = $0a;
  GR_MAX_TEXTURE_ASPECT_RATIO     = $0b;
  GR_MEMORY_FB                    = $0c;
  GR_MEMORY_TMU                   = $0d;
  GR_MEMORY_UMA                   = $0e;
  GR_NUM_BOARDS                   = $0f;
  GR_NON_POWER_OF_TWO_TEXTURES    = $10;
  GR_NUM_FB                       = $11;
  GR_NUM_SWAP_HISTORY_BUFFER      = $12;
  GR_NUM_TMU                      = $13;
  GR_PENDING_BUFFERSWAPS          = $14;
  GR_REVISION_FB                  = $15;
  GR_REVISION_TMU                 = $16;
  GR_STATS_LINES                  = $17;  (* grGet/grReset *)
  GR_STATS_PIXELS_AFUNC_FAIL      = $18;
  GR_STATS_PIXELS_CHROMA_FAIL     = $19;
  GR_STATS_PIXELS_DEPTHFUNC_FAIL  = $1a;
  GR_STATS_PIXELS_IN              = $1b;
  GR_STATS_PIXELS_OUT             = $1c;
  GR_STATS_PIXELS                 = $1d;  (* grReset *)
  GR_STATS_POINTS                 = $1e;  (* grGet/grReset *)
  GR_STATS_TRIANGLES_IN           = $1f;
  GR_STATS_TRIANGLES_OUT          = $20;
  GR_STATS_TRIANGLES              = $21;  (* grReset *)
  GR_SWAP_HISTORY                 = $22;
  GR_SUPPORTS_PASSTHRU            = $23;
  GR_TEXTURE_ALIGN                = $24;
  GR_VIDEO_POSITION               = $25;
  GR_VIEWPORT                     = $26;
  GR_WDEPTH_MIN_MAX               = $27;
  GR_ZDEPTH_MIN_MAX               = $28;
  GR_VERTEX_PARAMETER             = $29;
  GR_BITS_GAMMA                   = $2a;

(*
** grGetString types
*)
  GR_EXTENSION                    = $a0;
  GR_HARDWARE                     = $a1;
  GR_RENDERER                     = $a2;
  GR_VENDOR                       = $a3;
  GR_VERSION                      = $a4;

(*
** -----------------------------------------------------------------------
** STRUCTURES
** -----------------------------------------------------------------------
*)
type
  TGrTexInfo = packed record
    smallLodLog2 : TGrLOD;
    largeLodLog2 : TGrLOD;
    aspectRatioLog2 : TGrAspectRatio;
    format : TGrTextureFormat;
    data : Pointer;
  end;

  TGrSstPerfStats = packed record
    pixelsIn : TFxU32;              // # pixels processed (minus buffer clears) */
    chromaFail : TFxU32;            // # pixels not drawn due to chroma key */
    zFuncFail : TFxU32;             // # pixels not drawn due to Z comparison */
    aFuncFail : TFxU32;             // # pixels not drawn due to alpha comparison */
    pixelsOut : TFxU32;             // # pixels drawn (including buffer clears) */
  end;

  PGrResolution = ^TGrResolution;
  TGrResolution = packed record
    resolution : TGrScreenResolution;
    refresh : TGrScreenRefresh;
    numColorBuffers : TFxI32;
    numAuxBuffers : TFxI32;
  end;

  TGlideResolution = TGrResolution;

const
  GR_QUERY_ANY = -1;

type
  TGrLfbSrcFmt = TFxU32;
const
  GR_LFB_SRC_FMT_565          = $00;
  GR_LFB_SRC_FMT_555          = $01;
  GR_LFB_SRC_FMT_1555         = $02;
  GR_LFB_SRC_FMT_888          = $04;
  GR_LFB_SRC_FMT_8888         = $05;
  GR_LFB_SRC_FMT_565_DEPTH    = $0c;
  GR_LFB_SRC_FMT_555_DEPTH    = $0d;
  GR_LFB_SRC_FMT_1555_DEPTH   = $0e;
  GR_LFB_SRC_FMT_ZA16         = $0f;
  GR_LFB_SRC_FMT_RLE16        = $80;

// H3D
  GR_HINT_H3DENABLE = 4;
  GR_HINTTYPE_MAX = 4;

(*
** -----------------------------------------------------------------------
** FUNCTION PROTOTYPES
** -----------------------------------------------------------------------
*)

(*
** rendering functions
*)
var
  grDrawPoint : procedure (const pt); stdcall;
  grDrawLine : procedure (const v1, v2); stdcall;
  grDrawTriangle : procedure (const a, b, c); stdcall;
  grVertexLayout : procedure (param: TFxU32; offset: TFxI32; mode: TFxU32); stdcall;
  grDrawVertexArray : procedure (mode, Count: TFxU32; var pointers: pointer); stdcall;
  grDrawVertexArrayContiguous : procedure(mode, Count: TFxU32; const vertex; stride: TFxU32); stdcall;

(*
**  Antialiasing Functions
*)
  grAADrawTriangle : procedure (const a,b,c; ab_antialias, bc_antialias, ca_antialias : TFxBool); stdcall;

(*
** buffer management
*)
  grBufferClear : procedure (Color: TGrColor; alpha: TGrAlpha; depth: TFxU32); stdcall;
  grBufferSwap : procedure (swap_interval: TFxI32); stdcall;
  grRenderBuffer : procedure (buffer: TGrBuffer); stdcall;

(*
** error management
*)
type
  TGrErrorCallbackFnc = procedure(str: PChar; fatal: TFxBool);
var
  grErrorSetCallback : procedure (fnc : TGrErrorCallbackFnc); stdcall;

(*
** SST routines
*)
  grFinish : procedure; stdcall;
  grFlush : procedure; stdcall;
  grSstWinOpen : function (hWnd : TFxU32;
      screen_resolution : TGrScreenResolution; refresh_rate : TGrScreenRefresh;
      color_format : TGrColorFormat; origin_location : TGrOriginLocation;
      nColBuffers, nAuxBuffers : TFxI32) : TGrContext; stdcall;
  grSstWinClose : function(context: TGrContext) : TFxBool; stdcall;
  grSelectContext : function(context: TGrContext) : TFxBool; stdcall;
  grSstOrigin : procedure (origin : TGrOriginLocation); stdcall;
  grSstSelect : procedure (which_sst : TFxI32); stdcall;

(*
** Glide configuration and special effect maintenance functions
*)
  grAlphaBlendFunction : procedure (rgb_sf, rgb_df, alpha_sf, alpha_df : TGrAlphaBlendFnc); stdcall;
  grAlphaCombine : procedure (Fnct : TGrCombineFunction; factor : TGrCombineFactor;
                              local : TGrCombineLocal; other : TGrCombineOther;
                              invert : TFxBool); stdcall;
  grAlphaControlsITRGBLighting : procedure (enable : TFxBool); stdcall;
  grAlphaTestFunction : procedure (fnct : TGrCmpFnc); stdcall;
  grAlphaTestReferenceValue : procedure (value : TGrAlpha); stdcall;
  grChromakeyMode : procedure (mode : TGrChromakeyMode); stdcall;
  grChromakeyValue : procedure (value : TGrColor); stdcall;
  grClipWindow : procedure (minx, miny, maxx, maxy : TFxU32); stdcall;
  grColorCombine : procedure (Fnct : TGrCombineFunction; factor : TGrCombineFactor;
                              local : TGrCombineLocal; other :TGrCombineOther;
                              invert : TFxBool); stdcall;
  grColorMask : procedure (rgb, a : TFxBool); stdcall;
  grCullMode : procedure (mode : TGrCullMode); stdcall;
  grConstantColorValue : procedure (value : TGrColor); stdcall;
  grDepthBiasLevel : procedure (level : TFxI32); stdcall;
  grDepthBufferFunction : procedure (fnct : TGrCmpFnc); stdcall;
  grDepthBufferMode : procedure (mode : TGrDepthBufferMode); stdcall;
  grDepthMask : procedure (mask : TFxBool); stdcall;
  grDisableAllEffects : procedure; stdcall;
  grDitherMode : procedure (mode : TGrDitherMode); stdcall;
  grFogColorValue : procedure (fogcolor : TGrColor); stdcall;
  grFogMode : procedure (mode : TGrFogMode); stdcall;
  grFogTable : procedure (ft : PGrFog); stdcall;

  grLoadGammaTable : procedure (nentries: TFxU32; var red,green,blue: TFxU32);
  grSplash : procedure (x, y, width, height : TFxFloat; frame : TFxU32); stdcall;
  grGet : function (pname, plength: TFxU32; var params) : TFxU32; stdcall;
  grGetString : function (pname: TFxU32) : PChar; stdcall;
  grQueryResolutions : function (var resTemplate: TGrResolution; output: PGrResolution) : TFxI32; stdcall;
  grReset : function (what: TFxU32) : TFxBool; stdcall;
  grGetProcAddress : function (procName: PChar) : TGrProc; stdcall;
  grEnable : procedure (mode: TGrEnableMode);  stdcall;
  grDisable : procedure (mode: TGrEnableMode); stdcall;
  grCoordinateSpace : procedure (mode: TGrCoordinateSpaceMode); stdcall;
  grDepthRange : procedure (n,f: TFxFloat); stdcall;
  grViewport : procedure (x,y,width,height: TFxI32); stdcall;

(*
** texture mapping control functions
*)
  grTexCalcMemRequired : function (lodmin, modmax : TGrLOD; aspect : TGrAspectRatio; fmt : TGrTextureFormat) : TFxU32; stdcall;
  grTexTextureMemRequired : function (evenOdd : TFxU32; var info : TGrTexInfo) : TFxU32; stdcall;
  grTexMinAddress : function (tmu: TGrChipID) : TFxU32; stdcall;
  grTexMaxAddress : function (tmu: TGrChipID) : TFxU32; stdcall;
  grTexNCCTable : procedure (table : TGrNCCTable); stdcall;
  grTexSource : procedure (tmu: TGrChipID; startAddess, evenOdd : TFxU32; var info : TGrTexInfo); stdcall;
  grTexClampMode : procedure (tmu: TGrChipID; s_clampmode, t_clampmode: TGrTextureClampMode); stdcall;
  grTexCombine : procedure (tmu: TGrChipID;
                            rgb_function : TGrCombineFunction;
                            rgb_factor : TGrCombineFactor;
                            alpha_function : TGrCombineFunction;
                            alpha_factor : TGrCombineFactor;
                            rgb_invert, alpha_invert : TFxBool); stdcall;
  grTexDetailControl : procedure (tmu: TGrChipID; lod_bias : Integer; detail_scale : TFxU8; detail_max : TFxFloat); stdcall;
  grTexFilterMode : procedure (tmu: TGrChipID; minfilter_mode,magfilter_mode : TGrTextureFilterMode); stdcall;
  grTexLodBiasValue : procedure (tmu: TGrChipID; bias : Single); stdcall;
  grTexDownloadMipMap : procedure (tmu: TGrChipID; startAddress, enenOdd : TFxU32; var info : TGrTexInfo); stdcall;
  grTexDownloadMipMapLevel : procedure (tmu: TGrChipID;
                                        startAddress: TFxU32;
                                        thisLod, largeLod: TGrLOD;
                                        aspectRatio: TGrAspectRatio;
                                        format: TGrTextureFormat;
                                        evenOdd: TFxU32;
                                        data: Pointer); stdcall;
  grTexDownloadMipMapLevelPartial : procedure (tmu: TGrChipID;
                                               startAdress: TFxU32;
                                               thisLod, largeLod : TGrLOD;
                                               aspectRatio : TGrAspectRatio;
                                               format: TGrTextureFormat;
                                               evenOdd: TFxU32;
                                               data: Pointer;
                                               _start, _end: TFxI32); stdcall;
  grTexDownloadTable : procedure (_type : TGrTexTable; data: Pointer); stdcall;
  grTexDownloadTablePartial : procedure (_type: TGrTexTable; data : Pointer;
                                         _start, _end: TFxI32); stdcall;
  grTexMipMapMode : procedure (tmu: TGrChipID; mode : TGrMipMapMode; lodBlend: TFxBool); stdcall;
  grTexMultibase : procedure (tmu: TGrChipID; enable: TFxBool); stdcall;
  grTexMultibaseAddress : procedure (tmu: TGrChipID; range: TGrTexBaseRange;
                                     startAddress, evenodd: TFxU32;
                                     var info : TGrTexInfo); stdcall;

(*
** linear frame buffer functions
*)
  grLfbLock : function (_type: TGrLock; buffer: TGrBuffer; writeMode: TGrLfbWriteMode;
                        origin: TGrOriginLocation; pixelPipeline: TFxBool;
                        var info: TGrLfbInfo) : TFxBool; stdcall;
  grLfbUnlock : function (_type: TGrLock; buffer: TGrBuffer) : TFxBool; stdcall;
  grLfbConstantAlpha : procedure (alpha: TGrAlpha); stdcall;
  grLfbConstantDepth : procedure (depth: TFxU16); stdcall;
  grLfbWriteColorSwizzle : procedure (swizzleBytes: TFxBool; swapWords: TFxBool); stdcall;
  grLfbWriteColorFormat : procedure (colorFormat : TGrColorFormat); stdcall;
  grLfbWriteRegion : function (dst_buffer: TGrBuffer;
                               dst_x, dst_y: TFxU32;
                               src_format: TGrLfbSrcFmt;
                               src_width, src_height: TFxU32;
                               pixelPipeline: TFxBool;
                               src_stride: TFxI32;
                               src_data: Pointer) : TFxBool; stdcall;
  grLfbReadRegion : function (src_buffer: TGrBuffer;
                              src_x, src_y: TFxU32;
                              src_width, src_height: TFxU32;
                              dst_stride: TFxU32;
                              dst_data: Pointer) : TFxBool; stdcall;

(*
** glide management functions
*)
  grGlideInit : procedure; stdcall;
  grGlideShutdown : procedure; stdcall;
  grGlideGetState : procedure (state: pointer); stdcall;
  grGlideSetState : procedure (state: pointer); stdcall;
  grGlideGetVertexLayout : procedure (layout: pointer); stdcall;
  grGlideSetVertexLayout : procedure (layout: pointer); stdcall;

////////////////////////////////////////////////////////////////////////////////
// GlideUtl.h
////////////////////////////////////////////////////////////////////////////////

(*
** Copyright (c) 1995, 3Dfx Interactive, Inc.
** All Rights Reserved.
**
** This is UNPUBLISHED PROPRIETARY SOURCE CODE of 3Dfx Interactive, Inc.;
** the contents of this file may not be disclosed to third parties, copied or
** duplicated in any form, in whole or in part, without the prior written
** permission of 3Dfx Interactive, Inc.
**
** RESTRICTED RIGHTS LEGEND:
** Use, duplication or disclosure by the Government is subject to restrictions
** as set forth in subdivision (c)(1)(ii) of the Rights in Technical Data
** and Computer Software clause at DFARS 252.227-7013, and/or in similar or
** successor clauses in the FAR, DOD or NASA FAR Supplement. Unpublished -
** rights reserved under the Copyright Laws of the United States.
**
** $Header: /devel/cvg/glide3/src/glideutl.h 4     7/24/98 1:41p Hohn $
** $Log: /devel/cvg/glide3/src/glideutl.h $
**
** 4     7/24/98 1:41p Hohn
** 
** 3     1/30/98 4:27p Atai
** gufog* prototype
** 
** 1     1/29/98 4:00p Atai
 * 
 * 1     1/16/98 4:29p Atai
 * create glide 3 src
 * 
 * 11    1/07/98 11:18a Atai
 * remove GrMipMapInfo and GrGC.mm_table in glide3
 * 
 * 10    1/06/98 6:47p Atai
 * undo grSplash and remove gu routines
 * 
 * 9     1/05/98 6:04p Atai
 * move 3df gu related data structure from glide.h to glideutl.h
 * 
 * 8     12/18/97 2:13p Peter
 * fogTable cataclysm
 * 
 * 7     12/15/97 5:52p Atai
 * disable obsolete glide2 api for glide3
 * 
 * 6     8/14/97 5:32p Pgj
 * remove dead code per GMT
 * 
 * 5     6/12/97 5:19p Pgj
 * Fix bug 578
 * 
 * 4     3/05/97 9:36p Jdt
 * Removed guFbWriteRegion added guEncodeRLE16
 * 
 * 3     1/16/97 3:45p Dow
 * Embedded fn protos in ifndef FX_GLIDE_NO_FUNC_PROTO 
*)

(* Glide Utility routines *)

(*
** 3DF texture file structs
*)

type
  TGu3dfHeader = packed record
    width, height : TFxU32;
    small_lod, large_lod : TFxI32;
    aspect_ratio : TGrAspectRatio;
    format : TGrTextureFormat;
  end;

  TGuNccTable = packed record
    yRGB : array [0..15] of TFxU8;
    iRGB : array [0..3, 0..2] of TFxI16;
    qRGB : array [0..3, 0..2] of TFxI16;
    packed_data : array [0..11] of TFxU32;
  end;

  TGuTexPalette = packed record
    data : array [0..255] of TFxU32;
  end;

  TGuTexTable = packed record
    case byte of
      0: (nccTable : TGuNccTable);
      1: (palette  : TGuTexPalette);
  end;

  TGu3dfInfo = packed record
    header : TGu3dfHeader;
    table : TGuTexTable;
    data : Pointer;
    mem_required : TFxU32;    (* memory required for mip map in bytes. *)
  end;

(*
** Gamma functions
*)
var
  guGammaCorrectionRGB : procedure (red,green,blue: TFxFloat); stdcall;

(*
** fog stuff
*)
  guFogTableIndexToW : function (i: TFxI32) : Single; stdcall;
  guFogGenerateExp : procedure (fogtable: PGrFog; density: TFxFloat); stdcall;
  guFogGenerateExp2 : procedure (fogtable: PGrFog; density: TFxFloat); stdcall;
  guFogGenerateLinear : procedure (fogtable: PGrFog; nearZ, farZ: TFxFloat); stdcall;

(*
** hi-level texture manipulation tools.
*)
  gu3dfGetInfo : function (filename: PChar; var info: TGu3dfInfo): TFxBool; stdcall;
  gu3dfLoad : function (filename: PChar; var data: TGu3dfInfo): TFxBool; stdcall;


// Unit managment:

const
  GlideHandle : THandle = 0;

function InitGlide : Boolean;
function InitGlideFromLibrary(PathDLL: string): Boolean;
procedure CloseGlide;
function CheckProcAddresses : integer;

// procedure ClearExtensions;
// procedure ReadExtensions;

implementation

procedure ClearProcAddresses;
begin
(*
** rendering functions
*)
    grDrawPoint := nil;
    grDrawLine := nil;
    grDrawTriangle := nil;
    grVertexLayout := nil;
    grDrawVertexArray := nil;
    grDrawVertexArrayContiguous := nil;

(*
**  Antialiasing Functions
*)
    grAADrawTriangle := nil;

(*
** buffer management
*)
    grBufferClear := nil;
    grBufferSwap := nil;
    grRenderBuffer := nil;

(*
** error management
*)
    grErrorSetCallback := nil;

(*
** SST routines
*)
    grFinish := nil;
    grFlush := nil;
    grSstWinOpen := nil; 
    grSstWinClose := nil;
    grSelectContext := nil;
    grSstOrigin := nil;
    grSstSelect := nil;

(*
** Glide configuration and special effect maintenance functions
*)
    grAlphaBlendFunction := nil;
    grAlphaCombine := nil;
    grAlphaControlsITRGBLighting := nil;
    grAlphaTestFunction := nil;
    grAlphaTestReferenceValue := nil;
    grChromakeyMode := nil;
    grChromakeyValue := nil;
    grClipWindow := nil;
    grColorCombine := nil;
    grColorMask := nil; 
    grCullMode := nil;
    grConstantColorValue := nil; 
    grDepthBiasLevel := nil;
    grDepthBufferFunction := nil;
    grDepthBufferMode := nil;
    grDepthMask := nil;
    grDisableAllEffects := nil;
    grDitherMode := nil;
    grFogColorValue := nil;
    grFogMode := nil;
    grFogTable := nil;
    grLoadGammaTable := nil;
    grSplash := nil;
    grGet := nil;
    grGetString := nil;
    grQueryResolutions := nil;
    grReset := nil;
    grGetProcAddress := nil;
    grEnable := nil;
    grDisable := nil;
    grCoordinateSpace := nil;
    grDepthRange := nil;
    grViewport := nil;

(*
** texture mapping control functions
*)
    grTexCalcMemRequired := nil;
    grTexTextureMemRequired := nil;
    grTexMinAddress := nil;
    grTexMaxAddress := nil;
    grTexNCCTable := nil;
    grTexSource := nil;
    grTexClampMode := nil;
    grTexCombine := nil;
    grTexDetailControl := nil; 
    grTexFilterMode := nil;
    grTexLodBiasValue := nil;
    grTexDownloadMipMap := nil;
    grTexDownloadMipMapLevel := nil;
    grTexDownloadMipMapLevelPartial := nil;
    grTexDownloadTable := nil;
    grTexDownloadTablePartial := nil;
    grTexMipMapMode := nil;
    grTexMultibase := nil;
    grTexMultibaseAddress := nil;

(*
** linear frame buffer functions
*)
    grLfbLock := nil;
    grLfbUnLock := nil;
    grLfbConstantAlpha := nil;
    grLfbConstantDepth := nil;
    grLfbWriteColorSwizzle := nil;
    grLfbWriteColorFormat := nil;
    grLfbWriteRegion := nil;
    grLfbReadRegion := nil;

(*
** glide management functions
*)
    grGlideInit := nil;
    grGlideShutdown := nil;
    grGlideGetState := nil;
    grGlideSetState := nil;
    grGlideGetVertexLayout := nil;
    grGlideSetVertexLayout := nil;

(*
** Gamma functions
*)
    guGammaCorrectionRGB := nil;

(*
** fog stuff
*)
    guFogTableIndexToW := nil;
    guFogGenerateExp := nil;
    guFogGenerateExp2 := nil;
    guFogGenerateLinear := nil;

(*
** hi-level texture manipulation tools.
*)
    gu3dfGetInfo := nil;
    gu3dfLoad := nil;
end;

function CheckProcAddresses : integer;
begin
  Result := 0;
(*
** rendering functions
*)
  if @grDrawPoint = nil then Inc(Result);
  if @grDrawLine = nil then Inc(Result);
  if @grDrawTriangle = nil then Inc(Result);
  if @grVertexLayout = nil then Inc(Result);
  if @grDrawVertexArray = nil then Inc(Result);
  if @grDrawVertexArrayContiguous = nil then Inc(Result);

(*
**  Antialiasing Functions
*)
  if @grAADrawTriangle = nil then Inc(Result);

(*
** buffer management
*)
  if @grBufferClear = nil then Inc(Result);
  if @grBufferSwap = nil then Inc(Result);
  if @grRenderBuffer = nil then Inc(Result);

(*
** error management
*)
  if @grErrorSetCallback = nil then Inc(Result);

(*
** SST routines
*)
  if @grFinish = nil then Inc(Result);
  if @grFlush = nil then Inc(Result);
  if @grSstWinOpen = nil then Inc(Result);
  if @grSstWinClose = nil then Inc(Result);
  if @grSelectContext = nil then Inc(Result);
  if @grSstOrigin = nil then Inc(Result);
  if @grSstSelect = nil then Inc(Result);

(*
** Glide configuration and special effect maintenance functions
*)
  if @grAlphaBlendFunction = nil then Inc(Result);
  if @grAlphaCombine = nil then Inc(Result);
  if @grAlphaControlsITRGBLighting = nil then Inc(Result);
  if @grAlphaTestFunction = nil then Inc(Result);
  if @grAlphaTestReferenceValue = nil then Inc(Result);
  if @grChromakeyMode = nil then Inc(Result);
  if @grChromakeyValue = nil then Inc(Result);
  if @grClipWindow = nil then Inc(Result);
  if @grColorCombine = nil then Inc(Result);
  if @grColorMask = nil then Inc(Result);
  if @grCullMode = nil then Inc(Result);
  if @grConstantColorValue = nil then Inc(Result);
  if @grDepthBiasLevel = nil then Inc(Result);
  if @grDepthBufferFunction = nil then Inc(Result);
  if @grDepthBufferMode = nil then Inc(Result);
  if @grDepthMask = nil then Inc(Result);
  if @grDisableAllEffects = nil then Inc(Result);
  if @grDitherMode = nil then Inc(Result);
  if @grFogColorValue = nil then Inc(Result);
  if @grFogMode = nil then Inc(Result);
  if @grFogTable = nil then Inc(Result);
  if @grLoadGammaTable = nil then Inc(Result);
  if @grSplash = nil then Inc(Result);
  if @grGet = nil then Inc(Result);
  if @grGetString = nil then Inc(Result);
  if @grQueryResolutions = nil then Inc(Result);
  if @grReset = nil then Inc(Result);
  if @grGetProcAddress = nil then Inc(Result);
  if @grEnable = nil then Inc(Result);
  if @grDisable = nil then Inc(Result);
  if @grCoordinateSpace = nil then Inc(Result);
  if @grDepthRange = nil then Inc(Result);
  if @grViewport = nil then Inc(Result);

(*
** texture mapping control functions
*)
  if @grTexCalcMemRequired = nil then Inc(Result);
  if @grTexTextureMemRequired = nil then Inc(Result);
  if @grTexMinAddress = nil then Inc(Result);
  if @grTexMaxAddress = nil then Inc(Result);
  if @grTexNCCTable = nil then Inc(Result);
  if @grTexSource = nil then Inc(Result);
  if @grTexClampMode = nil then Inc(Result);
  if @grTexCombine = nil then Inc(Result);
  if @grTexDetailControl = nil then Inc(Result);
  if @grTexFilterMode = nil then Inc(Result);
  if @grTexLodBiasValue = nil then Inc(Result);
  if @grTexDownloadMipMap = nil then Inc(Result);
  if @grTexDownloadMipMapLevel = nil then Inc(Result);
  if @grTexDownloadMipMapLevelPartial = nil then Inc(Result);
  if @grTexDownloadTable = nil then Inc(Result);
  if @grTexDownloadTablePartial = nil then Inc(Result);
  if @grTexMipMapMode = nil then Inc(Result);
  if @grTexMultibase = nil then Inc(Result);
  if @grTexMultibaseAddress = nil then Inc(Result);

(*
** linear frame buffer functions
*)
  if @grLfbLock = nil then Inc(Result);
  if @grLfbUnLock = nil then Inc(Result);
  if @grLfbConstantAlpha = nil then Inc(Result);
  if @grLfbConstantDepth = nil then Inc(Result);
  if @grLfbWriteColorSwizzle = nil then Inc(Result);
  if @grLfbWriteColorFormat = nil then Inc(Result);
  if @grLfbWriteRegion = nil then Inc(Result);
  if @grLfbReadRegion = nil then Inc(Result);

(*
** glide management functions
*)
  if @grGlideInit = nil then Inc(Result);
  if @grGlideShutdown = nil then Inc(Result);
  if @grGlideGetState = nil then Inc(Result);
  if @grGlideSetState = nil then Inc(Result);
  if @grGlideGetVertexLayout = nil then Inc(Result);
  if @grGlideSetVertexLayout = nil then Inc(Result);

(*
** Gamma functions
*)
  if @guGammaCorrectionRGB = nil then Inc(Result);

(*
** fog stuff
*)
  if @guFogTableIndexToW = nil then Inc(Result);
  if @guFogGenerateExp = nil then Inc(Result);
  if @guFogGenerateExp2 = nil then Inc(Result);
  if @guFogGenerateLinear = nil then Inc(Result);

(*
** hi-level texture manipulation tools.
*)
  if @gu3dfGetInfo = nil then Inc(Result);
  if @gu3dfLoad = nil then Inc(Result);
end;

procedure LoadProcAddresses;
begin
  if GlideHandle <> 0 then
  begin
(*
** rendering functions
*)
    grDrawPoint := GetProcAddress (GlideHandle, '_grDrawPoint@4');
    grDrawLine := GetProcAddress (GlideHandle, '_grDrawLine@8');
    grDrawTriangle := GetProcAddress (GlideHandle, '_grDrawTriangle@12');
    grVertexLayout := GetProcAddress (GlideHandle, '_grVertexLayout@12');
    grDrawVertexArray := GetProcAddress (GlideHandle, '_grDrawVertexArray@12');
    grDrawVertexArrayContiguous := GetProcAddress (GlideHandle, '_grDrawVertexArrayContiguous@16');

(*
**  Antialiasing Functions
*)
    grAADrawTriangle := GetProcAddress (GlideHandle,  '_grAADrawTriangle@24');

(*
** buffer management
*)
    grBufferClear := GetProcAddress (GlideHandle, '_grBufferClear@12');
    grBufferSwap := GetProcAddress (GlideHandle, '_grBufferSwap@4');
    grRenderBuffer := GetProcAddress (GlideHandle, '_grRenderBuffer@4');

(*
** error management
*)
    grErrorSetCallback := GetProcAddress (GlideHandle, '_grErrorSetCallback@4');

(*
** SST routines
*)
    grFinish := GetProcAddress (GlideHandle, '_grFinish@0');
    grFlush := GetProcAddress (GlideHandle, '_grFlush@0');
    grSstWinOpen := GetProcAddress (GlideHandle, '_grSstWinOpen@28');
    grSstWinClose := GetProcAddress (GlideHandle, '_grSstWinClose@4');
    grSelectContext := GetProcAddress (GlideHandle, '_grSelectContext@4');
    grSstOrigin := GetProcAddress (GlideHandle, '_grSstOrigin@4');
    grSstSelect := GetProcAddress (GlideHandle, '_grSstSelect@4');

(*
** Glide configuration and special effect maintenance functions
*)
    grAlphaBlendFunction := GetProcAddress (GlideHandle, '_grAlphaBlendFunction@16');
    grAlphaCombine := GetProcAddress (GlideHandle, '_grAlphaCombine@20');
    grAlphaControlsITRGBLighting := GetProcAddress (GlideHandle, '_grAlphaControlsITRGBLighting@4');
    grAlphaTestFunction := GetProcAddress (GlideHandle, '_grAlphaTestFunction@4');
    grAlphaTestReferenceValue := GetProcAddress (GlideHandle, '_grAlphaTestReferenceValue@4');
    grChromakeyMode := GetProcAddress (GlideHandle, '_grChromakeyMode@4');
    grChromakeyValue := GetProcAddress (GlideHandle, '_grChromakeyValue@4');
    grClipWindow := GetProcAddress (GlideHandle, '_grClipWindow@16');
    grColorCombine := GetProcAddress (GlideHandle, '_grColorCombine@20');
    grColorMask := GetProcAddress (GlideHandle, '_grColorMask@8');
    grCullMode := GetProcAddress (GlideHandle, '_grCullMode@4');
    grConstantColorValue := GetProcAddress (GlideHandle, '_grConstantColorValue@4');
    grDepthBiasLevel := GetProcAddress (GlideHandle, '_grDepthBiasLevel@4');
    grDepthBufferFunction := GetProcAddress (GlideHandle, '_grDepthBufferFunction@4');
    grDepthBufferMode := GetProcAddress (GlideHandle, '_grDepthBufferMode@4');
    grDepthMask := GetProcAddress (GlideHandle, '_grDepthMask@4');
    grDisableAllEffects := GetProcAddress (GlideHandle, '_grDisableAllEffects@0');
    grDitherMode := GetProcAddress (GlideHandle, '_grDitherMode@4');
    grFogColorValue := GetProcAddress (GlideHandle, '_grFogColorValue@4');
    grFogMode := GetProcAddress (GlideHandle, '_grFogMode@4');
    grFogTable := GetProcAddress (GlideHandle, '_grFogTable@4');
    grLoadGammaTable := GetProcAddress (GlideHandle, '_grLoadGammaTable@16');
    grSplash := GetProcAddress (GlideHandle, '_grSplash@20');
    grGet := GetProcAddress (GlideHandle, '_grGet@12');
    grGetString := GetProcAddress (GlideHandle, '_grGetString@4');
    grQueryResolutions := GetProcAddress (GlideHandle, '_grQueryResolutions@8');
    grReset := GetProcAddress (GlideHandle, '_grReset@4');
    grGetProcAddress := GetProcAddress (GlideHandle, '_grGetProcAddress@4');
    grEnable := GetProcAddress (GlideHandle, '_grEnable@4');
    grDisable := GetProcAddress (GlideHandle, '_grDisable@4');
    grCoordinateSpace := GetProcAddress (GlideHandle, '_grCoordinateSpace@4');
    grDepthRange := GetProcAddress (GlideHandle, '_grDepthRange@8');
    grViewport := GetProcAddress (GlideHandle, '_grViewport@16');

(*
** texture mapping control functions
*)
    grTexCalcMemRequired := GetProcAddress (GlideHandle, '_grTexCalcMemRequired@16');
    grTexTextureMemRequired := GetProcAddress (GlideHandle, '_grTexTextureMemRequired@8');
    grTexMinAddress := GetProcAddress (GlideHandle, '_grTexMinAddress@4');
    grTexMaxAddress := GetProcAddress (GlideHandle, '_grTexMaxAddress@4');
    grTexNCCTable := GetProcAddress (GlideHandle, '_grTexNCCTable@4');
    grTexSource := GetProcAddress (GlideHandle, '_grTexSource@16');
    grTexClampMode := GetProcAddress (GlideHandle, '_grTexClampMode@12');
    grTexCombine := GetProcAddress (GlideHandle, '_grTexCombine@28');
    grTexDetailControl := GetProcAddress (GlideHandle, '_grTexDetailControl@16');
    grTexFilterMode := GetProcAddress (GlideHandle, '_grTexFilterMode@12');
    grTexLodBiasValue := GetProcAddress (GlideHandle, '_grTexLodBiasValue@8');
    grTexDownloadMipMap := GetProcAddress (GlideHandle, '_grTexDownloadMipMap@16');
    grTexDownloadMipMapLevel := GetProcAddress (GlideHandle, '_grTexDownloadMipMapLevel@32');
    grTexDownloadMipMapLevelPartial := GetProcAddress (GlideHandle, '_grTexDownloadMipMapLevelPartial@40');
    grTexDownloadTable := GetProcAddress (GlideHandle, '_grTexDownloadTable@8');
    grTexDownloadTablePartial := GetProcAddress (GlideHandle, '_grTexDownloadTablePartial@16');
    grTexMipMapMode := GetProcAddress (GlideHandle, '_grTexMipMapMode@12');
    grTexMultibase := GetProcAddress (GlideHandle, '_grTexMultibase@8');
    grTexMultibaseAddress := GetProcAddress (GlideHandle, '_grTexMultibaseAddress@20');

(*
** linear frame buffer functions
*)
    grLfbLock := GetProcAddress (GlideHandle,  '_grLfbLock@24');
    grLfbUnLock := GetProcAddress (GlideHandle,  '_grLfbUnlock@8');
    grLfbConstantAlpha := GetProcAddress (GlideHandle,  '_grLfbConstantAlpha@4');
    grLfbConstantDepth := GetProcAddress (GlideHandle,  '_grLfbConstantDepth@4');
    grLfbWriteColorSwizzle := GetProcAddress (GlideHandle,  '_grLfbWriteColorSwizzle@8');
    grLfbWriteColorFormat := GetProcAddress (GlideHandle,  '_grLfbWriteColorFormat@4');
    grLfbWriteRegion := GetProcAddress (GlideHandle,  '_grLfbWriteRegion@36');
    grLfbReadRegion := GetProcAddress (GlideHandle,  '_grLfbReadRegion@28');

(*
** glide management functions
*)
    grGlideInit := GetProcAddress (GlideHandle,  '_grGlideInit@0');
    grGlideShutdown := GetProcAddress (GlideHandle,  '_grGlideShutdown@0');
    grGlideGetState := GetProcAddress (GlideHandle,  '_grGlideGetState@4');
    grGlideSetState := GetProcAddress (GlideHandle,  '_grGlideSetState@4');
    grGlideGetVertexLayout := GetProcAddress (GlideHandle,  '_grGlideGetVertexLayout@4');
    grGlideSetVertexLayout := GetProcAddress (GlideHandle,  '_grGlideSetVertexLayout@4');

(*
** Gamma functions
*)
    guGammaCorrectionRGB := GetProcAddress (GlideHandle,  '_guGammaCorrectionRGB@12');

(*
** fog stuff
*)
    guFogTableIndexToW := GetProcAddress (GlideHandle,  '_guFogTableIndexToW@4');
    guFogGenerateExp := GetProcAddress (GlideHandle,  '_guFogGenerateExp@8');
    guFogGenerateExp2 := GetProcAddress (GlideHandle,  '_guFogGenerateExp2@8');
    guFogGenerateLinear := GetProcAddress (GlideHandle,  '_guFogGenerateLinear@12');

(*
** hi-level texture manipulation tools.
*)
    gu3dfGetInfo := GetProcAddress (GlideHandle,  '_gu3dfGetInfo@8');
    gu3dfLoad := GetProcAddress (GlideHandle,  '_gu3dfLoad@8');
  end;
end;

const
  GlideName = 'Glide3x.dll';
  
function InitGlideFromLibrary(PathDLL: string): Boolean;
begin
  Result := False;
  if PathDLL = '' then PathDLL := GlideName;
  if PathDLL[Length(PathDLL)-1] = '\' then PathDLL := PathDLL + GlideName;
  if GlideHandle <> 0 then CloseGlide;
  GlideHandle := LoadLibrary(PChar(PathDLL));
  if (GlideHandle = 0) then exit;
  LoadProcAddresses;
  Result := CheckProcAddresses = 0;
end;

function InitGlide : Boolean;
begin
  Result := InitGlideFromLibrary('');
end;

procedure CloseGlide;
begin
  ClearProcAddresses;
  if GlideHandle <> 0 then
  begin
    FreeLibrary(GlideHandle);
    GlideHandle := 0;
  end;
end;

function GR_GENERATE_FIFOCHECK_HINT_MASK(swHWM, swLWM: variant) : variant;
begin
  Result := ((swHWM and $ffff) shl 16) or (swLWM and $ffff)
end;

function FXBIT(i: variant) : variant;
begin
  Result := 1 shl i;
end;

function FXUNUSED(a: variant) : boolean;
begin
  Result := (a = 0);
end;

initialization
begin
  ClearProcAddresses;
  InitGlide;
end

finalization
begin
  CloseGlide;
end

end.
