unit RCP;

(**************************************************************************;
 *                                                                        *
 *               Copyright (C) 1995, Silicon Graphics, Inc.               *
 *                                                                        *
 *  These coded instructions, statements, and computer programs  contain  *
 *  unpublished  proprietary  information of Silicon Graphics, Inc., and  *
 *  are protected by Federal copyright law.  They  may  not be disclosed  *
 *  to  third  parties  or copied or duplicated in any form, in whole or  *
 *  in part, without the prior written consent of Silicon Graphics, Inc.  *
 *                                                                        *
 **************************************************************************)


(*************************************************************************
 *
 *  File: rcp.h
 *
 *  This file contains register and bit definitions for RCP memory map.
 *
 *  $Header: /mdev2/PR/include/RCS/rcp.h,v 1.16 1996/08/05 21:06:18 tsai Exp $
 *
 *)


(**********************************************************************
 *
 * Here is a quick overview of the RCP memory map:
 *

$0000_0000 .. $03ef_ffff	RDRAM memory
$03f0_0000 .. $03ff_ffff	RDRAM registers

				RCP registers (see below)
$0400_0000 .. $040f_ffff	SP registers
$0410_0000 .. $041f_ffff	DP command registers
$0420_0000 .. $042f_ffff	DP span registers
$0430_0000 .. $043f_ffff	MI registers
$0440_0000 .. $044f_ffff	VI registers
$0450_0000 .. $045f_ffff	AI registers
$0460_0000 .. $046f_ffff	PI registers
$0470_0000 .. $047f_ffff	RI registers
$0480_0000 .. $048f_ffff	SI registers
$0490_0000 .. $04ff_ffff	unused

$0500_0000 .. $05ff_ffff	cartridge domain 2
$0600_0000 .. $07ff_ffff	cartridge domain 1
$0800_0000 .. $0fff_ffff	cartridge domain 2
$1000_0000 .. $1fbf_ffff	cartridge domain 1

$1fc0_0000 .. $1fc0_07bf      PIF Boot Rom (1984 bytes)
$1fc0_07c0 .. $1fc0_07ff      PIF (JoyChannel) RAM (64 bytes)
$1fc0_0800 .. $1fcf_ffff      Reserved
$1fd0_0000 .. $7fff_ffff      cartridge domain 1
$8000_0000 .. $ffff_ffff      external SysAD device

The Indy development board use cartridge domain 1:
$1000_0000 .. $10ff_ffff	RAMROM
$1800_0000 .. $1800_0003	GIO interrupt (6 bits valid in 4 bytes)	
$1800_0400 .. $1800_0403	GIO sync (6 bits valid in 4 bytes)	
$1800_0800 .. $1800_0803	CART interrupt (6 bits valid in 4 bytes)	



**************************************************************************)

interface


(*************************************************************************
 * RDRAM Memory (Assumes that maximum size is 4 MB)
 *)
CONST RDRAM_0_START=		$00000000;
CONST RDRAM_0_END=		$001FFFFF;
CONST RDRAM_1_START=		$00200000;
CONST RDRAM_1_END=		$003FFFFF;

CONST RDRAM_START=		RDRAM_0_START;
CONST RDRAM_END=		RDRAM_1_END;


(*************************************************************************
 * RDRAM Registers ($03f0_0000 .. $03ff_ffff)
 *)
CONST RDRAM_BASE_REG=		$03F00000;

CONST RDRAM_CONFIG_REG=         (RDRAM_BASE_REG+$00);
CONST RDRAM_DEVICE_TYPE_REG=    (RDRAM_BASE_REG+$00);
CONST RDRAM_DEVICE_ID_REG=      (RDRAM_BASE_REG+$04);
CONST RDRAM_DELAY_REG=          (RDRAM_BASE_REG+$08);
CONST RDRAM_MODE_REG=           (RDRAM_BASE_REG+$0c);
CONST RDRAM_REF_INTERVAL_REG=   (RDRAM_BASE_REG+$10);
CONST RDRAM_REF_ROW_REG=        (RDRAM_BASE_REG+$14);
CONST RDRAM_RAS_INTERVAL_REG=   (RDRAM_BASE_REG+$18);
CONST RDRAM_MIN_INTERVAL_REG=   (RDRAM_BASE_REG+$1c);
CONST RDRAM_ADDR_SELECT_REG=    (RDRAM_BASE_REG+$20);
CONST RDRAM_DEVICE_MANUF_REG=   (RDRAM_BASE_REG+$24);

CONST RDRAM_0_DEVICE_ID=       0;
CONST RDRAM_1_DEVICE_ID=       1;

CONST RDRAM_RESET_MODE=        0;
CONST RDRAM_ACTIVE_MODE=       1;
CONST RDRAM_STANDBY_MODE=      2;

CONST RDRAM_LENGTH=		(2*512*2048);
CONST RDRAM_0_BASE_ADDRESS=	(RDRAM_0_DEVICE_ID*RDRAM_LENGTH);
CONST RDRAM_1_BASE_ADDRESS=	(RDRAM_1_DEVICE_ID*RDRAM_LENGTH);

CONST RDRAM_0_CONFIG=		$00000;
CONST RDRAM_1_CONFIG=		$00400;
CONST RDRAM_GLOBAL_CONFIG=	$80000;


(*************************************************************************
 * PIF Physical memory map (total size = 2 KB)
 *
 *			  Size	    Description     Mode
 *	1FC007FF	+-------+-----------------+-----+
 *			 OR   64 B  OR  JoyChannel RAM   OR  R/W  OR
 *	1FC007C0	+-------+-----------------+-----+
 *			 OR 1984 B  OR     Boot ROM      OR   *   OR   * = Reserved
 *	1FC00000	+-------+-----------------+-----+
 *
 *)
CONST PIF_ROM_START=		$1FC00000;
CONST PIF_ROM_END=		$1FC007BF;
CONST PIF_RAM_START=		$1FC007C0;
CONST PIF_RAM_END=		$1FC007FF;


(*************************************************************************
 * Controller channel 
 * Each game controller channel has 4 error bits that are defined in bit 6-7 of
 * the Rx and Tx data size area bytes. Programmers need to clear these bits
 * when setting the Tx/Rx size area values for a channel
 *)
CONST CHNL_ERR_NORESP=		$80;	(* Bit 7 (Rx): No response error *)
CONST CHNL_ERR_OVERRUN=	$40;	(* Bit 6 (Rx): Overrun error *)
CONST CHNL_ERR_FRAME=		$80;	(* Bit 7 (Tx): Frame error *)
CONST CHNL_ERR_COLLISION=	$40;	(* Bit 6 (Tx): Collision error *)

CONST CHNL_ERR_MASK=		$C0;	(* Bit 6-7: channel errors *)


(*************************************************************************
 * External device info
 *)
CONST DEVICE_TYPE_CART=	0;	(* ROM cartridge *)
CONST DEVICE_TYPE_BULK=	1;	(* ROM bulk *)
CONST DEVICE_TYPE_64DD=	2;	(* 64 Disk Drive *)


(*************************************************************************
 * SP Memory
 *)
CONST SP_DMEM_START=		$04000000;	(* read/write *)
CONST SP_DMEM_END=		$04000FFF;
CONST SP_IMEM_START=		$04001000;	(* read/write *)
CONST SP_IMEM_END=		$04001FFF;

(*************************************************************************
 * SP CP0 Registers
 *)

CONST SP_BASE_REG=		$04040000;

(* SP memory address (R/W): [11:0] DMEM/IMEM address; [12] 0=DMEM,1=IMEM *)
CONST SP_MEM_ADDR_REG=		(SP_BASE_REG+$00);	(* Master *)
						
(* SP DRAM DMA address (R/W): [23:0] RDRAM address *)
CONST SP_DRAM_ADDR_REG=	(SP_BASE_REG+$04);	(* Slave *)

(* SP read DMA length (R/W): [11:0] length, [19:12] count, [31:20] skip *)
(* direction: I/DMEM <- RDRAM *)
CONST SP_RD_LEN_REG=		(SP_BASE_REG+$08);	(* R/W: read len *)

(* SP write DMA length (R/W): [11:0] length, [19:12] count, [31:20] skip *)
(* direction: I/DMEM -> RDRAM *)
CONST SP_WR_LEN_REG=		(SP_BASE_REG+$0C);	(* R/W: write len *)

(* SP status (R/W): [14:0] valid bits; see below for write/read mode *)
CONST SP_STATUS_REG=		(SP_BASE_REG+$10);

(* SP DMA full (R): [0] valid bit; dma full *)
CONST SP_DMA_FULL_REG=		(SP_BASE_REG+$14);

(* SP DMA busy (R): [0] valid bit; dma busy *)
CONST SP_DMA_BUSY_REG=		(SP_BASE_REG+$18);

(* SP semaphore (R/W): Read:  [0] semaphore flag (set on read) *)
(*                     Write: [] clear semaphore flag *)
CONST SP_SEMAPHORE_REG=	(SP_BASE_REG+$1C);

(* SP PC (R/W): [11:0] program counter *)
CONST SP_PC_REG=		$04080000;

(* SP MEM address: bit 12 specifies if address is IMEM or DMEM *)
CONST SP_DMA_DMEM=		$0000;		(* Bit 12: 0=DMEM, 1=IMEM *)
CONST SP_DMA_IMEM=		$1000;		(* Bit 12: 0=DMEM, 1=IMEM *)

(*
 * Values to clear/set bit in status reg (SP_STATUS_REG - write)
 *)
CONST SP_CLR_HALT=		$00001;	    (* Bit  0: clear halt *)
CONST SP_SET_HALT=		$00002;	    (* Bit  1: set halt *)
CONST SP_CLR_BROKE=		$00004;	    (* Bit  2: clear broke *)
CONST SP_CLR_INTR=		$00008;	    (* Bit  3: clear intr *)
CONST SP_SET_INTR=		$00010;	    (* Bit  4: set intr *)
CONST SP_CLR_SSTEP=		$00020;	    (* Bit  5: clear sstep *)
CONST SP_SET_SSTEP=		$00040;	    (* Bit  6: set sstep *)
CONST SP_CLR_INTR_BREAK=	$00080;	    (* Bit  7: clear intr on break *)
CONST SP_SET_INTR_BREAK=	$00100;	    (* Bit  8: set intr on break *)
CONST SP_CLR_SIG0=		$00200;	    (* Bit  9: clear signal 0 *)
CONST SP_SET_SIG0=		$00400;	    (* Bit 10: set signal 0 *)
CONST SP_CLR_SIG1=		$00800;	    (* Bit 11: clear signal 1 *)
CONST SP_SET_SIG1=		$01000;	    (* Bit 12: set signal 1 *)
CONST SP_CLR_SIG2=		$02000;	    (* Bit 13: clear signal 2 *)
CONST SP_SET_SIG2=		$04000;	    (* Bit 14: set signal 2 *)
CONST SP_CLR_SIG3=		$08000;	    (* Bit 15: clear signal 3 *)
CONST SP_SET_SIG3=		$10000;	    (* Bit 16: set signal 3 *)
CONST SP_CLR_SIG4=		$20000;	    (* Bit 17: clear signal 4 *)
CONST SP_SET_SIG4=		$40000;	    (* Bit 18: set signal 4 *)
CONST SP_CLR_SIG5=		$80000;	    (* Bit 19: clear signal 5 *)
CONST SP_SET_SIG5=	       $100000;	    (* Bit 20: set signal 5 *)
CONST SP_CLR_SIG6=	       $200000;	    (* Bit 21: clear signal 6 *)
CONST SP_SET_SIG6=	       $400000;	    (* Bit 22: set signal 6 *)
CONST SP_CLR_SIG7=	       $800000;	    (* Bit 23: clear signal 7 *)
CONST SP_SET_SIG7=	      $1000000;	    (* Bit 24: set signal 7 *)

(*
 * Patterns to interpret status reg (SP_STATUS_REG - read)
 *)
CONST SP_STATUS_HALT=		$001;		(* Bit  0: halt *)
CONST SP_STATUS_BROKE=		$002;		(* Bit  1: broke *)
CONST SP_STATUS_DMA_BUSY=	$004;		(* Bit  2: dma busy *)
CONST SP_STATUS_DMA_FULL=	$008;		(* Bit  3: dma full *)
CONST SP_STATUS_IO_FULL=	$010;		(* Bit  4: io full *)
CONST SP_STATUS_SSTEP=		$020;		(* Bit  5: single step *)
CONST SP_STATUS_INTR_BREAK=	$040;		(* Bit  6: interrupt on break *)
CONST SP_STATUS_SIG0=		$080;		(* Bit  7: signal 0 set *)
CONST SP_STATUS_SIG1=		$100;		(* Bit  8: signal 1 set *)
CONST SP_STATUS_SIG2=		$200;		(* Bit  9: signal 2 set *)
CONST SP_STATUS_SIG3=		$400;		(* Bit 10: signal 3 set *)
CONST SP_STATUS_SIG4=		$800;		(* Bit 11: signal 4 set *)
CONST SP_STATUS_SIG5=	       $1000;		(* Bit 12: signal 5 set *)
CONST SP_STATUS_SIG6=	       $2000;		(* Bit 13: signal 6 set *)
CONST SP_STATUS_SIG7=	       $4000;		(* Bit 14: signal 7 set *)

(* 
 * Use of SIG bits
 *)
CONST SP_CLR_YIELD=		SP_CLR_SIG0;
CONST SP_SET_YIELD=		SP_SET_SIG0;
CONST SP_STATUS_YIELD=		SP_STATUS_SIG0;
CONST SP_CLR_YIELDED=		SP_CLR_SIG1;
CONST SP_SET_YIELDED=		SP_SET_SIG1;
CONST SP_STATUS_YIELDED=	SP_STATUS_SIG1;
CONST SP_CLR_TASKDONE=		SP_CLR_SIG2;
CONST SP_SET_TASKDONE=		SP_SET_SIG2;
CONST SP_STATUS_TASKDONE=	SP_STATUS_SIG2;


(* SP IMEM BIST REG (R/W): [6:0] BIST status bits; see below for detail *)
CONST SP_IBIST_REG=	$04080004;

(*
 * Patterns to interpret status reg (SP_BIST_REG - write)
 *)
CONST SP_IBIST_CHECK=		$01;		(* Bit 0: BIST check *)
CONST SP_IBIST_GO=		$02;		(* Bit 1: BIST go *)
CONST SP_IBIST_CLEAR=		$04;		(* Bit 2: BIST clear *)

(*
 * Patterns to interpret status reg (SP_BIST_REG - read)
 *)
(* First 2 bits are same as in write mode:  
 * Bit 0: BIST check; Bit 1: BIST go 
 *)
CONST SP_IBIST_DONE=		$04;		(* Bit 2: BIST done *)
CONST SP_IBIST_FAILED=		$78;		(* Bit [6:3]: BIST fail *)


(*************************************************************************
 * DP Command Registers 
 *)
CONST DPC_BASE_REG=		$04100000;

(* DP CMD DMA start (R/W): [23:0] DMEM/RDRAM start address *)
CONST DPC_START_REG=		(DPC_BASE_REG+$00);

(* DP CMD DMA end (R/W): [23:0] DMEM/RDRAM end address *)
CONST DPC_END_REG=		(DPC_BASE_REG+$04);

(* DP CMD DMA end (R): [23:0] DMEM/RDRAM current address *)
CONST DPC_CURRENT_REG=		(DPC_BASE_REG+$08);	

(* DP CMD status (R/W): [9:0] valid bits - see below for definitions *)
CONST DPC_STATUS_REG=		(DPC_BASE_REG+$0C);

(* DP clock counter (R): [23:0] clock counter *)
CONST DPC_CLOCK_REG=		(DPC_BASE_REG+$10);	

(* DP buffer busy counter (R): [23:0] clock counter *)
CONST DPC_BUFBUSY_REG=		(DPC_BASE_REG+$14);

(* DP pipe busy counter (R): [23:0] clock counter *)
CONST DPC_PIPEBUSY_REG=	(DPC_BASE_REG+$18);

(* DP TMEM load counter (R): [23:0] clock counter *)
CONST DPC_TMEM_REG=		(DPC_BASE_REG+$1C);

(*
 * Values to clear/set bit in status reg (DPC_STATUS_REG - write)
 *)
CONST DPC_CLR_XBUS_DMEM_DMA=	$0001;		(* Bit 0: clear xbus_dmem_dma *)
CONST DPC_SET_XBUS_DMEM_DMA=	$0002;		(* Bit 1: set xbus_dmem_dma *)
CONST DPC_CLR_FREEZE=		$0004;		(* Bit 2: clear freeze *)
CONST DPC_SET_FREEZE=		$0008;		(* Bit 3: set freeze *)
CONST DPC_CLR_FLUSH=		$0010;		(* Bit 4: clear flush *)
CONST DPC_SET_FLUSH=		$0020;		(* Bit 5: set flush *)
CONST DPC_CLR_TMEM_CTR=	        $0040;		(* Bit 6: clear tmem ctr *)
CONST DPC_CLR_PIPE_CTR=	        $0080;		(* Bit 7: clear pipe ctr *)
CONST DPC_CLR_CMD_CTR=		$0100;		(* Bit 8: clear cmd ctr *)
CONST DPC_CLR_CLOCK_CTR=	$0200;		(* Bit 9: clear clock ctr *)

(*
 * Patterns to interpret status reg (DPC_STATUS_REG - read)
 *)
CONST DPC_STATUS_XBUS_DMEM_DMA=	        $001;	(* Bit  0: xbus_dmem_dma *)
CONST DPC_STATUS_FREEZE=		$002;	(* Bit  1: freeze *)
CONST DPC_STATUS_FLUSH=		        $004;	(* Bit  2: flush *)
(*CONST DPC_STATUS_FROZEN=		$008*)	(* Bit  3: frozen *)
CONST DPC_STATUS_START_GCLK=		$008;	(* Bit  3: start gclk *)
CONST DPC_STATUS_TMEM_BUSY=		$010;	(* Bit  4: tmem busy *)
CONST DPC_STATUS_PIPE_BUSY=		$020;	(* Bit  5: pipe busy *)
CONST DPC_STATUS_CMD_BUSY=		$040;	(* Bit  6: cmd busy *)
CONST DPC_STATUS_CBUF_READY=		$080;	(* Bit  7: cbuf ready *)
CONST DPC_STATUS_DMA_BUSY=		$100;	(* Bit  8: dma busy *)
CONST DPC_STATUS_END_VALID=		$200;	(* Bit  9: end valid *)
CONST DPC_STATUS_START_VALID=		$400;	(* Bit 10: start valid *)


(*************************************************************************
 * DP Span Registers 
 *)
CONST DPS_BASE_REG=		$04200000;

(* DP tmem bist (R/W): [10:0] BIST status bits; see below for detail *)
CONST DPS_TBIST_REG=		(DPS_BASE_REG+$00);

(* DP span test mode (R/W): [0] Span buffer test access enable *)
CONST DPS_TEST_MODE_REG=	(DPS_BASE_REG+$04);

(* DP span buffer test address (R/W): [6:0] bits; see below for detail *)
CONST DPS_BUFTEST_ADDR_REG=	(DPS_BASE_REG+$08);

(* DP span buffer test data (R/W): [31:0] span buffer data *)
CONST DPS_BUFTEST_DATA_REG=	(DPS_BASE_REG+$0C);

(*
 * Patterns to interpret status reg (DPS_TMEM_BIST_REG - write)
 *)
CONST DPS_TBIST_CHECK=		$01;		(* Bit 0: BIST check *)
CONST DPS_TBIST_GO=		$02;		(* Bit 1: BIST go *)
CONST DPS_TBIST_CLEAR=		$04;		(* Bit 2: BIST clear *)

(*
 * Patterns to interpret status reg (DPS_TMEM_BIST_REG - read)
 *)
(* First 2 bits are same as in write mode:  
 * Bit 0: BIST check; Bit 1: BIST go 
 *)
CONST DPS_TBIST_DONE=		$004;		(* Bit 2: BIST done *)
CONST DPS_TBIST_FAILED=	$7F8;		(* Bit [10:3]: BIST fail *)


(*************************************************************************
 * MIPS Interface (MI) Registers 
 *)
CONST MI_BASE_REG=		$04300000;

(*
 * MI init mode (W): [6:0] init length, [7] clear init mode, [8] set init mode
 *                   [9/10] clear/set ebus test mode, [11] clear DP interrupt
 *              (R): [6:0] init length, [7] init mode, [8] ebus test mode
 *)
CONST MI_INIT_MODE_REG=	(MI_BASE_REG+$00);
CONST MI_MODE_REG=     	MI_INIT_MODE_REG;

(*
 * Values to clear/set bit in mode reg (MI_MODE_REG - write)
 *)
CONST MI_CLR_INIT=		$0080;		(* Bit  7: clear init mode *)
CONST MI_SET_INIT=		$0100;		(* Bit  8: set init mode *)
CONST MI_CLR_EBUS=		$0200;		(* Bit  9: clear ebus test *)
CONST MI_SET_EBUS=		$0400;		(* Bit 10: set ebus test mode *)
CONST MI_CLR_DP_INTR=		$0800;		(* Bit 11: clear dp interrupt *)
CONST MI_CLR_RDRAM=		$1000;		(* Bit 12: clear RDRAM reg *)
CONST MI_SET_RDRAM=		$2000;		(* Bit 13: set RDRAM reg mode *)

(*
 * Patterns to interpret mode reg (MI_MODE_REG - read)
 *)
CONST MI_MODE_INIT=		$0080;		(* Bit  7: init mode *)
CONST MI_MODE_EBUS=		$0100;		(* Bit  8: ebus test mode *)
CONST MI_MODE_RDRAM=		$0200;		(* Bit  9: RDRAM reg mode *)

(* MI version (R): [7:0] io, [15:8] rac, [23:16] rdp, [31:24] rsp *)
CONST MI_VERSION_REG=		(MI_BASE_REG+$04);
CONST MI_NOOP_REG=		MI_VERSION_REG;

(* MI interrupt (R): [5:0] valid bits - see below for bit patterns *)
CONST MI_INTR_REG=		(MI_BASE_REG+$08);

(* 
 * MI interrupt mask (W): [11:0] valid bits - see below for bit patterns 
 *                   (R): [5:0] valid bits - see below for bit patterns 
 *)
CONST MI_INTR_MASK_REG=	(MI_BASE_REG+$0C);

(*
 * The following are values to check for interrupt setting (MI_INTR_REG)
 *)
CONST MI_INTR_SP=		$01;		(* Bit 0: SP intr *)
CONST MI_INTR_SI=		$02;		(* Bit 1: SI intr *)
CONST MI_INTR_AI=		$04;		(* Bit 2: AI intr *)
CONST MI_INTR_VI=		$08;		(* Bit 3: VI intr *)
CONST MI_INTR_PI=		$10;		(* Bit 4: PI intr *)
CONST MI_INTR_DP=		$20;		(* Bit 5: DP intr *)

(*
 * The following are values to clear/set various interrupt bit mask
 * They can be ORed together to manipulate multiple bits 
 * (MI_INTR_MASK_REG - write)
 *)
CONST MI_INTR_MASK_CLR_SP=	$0001;		(* Bit  0: clear SP mask *)
CONST MI_INTR_MASK_SET_SP=	$0002;		(* Bit  1: set SP mask *)
CONST MI_INTR_MASK_CLR_SI=	$0004;		(* Bit  2: clear SI mask *)
CONST MI_INTR_MASK_SET_SI=	$0008;		(* Bit  3: set SI mask *)
CONST MI_INTR_MASK_CLR_AI=	$0010;		(* Bit  4: clear AI mask *)
CONST MI_INTR_MASK_SET_AI=	$0020;		(* Bit  5: set AI mask *)
CONST MI_INTR_MASK_CLR_VI=	$0040;		(* Bit  6: clear VI mask *)
CONST MI_INTR_MASK_SET_VI=	$0080;		(* Bit  7: set VI mask *)
CONST MI_INTR_MASK_CLR_PI=	$0100;		(* Bit  8: clear PI mask *)
CONST MI_INTR_MASK_SET_PI=	$0200;		(* Bit  9: set PI mask *)
CONST MI_INTR_MASK_CLR_DP=	$0400;		(* Bit 10: clear DP mask *)
CONST MI_INTR_MASK_SET_DP=	$0800;		(* Bit 11: set DP mask *)

(*
 * The following are values to check for interrupt mask setting 
 * (MI_INTR_MASK_REG - read)
 *)
CONST MI_INTR_MASK_SP=		$01;		(* Bit 0: SP intr mask *)
CONST MI_INTR_MASK_SI=		$02;		(* Bit 1: SI intr mask *)
CONST MI_INTR_MASK_AI=		$04;		(* Bit 2: AI intr mask *)
CONST MI_INTR_MASK_VI=		$08;		(* Bit 3: VI intr mask *)
CONST MI_INTR_MASK_PI=		$10;		(* Bit 4: PI intr mask *)
CONST MI_INTR_MASK_DP=		$20;		(* Bit 5: DP intr mask *)


(*************************************************************************
 * Video Interface (VI) Registers 
 *)
CONST VI_BASE_REG=		$04400000;

(* VI status/control (R/W): [15-0] valid bits: 
 *	[1:0]   = type[1:0] (pixel size) 
 *			0: blank (no data, no sync)
 *                      1: reserved
 *                      2: 5/5/5/3 ("16" bit)
 *                      3: 8/8/8/8 (32 bit)
 *	[2]     = gamma_dither_enable (normally on, unless "special effect")
 *	[3]     = gamma_enable (normally on, unless MPEG/JPEG)
 *	[4]     = divot_enable (normally on if antialiased, unless decal lines)
 *	[5]     = reserved - always off
 *	[6]     = serrate (always on if interlaced, off if not)
 *	[7]     = reserved - diagnostics only
 *	[9:8]   = anti-alias (aa) mode[1:0]
 *			0: aa  AND  resamp (always fetch extra lines)
 *                      1: aa  AND  resamp (fetch extra lines if needed)
 *                      2: resamp only (treat as all fully covered)
 *                      3: neither (replicate pixels, no interpolate)
 *	[11]    = reserved - diagnostics only
 *	[15:12] = reserved
 *
 *)
CONST VI_STATUS_REG=		(VI_BASE_REG+$00);
CONST VI_CONTROL_REG=		VI_STATUS_REG;

(* VI origin (R/W): [23:0] frame buffer origin in bytes *)
CONST VI_ORIGIN_REG=		(VI_BASE_REG+$04);
CONST VI_DRAM_ADDR_REG=	        VI_ORIGIN_REG;

(* VI width (R/W): [11:0] frame buffer line width in pixels *)
CONST VI_WIDTH_REG=		(VI_BASE_REG+$08);
CONST VI_H_WIDTH_REG=		VI_WIDTH_REG;

(* VI vertical intr (R/W): [9:0] interrupt when current half-line = V_INTR *)
CONST VI_INTR_REG=		(VI_BASE_REG+$0C);
CONST VI_V_INTR_REG=		VI_INTR_REG;

(* 
 * VI current vertical line (R/W): [9:0] current half line, sampled once per
 *	line (the lsb of V_CURRENT is constant within a field, and in
 *	interlaced modes gives the field number - which is constant for non-
 *	interlaced modes)
 * 	- Any write to this register will clear interrupt line
 *)
CONST VI_CURRENT_REG=		(VI_BASE_REG+$10);
CONST VI_V_CURRENT_LINE_REG=	VI_CURRENT_REG;

(* 
 * VI video timing (R/W): [ 7: 0] horizontal sync width in pixels, 
 *			  [15: 8] color burst width in pixels,
 *                        [19:16] vertical sync width in half lines,   
 *			  [29:20] start of color burst in pixels from h-sync
 *)
CONST VI_BURST_REG=		(VI_BASE_REG+$14);
CONST VI_TIMING_REG=		VI_BURST_REG;

(* VI vertical sync (R/W): [9:0] number of half-lines per field *)
CONST VI_V_SYNC_REG=		(VI_BASE_REG+$18);

(* VI horizontal sync (R/W): [11: 0] total duration of a line in 1/4 pixel 
 *			     [20:16] a 5-bit leap pattern used for PAL only 
 *				     (h_sync_period)
 *)
CONST VI_H_SYNC_REG=		(VI_BASE_REG+$1C);

(* 
 * VI horizontal sync leap (R/W): [11: 0] identical to h_sync_period
 *                                [27:16] identical to h_sync_period
 *)
CONST VI_LEAP_REG=		(VI_BASE_REG+$20);
CONST VI_H_SYNC_LEAP_REG=	VI_LEAP_REG;

(* 
 * VI horizontal video (R/W): [ 9: 0] end of active video in screen pixels
 *                          : [25:16] start of active video in screen pixels
 *)
CONST VI_H_START_REG=		(VI_BASE_REG+$24);
CONST VI_H_VIDEO_REG=		VI_H_START_REG;

(* 
 * VI vertical video (R/W): [ 9: 0] end of active video in screen half-lines
 *                        : [25:16] start of active video in screen half-lines
 *)
CONST VI_V_START_REG=		(VI_BASE_REG+$28);
CONST VI_V_VIDEO_REG=		VI_V_START_REG;

(* 
 * VI vertical burst (R/W): [ 9: 0] end of color burst enable in half-lines
 *                        : [25:16] start of color burst enable in half-lines
 *)
CONST VI_V_BURST_REG=		(VI_BASE_REG+$2C);

(* VI x-scale (R/W): [11: 0] 1/horizontal scale up factor (2.10 format)
 *		     [27:16] horizontal subpixel offset (2.10 format)
 *)
CONST VI_X_SCALE_REG=		(VI_BASE_REG+$30);

(* VI y-scale (R/W): [11: 0] 1/vertical scale up factor (2.10 format)
 *		     [27:16] vertical subpixel offset (2.10 format)
 *)
CONST VI_Y_SCALE_REG=		(VI_BASE_REG+$34);

(*
 * Patterns to interpret VI_CONTROL_REG
 *)
CONST VI_CTRL_TYPE_16=		 $00002;    (* Bit [1:0] pixel size: 16 bit *)
CONST VI_CTRL_TYPE_32=		 $00003;    (* Bit [1:0] pixel size: 32 bit *)
CONST VI_CTRL_GAMMA_DITHER_ON=	 $00004;    (* Bit 2: default = on *)
CONST VI_CTRL_GAMMA_ON=	         $00008;    (* Bit 3: default = on *)
CONST VI_CTRL_DIVOT_ON=	         $00010;    (* Bit 4: default = on *)
CONST VI_CTRL_SERRATE_ON=	 $00040;    (* Bit 6: on if interlaced *)
CONST VI_CTRL_ANTIALIAS_MASK=	 $00300;    (* Bit [9:8] anti-alias mode *)
CONST VI_CTRL_DITHER_FILTER_ON=  $10000;    (* Bit 16: dither-filter mode *)

(*
 * Possible video clocks (NTSC or PAL)
 *)
CONST VI_NTSC_CLOCK=		48681812;        (* Hz = 48.681812 MHz *)
CONST VI_PAL_CLOCK=		49656530;        (* Hz = 49.656530 MHz *)
CONST VI_MPAL_CLOCK=		48628316;        (* Hz = 48.628316 MHz *)


(*************************************************************************
 * Audio Interface (AI) Registers 
 *
 * The address and length registers are double buffered; that is, they
 * can be written twice before becoming full.
 * The address must be written before the length.
 *)
CONST AI_BASE_REG=		$04500000;

(* AI DRAM address (W): [23:0] starting RDRAM address (8B-aligned) *)
CONST AI_DRAM_ADDR_REG=	(AI_BASE_REG+$00);	(* R0: DRAM address *)

(* AI length (R/W): [14:0] transfer length (v1.0) - Bottom 3 bits are ignored *)
(*                  [17:0] transfer length (v2.0) - Bottom 3 bits are ignored *)
CONST AI_LEN_REG=		(AI_BASE_REG+$04);	(* R1: Length *)

(* AI control (W): [0] DMA enable - if LSB == 1, DMA is enabled *)
CONST AI_CONTROL_REG=		(AI_BASE_REG+$08);	(* R2: DMA Control *)

(*
 * AI status (R): [31]/[0] ai_full (addr  AND  len buffer full), [30] ai_busy
 *		  Note that a 1->0 transition in ai_full will set interrupt
 *           (W): clear audio interrupt
 *)
CONST AI_STATUS_REG=		(AI_BASE_REG+$0C);	(* R3: Status *)

(*
 * AI DAC sample period register (W): [13:0] dac rate
 *   - vid_clock/(dperiod + 1) is the DAC sample rate
 *   - (dperiod + 1) >= 66 * (aclockhp + 1) must be true
 *)
CONST AI_DACRATE_REG=		(AI_BASE_REG+$10);	(* R4: DAC rate 14-lsb*)

(*
 * AI bit rate (W): [3:0] bit rate (abus clock half period register - aclockhp)
 *   - vid_clock/(2 * (aclockhp + 1)) is the DAC clock rate
 *   - The abus clock stops if aclockhp is zero
 *)
CONST AI_BITRATE_REG=		(AI_BASE_REG+$14);	(* R5: Bit rate 4-lsb *)

(* Value for control register *)
CONST AI_CONTROL_DMA_ON=	$01;			(* LSB = 1: DMA enable*)
CONST AI_CONTROL_DMA_OFF=	$00;			(* LSB = 1: DMA enable*)

(* Value for status register *)
CONST AI_STATUS_FIFO_FULL=	$80000000;		(* Bit 31: full *)
CONST AI_STATUS_DMA_BUSY=	$40000000;		(* Bit 30: busy *)

(* DAC rate = video clock / audio frequency
 *   - DAC rate >= (66 * Bit rate) must be true
 *)
CONST AI_MAX_DAC_RATE=         16384;           (* 14-bit+1 *)
CONST AI_MIN_DAC_RATE=         132;

(* Bit rate <= (DAC rate / 66) *)
CONST AI_MAX_BIT_RATE=         16;              (* 4-bit+1 *)
CONST AI_MIN_BIT_RATE=         2;

(*
 * Maximum and minimum values for audio frequency based on video clocks
 *   max frequency = (video clock / min dac rate)
 *   min frequency = (video clock / max dac rate)
 *)
CONST AI_NTSC_MAX_FREQ=        368000;          (* 368 KHz *)
CONST AI_NTSC_MIN_FREQ=        3000;            (*   3 KHz ~ 2971 Hz *)

CONST AI_PAL_MAX_FREQ=         376000;          (* 376 KHz *)
CONST AI_PAL_MIN_FREQ=         3050;            (*   3 KHz ~ 3031 Hz *)

CONST AI_MPAL_MAX_FREQ=        368000;          (* 368 KHz *)
CONST AI_MPAL_MIN_FREQ=        3000;            (*   3 KHz ~ 2968 Hz *)


(*************************************************************************
 * Peripheral Interface (PI) Registers 
 *)
CONST PI_BASE_REG=		$04600000;

(* PI DRAM address (R/W): [23:0] starting RDRAM address *)
CONST PI_DRAM_ADDR_REG=	(PI_BASE_REG+$00);	(* DRAM address *)

(* PI pbus (cartridge) address (R/W): [31:0] starting AD16 address *)
CONST PI_CART_ADDR_REG=	(PI_BASE_REG+$04);

(* PI read length (R/W): [23:0] read data length *)
CONST PI_RD_LEN_REG=		(PI_BASE_REG+$08);

(* PI write length (R/W): [23:0] write data length *)
CONST PI_WR_LEN_REG=		(PI_BASE_REG+$0C);

(*
 * PI status (R): [0] DMA busy, [1] IO busy, [2], error
 *           (W): [0] reset controller (and abort current op), [1] clear intr
 *)
CONST PI_STATUS_REG=		(PI_BASE_REG+$10);

(* PI dom1 latency (R/W): [7:0] domain 1 device latency *)
CONST PI_BSD_DOM1_LAT_REG=	(PI_BASE_REG+$14);

(* PI dom1 pulse width (R/W): [7:0] domain 1 device R/W strobe pulse width *)
CONST PI_BSD_DOM1_PWD_REG=	(PI_BASE_REG+$18);

(* PI dom1 page size (R/W): [3:0] domain 1 device page size *)
CONST PI_BSD_DOM1_PGS_REG=	(PI_BASE_REG+$1C);    (*   page size *)

(* PI dom1 release (R/W): [1:0] domain 1 device R/W release duration *)
CONST PI_BSD_DOM1_RLS_REG=	(PI_BASE_REG+$20);

(* PI dom2 latency (R/W): [7:0] domain 2 device latency *)
CONST PI_BSD_DOM2_LAT_REG=	(PI_BASE_REG+$24);    (* Domain 2 latency *)

(* PI dom2 pulse width (R/W): [7:0] domain 2 device R/W strobe pulse width *)
CONST PI_BSD_DOM2_PWD_REG=	(PI_BASE_REG+$28);    (*   pulse width *)

(* PI dom2 page size (R/W): [3:0] domain 2 device page size *)
CONST PI_BSD_DOM2_PGS_REG=	(PI_BASE_REG+$2C);    (*   page size *)

(* PI dom2 release (R/W): [1:0] domain 2 device R/W release duration *)
CONST PI_BSD_DOM2_RLS_REG=	(PI_BASE_REG+$30);    (*   release duration *)

CONST PI_DOMAIN1_REG=		PI_BSD_DOM1_LAT_REG;
CONST PI_DOMAIN2_REG=		PI_BSD_DOM2_LAT_REG;

CONST PI_DOM_LAT_OFS=		$00;
CONST PI_DOM_PWD_OFS=		$04;
CONST PI_DOM_PGS_OFS=		$08;
CONST PI_DOM_RLS_OFS=		$0C;

(*
 * PI status register has 3 bits active when read from (PI_STATUS_REG - read)
 *	Bit 0: DMA busy - set when DMA is in progress
 *	Bit 1: IO busy  - set when IO is in progress
 *	Bit 2: Error    - set when CPU issues IO request while DMA is busy
 *)
CONST	PI_STATUS_ERROR=    	$04;
CONST	PI_STATUS_IO_BUSY=	$02;
CONST	PI_STATUS_DMA_BUSY=	$01;

(* PI status register has 2 bits active when written to:
 *	Bit 0: When set, reset PIC
 *	Bit 1: When set, clear interrupt flag
 * The values of the two bits can be ORed together to both reset PIC and
 * clear interrupt at the same time.
 *
 * Note:
 *	- The PIC does generate an interrupt at the end of each DMA. CPU
 *	needs to clear the interrupt flag explicitly (from an interrupt
 *	handler) by writing into the STATUS register with bit 1 set.
 *
 *	- When a DMA completes, the interrupt flag is set.  CPU can issue
 *	another request even while the interrupt flag is set (as long as
 *	PIC is idle).  However, it is the CPU's responsibility for
 *	maintaining accurate correspondence between DMA completions and
 *	interrupts.
 *
 *	- When PIC is reset, if PIC happens to be busy, an interrupt will
 *	be generated as PIC returns to idle.  Otherwise, no interrupt will
 *	be generated and PIC remains idle.
 *)
(*
 * Values to clear interrupt/reset PIC (PI_STATUS_REG - write)
 *)
CONST PI_STATUS_RESET=		$01;
CONST PI_SET_RESET=		PI_STATUS_RESET;

CONST PI_STATUS_CLR_INTR=	$02;
CONST PI_CLR_INTR=		PI_STATUS_CLR_INTR;

CONST PI_DMA_BUFFER_SIZE=	128;

CONST PI_DOM1_ADDR1=		$06000000;	(* to $07FFFFFF *)
CONST PI_DOM1_ADDR2=		$10000000;	(* to $1FBFFFFF *)
CONST PI_DOM1_ADDR3=		$1FD00000;	(* to $7FFFFFFF *)
CONST PI_DOM2_ADDR1=		$05000000;	(* to $05FFFFFF *)
CONST PI_DOM2_ADDR2=		$08000000;	(* to $0FFFFFFF *)


(*************************************************************************
 * RDRAM Interface (RI) Registers 
 *)
CONST RI_BASE_REG=		$04700000;

(* RI mode (R/W): [1:0] operating mode, [2] stop T active, [3] stop R active *)
CONST RI_MODE_REG=		(RI_BASE_REG+$00);

(* RI config (R/W): [5:0] current control input, [6] current control enable *)
CONST RI_CONFIG_REG=		(RI_BASE_REG+$04);

(* RI current load (W): [] any write updates current control register *)
CONST RI_CURRENT_LOAD_REG=	(RI_BASE_REG+$08);

(* RI select (R/W): [2:0] receive select, [2:0] transmit select *)
CONST RI_SELECT_REG=		(RI_BASE_REG+$0C);

(* RI refresh (R/W): [7:0] clean refresh delay, [15:8] dirty refresh delay,
 *		     [16] refresh bank, [17] refresh enable 
 *		     [18] refresh optimize 
 *)
CONST RI_REFRESH_REG=		(RI_BASE_REG+$10);
CONST RI_COUNT_REG=		RI_REFRESH_REG;

(* RI latency (R/W): [3:0] DMA latency/overlap *)
CONST RI_LATENCY_REG=		(RI_BASE_REG+$14);

(* RI error (R): [0] nack error, [1] ack error *)
CONST RI_RERROR_REG=		(RI_BASE_REG+$18);

(* RI error (W): [] any write clears all error bits *)
CONST RI_WERROR_REG=		(RI_BASE_REG+$1C);


(*************************************************************************
 * Serial Interface (SI) Registers 
 *)
CONST SI_BASE_REG=		$04800000;

(* SI DRAM address (R/W): [23:0] starting RDRAM address *)
CONST SI_DRAM_ADDR_REG=	(SI_BASE_REG+$00);	(* R0: DRAM address *)

(* SI address read 64B (W): [] any write causes a 64B DMA write *)
CONST SI_PIF_ADDR_RD64B_REG=	(SI_BASE_REG+$04);	(* R1: 64B PIF->DRAM *)

(* Address SI_BASE_REG + ($08, $0c, $14) are reserved *)

(* SI address write 64B (W): [] any write causes a 64B DMA read *)
CONST SI_PIF_ADDR_WR64B_REG=	(SI_BASE_REG+$10);	(* R4: 64B DRAM->PIF *)

(* 
 * SI status (W): [] any write clears interrupt
 *           (R): [0] DMA busy, [1] IO read busy, [2] reserved
 *                [3] DMA error, [12] interrupt
 *)
CONST SI_STATUS_REG=		(SI_BASE_REG+$18);	(* R6: Status *)

(* SI status register has the following bits active:
 *	0:   DMA busy		- set when DMA is in progress
 *	1:   IO busy		- set when IO access is in progress
 *	3:   DMA error		- set when there are overlapping DMA requests
 *     12:   Interrupt		- Interrupt set
 *)
CONST	SI_STATUS_DMA_BUSY=	$0001;
CONST	SI_STATUS_RD_BUSY=	$0002;
CONST	SI_STATUS_DMA_ERROR=	$0008;
CONST	SI_STATUS_INTERRUPT=	$1000;

(*************************************************************************
 * Development Board GIO Control Registers 
 *)

CONST GIO_BASE_REG=		$18000000;

(* Game to Host Interrupt *)
CONST GIO_GIO_INTR_REG=	(GIO_BASE_REG+$000);

(* Game to Host SYNC *)
CONST GIO_GIO_SYNC_REG=	(GIO_BASE_REG+$400);

(* Host to Game Interrupt *)
CONST GIO_CART_INTR_REG=	(GIO_BASE_REG+$800);


implementation

end.
