/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL/SDL.h>
#include <SDL/SDL_mixer.h>
#include <SDL/SDL_image.h>
#include <SDL/SDL_ttf.h>
#include <png.h>

#include "global.h"
#include "psp_sdl.h"
#include "psp_danzeff.h"
#include "psp_ti99.h"

int psp_screenshot_mode = 0;
int ti99_in_menu = 0;

static void
ti99_render_normal()
{
  int x; 
  int y;
  u16 *src_pixel = (u16*)blit_surface->pixels;
  u16 *dst_pixel = (u16*)back_surface->pixels;
  dst_pixel += (320 - TI99_SCREEN_W) / 2;
  dst_pixel += ((240 - TI99_SCREEN_H) * PSP_LINE_SIZE / 2);
  for (y = 0; y < TI99_SCREEN_H; ++y) {
    for (x = 0; x < TI99_SCREEN_W; ++x) {
      dst_pixel[x] = src_pixel[x];
    }
    dst_pixel += PSP_LINE_SIZE;
    src_pixel += TI99_SCREEN_W;
  }
}

/* 

  LUDO: 16-bit HiColor (565 format) 
  see http://www.compuphase.com/graphic/scale3.htm

 */
static inline u16 loc_coloraverage(u16 a, u16 b)
{
  return (u16)(((a ^ b) & 0xf7deU) >> 1) + (a & b);
}

static inline void 
ti99_X125_pixel(u16 *dist, const u16 *src)
{
  dist[0] = src[0];
  dist[1] = loc_coloraverage(src[0], src[1]);
  dist[2] = src[1];
  dist[3] = src[2];
  dist[4] = src[3];
}

static void
ti99_render_fit()
{
  u16 *src_pixel = (u16*)blit_surface->pixels;
  u16 *dst_pixel = (u16*)back_surface->pixels;

  int src_y;
  int dst_x;
  int dst_y;
  int count;

  for (dst_y = 0; dst_y < 240; dst_y++) {
    src_y = (dst_y * TI99_SCREEN_H) / 240;
    u16* src_line = &src_pixel[(src_y * TI99_SCREEN_W)];
    u16* dst_line = dst_pixel;

    count = 320;
    while (count > 0) {
      ti99_X125_pixel(dst_line, src_line);
      src_line += 4;
      dst_line += 5;
      count -= 4;
    }
    dst_pixel += PSP_LINE_SIZE;
  }
}

static void
ti99_synchronize(void)
{
	static u32 nextclock = 1;

  if (TI99.ti99_speed_limiter) {

	  if (nextclock) {
		  u32 curclock;
		  do {
        curclock = SDL_GetTicks();
		  } while (curclock < nextclock);
  
      nextclock = curclock + (u32)( 1000 / TI99.ti99_speed_limiter);
    }
  }
}

void
ti99_update_fps()
{
  static u32 next_sec_clock = 0;
  static u32 cur_num_frame = 0;
  cur_num_frame++;
  u32 curclock = SDL_GetTicks();
  if (curclock > next_sec_clock) {
    next_sec_clock = curclock + 1000;
    TI99.ti99_current_fps = cur_num_frame;
    cur_num_frame = 0;
  }
}

void
psp_sdl_render()
{
  if (ti99_in_menu) return;

  if (TI99.psp_skip_cur_frame <= 0) {

    TI99.psp_skip_cur_frame = TI99.psp_skip_max_frame;

    if (TI99.ti99_render_mode == TI99_RENDER_NORMAL) ti99_render_normal();
    else                          
    if (TI99.ti99_render_mode == TI99_RENDER_FIT) ti99_render_fit();

    if (psp_kbd_is_danzeff_mode()) {

      danzeff_moveTo(-50, -50);
      danzeff_render();
    }

    if (TI99.ti99_view_fps) {
      char buffer[32];
      sprintf(buffer, "%3d", (int)TI99.ti99_current_fps);
      psp_sdl_fill_print(0, 0, buffer, 0xffffff, 0 );
    }

    if (TI99.psp_display_lr) {
      psp_kbd_display_active_mapping();
    }
    psp_sdl_flip();
  
    if (psp_screenshot_mode) {
      psp_screenshot_mode--;
      if (psp_screenshot_mode <= 0) {
        psp_sdl_save_screenshot();
        psp_screenshot_mode = 0;
      }
    }

  } else if (TI99.psp_skip_max_frame) {
    TI99.psp_skip_cur_frame--;
  }

  if (TI99.ti99_speed_limiter) {
    ti99_synchronize();
  }

  if (TI99.ti99_view_fps) {
    ti99_update_fps();
  }
}
