/*
 * Texture Filtering
 * Version:  1.0
 *
 * Copyright (C) 2007  Hiroshi Morii   All Rights Reserved.
 * Email koolsmoky(at)users.sourceforge.net
 * Web   http://www.3dfxzone.it/koolsmoky
 *
 * this is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * this is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Make; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __MSC__
#pragma warning(disable: 4786)
#endif

/* dump processed hirestextures to disk 
 * (0:disable, 1:enable) */
#define DUMP_CACHE 1

/* handle oversized textures by
 *   0: minification
 *   1: Glide64 style tiling
 */
#define TEXTURE_TILING 1

/* use power of 2 texture size
 * (0:disable, 1:enable, 2:3dfx) */
#define POW2_TEXTURES 2

#if TEXTURE_TILING
#undef POW2_TEXTURES
#define POW2_TEXTURES 2
#endif

/* hack to reduce texture footprint to achieve
 * better performace on midrange gfx cards.
 * (0:disable, 1:enable) */
#define REDUCE_TEXTURE_FOOTPRINT 0

/* use aggressive format assumption for quantization
 * (0:disable, 1:enable, 2:extreme) */
#define AGGRESSIVE_QUANTIZATION 1

#include "TxHiResCache.h"
#include "TxDbg.h"
#include <zlib.h>
#include <string>

TxHiResCache::~TxHiResCache()
{
#if DUMP_CACHE
  if (_options & DUMP_HIRESTEXCACHE && !_haveCache && !_abortLoad) {
    /* dump cache to disk */
    std::wstring filename = _ident + L"_HIRESTEXTURES.dat";
    boost::filesystem::wpath cachepath(_path);
    cachepath /= boost::filesystem::wpath(L"cache");
    int config = _options & (HIRESTEXTURES_MASK|COMPRESS_HIRESTEX|COMPRESSION_MASK|TILE_HIRESTEX|FORCE16BPP_HIRESTEX|GZ_HIRESTEXCACHE|LET_TEXARTISTS_FLY);

    TxCache::save(cachepath.string().c_str(), filename.c_str(), config);
  }
#endif

  delete _txImage;
  delete _txQuantize;
  delete _txReSample;
}

TxHiResCache::TxHiResCache(int maxwidth, int maxheight, int maxbpp, int options,
                           const wchar_t *path, const wchar_t *ident,
                           dispInfoFuncExt callback
                           ) : TxCache((options & ~GZ_TEXCACHE), 0, path, ident, callback)
{
  _txImage = new TxImage();
  _txQuantize  = new TxQuantize();
  _txReSample = new TxReSample();

  _maxwidth  = maxwidth;
  _maxheight = maxheight;
  _maxbpp    = maxbpp;
  _abortLoad = 0;

  /* assert local options */
  if (!(_options & COMPRESS_HIRESTEX))
    _options &= ~COMPRESSION_MASK;

  if (_path.empty() || _ident.empty())
    _options &= ~DUMP_HIRESTEXCACHE;

  /* read in hires textures */
  if (!_path.empty() && !_ident.empty()) {
    _haveCache = 0;

#if DUMP_CACHE
    if (_options & DUMP_HIRESTEXCACHE) {
      /* find it on disk */
      std::wstring filename = _ident + L"_HIRESTEXTURES.dat";
      boost::filesystem::wpath cachepath(_path);
      cachepath /= boost::filesystem::wpath(L"cache");
      int config = _options & (HIRESTEXTURES_MASK|COMPRESS_HIRESTEX|COMPRESSION_MASK|TILE_HIRESTEX|FORCE16BPP_HIRESTEX|GZ_HIRESTEXCACHE|LET_TEXARTISTS_FLY);

      _haveCache = TxCache::load(cachepath.string().c_str(), filename.c_str(), config);
    }
#endif

    if (!_haveCache) {
      boost::filesystem::wpath dir_path(path);
      switch (_options & HIRESTEXTURES_MASK) {
      case GHQ_HIRESTEXTURES:
        break;
      case RICE_HIRESTEXTURES:
        dir_path /= boost::filesystem::wpath(L"hires_texture");
        dir_path /= boost::filesystem::wpath(_ident);
        loadHiResTextures(dir_path);
        break;
      case JABO_HIRESTEXTURES:
        ;
      }
    }
  }
}

boolean
TxHiResCache::empty()
{
  return _cache.empty();
}

boolean
TxHiResCache::loadHiResTextures(boost::filesystem::wpath dir_path)
{
  DBG_INFO(80, L"%s\n", dir_path.string().c_str());

  /* find it on disk */
  if (!boost::filesystem::exists(dir_path)) {
    DBG_INFO(80, L"Error: directory not found!\n");
    return 0;
  }

  /* XXX: deal with UNICODE fiasco!
   * stupidity flows forth beneath this...
   *
   * I opted to use chdir in order to use fopen() for windows 9x.
   */
  wchar_t curpath[MAX_PATH];
  GETCWD(MAX_PATH, curpath);
  CHDIR(dir_path.string().c_str());

  /* NOTE: I could use the boost::wdirectory_iterator and boost::wpath
   * to resolve UNICODE file names and paths. But then, _wfopen() is
   * required to get the file descriptor for MS Windows to pass into
   * libpng, which is incompatible with Win9x. Win9x's fopen() cannot
   * handle UNICODE names. UNICODE capable boost::filesystem is available
   * with Boost1.34.1 built with VC8.0 (bjam --toolset=msvc-8.0 stage).
   *
   * RULE OF THUMB: NEVER save texture packs in NON-ASCII names!!
   */
  boost::filesystem::wdirectory_iterator it(dir_path);
  boost::filesystem::wdirectory_iterator end_it; /* default construction yields past-the-end */

  for (; it != end_it; ++it) {

    if (KBHIT(0x1B)) {
      _abortLoad = 1;
      if (_callback) (*_callback)(L"Aborted loading hiresolution texture!\n");
      DBG_INFO(80, L"Error: Aborted loading hiresolution texture!\n");
    }
    if (_abortLoad) break;

    DBG_INFO(80, L"-----\n");

    /* recursive read into sub-directory */
    if (boost::filesystem::is_directory(it->status())) {
      loadHiResTextures(it->path());
      continue;
    }

    int width = 0, height = 0;
    uint16 format = 0;
    uint8 *tex = NULL;
    int tmpwidth = 0, tmpheight = 0;
    uint16 tmpformat = 0;
    uint8 *tmptex= NULL;
    int untiled_width = 0, untiled_height = 0;
    uint16 destformat = 0;

    /* Rice hi-res textures: begin
     */
    uint32 chksum = 0, fmt = 0, siz = 0, palchksum = 0;
    char *pfname = NULL, fname[MAX_PATH];
    std::string ident, leaf;
    FILE *fp = NULL;

    wcstombs(fname, _ident.c_str(), MAX_PATH);
    ident.assign(fname);
    wcstombs(fname, it->path().leaf().c_str(), MAX_PATH);
    leaf.assign(fname);

    /* read in Rice's file naming convention */
    pfname = strstr(fname, leaf.c_str());
    pfname = strstr(pfname, ident.c_str());
    if (!(pfname && sscanf(pfname + ident.size(), "#%08X#%01x#%01x", &chksum, &fmt, &siz) != EOF)) {
      DBG_INFO(80, L"%s\n", it->path().leaf().c_str());
      DBG_INFO(80, L"Error: Not Rice texture name format!\n");
      continue;
    }
    if (!chksum) {
      DBG_INFO(80, L"%s\n", it->path().leaf().c_str());
      DBG_INFO(80, L"Error: checksum 0!\n");
      continue;
    }

    /* check if we already have it in hires texture cache */
    GHQTexInfo tmpInfo;
    memset(&tmpInfo, 0, sizeof(GHQTexInfo));
    if (get((uint64)chksum, &tmpInfo)) {
      DBG_INFO(80, L"%s\n", it->path().leaf().c_str());
      DBG_INFO(80, L"Warning: already cached.\n");
      continue; /* yep, we've got it */
    }

    DBG_INFO(80, L"chksum:%08X fmt:%x size:%x\n", chksum, fmt, siz);

    /* Deal with the wackiness some texture packs utilize Rice format.
     * Read in the following order: _a.* + _rgb.*, _all.png _ciByRGBA.png,
     * _allciByRGBA.png, and _ci.bmp. PNG are prefered over BMP.
     *
     * For some reason there are texture packs that include them all. Some
     * even have RGB textures named as _all.* and ARGB textures named as
     * _rgb.*... Someone pleeeez write a GOOD guideline for the texture
     * designers!!!
     *
     * We allow hires textures to have higher bpp than the N64 originals.
     */
    /* N64 formats
     * Format: 0 - RGBA, 1 - YUV, 2 - CI, 3 - IA, 4 - I
     * Size:   0 - 4bit, 1 - 8bit, 2 - 16bit, 3 - 32 bit
     */

#define CRCFMTSIZ_LEN 13
#define PALCRC_LEN 9

    pfname += (ident.size() + CRCFMTSIZ_LEN);

    /*
     * read in _rgb.* and _a.*
     */
    /* _a.png */
    strcpy(pfname, "_a.png");
    if ((fp = fopen(fname, "rb")) != NULL) {
      tmptex = _txImage->readPNG(fp, &tmpwidth, &tmpheight, &tmpformat);
      fclose(fp);
    }
    if (!tmptex) {
      /* _a.bmp */
      strcpy(pfname, "_a.bmp");
      if ((fp = fopen(fname, "rb")) != NULL) {
        tmptex = _txImage->readBMP(fp, &tmpwidth, &tmpheight, &tmpformat);
        fclose(fp);
      }
    }
#ifdef DEBUG
    if (tmptex) {
      wchar_t tmpbuf[MAX_PATH];
      mbstowcs(tmpbuf, fname, MAX_PATH);
      DBG_INFO(80, L"%s\n", tmpbuf);
      DBG_INFO(80, L"read in as %d x %d gfmt:%x\n", tmpwidth, tmpheight, tmpformat);
    }
#endif
    /* _rgb.png */
    strcpy(pfname, "_rgb.png");
    if ((fp = fopen(fname, "rb")) != NULL) {
      tex = _txImage->readPNG(fp, &width, &height, &format);
      fclose(fp);
    }
    if (!tex) {
      /* _rgb.bmp */
      strcpy(pfname, "_rgb.bmp");
      if ((fp = fopen(fname, "rb")) != NULL) {
        tex = _txImage->readBMP(fp, &tmpwidth, &tmpheight, &tmpformat);
        fclose(fp);
      }
    }
#ifdef DEBUG
    if (tex) {
      wchar_t tmpbuf[MAX_PATH];
      mbstowcs(tmpbuf, fname, MAX_PATH);
      DBG_INFO(80, L"%s\n", tmpbuf);
      DBG_INFO(80, L"read in as %d x %d gfmt:%x\n", width, height, format);
    }
#endif
    if (tmptex) {
      /* check if _rgb.* and _a.* have matching size and format. */
      if (!tex || width != tmpwidth || height != tmpheight ||
          format != GR_TEXFMT_ARGB_8888 || tmpformat != GR_TEXFMT_ARGB_8888) {
        if (tex) free(tex);
        if (tmptex) free(tmptex);
        tex = NULL;
        tmptex = NULL;
        DBG_INFO(80, L"Error: mismatched size or format!\n");
      }
    }
    /* make adjustments */
    if (tex) {
      if (tmptex) {
        /* merge (A)RGB and A comp */
        DBG_INFO(80, L"merge (A)RGB and A comp\n");
        int i;
        for (i = 0; i < height * width; i++) {
#if 1
          /* use R comp for alpha. this is what Rice uses. sigh... */
          ((uint32*)tex)[i] &= 0x00ffffff;
          ((uint32*)tex)[i] |= ((((uint32*)tmptex)[i] & 0x00ff0000) << 8);
#endif
#if 0
          /* use libpng style grayscale conversion */
          uint32 texel = ((uint32*)tmptex)[i];
          uint32 acomp = (((texel >> 16) & 0xff) * 6969 +
                          ((texel >>  8) & 0xff) * 23434 +
                          ((texel      ) & 0xff) * 2365) / 32768;
          ((uint32*)tex)[i] = (acomp << 24) | (((uint32*)tex)[i] & 0x00ffffff);
#endif
#if 0
          /* use the standard NTSC gray scale conversion */
          uint32 texel = ((uint32*)tmptex)[i];
          uint32 acomp = (((texel >> 16) & 0xff) * 299 +
                          ((texel >>  8) & 0xff) * 587 +
                          ((texel      ) & 0xff) * 114) / 1000;
          ((uint32*)tex)[i] = (acomp << 24) | (((uint32*)tex)[i] & 0x00ffffff);
#endif
        }
        free(tmptex);
        tmptex = NULL;
      } else {
        /* clobber A comp. never a question of alpha. only RGB used. */
        DBG_INFO(80, L"clobber A comp\n");
        int i;
        for (i = 0; i < height * width; i++) {
          ((uint32*)tex)[i] |= 0xff000000;
        }
      }
    }

    /*
     * read in _all.png. BMP variants are not used.
     */
    /* _all.png */
    if (!tex) {
      strcpy(pfname, "_all.png");
      if ((fp = fopen(fname, "rb")) != NULL) {
        tex = _txImage->readPNG(fp, &width, &height, &format);
        fclose(fp);
      }
    }
#ifdef DEBUG
    if (tex) {
      wchar_t tmpbuf[MAX_PATH];
      mbstowcs(tmpbuf, fname, MAX_PATH);
      DBG_INFO(80, L"%s\n", tmpbuf);
      DBG_INFO(80, L"read in as %d x %d gfmt:%x\n", width, height, format);
    }
#endif

    /*
     * read in _ciByRGBA.png, _allciByRGBA.png. BMP variants are not used.
     */
    /* _ciByRGBA.*, _allciByRGBA.* */
    if (!tex) {
      if (strstr(leaf.c_str(), "ciByRGBA.png")) {
        strcpy(pfname, leaf.c_str() + ident.size() + CRCFMTSIZ_LEN);
        if (sscanf(pfname, "#%08X", &palchksum) != EOF) {
          if ((fp = fopen(fname, "rb")) != NULL) {
            tex = _txImage->readPNG(fp, &width, &height, &format);
            fclose(fp);
#ifdef DEBUG
            if (tex) {
              wchar_t tmpbuf[MAX_PATH];
              mbstowcs(tmpbuf, fname, MAX_PATH);
              DBG_INFO(80, L"%s\n", tmpbuf);
              DBG_INFO(80, L"read in as %d x %d gfmt:%x palchksum:%08X\n", width, height, format, palchksum);
            }
#endif
          }
        }
      }
    }

    /* check if size and format are OK */
    if (tex) {
      if (format != GR_TEXFMT_ARGB_8888 ||
          (width * height) < 4) { /* TxQuantize requirement: width * height must be 4 or larger. */
        free(tex);
        tex = NULL;
      }
    }

    /*
     * read in CI bitmaps
     */
    if (!tex) {
      /* _ci.bmp */
      strcpy(pfname, "_ci.bmp");
      if ((fp = fopen(fname, "rb")) != NULL) {
        tex = _txImage->readBMP(fp, &width, &height, &format);
        fclose(fp);
      }
      if (tex) {
        wchar_t tmpbuf[MAX_PATH];
        mbstowcs(tmpbuf, fname, MAX_PATH);
        DBG_INFO(80, L"%s\n", tmpbuf);
        DBG_INFO(80, L"read in as %d x %d gfmt:%x\n", width, height, format);

        /* check if size and format are OK */
        if (format != GR_TEXFMT_P_8 ||
            (width * height) < 4) { /* TxQuantize requirement: width * height must be 4 or larger. */
          free(tex);
          tex = NULL;
        }
      }
    }

#ifdef DEBUG
    if (!tex) {
      wchar_t tmpbuf[MAX_PATH];
      mbstowcs(tmpbuf, fname, MAX_PATH);
      DBG_INFO(80, L"%s\n", tmpbuf);
      DBG_INFO(80, L"Error: bad format!\n");
    }
#endif

    /* analyze and determine best format to quantize */
    if (tex && format == GR_TEXFMT_ARGB_8888) {
      int i;
      int alphabits = 0;
      int fullalpha = 0;
      boolean intensity = 1;

      if (!(_options & LET_TEXARTISTS_FLY)) {
        /* HACK ALERT! */
        /* Account for Rice's weirdness with fmt:0 siz:2 textures.
         * Although the conditions are relaxed with other formats,
         * the D3D RGBA5551 surface is used for this format in certain
         * cases. See Nintemod's SuperMario64 life gauge and power
         * meter. The same goes for fmt:2 textures. See Mollymutt's
         * PaperMario text. */
        if ((fmt == 0 && siz == 2) || fmt == 2) {
          DBG_INFO(80, L"Remove black, white, etc borders along the alpha edges.\n");
          /* round A comp */
          for (i = 0; i < height * width; i++) {
            uint32 texel = ((uint32*)tex)[i];
            ((uint32*)tex)[i] = ((texel & 0xff000000) == 0xff000000 ? 0xff000000 : 0) |
                                (texel & 0x00ffffff);
          }
          /* Substitute texel color with the average of the surrounding
           * opaque texels. This removes borders regardless of hardware
           * texture filtering (bilinear, etc). */
          int j;
          for (i = 0; i < height; i++) {
            for (j = 0; j < width; j++) {
              uint32 texel = ((uint32*)tex)[i * width + j];
              if ((texel & 0xff000000) != 0xff000000) {
                uint32 tmptexel[8];
                uint32 k, numtexel, r, g, b;
                numtexel = r = g = b = 0;
                memset(&tmptexel, 0, sizeof(tmptexel));
                if (i > 0) {
                  /* north */
                  tmptexel[0] = ((uint32*)tex)[(i - 1) * width + j];
                  /* north-west */
                  if (j > 0)
                    tmptexel[1] = ((uint32*)tex)[(i - 1) * width + j - 1];
                  /* north-east */
                  if (j < width - 1)
                    tmptexel[2] = ((uint32*)tex)[(i - 1) * width + j + 1];
                }
                if (i < height - 1) {
                  /* south */
                  tmptexel[3] = ((uint32*)tex)[(i + 1) * width + j];
                  /* south-west */
                  if (j > 0)
                    tmptexel[4] = ((uint32*)tex)[(i + 1) * width + j - 1];
                  /* south-east */
                  if (j < width - 1)
                    tmptexel[5] = ((uint32*)tex)[(i + 1) * width + j + 1];
                }
                if (j > 0) {
                  /* west */
                  tmptexel[6] = ((uint32*)tex)[i * width + j - 1];
                }
                if (j < width - 1) {
                  /* east */
                  tmptexel[7] = ((uint32*)tex)[i * width + j + 1];
                }
                for (k = 0; k < 8; k++) {
                  if ((tmptexel[k] & 0xff000000) == 0xff000000) {
                    r += ((tmptexel[k] & 0x00ff0000) >> 16);
                    g += ((tmptexel[k] & 0x0000ff00) >>  8);
                    b += ((tmptexel[k] & 0x000000ff)      );
                    numtexel++;
                  }
                }
                if (numtexel) {
                  ((uint32*)tex)[i * width + j] = ((r / numtexel) << 16) |
                                                  ((g / numtexel) <<  8) |
                                                  ((b / numtexel)      );
                } else {
                  ((uint32*)tex)[i * width + j] = texel & 0x00ffffff;
                }
              }
            }
          }
        }
      }

      /* simple analysis of texture */
      for (i = 0; i < height * width; i++) {
        uint32 texel = ((uint32*)tex)[i];
        if (alphabits != 8) {
#if AGGRESSIVE_QUANTIZATION
          if ((texel & 0xff000000) < 0x00000003) {
            alphabits = 1;
            fullalpha++;
          } else if ((texel & 0xff000000) < 0xfe000000) {
            alphabits = 8;
          }
#else
          if ((texel & 0xff000000) == 0x00000000) {
            alphabits = 1;
            fullalpha++;
          } else if ((texel & 0xff000000) != 0xff000000) {
            alphabits = 8;
          }
#endif
        }
        if (intensity) {
          int rcomp = (texel >> 16) & 0xff;
          int gcomp = (texel >>  8) & 0xff;
          int bcomp = (texel      ) & 0xff;
#if AGGRESSIVE_QUANTIZATION
          if (abs(rcomp - gcomp) > 8 || abs(rcomp - bcomp) > 8 || abs(gcomp - bcomp) > 8) intensity = 0;
#else
          if (rcomp != gcomp || rcomp != bcomp || gcomp != bcomp) intensity = 0;
#endif
        }
        if (!intensity && alphabits == 8) break;
      }
      DBG_INFO(80, L"required alpha bits:%d zero acomp texels:%d rgb as intensity:%d\n", alphabits, fullalpha, intensity);

      /* preparations based on above analysis */
      if (intensity) {
        switch (alphabits) {
        case 0:
          if (fmt == 4) {
            /* N64-I8 */
            destformat = GR_TEXFMT_ALPHA_8;
            for (i = 0; i < height * width; i++) {
              uint32 icomp = ((uint32*)tex)[i] & 0x00ffffff;
              ((uint32*)tex)[i] = (icomp << 24) | icomp;
            }
          } else {
            destformat = GR_TEXFMT_INTENSITY_8;
          }
          break;
        case 1:
          destformat = GR_TEXFMT_ALPHA_INTENSITY_88;
#if (AGGRESSIVE_QUANTIZATION == 2)
          /* NOTE: in rare cases where the designer creates a #04#xx_all.xxx 
           * texture that has all zero alpha texels, we handle it as an IA format.
           */
          if (fmt == 4 &&
              fullalpha < width * height /* XXX: check for problematic all zero intensity textures. */
              ) {
            /* good intensity texture. prehaps the condition is too aggressive... */

            /* N64-I8 */
            destformat = GR_TEXFMT_ALPHA_8;
            for (i = 0; i < height * width; i++) {
              if (((uint32*)tex)[i] & 0xff000000) {
                uint32 icomp = ((uint32*)tex)[i] & 0x00ffffff;
                ((uint32*)tex)[i] = (icomp << 24) | icomp;
              }
              else {
                ((uint32*)tex)[i] = 0;
              }
            }
          }
#else
          if (fmt == 4) {
            /* N64-I8 */
            for (i = 0; i < height * width; i++) {
              if (((uint32*)tex)[i] & 0xff000000) {
                uint32 icomp = ((uint32*)tex)[i] & 0x00ffffff;
                ((uint32*)tex)[i] = (icomp << 24) | icomp;
              }
            }
          }
#endif
          break;
        default:
          destformat = GR_TEXFMT_ALPHA_INTENSITY_88;
          if (fmt == 4) {
            /* N64-I8 */
            for (i = 0; i < height * width; i++) {
              uint32 texel = ((uint32*)tex)[i];
              if (texel & 0xff000000) {
                uint32 icomp = ((texel >> 24) * (texel & 0xff)) / 255;
                ((uint32*)tex)[i] = (icomp << 24) | (texel & 0x00ffffff);
              }
            }
          }
        }
      } else {
        switch (alphabits) {
        case 0:
          if (fmt == 4) {
            /* N64-I8 */
#if (AGGRESSIVE_QUANTIZATION == 2)
            destformat = GR_TEXFMT_ALPHA_8;
#else
            destformat = GR_TEXFMT_ARGB_8888;
#endif
            for (i = 0; i < height * width; i++) {
              uint32 texel = ((uint32*)tex)[i];
              uint32 icomp = (((texel >> 16) & 0xFF) * 299 +
                              ((texel >>  8) & 0xFF) * 587 +
                              ((texel      ) & 0xFF) * 114) / 1000;
              ((uint32*)tex)[i] = (icomp << 24) | (texel & 0x00ffffff);
            }
          } else {
            destformat = GR_TEXFMT_RGB_565;
          }
          break;
        case 1:
          if (fmt == 4) {
            /* N64-I8 */
#if (AGGRESSIVE_QUANTIZATION == 2)
            destformat = GR_TEXFMT_ALPHA_INTENSITY_88;
#else
            destformat = GR_TEXFMT_ARGB_8888;
#endif
            for (i = 0; i < height * width; i++) {
              uint32 texel = ((uint32*)tex)[i];
              if (texel & 0xff000000) {
                uint32 icomp = (((texel >> 16) & 0xFF) * 299 +
                                ((texel >>  8) & 0xFF) * 587 +
                                ((texel      ) & 0xFF) * 114) / 1000;
                ((uint32*)tex)[i] = (icomp << 24) | (texel & 0x00ffffff);
              }
            }
          } else {
            destformat = GR_TEXFMT_ARGB_1555;
          }
          break;
        default:
          destformat = GR_TEXFMT_ARGB_8888;
          if (fmt == 4) {
            /* N64-I8 */
#if (AGGRESSIVE_QUANTIZATION == 2)
            destformat = GR_TEXFMT_ALPHA_INTENSITY_88;
#endif
            for (i = 0; i < height * width; i++) {
              uint32 texel = ((uint32*)tex)[i];
              if (texel & 0xff000000) {
                uint32 icomp = ((((texel >> 16) & 0xFF) * 299 +
                                 ((texel >>  8) & 0xFF) * 587 +
                                 ((texel      ) & 0xFF) * 114) / 1000) * (texel >> 24) / 255;
                ((uint32*)tex)[i] = (icomp << 24) | (texel & 0x00ffffff);
              }
            }
          }
        }
      }
      DBG_INFO(80, L"best gfmt:%x\n", destformat);
    }
    /*
     * Rice hi-res textures: end */

#if TEXTURE_TILING

    /* Glide64 style texture tiling */
    /* NOTE: narrow wide textures can be tiled into 256x256 size textures */

    /* adjust texture size to allow tiling for V1, Rush, V2, Banshee, V3 */
    /* NOTE: we skip this for palette textures that need minification
     * becasue it will look ugly. */

    /* minification */
    if (tex && format == GR_TEXFMT_ARGB_8888) { /* XXX: only ARGB8888. comeback to this!! */
      int ratio = 1;
      int ratio_tile = 1;

      /* normal minification to fit max texture size */
      if (width > _maxwidth || height > _maxheight) {
        DBG_INFO(80, L"determine minification ratio to fit max texture size\n");
        tmpwidth = width;
        tmpheight = height;
        while (tmpwidth > _maxwidth) {
          tmpheight >>= 1;
          tmpwidth >>= 1;
          ratio <<= 1;
        }
        while (tmpheight > _maxheight) {
          tmpheight >>= 1;
          tmpwidth >>= 1;
          ratio <<= 1;
        }
        DBG_INFO(80, L"minification ratio:%d %d x %d -> %d x %d\n",
                 ratio, width, height, tmpwidth, tmpheight);
      }

      /* minification to enable glide64 style texture tiling */
      /* determine the minification ratio to tile the texture into 256x256 size */
      if ((_options & TILE_HIRESTEX) && _maxwidth >= 256 && _maxheight >= 256) {
        DBG_INFO(80, L"determine minification ratio to tile\n");
        tmpwidth = width;
        tmpheight = height;
        if (height > 256) {
          ratio_tile = ((height - 1) >> 8) + 1;
          tmpwidth = width / ratio_tile;
          tmpheight = height / ratio_tile;
          DBG_INFO(80, L"height > 256, minification ratio:%d %d x %d -> %d x %d\n",
                   ratio_tile, width, height, tmpwidth, tmpheight);
        }
        if (tmpwidth > 256 && (((tmpwidth - 1) >> 8) + 1) * tmpheight > 256) {
          ratio_tile *= ((((((tmpwidth - 1) >> 8) + 1) * tmpheight) - 1) >> 8) + 1;
#ifdef DEBUG
          tmpwidth = width / ratio_tile;
          tmpheight = height / ratio_tile;
          DBG_INFO(80, L"width > 256, minification ratio:%d %d x %d -> %d x %d\n",
                   ratio_tile, width, height, tmpwidth, tmpheight);
#endif
        }
        if (ratio_tile < ratio) ratio = ratio_tile;
      }

      if (ratio > 1) {
        if (!_txReSample->minify(&tex, &width, &height, ratio)) {
          if (tex) {
            free(tex);
            tex = NULL;
          }
        }
      }
    }

    /* tiling */
    if (tex && (_options & TILE_HIRESTEX) && _maxwidth >= 256 && _maxheight >= 256) {
      boolean usetile = 0;

      /* to tile or not to tile, that is the question */
      if (width > 256 && height <= 128 && (((width - 1) >> 8) + 1) * height <= 256) {

        if (width > _maxwidth) usetile = 1;
        else {
          /* tile if the tiled texture memory footprint is smaller */
          int tilewidth  = 256;
          int tileheight = _txReSample->nextPow2((((width - 1) >> 8) + 1) * height);
          tmpwidth  = width;
          tmpheight = height;

          /* 3dfx Glide3 tmpheight, W:H aspect ratio range (8:1 - 1:8) */
          if (tilewidth > (tileheight << 3)) tileheight = tilewidth >> 3;

          /* HACKALERT: see TxReSample::pow2(); */
          if (tmpwidth > 64)       tmpwidth -= 4;
          else if (tmpwidth > 16)  tmpwidth -= 2;
          else if (tmpwidth > 4)   tmpwidth -= 1;

          if (tmpheight > 64)      tmpheight -= 4;
          else if (tmpheight > 16) tmpheight -= 2;
          else if (tmpheight > 4)  tmpheight -= 1;

          tmpwidth  = _txReSample->nextPow2(tmpwidth);
          tmpheight = _txReSample->nextPow2(tmpheight);

          /* 3dfx Glide3 tmpheight, W:H aspect ratio range (8:1 - 1:8) */
          if (tmpwidth > tmpheight) {
            if (tmpwidth > (tmpheight << 3)) tmpheight = tmpwidth >> 3;
          } else {
            if (tmpheight > (tmpwidth << 3)) tmpwidth = tmpheight >> 3;
          }

          usetile = (tilewidth * tileheight < tmpwidth * tmpheight);
        }

      }

      /* tile it! do the actual tiling into 256x256 size */
      if (usetile) {
        DBG_INFO(80, L"Glide64 style texture tiling\n");

        int x, y, z, ratio, offset;
        offset = 0;
        ratio = ((width - 1) >> 8) + 1;
        tmptex = (uint8 *)malloc(_txUtil->sizeofTx(256, height * ratio, format));
        if (tmptex) {
          switch (format) {
          case GR_TEXFMT_ARGB_8888:
            for (x = 0; x < ratio; x++) {
              for (y = 0; y < height; y++) {
                if (x < ratio - 1) {
                  memcpy(&tmptex[offset << 2], &tex[(x * 256 + y * width) << 2], 256 << 2);
                } else {
                  for (z = 0; z < width - 256 * (ratio - 1); z++) {
                    ((uint32*)tmptex)[offset + z] = ((uint32*)tex)[x * 256 + y * width + z];
                  }
                  for (; z < 256; z++) {
                    ((uint32*)tmptex)[offset + z] = ((uint32*)tmptex)[offset + z - 1];
                  }
                }
                offset += 256;
              }
            }
            free(tex);
            tex = tmptex;
            untiled_width = width;
            untiled_height = height;
            width = 256;
            height *= ratio;
            DBG_INFO(80, L"Tiled: %d x %d -> %d x %d\n", untiled_width, untiled_height, width, height);
            break;
          case GR_TEXFMT_P_8:
            for (x = 0; x < ratio; x++) {
              for (y = 0; y < height; y++) {
                if (x < ratio - 1) {
                  memcpy(&tmptex[offset], &tex[x * 256 + y * width], 256);
                } else {
                  for (z = 0; z < width - 256 * (ratio - 1); z++) {
                    tmptex[offset + z] = tex[x * 256 + y * width + z];
                  }
                  for (; z < 256; z++) {
                    tmptex[offset + z] = tmptex[offset + z - 1];
                  }
                }
                offset += 256;
              }
            }
            free(tex);
            tex = tmptex;
            untiled_width = width;
            untiled_height = height;
            width = 256;
            height *= ratio;
            DBG_INFO(80, L"Tiled: %d x %d -> %d x %d\n", untiled_width, untiled_height, width, height);
            break;
          default:
            free(tmptex);
          }
        }
      }
    }

    /* expand to 3dfx Glide3x aspect ratio (8:1 - 1:8) */
    if (tex) {
      int bpp = 0;
      switch (format) {
      case GR_TEXFMT_ARGB_8888:
        bpp = 32;
        break;
      case GR_TEXFMT_P_8:
        bpp = 8;
      }
      if (!_txReSample->nextPow2(&tex, &width , &height, bpp, 1)) {
        if (tex) {
          free(tex);
          tex = NULL;
        }
      }
    }
#else  /* TEXTURE_TILING */

#if POW2_TEXTURES
    if (tex) {
      int bpp = 0;
      switch (format) {
      case GR_TEXFMT_ARGB_8888:
        bpp = 32;
        break;
      case GR_TEXFMT_P_8:
        bpp = 8;
      }
#if (POW2_TEXTURES == 2)
      /* 3dfx Glide3x aspect ratio (8:1 - 1:8) */
      if (!_txImage->nextPow2(&tex, &width , &height, bpp, 1)) {
#else
      /* normal pow2 expansion */
      if (!_txImage->nextPow2(&tex, &width , &height, bpp, 0)) {
#endif
        if (tex) {
          free(tex);
          tex = NULL;
        }
      }
    }
#endif

    /* minification */
    if (tex &&
        (width > _maxwidth || height > _maxheight) &&
        format == GR_TEXFMT_ARGB_8888) {
      int ratio = 1;
      if (width / _maxwidth > height / _maxheight) {
        ratio = (int)((double)width / _maxwidth + 0.5);
      } else {
        ratio = (int)((double)height / _maxheight + 0.5);
      }
      if (!_txReSample->minify(&tex, &width, &height, ratio)) {
        if (tex) {
          free(tex);
          tex = NULL;
        }
      }
    }

#endif /* TEXTURE_TILING */

    /* texture compression */
    if (tex && format == GR_TEXFMT_ARGB_8888 &&
        (_options & COMPRESSION_MASK)) {
      uint32 alpha = 0;
      int dataSize = 0;
      int compressionType = _options & COMPRESSION_MASK;

      switch (_options & COMPRESSION_MASK) {
      case S3TC_COMPRESSION:
        switch (destformat) {
        case GR_TEXFMT_ARGB_8888:
        case GR_TEXFMT_ARGB_1555: /* for ARGB1555 use DXT5 instead of DXT1 */
        case GR_TEXFMT_ALPHA_INTENSITY_88:
          dataSize = width * height;
          break;
        case GR_TEXFMT_RGB_565:
        case GR_TEXFMT_INTENSITY_8:
          dataSize = (width * height) >> 1;
          break;
        case GR_TEXFMT_ALPHA_8: /* no size benefit with dxtn */
          ;
        }
        break;
      case FXT1_COMPRESSION:
        switch (destformat) {
        case GR_TEXFMT_ARGB_8888:
        case GR_TEXFMT_ARGB_1555:
        case GR_TEXFMT_RGB_565:
        case GR_TEXFMT_INTENSITY_8:
        case GR_TEXFMT_ALPHA_8:
          dataSize = (width * height) >> 1;
          break;
        case GR_TEXFMT_ALPHA_INTENSITY_88: /* looks bad for textures that use the alpha channel as a mask */
          /* XXX: afaik all gfx cards that support fxt1 also support dxtn. (3dfx and Intel) */
          compressionType = S3TC_COMPRESSION;
          dataSize = width * height;
        }
      }
      /* compress it! */
      if (dataSize) {
        tmptex = (uint8 *)malloc(dataSize);
        if (tmptex) {
          if (_txQuantize->compress(tex, tmptex,
                                    width, height, destformat,
                                    &tmpwidth, &tmpheight, &tmpformat,
                                    compressionType)) {
            free(tex);
            tex = tmptex;
            width = tmpwidth;
            height = tmpheight;
            format = destformat = tmpformat;
          } else {
            free(tmptex);
          }
        }
      }
    }

    /* quantize */
    if (tex && format == GR_TEXFMT_ARGB_8888) {
      tmptex = (uint8 *)malloc(_txUtil->sizeofTx(width, height, destformat));
      if (tmptex) {
        switch (destformat) {
        case GR_TEXFMT_ARGB_8888:
        case GR_TEXFMT_ARGB_4444:
#if !REDUCE_TEXTURE_FOOTPRINT
          if (_maxbpp < 32 || _options & FORCE16BPP_HIRESTEX) {
#endif
            _txQuantize->ARGB8888_ARGB4444_ErrD((uint32*)tex, (uint32*)tmptex, width, height, 0);
            format = GR_TEXFMT_ARGB_4444;
#if !REDUCE_TEXTURE_FOOTPRINT
          } else {
            free(tmptex);
            tmptex = NULL;
          }
#endif
          break;
        case GR_TEXFMT_ARGB_1555:
#if !REDUCE_TEXTURE_FOOTPRINT
          if (_maxbpp < 32 || _options & FORCE16BPP_HIRESTEX) {
#endif
            _txQuantize->ARGB8888_ARGB1555_ErrD((uint32*)tex, (uint32*)tmptex, width, height);
            format = GR_TEXFMT_ARGB_1555;
#if !REDUCE_TEXTURE_FOOTPRINT
          } else {
            free(tmptex);
            tmptex = NULL;
          }
#endif
          break;
        case GR_TEXFMT_RGB_565:
#if !REDUCE_TEXTURE_FOOTPRINT
          if (_maxbpp < 32 || _options & FORCE16BPP_HIRESTEX) {
#endif
            _txQuantize->ARGB8888_RGB565_ErrD((uint32*)tex, (uint32*)tmptex, width, height);
            format = GR_TEXFMT_RGB_565;
#if !REDUCE_TEXTURE_FOOTPRINT
          } else {
            free(tmptex);
            tmptex = NULL;
          }
#endif
          break;
        case GR_TEXFMT_ALPHA_INTENSITY_88:
        case GR_TEXFMT_ALPHA_INTENSITY_44:
#if !REDUCE_TEXTURE_FOOTPRINT
          _txQuantize->ARGB8888_AI44_ErrD((uint32*)tex, (uint32*)tmptex, width, height, 0);
          format = GR_TEXFMT_ALPHA_INTENSITY_44;
#else
          _txQuantize->ARGB8888_AI88_Slow((uint32*)tex, (uint32*)tmptex, width, height);
          format = GR_TEXFMT_ALPHA_INTENSITY_88;
#endif
          break;
        case GR_TEXFMT_ALPHA_8:
          _txQuantize->ARGB8888_I8_Slow((uint32*)tex, (uint32*)tmptex, width, height);
          format = GR_TEXFMT_ALPHA_8; /* yes, this is correct. ALPHA_8 instead of INTENSITY_8 */
          break;
        case GR_TEXFMT_INTENSITY_8:
          _txQuantize->ARGB8888_I8_Slow((uint32*)tex, (uint32*)tmptex, width, height);
          format = GR_TEXFMT_INTENSITY_8;
        }
        if (tmptex) {
          free(tex);
          tex = tmptex;
        }
      }
    }

    /* last minute validations */
    if (!chksum || !width || !height || !format || width > _maxwidth || height > _maxheight) {
      if (tex) {
        free(tex);
        tex = NULL;
      }
      DBG_INFO(80, L"Error: bad format or size. crc:%08X, %d x %d gfmt:%x\n", chksum, width, height, format);
    }

    /* load it into hires texture cache. */
    if (tex) {
      uint64 chksum64 = (uint64)palchksum;
      chksum64 <<= 32;
      chksum64 |= (uint64)chksum;

      GHQTexInfo tmpInfo;
      memset(&tmpInfo, 0, sizeof(GHQTexInfo));

      tmpInfo.data = tex;
      tmpInfo.width = width;
      tmpInfo.height = height;
      tmpInfo.format = format;
      tmpInfo.largeLodLog2 = _txUtil->grLodLog2(width, height);
      tmpInfo.smallLodLog2 = tmpInfo.largeLodLog2;
      tmpInfo.aspectRatioLog2 = _txUtil->grAspectRatioLog2(width, height);

#if TEXTURE_TILING
      /* Glide64 style texture tiling. */
      if (untiled_width && untiled_height) {
        tmpInfo.tiles = ((untiled_width - 1) >> 8) + 1;
        tmpInfo.untiled_width = untiled_width;
        tmpInfo.untiled_height = untiled_height;
      }
#endif

      if (TxCache::add(chksum64, &tmpInfo)) {
        /* Callback to display hires texture info.
         * Gonetz <gonetz(at)ngs.ru> */
        if (_callback) {
          wchar_t tmpbuf[MAX_PATH];
          mbstowcs(tmpbuf, fname, MAX_PATH);
          (*_callback)(L"[%d] total mem:%.2fmb - %s\n", _cache.size(), (float)_totalSize/1000000, tmpbuf);
        }
      }
      free(tex);
    }

  }

  CHDIR(curpath);

  return 1;
}
