/*
 * Texture Filtering
 * Version:  1.0
 *
 * Copyright (C) 2007  Hiroshi Morii   All Rights Reserved.
 * Email koolsmoky(at)users.sourceforge.net
 * Web   http://www.3dfxzone.it/koolsmoky
 *
 * this is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * this is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Make; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef __MSC__
#pragma warning(disable: 4786)
#endif

#include "TxFilter.h"
#include "TextureFilters.h"
#include "TxDbg.h"
#include "bldno.h"

TxFilter::~TxFilter()
{
  /* clear hires texture cache */
  delete _txHiResCache;

  /* clear texture cache */
  delete _txTexCache;

  /* free memory */
  TxMemBuf::getInstance()->shutdown();

  /* clear other stuff */
  delete _txQuantize;
  delete _txUtil;
}

TxFilter::TxFilter(int maxwidth, int maxheight, int maxbpp, int options,
                   int cachesize, wchar_t *path, wchar_t *ident,
                   dispInfoFuncExt callback)
{
  /* shamelessness :P this first call to the debug output message creates
   * a file in the executable directory. */
  DBG_INFO(0, L"------------------------------------------------------------------\n");
  DBG_INFO(0, L" GlideHQ 1.00.00.%d Copyright 2007\n", BUILD_NUMBER);
  DBG_INFO(0, L" Hiroshi Morii aka KoolSmoky <koolsmoky(at)users.sourceforge.net>\n");
  DBG_INFO(0, L"------------------------------------------------------------------\n");

  _txQuantize   = new TxQuantize();
  _txUtil       = new TxUtil();

  _initialized = 0;

  _tex1 = NULL;
  _tex2 = NULL;

  _maxwidth  = maxwidth;
  _maxheight = maxheight;
  _maxbpp    = maxbpp;
  _options   = options;
  _cacheSize = cachesize;

  /* TODO: validate options and do overrides here*/

  /* save path name */
  if (path)
    _path.assign(path);

  /* save ROM name */
  if (ident && wcscmp(ident, L"DEFAULT") != 0)
    _ident.assign(ident);

  /* check for dxtn extensions */
  if (!TxLoadLib::getInstance()->getdxtCompressTexFuncExt() &&
      (options & COMPRESSION_MASK) == S3TC_COMPRESSION)
    _options &= ~COMPRESSION_MASK;

  if (!TxLoadLib::getInstance()->getfxtCompressTexFuncExt() &&
      (options & COMPRESSION_MASK) == FXT1_COMPRESSION)
    _options &= ~COMPRESSION_MASK;

  switch (options & COMPRESSION_MASK) {
  case FXT1_COMPRESSION:
  case S3TC_COMPRESSION:
    break;
  case NCC_COMPRESSION:
  default:
    _options &= ~COMPRESSION_MASK;
  }

  if (TxMemBuf::getInstance()->init(maxwidth, maxheight)) {
    if (!_tex1)
      _tex1 = TxMemBuf::getInstance()->get(0);

    if (!_tex2)
      _tex2 = TxMemBuf::getInstance()->get(1);
  }

#if !_16BPP_HACK
  /* initialize hq4x filter */
  hq4x_init();
#endif

  /* initialize texture cache in bytes. 128Mb will do nicely in most cases */
  _txTexCache = new TxTexCache(_options, _cacheSize, _path.c_str(), _ident.c_str(), callback);

  /* hires texture */
#if HIRES_TEXTURE
  _txHiResCache = new TxHiResCache(_maxwidth, _maxheight, _maxbpp, _options, _path.c_str(), _ident.c_str(), callback);

  if (_txHiResCache->empty())
    _options &= ~HIRESTEXTURES_MASK;
#endif

  /* assert local options */
  if (!(_options & COMPRESS_TEX))
    _options &= ~COMPRESSION_MASK;

  /* limit texture size */
  /*if (_maxwidth  > 512) _maxwidth  = 512;
  if (_maxheight > 512) _maxheight = 512;*/

  if (_tex1 && _tex2)
      _initialized = 1;
}

boolean
TxFilter::filter(uint8 *src, int srcwidth, int srcheight, uint16 srcformat, uint64 g64crc, GHQTexInfo *info)
{
  uint8 *texture = src;
  uint8 *tmptex = _tex1;
  uint16 destformat = srcformat;

  /* We need to be initialized first! */
  /* bypass _options to do ARGB8888->ARGB4444 if _maxbpp=16 */
  /* ignore if we only have texture compression option on. */
  if (!_initialized ||
      !((_options & (FILTER_MASK|ENHANCEMENT_MASK/*|COMPRESSION_MASK*/)) ||
        (srcformat == GR_TEXFMT_ARGB_8888 && (_maxbpp < 32 || _options & FORCE16BPP_TEX))))
    return 0;

  /* Leave small textures alone because filtering makes little difference.
   * Moreover, some filters require at least 4 * 4 to work.
   */
  if (srcwidth >= 4 && srcheight >= 4) {

    /* check if width * height >= 4
     * TexConv requirement.
     */
    /*if ((srcwidth * srcheight) < 4)
      return NULL;*//* intentionally skipped! we get here only if have 4x4 or larger textures */

    /* find cached textures */
    if (_cacheSize) {

      /* calculate checksum of source texture */
      if (!g64crc)
        g64crc = (uint64)(_txUtil->checksumTx(texture, srcwidth, srcheight, srcformat));

      DBG_INFO(80, L"filter: crc:%08X %08X %d x %d gfmt:%x\n",
               (uint32)(g64crc >> 32), (uint32)(g64crc & 0xffffffff), srcwidth, srcheight, srcformat);

#if 0 /* use hirestex to retrieve cached textures. */
      /* check if we have it in cache */
      if (!(g64crc & 0xffffffff00000000) && /* we reach here only when there is no hires texture for this crc */
          _txTexCache->get(g64crc, info)) {
        DBG_INFO(80, L"cache hit: %d x %d gfmt:%x\n", info->width, info->height, info->format);
        return 1; /* yep, we've got it */
      }
#endif
    }

#if !_16BPP_HACK
    /* convert textures to a format that the compressor accepts (ARGB8888) */
    if (_options & COMPRESSION_MASK) {
#endif
      if (srcformat != GR_TEXFMT_ARGB_8888) {
        if (!_txQuantize->quantize(texture, tmptex, srcwidth, srcheight, srcformat, GR_TEXFMT_ARGB_8888)) {
          DBG_INFO(80, L"Error: unsupported format! gfmt:%x\n", srcformat);
          return 0;
        }
        texture = tmptex;
        destformat = GR_TEXFMT_ARGB_8888;
      }
#if !_16BPP_HACK
    }
#endif

    switch (destformat) {
    case GR_TEXFMT_ARGB_8888:

      /*
       * texture enhancements (x2, x4 scalers)
       */
      int scale_shift = 0;
      tmptex = (texture == _tex1) ? _tex2 : _tex1;
      
      switch (_options & ENHANCEMENT_MASK) {
      case HQ4X_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 2) && srcheight <= (_maxheight >> 2)) {
          hq4x_8888((uint8*)texture, (uint8*)tmptex, srcwidth, srcheight, srcwidth, (srcwidth << 4));
          scale_shift = 2;
        } else if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          hq2x_32((uint8*)texture, (srcwidth << 2), (uint8*)tmptex, (srcwidth << 3), srcwidth, srcheight);
          scale_shift = 1;
        }
        break;
      case HQ2X_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          hq2x_32((uint8*)texture, (srcwidth << 2), (uint8*)tmptex, (srcwidth << 3), srcwidth, srcheight);
          scale_shift = 1;
        }
        break;
      case LQ2X_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          lq2x_32((uint8*)texture, (srcwidth << 2), (uint8*)tmptex, (srcwidth << 3), srcwidth, srcheight);
          scale_shift = 1;
        }
        break;
      case X2SAI_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          Super2xSaI_8888((uint32*)texture, (uint32*)tmptex, srcwidth, srcheight, srcwidth);
          scale_shift = 1;
        }
        break;
      case X2_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          Texture2x_32((uint8*)texture, (srcwidth << 2), (uint8*)tmptex, (srcwidth << 3), srcwidth, srcheight);
          scale_shift = 1;
        }
      }
      if (scale_shift) {
        srcwidth <<= scale_shift;
        srcheight <<= scale_shift;
        texture = tmptex;
      }


      /*
       * texture filtering
       */
      if (_options & SMOOTH_FILTER_MASK) {
        tmptex = (texture == _tex1) ? _tex2 : _tex1;
        SmoothFilter_8888((uint32*)texture, srcwidth, srcheight, (uint32*)tmptex, (_options & SMOOTH_FILTER_MASK));
        texture = tmptex;
      } else if (_options & SHARP_FILTER_MASK) {
        tmptex = (texture == _tex1) ? _tex2 : _tex1;
        SharpFilter_8888((uint32*)texture, srcwidth, srcheight, (uint32*)tmptex, (_options & SHARP_FILTER_MASK));
        texture = tmptex;
      }


      /*
       * texture compression
       */
      /* ignore if we only have texture compression option on. */
      if (_options & (FILTER_MASK|ENHANCEMENT_MASK) &&
          _options & COMPRESSION_MASK) {
        int tmpwidth, tmpheight;
        uint16 tmpformat;
        
        tmptex = (texture == _tex1) ? _tex2 : _tex1;
        if (_txQuantize->compress(texture, tmptex,
                                  srcwidth, srcheight, srcformat,
                                  &tmpwidth, &tmpheight, &tmpformat,
                                  _options & COMPRESSION_MASK)) {
          srcwidth = tmpwidth;
          srcheight = tmpheight;
          destformat = tmpformat;
          texture = tmptex;
        }
      }


      /*
       * texture (re)conversions
       */
      if (destformat == GR_TEXFMT_ARGB_8888) {
        if (srcformat == GR_TEXFMT_ARGB_8888 && (_maxbpp < 32 || _options & FORCE16BPP_TEX)) srcformat = GR_TEXFMT_ARGB_4444;
        if (srcformat != GR_TEXFMT_ARGB_8888) {
          tmptex = (texture == _tex1) ? _tex2 : _tex1;
          if (!_txQuantize->quantize(texture, tmptex, srcwidth, srcheight, GR_TEXFMT_ARGB_8888, srcformat)) {
            DBG_INFO(80, L"Error: unsupported format! gfmt:%x\n", srcformat);
            return 0;
          }
          texture = tmptex;
          destformat = srcformat;
        }
      }

      break;
#if !_16BPP_HACK
    case GR_TEXFMT_ARGB_4444:

      int scale_shift = 0;
      tmptex = (texture == _tex1) ? _tex2 : _tex1;

      switch (_options & ENHANCEMENT_MASK) {
      case HQ4X_ENHANCEMENT:
        if (srcwidth <= (_maxwidth >> 2) && srcheight <= (_maxheight >> 2)) {
          hq4x_4444((uint8*)texture, (uint8*)tmptex, srcwidth, srcheight, srcwidth, srcwidth * 4 * 2);
          scale_shift = 2;
        }/* else if (srcwidth <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          hq2x_16((uint8*)texture, srcwidth * 2, (uint8*)tmptex, srcwidth * 2 * 2, srcwidth, srcheight);
          scale_shift = 1;
        }*/
        break;
      case HQ2X_ENHANCEMENT:
        if (srcwidth <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          hq2x_16((uint8*)texture, srcwidth * 2, (uint8*)tmptex, srcwidth * 2 * 2, srcwidth, srcheight);
          scale_shift = 1;
        }
        break;
      case LQ2X_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          lq2x_16((uint8*)texture, srcwidth * 2, (uint8*)tmptex, srcwidth * 2 * 2, srcwidth, srcheight);
          scale_shift = 1;
        }
        break;
      case X2SAI_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          Super2xSaI_4444((uint16*)texture, (uint16*)tmptex, srcwidth, srcheight, srcwidth);
          scale_shift = 1;
        }
        break;
      case X2_ENHANCEMENT:
        if (srcwidth  <= (_maxwidth >> 1) && srcheight <= (_maxheight >> 1)) {
          Texture2x_16((uint8*)texture, srcwidth * 2, (uint8*)tmptex, srcwidth * 2 * 2, srcwidth, srcheight);
          scale_shift = 1;
        }
      }
      if (scale_shift) {
        srcwidth <<= scale_shift;
        srcheight <<= scale_shift;
        texture = tmptex;
      }

      if (_options & SMOOTH_FILTER_MASK) {
        tmptex = (texture == _tex1) ? _tex2 : _tex1;
        SmoothFilter_4444((uint16*)texture, srcwidth, srcheight, (uint16*)tmptex, (_options & SMOOTH_FILTER_MASK));
        texture = tmptex;
      } else if (_options & SHARP_FILTER_MASK) {
        tmptex = (texture == _tex1) ? _tex2 : _tex1;
        SharpFilter_4444((uint16*)texture, srcwidth, srcheight, (uint16*)tmptex, (_options & SHARP_FILTER_MASK));
        texture = tmptex;
      }

      break;
    case GR_TEXFMT_ARGB_1555:
      break;
    case GR_TEXFMT_RGB_565:
      break;
    case GR_TEXFMT_ALPHA_8:
      break;
#endif /* _16BPP_HACK */
    }
  }

  /* if texture == src, then nothing was done. */
  if (texture != src) {

    /* cache the filtered texture. */
    info->data = texture;
    info->width  = srcwidth;
    info->height = srcheight;
    info->format = destformat;
    info->smallLodLog2 = _txUtil->grLodLog2(srcwidth, srcheight);
    info->largeLodLog2 = info->smallLodLog2;
    info->aspectRatioLog2 = _txUtil->grAspectRatioLog2(srcwidth, srcheight);

    _txTexCache->add(g64crc, info);

    DBG_INFO(80, L"filtered texture: %d x %d gfmt:%x\n", info->width, info->height, info->format);

    return 1;
  }

  return 0;
}

boolean
TxFilter::hirestex(uint64 g64crc, uint64 r_crc64, uint16 *palette, GHQTexInfo *info)
{
  /* NOTE: Rice CRC32 sometimes return the same value for different textures.
   * As a workaround, Glide64 CRC32 is used for the key for NON-hires
   * texture cache.
   *
   * r_crc64 = hi:palette low:texture
   *           (separate crc. doesn't necessary have to be rice crc)
   * g64crc  = texture + palette glide64 crc32
   *           (can be any other crc if robust)
   */

  DBG_INFO(80, L"hirestex: r_crc64:%08X %08X, g64crc:%08X %08X\n",
           (uint32)(r_crc64 >> 32), (uint32)(r_crc64 & 0xffffffff),
           (uint32)(g64crc >> 32), (uint32)(g64crc & 0xffffffff));

  /* check if we have it in memory cache */
  if (_cacheSize && g64crc) {
    if (_txTexCache->get(g64crc, info)) {
      DBG_INFO(80, L"cache hit: %d x %d gfmt:%x\n", info->width, info->height, info->format);
      return 1; /* yep, we've got it */
    }
  }

#if HIRES_TEXTURE
  /* check if we have it in hires memory cache. */
  if ((_options & HIRESTEXTURES_MASK) && r_crc64) {
    if (_txHiResCache->get(r_crc64, info)) {
      DBG_INFO(80, L"hires hit: %d x %d gfmt:%x\n", info->width, info->height, info->format);

      /* TODO: Enable emulation for special N64 combiner modes. There are few ways
       * to get this done. Also applies for CI textures below.
       *
       * Solution 1. Load the hiresolution textures in ARGB8888 (or A8, IA88) format
       * to cache. When a cache is hit, then we take the modes passed in from Glide64
       * (also TODO) and apply the modification. Then we do color reduction or format
       * conversion or compression if desired and stuff it into the non-hires texture
       * cache.
       *
       * Solution 2. When a cache is hit and if the combiner modes are present,
       * convert the texture to ARGB4444 and pass it back to Glide64 to process.
       * If a texture is compressed, it needs to be decompressed first. Then add
       * the processed texture to the non-hires texture cache.
       *
       * Solution 3. Hybrid of the above 2. Load the textures in ARGB8888 (A8, IA88)
       * format. Convert the texture to ARGB4444 and pass it back to Glide64 when
       * the combiner modes are present. Get the processed texture back from Glide64
       * and compress if desired and add it to the non-hires texture cache.
       *
       * Solution 4. Take the easy way out and forget about this whole thing.
       */

      return 1; /* yep, got it */
    }
    /* XXX: perhaps it would be better to move this to TxHiResTexture::get */
    if (palette &&
        _txHiResCache->get((r_crc64 & 0xffffffff), info)) {
      DBG_INFO(80, L"hires hit: %d x %d gfmt:%x\n", info->width, info->height, info->format);

      /* for true CI textures, we use the passed in palette to convert to
       * ARGB1555 and add it to memory cache.
       *
       * NOTE: we do this AFTER all other texture cache searches because
       * only a few texture packs actually use true CI textures.
       *
       * NOTE: the pre-converted palette from Glide64 is in RGBA5551 format.
       * A comp comes before RGB comp.
       */
      if (info->format == GR_TEXFMT_P_8) {
        DBG_INFO(80, L"found GR_TEXFMT_P_8 format. Need conversion!!\n");

        int width = info->width;
        int height = info->height;
        uint16 format = info->format;
        /* XXX: avoid collision with zlib compression buffer in TxHiResTexture::get */
        uint8 *texture = info->data;
        uint8 *tmptex = (texture == _tex1) ? _tex2 : _tex1;

        /* use palette and convert to 16bit format */
        _txQuantize->P8_16BPP((uint32*)texture, (uint32*)tmptex, info->width, info->height, (uint32*)palette);
        texture = tmptex;
        format = GR_TEXFMT_ARGB_1555;

#if 1
        /* XXX: compressed if memory cache compression is ON (not hires cache) */
        if (_options & COMPRESSION_MASK) {
          tmptex = (texture == _tex1) ? _tex2 : _tex1;
          if (_txQuantize->quantize(texture, tmptex, info->width, info->height, format, GR_TEXFMT_ARGB_8888)) {
            texture = tmptex;
            format = GR_TEXFMT_ARGB_8888;
          }
          if (format == GR_TEXFMT_ARGB_8888) {
            tmptex = (texture == _tex1) ? _tex2 : _tex1;
            if (_txQuantize->compress(texture, tmptex,
                                      info->width, info->height, GR_TEXFMT_ARGB_1555,
                                      &width, &height, &format,
                                      _options & COMPRESSION_MASK)) {
              texture = tmptex;
            } else {
              /*if (!_txQuantize->quantize(texture, tmptex, info->width, info->height, GR_TEXFMT_ARGB_8888, GR_TEXFMT_ARGB_1555)) {
                DBG_INFO(80, L"Error: unsupported format! gfmt:%x\n", format);
                return 0;
              }*/
              texture = tmptex;
              format = GR_TEXFMT_ARGB_1555;
            }
          }
        }
#endif

        /* fill in the required info to return */
        info->data = texture;
        info->width = width;
        info->height = height;
        info->format = format;
        info->smallLodLog2 = _txUtil->grLodLog2(width, height);
        info->largeLodLog2 = info->smallLodLog2;
        info->aspectRatioLog2 = _txUtil->grAspectRatioLog2(width, height);

        /* try adding it to memory cache. */
        if (_cacheSize && g64crc) {
          DBG_INFO(80, L"Yes! the user has memory cache ON. Let's use it.\n");

          _txTexCache->add(g64crc, info);
        }

        DBG_INFO(80, L"GR_TEXFMT_P_8 loaded as GR_TEXFMT_ARGB_1555!\n");

        return 1;
      }
    }
  }
#endif

  DBG_INFO(80, L"no cache hits.\n");

  return 0;
}

uint64
TxFilter::checksum64(uint8 *src, int width, int height, int size, int rowStride, uint8 *palette)
{
  if (_options & HIRESTEXTURES_MASK)
    return _txUtil->checksum64(src, width, height, size, rowStride, palette);

  return 0;
}

boolean
TxFilter::dmptx(uint8 *src, int width, int height, uint16 format, int rowStride)
{
  if (!_initialized)
    return 0;

#if 1
  return 0;
#else
  DBG_INFO(80, L"fmt = %04x\n", format);

  uint64 checksum = _txUtil->RiceCRC32(src, width, height, (format & 0x00ff), rowStride);

  DBG_INFO(80, L"dmptx: %08X\n", checksum);

  if (!checksum)
    return 0;

  if (format != 0x0401)
    return 0;

  /*if (!_txQuantize->quantize(src, _tex1, width, height, GR_TEXFMT_ALPHA_8, GR_TEXFMT_ARGB8888))
    return 0;

  src = _tex1;

  format = 0x0003;

  rowStride <<= 2;*/

  //if (format == GR_TEXFMT_ARGB_8888) {
    if (!_path.empty() && !_ident.empty()) {
      /* dump it to disk */
      FILE *fp = NULL;
      wchar_t tmpbuf[MAX_PATH];
      swprintf(tmpbuf, MAX_PTH, L"%s\\texture_dump\\%s\\GlideHQ\\%08X%08X.png",
               _path.c_str(), _ident.c_str(), (uint32)(checksum >> 32), (uint32)(checksum & 0xffffffff));
      if ((fp = _wfopen(tmpbuf, L"wb")) != NULL) {
        txImage.writePNG(src, fp, width, height, format, rowStride);
        fclose(fp);
        return 1;
      }
    }
  //}

  return 0;
#endif
}
