/*
 *  Copyright (C) 2006 Ludovic Jacomme (ludovic.jacomme@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <SDL/SDL.h>

#include "global.h"
#include "psp_sdl.h"
#include "psp_kbd.h"
#include "psp_menu.h"
#include "psp_fmgr.h"
#include "psp_menu_kbd.h"
#include "psp_menu_help.h"
#include "gp2x_psp.h"

extern SDL_Surface *back_surface;
static int psp_menu_dirty = 1;

# define MENU_JOYSTICK     0
# define MENU_CLOCK        1
# define MENU_RENDER       2
# define MENU_SCREENSHOT   3

# define MENU_HELP         4

# define MENU_LOAD_PRG     5
# define MENU_SAVE_CFG     6
# define MENU_KEYBOARD     7

# define MENU_SOFT_RESET   8
# define MENU_HARD_RESET   9
# define MENU_BACK        10
# define MENU_EXIT        11

# define MAX_MENU_ITEM (MENU_EXIT + 1)


  static menu_item_t menu_list[] =
  {
    { "Joystick as cursor :" },
    { "Clock frequency    :" },
    { "Render mode        :" },
    { "Save Screenshot    :" },

    { "Help" },

    { "Load Program" },
    { "Save Config" },

    { "Keyboard" },

    { "Reset TI-92" },
    { "Hard Reset TI-92" },
    { "Back to TI-92" },
    { "Exit" }
  };

  static int cur_menu_id = MENU_LOAD_PRG;

  static int psp_joystick_cursor   = 1;
  static int ti92_render_mode      = TI92_RENDER_X125;
  static int psp_cpu_clock         = 200;

void
string_fill_with_space(char *buffer, int size)
{
  int length = strlen(buffer);
  int index;

  for (index = length; index < size; index++) {
    buffer[index] = ' ';
  }
  buffer[size] = 0;
}

static void 
psp_display_screen_menu(void)
{
  char buffer[64];
  int menu_id = 0;
  int color   = 0;
  int x       = 0;
  int y       = 0;
  int y_step  = 0;

  if (psp_menu_dirty) 
  {
    psp_sdl_blit_background();
    //psp_menu_dirty = 0;
  }
  
  x      = 10;
  y      = 15;
  y_step = 10;
  
  for (menu_id = 0; menu_id < MAX_MENU_ITEM; menu_id++) {
    color = PSP_MENU_TEXT_COLOR;
    if (cur_menu_id == menu_id) color = PSP_MENU_SEL_COLOR;
    else
    if (menu_id == MENU_EXIT) color = PSP_MENU_WARNING_COLOR;
    else
    if (menu_id == MENU_HELP) color = PSP_MENU_GREEN_COLOR;

    psp_sdl_back2_print(x, y, menu_list[menu_id].title, color);

    if (menu_id == MENU_RENDER) {

      if (ti92_render_mode == TI92_RENDER_NORMAL) strcpy(buffer, "normal");
      else
      if (ti92_render_mode == TI92_RENDER_WIDTH) strcpy(buffer, "width");
      else
      if (ti92_render_mode == TI92_RENDER_X125)   strcpy(buffer, "x1.25");
      else
      if (ti92_render_mode == TI92_RENDER_MAX)    strcpy(buffer, "max");

      string_fill_with_space(buffer, 13);
      psp_sdl_back2_print(140, y, buffer, color);
    } else
    if (menu_id == MENU_JOYSTICK) {
      if (psp_joystick_cursor) strcpy(buffer,"yes");
      else                     strcpy(buffer,"no ");
      psp_sdl_back2_print(140, y, buffer, color);
    } else
    if (menu_id == MENU_CLOCK) {
      sprintf(buffer,"%d", psp_cpu_clock);
      psp_sdl_back2_print(140, y, buffer, color);
    } else
    if (menu_id == MENU_SCREENSHOT) {
      sprintf(buffer,"%d", TI92.psp_screenshot_id);
      psp_sdl_back2_print(140, y, buffer, color);
      y += y_step;
    } else
    if (menu_id == MENU_HARD_RESET) {
      y += y_step;
    } else
    if (menu_id == MENU_BACK) {
      y += y_step;
    } else
    if (menu_id == MENU_KEYBOARD) {
      y += y_step;
    } else
    if (menu_id == MENU_HELP) {
      y += y_step;
    }

    y += y_step;
  }
}

static void
psp_main_menu_reset(void)
{
  /* Reset ! */
  psp_display_screen_menu();
  psp_sdl_back2_print(180,  70, "Reset TI-92 !", 
                     PSP_MENU_WARNING_COLOR);
  psp_menu_dirty = 1;
  psp_sdl_flip();
  MC68000_reset();
  sleep(1);
}

static int
psp_main_menu_hard_reset(void)
{
  gp2xCtrlData c;
  int          do_hard_reset = 0;

  /* Reset ! */
  psp_display_screen_menu();
  psp_sdl_back2_print(140,  70, "press X to hard reset ", PSP_MENU_WARNING_COLOR);
  psp_menu_dirty = 1;
  psp_sdl_flip();

  psp_kbd_wait_no_button();

  do
  {
    gp2xCtrlReadBufferPositive(&c, 1);
    c.Buttons &= PSP_ALL_BUTTON_MASK;

    if (c.Buttons & GP2X_CTRL_CROSS) {
      psp_display_screen_menu();
      psp_sdl_back2_print(180,  70, "Hard Reset TI-92 !", 
                      PSP_MENU_WARNING_COLOR);
      psp_menu_dirty = 1;
      psp_sdl_flip();
      hard_reset_emulator();
      do_hard_reset = 1;
      sleep(1);
    }

  } while (c.Buttons == 0);

  psp_kbd_wait_no_button();

  return do_hard_reset;
}

static void
psp_main_menu_clock(int step)
{
  if (step > 0) {
    psp_cpu_clock += 10;
    if (psp_cpu_clock > 260) psp_cpu_clock = 260;
  } else {
    psp_cpu_clock -= 10;
    if (psp_cpu_clock < 60) psp_cpu_clock = 60;
  }
}


static void
psp_main_menu_load()
{
  int ret;

  ret = psp_fmgr_menu(FMGR_FORMAT_92);
  if (ret ==  1) /* load OK */
  {
    psp_menu_dirty = 1;
    psp_display_screen_menu();
    psp_sdl_back2_print(180, 70, "File loaded !", 
                       PSP_MENU_NOTE_COLOR);
    psp_menu_dirty = 1;
    psp_sdl_flip();
    sleep(1);
  }
  else 
  if (ret == -1) /* Load Error */
  {
    psp_menu_dirty = 1;
    psp_display_screen_menu();
    psp_sdl_back2_print(180,  70, "Can't load file !", 
                       PSP_MENU_WARNING_COLOR);
    psp_menu_dirty = 1;
    psp_sdl_flip();
    sleep(1);
  }
}

static void
psp_main_menu_screenshot(void)
{
  psp_screenshot_mode = 10;
}

static void
psp_main_menu_validate(int end_menu)
{
  /* Validate */
  TI92.psp_cpu_clock = psp_cpu_clock;
  TI92.psp_joystick_cursor = psp_joystick_cursor;
  TI92.ti92_render_mode = ti92_render_mode;
  gp2xPowerSetClockFrequency(TI92.psp_cpu_clock);
}

static void
psp_main_menu_save_config()
{
  int error;
  psp_main_menu_validate(0);

  error = ti92_save_configuration();

  if (! error) /* save OK */
  {
    psp_display_screen_menu();
    psp_sdl_back2_print(180,  70, "File saved !", 
                       PSP_MENU_NOTE_COLOR);
    psp_menu_dirty = 1;
    psp_sdl_flip();
    sleep(1);
  }
  else 
  {
    psp_display_screen_menu();
    psp_sdl_back2_print(180,  70, "Can't save file !", 
                       PSP_MENU_WARNING_COLOR);
    psp_menu_dirty = 1;
    psp_sdl_flip();
    sleep(1);
  }
}

int
psp_main_menu_exit(void)
{
  gp2xCtrlData c;
  int          do_exit = 0;

  psp_display_screen_menu();
  psp_sdl_back2_print(180,  70, "press X to exit ", PSP_MENU_WARNING_COLOR);
  psp_sdl_back2_print(150,  80, "press B to save and exit", PSP_MENU_WARNING_COLOR);
  psp_menu_dirty = 1;
  psp_sdl_flip();

  psp_kbd_wait_no_button();

  do
  {
    gp2xCtrlReadBufferPositive(&c, 1);
    c.Buttons &= PSP_ALL_BUTTON_MASK;

    if (c.Buttons & GP2X_CTRL_CROSS) {
      do_exit = 1;
    } else 
    if (c.Buttons & GP2X_CTRL_CIRCLE) {
      do_exit = 2;
    }
    if (do_exit) {
  
      if (do_exit == 2) {
        ti92_save_configuration();
        ti92_save_emulator_stuff();
      }
      psp_sdl_clear_screen(0);
      psp_sdl_flip();
      psp_sdl_clear_screen(0);
      psp_sdl_flip();
      psp_sdl_exit(0);
    }

  } while (c.Buttons == 0);

  psp_kbd_wait_no_button();

  return 0;
}

static void
psp_menu_render(int step)
{
  if (step > 0) {
    if (ti92_render_mode < TI92_LAST_RENDER) ti92_render_mode++;
    else                                   ti92_render_mode = 0;
  } else {
    if (ti92_render_mode > 0) ti92_render_mode--;
    else                     ti92_render_mode = TI92_LAST_RENDER;
  }
}

int 
psp_main_menu(void)
{
  gp2xCtrlData c;
  long        new_pad;
  long        old_pad;
  int         last_time;
  int         end_menu;

  psp_kbd_wait_no_button();

  old_pad   = 0;
  last_time = 0;
  end_menu  = 0;

  psp_menu_dirty = 1;

  psp_cpu_clock = TI92.psp_cpu_clock;
  psp_joystick_cursor = TI92.psp_joystick_cursor;
  ti92_render_mode = TI92.ti92_render_mode;

  while (! end_menu)
  {
    psp_display_screen_menu();
    psp_sdl_flip();

    while (1)
    {
      gp2xCtrlPeekBufferPositive(&c, 1);
      c.Buttons &= PSP_ALL_BUTTON_MASK;

      if (c.Buttons) break;
    }

    new_pad = c.Buttons;

    if ((old_pad != new_pad) || ((c.TimeStamp - last_time) > PSP_MENU_MIN_TIME)) {
      last_time = c.TimeStamp;
      old_pad = new_pad;

    } else continue;

    if ((c.Buttons & GP2X_CTRL_RTRIGGER) == GP2X_CTRL_RTRIGGER) {
      psp_main_menu_reset();
      end_menu = 1;
    } else
    if ((new_pad & GP2X_CTRL_CROSS ) || 
        (new_pad & GP2X_CTRL_CIRCLE) ||
        (new_pad & GP2X_CTRL_TRIANGLE))
    {
      int step;

      if (new_pad & GP2X_CTRL_CROSS)  step = -1;
      else 
      if (new_pad & GP2X_CTRL_CIRCLE) step =  1;
      else                            step =  0;

      switch (cur_menu_id ) 
      {
        case MENU_JOYSTICK  : psp_joystick_cursor = ! psp_joystick_cursor;
        break;              
        case MENU_RENDER    : psp_menu_render( step );
        break;              
        case MENU_CLOCK     : psp_main_menu_clock( step );
        break;

        case MENU_LOAD_PRG  : psp_menu_dirty = 1;
                              psp_main_menu_load();
                              old_pad = new_pad = 0;
                              psp_menu_dirty = 1;
        break;              
        case MENU_SCREENSHOT : psp_main_menu_screenshot();
                               end_menu = 1;
        break;              

        case MENU_SAVE_CFG   : psp_main_menu_save_config();
        break;

        case MENU_HARD_RESET : if (psp_main_menu_hard_reset()) end_menu = 1;
        break;
        case MENU_KEYBOARD   : psp_keyboard_menu();
                               psp_menu_dirty = 1;
                               old_pad = new_pad = 0;
        break;
        case MENU_SOFT_RESET : psp_main_menu_reset();
                               end_menu = 1;
        break;

        case MENU_BACK      : end_menu = 1;
        break;

        case MENU_EXIT      : psp_main_menu_exit();
        break;

        case MENU_HELP      : psp_menu_dirty = 1;
                              psp_help_menu();
                              old_pad = new_pad = 0;
                              psp_menu_dirty = 1;
        break;              
      }

    } else
    if(new_pad & GP2X_CTRL_UP) {

      if (cur_menu_id > 0) cur_menu_id--;
      else                 cur_menu_id = MAX_MENU_ITEM-1;

    } else
    if(new_pad & GP2X_CTRL_DOWN) {

      if (cur_menu_id < (MAX_MENU_ITEM-1)) cur_menu_id++;
      else                                 cur_menu_id = 0;

    } else  
    if(new_pad & GP2X_CTRL_SQUARE) {
      /* Cancel */
      end_menu = -1;
    } else 
    if(new_pad & GP2X_CTRL_SELECT) {
      /* Back to TI-92 */
      end_menu = 1;
    }
  }
 
  psp_kbd_wait_no_button();

  psp_sdl_clear_screen( PSP_MENU_BLACK_COLOR );
  psp_sdl_flip();
  psp_sdl_clear_screen( PSP_MENU_BLACK_COLOR );
  psp_sdl_flip();

  if (end_menu > 0) {
    psp_main_menu_validate(1);
  }

  return 1;
}

