/*
	YAPE - Yet Another Plus/4 Emulator

	The program emulates the Commodore 264 family of 8 bit microcomputers

	This program is free software, you are welcome to distribute it,
	and/or modify it under certain conditions. For more information,
	read 'Copying'.

	(c) 2000, 2001, 2004 Attila Grsz
*/

#include <algorithm>
#include <psptypes.h>
#include "main.h"


#include "psp_sdl.h"
#include "psp_kbd.h"
#include "psp_danzeff.h"
#include "global.h"

// SDL stuff
static short     g_pal[256];
// color
static Uint8	p4col_calc[768];

///////////////////
// Screen variables
static Uint8	*pixel;

////////////////
// Supplementary
static Uint32			g_TotFrames = 0;
static Uint32			timeelapsed;

static MEM				*ted8360 = new MEM;
static CPU				*machine = new CPU(ted8360);
static Uint32			i;
static char				textout[40];

static Uint8			*screenptr;
static bool				g_FrameRate = true;
static bool				g_50Hz = true;
static bool				g_bSaveSettings = true;

//-----------------------------------------------------------------------------
// Name: ShowFrameRate()
//-----------------------------------------------------------------------------
# if 0
inline static void 
ShowFrameRate()
{
    static Uint32 fps = 50;
	static Uint32 g_TotElapsed = 0;

    g_TotFrames++;
	if (g_TotElapsed + 2000 < timeelapsed) {
     	g_TotElapsed = timeelapsed;
     	fps = g_TotFrames / 2;
     	g_TotFrames = 0;
  	}
	sprintf(textout, "%3i FPS", fps);
# if 1 //LUDO:
  psp_sdl_fill_print(10, 10, buffer, psp_sdl_rgb(0xff, 0xff, 0xff), 0x00);
# else
	ted8360->texttoscreen(324,40,textout);
# endif
}
# endif


//-----------------------------------------------------------------------------
// Name: load_file()
// Desc: The Load File Procedure
//-----------------------------------------------------------------------------
static bool 
load_file(char *fname)
{
	Uint8   *lpBufPtr;
	Uint16	loadaddr;
	FILE	*prg;
	Uint32	fsize;

	if ((prg = fopen(fname, "rb"))== NULL) {
    	return false;
	} else {
		// load PRG file
		fseek(prg, 0L, SEEK_END);
		fsize=ftell(prg);
		fseek(prg, 0L, SEEK_SET);
		lpBufPtr=(Uint8 *) malloc(fsize);
		fread(lpBufPtr,fsize,1,prg);
		// copy to memory
		loadaddr=lpBufPtr[0]|(lpBufPtr[1]<<8);
		for (i=0;i<fsize;i++)
			ted8360->wrt(loadaddr+i,lpBufPtr[2+i]);

		ted8360->wrt(0x2D,(loadaddr+fsize-2)&0xFF);
		ted8360->wrt(0x2E,(loadaddr+fsize-2)>>8);
		ted8360->wrt(0x2F,(loadaddr+fsize-2)&0xFF);
		ted8360->wrt(0x30,(loadaddr+fsize-2)>>8);
		ted8360->wrt(0x31,(loadaddr+fsize-2)&0xFF);
		ted8360->wrt(0x32,(loadaddr+fsize-2)>>8);
		ted8360->wrt(0x9D,(loadaddr+fsize-2)&0xFF);
		ted8360->wrt(0x9E,(loadaddr+fsize-2)>>8);
		fclose(prg);
	};

	return true;
}
//-----------------------------------------------------------------------------
// Name: StartFile( )
// Desc: The Load PRG File Dialog Box Procedure
//-----------------------------------------------------------------------------
bool StartFile()
{
	ted8360->wrt(0xEF,4);
	ted8360->wrt(0x0527,82);
	ted8360->wrt(0x0528,85);
	ted8360->wrt(0x0529,78);
	ted8360->wrt(0x052A,13);
	return true;
}

//-----------------------------------------------------------------------------

static bool 
start_file(char *szFile )
{
	char			fileext[4];

	if (szFile[0]!='\0') {
		for (i=0;i<4;++i) fileext[i]=tolower(szFile[strlen(szFile)-3+i]);
		if (!strncmp(fileext,"d64",3)) {
			/*if (!LoadD64Image(szFile)) {
				return false;
			}*/
			ted8360->wrt(0xEF,9);
			ted8360->wrt(0x0527,68);
			ted8360->wrt(0x0528,204);
			ted8360->wrt(0x0529,34);
			ted8360->wrt(0x052A,42);
			ted8360->wrt(0x052B,13);
			ted8360->wrt(0x052C,82);
			ted8360->wrt(0x052D,85);
			ted8360->wrt(0x052E,78);
			ted8360->wrt(0x052F,13);
			return true;
		}
		if (!strcmp(fileext,"prg")) {
			load_file( szFile );
			StartFile();
			return true;
		}
		if (!strcmp(fileext,"tap")) {

			/*ted8360->wrt(0xEF,3);
			ted8360->wrt(0x0527,76);
			ted8360->wrt(0x0528,207);
			ted8360->wrt(0x0529,13);*/
			ted8360->tap->detach_tap();
			strcpy(ted8360->tap->tapefilename, szFile);
			ted8360->tap->attach_tap();
			/*ted8360->wrtDMA(0xFD10,ted8360->readDMA(0xFD10)&~0x04);
			ted8360->tapeload=true;*/
			return true;
		}
		return false;
    }
	return false;
}

//-----------------------------------------------------------------------------

void init_palette(void)
{
	Uint32 colorindex;
	double	Uc, Vc, Yc,  PI = 3.14159265 ;
	double bsat = 45.0;

    /* Allocate 256 color palette */
    int ncolors = 256;
# if 0 //LUDO:
    g_pal  = (SDL_Color *)malloc(ncolors*sizeof(SDL_Color));
# endif

	// calculate palette based on the HUE values
	memset(p4col_calc, 0, 768);
	for ( i=1; i<16; i++)
		for ( register int j = 0; j<8; j++) {
		    Uint8 col;
			if (i == 1)
				Uc = Vc = 0;
			else {
				Uc = bsat * ((float) cos( HUE[i] * PI / 180.0 ));
				Vc = bsat * ((float) sin( HUE[i] * PI / 180.0 ));
			}
			Yc = (luma[j+1] - 2.0)* 255.0 / (5.0 - 2.0); // 5V is the base voltage
			// RED, GREEN and BLUE component
			colorindex = (j)*16*3 + i*3;
			col = (Uint8) std::max(std::min((Yc + 1.367 * Vc),255.0),0.0);
			p4col_calc[ colorindex ] = p4col_calc[ 384 + colorindex ] = col;
			col = (Uint8) std::max(std::min((Yc - 0.336 * Uc - 0.698 * Vc ),255.0),0.0);
			p4col_calc[ colorindex + 1] = p4col_calc[ 384 + colorindex + 1] = col;
			col = (Uint8) std::max(std::min((Yc + 1.732 * Uc),255.0),0.0);
			p4col_calc[ colorindex + 2] = p4col_calc[ 384 + colorindex + 2] = col;
		}

	for (i=0 ; i<256; ++i) {
		// Creating an 8 bit SDL_Color structure
		unsigned char b=p4col_calc[i*3+2];
		unsigned char g=p4col_calc[i*3+1];
		unsigned char r=p4col_calc[i*3];
		g_pal[i] = psp_sdl_rgb(r, g, b);
	}

# if 0 //LUDO:
	palette.ncolors = 256;
	palette.colors = g_pal;
# endif
}

# if 0
#define SCR_HSIZE 456
#define SCR_VSIZE 312
# endif
void
yape_display_screen_normal(void)
{
  unsigned char  *src_vram = ted8360->screen;
  unsigned short *dst_vram = psp_sdl_get_vram_addr(12,0);
  int x;
  int y;
  src_vram += 20 * SCR_HSIZE;
  for (y = 20; y < SCR_VSIZE - 20; y++) {
    for (x = 0; x < SCR_HSIZE; x++) {
      dst_vram[x] = g_pal[ src_vram[x] ];
    }
    src_vram += SCR_HSIZE;
    dst_vram += PSP_LINE_SIZE;
  }
}

typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned int dword;

/* 

  LUDO: 16-bit HiColor (565 format) 
  see http://www.compuphase.com/graphic/scale3.htm

 */
static inline word loc_coloraverage(word a, word b)
{
  return (word)(((a ^ b) & 0xf7deU) >> 1) + (a & b);
}

static inline void 
render16bpp_X15_pixel(word *dist, byte *b_src)
{
  word src[4];

  src[0] = g_pal[ b_src[0] ];
  src[1] = g_pal[ b_src[1] ];
  src[2] = g_pal[ b_src[2] ];
  src[3] = g_pal[ b_src[3] ];

  dist[0] = src[0];
  dist[1] = loc_coloraverage(src[0], src[1]);
  dist[2] = src[1];
  dist[3] = src[2];
  dist[4] = loc_coloraverage(src[2], src[3]);
  dist[5] = src[3];
}

static inline void 
render16bpp_X125_pixel(word *dist, byte *b_src)
{
  word src[4];

  src[0] = g_pal[ b_src[0] ];
  src[1] = g_pal[ b_src[1] ];
  src[2] = g_pal[ b_src[2] ];
  src[3] = g_pal[ b_src[3] ];
  
  dist[0] = src[0];
  dist[1] = loc_coloraverage(src[0], src[1]);
  dist[2] = src[1];
  dist[3] = src[2];
  dist[4] = src[3];
}


# if 0
void
yape_display_screen_x125(void)
{
  unsigned char  *src_vram = ted8360->screen;
  unsigned short *dst_vram = psp_sdl_get_vram_addr(0,0);
  unsigned char  *src_vram_line = src_vram;
  unsigned short *dst_vram_line = dst_vram;

  int x;
  int y;
  int count;

  src_vram += (20 * SCR_HSIZE) + 36;
  for (y = 20; y < SCR_VSIZE - 20; y++) {
    count = 384;

    src_vram_line = src_vram;
    dst_vram_line = dst_vram;

    while (count > 0) {
      render16bpp_X125_pixel(dst_vram_line, src_vram_line);
      src_vram_line += 4;
      dst_vram_line += 5;
      count -= 4;
    }
    src_vram += SCR_HSIZE;
    dst_vram += PSP_LINE_SIZE;
  }
  
}
# endif

void
yape_display_screen_x125(void)
{
  unsigned char  *src_vram = ted8360->screen;
  unsigned short *dst_vram = psp_sdl_get_vram_addr(0,10);
  unsigned char  *src_vram_line = src_vram;
  unsigned short *dst_vram_line = dst_vram;

  int src_y;
  int dst_x;
  int dst_y;
  int count;

  src_vram += (56 * SCR_HSIZE) + 36;

  for (dst_y = 0; dst_y < 250; dst_y++) {

    src_y = (dst_y * 200) / 250;
    src_vram_line = &src_vram[(src_y * SCR_HSIZE)];
    dst_vram_line = dst_vram;

    count = 384;
    while (count > 0) {
      render16bpp_X125_pixel(dst_vram_line, src_vram_line);
      src_vram_line += 4;
      dst_vram_line += 5;
      count -= 4;
    }
    dst_vram += PSP_LINE_SIZE;
  }
  
}

void
yape_display_screen_x15(void)
{
  unsigned char  *src_vram = ted8360->screen;
  unsigned short *dst_vram = psp_sdl_get_vram_addr(0,0);
  unsigned char  *src_vram_line = src_vram;
  unsigned short *dst_vram_line = dst_vram;

  int src_y;
  int dst_x;
  int dst_y;
  int count;

  src_vram += (66 * SCR_HSIZE) + 68;

  for (dst_y = 0; dst_y < 270; dst_y++) {

    src_y = (dst_y * 180) / 270;
    src_vram_line = &src_vram[(src_y * SCR_HSIZE)];
    dst_vram_line = dst_vram;

    count = 320;
    while (count > 0) {
      render16bpp_X15_pixel(dst_vram_line, src_vram_line);
      src_vram_line += 4;
      dst_vram_line += 6;
      count -= 4;
    }
    dst_vram += PSP_LINE_SIZE;
  }
}

void
yape_display_screen_fit(void)
{
  unsigned char  *src_vram = ted8360->screen;
  unsigned short *dst_vram = psp_sdl_get_vram_addr(0,0);
  unsigned char  *src_vram_line = src_vram;
  unsigned short *dst_vram_line = dst_vram;

  int src_y;
  int dst_x;
  int dst_y;
  int count;

  src_vram += (56 * SCR_HSIZE) + 68;

  for (dst_y = 0; dst_y < 270; dst_y++) {

    src_y = (dst_y * 200) / 270;
    src_vram_line = &src_vram[(src_y * SCR_HSIZE)];
    dst_vram_line = dst_vram;

    count = 320;
    while (count > 0) {
      render16bpp_X15_pixel(dst_vram_line, src_vram_line);
      src_vram_line += 4;
      dst_vram_line += 6;
      count -= 4;
    }
    dst_vram += PSP_LINE_SIZE;
  }
}

static void
yape_synchronize(void)
{
	static u32 nextclock = 1;

  if (YAPE.yape_speed_limiter) {

	  if (nextclock) {
		  u32 curclock;
		  do {
			  curclock = clock();
		  } while (curclock < nextclock);
  
		  nextclock = curclock + ((CLK_TCK * YAPE.yape_speed_limiter) / 60);
    }
  }
}


static inline void 
FrameUpdate(void)
{
  if (YAPE.psp_skip_cur_frame <= 0) {

    YAPE.psp_skip_cur_frame = YAPE.psp_skip_max_frame;

    if (YAPE.yape_render_mode == YAPE_RENDER_NORMAL    ) yape_display_screen_normal(); 
    else
    if (YAPE.yape_render_mode == YAPE_RENDER_X125      ) yape_display_screen_x125(); 
    else
    if (YAPE.yape_render_mode == YAPE_RENDER_X15       ) yape_display_screen_x15(); 
    else
    if (YAPE.yape_render_mode == YAPE_RENDER_FIT       ) yape_display_screen_fit(); 

    if (YAPE.yape_speed_limiter) {
      yape_synchronize();
    }

    if (psp_kbd_is_danzeff_mode()) {
      danzeff_moveTo(-165, -50);
      danzeff_render();
    }

    psp_kbd_display_active_mapping();
    psp_sdl_flip();
  
    if (psp_screenshot_mode) {
      psp_screenshot_mode--;
      if (psp_screenshot_mode <= 0) {
        psp_sdl_save_screenshot();
        psp_screenshot_mode = 0;
      }
    }

    if (yape_prg_to_load) {
      yape_prg_to_load--;
      if (! yape_prg_to_load) {
        yape_load_prg();
      }
    }

  } else if (YAPE.psp_skip_max_frame) {
    YAPE.psp_skip_cur_frame--;
  }
}

static void 
app_initialise()
{
  psp_sdl_init();

	init_palette();

	init_audio(ted8360);
}

inline Uint32 time_elapsed(void)
{
    return SDL_GetTicks();
}

extern "C" int
c_yape_load_prg(char *filename)
{
  int error = 1;

  if (load_file(filename)) {
    if (start_file(filename)) error = 0;
  }

  return error;
}

extern "C" void
c_yape_push_key(int key_id)
{
  ted8360->keys->pushkey(key_id);
}

extern "C" void
c_yape_release_key(int key_id)
{
	ted8360->keys->releasekey(key_id);
}

extern "C" void
c_yape_reset(void)
{
  machine->reset();
}

extern "C" void
c_yape_push_joystick(int joy_id)
{
  ted8360->keys->pushjoy(joy_id);
}

extern "C" void
c_yape_release_joystick(int joy_id)
{
	ted8360->keys->releasejoy(joy_id);
}

extern "C" void
c_yape_reset_keyboard()
{
	ted8360->keys->empty();
}

// main program
int
SDL_main(int argc, char *argv[])
{
	app_initialise();

  yape_initialize();

	//--------------------------------------------------------------
# if 0
	timeelapsed = time_elapsed();
# endif
	ted8360->cpuptr = machine;
	for (;;) {
		// hook into the emulation loop if active
	 
	  ted8360->ted_process();
    psp_update_keys();
# if 0
	  unsigned int time_limit = timeelapsed + 20;
	  timeelapsed = time_elapsed();
	  if (g_50Hz) {
	      if (time_limit>timeelapsed) {
	      	int nr10ms = ((time_limit-timeelapsed)/10) * 10;
	      	SDL_Delay(nr10ms);
	      	timeelapsed = time_elapsed();
	  	    while (time_limit>timeelapsed) {
	  	    	SDL_Delay(0);
	  	    	timeelapsed = time_elapsed();
	  		}
	  	}
	  }
	  if (g_FrameRate) {
	  	ShowFrameRate();
    }
# endif
	  // frame update
	  FrameUpdate();
	 
	}
	return 0;
}

