/*
 *  Copyright (C) 2006 Ludovic Jacomme (ludovic.jacomme@gmail.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/time.h>
#include <fcntl.h>

#include <pspctrl.h>
#include <pspkernel.h>
#include <pspdebug.h>
#include <SDL.h>

#include "global.h"
#include "psp_kbd.h"
#include "psp_menu.h"
#include "psp_sdl.h"
#include "psp_danzeff.h"

# define KBD_MIN_ANALOG_TIME  150000
# define KBD_MIN_START_TIME   800000
# define KBD_MAX_EVENT_TIME   500000
# define KBD_MIN_PENDING_TIME 300000
# define KBD_MIN_DANZEFF_TIME 150000
# define KBD_MIN_COMMAND_TIME 100000
# define KBD_MIN_BATTCHECK_TIME 90000000 

 static SceCtrlData    loc_button_data;
 static unsigned int   loc_last_event_time = 0;
 static unsigned int   loc_last_analog_time = 0;
 static long           first_time_stamp = -1;
 static char           loc_button_press[ KBD_MAX_BUTTONS ]; 
 static char           loc_button_release[ KBD_MAX_BUTTONS ]; 
 static unsigned int   loc_button_mask[ KBD_MAX_BUTTONS ] =
 {
   PSP_CTRL_UP         , /*  KBD_UP         */
   PSP_CTRL_RIGHT      , /*  KBD_RIGHT      */
   PSP_CTRL_DOWN       , /*  KBD_DOWN       */
   PSP_CTRL_LEFT       , /*  KBD_LEFT       */
   PSP_CTRL_TRIANGLE   , /*  KBD_TRIANGLE   */
   PSP_CTRL_CIRCLE     , /*  KBD_CIRCLE     */
   PSP_CTRL_CROSS      , /*  KBD_CROSS      */
   PSP_CTRL_SQUARE     , /*  KBD_SQUARE     */
   PSP_CTRL_SELECT     , /*  KBD_SELECT     */
   PSP_CTRL_START      , /*  KBD_START      */
   PSP_CTRL_HOME       , /*  KBD_HOME       */
   PSP_CTRL_HOLD       , /*  KBD_HOLD       */
   PSP_CTRL_LTRIGGER   , /*  KBD_LTRIGGER   */
   PSP_CTRL_RTRIGGER   , /*  KBD_RTRIGGER   */
 };

 static char loc_button_name[ KBD_ALL_BUTTONS ][20] =
 {
   "UP",
   "RIGHT",
   "DOWN",
   "LEFT",
   "TRIANGLE",
   "CIRCLE",
   "CROSS",
   "SQUARE",
   "SELECT",
   "START",
   "HOME",
   "HOLD",
   "LTRIGGER",
   "RTRIGGER",
   "JOY_UP",
   "JOY_RIGHT",
   "JOY_DOWN",
   "JOY_LEFT"
 };

 static char loc_button_name_L[ KBD_ALL_BUTTONS ][20] =
 {
   "L_UP",
   "L_RIGHT",
   "L_DOWN",
   "L_LEFT",
   "L_TRIANGLE",
   "L_CIRCLE",
   "L_CROSS",
   "L_SQUARE",
   "L_SELECT",
   "L_START",
   "L_HOME",
   "L_HOLD",
   "L_LTRIGGER",
   "L_RTRIGGER",
   "L_JOY_UP",
   "L_JOY_RIGHT",
   "L_JOY_DOWN",
   "L_JOY_LEFT"
 };
 
  static char loc_button_name_R[ KBD_ALL_BUTTONS ][20] =
 {
   "R_UP",
   "R_RIGHT",
   "R_DOWN",
   "R_LEFT",
   "R_TRIANGLE",
   "R_CIRCLE",
   "R_CROSS",
   "R_SQUARE",
   "R_SELECT",
   "R_START",
   "R_HOME",
   "R_HOLD",
   "R_LTRIGGER",
   "R_RTRIGGER",
   "R_JOY_UP",
   "R_JOY_RIGHT",
   "R_JOY_DOWN",
   "R_JOY_LEFT"
 };
 
  struct yape_key_trans psp_yape_key_info[YAPEK_MAX_KEY]=
  {
    // YAPEK            SHIFT            NAME 
    { YAPEK_1,          YAPE_KEY_1,           0,     "1" },
    { YAPEK_2,          YAPE_KEY_2,           0,     "2" },
    { YAPEK_3,          YAPE_KEY_3,           0,     "3" },
    { YAPEK_4,          YAPE_KEY_4,           0,     "4" },
    { YAPEK_5,          YAPE_KEY_5,           0,     "5" },
    { YAPEK_6,          YAPE_KEY_6,           0,     "6" },
    { YAPEK_7,          YAPE_KEY_7,           0,     "7" },
    { YAPEK_8,          YAPE_KEY_8,           0,     "8" },
    { YAPEK_9,          YAPE_KEY_9,           0,     "9" },
    { YAPEK_0,          YAPE_KEY_0,           0,     "0" },
    { YAPEK_MINUS    ,  YAPE_KEY_MINUS    ,   0,     "-" },
    { YAPEK_PLUS,       YAPE_KEY_PLUS,        0,     "+" },
    { YAPEK_SLASH,      YAPE_KEY_SLASH,       0,     "/" },
    { YAPEK_ASTERISK,   YAPE_KEY_ASTERISK,    0,     "*" },
    { YAPEK_COMMA,      YAPE_KEY_COMMA,       0,     "," },
    { YAPEK_PERIOD,     YAPE_KEY_PERIOD,      0,     "." },
    { YAPEK_AT,         YAPE_KEY_AT,          0,     "@" },
    { YAPEK_EXCLAMATN,  YAPE_KEY_1,           1,     "!" },
    { YAPEK_DBLQUOTE,   YAPE_KEY_2,           1,     "\"" },
    { YAPEK_HASH,       YAPE_KEY_3,           1,     "#" },
    { YAPEK_DOLLAR,     YAPE_KEY_4,           1,     "$" },
    { YAPEK_PERCENT,    YAPE_KEY_5,           1,     "%" },
    { YAPEK_AMPERSAND,  YAPE_KEY_6,           1,     "&" },
    { YAPEK_QUOTE,      YAPE_KEY_7,           1,     "'" },
    { YAPEK_LEFTPAREN,  YAPE_KEY_8,           1,     "(" },
    { YAPEK_RIGHTPAREN, YAPE_KEY_9,           1,     ")" },
    { YAPEK_EQUAL,      YAPE_KEY_EQUAL,       0,    "=" },
    { YAPEK_SEMICOLON,  YAPE_KEY_SEMICOLON,   0,     ";" },
    { YAPEK_QUESTION,   YAPE_KEY_SLASH,       1,     "?" },
    { YAPEK_COLON,      YAPE_KEY_COLON,       0,     ":" },
    { YAPEK_LESS,       YAPE_KEY_COMMA,       1,     "<" },
    { YAPEK_GREATER,    YAPE_KEY_PERIOD,      1,     ">" },
    { YAPEK_SHIFT,      YAPE_KEY_SHIFT,       0,     "SHIFT" },
    { YAPEK_A,          YAPE_KEY_A,           0,     "A" },
    { YAPEK_B,          YAPE_KEY_B,           0,     "B" },
    { YAPEK_C,          YAPE_KEY_C,           0,     "C" },
    { YAPEK_D,          YAPE_KEY_D,           0,     "D" },
    { YAPEK_E,          YAPE_KEY_E,           0,     "E" },
    { YAPEK_F,          YAPE_KEY_F,           0,     "F" },
    { YAPEK_G,          YAPE_KEY_G,           0,     "G" },
    { YAPEK_H,          YAPE_KEY_H,           0,     "H" },
    { YAPEK_I,          YAPE_KEY_I,           0,     "I" },
    { YAPEK_J,          YAPE_KEY_J,           0,     "J" },
    { YAPEK_K,          YAPE_KEY_K,           0,     "K" },
    { YAPEK_L,          YAPE_KEY_L,           0,     "L" },
    { YAPEK_M,          YAPE_KEY_M,           0,     "M" },
    { YAPEK_N,          YAPE_KEY_N,           0,     "N" },
    { YAPEK_O,          YAPE_KEY_O,           0,     "O" },
    { YAPEK_P,          YAPE_KEY_P,           0,     "P" },
    { YAPEK_Q,          YAPE_KEY_Q,           0,     "Q" },
    { YAPEK_R,          YAPE_KEY_R,           0,     "R" },
    { YAPEK_S,          YAPE_KEY_S,           0,     "S" },
    { YAPEK_T,          YAPE_KEY_T,           0,     "T" },
    { YAPEK_U,          YAPE_KEY_U,           0,     "U" },
    { YAPEK_V,          YAPE_KEY_V,           0,     "V" },
    { YAPEK_W,          YAPE_KEY_W,           0,     "W" },
    { YAPEK_X,          YAPE_KEY_X,           0,     "X" },
    { YAPEK_Y,          YAPE_KEY_Y,           0,     "Y" },
    { YAPEK_Z,          YAPE_KEY_Z,           0,     "Z" },
    { YAPEK_SPACE,      YAPE_KEY_SPACE,       0,     "SPACE" },
    { YAPEK_RETURN,     YAPE_KEY_RETURN,      0,     "RETURN" },
    { YAPEK_BACKSPACE,  YAPE_KEY_BACKSPACE,   0,     "BACKSPACE" },
    { YAPEK_POUND,      YAPE_KEY_POUND,       0,     "POUND" },
    { YAPEK_INSERT,     YAPE_KEY_INSERT,      0,     "INSERT" },
    { YAPEK_CAPSLOCK,   YAPE_KEY_CAPSLOCK,    0,     "CAPSLOCK" },
    { YAPEK_LEFT,       YAPE_KEY_LEFT,        0,     "LEFT" },
    { YAPEK_UP,         YAPE_KEY_UP,          0,     "UP" },
    { YAPEK_RIGHT,      YAPE_KEY_RIGHT,       0,     "RIGHT" },
    { YAPEK_DOWN,       YAPE_KEY_DOWN,        0,     "DOWN" },
    { YAPEK_ESCAPE,     YAPE_KEY_ESCAPE,      0,     "ESCAPE" },
    { YAPEK_HOME,       YAPE_KEY_HOME,        0,     "HOME" },
    { YAPEK_CLEAR,      YAPE_KEY_HOME,        1,     "CLEAR" },
    { YAPEK_CONTROL,    YAPE_KEY_CONTROL,     0,     "CONTROL" },
    { YAPEK_COMMODORE,  YAPE_KEY_COMMODORE,   0,     "COMMODORE" },
    { YAPEK_F1,         YAPE_KEY_F1,          0,     "F1" },
    { YAPEK_F2,         YAPE_KEY_F2,          0,     "F2" },
    { YAPEK_F3,         YAPE_KEY_F3,          0,     "F3" },
    { YAPEK_F4,         YAPE_KEY_F1,          1,     "F4" },
    { YAPEK_F5,         YAPE_KEY_F2,          1,     "F5" },
    { YAPEK_F6,         YAPE_KEY_F3,          1,     "F6" },
    { YAPEK_F7,         YAPE_KEY_HELP,        1,     "F7" },
    { YAPEK_HELP,       YAPE_KEY_HELP,        0,     "HELP" },
    { YAPEK_JOY_UP,     -1,    0,   "JOY_UP" },
    { YAPEK_JOY_DOWN,   -1,    0,   "JOY_DOWN" },
    { YAPEK_JOY_LEFT,   -1,    0,   "JOY_LEFT" },
    { YAPEK_JOY_RIGHT,  -1,    0,   "JOY_RIGHT" },
    { YAPEK_JOY_FIRE,   -1 ,   0,   "JOY_FIRE" } 
  };

  static int loc_default_mapping[ KBD_ALL_BUTTONS ] = {
    YAPEK_UP              , /*  KBD_UP         */
    YAPEK_RIGHT           , /*  KBD_RIGHT      */
    YAPEK_DOWN            , /*  KBD_DOWN       */
    YAPEK_LEFT            , /*  KBD_LEFT       */
    YAPEK_RETURN          , /*  KBD_TRIANGLE   */
    YAPEK_1               , /*  KBD_CIRCLE     */
    YAPEK_JOY_FIRE        , /*  KBD_CROSS      */
    YAPEK_SPACE           , /*  KBD_SQUARE     */
    -1                    , /*  KBD_SELECT     */
    -1                    , /*  KBD_START      */
    -1                    , /*  KBD_HOME       */
    -1                    , /*  KBD_HOLD       */
    KBD_LTRIGGER_MAPPING  , /*  KBD_LTRIGGER   */
    KBD_RTRIGGER_MAPPING  , /*  KBD_RTRIGGER   */
    YAPEK_JOY_UP          , /*  KBD_JOY_UP     */
    YAPEK_JOY_RIGHT       , /*  KBD_JOY_RIGHT  */
    YAPEK_JOY_DOWN        , /*  KBD_JOY_DOWN   */
    YAPEK_JOY_LEFT          /*  KBD_JOY_LEFT   */
  };

  static int loc_default_mapping_L[ KBD_ALL_BUTTONS ] = {
    YAPEK_UP              , /*  KBD_UP         */
    YAPEK_RIGHT           , /*  KBD_RIGHT      */
    YAPEK_DOWN            , /*  KBD_DOWN       */
    YAPEK_LEFT            , /*  KBD_LEFT       */
    YAPEK_RETURN          , /*  KBD_TRIANGLE   */
    YAPEK_ESCAPE          , /*  KBD_CIRCLE     */
    YAPEK_JOY_FIRE        , /*  KBD_CROSS      */
    YAPEK_SPACE           , /*  KBD_SQUARE     */
    -1                    , /*  KBD_SELECT     */
    -1                    , /*  KBD_START      */
    -1                    , /*  KBD_HOME       */
    -1                    , /*  KBD_HOLD       */
    KBD_LTRIGGER_MAPPING  , /*  KBD_LTRIGGER   */
    KBD_RTRIGGER_MAPPING  , /*  KBD_RTRIGGER   */
    YAPEK_JOY_UP          , /*  KBD_JOY_UP     */
    YAPEK_JOY_RIGHT       , /*  KBD_JOY_RIGHT  */
    YAPEK_JOY_DOWN        , /*  KBD_JOY_DOWN   */
    YAPEK_JOY_LEFT          /*  KBD_JOY_LEFT   */
  };

  static int loc_default_mapping_R[ KBD_ALL_BUTTONS ] = {
    YAPEK_UP              , /*  KBD_UP         */
    YAPEK_RIGHT           , /*  KBD_RIGHT      */
    YAPEK_DOWN            , /*  KBD_DOWN       */
    YAPEK_LEFT            , /*  KBD_LEFT       */
    YAPEK_RETURN          , /*  KBD_TRIANGLE   */
    YAPEK_Y               , /*  KBD_CIRCLE     */
    YAPEK_N               , /*  KBD_CROSS      */
    YAPEK_SPACE           , /*  KBD_SQUARE     */
    -1                    , /*  KBD_SELECT     */
    -1                    , /*  KBD_START      */
    -1                    , /*  KBD_HOME       */
    -1                    , /*  KBD_HOLD       */
    KBD_LTRIGGER_MAPPING  , /*  KBD_LTRIGGER   */
    KBD_RTRIGGER_MAPPING  , /*  KBD_RTRIGGER   */
    YAPEK_JOY_UP          , /*  KBD_JOY_UP     */
    YAPEK_JOY_RIGHT       , /*  KBD_JOY_RIGHT  */
    YAPEK_JOY_DOWN        , /*  KBD_JOY_DOWN   */
    YAPEK_JOY_LEFT          /*  KBD_JOY_LEFT   */
  };

# define KBD_MAX_ENTRIES   82

  int kbd_layout[KBD_MAX_ENTRIES][2] = {
    /* Key            Ascii */
    { YAPEK_1,          '1' },
    { YAPEK_2,          '2' },
    { YAPEK_3,          '3' },
    { YAPEK_4,          '4' },
    { YAPEK_5,          '5' },
    { YAPEK_6,          '6' },
    { YAPEK_7,          '7' },
    { YAPEK_8,          '8' },
    { YAPEK_9,          '9' },
    { YAPEK_0,          '0' },
    { YAPEK_MINUS,      '-' },
    { YAPEK_PLUS,       '+' },
    { YAPEK_SLASH,      '/' },
    { YAPEK_ASTERISK,   '*' },
    { YAPEK_COMMA,      ',' },
    { YAPEK_PERIOD,     '.' },
    { YAPEK_AT,         '@' },
    { YAPEK_EXCLAMATN,  '!' },
    { YAPEK_DBLQUOTE,   '"' },
    { YAPEK_HASH,       '#' },
    { YAPEK_DOLLAR,     '$' },
    { YAPEK_PERCENT,    '%' },
    { YAPEK_AMPERSAND,  '&' },
    { YAPEK_QUOTE,      '\'' },
    { YAPEK_LEFTPAREN,  '(' },
    { YAPEK_RIGHTPAREN, ')' },
    { YAPEK_EQUAL,      '=' },
    { YAPEK_SEMICOLON,  ';' },
    { YAPEK_QUESTION,   '?' },
    { YAPEK_COLON,      ':' },
    { YAPEK_LESS,       '<' },
    { YAPEK_GREATER,    '>' },
    { YAPEK_SHIFT,      DANZEFF_SHIFT  },
    { YAPEK_A,          'A' },
    { YAPEK_B,          'B' },
    { YAPEK_C,          'C' },
    { YAPEK_D,          'D' },
    { YAPEK_E,          'E' },
    { YAPEK_F,          'F' },
    { YAPEK_G,          'G' },
    { YAPEK_H,          'H' },
    { YAPEK_I,          'I' },
    { YAPEK_J,          'J' },
    { YAPEK_K,          'K' },
    { YAPEK_L,          'L' },
    { YAPEK_M,          'M' },
    { YAPEK_N,          'N' },
    { YAPEK_O,          'O' },
    { YAPEK_P,          'P' },
    { YAPEK_Q,          'Q' },
    { YAPEK_R,          'R' },
    { YAPEK_S,          'S' },
    { YAPEK_T,          'T' },
    { YAPEK_U,          'U' },
    { YAPEK_V,          'V' },
    { YAPEK_W,          'W' },
    { YAPEK_X,          'X' },
    { YAPEK_Y,          'Y' },
    { YAPEK_Z,          'Z' },
    { YAPEK_SPACE,      ' '         },
    { YAPEK_RETURN,     DANZEFF_RETURN   },
    { YAPEK_BACKSPACE,  DANZEFF_BACKSPACE },
    { YAPEK_POUND,      DANZEFF_POUND     },
    { YAPEK_INSERT,     DANZEFF_INS },
    { YAPEK_CAPSLOCK,   DANZEFF_CAPSLOCK },
    { YAPEK_LEFT,       -1  },
    { YAPEK_UP,         -1  },
    { YAPEK_RIGHT,      -1  },
    { YAPEK_DOWN,       -1  },
    { YAPEK_ESCAPE,     DANZEFF_ESCAPE  },
    { YAPEK_HOME,       DANZEFF_HOME    },
    { YAPEK_CLEAR,      DANZEFF_CLEAR   },
    { YAPEK_CONTROL,    DANZEFF_CONTROL },
# if 0
    { YAPEK_RUN,        DANZEFF_RUN     },
# endif
    { YAPEK_COMMODORE,  DANZEFF_COMMODORE },
    { YAPEK_F1,         DANZEFF_F1      },
    { YAPEK_F2,         DANZEFF_F2      },
    { YAPEK_F3,         DANZEFF_F3      },
    { YAPEK_F4,         DANZEFF_F4      },
    { YAPEK_F5,         DANZEFF_F5      },
    { YAPEK_F6,         DANZEFF_F6      },
    { YAPEK_F7,         DANZEFF_F7      },
    { YAPEK_HELP,       DANZEFF_HELP    },
  };

 int psp_kbd_mapping[ KBD_ALL_BUTTONS ];
 int psp_kbd_mapping_L[ KBD_ALL_BUTTONS ];
 int psp_kbd_mapping_R[ KBD_ALL_BUTTONS ];
 int psp_kbd_press[ KBD_ALL_BUTTONS ];
 int kbd_ltrigger_mapping_active;
 int kbd_rtrigger_mapping_active;

 static int danzeff_yape_key     = 0;
 static int danzeff_yape_pending = 0;
 static int danzeff_mode        = 0;



int
yape_key_event(int yape_idx, int press)
{
  int yape_id = 0;
  int shift   = 0;

  if ((yape_idx >=           0) && 
      (yape_idx < YAPEK_JOY_UP )) {
    yape_id = psp_yape_key_info[yape_idx].yape_id;
    shift   = psp_yape_key_info[yape_idx].shift;

    if (press) {
      if (shift) c_yape_push_key(YAPE_KEY_SHIFT);
      c_yape_push_key(yape_id);
    } else {
      c_yape_release_key(yape_id);
      if (shift) c_yape_release_key(YAPE_KEY_SHIFT);
    }

  } else
  if ((yape_idx >= YAPEK_JOY_UP) &&
      (yape_idx <  YAPEK_MAX_KEY)) {
    yape_id = yape_idx - YAPEK_JOY_UP;
    if (press) c_yape_push_joystick(yape_id);
    else       c_yape_release_joystick(yape_id);

  }
  return 0;
}

int 
yape_kbd_reset()
{
  c_yape_reset_keyboard();
  return 0;
}

int
yape_get_key_from_ascii(int key_ascii)
{
  int index;
  for (index = 0; index < KBD_MAX_ENTRIES; index++) {
   if (kbd_layout[index][1] == key_ascii) return kbd_layout[index][0];
  }
  return -1;
}

int
psp_kbd_reset_mapping(void)
{
  memcpy(psp_kbd_mapping  , loc_default_mapping, sizeof(loc_default_mapping));
  memcpy(psp_kbd_mapping_L, loc_default_mapping_L, sizeof(loc_default_mapping_L));
  memcpy(psp_kbd_mapping_R, loc_default_mapping_R, sizeof(loc_default_mapping_R));
  return 0;
}

int
psp_kbd_load_mapping(char *kbd_filename)
{
  FILE    *KbdFile;
  int      error = 0;

  KbdFile = fopen(kbd_filename, "r");
  error   = 1;

  if (KbdFile != (FILE*)0) {
  psp_kbd_load_mapping_file(KbdFile);
  error = 0;
    fclose(KbdFile);
  }

  kbd_ltrigger_mapping_active = 0;
  kbd_rtrigger_mapping_active = 0;
    
  return error;
}

int
psp_kbd_load_mapping_file(FILE *KbdFile)
{
  char     Buffer[512];
  char    *Scan;
  int      tmp_mapping[KBD_ALL_BUTTONS];
  int      tmp_mapping_L[KBD_ALL_BUTTONS];
  int      tmp_mapping_R[KBD_ALL_BUTTONS];
  int      yape_key_id = 0;
  int      kbd_id = 0;

  memcpy(tmp_mapping  , loc_default_mapping  , sizeof(loc_default_mapping));
  memcpy(tmp_mapping_L, loc_default_mapping_L, sizeof(loc_default_mapping_R));
  memcpy(tmp_mapping_R, loc_default_mapping_R, sizeof(loc_default_mapping_R));

  while (fgets(Buffer,512,KbdFile) != (char *)0) {
      
      Scan = strchr(Buffer,'\n');
      if (Scan) *Scan = '\0';
      /* For this #@$% of windows ! */
      Scan = strchr(Buffer,'\r');
      if (Scan) *Scan = '\0';
      if (Buffer[0] == '#') continue;

      Scan = strchr(Buffer,'=');
      if (! Scan) continue;
    
      *Scan = '\0';
      yape_key_id = atoi(Scan + 1);

      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name[kbd_id])) {
          tmp_mapping[kbd_id] = yape_key_id;
          //break;
        }
      }
      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name_L[kbd_id])) {
          tmp_mapping_L[kbd_id] = yape_key_id;
          //break;
        }
      }
      for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++) {
        if (!strcasecmp(Buffer,loc_button_name_R[kbd_id])) {
          tmp_mapping_R[kbd_id] = yape_key_id;
          //break;
        }
      }
  }

  memcpy(psp_kbd_mapping, tmp_mapping, sizeof(psp_kbd_mapping));
  memcpy(psp_kbd_mapping_L, tmp_mapping_L, sizeof(psp_kbd_mapping_L));
  memcpy(psp_kbd_mapping_R, tmp_mapping_R, sizeof(psp_kbd_mapping_R));
  
  return 0;
}

int
psp_kbd_save_mapping(char *kbd_filename)
{
  FILE    *KbdFile;
  int      kbd_id = 0;
  int      error = 0;

  KbdFile = fopen(kbd_filename, "w");
  error   = 1;

  if (KbdFile != (FILE*)0) {

    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name[kbd_id], psp_kbd_mapping[kbd_id]);
    }
    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name_L[kbd_id], psp_kbd_mapping_L[kbd_id]);
    }
    for (kbd_id = 0; kbd_id < KBD_ALL_BUTTONS; kbd_id++)
    {
      fprintf(KbdFile, "%s=%d\n", loc_button_name_R[kbd_id], psp_kbd_mapping_R[kbd_id]);
    }
    error = 0;
    fclose(KbdFile);
  }

  return error;
}

int 
psp_kbd_is_danzeff_mode()
{
  return danzeff_mode;
}

int
psp_kbd_enter_danzeff()
{
  unsigned int danzeff_key = 0;
  int          yape_key     = 0;
  int          key_event   = 0;
  SceCtrlData  c;

  if (! danzeff_mode) {
    psp_init_keyboard();
    danzeff_mode = 1;
  }

  sceCtrlPeekBufferPositive(&c, 1);

  if (danzeff_yape_pending) 
  {
    if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_PENDING_TIME) {
      loc_last_event_time = c.TimeStamp;
      danzeff_yape_pending = 0;
      yape_key_event(danzeff_yape_key, 0);
    }

    return 0;
  }

  if ((c.TimeStamp - loc_last_event_time) > KBD_MIN_DANZEFF_TIME) {
    loc_last_event_time = c.TimeStamp;
  
    sceCtrlPeekBufferPositive(&c, 1);
    danzeff_key = danzeff_readInput(c);
  }

  if (danzeff_key > DANZEFF_START) {
    yape_key = yape_get_key_from_ascii(danzeff_key);

    if (yape_key != -1) {
      danzeff_yape_key     = yape_key;
      danzeff_yape_pending = 1;
      yape_key_event(danzeff_yape_key, 1);
    }

    return 1;

  } else if (danzeff_key == DANZEFF_START) {
    danzeff_mode       = 0;
    danzeff_yape_pending = 0;
    danzeff_yape_key     = 0;

    psp_kbd_wait_no_button();

  } else if (danzeff_key == DANZEFF_SELECT) {
    danzeff_mode       = 0;
    danzeff_yape_pending = 0;
    danzeff_yape_key     = 0;
    psp_main_menu();
    psp_init_keyboard();

    psp_kbd_wait_no_button();
  }

  return 0;
}

void
psp_kbd_display_active_mapping()
{
  if (kbd_ltrigger_mapping_active) {
    psp_sdl_fill_rectangle(0, 0, 10, 3, psp_sdl_rgb(0x0, 0x0, 0xff), 0);
  } else {
    psp_sdl_fill_rectangle(0, 0, 10, 3, 0x0, 0);
  }

  if (kbd_rtrigger_mapping_active) {
    psp_sdl_fill_rectangle(470, 0, 10, 3, psp_sdl_rgb(0x0, 0x0, 0xff), 0);
  } else {
    psp_sdl_fill_rectangle(470, 0, 10, 3, 0x0, 0);
  }
}

int
yape_decode_key(int psp_b, int button_pressed)
{
  int wake = 0;
  int reverse_analog = YAPE.psp_reverse_analog;

  if (reverse_analog) {
    if ((psp_b >= KBD_JOY_UP  ) &&
        (psp_b <= KBD_JOY_LEFT)) {
      psp_b = psp_b - KBD_JOY_UP + KBD_UP;
    } else
    if ((psp_b >= KBD_UP  ) &&
        (psp_b <= KBD_LEFT)) {
      psp_b = psp_b - KBD_UP + KBD_JOY_UP;
    }
  }

  if (psp_b == KBD_START) {
     if (button_pressed) psp_kbd_enter_danzeff();
  } else
  if (psp_b == KBD_SELECT) {
    if (button_pressed) {
      psp_main_menu();
      psp_init_keyboard();
    }
  } else {
 
    if (psp_kbd_mapping[psp_b] >= 0) {
      wake = 1;
      if (button_pressed) {
        // Determine which buton to press first (ie which mapping is currently active)
        if (kbd_ltrigger_mapping_active) {
          // Use ltrigger mapping
          psp_kbd_press[psp_b] = psp_kbd_mapping_L[psp_b];
          yape_key_event(psp_kbd_press[psp_b], button_pressed);
        } else
        if (kbd_rtrigger_mapping_active) {
          // Use rtrigger mapping
          psp_kbd_press[psp_b] = psp_kbd_mapping_R[psp_b];
          yape_key_event(psp_kbd_press[psp_b], button_pressed);
        } else {
          // Use standard mapping
          psp_kbd_press[psp_b] = psp_kbd_mapping[psp_b];
          yape_key_event(psp_kbd_press[psp_b], button_pressed);
        }
      } else {
          // Determine which button to release (ie what was pressed before)
          yape_key_event(psp_kbd_press[psp_b], button_pressed);
      }

    } else {
      if (psp_kbd_mapping[psp_b] == KBD_LTRIGGER_MAPPING) {
        kbd_ltrigger_mapping_active = button_pressed;
        kbd_rtrigger_mapping_active = 0;
      } else
      if (psp_kbd_mapping[psp_b] == KBD_RTRIGGER_MAPPING) {
        kbd_rtrigger_mapping_active = button_pressed;
        kbd_ltrigger_mapping_active = 0;
      }
    }
  }
  return 0;
}

# define ANALOG_THRESHOLD 60

void 
kbd_get_analog_direction(int Analog_x, int Analog_y, int *x, int *y)
{
  int DeltaX = 255;
  int DeltaY = 255;
  int DirX   = 0;
  int DirY   = 0;

  *x = 0;
  *y = 0;

  if (Analog_x <=        ANALOG_THRESHOLD)  { DeltaX = Analog_x; DirX = -1; }
  else 
  if (Analog_x >= (255 - ANALOG_THRESHOLD)) { DeltaX = 255 - Analog_x; DirX = 1; }

  if (Analog_y <=        ANALOG_THRESHOLD)  { DeltaY = Analog_y; DirY = -1; }
  else 
  if (Analog_y >= (255 - ANALOG_THRESHOLD)) { DeltaY = 255 - Analog_y; DirY = 1; }

  *x = DirX;
  *y = DirY;
}

static int 
kbd_reset_button_status(void)
{
  int b = 0;
  /* Reset Button status */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    loc_button_press[b]   = 0;
    loc_button_release[b] = 0;
  }
  psp_init_keyboard();
  return 0;
}

int
kbd_scan_keyboard(void)
{
  SceCtrlData c;
  long        delta_stamp;
  int         event;
  int         b;

  int new_Lx;
  int new_Ly;
  int old_Lx;
  int old_Ly;

  event = 0;
  sceCtrlPeekBufferPositive( &c, 1 );

  if ((c.Buttons & (PSP_CTRL_LTRIGGER|PSP_CTRL_RTRIGGER|PSP_CTRL_START)) ==
      (PSP_CTRL_LTRIGGER|PSP_CTRL_RTRIGGER|PSP_CTRL_START)) {
    /* Exit ! */
    psp_sdl_exit(0);
  }

  delta_stamp = c.TimeStamp - first_time_stamp;
  if ((delta_stamp < 0) || (delta_stamp > KBD_MIN_BATTCHECK_TIME)) {
    first_time_stamp = c.TimeStamp;
    if (psp_is_low_battery()) {
      psp_main_menu();
      psp_init_keyboard();
      return 0;
    }
  }

  /* Check Analog Device */
  kbd_get_analog_direction(loc_button_data.Lx,loc_button_data.Ly,&old_Lx,&old_Ly);
  kbd_get_analog_direction( c.Lx, c.Ly, &new_Lx, &new_Ly);

  /* Analog device has moved */
  if (new_Lx > 0) {
    if (old_Lx <  0) yape_decode_key(KBD_JOY_LEFT , 0);
    if (old_Lx <= 0) yape_decode_key(KBD_JOY_RIGHT, 1);
  } else 
  if (new_Lx < 0) {
    if (old_Lx >  0) yape_decode_key(KBD_JOY_RIGHT, 0);
    if (old_Lx >= 0) yape_decode_key(KBD_JOY_LEFT , 1);
  } else {
    if (old_Lx >  0) yape_decode_key(KBD_JOY_RIGHT , 0);
    else
    if (old_Lx <  0) yape_decode_key(KBD_JOY_LEFT, 0);
  }

  if (new_Ly > 0) {
    if (old_Ly <  0) yape_decode_key(KBD_JOY_UP , 0);
    if (old_Ly <= 0) yape_decode_key(KBD_JOY_DOWN, 1);
  } else 
  if (new_Ly < 0) {
    if (old_Ly >  0) yape_decode_key(KBD_JOY_DOWN, 0);
    if (old_Ly >= 0) yape_decode_key(KBD_JOY_UP , 1);
  } else {
    if (old_Ly >  0) yape_decode_key(KBD_JOY_DOWN , 0);
    else
    if (old_Ly <  0) yape_decode_key(KBD_JOY_UP, 0);
  }

  for (b = 0; b < KBD_MAX_BUTTONS; b++) 
  {
    if (c.Buttons & loc_button_mask[b]) {
      if (!(loc_button_data.Buttons & loc_button_mask[b])) 
      {
        loc_button_press[b] = 1;
        event = 1;
      }
    } else {
      if (loc_button_data.Buttons & loc_button_mask[b]) {
        loc_button_release[b] = 1;
        event = 1;
      }
    }
  }
  memcpy(&loc_button_data,&c,sizeof(SceCtrlData));

  return event;
}

void
psp_kbd_wait_start(void)
{
  while (1)
  {
    SceCtrlData c;
    sceCtrlReadBufferPositive(&c, 1);
    if (c.Buttons & PSP_CTRL_START) break;
  }
  psp_kbd_wait_no_button();
}

void
psp_init_keyboard(void)
{
  yape_kbd_reset();
  kbd_ltrigger_mapping_active = 0;
  kbd_rtrigger_mapping_active = 0;
}

void
psp_kbd_wait_no_button(void)
{
  SceCtrlData c;

  do {
   sceCtrlPeekBufferPositive(&c, 1);
  } while (c.Buttons != 0);
} 

void
psp_kbd_wait_button(void)
{
  SceCtrlData c;

  do {
   sceCtrlReadBufferPositive(&c, 1);
  } while (c.Buttons == 0);
} 

int
psp_update_keys(void)
{
  int         b;

  static char first_time = 1;
  static int release_pending = 0;

  if (first_time) {

    memcpy(psp_kbd_mapping, loc_default_mapping, sizeof(loc_default_mapping));
    memcpy(psp_kbd_mapping_L, loc_default_mapping_L, sizeof(loc_default_mapping_L));
    memcpy(psp_kbd_mapping_R, loc_default_mapping_R, sizeof(loc_default_mapping_R));

    yape_kbd_load();

    SceCtrlData c;
    sceCtrlPeekBufferPositive(&c, 1);

    if (first_time_stamp == -1) first_time_stamp = c.TimeStamp;
    if ((! c.Buttons) && ((c.TimeStamp - first_time_stamp) < KBD_MIN_START_TIME)) return 0;

    first_time      = 0;
    release_pending = 0;

    for (b = 0; b < KBD_MAX_BUTTONS; b++) {
      loc_button_release[b] = 0;
      loc_button_press[b] = 0;
    }
    sceCtrlPeekBufferPositive(&loc_button_data, 1);

    psp_main_menu();
    psp_init_keyboard();

    return 0;
  }

  if (danzeff_mode) {
    return psp_kbd_enter_danzeff();
  }

  if (release_pending)
  {
    release_pending = 0;
    for (b = 0; b < KBD_MAX_BUTTONS; b++) {
      if (loc_button_release[b]) {
        loc_button_release[b] = 0;
        loc_button_press[b] = 0;
        yape_decode_key(b, 0);
      }
    }
  }

  kbd_scan_keyboard();

  /* check press event */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    if (loc_button_press[b]) {
      loc_button_press[b] = 0;
      release_pending     = 0;
      yape_decode_key(b, 1);
    }
  }
  /* check release event */
  for (b = 0; b < KBD_MAX_BUTTONS; b++) {
    if (loc_button_release[b]) {
      release_pending = 1;
      break;
    } 
  }

  return 0;
}
