/***********************************************************************************
 **
 ** Atari++ emulator (c) 2002 THOR-Software, Thomas Richter
 **
 ** $Id: display.hpp,v 1.21 2005/09/10 12:55:40 thor Exp $
 **
 ** In this module: Interface definition for the visual display frontend
 **********************************************************************************/

#ifndef DISPLAY_HPP
#define DISPLAY_HPP

/// Includes
#include "machine.hpp"
#include "chip.hpp"
#include "vbiaction.hpp"
///

/// Forwards
class ArgParser;
class Monitor;
///

/// Class AtariDisplay
// This class defines the generic interface for all graphic front-ends
class AtariDisplay : public Chip, public VBIAction {
  //
  //
  // Implement the frequent VBI activity of the display: This should
  // update the display and read inputs
  virtual void VBI(class Timer *time,bool quick,bool pause) = 0;
  //
protected:
  // Unit number of this display. Zero is the regular display, 
  // One is the 80 character display.
  int Unit;
  //
public:
  AtariDisplay(class Machine *mach,int unit);
  virtual ~AtariDisplay(void);
  //
  // Packed RGB color
  typedef LONG PackedRGB;
  //
  // Coldstart and warmstart methods for the chip class.
  virtual void ColdStart(void) = 0;
  virtual void WarmStart(void) = 0;
  //
  // Argument parser frontends
  virtual void ParseArgs(class ArgParser *args)  = 0;
  virtual void DisplayStatus(class Monitor *mon) = 0;
  //
  // Return the active buffer we must render into
  virtual UBYTE *ActiveBuffer(void) = 0;
  //
  // Return the next output buffer for the next scanline to generate.
  // This either returns a temporary buffer to place the data in, or
  // a row in the final output buffer, depending on how the display generation
  // process works.
  virtual UBYTE *NextScanLine(void) = 0;
  //
  // Similar to the above, but provide the buffer for RGB true color
  // display. If there is no true color, return NULL.
  virtual PackedRGB *NextRGBScanLine(void)
  {
    return NULL;
  }
  //
  // Push the output buffer back into the display process. This signals that the
  // currently generated display line is ready.
  virtual void PushLine(UBYTE *buffer,int size) = 0;
  //
  // Ditto for RGB/TrueColor displays
  virtual void PushRGBLine(PackedRGB *,int)
  {
  }
  //
  // Signal that we start the display generation again from top. Hence, this implements
  // some kind of "vertical sync" signal for the display generation.
  virtual void ResetVertical(void) = 0;
  //
  // If we want to emulate a flickering drive LED, do it here.
  virtual void SetLED(bool on) = 0;
  //
  // Enforce a full display refresh, i.e. after the monitor returnes
  virtual void EnforceFullRefresh(void) = 0;
  //
  // Enforce the screen to foreground or background to run the monitor
  virtual void SwitchScreen(bool foreground) = 0;
  //
  // Enable or disable the mouse pointer
  virtual void ShowPointer(bool showit) = 0;
  //
  // Get the layout of the buffer
  virtual void BufferDimensions(LONG &leftedge,LONG &topedge,
				LONG &width,LONG &height,LONG &modulo) = 0;
  //
  // For GUI frontends within this buffer: Get the position and the status
  // of the mouse within this buffer measured in internal coordinates.
  virtual void MousePosition(LONG &x,LONG &y,bool &button) = 0;
  //
  // Return the current scroll distance generated by a mouse wheel.
  virtual int ScrollDistance() = 0;
  //
  // For GUI frontends: Check whether the mouse is available as input
  // device. This returns false in case the mouse is an input device
  // and has been grabbed on its "GamePort".
  virtual bool MouseIsAvailable(void) = 0;
  //
  // Signal the requirement for refresh in the indicated region. This is only
  // used by the GUI engine as the emulator core uses PushLine for that.
  virtual void SignalRect(LONG leftedge,LONG topedge,LONG width,LONG height) = 0;
  //
  // Disable or enable double buffering temporarely for the user
  // menu. Default is to have it enabled.
  virtual void EnableDoubleBuffer(bool enable) = 0;
  //
  // Test whether the user requests the menu by pointing the mouse into
  // the menu bar and holding the button. Returns true if so.
  bool MenuVerify(void);
};
///

///
#endif
