// Part of SimCoupe - A SAM Coupe emulator
//
// Display.cpp: SDL display rendering
//
//  Copyright (c) 1999-2006  Simon Owen
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// ToDo:
//  - change to handle multiple dirty regions

#include "SimCoupe.h"
#include "simcoupec.h"
#include <pspdisplay.h>

#include "Display.h"
#include "Frame.h"
#include "GUI.h"
#include "Options.h"
#include "Profile.h"
#include "Video.h"
#include "UI.h"

#include "psp_kbd.h"
#include "psp_sdl.h"
#include "psp_danzeff.h"

bool* Display::pafDirty;
SDL_Rect rSource, rTarget;

////////////////////////////////////////////////////////////////////////////////

// Writing to the display in DWORDs makes it endian sensitive, so we need to cover both cases
#if SDL_BYTEORDER == SDL_LIL_ENDIAN

inline DWORD PaletteDWORD (BYTE b1_, BYTE b2_, BYTE b3_, BYTE b4_, DWORD* pulPalette_)
    { return (pulPalette_[b4_] << 24) | (pulPalette_[b3_] << 16) | (pulPalette_[b2_] << 8) | pulPalette_[b1_]; }
inline DWORD PaletteDWORD (BYTE b1_, BYTE b2_, DWORD* pulPalette_)
    { return (pulPalette_[b2_] << 16) | pulPalette_[b1_]; }
inline DWORD MakeDWORD (BYTE b1_, BYTE b2_, BYTE b3_, BYTE b4_)
    { return (b4_ << 24) | (b3_ << 16) | (b2_ << 8) | b1_; }
inline DWORD MakeDWORD (BYTE b1_, BYTE b2_)
    { return ((b2_ << 16) | b1_) * 0x0101UL; }
inline DWORD JoinWORDs (DWORD w1_, DWORD w2_)
    { return ((w2_ << 16) | w1_); }
#else

inline DWORD PaletteDWORD (BYTE b1_, BYTE b2_, BYTE b3_, BYTE b4_, DWORD* pulPalette_)
    { return (pulPalette_[b1_] << 24) | (pulPalette_[b2_] << 16) | (pulPalette_[b3_] << 8) | pulPalette_[b4_]; }
inline DWORD PaletteDWORD (BYTE b1_, BYTE b2_, DWORD* pulPalette_)
    { return (pulPalette_[b1_] << 16) | pulPalette_[b2_]; }
inline DWORD MakeDWORD (BYTE b1_, BYTE b2_, BYTE b3_, BYTE b4_)
    { return (b1_ << 24) | (b2_ << 16) | (b3_ << 8) | b4_; }
inline DWORD MakeDWORD (BYTE b1_, BYTE b2_)
    { return ((b1_ << 16) | b2_) * 0x0101UL; }
inline DWORD JoinWORDs (DWORD w1_, DWORD w2_)
    { return ((w1_ << 16) | w2_); }
#endif

////////////////////////////////////////////////////////////////////////////////

bool Display::Init (bool fFirstInit_/*=false*/)
{
    Exit(true);

    pafDirty = new bool[Frame::GetHeight()];

    // These will be updated to the appropriate values on the first draw
    rSource.w = rTarget.w = Frame::GetWidth();
    rSource.h = rTarget.h = Frame::GetHeight() << 1;

    return Video::Init(fFirstInit_);
}

void Display::Exit (bool fReInit_/*=false*/)
{
    Video::Exit(fReInit_);

    if (pafDirty) { delete[] pafDirty; pafDirty = NULL; }
}

void Display::SetDirty ()
{
    // Mark all display lines dirty
    for (int i = 0, nHeight = Frame::GetHeight() ; i < nHeight ; i++)
        pafDirty[i] = true;
}

static void
loc_Draw_blit(CScreen* pScreen_)
{
  SDL_Surface* pSurface_ = blit_surface;
  short *pdsStart = reinterpret_cast<short *>(pSurface_->pixels);
  short *pdsBack  = pdsStart;

  long lPitchDW = pSurface_->pitch >> 1;
  bool *pfDirty = Display::pafDirty, *pfHiRes = pScreen_->GetHiRes();

  BYTE *pbSAM = pScreen_->GetLine(0);
  BYTE *pb    = pbSAM;
  long lPitch = pScreen_->GetPitch();

  int nShift  = 1;
  int nDepth  = pSurface_->format->BitsPerPixel;
  int nBottom = pScreen_->GetHeight() >> nShift;
  int nWidth  = pScreen_->GetPitch(), nRightHi = nWidth, nRightLo = nRightHi >> 1;

  nWidth <<= 1;

  int delta_y = (272 - nBottom) / 2;
  int color   = 0;

  for (int y = 0 ; y < nBottom; y++)
  {
    pdsBack = pdsStart + (lPitchDW * (y + delta_y));
    short *pds = pdsBack;

    int delta_x = 0;
    int max_x   = 0;

    if (pfHiRes[y]) { 
      delta_x = (480 - nRightHi)  / 2;
      if (delta_x < 0) delta_x = 0;
      max_x   = nRightHi;
    } else {
      delta_x = (480 - nRightLo)  / 2;
      if (delta_x < 0) delta_x = 0;
      max_x   = nRightLo;
    }

    if (delta_x > 0) {
      memset(pds, 0, delta_x * sizeof(short));
    }

    for (int x = 0 ; x < (480 - delta_x); x++)
    {
      if (x >= max_x) color = 0;
      else            color = aulPalette[pb[x]];
      pds[x + delta_x] = color;
    }
    pb  = pbSAM;
    pds = (short *)(pdsBack + lPitchDW);

    if (delta_x > 0) {
      memset(pds, 0, delta_x * sizeof(short));
    }
  
    for (int x = 0 ; x < (480 - delta_x); x++)
    {
      if (x >= max_x) color = 0;
      else            color = aulScanline[pb[x]];
      pds[x + delta_x] = color;
    }
    pb  = (pbSAM   += lPitch);
  }
}

static void
loc_Draw_gu_Normal(CScreen* pScreen_, SDL_Surface* pSurface_)
{
  SDL_Rect srcRect;
  SDL_Rect dstRect;

  srcRect.x = 0;
  srcRect.y = 0;
  srcRect.w = 480;
  srcRect.h = 272;
  dstRect.x = 0;
  dstRect.y = 0;
  dstRect.w = 480;
  dstRect.h = 272;

  loc_Draw_blit(pScreen_);
  psp_sdl_gu_stretch(&srcRect, &dstRect);
}

static void
loc_Draw_gu_X15(CScreen* pScreen_, SDL_Surface* pSurface_)
{
  SDL_Rect srcRect;
  SDL_Rect dstRect;

  int nShift  = 1;
  int nBottom = pScreen_->GetHeight() >> nShift;

  srcRect.x = 80;
  srcRect.y = 16;
  srcRect.w = 320;
  srcRect.h = 240;
  dstRect.x = 0;
  dstRect.y = 0;
  dstRect.w = 480;
  dstRect.h = 272;

  loc_Draw_blit(pScreen_);
  psp_sdl_gu_stretch(&srcRect, &dstRect);
}

static void
loc_Draw_gu_X175(CScreen* pScreen_, SDL_Surface* pSurface_)
{
  SDL_Rect srcRect;
  SDL_Rect dstRect;

  srcRect.x = 102;
  srcRect.y = 33;
  srcRect.w = 274;
  srcRect.h = 206;
  dstRect.x = 0;
  dstRect.y = 0;
  dstRect.w = 480;
  dstRect.h = 272;

  loc_Draw_blit(pScreen_);
  psp_sdl_gu_stretch(&srcRect, &dstRect);
}

static void
loc_Draw_gu_X2(CScreen* pScreen_, SDL_Surface* pSurface_)
{
  SDL_Rect srcRect;
  SDL_Rect dstRect;

  srcRect.x = 120;
  srcRect.y = 68;
  srcRect.w = 240;
  srcRect.h = 136;
  dstRect.x = 0;
  dstRect.y = 0;
  dstRect.w = 480;
  dstRect.h = 272;

  loc_Draw_blit(pScreen_);
  psp_sdl_gu_stretch(&srcRect, &dstRect);
}

// Draw the changed lines in the appropriate colour depth and hi/low resolution
bool 
DrawChanges (CScreen* pScreen_, SDL_Surface* pSurface_)
{
  ProfileStart(Gfx);

  int RenderMode = GetOption(render_mode);

  if (RenderMode == SIM_RENDER_NORMAL) loc_Draw_gu_Normal(pScreen_, pSurface_);
  else
  if (RenderMode == SIM_RENDER_X15) loc_Draw_gu_X15(pScreen_, pSurface_);
  else
  if (RenderMode == SIM_RENDER_X175) loc_Draw_gu_X175(pScreen_, pSurface_);
  else
  if (RenderMode == SIM_RENDER_X2) loc_Draw_gu_X2(pScreen_, pSurface_);

  //LUDO:
  if (psp_kbd_is_danzeff_mode()) {
    sceDisplayWaitVblankStart();
    danzeff_moveTo(-130, -50);
    danzeff_render();
  }

  ProfileEnd();

  // Success
  return true;
}


void
psp_sim_synchronize(int speed_limiter)
{
	static u32 nextclock = 1;

  if (speed_limiter) {

	  if (nextclock) {
		  u32 curclock;
		  do {
        curclock = SDL_GetTicks();
		  } while (curclock < nextclock);
  
      nextclock = curclock + (u32)( 1000 / speed_limiter);
    }
  }
}

static int sim_current_fps = 0;

void
psp_sim_update_fps()
{
  static u32 next_sec_clock = 0;
  static u32 cur_num_frame = 0;
  cur_num_frame++;
  u32 curclock = SDL_GetTicks();
  if (curclock > next_sec_clock) {
    next_sec_clock = curclock + 1000;
    sim_current_fps = cur_num_frame;
    cur_num_frame = 0;
  }
}

// Update the display to show anything that's changed since last time
void Display::Update (CScreen* pScreen_)
{
# if 0 //LUDO:
    // Don't draw if fullscreen but not active
    if (GetOption(fullscreen) && !g_fActive)
        return;
# endif
    // Draw any changed lines
    DrawChanges(pScreen_, back_surface);

    int view_fps = GetOption(view_fps);
    if (view_fps) {
      char buffer[32];
      sprintf(buffer, "%3d", (int)sim_current_fps);
      psp_sdl_fill_print(0, 0, buffer, 0xffffff, 0 );
    }

    int display_lr = GetOption(display_lr);
    if (display_lr) {
      psp_kbd_display_active_mapping();
    }
    psp_sdl_flip();

    if (psp_screenshot_mode) {
      psp_screenshot_mode--;
      if (psp_screenshot_mode <= 0) {
        psp_sdl_save_screenshot();
        psp_screenshot_mode = 0;
      }
    }
}


// Scale a client size/movement to one relative to the SAM view port size
// Should round down and be consistent with positive and negative values
void Display::DisplayToSamSize (int* pnX_, int* pnY_)
{
    int nHalfWidth = !GUI::IsActive(), nHalfHeight = 0;

    *pnX_ = *pnX_ * (rSource.w >> nHalfWidth)  / rTarget.w;
    *pnY_ = *pnY_ * (rSource.h >> nHalfHeight) / rTarget.h;
}

// Scale a size/movement in the SAM view port to one relative to the client
// Should round down and be consistent with positive and negative values
void Display::SamToDisplaySize (int* pnX_, int* pnY_)
{
    int nHalfWidth = !GUI::IsActive(), nHalfHeight = 0;

    *pnX_ = *pnX_ * rTarget.w / (rSource.w >> nHalfWidth);
    *pnY_ = *pnY_ * rTarget.h / (rSource.h >> nHalfHeight);
}

// Map a client point to one relative to the SAM view port
void Display::DisplayToSamPoint (int* pnX_, int* pnY_)
{
    *pnX_ -= rTarget.x;
    *pnY_ -= rTarget.y;
    DisplayToSamSize(pnX_, pnY_);
}

// Map a point in the SAM view port to a point relative to the client position
void Display::SamToDisplayPoint (int* pnX_, int* pnY_)
{
    SamToDisplaySize(pnX_, pnY_);
    *pnX_ += rTarget.x;
    *pnY_ += rTarget.y;
}
