/* gameplaySP
 *
 * Copyright (C) 2006 Exophase <exophase@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "SDL.h"
#include "common.h"

#include <stdio.h>

u16 palette_ram[512];
u16 oam_ram[512];

// Special thanks to psp298 for the analog->dpad code!

void trigger_key(u32 key)
{
  u32 p1_cnt = io_registers[REG_P1CNT];
  u32 p1;

  if((p1_cnt >> 14) & 0x01)
  {
    u32 key_intersection = (p1_cnt & key) & 0x3FF;

    if(p1_cnt >> 15)
    {
      if(key_intersection == (p1_cnt & 0x3FF))
        raise_interrupt(IRQ_KEYPAD);
    }
    else
    {
      if(key_intersection)
        raise_interrupt(IRQ_KEYPAD);
    }
  }
}

u32 key = 0;

u32 gamepad_config_map[12] =
{
  BUTTON_ID_FRAMESKIP,
  BUTTON_ID_A,
  BUTTON_ID_B,
  BUTTON_ID_START,
  BUTTON_ID_L,
  BUTTON_ID_R,
  BUTTON_ID_DOWN,
  BUTTON_ID_LEFT,
  BUTTON_ID_UP,
  BUTTON_ID_RIGHT,
  BUTTON_ID_SELECT,
  BUTTON_ID_START
};

#ifdef PSP_BUILD

#define PSP_ALL_BUTTON_MASK 0xFFFF

u32 last_buttons = 0;
u32 button_repeat_timestamp;

typedef enum
{
  BUTTON_NOT_HELD,
  BUTTON_HELD_INITIAL,
  BUTTON_HELD_REPEAT
} button_repeat_state_type;

button_repeat_state_type button_repeat_state = BUTTON_NOT_HELD;
u32 button_repeat = 0;
gui_action_type cursor_repeat = CURSOR_NONE;

#define BUTTON_REPEAT_START    200
#define BUTTON_REPEAT_CONTINUE 50

gui_action_type get_gui_input()
{
  SceCtrlData ctrl_data;
  gui_action_type new_button = CURSOR_NONE;
  u32 new_buttons;

  SDL_Delay(30);

  sceCtrlPeekBufferPositive(&ctrl_data, 1);
  ctrl_data.Buttons &= PSP_ALL_BUTTON_MASK;
  new_buttons = (last_buttons ^ ctrl_data.Buttons) & ctrl_data.Buttons;
  last_buttons = ctrl_data.Buttons;

  if((new_buttons & PSP_CTRL_LEFT) || (ctrl_data.Lx < 64))
    new_button = CURSOR_LEFT;

  if((new_buttons & PSP_CTRL_RIGHT) || (ctrl_data.Lx > 196))
    new_button = CURSOR_RIGHT;

  if((new_buttons & PSP_CTRL_UP) || (ctrl_data.Ly < 64))
    new_button = CURSOR_UP;

  if((new_buttons & PSP_CTRL_DOWN) || (ctrl_data.Ly > 196))
    new_button = CURSOR_DOWN;

  if(new_buttons & PSP_CTRL_START)
    new_button = CURSOR_SELECT;

  if(new_buttons & PSP_CTRL_CIRCLE)
    new_button = CURSOR_SELECT;

  if(new_buttons & PSP_CTRL_CROSS)
    new_button = CURSOR_EXIT;

  if(new_buttons & PSP_CTRL_SQUARE)
    new_button = CURSOR_BACK;

  if(new_button != CURSOR_NONE)
  {
    button_repeat_timestamp = SDL_GetTicks();
    button_repeat_state = BUTTON_HELD_INITIAL;
    button_repeat = new_buttons;
    cursor_repeat = new_button;
  }
  else
  {
    if(ctrl_data.Buttons & button_repeat)
    {
      u32 new_ticks = SDL_GetTicks();

      if(button_repeat_state == BUTTON_HELD_INITIAL)
      {
        if((new_ticks - button_repeat_timestamp) >
         BUTTON_REPEAT_START)
        {
          new_button = cursor_repeat;
          button_repeat_timestamp = new_ticks;
          button_repeat_state = BUTTON_HELD_REPEAT;
        }
      }

      if(button_repeat_state == BUTTON_HELD_REPEAT)
      {
        if((new_ticks - button_repeat_timestamp) >
         BUTTON_REPEAT_CONTINUE)
        {
          new_button = cursor_repeat;
          button_repeat_timestamp = new_ticks;
        }
      }
    }
  }

  return new_button;
}

u32 button_psp_mask_to_config[] =
{
  PSP_CTRL_TRIANGLE,
  PSP_CTRL_CIRCLE,
  PSP_CTRL_CROSS,
  PSP_CTRL_SQUARE,
  PSP_CTRL_LTRIGGER,
  PSP_CTRL_RTRIGGER,
  PSP_CTRL_DOWN,
  PSP_CTRL_LEFT,
  PSP_CTRL_UP,
  PSP_CTRL_RIGHT,
  PSP_CTRL_SELECT,
  PSP_CTRL_START
};

u32 button_id_to_gba_mask[] =
{
  BUTTON_UP,
  BUTTON_DOWN,
  BUTTON_LEFT,
  BUTTON_RIGHT,
  BUTTON_A,
  BUTTON_B,
  BUTTON_L,
  BUTTON_R,
  BUTTON_START,
  BUTTON_SELECT,
  BUTTON_NONE,
  BUTTON_NONE,
  BUTTON_NONE,
  BUTTON_NONE
};

gui_action_type get_gui_input_fs_hold(u32 button_id)
{
  gui_action_type new_button = get_gui_input();
  if((last_buttons & button_psp_mask_to_config[button_id]) == 0)
    return CURSOR_BACK;

  return new_button;
}

u32 update_input()
{
  SceCtrlData ctrl_data;
  u32 buttons;
  u32 button_id;
  u32 i;
  u32 new_key = 0;

  sceCtrlPeekBufferPositive(&ctrl_data, 1);

  buttons = ctrl_data.Buttons & PSP_ALL_BUTTON_MASK;

  if(ctrl_data.Lx < 64)
    buttons |= PSP_CTRL_LEFT;

  if(ctrl_data.Lx > 196)
    buttons |= PSP_CTRL_RIGHT;

  if(ctrl_data.Ly < 64)
    buttons |= PSP_CTRL_UP;

  if(ctrl_data.Ly > 196)
    buttons |= PSP_CTRL_DOWN;

  for(i = 0; i < 12; i++)
  {
    if(buttons & (button_psp_mask_to_config[i]))
    {
      button_id = gamepad_config_map[i];
      switch(button_id)
      {
        case BUTTON_ID_FRAMESKIP:
          return adjust_frameskip(i);

        case BUTTON_ID_MENU:
          return menu();

        case BUTTON_ID_FASTFORWARD:
          synchronize_flag ^= 1;
          return 0;

        default:
          new_key |= button_id_to_gba_mask[button_id];
          break;
      }
    }
  }

  if((new_key | key) != key)
    trigger_key(new_key);

  key = new_key;

  io_registers[REG_P1] = (~key) & 0x3FF;

  return 0;
}

void init_input()
{
  sceCtrlSetSamplingCycle(0);
  sceCtrlSetSamplingMode(PSP_CTRL_MODE_ANALOG);
}

#else

u32 key_map(SDLKey key_sym)
{
  switch(key_sym)
  {
    case SDLK_LSHIFT:
      return BUTTON_L;

    case SDLK_x:
      return BUTTON_R;

    case SDLK_DOWN:
      return BUTTON_DOWN;

    case SDLK_UP:
      return BUTTON_UP;

    case SDLK_LEFT:
      return BUTTON_LEFT;

    case SDLK_RIGHT:
      return BUTTON_RIGHT;

    case SDLK_RETURN:
      return BUTTON_START;

    case SDLK_RSHIFT:
      return BUTTON_SELECT;

    case SDLK_LCTRL:
      return BUTTON_B;

    case SDLK_LALT:
      return BUTTON_A;

    default:
      return BUTTON_NONE;
  }
}

u32 joy_map(u32 button)
{
  switch(button)
  {
    case 4:
      return BUTTON_L;

    case 5:
      return BUTTON_R;

    case 9:
      return BUTTON_START;

    case 8:
      return BUTTON_SELECT;

    case 0:
      return BUTTON_B;

    case 1:
      return BUTTON_A;

    default:
      return BUTTON_NONE;
  }
}


u32 joy_hat_map(u32 value)
{
  u32 result = BUTTON_NONE;

  if(value & SDL_HAT_UP)
    result |= BUTTON_UP;

  if(value & SDL_HAT_RIGHT)
    result |= BUTTON_RIGHT;

  if(value & SDL_HAT_DOWN)
    result |= BUTTON_DOWN;

  if(value & SDL_HAT_LEFT)
    result |= BUTTON_LEFT;

  return result;
}

u32 last_axis_values[16];

u32 joy_axis_map(u32 value, u32 axis)
{
  s32 digital_value = -1;
  u32 last_axis_value = last_axis_values[axis];

  if(value > 5000)
    digital_value = 1;
  else

  if(value < -5000)
    digital_value = 0;

  if(digital_value != -1)
  {
    last_axis_values[axis] = digital_value;

    if(last_axis_value == (digital_value ^ 1))
      return BUTTON_NONE;

    switch((axis << 1) | digital_value)
    {
      case 0x00:
        return BUTTON_LEFT;

      case 0x01:
        return BUTTON_RIGHT;

      case 0x02:
        return BUTTON_UP;

      case 0x03:
        return BUTTON_DOWN;
    }
  }

  return BUTTON_NONE;
}

gui_action_type get_gui_input()
{
  SDL_Event event;
  gui_action_type gui_action = CURSOR_NONE;

  SDL_Delay(30);

  while(SDL_PollEvent(&event))
  {
    switch(event.type)
    {
      case SDL_QUIT:
        quit();

      case SDL_KEYDOWN:
      {
        switch(event.key.keysym.sym)
        {
          case SDLK_ESCAPE:
            gui_action = CURSOR_EXIT;
            break;

          case SDLK_DOWN:
            gui_action = CURSOR_DOWN;
            break;

          case SDLK_UP:
            gui_action = CURSOR_UP;
            break;

          case SDLK_LEFT:
            gui_action = CURSOR_LEFT;
            break;

          case SDLK_RIGHT:
            gui_action = CURSOR_RIGHT;
            break;

          case SDLK_RETURN:
            gui_action = CURSOR_SELECT;
            break;

          case SDLK_BACKSPACE:
            gui_action = CURSOR_BACK;
            break;
        }
        break;
      }
    }
  }

  return gui_action;
}

// FIXME: Not implemented properly for x86 version.

gui_action_type get_gui_input_fs_hold(u32 button_id)
{
  return get_gui_input();
}

u32 update_input()
{
  SDL_Event event;

  while(SDL_PollEvent(&event))
  {
    switch(event.type)
    {
      case SDL_QUIT:
        quit();

      case SDL_KEYDOWN:
      {
        if(event.key.keysym.sym == SDLK_ESCAPE)
        {
          quit();
        }

        if(event.key.keysym.sym == SDLK_BACKSPACE)
        {
          return adjust_frameskip(0);
        }
        else

        if(event.key.keysym.sym == SDLK_F1)
        {
          current_debug_state = STEP;
        }
        else

        if(event.key.keysym.sym == SDLK_F2)
        {
          FILE *fp = fopen("palette_ram.bin", "wb");
          printf("writing palette RAM\n");
          fwrite(palette_ram, 1024, 1, fp);
          fclose(fp);
          printf("writing palette VRAM\n");
          fp = fopen("vram.bin", "wb");
          fwrite(vram, 1024 * 96, 1, fp);
          fclose(fp);
          printf("writing palette OAM RAM\n");
          fp = fopen("oam_ram.bin", "wb");
          fwrite(oam_ram, 1024, 1, fp);
          fclose(fp);
          printf("writing palette I/O registers\n");
          fp = fopen("io_registers.bin", "wb");
          fwrite(io_registers, 1024, 1, fp);
          fclose(fp);
        }
        else

        if(event.key.keysym.sym == SDLK_F3)
        {
          dump_translation_cache();
        }
        else

        if(event.key.keysym.sym == SDLK_BACKQUOTE)
        {
          synchronize_flag ^= 1;
        }
        else
        {
          key |= key_map(event.key.keysym.sym);
          trigger_key(key);
        }

        break;
      }

      case SDL_KEYUP:
      {
        key &= ~(key_map(event.key.keysym.sym));
        break;
      }

      case SDL_JOYAXISMOTION:
      {
/*        key = (key & 0xFF0F) | joy_axis_map(event.jaxis.value,
         event.jaxis.axis);
        trigger_key(key); */
        break;
      }

      case SDL_JOYHATMOTION:
      {
        key = (key & 0xFF0F) | joy_hat_map(event.jhat.value);
        trigger_key(key);
        break;
      }

      case SDL_JOYBUTTONDOWN:
      {
        key |= joy_map(event.jbutton.button);
        trigger_key(key);
        break;
      }

      case SDL_JOYBUTTONUP:
      {
        key &= ~(joy_map(event.jbutton.button));
        break;
      }
    }
  }

  io_registers[REG_P1] = (~key) & 0x3FF;

  return 0;
}

void init_input()
{
  u32 joystick_count = SDL_NumJoysticks();

  if(joystick_count > 0)
  {
    SDL_JoystickOpen(0);
    SDL_JoystickEventState(SDL_ENABLE);
  }
}

#endif
