/* gameplaySP
 *
 * Copyright (C) 2006 Exophase <exophase@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef CPU_H
#define CPU_H

// System mode and user mode are represented as the same here

typedef enum
{
  MODE_USER,
  MODE_IRQ,
  MODE_FIQ,
  MODE_SUPERVISOR,
  MODE_ABORT,
  MODE_UNDEFINED,
  MODE_INVALID
} cpu_mode_type;

typedef enum
{
  CPU_ALERT_NONE,
  CPU_ALERT_HALT,
  CPU_ALERT_SMC
} cpu_alert_type;

typedef enum
{
  CPU_ACTIVE,
  CPU_HALT,
  CPU_STOP
} cpu_halt_type;

typedef enum
{
  IRQ_NONE = 0x0000,
  IRQ_VBLANK = 0x0001,
  IRQ_HBLANK = 0x0002,
  IRQ_VCOUNT = 0x0004,
  IRQ_TIMER0 = 0x0008,
  IRQ_TIMER1 = 0x0010,
  IRQ_TIMER2 = 0x0020,
  IRQ_TIMER3 = 0x0040,
  IRQ_SERIAL = 0x0080,
  IRQ_DMA0 = 0x0100,
  IRQ_DMA1 = 0x0200,
  IRQ_DMA2 = 0x0400,
  IRQ_DMA3 = 0x0800,
  IRQ_KEYPAD = 0x1000,
  IRQ_GAMEPAK = 0x2000,
} irq_type;

typedef enum
{
  REG_SP     = 13,
  REG_LR     = 14,
  REG_PC     = 15,
  REG_N_FLAG = 16,
  REG_Z_FLAG = 17,
  REG_C_FLAG = 18,
  REG_V_FLAG = 19
} ext_reg_numbers;

typedef enum
{
  STEP,
  PC_BREAKPOINT,
  VCOUNT_BREAKPOINT,
  Z_BREAKPOINT,
  COUNTDOWN_BREAKPOINT,
  COUNTDOWN_BREAKPOINT_B,
  STEP_RUN,
  RUN
} debug_state;

typedef enum
{
  TRANSLATION_REGION_RAM,
  TRANSLATION_REGION_ROM,
  TRANSLATION_REGION_BIOS
} translation_region_type;

extern debug_state current_debug_state;
extern u32 instruction_count;

u32 function_cc step_debug(u32 pc, u32 cycles);
u32 execute_arm(u32 cycles);
void raise_interrupt(irq_type irq_raised);

u32 function_cc execute_load_u8(u32 address);
u32 function_cc execute_load_u16(u32 address);
u32 function_cc execute_load_u32(u32 address);
u32 function_cc execute_load_s8(u32 address);
u32 function_cc execute_load_s16(u32 address);
void function_cc execute_store_u8(u32 address, u32 source);
void function_cc execute_store_u16(u32 address, u32 source);
void function_cc execute_store_u32(u32 address, u32 source);
u32 function_cc execute_arm_translate(u32 cycles);
void init_translater();

u8 function_cc *block_lookup_address_arm(u32 pc);
u8 function_cc *block_lookup_address_thumb(u32 pc);
s32 translate_block_arm(u32 pc, translation_region_type translation_region,
 u8 *smc_pointer);
s32 translate_block_thumb(u32 pc, translation_region_type translation_region,
 u8 *smc_pointer);

#define ROM_TRANSLATION_CACHE_SIZE (1024 * 512 * 4)
#define RAM_TRANSLATION_CACHE_SIZE (1024 * 384)
#define BIOS_TRANSLATION_CACHE_SIZE (1024 * 128)
#define TRANSLATION_CACHE_LIMIT_THRESHOLD (1024)

extern u8 rom_translation_cache[ROM_TRANSLATION_CACHE_SIZE];
extern u8 ram_translation_cache[RAM_TRANSLATION_CACHE_SIZE];
extern u8 bios_translation_cache[BIOS_TRANSLATION_CACHE_SIZE];
extern u8 *rom_translation_ptr;
extern u8 *ram_translation_ptr;
extern u8 *bios_translation_ptr;

#define MAX_TRANSLATION_GATES 8

extern u32 idle_loop_target_pc;
extern u32 iwram_stack_optimize;
extern u32 translation_gate_targets;
extern u32 translation_gate_target_pc[MAX_TRANSLATION_GATES];

#define ROM_BRANCH_HASH_SIZE (1024 * 64)

u32 *rom_branch_hash[ROM_BRANCH_HASH_SIZE];

void flush_translation_cache_rom();
void flush_translation_cache_ram();
void flush_translation_cache_bios();
void dump_translation_cache();

extern u32 reg_mode[6][7];
extern u32 cpsr;
extern u32 spsr[6];
extern cpu_mode_type cpu_mode;
extern cpu_halt_type cpu_halt;

extern u32 cpu_modes[32];
extern const u32 psr_masks[16];

extern u32 branch_targets;
extern u32 high_frequency_branch_targets;

extern u32 breakpoint_value;



#endif
