/*
 *  xtiger port on PSP 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

 /* 
  * UAE - The Un*x Amiga Emulator
  * 
  * MC68000 emulation
  *
  * Copyright 1995 Bernd Schmidt
  */


#include "memory.h"
#include "sysdeps.h"

extern unsigned long specialflags;

#define SPCFLAG_DBTRACE 1
#define SPCFLAG_STOP 2
#define SPCFLAG_DBSKIP 4
#define SPCFLAG_INT  8
#define SPCFLAG_BRK  16
#define SPCFLAG_EXTRA_CYCLES 32
#define SPCFLAG_TRACE 64
#define SPCFLAG_DOTRACE 128
#define SPCFLAG_DOINT 256
#define SPCFLAG_TIMER 512
#define SPCFLAG_DELAY 1024
#define SPCFLAG_ADRERR 2048

extern int speedyLink;
extern int delayTime;

extern int areg_byteinc[];
extern int imm8_table[];

extern int broken_in;

extern int currIntLev;

typedef void cpuop_func(ULONG);

struct cputbl {
    cpuop_func *handler;
    int specific;
    UWORD opcode;
};

extern struct cputbl smallcputbl[];

extern cpuop_func *cpufunctbl[65536];
extern void op_illg(ULONG);

typedef char flagtype; 

union flagu {
    struct {
        /* v must be at the start so that the x86 seto instruction
         * changes the V flag. C must follow after V. */
        char v;
        char c;
        char n;
        char z;
    } flags;
    ULONG longflags;
};

extern struct regstruct 
{
    ULONG d[8];
    CPTR  a[8],usp;
    UWORD sr;
    flagtype t;
    flagtype s;
    flagtype x;
    flagtype stopped;
    int intmask;
    ULONG pc;
    UBYTE *pc_p;
    UBYTE *pc_oldp;
    
    ULONG vbr,sfc,dfc;
} regs;

#ifdef INTEL_FLAG_OPT
extern union flagu intel_flag_lookup[256] __asm__ ("intel_flag_lookup");
extern union flagu regflags __asm__ ("regflags");
#else
extern union flagu regflags;
#endif

#define ZFLG (regflags.flags.z)
#define NFLG (regflags.flags.n)
#define CFLG (regflags.flags.c)
#define VFLG (regflags.flags.v)


extern void MC68000_oldstep(UWORD opcode);

static __inline__ UWORD nextiword(void)
{
  UWORD r = (((UWORD)regs.pc_p[0])<<8) | regs.pc_p[1];
  regs.pc_p += 2;
  return r;
}

static __inline__ ULONG nextilong(void)
{
    return (nextiword()<<16) | nextiword();
}

static __inline__ void m68k_setpc(CPTR newpc)
{
    regs.pc = newpc&0xffffff;
    regs.pc_p = regs.pc_oldp = get_real_address(regs.pc);
}

static __inline__ CPTR m68k_getpc(void)
{
    return regs.pc + regs.pc_p - regs.pc_oldp;
}

static __inline__ void m68k_setstopped(int stop)
{
    regs.stopped = stop;
    if (stop)
        specialflags |= SPCFLAG_STOP;
}

static __inline__ int cctrue(const int cc)
{
    switch(cc){
     case 0: return 1;                       /* T */
     case 1: return 0;                       /* F */
     case 2: return !CFLG && !ZFLG;          /* HI */
     case 3: return CFLG || ZFLG;            /* LS */
     case 4: return !CFLG;                   /* CC */
     case 5: return CFLG;                    /* CS */
     case 6: return !ZFLG;                   /* NE */
     case 7: return ZFLG;                    /* EQ */
     case 8: return !VFLG;                   /* VC */
     case 9: return VFLG;                    /* VS */
     case 10:return !NFLG;                   /* PL */
     case 11:return NFLG;                    /* MI */
     case 12:return NFLG == VFLG;            /* GE */
     case 13:return NFLG != VFLG;            /* LT */
     case 14:return !ZFLG && (NFLG == VFLG); /* GT */
     case 15:return ZFLG || (NFLG != VFLG);  /* LE */
    }
    abort();
    return 0;
}

static __inline__ ULONG get_disp_ea (ULONG base, UWORD dp)
{
    int reg = (dp >> 12) & 7;
    LONG regd;
    
    if (dp & 0x8000)
        regd = regs.a[reg];
    else
        regd = regs.d[reg];
    if (!(dp & 0x800))
        regd = (LONG)(WORD)regd;
    return base + (BYTE)(dp) + regd;
}


extern void MakeSR(void);
extern void MakeFromSR(void);
extern void Exception(int);
extern void m68k_move2c(int, ULONG *);
extern void m68k_movec2(int, ULONG *);
extern void m68k_divl (UWORD, ULONG, UWORD);
extern void m68k_mull (UWORD, ULONG, UWORD);
extern void init_m68k (void);
extern void MC68000_step(void);
extern void MC68000_run(void);
extern void MC68000_skip(CPTR);
extern void MC68000_dumpstate(CPTR *);
extern void MC68000_disasm(CPTR,CPTR *,int);
extern void MC68000_reset(void);
extern int intlev(void);
