/*
Copyright (C) 2001,2005 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/


#include "stdafx.h"
#include "CodeGeneratorX86.h"
#include "DynaRec/Trace.h"
#include "DynaRec/IndirectExitMap.h"

#include "Debug/DebugLog.h"
#include "Debug/DBGConsole.h"

#include "Core/CPU.h"
#include "Core/Registers.h"
#include "Core/R4300.h"

#include "OSHLE/ultra_R4300.h"

#include "DynaRec/AssemblyUtils.h"


using namespace AssemblyUtils;

static const u32		NUM_MIPS_REGISTERS( 32 );
static const u32		INTEL_REG_UNUSED( ~0 );

//*****************************************************************************
//	XXXX
//*****************************************************************************
void Dynarec_ClearedCPUStuffToDo()
{
}
void Dynarec_SetCPUStuffToDo()
{
}

//*****************************************************************************
//	Register Caching
//*****************************************************************************
class CRegisterStatus
{
	struct CachedRegInfo
	{
		EIntelReg iCachedIReg;	// INVALID_CODE if uncached, else ???_CODE of intel reg we're cached in
		bool	BLoValid;		// If cached, TRUE if value in intel register is valid
		bool	BLoDirty;		// If cached, TRUE if value has been modified

		u32		HiValue;		// If BHiUnknown is FALSE, this is the value of the register (through setmipshi..)
		bool	BHiDirty;		// Do we need to write information about this register back to memory?
		bool	BHiUnknown;		// If TRUE, we don't know the value of this register
	};
public:

	void Reset()
	{
		for (u32 i = 0; i < NUM_REGISTERS; i++)
		{
			SetCachedReg( i, INVALID_CODE );
			MarkAsValid( i, false );
			MarkAsDirty( i, false );

			MarkHiAsDirty( i, false );
			MarkHiAsUnknown( i, true );
			SetHiValue( i, 0 );		// Ignored
		}
	}

	inline void MarkAsValid( u32 mreg, bool valid )
	{
		mCachedRegisterInfo[mreg].BLoValid = valid;
	}

	inline bool IsValid( u32 mreg ) const
	{
		return mCachedRegisterInfo[mreg].BLoValid;
	}

	inline bool IsDirty( u32 mreg ) const
	{
		return mCachedRegisterInfo[mreg].BLoDirty;
	}

	inline void MarkAsDirty( u32 mreg, bool dirty )
	{
		mCachedRegisterInfo[mreg].BLoDirty = dirty;	
	}

	inline EIntelReg GetCachedReg( u32 mreg ) const
	{
		return mCachedRegisterInfo[mreg].iCachedIReg;
	}

	inline void SetCachedReg( u32 mreg, EIntelReg reg )
	{
		mCachedRegisterInfo[mreg].iCachedIReg = reg;
	}

	inline void MarkHiAsUnknown( u32 mreg, bool unk )
	{
		mCachedRegisterInfo[mreg].BHiUnknown = unk;
	}

	inline void MarkHiAsDirty( u32 mreg, bool dirty )
	{
		mCachedRegisterInfo[mreg].BHiDirty = dirty;	
	}

	inline bool IsHiDirty( u32 mreg ) const
	{
		return mCachedRegisterInfo[mreg].BHiDirty;
	}
	
	inline bool IsHiUnknown( u32 mreg ) const
	{
		return mCachedRegisterInfo[mreg].BHiUnknown;
	}

	inline void SetHiValue( u32 mreg, u32 value )
	{
		mCachedRegisterInfo[mreg].HiValue = value;
	}

	inline u32 GetHiValue( u32 mreg ) const
	{
		return mCachedRegisterInfo[mreg].HiValue;
	}

private:
	static const u32	NUM_REGISTERS = 32;
	CachedRegInfo		mCachedRegisterInfo[NUM_REGISTERS];
};


static CRegisterStatus	gRegisterStatus;
static u32				gIntelRegUsageMap[NUM_X86_REGISTERS];
static u32				gWriteCheck[NUM_MIPS_REGISTERS];

//*****************************************************************************
//
//*****************************************************************************
CCodeGeneratorX86::CCodeGeneratorX86( CAssemblyBuffer * p_primary, CAssemblyBuffer * p_secondary )
:	CCodeGenerator( )
,	CAssemblyWriterX86( p_primary )
,	mSpCachedInESI( false )
,	mSetSpPostUpdate( 0 )
,	mpPrimary( p_primary )
,	mpSecondary( p_secondary )
{
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::Finalise( ExceptionHandlerFn p_exception_handler_fn, const std::vector< CJumpLocation > & exception_handler_jumps )
{
	if( !exception_handler_jumps.empty() )
	{
		GenerateExceptionHander( p_exception_handler_fn, exception_handler_jumps );
	}

	SetAssemblyBuffer( NULL );
	mpPrimary = NULL;
	mpSecondary = NULL;
}

//*****************************************************************************
//
//*****************************************************************************
#if 0
u32 gNumFragmentsExecuted = 0;
extern "C"
{

void LogFragmentEntry( u32 entry_address )
{
	gNumFragmentsExecuted++;
	if(gNumFragmentsExecuted >= 0x99990)
	{
		OutputDebugString( DSPrintf( "Address %08x\n", entry_address ) );
	}
}

}
#endif

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::Initialise( u32 entry_address, u32 exit_address, u32 * hit_counter, const void * p_base, const SRegisterUsageInfo & register_usage )
{
	//MOVI(ECX_CODE, entry_address);
	// CALL( CCodeLabel( LogFragmentEntry ) );

	if( hit_counter != NULL )
	{
		MOV_REG_MEM( EAX_CODE, hit_counter );
		ADDI( EAX_CODE, 1 );
		MOV_MEM_REG( hit_counter, EAX_CODE );
	}

	// p_base/span_list ignored for now
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::UpdateRegisterCaching( u32 instruction_idx )
{
	// This is ignored for now
}

//*****************************************************************************
//
//*****************************************************************************
RegisterSnapshotHandle	CCodeGeneratorX86::GetRegisterSnapshot()
{
	// This doesn't do anything useful yet.
	return RegisterSnapshotHandle( 0 );
}

//*****************************************************************************
//
//*****************************************************************************
CCodeLabel	CCodeGeneratorX86::GetEntryPoint() const
{
	return mpPrimary->GetStartAddress();
}

//*****************************************************************************
//
//*****************************************************************************
CCodeLabel	CCodeGeneratorX86::GetCurrentLocation() const
{
	return mpPrimary->GetLabel();
}

//*****************************************************************************
//
//*****************************************************************************
u32	CCodeGeneratorX86::GetCompiledCodeSize() const
{
	return mpPrimary->GetSize() + mpSecondary->GetSize();
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation CCodeGeneratorX86::GenerateExitCode( u32 exit_address, u32 jump_address, u32 num_instructions, CCodeLabel next_fragment )
{
	//DAEDALUS_ASSERT( exit_address != u32( ~0 ), "Invalid exit address" );
	DAEDALUS_ASSERT( !next_fragment.IsSet() || jump_address == 0, "Shouldn't be specifying a jump address if we have a next fragment?" );

#ifdef _DEBUG
	if(exit_address == u32(~0))
	{
		INT3();
	}
#endif

	MOVI(ECX_CODE, num_instructions);
	CALL( CCodeLabel( CPU_UpdateCounter ) );

	// This jump may be NULL, in which case we patch it below
	// This gets patched with a jump to the next fragment if the target is later found
	CJumpLocation jump_to_next_fragment( GenerateBranchIfNotSet( &gCPUState.StuffToDo, next_fragment ) );

	// If the flag was set, we need in initialise the pc/delay to exit with
	CCodeLabel interpret_next_fragment( GetAssemblyBuffer()->GetLabel() );

	u8		exit_delay;

	if( jump_address != 0 )
	{
		SetVar( &gCPUState.TargetPC, jump_address );
		exit_delay = EXEC_DELAY;
	}
	else
	{
		exit_delay = NO_DELAY;
	}

	SetVar8( &gCPUState.Delay, exit_delay );
	SetVar( &gCPUState.CurrentPC, exit_address );
	
	// No need to call CPU_SetPC(), as this is handled by CFragment when we exit
	RET();

	// Patch up the exit jump
	if( !next_fragment.IsSet() )
	{
		PatchJumpLong( jump_to_next_fragment, interpret_next_fragment );
	}

	return jump_to_next_fragment;
}

//*****************************************************************************
// Handle branching back to the interpreter after an ERET
//*****************************************************************************
void CCodeGeneratorX86::GenerateEretExitCode( u32 num_instructions, CIndirectExitMap * p_map )
{
	MOVI(ECX_CODE, num_instructions);
	CALL( CCodeLabel( CPU_UpdateCounter ) );

	// We always exit to the interpreter, regardless of the state of gCPUState.StuffToDo

	// Eret is a bit bodged so we exit at PC + 4
	MOV_REG_MEM( EAX_CODE, &gCPUState.CurrentPC );
	ADDI( EAX_CODE, 4 );
	MOV_MEM_REG( &gCPUState.CurrentPC, EAX_CODE );
	SetVar8( &gCPUState.Delay, NO_DELAY );
	
	// No need to call CPU_SetPC(), as this is handled by CFragment when we exit

	RET();
}

//*****************************************************************************
// Handle branching back to the interpreter after an indirect jump
//*****************************************************************************
void CCodeGeneratorX86::GenerateIndirectExitCode( u32 num_instructions, CIndirectExitMap * p_map )
{

	MOVI(ECX_CODE, num_instructions);
	CALL( CCodeLabel( CPU_UpdateCounter ) );

	CCodeLabel		no_target( NULL );
	CJumpLocation	jump_to_next_fragment( GenerateBranchIfNotSet( &gCPUState.StuffToDo, no_target ) );

	CCodeLabel		exit_dynarec( GetAssemblyBuffer()->GetLabel() );
	// New return address is in gCPUState.TargetPC
	MOV_REG_MEM( EAX_CODE, &gCPUState.TargetPC );
	MOV_MEM_REG( &gCPUState.CurrentPC, EAX_CODE );
	SetVar8( &gCPUState.Delay, NO_DELAY );
	
	// No need to call CPU_SetPC(), as this is handled by CFragment when we exit

	RET();

	// gCPUState.StuffToDo == 0, try to jump to the indirect target
	PatchJumpLong( jump_to_next_fragment, GetAssemblyBuffer()->GetLabel() );

	MOVI( ECX_CODE, reinterpret_cast< u32 >( p_map ) );
	MOV_REG_MEM( EDX_CODE, &gCPUState.TargetPC );
	CALL( CCodeLabel( IndirectExitMap_Lookup ) );

	// If the target was not found, exit
	TEST( EAX_CODE, EAX_CODE );
	JELong( exit_dynarec );

	JMP_REG( EAX_CODE );
}

//*****************************************************************************
//
//*****************************************************************************
void CCodeGeneratorX86::GenerateExceptionHander( ExceptionHandlerFn p_exception_handler_fn, const std::vector< CJumpLocation > & exception_handler_jumps )
{
	CCodeLabel exception_handler( GetAssemblyBuffer()->GetLabel() );

	CALL( CCodeLabel( p_exception_handler_fn ) );
	RET();

	for( std::vector< CJumpLocation >::const_iterator it = exception_handler_jumps.begin(); it != exception_handler_jumps.end(); ++it )
	{
		CJumpLocation	jump( *it );
		PatchJumpLong( jump, exception_handler );
	}
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::SetVar( u32 * p_var, u32 value )
{
	MOVI_MEM( p_var, value );
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::SetVar8( u32 * p_var, u8 value )
{
	MOVI_MEM8( p_var, value );
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::GenerateBranchHandler( CJumpLocation branch_handler_jump, RegisterSnapshotHandle snapshot )
{
	PatchJumpLong( branch_handler_jump, GetAssemblyBuffer()->GetLabel() );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchAlways( CCodeLabel target )
{
	return JMPLong( target );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchIfSet( const u32 * p_var, CCodeLabel target )
{
	MOV_REG_MEM( EAX_CODE, p_var );
	TEST( EAX_CODE, EAX_CODE );

	return JNELong( target );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchIfNotSet( const u32 * p_var, CCodeLabel target )
{
	MOV_REG_MEM( EAX_CODE, p_var );
	TEST( EAX_CODE, EAX_CODE );

	return JELong( target );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchIfEqual32( const u32 * p_var, u32 value, CCodeLabel target )
{
	CMP_MEM32_I32( p_var, value );
	
	return JELong( target );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchIfEqual8( const u32 * p_var, u8 value, CCodeLabel target )
{
	CMP_MEM32_I8( p_var, value );
	
	return JELong( target );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchIfNotEqual32( const u32 * p_var, u32 value, CCodeLabel target )
{
	CMP_MEM32_I32( p_var, value );
	
	return JNELong( target );
}

//*****************************************************************************
//
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateBranchIfNotEqual8( const u32 * p_var, u8 value, CCodeLabel target )
{
	CMP_MEM32_I8( p_var, value );
	
	return JNELong( target );
}

//*****************************************************************************
//	Generates instruction handler for the specified op code.
//	Returns a jump location if an exception handler is required
//*****************************************************************************
CJumpLocation	CCodeGeneratorX86::GenerateOpCode( u32 address, bool branch_delay_slot, OpCode op_code, const SBranchDetails * p_branch, CJumpLocation * p_branch_jump )
{
	bool	need_pc( R4300_InstructionHandlerNeedsPC( op_code ) );

	if( !need_pc )
	{
		address = 0;
	}

	if( branch_delay_slot )
	{
		SetVar8( &gCPUState.Delay, EXEC_DELAY );
	}

	GenerateGenericR4300( address, op_code, R4300_GetInstructionHandler( op_code ) );

	CJumpLocation	exception_handler;
	CCodeLabel		no_target( NULL );

	if( need_pc )
	{
		exception_handler = GenerateBranchIfSet( &gCPUState.StuffToDo, no_target );
	}

	// Check whether we want to invert the status of this branch
	if( p_branch != NULL )
	{
		//
		// Check if the branch has been taken
		//
		if( p_branch->Direct )
		{
			if( p_branch->ConditionalBranchTaken )
			{
				*p_branch_jump = GenerateBranchIfNotEqual8( &gCPUState.Delay, DO_DELAY, no_target );
			}
			else
			{
				*p_branch_jump = GenerateBranchIfEqual8( &gCPUState.Delay, DO_DELAY, no_target );
			}
		}
		else
		{
			// XXXX eventually just exit here, and skip default exit code below
			if( p_branch->Eret )
			{
				*p_branch_jump = GenerateBranchAlways( no_target );
			}
			else
			{
				*p_branch_jump = GenerateBranchIfNotEqual32( &gCPUState.TargetPC, p_branch->TargetAddress, no_target );
			}
		}
	}
	else
	{
		if( branch_delay_slot )
		{
			SetVar8( &gCPUState.Delay, NO_DELAY );
		}
	}

	
	return exception_handler;
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::GenerateGenericR4300( u32 address, OpCode op_code, CPU_Instruction p_instruction )
{
	// XXXX Flush all fp registers before a generic call

	if( address != 0 )
	{
		SetVar( &gCPUState.CurrentPC, address );
	}
	
	// Call function - __fastcall
	MOVI(ECX_CODE, op_code._u32);
	CALL( CCodeLabel( p_instruction ) );
}

//*****************************************************************************
//
//*****************************************************************************
void	CCodeGeneratorX86::ExecuteSpeedHack( CCodeLabel speed_hack )
{
	CALL( speed_hack );
}
