/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
	  
*/

// Manages textures for RDP code
// Uses a HashTable (hashing on TImg) to allow quick access
//  to previously used textures

#include "stdafx.h"

#include "TextureCache.h"
#include "TextureDescriptor.h"

#include "Utility/Profiler.h"

#include "DebugDisplayList.h"
#include "DaedCritSect.h"

#include <vector>
#include <algorithm>

//*************************************************************************************
//
//*************************************************************************************
class ITextureCache : public CTextureCache
{
public:
	ITextureCache();
	~ITextureCache();
	
			void			PurgeOldTextures();
			void			DropTextures();

			void			SetDumpTextures( bool dump_textures )	{ mDumpTextures = dump_textures; }
			bool			GetDumpTextures( ) const				{ return mDumpTextures; }
	
			CRefPtr<CTexture>GetTexture(const TextureInfo * pti);
	
	virtual	void			DisplayStats();

		#ifdef DAEDALUS_DEBUG_DISPLAYLIST
			void			Snapshot( std::vector< STextureInfoSnapshot > & snapshot ) const;
		#endif

protected:
			void			GetUsedTextureStats( u32 * p_num_textures, u32 * p_video_memory_used, u32 * p_system_memory_used ) const;

protected:
	struct STextureEntry
	{
		TextureInfo				Info;
		CRefPtr<CTexture>		Texture;

		explicit STextureEntry( const TextureInfo & info, CTexture * texture )
			:	Info( info )
			,	Texture( texture )
		{
		}

		bool operator<( const STextureEntry & rhs ) const
		{
			return Info < rhs.Info;
		}
	};

	typedef std::vector< STextureEntry >	TextureVec;
	TextureVec				mTextures;

	bool					mDumpTextures;
	CCritSect				mCritSect;
};


// Interface

//*****************************************************************************
//
//*****************************************************************************
namespace daedalus
{
template<> bool CSingleton< CTextureCache >::Create()
{
	DAEDALUS_ASSERT_Q(mpInstance == NULL);
	
	mpInstance = new ITextureCache();
	return mpInstance != NULL;
}
}

//*************************************************************************************
//
//*************************************************************************************
ITextureCache::ITextureCache()
:	mDumpTextures(false)
{
}

//*************************************************************************************
//
//*************************************************************************************
ITextureCache::~ITextureCache()
{
	DropTextures();
}


//*************************************************************************************
// Purge any textures that haven't been used recently
//*************************************************************************************
void ITextureCache::PurgeOldTextures()
{
	AUTO_CRIT_SECT( mCritSect );

	//
	//	Erase expired textures in reverse order, which should require less
	//	copying when large clumps of textures are released simultaneously.
	//
	for( s32 i = mTextures.size() - 1; i >= 0; --i )
	{
		CTexture * texture( mTextures[ i ].Texture );
		if ( texture->HasExpired() )
		{
			mTextures.erase( mTextures.begin() + i );
		}
	}
}

//*************************************************************************************
//
//*************************************************************************************
void ITextureCache::DropTextures()
{
	AUTO_CRIT_SECT( mCritSect );

	mTextures.clear();
}

//*************************************************************************************
// If already in table, return cached copy
// Otherwise, create surfaces, and load texture into memory
//*************************************************************************************
CRefPtr<CTexture> ITextureCache::GetTexture(const TextureInfo * pti)
{
	DAEDALUS_PROFILE( "ITextureCache::GetTexture" );

	// Check for 0 width/height textures
	if (pti->GetWidth() == 0 || pti->GetHeight() == 0)
	{
		DAEDALUS_DL_ERROR( "Loading texture with 0 width/height" );
		return NULL;
	}

	//
	// Retrieve the texture from the cache (if it already exists)
	//
	AUTO_CRIT_SECT( mCritSect );

	CRefPtr<CTexture>		texture;
	STextureEntry			entry( *pti, NULL );
	TextureVec::iterator	it( std::lower_bound( mTextures.begin(), mTextures.end(), entry ) );
	if( it != mTextures.end() && it->Info == *pti )
	{
		texture = it->Texture;
	}
	else
	{
		texture = CTexture::Create( *pti );
		if (texture != NULL)
		{
			if ( mDumpTextures )
			{
				texture->DumpTexture();
			}
			entry.Texture = texture;

			mTextures.insert( it, entry );
		}
	}

	return texture;
}

//*************************************************************************************
//
//*************************************************************************************
void	ITextureCache::GetUsedTextureStats( u32 * p_num_textures,
											u32 * p_video_memory_used,
											u32 * p_system_memory_used ) const
{
	u32	num_textures( 0 );
	u32	video_memory_used( 0 );
	u32	system_memory_used( 0 );

	for( TextureVec::const_iterator it = mTextures.begin(); it != mTextures.end(); ++it )
	{
		CTexture * texture( it->Texture );
		num_textures++;
		video_memory_used += texture->GetVideoMemoryUsage();
		system_memory_used += texture->GetSystemMemoryUsage();
	}

	*p_num_textures = num_textures;
	*p_video_memory_used = video_memory_used;
	*p_system_memory_used = system_memory_used;
}

//*************************************************************************************
//	Display info on the number of textures in the cache
//*************************************************************************************
void	ITextureCache::DisplayStats()
{
	u32		used_textures;
	u32		used_texture_video_mem;
	u32		used_texture_system_mem;

	GetUsedTextureStats( &used_textures, &used_texture_video_mem, &used_texture_system_mem );

	printf( "Texture Cache Stats\n" );
	printf( " Used: %3d v:%4dKB s:%4dKB\n", used_textures, used_texture_video_mem / 1024, used_texture_system_mem / 1024 );
}

#ifdef DAEDALUS_DEBUG_DISPLAYLIST
//*************************************************************************************
//
//*************************************************************************************
void	ITextureCache::Snapshot( std::vector< STextureInfoSnapshot > & snapshot ) const
{
	snapshot.erase( snapshot.begin(), snapshot.end() );

	for( TextureVec::const_iterator it = mTextures.begin(); it != mTextures.end(); ++it )
	{
		STextureInfoSnapshot	info( it->Texture );
		snapshot.push_back( info );
	}
}

CTextureCache::STextureInfoSnapshot::STextureInfoSnapshot( CTexture * p_texture )
:	Texture( p_texture )
{
}

CTextureCache::STextureInfoSnapshot::STextureInfoSnapshot( const STextureInfoSnapshot & rhs )
:	Texture( rhs.Texture )
{
}

CTextureCache::STextureInfoSnapshot & CTextureCache::STextureInfoSnapshot::operator=( const STextureInfoSnapshot & rhs )
{
	if( this != &rhs )
	{
		Texture = rhs.Texture;
	}

	return *this;
}

CTextureCache::STextureInfoSnapshot::~STextureInfoSnapshot()
{
}


#endif
