/*

  Copyright (C) 2001 StrmnNrmn

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "GraphicsContext.h"
#include "PngUtil.h"
#include "Debug/DBGConsole.h"
#include "Debug/Dump.h"
#include "Utility/Profiler.h"

#include "Core/ROM.h"

#include "Utility/Preferences.h"

#include "DaedIO.h"

#include "TextureCache.h"
#include "VideoMemoryManager.h"

#include <pspgu.h>
#include <pspdisplay.h>
#include <pspdebug.h>

#ifndef SAFE_RELEASE
#define SAFE_RELEASE(p)  { if(p) { (p)->Release();     (p)=NULL; } }
#endif

namespace
{
	const char *	gScreenDumpRootPath = "ScreenShots";
	const char *	gScreenDumpDumpPathFormat = "sd%04d.png";
}

unsigned int __attribute__((aligned(16))) list[262144];

#define BUF_WIDTH (512)
#define SCR_WIDTH (480)
#define SCR_HEIGHT (272)
#define PIXEL_SIZE (4) /* change this if you change to another screenmode */
#define FRAME_SIZE (BUF_WIDTH * SCR_HEIGHT * PIXEL_SIZE)
#define DEPTH_SIZE (BUF_WIDTH * SCR_HEIGHT * 2)


// Implementation
class IGraphicsContext : public CGraphicsContext
{
public:
	virtual ~IGraphicsContext();
	
	BOOL				Ready() { return m_bReady; }
	
	void				SetReady(BOOL bReady) { m_bReady = bReady; }

	void				ClearAllSurfaces();
	
	void				Clear(bool clear_screen, bool clear_depth);
	void				Clear(u32 frame_buffer_col, u32 depth);

	void				BeginFrame();
	void				EndFrame();
	bool				UpdateFrame( bool wait_for_vbl );
	bool				GetBufferSize( u32 * p_width, u32 * p_height );
	bool				Initialise();
		
	void				SetDebugScreenTarget( ETargetSurface buffer );

	void				DumpScreenShot();
	void				DumpNextScreen()			{ mDumpNextScreen = true; }

private:
	void				SaveScreenshot( const char* filename, s32 x, s32 y, u32 width, u32 height );
	
protected:
	friend class CSingleton< CGraphicsContext >;

	IGraphicsContext();

	BOOL				m_bReady;
	BOOL				m_bActive;

	void *				mpBuffers[2];
	void *				mpCurrentBackBuffer;

	bool				mDumpNextScreen;
};

//*************************************************************************************
// 
//*************************************************************************************
namespace daedalus
{
template<> bool CSingleton< CGraphicsContext >::Create()
{	
	IGraphicsContext * pNewInstance;
	
	DAEDALUS_ASSERT_Q(mpInstance == NULL);
	
	pNewInstance = new IGraphicsContext();
	if (!pNewInstance)
	{
		return false;
	}

	pNewInstance->Initialise();
	
	mpInstance = pNewInstance;
	
	return true;
}
}


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

IGraphicsContext::IGraphicsContext()
:	m_bReady(FALSE)
, 	m_bActive(FALSE)
,	mpCurrentBackBuffer(NULL)
,	mDumpNextScreen( false )
{
	mpBuffers[ 0 ] = NULL;
	mpBuffers[ 1 ] = NULL;
}

//*****************************************************************************
//
//*****************************************************************************
IGraphicsContext::~IGraphicsContext()
{
	sceGuTerm();
}

//*****************************************************************************
//
//*****************************************************************************
void IGraphicsContext::ClearAllSurfaces()
{
	for( u32 i = 0; i < 2; ++i )
	{
		BeginFrame();

		sceGuOffset(2048 - (SCR_WIDTH/2),2048 - (SCR_HEIGHT/2));
		sceGuViewport(2048,2048,SCR_WIDTH,SCR_HEIGHT);
		sceGuScissor(0,0,SCR_WIDTH,SCR_HEIGHT);

		Clear( true, true );
		EndFrame();
		UpdateFrame( false );
	}
}

//*****************************************************************************
//
//*****************************************************************************
void IGraphicsContext::Clear(bool clear_screen, bool clear_depth)
{
	int	flags( 0 );

	if(clear_screen)
		flags |= GU_COLOR_BUFFER_BIT;
	if(clear_depth)
		flags |= GU_DEPTH_BUFFER_BIT;

	sceGuClearColor(0xff000000);
	sceGuClearDepth(0);				// 1?
	sceGuClear(flags);
}

//*****************************************************************************
//
//*****************************************************************************
void IGraphicsContext::Clear(u32 frame_buffer_col, u32 depth)
{
	sceGuClearColor(frame_buffer_col);
	sceGuClearDepth(depth);				// 1?
	sceGuClear(GU_COLOR_BUFFER_BIT|GU_DEPTH_BUFFER_BIT);
}

//*****************************************************************************
//
//*****************************************************************************
void IGraphicsContext::BeginFrame()
{
	sceGuStart(GU_DIRECT,list);
}

//*****************************************************************************
//
//*****************************************************************************
void IGraphicsContext::EndFrame()
{
	sceGuFinish();
}


//*****************************************************************************
//
//*****************************************************************************
bool IGraphicsContext::UpdateFrame( bool wait_for_vbl )
{
	DAEDALUS_PROFILE( "IGraphicsContext::UpdateFrame" );

	sceGuSync(0,0);

	if(wait_for_vbl)
	{
		sceDisplayWaitVblankStart();
	}

	void * p_back = sceGuSwapBuffers();

	if ( mDumpNextScreen )
	{
		DumpScreenShot();
		mDumpNextScreen = false;
	}

	mpCurrentBackBuffer = p_back;

	SetDebugScreenTarget( TS_BACKBUFFER );

	return true;
}

//*****************************************************************************
//	Set the target for the debug screen
//*****************************************************************************
void IGraphicsContext::SetDebugScreenTarget( ETargetSurface buffer )
{
	void *	p_target;

	if( buffer == TS_BACKBUFFER )
	{
		p_target = mpCurrentBackBuffer;
	}
	else if( buffer == TS_FRONTBUFFER )
	{
		if(mpCurrentBackBuffer == mpBuffers[ 0 ])
		{
			p_target = mpBuffers[ 1 ];
		}
		else
		{
			p_target = mpBuffers[ 0 ];
		}
	}
	else
	{
		DAEDALUS_ERROR( "Unknown buffer" );
		p_target = mpCurrentBackBuffer;
	}

	pspDebugScreenSetOffset( int(p_target) & ~0x40000000 );
}

//*****************************************************************************
// Save current visible screen as PNG
// From Shazz/71M - thanks guys!
//*****************************************************************************
void IGraphicsContext::SaveScreenshot( const char* filename, s32 x, s32 y, u32 width, u32 height )
{
	void * buffer;
	int bufferwidth;
	int pixelformat;
	int unknown = 0;

	sceDisplayWaitVblankStart();  // if framebuf was set with PSP_DISPLAY_SETBUF_NEXTFRAME, wait until it is changed
	sceDisplayGetFrameBuf( &buffer, &bufferwidth, &pixelformat, unknown );

	ETextureFormat		texture_format;
	u32					bpp;

	switch( pixelformat )
	{
	case PSP_DISPLAY_PIXEL_FORMAT_565:
		texture_format = TexFmt_5650;
		bpp = 2;
		break;
	case PSP_DISPLAY_PIXEL_FORMAT_5551:
		texture_format = TexFmt_5551;
		bpp = 2;
		break;
	case PSP_DISPLAY_PIXEL_FORMAT_4444:
		texture_format = TexFmt_4444;
		bpp = 2;
		break;
	case PSP_DISPLAY_PIXEL_FORMAT_8888:
		texture_format = TexFmt_8888;
		bpp = 4;
		break;
	default:
		texture_format = TexFmt_8888;
		bpp = 4;
		break;
	}

	u32 pitch( bufferwidth * bpp );

	buffer = reinterpret_cast< u8 * >( buffer ) + (y * pitch) + (x * bpp);

	PngSaveImage( filename, buffer, NULL, texture_format, pitch, width, height, false );
}

//*****************************************************************************
//
//*****************************************************************************
void IGraphicsContext::DumpScreenShot()
{
	char szFilePath[MAX_PATH+1];
	char szDumpDir[MAX_PATH+1];

	IO::Path::Combine(szDumpDir, g_ROM.settings.GameName.c_str(), gScreenDumpRootPath);

	Dump_GetDumpDirectory(szFilePath, szDumpDir);

	char	unique_filename[MAX_PATH+1];
	u32 count = 0;
	do
	{
		char	test_name[MAX_PATH+1];

		sprintf(test_name, "sd%04d.png", count++);

		IO::Path::Combine( unique_filename, szFilePath, test_name );

	} while( IO::File::Exists( unique_filename ) );

	u32		display_width( 0 );
	u32		display_height( 0 );
	switch ( gGlobalPreferences.ViewportType )
	{
	case VT_UNSCALED_4_3:		// 1:1
		display_width = 320;
		display_height = 240;
		break;
	case VT_SCALED_4_3:		// Largest 4:3
		display_height = 272;
		display_width = (display_height * 4) / 3; 
		break;
	case VT_FULLSCREEN:		// Fullscreen
		display_width = 480;
		display_height = 272;
		break;
 	}
	DAEDALUS_ASSERT( display_width != 0 && display_height != 0, "Unhandled viewport type" );

	s32		display_x( (480 - display_width)/2 );
	s32		display_y( (272 - display_height)/2 );

	SaveScreenshot( unique_filename, display_x, display_y, display_width, display_height );
}

//*****************************************************************************
//
//*****************************************************************************
bool IGraphicsContext::GetBufferSize(u32 * p_width, u32 * p_height)
{
	*p_width = SCR_WIDTH;
	*p_height = SCR_HEIGHT;

	return true;
}
//*****************************************************************************
//
//*****************************************************************************
namespace
{
	void *	VramAddrAsOffset( void * ptr )
	{
		return reinterpret_cast< void * >( reinterpret_cast< u32 >( ptr ) - reinterpret_cast< u32 >( sceGeEdramGetAddr() ) );
	}
}

bool IGraphicsContext::Initialise()
{
	sceGuInit();

	bool	is_videmem;		// Result is ignored

	void *	draw_buffer;
	void *	disp_buffer;
	void *	depth_buffer;

	CVideoMemoryManager::Get()->Alloc( FRAME_SIZE, &draw_buffer, &is_videmem );
	CVideoMemoryManager::Get()->Alloc( FRAME_SIZE, &disp_buffer, &is_videmem );
	CVideoMemoryManager::Get()->Alloc( DEPTH_SIZE, &depth_buffer, &is_videmem );

	void *	draw_buffer_rel( VramAddrAsOffset( draw_buffer ) );
	void *	disp_buffer_rel( VramAddrAsOffset( disp_buffer ) );
	void *	depth_buffer_rel( VramAddrAsOffset( depth_buffer ) );

	printf( "Allocated %d bytes of memory for draw buffer at %p\n", FRAME_SIZE, draw_buffer );
	printf( "Allocated %d bytes of memory for draw buffer at %p\n", FRAME_SIZE, disp_buffer );
	printf( "Allocated %d bytes of memory for draw buffer at %p\n", DEPTH_SIZE, depth_buffer );

	sceGuStart(GU_DIRECT,list);
	sceGuDrawBuffer(GU_PSM_8888,draw_buffer_rel,BUF_WIDTH);
	sceGuDispBuffer(SCR_WIDTH,SCR_HEIGHT,disp_buffer_rel,BUF_WIDTH);
	sceGuDepthBuffer(depth_buffer_rel,BUF_WIDTH);
	sceGuOffset(2048 - (SCR_WIDTH/2),2048 - (SCR_HEIGHT/2));
	sceGuViewport(2048,2048,SCR_WIDTH,SCR_HEIGHT);
	sceGuDepthRange(65535,0);
	sceGuScissor(0,0,SCR_WIDTH,SCR_HEIGHT);
	sceGuEnable(GU_SCISSOR_TEST);
	sceGuDepthFunc(GU_GEQUAL);
	sceGuEnable(GU_DEPTH_TEST);
	sceGuFrontFace(GU_CW);
	sceGuShadeModel(GU_SMOOTH);
	sceGuEnable(GU_CULL_FACE);
	sceGuEnable(GU_TEXTURE_2D);
	sceGuEnable(GU_CLIP_PLANES);
	sceGuFinish();
	sceGuSync(0,0);

	sceDisplayWaitVblankStart();
	sceGuDisplay(GU_TRUE);

	mpBuffers[ 0 ] = draw_buffer_rel;
	mpBuffers[ 1 ] = disp_buffer_rel;
	mpCurrentBackBuffer= mpBuffers[ 0 ];

	// The app is ready to go
	m_bReady = TRUE;
    m_bActive = TRUE;
    return true;
}


