/*

  Copyright (C) 2001 StrmnNrmn

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "StdAfx.h"

#include "RomDB.h"

#include "Core/ROM.h"
#include "Core/ROMImage.h"

#include "Debug/DBGConsole.h"

#include "Utility/ROMFile.h"

#include "DaedStream.h"

#include <map>
#include <string>

static const u64 ROMDB_MAGIC_NO	= 0x42444D5244454144LL; //DAEDRMDB		// 44 41 45 44 52 4D 44 42
static const u32 ROMDB_CURRENT_VERSION = 3;

static const u32 MAX_SENSIBLE_FILES = 2048;
static const u32 MAX_SENSIBLE_DETAILS = 2048;


//*****************************************************************************
//
//*****************************************************************************
CRomDB::~CRomDB()
{
}

//*****************************************************************************
// Class implementations
//*****************************************************************************
class IRomDB : public CRomDB
{
	public:
		IRomDB();
		~IRomDB();

		//
		// CRomDB implementation
		//
		bool			OpenDB( const char * filename );
		void			Reset();
		bool			Commit();

		bool			QueryByFilename( const char * filename, RomID * id, u32 * rom_size, ECicType * cic_type );
		bool			QueryByID( const RomID & id, u32 * rom_size, ECicType * cic_type ) const;
		const char *	QueryFilenameFromID( const RomID & id ) const;

	private:
		void			AddRom( const char * filename, const RomID & id, u32 rom_size, ECicType cic_type );

	private:
		struct RomDetails
		{
			RomDetails()
				:	RomSize( 0 )
				,	CicType( CIC_UNKNOWN )
			{
			}

			RomDetails( u32 rom_size, ECicType cic_type )
				:	RomSize( rom_size )
				,	CicType( cic_type )
			{
			}

			u32			RomSize;
			ECicType	CicType;
		};

		// For serialisation we used a fixed size struct for ease of reading
		struct RomFilesKeyValue
		{
			RomFilesKeyValue()
			{
				memset( FileName, 0, sizeof( FileName ) );
			}
			RomFilesKeyValue( const char * filename, const RomID & id )
			{
				memset( FileName, 0, sizeof( FileName ) );
				strcpy( FileName, filename );
				ID = id;
			}

			char		FileName[ MAX_PATH + 1 ];
			RomID		ID;
		};

		struct RomDetailsKeyValue
		{
			RomDetailsKeyValue()
				:	ID()
				,	Details()
			{
			}

			RomDetailsKeyValue( const RomID & id, const RomDetails & details )
				:	ID( id )
				,	Details( details )
			{
			}

			RomID				ID;
			RomDetails			Details;
		};

		typedef std::map< std::string, RomID >	FilenameMap;
		typedef std::map< RomID, RomDetails >	DetailsMap;

		char							mRomDBFileName[ MAX_PATH + 1 ];
		FilenameMap						mRomFiles;
		DetailsMap						mRomDetails;
		bool							mDirty;
};

//*****************************************************************************
// Singleton creator
//*****************************************************************************
namespace daedalus
{
template<> bool	CSingleton< CRomDB >::Create()
{
	DAEDALUS_ASSERT_Q(mpInstance == NULL);
	
	mpInstance = new IRomDB();

	return true;
}
}

//*****************************************************************************
// Constructor
//*****************************************************************************
IRomDB::IRomDB()
:	mDirty( false )
{
	mRomDBFileName[ 0 ] = '\0';
}

//*****************************************************************************
//
//*****************************************************************************
IRomDB::~IRomDB()
{
	Commit();
}

//*****************************************************************************
//
//*****************************************************************************
void	IRomDB::Reset()
{
	mRomFiles.clear();
	mRomDetails.clear();
	mDirty = true;
}

//*****************************************************************************
//
//*****************************************************************************
bool	IRomDB::OpenDB( const char * filename )
{
	u32 num_read;

	//
	// Remember the filename
	//
	strcpy( mRomDBFileName, filename );

	FILE * fh = fopen( filename, "rb" );
	if ( !fh )
		return false;
	
	//
	// Check the magic number
	//
	u64 magic;
	num_read = fread( &magic, sizeof( magic ), 1, fh );
	if ( num_read != 1 || magic != ROMDB_MAGIC_NO )
	{
		DBGConsole_Msg( 0, "RomDB has wrong magic number." );
		goto fail;
	}

	//
	// Check the version number
	//
	u32 version;
	num_read = fread( &version, sizeof( version ), 1, fh );
	if ( num_read != 1 || version != ROMDB_CURRENT_VERSION )
	{
		DBGConsole_Msg( 0, "RomDB has wrong version for this build of Daedalus." );
		goto fail;
	}

	while ( true )
	{
		u32		num_files;
		num_read = fread( &num_files, sizeof( num_files ), 1, fh );
		if ( num_read != 1 || num_files > MAX_SENSIBLE_FILES )
		{
			DBGConsole_Msg( 0, "RomDB has unexpectedly large number of files." );
			goto fail;
		}

		for( u32 i = 0; i < num_files; ++i )
		{
			RomFilesKeyValue	keyvalue;

			num_read = fread( &keyvalue, sizeof( keyvalue ), 1, fh );
			if ( num_read != 1 )
			{
				goto fail;
			}

			mRomFiles.insert( FilenameMap::value_type( keyvalue.FileName, keyvalue.ID ) );
		}



		u32		num_details;
		num_read = fread( &num_details, sizeof( num_details ), 1, fh );
		if ( num_read != 1 || num_details > MAX_SENSIBLE_DETAILS )
		{
			DBGConsole_Msg( 0, "RomDB has unexpectedly large number of details." );
			goto fail;
		}

		for( u32 i = 0; i < num_details; ++i )
		{
			RomDetailsKeyValue		keyvalue;
			num_read = fread( &keyvalue, sizeof( keyvalue ), 1, fh );
			if ( num_read != 1 )
			{
				goto fail;
			}

			mRomDetails.insert( DetailsMap::value_type( keyvalue.ID, keyvalue.Details ) );
		}

	}

	DBGConsole_Msg( 0, "RomDB initialised with %d files and %d details.", mRomFiles.size(), mRomDetails.size() );

	fclose( fh );
	return true;

fail:
	fclose( fh );
	return false;
}

//*****************************************************************************
//
//*****************************************************************************
bool	IRomDB::Commit()
{
	if( !mDirty )
		return true;

	//
	// Check if we have a valid filename
	// 
	if ( strlen( mRomDBFileName ) <= 0 )
		return false;

	FILE * fh = fopen( mRomDBFileName, "wb" );

	if ( !fh )
		return false;

	//
	// Write the magic
	//
	fwrite( &ROMDB_MAGIC_NO, sizeof( ROMDB_MAGIC_NO ), 1, fh );

	//
	// Write the version
	//
	fwrite( &ROMDB_CURRENT_VERSION, sizeof( ROMDB_CURRENT_VERSION ), 1, fh );
	
	{
		u32 num_files( mRomFiles.size() );
		fwrite( &num_files, sizeof( num_files ), 1, fh );

		u32 count( 0 );
		for ( FilenameMap::const_iterator it = mRomFiles.begin(); it != mRomFiles.end(); ++it )
		{
			RomFilesKeyValue	keyvalue( it->first.c_str(), it->second );
			fwrite( &keyvalue, sizeof( keyvalue ), 1, fh );
			count++;
		}

		DAEDALUS_ASSERT( count == num_files, "Didn't write the expected number of entries" );
	}

	{
		u32 num_details( mRomDetails.size() );
		fwrite( &num_details, sizeof( num_details ), 1, fh );

		u32 count( 0 );
		for ( DetailsMap::const_iterator it = mRomDetails.begin(); it != mRomDetails.end(); ++it )
		{
			RomDetailsKeyValue	keyvalue( it->first, it->second );
			fwrite( &keyvalue, sizeof( keyvalue ), 1, fh );
			count++;
		}

		DAEDALUS_ASSERT( count == num_details, "Didn't write the expected number of entries" );
	}

	fclose( fh );

	mDirty = true;
	return true;
}

//*****************************************************************************
//
//*****************************************************************************
void	IRomDB::AddRom( const char * filename, const RomID & id, u32 rom_size, ECicType cic_type )
{
	mRomFiles[ filename ] = id;
	mRomDetails[ id ] = RomDetails( rom_size, cic_type );
	mDirty = true;
}

namespace
{
bool	GenerateRomDetails( const char * filename, RomID * id, u32 * rom_size, ECicType * cic_type )
{
	//
	//	Haven't seen this rom before - try to add it to the database
	//
	ROMFile * rom_file( ROMFile::Create( filename ) );
	if( rom_file == NULL )
	{
		return false;
	}

	COutputStringStream messages;

	if( !rom_file->Open( messages ) )
	{
		delete rom_file;
		return false;
	}

	//
	// They weren't there - so we need to find this info out for ourselves
	// Only read in the header + bootcode
	//
	u32			bytes_to_read( RAMROM_GAME_OFFSET );
	u32			buffer_size;
	u8 *		bytes;
	if(!rom_file->LoadDataChunk( bytes_to_read, &bytes, &buffer_size, messages ))
	{
		// Lots of files don't have any info - don't worry about it
		delete rom_file;
		return false;
	}

	//
	// Get the address of the rom header
	// Setup the rom id and size
	//
	*rom_size = rom_file->GetRomSize();
	if (*rom_size >= RAMROM_GAME_OFFSET)
	{
		*cic_type = ROM_GenerateCICType( bytes );
	}
	else
	{
		*cic_type = CIC_UNKNOWN;
	}

	//
	//	Swap into native format
	//
	ROMFile::ByteSwap_3210( bytes, buffer_size );

	const ROMHeader * prh( reinterpret_cast<const ROMHeader *>( bytes ) );
	*id = RomID( prh->CRC1, prh->CRC2, prh->CountryID );

	delete [] bytes;
	delete rom_file;
	return true;
}
}

//*****************************************************************************
//
//*****************************************************************************
bool	IRomDB::QueryByFilename( const char * filename, RomID * id, u32 * rom_size, ECicType * cic_type )
{
	//
	// First of all, check if we have these details cached in the rom database
	//
	FilenameMap::const_iterator	fit( mRomFiles.find( filename ) );
	if( fit != mRomFiles.end() )
	{
		DetailsMap::const_iterator	dit( mRomDetails.find( fit->second ) );
		if( dit != mRomDetails.end() )
		{
			*id = dit->first;
			*rom_size = dit->second.RomSize;
			*cic_type = dit->second.CicType;
			return true;
		}
	}

	if( GenerateRomDetails( filename, id, rom_size, cic_type ) )
	{
		//
		// Store this information for future reference
		//
		AddRom( filename, *id, *rom_size, *cic_type );
		return true;
	}

	return false;
}

//*****************************************************************************
//
//*****************************************************************************
bool	IRomDB::QueryByID( const RomID & id, u32 * rom_size, ECicType * cic_type ) const
{
	DetailsMap::const_iterator	dit( mRomDetails.find( id ) );
	if( dit != mRomDetails.end() )
	{
		*rom_size = dit->second.RomSize;
		*cic_type = dit->second.CicType;
		return true;
	}

	// Can't generate the details, as we have no filename to work with
	return false;
}

//*****************************************************************************
//
//*****************************************************************************
const char *	IRomDB::QueryFilenameFromID( const RomID & id ) const
{
	for( FilenameMap::const_iterator it = mRomFiles.begin(); it != mRomFiles.end(); ++it )
	{
		if( it->second == id )
		{
			return it->first.c_str();
		}
	}

	return NULL;
}
