/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Display stuff like registers, instructions, memory usage and so on
#include "stdafx.h"

#include "Dump.h"
#include "DebugLog.h"
#include "DBGConsole.h"

#include "Core/Memory.h"
#include "Core/CPU.h"
#include "Core/RSP.h"
#include "Core/Interrupt.h"
#include "Core/ROMBuffer.h"
#include "Utility/PrintOpCode.h"
#include "OSHLE/Patch.h"		// For GetCorrectOp
#include "OSHLE/ultra_r4300.h"
#include "Interface/MainWindow.h"	// Main_SelectSaveDir()

#include "DaedIO.h"


static char gDumpDir[MAX_PATH+1] = "";

//*****************************************************************************
// Initialise the directory where files are dumped
//*****************************************************************************
void Dump_GetDumpDirectory(char * p_file_path, const char * p_sub_dir)
{
	// Appends szBase to the global dump base. Stores in p_file_path


	if (strlen(gDumpDir) == 0)
	{
		// Initialise
		IO::Path::Combine(gDumpDir, gDaedalusExePath, "Dumps");
	}

	// If a subdirectory was specified, append 
	if (strlen(p_sub_dir) > 0)
	{
		IO::Path::Combine(p_file_path, gDumpDir, p_sub_dir);
	}
	else
	{
		strncpy(p_file_path, gDumpDir, MAX_PATH);
		p_file_path[MAX_PATH-1] = '\0';
	}

	if(CDebugConsole::IsAvailable())
	{
		DBGConsole_Msg( 0, "Dump dir: [C%s]", p_file_path );
	}

	IO::Directory::EnsureExists(p_file_path);

}


//*****************************************************************************
// E.g. Dump_GetSaveDirectory([out], "c:\roms\test.rom", ".sra")
// would first try to find the save in g_DaedalusConfig.szSaveDir. If this is not
// found, g_DaedalusConfig.szRomsDir is checked.
//*****************************************************************************
void Dump_GetSaveDirectory(char * p_file_path, const char * p_rom_name, const char * p_ext)
{
	char file_name[MAX_PATH+1];

	// If the Save path has not yet been set up, prompt user
	if (strlen(g_DaedalusConfig.szSaveDir) == 0)
	{
	#ifdef DAEDALUS_W32
		CMainWindow::Get()->SelectSaveDir( CMainWindow::Get()->GetWindow() );
	#endif

		// User may have cancelled
		if (strlen(g_DaedalusConfig.szSaveDir) == 0)
		{
			// Default to rom path
			strncpy(g_DaedalusConfig.szSaveDir, p_rom_name, MAX_PATH);
			g_DaedalusConfig.szSaveDir[MAX_PATH-1] = '\0';
			IO::Path::RemoveFileSpec(g_DaedalusConfig.szSaveDir);

			if(CDebugConsole::IsAvailable())
			{
				DBGConsole_Msg(0, "SaveDir is still empty - defaulting to [C%s]", g_DaedalusConfig.szSaveDir);
			}
		}
	}

	// Form the filename from the file spec (i.e. strip path)
	strcpy( file_name, IO::Path::FindFileName( p_rom_name ) );
	IO::Path::RemoveExtension( file_name );
	IO::Path::AddExtension(file_name, p_ext);

	IO::Path::Combine(p_file_path, g_DaedalusConfig.szSaveDir, file_name);
	if (IO::File::Exists(p_file_path))
		return;

	IO::Path::Combine(p_file_path, g_DaedalusConfig.szRomsDirs[ 0 ], file_name);
	if (IO::File::Exists(p_file_path))
		return;

	// Not in either dir - default to SaveDir
	IO::Path::Combine(p_file_path, g_DaedalusConfig.szSaveDir, file_name);

}


#ifdef DAEDALUS_W32
//*****************************************************************************
// p_base[0] = instruction at start
//*****************************************************************************
static void DumpRange(const char * p_file_name,
					  const char * p_mode,
					  u32 * p_base,
					  u32 start,
					  u32 end,
					  BOOL bTranslatePatches)
{
	char opinfo[400];
	FILE * fp;	

	fp = fopen(p_file_name, p_mode);
	if (fp == NULL)
		return;

	for (u32 current_pc = start; current_pc < end; current_pc+=4)
	{
		OpCode original_op;
		OpCode op;

		original_op._u32 = p_base[(current_pc-start)/4];

		if (bTranslatePatches)
		{
			op = GetCorrectOp( original_op );
#ifdef DAEDALUS_ENABLE_OS_HOOKS
			if (IsJumpTarget( original_op ))
			{
				fprintf(fp, "\n");
				fprintf(fp, "\n");
				fprintf(fp, "// %s():\n", Patch_GetJumpAddressName(current_pc));
			}
#endif
		}			
		else
		{
			op = GetCorrectOp( original_op );
		}

		SprintOpCodeInfo( opinfo, current_pc, op );
		fprintf(fp, "0x%08x: <0x%08x> %s\n", current_pc, op._u32, opinfo);
	}

	fclose(fp);
}
#endif

#ifdef DAEDALUS_W32
//*****************************************************************************
//
//*****************************************************************************
void Dump_Disassemble(u32 start, u32 end, const char * p_file_name)
{
	char file_path[MAX_PATH+1];

	// Cute hack - if the end of the range is less than the start,
	// assume it is a length to disassemble
	if (end < start)
		end = start + end;

	if (p_file_name == NULL || strlen(p_file_name) == 0)
	{
		Dump_GetDumpDirectory(file_path, "");
		IO::Path::Append(file_path, "dis.txt");
	}
	else
	{
		strncpy(file_path, p_file_name, MAX_PATH);
		file_path[MAX_PATH-1] = '\0';
	}

		
	u32 * p_base;
	if (!Memory_GetInternalReadAddress(start, (void**)&p_base))
	{
		DBGConsole_Msg(0, "[Ydis: Invalid base 0x%08x]", start);
	}
	else
	{
		DBGConsole_Msg(0, "Disassembling from 0x%08x to 0x%08x ([C%s])",
			start, end, file_path);

		DumpRange(file_path, "w", p_base, start, end, TRUE);
	}
}
#endif


#ifdef DAEDALUS_W32

//*****************************************************************************
// p_base[0] = instruction at start
//*****************************************************************************
static void DumpRSPRange(const char * p_file_name,
						 const char * p_mode,
						 u32 * p_base,
						 u32 start,
						 u32 end)
{
	char opinfo[400];
	FILE * fp;	

	fp = fopen(p_file_name, p_mode);
	if (fp == NULL)
		return;

	for (u32 current_pc = start; current_pc < end; current_pc+=4)
	{
		OpCode op;
		op._u32 = p_base[(current_pc-start)/4];

		SprintRSPOpCodeInfo( opinfo, current_pc, op );
		fprintf(fp, "0x%08x: <0x%08x> %s\n", current_pc, op._u32, opinfo);
	}

	fclose(fp);
}
#endif

#ifdef DAEDALUS_W32
//*****************************************************************************
//
//*****************************************************************************
void Dump_RSPDisassemble(const char * p_file_name)
{
	char file_path[MAX_PATH+1];
	FILE * fp;
	u32 start = 0xa4000000;
	u32 end = 0xa4002000;

	if (p_file_name == NULL || strlen(p_file_name) == 0)
	{
		Dump_GetDumpDirectory(file_path, "");
		IO::Path::Append(file_path, "rdis.txt");
	}
	else
	{
		strncpy(file_path, p_file_name, MAX_PATH);
		file_path[MAX_PATH-1] = '\0';
	}


	u32 * p_base;
	if (!Memory_GetInternalReadAddress(start, (void**)&p_base))
	{
		DBGConsole_Msg(0, "[Yrdis: Invalid base 0x%08x]", start);
		return;
	}

	DBGConsole_Msg(0, "Disassembling from 0x%08x to 0x%08x ([C%s])",
		start, end, file_path);


	// Overwrite here
	fp = fopen(file_path, "w");
	if (fp == NULL)
		return;
	
	// Memory dump
	u32 current_pc;
	for (current_pc = 0xa4000000; current_pc < 0xa4001000; current_pc += 16)
	{
		fprintf(fp, "0x%08x: %08x %08x %08x %08x ", current_pc, 
			p_base[(current_pc-start)/4 + 0],
			p_base[(current_pc-start)/4 + 1],
			p_base[(current_pc-start)/4 + 2],
			p_base[(current_pc-start)/4 + 3]);

		// Concat characters
		for (int i = 0; i < 16; i++)
		{
			char c = ((u8*)p_base)[((current_pc-start)+i) ^ 0x3];
			// The cast to unsigned int avoids widening char to a signed int.
			// This cast avoids a failed assertion popup in MSVC 7.0
			if (isprint((unsigned int)(unsigned char)c))
				fprintf(fp, "%c", c);
			else 
				fprintf(fp, ".");

			if ((i%4)==3)
				fprintf(fp, " ");
		}
		fprintf(fp, "\n");
	}
	fclose(fp);

	// Append here
	DumpRSPRange(file_path, "a", p_base+(0x1000/4), start+0x1000, end);

}
#endif


//*****************************************************************************
//
//*****************************************************************************
void Dump_Strings( const char * p_file_name )
{
	char file_path[ MAX_PATH+1 ];
	FILE * fp;

	static const u32 MIN_LENGTH = 5;

	if (p_file_name == NULL || strlen(p_file_name) == 0)
	{
		Dump_GetDumpDirectory(file_path, "");
		IO::Path::Append(file_path, "strings.txt");
	}
	else
	{
		strncpy(file_path, p_file_name, MAX_PATH);
		file_path[MAX_PATH-1] = '\0';
	}

	DBGConsole_Msg(0, "Dumping strings in rom ([C%s])", file_path);
	
	// Overwrite here
	fp = fopen(file_path, "w");
	if (fp == NULL)
		return;
	
	// Memory dump
	u32 ascii_start = 0;
	u32 ascii_count = 0;
	for ( u32 i = 0; i < RomBuffer::GetRomSize(); i ++)
	{
		if ( RomBuffer::ReadValueRaw< u8 >( i ^ 0x3 ) >= ' ' &&
			 RomBuffer::ReadValueRaw< u8 >( i ^ 0x3 ) < 200 )
		{
			if ( ascii_count == 0 )
			{
				ascii_start = i;
			}
			ascii_count++;
		}
		else
		{
			if ( ascii_count >= MIN_LENGTH )
			{
				fprintf( fp, "0x%08x: ", ascii_start );

				for ( u32 j = 0; j < ascii_count; j++ )
				{
					fprintf( fp, "%c", RomBuffer::ReadValueRaw< u8 >( (ascii_start + j ) ^ 0x3 ) );
				}

				fprintf( fp, "\n");
			}

			ascii_count = 0;
		}
	}
	fclose(fp);

}


