/*
Copyright (C) 2007 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "AudioBuffer.h"

#include "Debug/DBGConsole.h"

//*****************************************************************************
//
//*****************************************************************************
static const u32	BUFFER_SIZE = 1024 * 16;		// Usually have to provide 1024 samples at a time, so this gives us 8 batches of them

static const u32	DESIRED_OUTPUT_FREQUENCY = 44100;
static const u32	MAX_OUTPUT_FREQUENCY = DESIRED_OUTPUT_FREQUENCY * 4;

static const u32	ADAPTIVE_FREQUENCY_ADJUST = 2000;

//*****************************************************************************
//
//*****************************************************************************
CAudioBuffer::CAudioBuffer()
	:	mBuffer( new Sample[ BUFFER_SIZE ] )
	,	mBufferLength( BUFFER_SIZE )
	,	mReadPos( 0 )
	,	mWritePos( 0 )
	,	mNumSamples( 0 )
	,	mOutputFrequency( DESIRED_OUTPUT_FREQUENCY )
	,	mAdaptFrequency( false )
{
}

//*****************************************************************************
//
//*****************************************************************************
CAudioBuffer::~CAudioBuffer()
{
	delete [] mBuffer;
}

//*****************************************************************************
//
//*****************************************************************************
void	CAudioBuffer::SetAdaptFrequency( bool adapt )
{
	mAdaptFrequency = adapt;
	mOutputFrequency = DESIRED_OUTPUT_FREQUENCY;
}

//*****************************************************************************
//
//*****************************************************************************
bool	CAudioBuffer::CanAdd( u32 num_samples, u32 frequency ) const
{
	u32		output_samples( ( num_samples * mOutputFrequency ) / frequency );
	u32		avail_space( mBufferLength - mNumSamples );

	return output_samples <= avail_space;
}

//*****************************************************************************
//
//*****************************************************************************
bool	CAudioBuffer::Append( Sample s )
{
	if( mNumSamples < mBufferLength )
	{
		mBuffer[ mWritePos ] = s;

		u32		next( mWritePos + 1 );
		if( next < mBufferLength )
			mWritePos = next;
		else
			mWritePos = 0;

		mNumSamples++;
		return true;
	}

	return false;
}

//*****************************************************************************
//
//*****************************************************************************
void	CAudioBuffer::AddSamples( const Sample * samples, u32 num_samples, u32 frequency )
{
	DAEDALUS_ASSERT( frequency <= mOutputFrequency, "Input frequency is too high" );

	u32			output_samples( ( num_samples * mOutputFrequency ) / frequency );

	//
	//	'r' is the number of input samples we progress through for each output sample.
	//	's' keeps track of how far between the current two input samples we are.
	//	We increment it by 'r' for each output sample we generate.
	//	When it reaches 1.0, we know we've hit the next sample, so we increment in_idx
	//	and reduce s by 1.0 (to keep it in the range 0.0 .. 1.0)
	//
	float		r( float( frequency ) / float( mOutputFrequency ) );
	float		s( 0.0f );
	u32			in_idx( 0 );

	for( u32 i = 0; i < output_samples; ++i )
	{
		DAEDALUS_ASSERT( in_idx + 1 < num_samples, "Input index out of range" );

		const Sample &	in_a( samples[ in_idx + 0 ] );
		const Sample &	in_b( samples[ in_idx + 1 ] );

		Sample	out( in_a.Interpolate( in_b, s ) );

		s += r;
		if( s > 1.0f )
		{
			s -= 1.0f;
			in_idx++;
		}

		DAEDALUS_ASSERT( s >= 0.0f && s < 1.0f, "s out of range" );

		if( !Append( out ) )
		{
			// The buffer is full - break out and drop the remaining samples
			DBGConsole_Msg( 0, "Buffer full - dropping %d samples", output_samples - i );
			break;
		}
	}

	if( mNumSamples > (mBufferLength - output_samples) )
	{
		DodgeBufferOverflow();
	}
}

//*****************************************************************************
//
//*****************************************************************************
void	CAudioBuffer::Fill( Sample * samples, u32 num_samples )
{
	u32		samples_required( num_samples );
	while( samples_required > 0 )
	{
		if( mNumSamples == 0 )
			break;

		*samples++ = mBuffer[ mReadPos ];

		u32		next( mReadPos + 1 );
		if( next < mBufferLength )
			mReadPos = next;
		else
			mReadPos = 0;

		mNumSamples--;
		samples_required--;
	}

	if( samples_required > 0 )
	{
		//DBGConsole_Msg( 0, "Buffer underflow (%d samples)", samples_required );

		memset( samples, 0, samples_required * sizeof( Sample ) );
	}

	if( mNumSamples < num_samples )
	{
		DodgeBufferUnderflow();
	}
}

//*****************************************************************************
//
//*****************************************************************************
void	CAudioBuffer::DodgeBufferUnderflow()
{
	if( mAdaptFrequency )
	{
		u32 adjust( ADAPTIVE_FREQUENCY_ADJUST );
		if(mOutputFrequency + adjust < MAX_OUTPUT_FREQUENCY)
		{
			mOutputFrequency += adjust;
			//DBGConsole_Msg( 0, "Adjusting freq to %d", mOutputFrequency );
		}
		else
		{
			mOutputFrequency = MAX_OUTPUT_FREQUENCY;
		}
	}
}

//*****************************************************************************
//
//*****************************************************************************
void	CAudioBuffer::DodgeBufferOverflow()
{
	if( mAdaptFrequency )
	{
		u32 adjust( ADAPTIVE_FREQUENCY_ADJUST );
		if( mOutputFrequency - adjust > DESIRED_OUTPUT_FREQUENCY)
		{
			mOutputFrequency -= adjust;
			//DBGConsole_Msg( 0, "Restoring freq to %d", mOutputFrequency );
		}
		else
		{
			mOutputFrequency = DESIRED_OUTPUT_FREQUENCY;
		}
	}
}
