/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifdef _MSC_VER
#include <crtdbg.h>
#endif

#include "DynaRecFP.h"

#include "OSHLE\ultra_R4300.h"

#include "CodeGeneratorX86.h"

struct fpRegInfo
{
	DWORD dwStackPos;		// ~0 indicates not loaded at present
							// 0 .. 7 indicates actual position on stack
	BOOL bDirty;			// TRUE if we need to write back to g_CPR[1][fp];

	DWORD dwTimestamp;		// Time of last use. Lower numbers indicate older 

	FPREG_TYPE nType;		// FP_SINGLE if cache of 32bit (single) value
							// FP_DOUBLE if cache of 64bit (double) value

};

static fpRegInfo g_FPReg[32];
static DWORD g_dwFPStackContents[8];	// What the current stack contains

u32 gFPTimeNow = 0;

static CCodeGeneratorX86 * g_pFPpCode = NULL;


void SR_FP_Init(CCodeGeneratorX86 * pCode)
{
	DWORD i;

	for (i = 0; i < 32; i++)
	{
		g_FPReg[i].bDirty = FALSE;
		g_FPReg[i].dwStackPos = u32(~0);	// Not on stack at the moment;
		g_FPReg[i].nType = FP_SINGLE;
	}

	for (i = 0; i < 8; i++)
	{
		g_dwFPStackContents[i] = u32(~0);
	}

	g_pFPpCode = pCode;
}

void SR_FP_FlushAllRegs()
{
	DWORD dwSafeCount;

	//for (i = 0; i < 32; i++)
	dwSafeCount = 0;
	while (g_dwFPStackContents[0] != u32(~0))
	{
		u32 reg_fs = g_dwFPStackContents[0];
		
		//if (g_FPReg[i].dwStackPos != ~0)
		{
			if (g_FPReg[reg_fs].bDirty)
				SR_FP_SaveReg(reg_fs);
			else
				SR_FP_DiscardReg(reg_fs);
		}

		// Make sure we don't keep looping forever!
		dwSafeCount++;
		if (dwSafeCount >= 8)
		{
			DPF(DEBUG_DYNREC, "!!!! Unable to flush FP regs!");
			break;
		}
	}

	if (dwSafeCount > 0)
	{
		DPF( DEBUG_DYNREC, DSPrintf( "Flushed %d fp regs", dwSafeCount ) );	
	}

	// Assert all regs saved?
#ifdef _DEBUG
	for (u32 i = 0; i < 32; i++)
	{
		DAEDALUS_ASSERT_Q(g_FPReg[i].bDirty == FALSE);
	}
#endif
}

void SR_FP_StackPicture()
{

	DPF(DEBUG_DYNREC, DSPrintf(
		"%02d %02d %02d %02d %02d %02d %02d %02d",
		g_dwFPStackContents[0] == ~0 ? -1 : g_dwFPStackContents[0],
		g_dwFPStackContents[1] == ~0 ? -1 : g_dwFPStackContents[1],
		g_dwFPStackContents[2] == ~0 ? -1 : g_dwFPStackContents[2],
		g_dwFPStackContents[3] == ~0 ? -1 : g_dwFPStackContents[3],
		g_dwFPStackContents[4] == ~0 ? -1 : g_dwFPStackContents[4],
		g_dwFPStackContents[5] == ~0 ? -1 : g_dwFPStackContents[5],
		g_dwFPStackContents[6] == ~0 ? -1 : g_dwFPStackContents[6],
		g_dwFPStackContents[7] == ~0 ? -1 : g_dwFPStackContents[7] ) );

}

// Swap positions of registers dwSP1 and dwSP2
// Assumes: both registers have valid contents!
// Assumes: dwSP1 is 0!
// Assumes: dwSP1 != dwSP2
void SR_FP_ExchReg(DWORD dwSP1, DWORD dwSP2)
{
	DAEDALUS_ASSERT_Q(dwSP1 == 0);
	DAEDALUS_ASSERT_Q(dwSP1 != dwSP2);

	// Swap stackpos of 
	DWORD dwFP1 = g_dwFPStackContents[dwSP1];
	DWORD dwFP2 = g_dwFPStackContents[dwSP2];

	// Update reg indicators:
	g_FPReg[dwFP2].dwStackPos = dwSP1;
	g_FPReg[dwFP1].dwStackPos = dwSP2;

	// Update stack indicators
	g_dwFPStackContents[dwSP2] = dwFP1;
	g_dwFPStackContents[dwSP1] = dwFP2;

	// Emit instructions for fxch!
	DPF( DEBUG_DYNREC, DSPrintf( "fxch	st(%d)", dwSP2 ) );

	g_pFPpCode->FXCH(dwSP2);

}

// If the register is not on the stack, it is loaded into position 0,
// Assumes: Any registers required for this operation have been "touched"
//          so that they won't be evicted
// Assumes: Register is not already loaded
void SR_FP_LoadReg(u32 reg_fs, FPREG_TYPE nType)
{
	DAEDALUS_ASSERT_Q(g_FPReg[reg_fs].dwStackPos == u32(~0));

	// Need to load into new position

	// Check if stack is full. If so, pop oldest
	if (g_dwFPStackContents[7] != u32(~0))
	{
		// This register will be evicted by the load - we have to either store or discard
		// the oldest register to prevent this...
		SR_FP_EvictOldestRegister();
	}

	// Move regs up stack - this is to simulate the fld
	SR_FP_PushRegs();

	// The contents of st(0) are the new register
	g_dwFPStackContents[0] = reg_fs;
	
	// Load register into position 0 and set timestamp
	g_FPReg[reg_fs].dwStackPos = 0;
	g_FPReg[reg_fs].bDirty = FALSE;
	g_FPReg[reg_fs].dwTimestamp = gFPTimeNow;
	g_FPReg[reg_fs].nType = nType;

	// Emit instruction for fld !!!!!!!
	if (nType == FP_SINGLE)
	{
		DPF( DEBUG_DYNREC, DSPrintf( "fld    dword ptr [FP%02d]", reg_fs ) );
		g_pFPpCode->FLD_MEMp32(&g_qwCPR[1][reg_fs]);
	}
	else
	{
		DPF( DEBUG_DYNREC, DSPrintf( "fld    qword ptr [FP%02d]", reg_fs ) );
		g_pFPpCode->FLD_MEMp64(&g_qwCPR[1][reg_fs], &g_qwCPR[1][reg_fs+1]);
	}
	
}

// This removes the oldest register from the stack.
// Assumes: Any registers required for this operation have been "touched"
// Assumes: Stack is not empty
void SR_FP_EvictOldestRegister()
{
	DWORD dwOldest = 0xFFFFFFFF;
	DWORD dwOldestIndex = u32(~0);

	DWORD i;
	DWORD dwFi;

	for (i = 0; i < 7; i++)
	{
		dwFi = g_dwFPStackContents[i];

		// Ignore reg if empty
		if (dwFi == u32(~0))
			continue;

		if (g_FPReg[dwFi].dwTimestamp < dwOldest)
		{
			dwOldest = g_FPReg[dwFi].dwTimestamp;
			dwOldestIndex = i;
		}
	}

	// dwOldestIndex should NOT be ~0 here, unless the stack is empty
	DAEDALUS_ASSERT_Q(dwOldestIndex != u32(~0));

	// Pop this value off the top of the stack.
	SR_FP_PopValueOffStack(dwOldestIndex);

}

// 
// If the register at the top of the stack is dirty, store it, else discard it
// Assumes: dwSI contains a register
void SR_FP_PopValueOffStack(DWORD dwSI)
{
	u32 reg_fs;
		
	DAEDALUS_ASSERT_Q(g_dwFPStackContents[dwSI] != u32(~0));
	
	// Exchange the oldest register with the top of the stack (if it is not top already)
	if (dwSI != 0)
		SR_FP_ExchReg(0, dwSI);

	reg_fs = g_dwFPStackContents[0];

	// Should be same value as assert above
	DAEDALUS_ASSERT_Q(reg_fs != u32(~0));

	if (g_FPReg[reg_fs].bDirty)
	{
		SR_FP_SaveReg(reg_fs);
	}
	else
	{
		SR_FP_DiscardReg(reg_fs);
	}

}

// Save contents of FP reg
// Assumes: reg_fs is currently on the stack
void SR_FP_SaveReg(u32 reg_fs)
{
	DWORD dwSI;
	DAEDALUS_ASSERT_Q(reg_fs != u32(~0));
	DAEDALUS_ASSERT_Q(g_FPReg[reg_fs].bDirty);

	// Move FD to top of stack
	dwSI = g_FPReg[reg_fs].dwStackPos;

	DAEDALUS_ASSERT_Q(dwSI != u32(~0));

	// Move to top of stack (if it's not there already)
	if (dwSI != 0)
		SR_FP_ExchReg(0, g_FPReg[reg_fs].dwStackPos);

	if (g_FPReg[reg_fs].nType == FP_SINGLE)
	{
		DPF( DEBUG_DYNREC, DSPrintf( "fstp   dword ptr [FP%02d]		// Save", reg_fs ) );
		g_pFPpCode->FSTP_MEMp32(&g_qwCPR[1][reg_fs]);
	}
	else
	{
		DPF( DEBUG_DYNREC, DSPrintf( "fstp   qword ptr [FP%02d]		// Save", reg_fs ) );
		g_pFPpCode->FSTP_MEMp64(&g_qwCPR[1][reg_fs], &g_qwCPR[1][reg_fs+1]);
	}
	
	g_FPReg[reg_fs].dwStackPos = u32(~0);
	g_dwFPStackContents[0] = u32(~0);
	g_FPReg[reg_fs].bDirty = FALSE;

	SR_FP_PopRegs();
	
}


// As above, but just discards contents
// Assumes: reg_fs is currently on the stack
void SR_FP_DiscardReg(u32 reg_fs)
{
	DWORD dwSI;
	DAEDALUS_ASSERT_Q(reg_fs != u32(~0));

	// Move FD to top of stack
	dwSI = g_FPReg[reg_fs].dwStackPos;

	DAEDALUS_ASSERT_Q(dwSI != u32(~0));

	// Move to top of stack (if it's not there already)
	if (dwSI != 0)
		SR_FP_ExchReg(0, g_FPReg[reg_fs].dwStackPos);

	DPF( DEBUG_DYNREC,"ffree	st(0)" );
	g_pFPpCode->FFREE(0);
	DPF( DEBUG_DYNREC, DSPrintf( "fincstp      //(discard FP%02d)", reg_fs ) );
	g_pFPpCode->FINCSTP();
	
	g_FPReg[reg_fs].dwStackPos = u32(~0);
	
	g_dwFPStackContents[0] = u32(~0);
	g_FPReg[reg_fs].bDirty = FALSE;
	
	SR_FP_PopRegs();

}


// Push registers up the stack
// Assumes: st(7) is empty
void SR_FP_PushRegs()
{
	DWORD i;

	DAEDALUS_ASSERT_Q(g_dwFPStackContents[7] == u32(~0));

	for (i = 7; i >= 1; i--)
		g_dwFPStackContents[i] = g_dwFPStackContents[i - 1];

	// Bottom is empty now
	g_dwFPStackContents[0] = u32(~0);

	for (i = 0; i < 32; i++)
		if (g_FPReg[i].dwStackPos != u32(~0))
			g_FPReg[i].dwStackPos++;

}

// Pop register from the stack.
// Assumes: St(0) is empty
void SR_FP_PopRegs()
{
	DWORD i;

	DAEDALUS_ASSERT_Q(g_dwFPStackContents[0] == u32(~0));

	for (i = 0; i < 7; i++)
		g_dwFPStackContents[i] = g_dwFPStackContents[i + 1];

	// Top is empty now
	g_dwFPStackContents[7] = u32(~0);

	for (i = 0; i < 32; i++)
		if (g_FPReg[i].dwStackPos != u32(~0))
			g_FPReg[i].dwStackPos--;

}

// "touch" a register to update it to the current time. This prevents it from 
// being evicted this "cycle"
void SR_FP_Touch(u32 reg_fs)
{
	g_FPReg[reg_fs].dwTimestamp = gFPTimeNow;
}

// Ensure that the two source registers are available for 
// the calculation, and discard the contents of the 
// destination register if it is not used
void SR_FP_Prepare_D_S_S(u32 reg_fd, u32 reg_fs, u32 reg_ft, FPREG_TYPE nType)
{
	SR_FP_Touch(reg_fs);
	SR_FP_Touch(reg_ft);

	// If the destination register is not used in the calculation, discard
	if (g_FPReg[reg_fd].dwStackPos != u32(~0) &&	// Register is on the stack
		reg_fd != reg_fs &&							// Register is not FS
		reg_fd != reg_ft)							// Register is not FT
		SR_FP_DiscardReg(reg_fd);

	// Ensure register is loaded
	if (g_FPReg[reg_fs].dwStackPos == u32(~0))
		SR_FP_LoadReg(reg_fs, nType);

	if (g_FPReg[reg_ft].dwStackPos == u32(~0))
		SR_FP_LoadReg(reg_ft, nType);
}


// Ensure that the two source registers are available for 
// the calculation
void SR_FP_Prepare_S_S(u32 reg_fs, u32 reg_ft, FPREG_TYPE nType)
{
	SR_FP_Touch(reg_fs);
	SR_FP_Touch(reg_ft);

	// Ensure register is loaded
	if (g_FPReg[reg_fs].dwStackPos == u32(~0))
		SR_FP_LoadReg(reg_fs, nType);

	if (g_FPReg[reg_ft].dwStackPos == u32(~0))
		SR_FP_LoadReg(reg_ft, nType);
}

// Ensure that the source register is available for 
// the calculation, and discard the contents of the 
// destination register if it is not used
void SR_FP_Prepare_D_S(u32 reg_fd, u32 reg_fs, FPREG_TYPE nType)
{
	SR_FP_Touch(reg_fs);

	// If the destination register is not used in the calculation, discard
	if (g_FPReg[reg_fd].dwStackPos != u32(~0) &&	// Register is on the stack
		reg_fd != reg_fs)							// Register is not FS
		SR_FP_DiscardReg(reg_fd);

	// Ensure register is loaded
	if (g_FPReg[reg_fs].dwStackPos == u32(~0))
		SR_FP_LoadReg(reg_fs, nType);

}


// Copies a register to st(0) in preparation for a calculation
// Assumes: reg_fs is already loaded
// ---Assumes: st(7) is empty -- We evict a register if necessary
void SR_FP_CopyRegToST0(u32 reg_fs)
{
	DWORD dwSI;
	
	if (g_dwFPStackContents[7] != u32(~0))
		SR_FP_EvictOldestRegister();

	dwSI = g_FPReg[reg_fs].dwStackPos;	// Keep copy of stack pos

	DAEDALUS_ASSERT_Q(dwSI != u32(~0));
	//DAEDALUS_ASSERT_Q(g_dwFPStackContents[7] == u32(~0));
	
	SR_FP_PushRegs();		// Make space at 0

	DPF( DEBUG_DYNREC, DSPrintf( "fld    st(%d)    //(FP%02d)", dwSI, reg_fs ) );
	g_pFPpCode->FLD(dwSI);
	
}

// Clears st(0) in preparation for a calculation
// Assumes: reg_fs is already loaded
// ---Assumes: st(7) is empty -- We evict a register if necessary
void SR_FP_LoadWordToST0(u32 reg_fs)
{
	if (g_dwFPStackContents[7] != u32(~0))
		SR_FP_EvictOldestRegister();

	SR_FP_PushRegs();		// Make space at 0

	DPF( DEBUG_DYNREC, DSPrintf( "fild    //(FP%02d)", reg_fs ) );
	g_pFPpCode->FILD_MEM(&g_qwCPR[1][reg_fs]);

}

// Assigns st(0) to reg FD
// Assumes: st(0) is currently unassigned
//
// If FD is currently assigned, this code will exchange the result with that reg
// on the top, and discard the old contents of st(FD)
void SR_FP_AssignToST0(u32 reg_fd, FPREG_TYPE nType)
{
	DAEDALUS_ASSERT_Q(g_dwFPStackContents[0] == u32(~0));

	if (g_FPReg[reg_fd].dwStackPos == u32(~0))
	{
		g_dwFPStackContents[0] = reg_fd;
		g_FPReg[reg_fd].dwStackPos = 0;
		g_FPReg[reg_fd].bDirty = TRUE;
		g_FPReg[reg_fd].dwTimestamp = gFPTimeNow;
		g_FPReg[reg_fd].nType = nType;
	}
	else
	{
		// Can't swap with ourselves!
		DAEDALUS_ASSERT_Q(g_FPReg[reg_fd].dwStackPos != 0);

		DWORD dwSI = g_FPReg[reg_fd].dwStackPos;

		// Emit instructions for fxch!
		DPF( DEBUG_DYNREC,"//Discarding old value!" );
		DPF( DEBUG_DYNREC, DSPrintf( "fxch	st(%d)", dwSI ) );
		g_pFPpCode->FXCH(dwSI);
		
		// Stackpos stays the same. Update dirty etc
		g_FPReg[reg_fd].bDirty = TRUE;
		g_FPReg[reg_fd].dwTimestamp = gFPTimeNow;
		g_FPReg[reg_fd].nType = nType;
		
		// Discard the top value
		DPF( DEBUG_DYNREC,"ffree	st(0)" );
		g_pFPpCode->FFREE(0);
		DPF( DEBUG_DYNREC,"fincstp      //(discard)" );
		g_pFPpCode->FINCSTP();
				
		SR_FP_PopRegs();
		
	}	
	
}

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////

void SR_FP_ADD(u32 reg_fd, u32 reg_fs, u32 reg_ft, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S_S(reg_fd, reg_fs, reg_ft, nType);

	SR_FP_CopyRegToST0(reg_fs);

	// Add 
	DWORD dwST = g_FPReg[reg_ft].dwStackPos;
	DPF( DEBUG_DYNREC, DSPrintf( "fadd   st(0),st(%d)    //(FP%02d)", dwST, reg_ft ) );
	g_pFPpCode->FADD(dwST);
	
	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}


void SR_FP_SUB(u32 reg_fd, u32 reg_fs, u32 reg_ft, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S_S(reg_fd, reg_fs, reg_ft, nType);

	SR_FP_CopyRegToST0(reg_fs);

	// Sub 
	DWORD dwST = g_FPReg[reg_ft].dwStackPos;
	DPF( DEBUG_DYNREC, DSPrintf( "fsub   st(0),st(%d)    //(FP%02d)", dwST, reg_ft ) );
	g_pFPpCode->FSUB(dwST);
	
	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}




void SR_FP_MUL(u32 reg_fd, u32 reg_fs, u32 reg_ft, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S_S(reg_fd, reg_fs, reg_ft, nType);

	SR_FP_CopyRegToST0(reg_fs);

	// Mul 
	DWORD dwST = g_FPReg[reg_ft].dwStackPos;
	DPF( DEBUG_DYNREC, DSPrintf( "fmul   st(0),st(%d)    //(FP%02d)", dwST, reg_ft ) );
	g_pFPpCode->FMUL(dwST);

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}


void SR_FP_DIV(u32 reg_fd, u32 reg_fs, u32 reg_ft, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S_S(reg_fd, reg_fs, reg_ft, nType);

	SR_FP_CopyRegToST0(reg_fs);

	// Div 
	DWORD dwST = g_FPReg[reg_ft].dwStackPos;
	DPF( DEBUG_DYNREC, DSPrintf( "fdiv   st(0),st(%d)    //(FP%02d)", dwST, reg_ft ) );
	g_pFPpCode->FDIV(dwST);

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////


void SR_FP_SQRT(u32 reg_fd, u32 reg_fs, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, nType);
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// Sqrt
	DPF(DEBUG_DYNREC,"fsqrt");
	g_pFPpCode->FSQRTp();

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}

void SR_FP_NEG(u32 reg_fd, u32 reg_fs, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, nType);
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// fchs
	DPF(DEBUG_DYNREC,"fchs");
	g_pFPpCode->FCHSp();

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}



void SR_FP_MOV(u32 reg_fd, u32 reg_fs, FPREG_TYPE nType)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, nType);
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// move (just assign st(0) to FD)
	DPF(DEBUG_DYNREC,"'fmov'");

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, nType);
}


///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////


void SR_FP_MTC1(u32 reg_fd)
{

	// If the value is currently on the stack, discard (even if dirty)
	if (g_FPReg[reg_fd].dwStackPos != u32(~0))
		SR_FP_DiscardReg(reg_fd);

	// TODO- copy value to FP (and load? - it's not worth doing this if the value
	//       is not used in subsequent instructions)
	DPF( DEBUG_DYNREC, DSPrintf( "mtc1 REG -> FP%02d", reg_fd ) );
	
}
void SR_FP_MFC1(u32 reg_fs)
{

	// If the value is currently on the stack, write it out
	// Usually it will not be used again (because the calculation is complete

	if (g_FPReg[reg_fs].dwStackPos != u32(~0) && g_FPReg[reg_fs].bDirty)
		SR_FP_SaveReg(reg_fs);
	

	DPF( DEBUG_DYNREC, DSPrintf( "mfc1 REG <- FP%02d", reg_fs ) );
}
void SR_FP_LWC1(u32 reg_fd)
{
	// If the value is currently on the stack, discard (even if dirty)
	if (g_FPReg[reg_fd].dwStackPos != u32(~0))
		SR_FP_DiscardReg(reg_fd);

	// TODO- copy value to FP (and load? - it's not worth doing this if the value
	//       is not used in subsequent instructions)
	DPF( DEBUG_DYNREC, DSPrintf( "lwc1 MEM -> FP%02d", reg_fd ) );
}

void SR_FP_SWC1(u32 reg_fs)
{
	// If the value is currently on the stack, write it out
	// Usually it will not be used again (because the calculation is complete)

	if (g_FPReg[reg_fs].dwStackPos != u32(~0) && g_FPReg[reg_fs].bDirty)
		SR_FP_SaveReg(reg_fs);
	

	DPF( DEBUG_DYNREC, DSPrintf( "swc1 MEM <- FP%02d", reg_fs ) );
}



void SR_FP_LDC1(u32 reg_fd)
{
	// If the value is currently on the stack, discard (even if dirty)
	if (g_FPReg[reg_fd].dwStackPos != u32(~0))
		SR_FP_DiscardReg(reg_fd);

	// TODO- copy value to FP (and load? - it's not worth doing this if the value
	//       is not used in subsequent instructions)
	DPF( DEBUG_DYNREC, DSPrintf( "ldc1 MEM -> FP%02d", reg_fd ) );
}

void SR_FP_SDC1(u32 reg_fs)
{
	// If the value is currently on the stack, write it out
	// Usually it will not be used again (because the calculation is complete)

	if (g_FPReg[reg_fs].dwStackPos != u32(~0) && g_FPReg[reg_fs].bDirty)
		SR_FP_SaveReg(reg_fs);
	

	DPF( DEBUG_DYNREC, DSPrintf( "sdc1 MEM <- FP%02d", reg_fs ) );
}

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////

void SR_FP_C(u32 reg_fs, u32 reg_ft, u16 nFlag, FPREG_TYPE nType)
{
	DWORD dwSS;
	SR_FP_Prepare_S_S(reg_fs, reg_ft, nType);
	
	// We need to swap the order of the test for some comparisons
	// e..g  le == !(b gt a) = !(a lt b)
	if (nFlag & FLAG_SWAP)
	{
		// Swap order
		SR_FP_CopyRegToST0(reg_fs);
		dwSS = g_FPReg[reg_ft].dwStackPos;
	}
	else
	{
		SR_FP_CopyRegToST0(reg_ft);
		dwSS = g_FPReg[reg_fs].dwStackPos;
	}

	// fcomp  - comp st(0),st(1) and pop stack
	g_pFPpCode->FCOMP(dwSS);
	SR_FP_PopRegs();		// pop st(0) - we're done with it now

	//fnstsw      ax
	g_pFPpCode->FNSTSW();

	// test		ah, 41
	g_pFPpCode->TEST_AH( u8(nFlag) );

	// Doesn't affect flags
	g_pFPpCode->MOV_REG_MEM(EAX_CODE, &g_qwCCR[1][31]);			// 6 bytes	

	// We need to swap the order of the test for some comparisons
	// e..g  le == !(b gt a) = !(a lt b)
	if (nFlag & FLAG_SWAP)
		g_pFPpCode->JE(6 + 2);
	else
		g_pFPpCode->JNE(6 + 2);

	// fX < fY - set bit
	g_pFPpCode->ORI(EAX_CODE, FPCSR_C);			// 6 bytes
	g_pFPpCode->JMP_SHORT(6);		// 2 bytes

	// fX >= fY - clear bit
	g_pFPpCode->ANDI(EAX_CODE, (~FPCSR_C));			// 6 bytes

	// Both branches end here:
	g_pFPpCode->MOV_MEM_REG(&g_qwCCR[1][31], EAX_CODE);			// 6 bytes	
}


// Convert to int, store, discard old dest
// May need to change rounding mode?
void SR_FP_S_TRUNC_W(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_SINGLE);
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// store (just assign st(0) to FD)
	DPF(DEBUG_DYNREC,"'fistp'");

	// FRNDINT?

	g_pFPpCode->FISTP_MEMp(&g_qwCPR[1][reg_fd]);
		
	SR_FP_PopRegs();		// pop st(0) - we're done with it now

}

// Convert to int, store, discard old dest
// May need to change rounding mode?
void SR_FP_S_CVT_W(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_SINGLE);
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// store (just assign st(0) to FD)
	DPF(DEBUG_DYNREC,"'fistp'");

	// FRNDINT?

	g_pFPpCode->FISTP_MEMp(&g_qwCPR[1][reg_fd]);
		
	SR_FP_PopRegs();		// pop st(0) - we're done with it now

}
void SR_FP_D_CVT_W(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_DOUBLE);
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// store (just assign st(0) to FD)
	DPF(DEBUG_DYNREC,"'fistp'");

	// FRNDINT?

	g_pFPpCode->FISTP_MEMp(&g_qwCPR[1][reg_fd]);
		
	SR_FP_PopRegs();		// pop st(0) - we're done with it now

}


// Convert single to double
// Simply load, then change nType field so it will be saved out as a double
// Simple!
void SR_FP_S_CVT_D(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_SINGLE);		// Set both as 32 - then save as 64
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// store (just assign st(0) to FD)
	SR_FP_AssignToST0(reg_fd, FP_DOUBLE);
}

void SR_FP_D_CVT_S(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_DOUBLE);		// Set both as 64 - then save as 32
	
	SR_FP_CopyRegToST0(reg_fs);
	
	// store (just assign st(0) to FD)
	SR_FP_AssignToST0(reg_fd, FP_SINGLE);
}


// Convert to float, store, discard old D
void SR_FP_W_CVT_S(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_SINGLE);
	
	SR_FP_LoadWordToST0(reg_fs);

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, FP_SINGLE);
}


// Convert to float, store, discard old D
void SR_FP_W_CVT_D(u32 reg_fd, u32 reg_fs)
{
	SR_FP_Prepare_D_S(reg_fd, reg_fs, FP_DOUBLE);
	
	SR_FP_LoadWordToST0(reg_fs);

	// Assign to reg_fd
	SR_FP_AssignToST0(reg_fd, FP_DOUBLE);
}
