/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Check patents: U.S. Pat. No. 6,394,905
//                U.S. Pat. No. 4,799,635
//                U.S. Pat. No. 5,426,762 


/*
In the exemplary embodiment, if processor 100 writes "0x00", "0xFD", "0xFE" or "0xFF" as TxData Size,
the data is not recognized as TxData size but has a special function as indicated below. They become
effective when processor 100 sets format bit (0x1FC007FC b0) by using Wr64B or Wr4B. 

"0x00"=Channel Skip 

If 0x00 is written as TxData Size, respective JoyChannel transaction is not executed. 

"0xFD"=Channel Reset 

If 0xFD is written as TxData Size, PIF outputs reset signal to respective JoyChannel. 

"0xFE"=Format End 

If 0xFE is written as TxData Size, TxData/RxData assignment is end at this ")xFE". In other words,
the TxData Size or RxData Size after "0xFE" is ignored. 

"0xFF"=Dummy Data 

TxData Size's 0xFF is used as the dummy data for word aligning the data area. 

Each Channel has four flags. Two of them have information from processor 100 to JoyChannel and
others from JoyChannel to processor 100. 

Skip=Channel Skip 

If processor 100 sets this flag to "1", respective JoyChannel transaction is not executed. This
flag becomes effective without formal flag. 

Res=Channel Reset 

If 64 bit CPU set this flag to "1", PIF outputs reset signal to respective JoyChannel. This flag
becomes effective without format flag. 

NR=No Response to JoyChannel 

When each JoyChannel's peripheral device does not respond, the respective NR bit is set to "1".
This is the way to detect the number of currently connected peripheral devices. 

Err=JoyChannel Error 

When communication error has occurred between PIF and peripheral device, Err flag is set to "1". 

If the 64 bit CPU 100 wants to change JoyChannel's Tx/RxData assignment, a 32 bit format flag is
used, where a certain bit(s) specify the desired format. For example, when Wr64B or Wr4B is
issued when this flag is "1", PIF executes each JoyChannel's Tx/RxData assignment based on each
channel's Tx/Rx Size. In other words, unless this flag is set to "1" with Wr64B or Wr4B, Tx/RxData
area assignment does not change. After Tx/RxData assignment, this flag is reset to "0" automatically. 
*/

// Typical command for a channel:

// TxDataSize: 0x01
// RxDataSize: 0x03
// TxData:     0x00 : Command: GetStatus
// RxData0:        
// RxData1:    
// RxData2:    


// Stuff to handle controllers

#include "stdafx.h"

#include "PIF.h"
#include "CPU.h"
#include "Memory.h"
#include "ROM.h"

#include "DaedMathUtil.h"

#include "Debug/DBGConsole.h"
#include "Input/InputManager.h"
#include "Debug/Dump.h"		// Dump_GetSaveDirectory()

#include "OSHLE/ultra_os.h"

#ifdef _MSC_VER
#pragma warning(default : 4002) 
#endif

//#define DEBUG_PIF_RAM

#ifdef DEBUG_PIF_RAM
	#define DPF_PIF( x )		{ if ( mDebugFile ) { fprintf( mDebugFile, "%s\n", x ); } }
#else
	#define DPF_PIF( x )
#endif

//*****************************************************************************
//
//*****************************************************************************
class	IController : public CController
{
	public:
		IController();
		~IController();

		//
		// CController implementation
		//
		bool			OnRomOpen( ESaveType save_type );
		void			OnRomClose();

		void			FlushSaveGames();

		void			Process();
		
	private:
		//
		//
		//
		void			LoadEeprom();
		void			SaveEeprom();
		void			LoadMempack();
		void			SaveMempack();

		void			FormatChannels();

		bool			ProcessCommand(u32 i, u32 iError, u32 channel, u32 ucWrite, u32 ucRead);

		bool			CommandStatusEeprom(u32 i, u32 iError, u32 ucWrite, u32 ucRead);
		bool			CommandReadEeprom(u32 i, u32 iError, u32 ucWrite, u32 ucRead);
		bool			CommandWriteEeprom(u32 i, u32 iError, u32 ucWrite, u32 ucRead);
		bool			CommandReadMemPack(u32 i, u32 iError, u32 channel, u32 ucWrite, u32 ucRead);
		bool			CommandWriteMemPack(u32 i, u32 iError, u32 channel, u32 ucWrite, u32 ucRead);

		void			InitMemPack();
		u8				CalculateDataCrc(u8 * pBuf);
		void			GenerateIDCheckSum(u16 * pBuf, u16 * pPos, u16 * pNot);


		bool			IsEepromPresent() const						{ return mpEepromData != NULL; }
		u16				GetEepromContType() const					{ return mEepromContType; }

		void			SetPifByte( u32 index, u8 value )			{ mpPifRam[ index ^ 0x3 ] = value; }
		u8				GetPifByte( u32 index ) const				{ return mpPifRam[ index ^ 0x3 ]; }

		void			WriteStatusBits( u32 index, u8 value );
		void			Write8Bits( u32 index, u8 value )			{ mpPifRam[ index ^ 0x3 ] = value; }
		void			Write16Bits( u32 index, u16 value );
		void			Write16Bits_Swapped( u32 index, u16 value );

	#ifdef DEBUG_PIF_RAM
		void			DumpInput() const;
	#endif

	private:
		
		enum
		{
			CONT_GET_STATUS      = 0x00,
			CONT_READ_CONTROLLER = 0x01,
			CONT_READ_MEMPACK    = 0x02,
			CONT_WRITE_MEMPACK   = 0x03,
			CONT_READ_EEPROM     = 0x04,
			CONT_WRITE_EEPROM    = 0x05,
			CONT_RESET           = 0xff
		};

		enum
		{
			CONT_TX_SIZE_CHANSKIP   = 0x00,					// Channel Skip
			CONT_TX_SIZE_DUMMYDATA  = 0xFF,					// Dummy Data
			CONT_TX_SIZE_FORMAT_END = 0xFE,					// Format End
			CONT_TX_SIZE_CHANRESET  = 0xFD,					// Channel Reset
		};

		u8 *			mpPifRam;

#ifdef DEBUG_PIF_RAM
		u8				mpInput[ 64 ];
#endif

		struct SChannelFormat
		{
			bool		Skipped;
			u32			StatusByte;
			u32			RxDataStart;
			u32			RxDataLength;
			u32			TxDataStart;
			u32			TxDataLength;
		};

		enum EPIFChannels
		{
			PC_CONTROLLER_0 = 0,
			PC_CONTROLLER_1,
			PC_CONTROLLER_2,
			PC_CONTROLLER_3,
			PC_EEPROM,
			PC_UNKNOWN_1,
			NUM_CHANNELS,
		};

		static const u32	NUM_CONTROLLERS = 4;

		OSContPad		mContPads[ NUM_CONTROLLERS ];
		bool			mContPresent[ NUM_CONTROLLERS ];
		bool			mContMemPackPresent[ NUM_CONTROLLERS ];

		u8 *			mpEepromData;
		u16				mEepromContType;					// 0, CONT_EEPROM or CONT_EEP16K
		u32				mEepromSize;

		u8				mMemPack[ NUM_CONTROLLERS ][(0x400+1) * 32];

		bool			mEepromDirty;
		bool			mMempackUsed;

		SChannelFormat	mChannelFormat[ NUM_CHANNELS ];

	#ifdef DEBUG_PIF_RAM
		FILE *			mDebugFile;
	#endif

};


//*****************************************************************************
// Singleton creator
//*****************************************************************************
namespace daedalus
{
template<> bool	CSingleton< CController >::Create()
{
	DAEDALUS_ASSERT_Q(mpInstance == NULL);
	
	mpInstance = new IController();

	return true;
}
}

//*****************************************************************************
// Constructor
//*****************************************************************************
IController::IController() :
	mpPifRam( NULL ),
	mpEepromData( NULL ),
	mEepromSize( 0 ),
	mEepromDirty( false ),
	mMempackUsed( false )
{
#ifdef DEBUG_PIF_RAM
	mDebugFile = fopen( "controller.txt", "w" );
#endif
	for ( u32 i = 0; i < NUM_CONTROLLERS; i++ )
	{
		mContPresent[ i ] = true;
		mContMemPackPresent[ i ] = false;
	}
	mContMemPackPresent[ 0 ] = true;

	memset( mChannelFormat, 0, sizeof( mChannelFormat ) );
	for ( u32 i = 0; i < NUM_CHANNELS; i++ )
	{
		mChannelFormat[ i ].Skipped = true;
	}
}

//*****************************************************************************
// Destructor
//*****************************************************************************
IController::~IController()
{
#ifdef DEBUG_PIF_RAM
	if( mDebugFile != NULL )
	{
		fclose( mDebugFile );
	}
#endif
}

//*****************************************************************************
// Called whenever a new rom is opened
//*****************************************************************************
bool IController::OnRomOpen( ESaveType save_type )
{
	mMempackUsed = false;

	mpPifRam = (u8 *)g_pMemoryBuffers[MEM_PIF_RAM] + 0x7C0;
	
	if ( mpEepromData )
	{
		delete [] mpEepromData;
		mpEepromData = NULL;
	}

	if ( save_type == SAVE_TYPE_EEP4K )
	{
		mEepromSize = 4096/8;					// 4k bits
		mpEepromData = new u8[mEepromSize];
		mEepromContType = CONT_EEPROM;

		DBGConsole_Msg( 0, DSPrintf( "Initialising EEPROM to [M%d] bytes", mEepromSize ) );
		LoadEeprom();
	}
	else if ( save_type == SAVE_TYPE_EEP16K )
	{
		mEepromSize = 16384/8;					// 16 kbits
		mpEepromData = new u8[mEepromSize];
		mEepromContType = CONT_EEP16K;

		DBGConsole_Msg( 0, DSPrintf( "Initialising EEPROM to [M%d] bytes", mEepromSize ) );
		LoadEeprom();
	}
	else
	{
		mEepromSize = 0;
		mpEepromData = NULL;
		mEepromContType = 0;

		DBGConsole_Msg( 0, "No EEPROM available" );
	}

	InitMemPack();

	return true;
}

//*****************************************************************************
// Called as a rom is closed
//*****************************************************************************
void IController::OnRomClose()
{
	if ( IsEepromPresent() && mEepromDirty )
	{
		// Write Eeprom to rom directory
		SaveEeprom();
	}

	if (mMempackUsed)
	{
		// Write Eeprom to rom directory
		SaveMempack();
		mMempackUsed = false;
	}

	if ( mpEepromData )
	{
		delete [] mpEepromData;
		mpEepromData = NULL;
	}
}

//*****************************************************************************
//
//*****************************************************************************
void	IController::FlushSaveGames()
{
	if ( IsEepromPresent() && mEepromDirty )
	{
		// Write Eeprom to rom directory
		SaveEeprom();
	}

	if (mMempackUsed)
	{
		// Write Eeprom to rom directory
		SaveMempack();
	}
}

//*****************************************************************************
//
//*****************************************************************************
void IController::FormatChannels()
{
#ifdef DEBUG_PIF_RAM
	DPF_PIF("");
	DPF_PIF("");
	DPF_PIF("*********************************************");
	DPF_PIF("**                                         **");
	DPF_PIF("Formatting:");

	for ( u32 x = 0; x < 64; x+=8 )
	{
		DPF_PIF( DSPrintf( "0x%02x%02x%02x%02x : 0x%02x%02x%02x%02x",
			mpPifRam[(x + 0) ^ 0x3],  mpPifRam[(x + 1) ^ 0x3],  mpPifRam[(x + 2) ^ 0x3],  mpPifRam[(x + 3) ^ 0x3],
			mpPifRam[(x + 4) ^ 0x3],  mpPifRam[(x + 5) ^ 0x3],  mpPifRam[(x + 6) ^ 0x3],  mpPifRam[(x + 7) ^ 0x3] ) );
	}
	DPF_PIF("");
	DPF_PIF("");

#endif

	// Ignore the status - just use the previous values
	u32		i( 0 );
	u32		channel( 0 );
	bool	finished( false );

	while ( i < 64 && channel < NUM_CHANNELS && !finished )
	{
		u8 tx_data_size;
		u8 rx_data_size;
		tx_data_size = GetPifByte( i );

		switch ( tx_data_size )
		{
		case CONT_TX_SIZE_CHANSKIP:
			DPF_PIF( DSPrintf( "Code 0x%02x - ChanSkip(%d)", tx_data_size, channel ) );
			mChannelFormat[ channel ].Skipped = true;
			channel++;
			i++;
			break;

		case CONT_TX_SIZE_DUMMYDATA:
			DPF_PIF( DSPrintf( "Code 0x%02x - DummyData", tx_data_size ) );
			i++;
			break;

		case CONT_TX_SIZE_FORMAT_END:
			DPF_PIF( DSPrintf( "Code 0x%02x - FormatEnd", tx_data_size ) );
			finished = true;
			break;
			
		case CONT_TX_SIZE_CHANRESET:
			DPF_PIF( DSPrintf( "Code 0x%02x - ChanReset (Ignoring)", tx_data_size ) );
			mChannelFormat[ channel ].Skipped = true;
			channel++;
			i++;
			break;

		default:
			rx_data_size = GetPifByte( i + 1 );

			// Set up error pointer and skip read/write bytes of input
			u32 iError = i + 1;
			i += 2;

			// Mask off high 2 bits. In rx this is used for error status. Not sure about tx, but we can only reference 1<6 bits anyway.
			tx_data_size &= 0x3f;
			rx_data_size &= 0x3f;

			mChannelFormat[ channel ].Skipped = false;
			mChannelFormat[ channel ].StatusByte = iError;
			mChannelFormat[ channel ].TxDataStart = i;
			mChannelFormat[ channel ].TxDataLength = tx_data_size;
			mChannelFormat[ channel ].RxDataStart = i + tx_data_size;
			mChannelFormat[ channel ].RxDataLength = rx_data_size;

			// Did skip 1 byte for write/read mempack here. Think this was wrong
			//if( GetPifByte( i ) == CONT_READ_MEMPACK || GetPifByte( i ) == CONT_WRITE_MEMPACK )
			//{
			//	i++;
			//}

			DPF_PIF( DSPrintf( "Channel %d, TxSize %d (@%d), RxSize %d (@%d)", channel, tx_data_size, i, rx_data_size, i+tx_data_size ) );

			i += tx_data_size + rx_data_size;
			channel++;
			break;
		}
	}

	// Set the remaining channels to skipped
	while( channel < NUM_CHANNELS )
	{
		mChannelFormat[ channel ].Skipped = true;
		channel++;
	}

#ifdef DEBUG_PIF_RAM

	DPF_PIF("Results:");

	for( u32 channel = 0; channel < NUM_CHANNELS; ++channel )
	{
		const SChannelFormat & format( mChannelFormat[ channel ] );
		if( format.Skipped )
		{
			DPF_PIF( DSPrintf( "%02d: Skipped", channel ) );
		}
		else
		{
			DPF_PIF( DSPrintf( "%02d: StatusIdx: %02d, TxSize %02d (@%02d), RxSize %02d (@%02d)",
								channel,
								format.StatusByte, 
								format.TxDataLength, format.TxDataStart,
								format.RxDataLength, format.RxDataStart ) );
		}
	}

	DPF_PIF("");
	DPF_PIF("");
	DPF_PIF("**                                         **");
	DPF_PIF("*********************************************");

#endif
}

//*****************************************************************************
//
//*****************************************************************************
void IController::Process()
{
#ifdef DEBUG_PIF_RAM
	memcpy(mpInput, mpPifRam, 64);
	DPF_PIF("");
	DPF_PIF("");
	DPF_PIF("*********************************************");
	DPF_PIF("**                                         **");
#endif
	
	// Clear to indicate success - we might set this again in the handler code
	if( GetPifByte( 63 ) == 1 )
	{
		FormatChannels();
	}
	SetPifByte( 63, 0 );

	CInputManager::Get()->GetState( mContPads );

	// Ignore the status - just use the previous values

	for( u32 channel = 0; channel < NUM_CHANNELS; ++channel )
	{
		const SChannelFormat & format( mChannelFormat[ channel ] );

		if( format.Skipped )
		{
			DPF_PIF( DSPrintf( "Skipping channel %d", channel ) );
			continue;
		}

		// XXXX Check tx/rx status flags here??

		if ( !ProcessCommand( format.TxDataStart, format.StatusByte, channel, format.TxDataLength, format.RxDataLength ) )
		{
			break;
		}
	}

#ifdef DEBUG_PIF_RAM
	DPF_PIF("Before | After:");

	for ( u32 x = 0; x < 64; x+=8 )
	{
		DPF_PIF( DSPrintf( "0x%02x%02x%02x%02x : 0x%02x%02x%02x%02x  |  0x%02x%02x%02x%02x : 0x%02x%02x%02x%02x",
			mpInput[(x + 0) ^ 0x3],  mpInput[(x + 1) ^ 0x3],  mpInput[(x + 2) ^ 0x3],  mpInput[(x + 3) ^ 0x3],
			mpInput[(x + 4) ^ 0x3],  mpInput[(x + 5) ^ 0x3],  mpInput[(x + 6) ^ 0x3],  mpInput[(x + 7) ^ 0x3],
			mpPifRam[(x + 0) ^ 0x3], mpPifRam[(x + 1) ^ 0x3], mpPifRam[(x + 2) ^ 0x3], mpPifRam[(x + 3) ^ 0x3],
			mpPifRam[(x + 4) ^ 0x3], mpPifRam[(x + 5) ^ 0x3], mpPifRam[(x + 6) ^ 0x3], mpPifRam[(x + 7) ^ 0x3] ) );
	}
	DPF_PIF("");
	DPF_PIF("");
	DPF_PIF("**                                         **");
	DPF_PIF("*********************************************");
#endif
}

#ifdef DEBUG_PIF_RAM
//*****************************************************************************
// Dump the PIF input
//*****************************************************************************
void IController::DumpInput() const
{
	DBGConsole_Msg( 0, "PIF:" );
	for ( u32 x = 0; x < 64; x+=8 )
	{
		DBGConsole_Msg( 0, DSPrintf( "0x%02x%02x%02x%02x : 0x%02x%02x%02x%02x",
			mpInput[(x + 0) ^ 0x3],  mpInput[(x + 1) ^ 0x3],  mpInput[(x + 2) ^ 0x3],  mpInput[(x + 3) ^ 0x3],
			mpInput[(x + 4) ^ 0x3],  mpInput[(x + 5) ^ 0x3],  mpInput[(x + 6) ^ 0x3],  mpInput[(x + 7) ^ 0x3] ) );
	}
}
#endif

//*****************************************************************************
//
//*****************************************************************************
void IController::LoadEeprom()
{
	u32 i, j;
	u32 nToDo;
	char szEepromFileName[MAX_PATH+1];
	FILE * fh;
	u8 b[2048];

	DAEDALUS_ASSERT( IsEepromPresent(), "Why are we loading EEPROM data when none is present?" );

	Dump_GetSaveDirectory(szEepromFileName, g_ROM.szFileName, ".sav");

	DBGConsole_Msg( 0, DSPrintf( "Loading eeprom from [C%s]", szEepromFileName ) );

	//
	// Initialise to zeros - this is incase the eeprom file is not found
	//
	memset(mpEepromData, 0, mEepromSize);

	fh = fopen(szEepromFileName, "rb");
	if ( fh )
	{
		// Read in chunks of 2KB
		for (i = 0; i < mEepromSize; )
		{
			nToDo = daedalus::Min((u32)(mEepromSize - i), (u32)2048);

			fread(b, nToDo, 1, fh);

			for (j = 0; j < nToDo; j++)
			{
				mpEepromData[i+j] = b[j^0x3];
			}

			i += nToDo;
		}

		fclose(fh);
	}
	else
	{
		DBGConsole_Msg( 0, "[RUnable to open eeprom %s", szEepromFileName );
	}

	mEepromDirty = false;
}

//*****************************************************************************
//
//*****************************************************************************
void IController::SaveEeprom()
{
	u32 i, j;
	u32 nToDo;
	char szEepromFileName[MAX_PATH+1];
	FILE * fh;
	u8 b[2048];

	DAEDALUS_ASSERT( IsEepromPresent(), "Why are we loading EEPROM data when none is present?" );

	Dump_GetSaveDirectory(szEepromFileName, g_ROM.szFileName, ".sav");

	DBGConsole_Msg( 0, DSPrintf( "Saving eeprom to [C%s]", szEepromFileName ) );

	fh = fopen(szEepromFileName, "wb");
	if ( fh )
	{
		for (i = 0; i < mEepromSize; )
		{
			nToDo = daedalus::Min((u32)(mEepromSize - i), (u32)2048);

			for(j = 0; j < nToDo; j++)
			{
				b[j] = mpEepromData[(i+j)^0x3];
			}

			fwrite(b, nToDo, 1, fh);

			i += nToDo;
		}

		fclose(fh);
	}

	mEepromDirty = false;
}

//*****************************************************************************
//
//*****************************************************************************
void IController::SaveMempack()
{
	char szMempackFileName[MAX_PATH+1];
	FILE * fp;
	
	Dump_GetSaveDirectory(szMempackFileName, g_ROM.szFileName, ".mpk");

	DBGConsole_Msg( 0, DSPrintf( "Saving mempack to [C%s]", szMempackFileName ) );

	fp = fopen(szMempackFileName, "wb");
	if (fp != NULL)
	{
		fwrite(&mMemPack[0][0], (0x400) * 32, 1, fp);
		fclose(fp);
	}
}

//*****************************************************************************
//
//*****************************************************************************
void IController::LoadMempack()
{
	char szMempackFileName[MAX_PATH+1];
	FILE * fp;

	Dump_GetSaveDirectory(szMempackFileName, g_ROM.szFileName, ".mpk");

	DBGConsole_Msg(0, DSPrintf( "Loading mempack from [C%s]", szMempackFileName ) );

	fp = fopen(szMempackFileName, "rb");
	if (fp != NULL)
	{
		fread(&mMemPack[0][0], (0x400) * 32, 1, fp);
		fclose(fp);
	}
	else
	{
		DBGConsole_Msg( 0, "[C%s] not found\n", szMempackFileName );
	}
}

//*****************************************************************************
// i points to start of command
//*****************************************************************************
bool	IController::ProcessCommand(u32 i, u32 iError, u32 channel, u32 ucWrite, u32 ucRead)
{
	bool	success( true );
	u8		command( GetPifByte( i + 0 ) );			// Read command

	i++;
	ucWrite--;

	WriteStatusBits( iError, 0 );					// Clear error flags
	

	// From the patent, it says that if a controller is plugged in and the memory card is removed, the CONT_CARD_PULL flag will be set.
	// Cleared if CONT_RESET or CONT_GET_STATUS is issued.
	// Might need to set this if mContMemPackPresent is false?

	// i Currently points to data to write to
	switch ( command )
	{
	case CONT_RESET:

		if (channel < NUM_CONTROLLERS)
		{
			DPF_PIF("Controller: Command is RESET");

			if (mContPresent[channel])
			{
				DAEDALUS_ASSERT( ucRead <= 3, "Reading too many bytes for RESET command" );

				if ( ucRead < 3 )
				{
					DAEDALUS_ASSERTMSG( "Overrun on RESET" );
					WriteStatusBits( iError, CONT_OVERRUN_ERROR );				// Transfer error...
				}

				Write16Bits( i, CONT_TYPE_NORMAL );
				Write8Bits( i+2, mContMemPackPresent[channel] ? CONT_CARD_ON : 0 );	// Is the mempack plugged in?
			}
			else
			{
				WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );				// Not connected
			}
		}
		else if (channel == PC_EEPROM)
		{
			DAEDALUS_ASSERTMSG( "Executing Reset on EEPROM - is this ok?" );
			success = CommandStatusEeprom( i, iError, ucWrite, ucRead );
		}
		else
		{
			//DPF_PIF(DSPrintf("Controller: UnkStatus, Channel = %d", channel));
			DAEDALUS_ASSERTMSG( "Trying to reset for invalid controller channel!" );
		}	
		break;

	case CONT_GET_STATUS:
		if (channel < NUM_CONTROLLERS)
		{
			DPF_PIF("Controller: Executing GET_STATUS");

			if (mContPresent[channel])
			{
				DAEDALUS_ASSERT( ucRead <= 3, "Reading too many bytes for STATUS command" );

				if (ucRead < 3)
				{
					DAEDALUS_ASSERTMSG( "Overrun on GET_STATUS" );
					WriteStatusBits( iError, CONT_OVERRUN_ERROR );				// Transfer error...
				}

				Write16Bits( i, CONT_TYPE_NORMAL );
				Write8Bits( i+2, mContMemPackPresent[channel] ? CONT_CARD_ON : 0 );	// Is the mempack plugged in?
			}
			else
			{
				WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );				// Not connected
			}
		}
		else if (channel == PC_EEPROM)
		{
			// This is eeprom status?
			DPF_PIF("Controller: Executing GET_EEPROM_STATUS?");

			success = CommandStatusEeprom( i, iError, ucWrite, ucRead );
		}
		else
		{
			//DPF_PIF("Controller: UnkStatus, Channel = %d", channel);
			DAEDALUS_ASSERTMSG( "Trying to get status for invalid controller channel!" );
		}	
		break;


	case CONT_READ_CONTROLLER:		// Controller
		if ( channel < NUM_CONTROLLERS )
		{
			DPF_PIF("Controller: Executing READ_CONTROLLER");
			// This is controller status
			if (mContPresent[channel])
			{
				DAEDALUS_ASSERT( ucRead <= 4, "Reading too many bytes for READ_CONT command" );

				if (ucRead < 4)
				{
					DAEDALUS_ASSERTMSG( "Overrun on READ_CONT" );
					WriteStatusBits( iError, CONT_OVERRUN_ERROR );
				}

				// Hack - we need to only write the number of bytes asked for!
				Write16Bits_Swapped( i, mContPads[channel].button );
				Write8Bits( i+2, mContPads[channel].stick_x );
				Write8Bits( i+3, mContPads[channel].stick_y );	
			}
			else
			{
				// Not connected			
				WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );
			}
		}
		else
		{
			DAEDALUS_ASSERTMSG( "Trying to read from invalid controller channel!" );
		}
		break;

	case CONT_READ_MEMPACK:
		if ( channel < NUM_CONTROLLERS )
		{
			DPF_PIF("Controller: Command is READ_MEMPACK");
			if (mContPresent[channel])
			{
				DPF_PIF( "Mempack present" );
				success = CommandReadMemPack(i, iError, channel, ucWrite, ucRead);
			}
			else
			{
				DPF_PIF( "Mempack not present" );
				WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );
			}
		}
		else
		{
			DAEDALUS_ASSERTMSG( "Trying to read mempack from invalid controller channel!" );
		}
		break;

	case CONT_WRITE_MEMPACK:
		if ( channel < NUM_CONTROLLERS )
		{
			DPF_PIF("Controller: Command is WRITE_MEMPACK");
			if (mContPresent[channel])
			{
				success = CommandWriteMemPack(i, iError, channel, ucWrite, ucRead);
			}
			else
			{
				WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );
			}
		}
		else
		{
			DAEDALUS_ASSERTMSG( "Trying to write mempack to invalid controller channel!" );
		}
		break;

	case CONT_READ_EEPROM:		return CommandReadEeprom(i, iError, ucWrite, ucRead);
	case CONT_WRITE_EEPROM:		return CommandWriteEeprom(i, iError, ucWrite, ucRead);

	default:
		DAEDALUS_ASSERTMSG( DSPrintf( "Unknown controller command: %02x", command ) );
		DPF_PIF( DSPrintf("Unknown controller command: %02x", command) );
		break;
	}

	return success;
}



//*****************************************************************************
// i points to start of command
//*****************************************************************************
bool	IController::CommandStatusEeprom(u32 i, u32 iError, u32 ucWrite, u32 ucRead)
{

	DPF_PIF("Controller: GetStatusEEPROM");

	if (ucWrite != 0 || ucRead > 3)
	{
		WriteStatusBits( iError, CONT_OVERRUN_ERROR );
		DAEDALUS_ASSERTMSG( "GetEepromStatus Overflow" );
		return false;
	}

	DAEDALUS_ASSERT( ucRead == 3, "Why is ucRead not 3 for an Eeprom read?" );

	if ( IsEepromPresent() )
	{
		Write16Bits(i, GetEepromContType() );
		Write8Bits(i+2, 0x00);

		i += 3;
		ucRead -= 3;
	}
	else
	{
		DAEDALUS_ASSERTMSG( "ROM is accessing the EEPROM, but none is present" );
		WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );
	}

	DAEDALUS_ASSERT( ucWrite == 0 && ucRead == 0, "GetEepromStatus Read / Write bytes remaining" );
	return true;

}

//*****************************************************************************
//
//*****************************************************************************
bool	IController::CommandReadEeprom(u32 i, u32 iError, u32 ucWrite, u32 ucRead)
{
	u8 block;

	DPF_PIF("Controller: ReadEEPROM");

	DAEDALUS_ASSERT( ucWrite+1 == 2, "Why is tx_data_size not 2 for READ_EEP?" );
	DAEDALUS_ASSERT( ucRead == 8, "Why is rx_data_size not 1 for WRITE_EEP?" );

	if (ucWrite != 1 || ucRead > 8)
	{
		WriteStatusBits( iError, CONT_OVERRUN_ERROR );
		DAEDALUS_ASSERTMSG( "ReadEeprom Overflow" );
		return false;
	}


	// Read the block 
	block = GetPifByte( i + 0 );
	i++;
	ucWrite--;

	if ( ucRead < 8 )
	{
		DAEDALUS_ASSERTMSG( "Overrun on READ_EEPROM" );
	}

	if ( IsEepromPresent() )
	{
		// TODO limit block to mEepromSize / 8
		if (block*8+ucRead > mEepromSize)
		{
			DAEDALUS_ASSERTMSG( "Reading outside of EEPROM bounds" );
		}

		u32 j = 0;
		while (ucRead)
		{
			SetPifByte( i, mpEepromData[(block*8 + j) ^ 0x3] );

			i++;
			j++;
			ucRead--;
		}
	}
	else
	{
		WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );
	}

	DAEDALUS_ASSERT( ucWrite == 0 && ucRead == 0, "ReadEeprom Read / Write bytes remaining" );
	return true;
}



//*****************************************************************************
//
//*****************************************************************************
bool	IController::CommandWriteEeprom(u32 i, u32 iError, u32 ucWrite, u32 ucRead)
{
	u8 block;

	DPF_PIF("Controller: WriteEEPROM");

	DAEDALUS_ASSERT( ucWrite+1 == 10, "Why is tx_data_size not 10 for WRITE_EEP?" );
	DAEDALUS_ASSERT( ucRead == 1, "Why is rx_data_size not 1 for WRITE_EEP?" );

	// 9 bytes of input remaining - 8 bytes data + block
	if (ucWrite != 9 /*|| ucRead != 1*/)
	{
		WriteStatusBits( iError, CONT_OVERRUN_ERROR );
		DAEDALUS_ASSERTMSG( "WriteEeprom Overflow" );
		return false;
	}

	// Read the block 
	block = GetPifByte( i + 0 );
	i++;
	ucWrite--;

	if ( IsEepromPresent() )
	{
		mEepromDirty = true;

		// TODO limit block to mEepromSize / 8
		if (block*8+ucWrite > mEepromSize)
		{
			DAEDALUS_ASSERTMSG( "Writing outside of EEPROM bounds" );
		}

		u32 j = 0;
		while (ucWrite)
		{
			mpEepromData[(block*8 + j) ^ 0x3] = GetPifByte( i );

			i++;
			j++;
			ucWrite--;
		}

#ifdef DAEDALUS_PSP
		// Immediately flush the EEPROM data out after writing
		//SaveEeprom();
#endif
	}
	else
	{
		WriteStatusBits( iError, CONT_NO_RESPONSE_ERROR );
	}

	// What on earth is this for??
	ucRead--;

	DAEDALUS_ASSERT( ucWrite == 0 && ucRead == 0, "WriteEeprom Read / Write bytes remaining" );
	return true;
}


//*****************************************************************************
// Returns new position to continue reading
// i is the address of the first write info (after command itself)
//*****************************************************************************
bool	IController::CommandReadMemPack( u32 i, u32 iError, u32 channel, u32 ucWrite, u32 ucRead )
{
	DPF_PIF( DSPrintf( "CommandReadMemPack(%d, %d, %d, %d, %d)", i, iError, channel, ucWrite, ucRead ) );
	u32 j;
	u32 address_crc;
	u32 address;
	u32 dwCRC;
	u8 ucDataCRC;
	u8 * pBuf;

	if (!mMempackUsed)
	{
		LoadMempack();
		mMempackUsed = true;
	}
	
	// There must be exactly 2 bytes to write, and 33 bytes to read
	if (ucWrite != 2 || ucRead != 33)
	{
		// TRANSFER ERROR!!!!
		DBGConsole_Msg( 0, "RMemPack with bad tx/rxdata size (%d/%d)\n", ucWrite, ucRead );
		WriteStatusBits( iError, CONT_OVERRUN_ERROR );
		return false;
	}

	DPF_PIF( DSPrintf("ReadMemPack: Channel %d, i is %d", channel, i) );

	// Get address..
	address_crc = (GetPifByte( i + 0 ) << 8) | GetPifByte( i + 1 );

	address = (address_crc >> 5);
	dwCRC = (address_crc & 0x1f);
	i += 2;	
	ucWrite -= 2;


	if (address > 0x400)
	{
		// SOME OTHER ERROR!
		DAEDALUS_ASSERTMSG( DSPrintf( "ReadMemPack: Address out of range: 0x%08x", address ) );
		return false;
	}
	else if ( address == 0x400 )
	{
		pBuf = &mMemPack[channel][address * 32];
		memset( pBuf, 0, 32 );

		for (j = 0; j < 32; j++)
		{
			if (i < 64)
			{
				SetPifByte( i, pBuf[j] );
			}
			i++;
			ucRead--;
		}

		// For some reason this seems to be negated when operating on this address
		// If I negate the result here, Zelda OoT stops working :/
		ucDataCRC = CalculateDataCrc(pBuf);
	}
	else
	{
		pBuf = &mMemPack[channel][address * 32];

		DPF_PIF( DSPrintf("Controller: Reading from block 0x%04x (crc: 0x%02x)", address, dwCRC) );
		
		for (j = 0; j < 32; j++)
		{
			if (i < 64)
			{
				SetPifByte( i, pBuf[j] );
			}
			i++;
			ucRead--;
		}

		ucDataCRC = CalculateDataCrc(pBuf);
	}
	
	DPF_PIF( DSPrintf("Controller: data crc is 0x%02x", ucDataCRC) );

	// Write the crc value:
	SetPifByte( i, ucDataCRC );
	i++;
	ucRead--;
	
	DPF_PIF( DSPrintf("Returning, setting i to %d", i + 1) );

	// With wetrix, there is still a padding byte?
	DAEDALUS_ASSERT( ucWrite == 0 && ucRead == 0, "ReadMemPack / Write bytes remaining" );
	return true;
}


//*****************************************************************************
// Returns new position to continue reading
// i is the address of the first write info (after command itself)
//*****************************************************************************
bool	IController::CommandWriteMemPack(u32 i, u32 iError, u32 channel, u32 ucWrite, u32 ucRead)
{
	u32 j;
	u32 address_crc;
	u32 address;
	u32 dwCRC;
	u8 ucDataCRC;
	u8 * pBuf;

	if (!mMempackUsed)
	{
		LoadMempack();
		mMempackUsed = true;
	}
	
	// There must be exactly 32+2 bytes to read

	if (ucWrite != 34 || ucRead != 1)
	{
		DAEDALUS_ASSERTMSG( DSPrintf( "WriteMemPack with bad tx/rxdata size (%d/%d)", ucWrite, ucRead ) );
		DBGConsole_Msg( 0, "WMemPack with bad tx/rxdata size (%d/%d)\n", ucWrite, ucRead );
		WriteStatusBits( iError, CONT_OVERRUN_ERROR );
		return false;
	}

	DPF_PIF( DSPrintf("WriteMemPack: Channel %d, i is %d", channel, i) );

	// Get address..
	address_crc = (GetPifByte( i + 0 ) << 8) | GetPifByte( i + 1 );

	address = (address_crc >>5);
	dwCRC = (address_crc & 0x1f);
	i += 2;	
	ucWrite -= 2;


	if (address > 0x400/* && address != 0x600*/)
	{
		// 0x600 is mempack enable address?
		/*if (address == 0x600)
		{
			pBuf = &arrTemp[0];
		}*/
		DAEDALUS_ASSERTMSG( DSPrintf( "Attempting to write to non-existing block 0x%08x", address ) );
		return false;
	}
	else if ( address == 0x400 )
	{
		pBuf = &mMemPack[channel][address * 32];

		// Do nothing - enable rumblepak eventually
		for (j = 0; j < 32; j++)
		{
			if (i < 64)
			{
				pBuf[j] = GetPifByte( i );
			}
			i++;
			ucWrite--;
		}
		
		// For some reason this seems to be negated when operating on this address
		// If I negate the result here, Zelda OoT stops working :/
		ucDataCRC = CalculateDataCrc(pBuf);
	}
	else
	{
		pBuf = &mMemPack[channel][address * 32];

		DPF_PIF( DSPrintf("Controller: Writing block 0x%04x (crc: 0x%02x)", address, dwCRC) );

		for (j = 0; j < 32; j++)
		{
			if (i < 64)
			{
				pBuf[j] = GetPifByte( i );
			}
			i++;
			ucWrite--;
		}
		ucDataCRC = CalculateDataCrc(pBuf);
	}

	DPF_PIF( DSPrintf("Controller: data crc is 0x%02x", ucDataCRC) );

	// Write the crc value:
	SetPifByte( i, ucDataCRC );
	i++;
	ucRead--;
	
	// With wetrix, there is still a padding byte?
	DAEDALUS_ASSERT( ucWrite == 0 && ucRead == 0, "WriteMemPack / Write bytes remaining" );

#ifdef DAEDALUS_PSP
	// Immediately write out the mempack on writing
	//SaveMempack();
#endif

	return true;
}

//*****************************************************************************
//
//*****************************************************************************
u8 IController::CalculateDataCrc(u8 * pBuf)
{
	u8 c;
	u8 x,s;
	u8 i;
	s8 z;

	c = 0;
	for (i = 0; i < 33; i++)
	{
		s = pBuf[i];

		for (z = 7; z >= 0; z--)
		{		
			x = (c & 0x80) ? 0x85 : 0;

			c <<= 1;

			if (i < 32)
			{
				if (s & (1<<z))
					c |= 1;
			}

			c = c ^ x;
		}
	}

	return c;
}

//*****************************************************************************
//
//*****************************************************************************
void IController::InitMemPack()
{
	u32 channel;

	//
	// Initialisation values taken from PJ64
	//
	u8 Initilize[] =
	{ 
		0x81,0x01,0x02,0x03, 0x04,0x05,0x06,0x07, 0x08,0x09,0x0a,0x0b, 0x0C,0x0D,0x0E,0x0F,
		0x10,0x11,0x12,0x13, 0x14,0x15,0x16,0x17, 0x18,0x19,0x1A,0x1B, 0x1C,0x1D,0x1E,0x1F,
		0xFF,0xFF,0xFF,0xFF, 0x05,0x1A,0x5F,0x13, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0x01,0xFF, 0x66,0x25,0x99,0xCD,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0xFF,0xFF,0xFF,0xFF, 0x05,0x1A,0x5F,0x13, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0x01,0xFF, 0x66,0x25,0x99,0xCD,
		0xFF,0xFF,0xFF,0xFF, 0x05,0x1A,0x5F,0x13, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0x01,0xFF, 0x66,0x25,0x99,0xCD,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0xFF,0xFF,0xFF,0xFF, 0x05,0x1A,0x5F,0x13, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0xFF,0xFF, 0xFF,0xFF,0x01,0xFF, 0x66,0x25,0x99,0xCD,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00, 0x00,0x00,0x00,0x00,
		0x00,0x71,0x00,0x03, 0x00,0x03,0x00,0x03, 0x00,0x03,0x00,0x03, 0x00,0x03,0x00,0x03,
	};
	
	for ( channel = 0; channel < NUM_CONTROLLERS; channel++ )
	{
		for ( u32 count2 = 0; count2 < 0x8000; count2 += 2 )
		{
			mMemPack[channel][count2 + 0] = 0x00;
			mMemPack[channel][count2 + 1] = 0x03;
		}
		memcpy(&mMemPack[channel][0],Initilize,sizeof(Initilize));
	}

	/*
	u32 address;
	for ( channel = 0; channel < NUM_CONTROLLERS; channel++ )
	{
		for (address = 0; address < 0x0400; address++)
		{
			u8 * pBuf = &mMemPack[channel][address * 32];

			// Clear values
			memset(pBuf, 0, 32);

			// Generate checksum if necessary
			if (address == 3 || address == 4 || address == 6)
			{
				u16 wPos, wNot;
				u16 * pwBuf = (u16 *)pBuf;

				GenerateIDCheckSum(pwBuf, &wPos, &wNot);

				pwBuf[14] = (wPos >> 8) | (wPos << 8);
				pwBuf[15] = (wNot >> 8) | (wNot << 8);

				DPF_PIF(DSPrintf("Hacking ID Values: 0x%04x, 0x%04x", wPos, wNot));
			}
		}
	}
	*/

/*	for ( u32 a = 0; a < 10*32; a+= 16 )
	{
	DBGConsole_Msg( 0, DSPrintf( "%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x",
									mMemPack[0][a+0], mMemPack[0][a+1], mMemPack[0][a+2], mMemPack[0][a+3], 
									mMemPack[0][a+4], mMemPack[0][a+5], mMemPack[0][a+6], mMemPack[0][a+7], 
									mMemPack[0][a+8], mMemPack[0][a+9], mMemPack[0][a+10], mMemPack[0][a+11], 
									mMemPack[0][a+12], mMemPack[0][a+13], mMemPack[0][a+14], mMemPack[0][a+15] ) );
	}
*/

}

//*****************************************************************************
//
//*****************************************************************************
void IController::GenerateIDCheckSum(u16 * pBuf, u16 * pPos, u16 * pNot)
{
	u16 wPos = 0;
	u16 wNot = 0;

	for (u32 i = 0; i < 14; i++)
	{
		wPos += pBuf[i];
		wNot += (~pBuf[i]);
	}

	*pPos = wPos;
	*pNot = wNot;
}

//*****************************************************************************
//
//*****************************************************************************
void IController::WriteStatusBits(u32 i, u8 val)
{
	mpPifRam[(i + 0) ^ 0x3] &= 0x3F;
	mpPifRam[(i + 0) ^ 0x3] |= val;
}

//*****************************************************************************
//
//*****************************************************************************
void IController::Write16Bits(u32 i, u16 val)
{
	mpPifRam[(i + 0) ^ 0x3] = (u8)(val   );	// Lo
	mpPifRam[(i + 1) ^ 0x3] = (u8)(val>>8);	// Hi
}

//*****************************************************************************
//
//*****************************************************************************
void IController::Write16Bits_Swapped(u32 i, u16 val)
{
	mpPifRam[(i + 0) ^ 0x3] = (u8)(val>>8);	// Hi
	mpPifRam[(i + 1) ^ 0x3] = (u8)(val   );	// Lo
}


