/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef CPU_H_
#define CPU_H_

//*****************************************************************************
// Forward Declarations
//*****************************************************************************
class CGraphicsPlugin;
class CAudioPlugin;

#include "R4300_Regs.h"
#include "R4300Instruction.h"
#include "R4300OpCode.h"
#include "Memory.h"

//*****************************************************************************
//
//*****************************************************************************
enum EDelayType
{
	NO_DELAY = 0,
	EXEC_DELAY,
	DO_DELAY
};


struct DBG_BreakPoint
{
	OpCode mOriginalOp;
	BOOL  bEnabled;
	BOOL  bTemporaryDisable;		// Set to TRUE when BP is activated - this lets us type 
									// go immediately after a bp and execution will resume.
									// otherwise it would keep activating the bp
									// The patch bp r4300 instruction clears this 
									// when it is next executed
};

//
// CPU Jobs
//
#define CPU_CHECK_EXCEPTIONS				0x00000001
#define CPU_CHECK_INTERRUPTS				0x00000002
#define CPU_STOP_RUNNING					0x00000008
#define CPU_CHANGE_CORE						0x00000010

//*****************************************************************************
// External declarations
//*****************************************************************************
extern CGraphicsPlugin * gGraphicsPlugin;
extern CAudioPlugin *		g_pAiPlugin;
#ifdef DAEDALUS_BREAKPOINTS_ENABLED
#include <vector>
extern std::vector< DBG_BreakPoint > g_BreakPoints;
#endif

// Arbitrary unique numbers for different timing related events:
enum	ECPUEventType
{
	CPU_EVENT_VBL = 1,
	CPU_EVENT_COMPARE,
};

struct CPUEvent
{
	s32						mCount;
	ECPUEventType			mEventType;
};
DAEDALUS_STATIC_ASSERT( sizeof( CPUEvent ) == 8 );


//
//	We declare various bits of the CPU state in a struct.
//	During dynarec compilation we can keep the base address of this 
//	structure cached in a spare register to avoid expensive
//	address-calculations (primarily on the PSP)
//
ALIGNED_TYPE(struct, SCPUState, CACHE_ALIGN)
{
	static const u32	MAX_CPU_EVENTS = 4;		// In practice there should only ever be 2

	SCPUState();

	register_set	_gGPR;				// 0x000 .. 0x100
	register_set _g_qwCPR0;				// 0x100 .. 0x200
	register_set _g_qwCPR1;				// 0x200 .. 0x300
	register_set _g_qwCPR2;				// 0x300 .. 0x400
	u32				CurrentPC;			// 0x400 ..			The current program counter
	u32				TargetPC;			// 0x404 ..			The PC to branch to
	u32				Delay;				// 0x408 ..			Delay state (NO_DELAY, EXEC_DELAY, DO_DELAY)
	u32				StuffToDo;			// 0x40c ..			CPU jobs (see above)

	REG64			MultLo;				// 0x410 ..
	REG64			MultHi;				// 0x418

	CPUEvent		Events[ MAX_CPU_EVENTS ];	// 0x420
	u32				NumEvents;


};

ALIGNED_EXTERN(SCPUState, gCPUState, CACHE_ALIGN);

#define gGPR (gCPUState._gGPR)
#define g_qwCPR ((register_set*)gCPUState._g_qwCPR0)


//*****************************************************************************
//
//*****************************************************************************
void	CPU_Reset( u32 new_pc );
void	CPU_Finalise();
void	CPU_Step();
void	CPU_Skip();
bool	CPU_Run();
bool	CPU_StartThread( char * p_failure_reason, u32 length );
void	CPU_StopThread();
bool	CPU_SaveState( const char * filename );
bool	CPU_LoadState( const char * filename );
void	CPU_Halt( const char * reason );
void	CPU_SelectCore();
u32		CPU_GetVideoInterruptEventCount();
void	CPU_SetVideoInterruptEventCount( u32 count );
void	CPU_DynarecEnable();
void	CPU_InvalidateICacheRange( u32 address, u32 length );
void	CPU_InvalidateICache();
void	CPU_SetCompare(u64 qwNewValue);
#ifdef DAEDALUS_BREAKPOINTS_ENABLED
void	CPU_AddBreakPoint( u32 address );						// Add a break point at address dwAddress
void	CPU_EnableBreakPoint( u32 address, bool enable );		// Enable/Disable the breakpoint as the specified address
#endif 
bool	CPU_IsRunning();
u32		CPU_GetVerticalInterruptCount();

void CPU_SkipToNextEvent();

inline bool CPU_IsJobSet( u32 job )
{
	return ( gCPUState.StuffToDo & job ) != 0;
}

void Dynarec_ClearedCPUStuffToDo();
void Dynarec_SetCPUStuffToDo();

inline void CPU_AddJob( u32 job )
{
	gCPUState.StuffToDo |= job;
	if(gCPUState.StuffToDo)
	{
		Dynarec_SetCPUStuffToDo();
	}
}

inline void CPU_ClearJob( u32 job )
{
	gCPUState.StuffToDo &= ~job;
	if(!gCPUState.StuffToDo)
	{
		Dynarec_ClearedCPUStuffToDo();
	}
}

inline void CPU_SetPC( u32 pc )		{ gCPUState.CurrentPC = pc; }
inline void INCREMENT_PC()			{ gCPUState.CurrentPC += 4; }
inline void DECREMENT_PC()			{ gCPUState.CurrentPC -= 4; }

enum ECPUBranchType
{
	CPU_BRANCH_DIRECT = 0,		// i.e. jump to a fixed address
	CPU_BRANCH_INDIRECT,		// i.e. jump to the contents of a register
};

void	CPU_TakeBranch( u32 new_pc, ECPUBranchType branch_type );
void	CPU_ExecuteOpRaw( u32 count, u32 address, OpCode op_code, CPU_Instruction p_instruction, bool * p_branch_taken );

// Needs to be callable from assembly
extern "C"
{
	void	CPU_UpdateCounter( u32 ops_executed ); 
	void	CPU_UpdateCounterNoInterrupt( u32 ops_exexuted );
}

#ifdef DAEDALUS_ENABLE_DYNAREC
	#ifdef DAEDALUS_DEBUG_DYNAREC
	void			CPU_DumpFragmentCache();
	#endif
	void			CPU_ResetFragmentCache();
#endif

#endif // CPU_H_
