/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef CSTRING_H__
#define CSTRING_H__

#if defined(DAEDALUS_PSP)
#define _strcmpi stricmp
#elif defined(DAEDALUS_OSX)
#define _strcmpi strcasecmp
#endif

namespace daedalus {

class CConstString
{
	public:
		CConstString()
		:	mpString( "" )
		{
		}

		CConstString( const char * string )
		:	mpString( string )
		{
		}

		operator const char * () const
		{
			return mpString;
		}

		CConstString & operator=( const char * string )
		{
			mpString = string;
			return *this;
		}

		bool operator==( const char * string )
		{
			return Equals( string );
		}

		//
		// Case sensitive compare
		//
		bool Equals( const char * string ) const
		{
			if ( mpString == string )
			{
				return true;
			}
			else if ( string == NULL )
			{
				return strlen( mpString ) == 0;
			}
			else if ( mpString == NULL )
			{
				return strlen( string ) == 0;
			}
			else
			{
				return strcmp( mpString, string ) == 0;
			}
		}

		//
		// Case insensitive compare
		//
		bool IEquals( const char * string ) const
		{
			if ( mpString == string )
			{
				return true;
			}
			else if ( string == NULL )
			{
				return strlen( mpString ) == 0;
			}
			else if ( mpString == NULL )
			{
				return strlen( string ) == 0;
			}
			else
			{
				return _strcmpi( mpString, string ) == 0;
			}
		}

		u32 Length() const
		{
			if ( mpString == NULL )
			{
				return 0;
			}
			else
			{
				return strlen( mpString );
			}
		}

		bool IsNull() const
		{
			return mpString == NULL;
		}

		bool IsEmpty() const
		{
			return Length() == 0;
		}


	private:
		const char *		mpString;
};


template< u32 MAX_LENGTH > class CStaticString
{
	public:
		CStaticString()
		{
			strcpy( mpString, "" );
		}

		CStaticString( CConstString string )
		{
			Copy( string );
		}

		CStaticString & operator=( CConstString string )
		{
			Copy( string );
			return *this;
		}

		operator const char * () const
		{
			return mpString;
		}

		bool operator==( CConstString string )
		{
			return Equals( string );
		}

		CStaticString & operator+=( CConstString string )
		{
			Append( string );
			return *this;
		}

		//
		// Case sensitive compare
		//
		bool Equals( CConstString string ) const
		{
			if ( string == NULL )
			{
				return strlen( mpString ) == 0;
			}
			else
			{
				return strcmp( mpString, string ) == 0;
			}
		}

		//
		// Case insensitive compare
		//
		bool IEquals( const char * string ) const
		{
			if ( string == NULL )
			{
				return strlen( mpString ) == 0;
			}
			else
			{
				return _strcmpi( mpString, string ) == 0;
			}
		}


		u32 Length() const
		{
			return strlen( mpString );
		}

		u32 MaxLength() const
		{
			return MAX_LENGTH;
		}

		bool IsEmpty() const
		{
			return Length() == 0;
		}

		//
		// Access for functions which need to write to our buffer. Should try to avoid these!
		//
		char * GetUnsafePtr()
		{
			return mpString;
		}


	private:

		void Copy( CConstString string )
		{
			DAEDALUS_ASSERT( string.Length() <= MAX_LENGTH, "String is too long for copy, truncation will occur" );
			strncpy( mpString, string, MAX_LENGTH );
			mpString[ MAX_LENGTH ] = '\0';
		}

		void Append( CConstString string )
		{
			DAEDALUS_ASSERT( Length() + string.Length() < MAX_LENGTH, "String is too long append, truncation will occur" );
			strncat( mpString, string, MAX_LENGTH );
			mpString[ MAX_LENGTH ] = '\0';
		}

	private:
		char			mpString[ MAX_LENGTH + 1 ];
};

class CString
{
	public:
		CString()
		:	mpString( NULL )
		,	mMaxLength( 0 )
		{
		}

		CString( CConstString string )
		{
			mMaxLength = string.Length();
			mpString = new char[ mMaxLength + 1 ];
			strcpy( mpString, string );
		}

		CString( const CString & string )
		{
			mMaxLength = string.MaxLength();
			mpString = new char[ mMaxLength + 1 ];
			strcpy( mpString, string );
		}


		CString & operator=( CConstString string )
		{
			Copy( string );
			return *this;
		}

		CString & operator+=( CConstString string )
		{
			Append( string );
			return *this;
		}


		CString & operator=( const CString & string )
		{
			// Check for a = a
			if ( this != &string )
			{
				Copy( string );
			}
			return *this;
		}

		CString & operator+=( const CString & string )
		{
			// Check for a += a
			DAEDALUS_ASSERT( this != &string, "Appending self - unhandled" );

			Append( string );
			return *this;
		}

		CString operator+( CConstString string ) const
		{
			CString	ret( *this );
			ret.Append( string );
			return ret;
		}


		CString operator+( const CString & string ) const
		{
			CString	ret( *this );
			ret.Append( string );
			return ret;
		}

		operator const char * () const
		{
			return mpString;
		}

		bool operator==( CConstString string )
		{
			return Equals( string );
		}

		//
		// Case sensitive compare
		//
		bool Equals( CConstString string ) const
		{
			if ( mpString == string )
			{
				return true;
			}
			else if ( string == NULL )
			{
				return strlen( mpString ) == 0;
			}
			else if ( mpString == NULL )
			{
				return strlen( string ) == 0;
			}
			else
			{
				return strcmp( mpString, string ) == 0;
			}
		}

		//
		// Case insensitive compare
		//
		bool IEquals( CConstString string ) const
		{
			if ( mpString == string )
			{
				return true;
			}
			else if ( string == NULL )
			{
				return strlen( mpString ) == 0;
			}
			else if ( mpString == NULL )
			{
				return strlen( string ) == 0;
			}
			else
			{
				return _strcmpi( mpString, string ) == 0;
			}
		}

		u32 Length() const
		{
			if ( mpString == NULL )
			{
				return 0;
			}
			else
			{
				return strlen( mpString );
			}
		}

		bool IsNull() const
		{
			return mpString == NULL;
		}

		bool IsEmpty() const
		{
			return Length() == 0;
		}

		u32 MaxLength() const
		{
			return mMaxLength;
		}

		//
		// Access for functions which need to write to our buffer. Should try to avoid these!
		//
		char * GetUnsafePtr()
		{
			return mpString;
		}

	private:

		void Size( u32 length )
		{
			delete [] mpString;
			mMaxLength = length;
			mpString = new char[ length + 1 ];
		}

		void Resize( u32 length )
		{
			DAEDALUS_ASSERT( length > Length(), "Resize should always increase buffer length" );

			char * p_new = new char[ length + 1 ];

			if ( mpString == NULL )
			{
				strcpy( p_new, "" );
			}
			else
			{
				strcpy( p_new, mpString );
				delete [] mpString;
			}

			mMaxLength = length;
			mpString = p_new;
		}

		void Copy( const char * string )
		{
			if ( string == NULL )
			{
				if ( mMaxLength == 0 )
				{
					Size( 0 );
				}

				strcpy( mpString, "" );
			}
			else
			{
				u32 length( strlen( string ) );

				if ( length > mMaxLength )
				{
					Size( length );
				}

				strcpy( mpString, string );
			}
		}

		void Append( const char * string )
		{
			if ( string == NULL )
			{
				// Nothing to do
			}
			else
			{
				u32 length( Length() + strlen( string ) );

				if ( length > mMaxLength || mMaxLength == 0 )
				{
					Resize( length );
				}

				strcat( mpString, string );
			}

		}


	private:
		char *			mpString;
		u32				mMaxLength;
};

};		// namespace daedalus

#endif //#ifndef CSTRING_H__
