/*
Copyright (C) 2001 CyRUS64 (http://www.boob.co.uk)
Copyright (C) 2006,2007 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "Preferences.h"
#include "IniFile.h"
#include "FramerateLimiter.h"

#include "Core/ROM.h"
#include "Interface/RomDB.h"

#include "DaedIO.h"

#ifdef DAEDALUS_PSP
#include "PSPGraphics/DaedalusGraphics.h"
#include "Input/InputManager.h"
#endif

#include <string>
#include <set>
#include <map>

//*****************************************************************************
//
//*****************************************************************************
SGlobalPreferences	gGlobalPreferences;

//*****************************************************************************
//
//*****************************************************************************
class IPreferences : public CPreferences
{
	public:
		IPreferences();
		virtual ~IPreferences();

		//
		// CPreferences implementation
		//
		bool						OpenPreferencesFile( const char * filename );
		void						Commit();

		bool						GetRomPreferences( const RomID & id, SRomPreferences * preferences ) const;
		void						SetRomPreferences( const RomID & id, const SRomPreferences & preferences );

	private:
		void						OutputSectionDetails( const RomID & id, const SRomPreferences & preferences, FILE * fh );

	private:
		typedef std::map<RomID, SRomPreferences>	PreferencesMap;

		PreferencesMap			mPreferences;

		bool					mDirty;				// (STRMNNRMN - Changed since read from disk?)
		std::string				mFilename;
};



//*****************************************************************************
// Singleton creator
//*****************************************************************************
namespace daedalus
{
template<> bool	CSingleton< CPreferences >::Create()
{
	DAEDALUS_ASSERT_Q(mpInstance == NULL);
	
	mpInstance = new IPreferences();

	return true;
}
}


//*****************************************************************************
// Constructor
//*****************************************************************************
IPreferences::IPreferences()
:	mDirty( false )
{
}

//*****************************************************************************
//
//*****************************************************************************
IPreferences::~IPreferences()
{
	if ( mDirty )
	{
		Commit();
	}
}

//*****************************************************************************
//
//*****************************************************************************
static RomID	RomIDFromString( const char * str )
{
	u32 crc1, crc2, country;
	sscanf( str, "%08x%08x-%02x", &crc1, &crc2, &country );
	return RomID( crc1, crc2, (u8)country );
}

//*****************************************************************************
//
//*****************************************************************************
bool IPreferences::OpenPreferencesFile( const char * filename )
{
	//
	// Remember the filename
	//
	mFilename = filename;

	CIniFile * p_ini_file( CIniFile::Create( filename ) );
	if( p_ini_file == NULL )
	{
		return false;
	}

	const CIniFileSection *	section( p_ini_file->GetDefaultSection() );
	if( section != NULL )
	{
		const CIniFileProperty * property;

#define BOOL_SETTING( b, nm, def )	if( section->FindProperty( #nm, &property ) ) { b.nm = property->GetBooleanValue( def.nm ); }
#define INT_SETTING( b, nm, def )	if( section->FindProperty( #nm, &property ) ) {	b.nm = property->GetIntValue( def.nm ); }
#define FLOAT_SETTING( b, nm, def ) if( section->FindProperty( #nm, &property ) ) {	b.nm = property->GetFloatValue( def.nm ); }

		const SGlobalPreferences	defaults;

		BOOL_SETTING( gGlobalPreferences, DisplayFramerate, defaults );
		BOOL_SETTING( gGlobalPreferences, SoftwareClipping, defaults );
		BOOL_SETTING( gGlobalPreferences, UseVFPUTnL, defaults );
		BOOL_SETTING( gGlobalPreferences, HighlightInexactBlendModes, defaults );
		FLOAT_SETTING( gGlobalPreferences, StickMinDeadzone, defaults );
		FLOAT_SETTING( gGlobalPreferences, StickMaxDeadzone, defaults );

		if( section->FindProperty( "ViewportType", &property ) )
		{
			u32	value( property->GetIntValue( defaults.ViewportType ) );
			if( value >= 0 && value < NUM_VIEWPORT_TYPES )
			{
				gGlobalPreferences.ViewportType = EViewportType( value );
			}
		}
	}

	for( u32 section_idx = 0; section_idx < p_ini_file->GetNumSections(); ++section_idx )
	{
		const CIniFileSection * section( p_ini_file->GetSection( section_idx ) );

		RomID			id( RomIDFromString( section->GetName() ) );
		SRomPreferences	preferences;

		const CIniFileProperty * property;
		if( section->FindProperty( "PatchesEnabled", &property ) )
		{
			preferences.PatchesEnabled = property->GetBooleanValue( false );
		}
		if( section->FindProperty( "SpeedSyncEnabled", &property ) )
		{
			preferences.SpeedSyncEnabled = property->GetBooleanValue( true );
		}
		if( section->FindProperty( "DynarecEnabled", &property ) )
		{
			preferences.DynarecEnabled = property->GetBooleanValue( true );
		}
		if( section->FindProperty( "DynarecStackOptimisation", &property ) )
		{
			preferences.DynarecStackOptimisation = property->GetBooleanValue( true );
		}
	#ifdef DAEDALUS_PSP
		if( section->FindProperty( "CheckTextureHashFrequency", &property ) )
		{
			preferences.CheckTextureHashFrequency = ROM_GetTextureHashFrequencyFromFrames( atoi( property->GetValue() ) );
		}
		if( section->FindProperty( "Frameskip", &property ) )
		{
			preferences.Frameskip = ROM_GetFrameskipValueFromInt( atoi( property->GetValue() ) );
		}
		if( section->FindProperty( "AudioEnabled", &property ) )
		{
			preferences.AudioEnabled = property->GetBooleanValue( false );
		}
		if( section->FindProperty( "AudioAdaptFrequency", &property ) )
		{
			preferences.AudioAdaptFrequency = property->GetBooleanValue( false );
		}
		if( section->FindProperty( "Controller", &property ) )
		{
			preferences.ControllerIndex = CInputManager::Get()->GetConfigurationFromName( property->GetValue() );
		}
	#endif

		mPreferences[ id ] = preferences;
	}

	mDirty = false;
	
	delete p_ini_file;
	return true;
}

//*****************************************************************************
//
//*****************************************************************************
void IPreferences::OutputSectionDetails( const RomID & id, const SRomPreferences & preferences, FILE * fh )
{
	// Generate the CRC-ID for this rom:
	RomSettings		settings;
	CRomSettingsDB::Get()->GetSettings( id, &settings );

	fprintf(fh, "{%08x%08x-%02x}\t\t// %s\n", id.CRC[0], id.CRC[1], id.CountryID, settings.GameName.c_str() );
	
	if (preferences.PatchesEnabled)								fprintf(fh, "PatchesEnabled=yes\n" );
	if (preferences.SpeedSyncEnabled)							fprintf(fh, "SpeedSyncEnabled=yes\n");
	if (!preferences.DynarecEnabled)							fprintf(fh, "DynarecEnabled=no\n");
	if (!preferences.DynarecStackOptimisation)					fprintf(fh, "DynarecStackOptimisation=no\n");
#ifdef DAEDALUS_PSP
	if (preferences.CheckTextureHashFrequency != THF_DISABLED)	fprintf(fh, "CheckTextureHashFrequency=%d\n", ROM_GetTexureHashFrequencyAsFrames( preferences.CheckTextureHashFrequency ) );
	if (preferences.Frameskip != FV_DISABLED)					fprintf(fh, "Frameskip=%d\n", ROM_GetFrameskipValueAsInt( preferences.Frameskip ) );
	if (preferences.AudioEnabled)								fprintf(fh, "AudioEnabled=yes\n");
	if (preferences.AudioAdaptFrequency)						fprintf(fh, "AudioAdaptFrequency=yes\n");
	if (preferences.ControllerIndex != 0)						fprintf(fh, "Controller=%s\n", CInputManager::Get()->GetConfigurationName( preferences.ControllerIndex ));
#endif

	fprintf(fh, "\n");			// Spacer
}

//*****************************************************************************
//	Write out the .ini file, keeping the original comments intact
//*****************************************************************************
void IPreferences::Commit()
{
	FILE * fh( fopen(mFilename.c_str(), "w") );
	if (fh != NULL)
	{
		const SGlobalPreferences	defaults;

#define OUTPUT_BOOL( b, nm, def )		if( b.nm != def.nm )					fprintf( fh, "%s=%s\n", #nm, b.nm ? "yes" : "no" );
#define OUTPUT_FLOAT( b, nm, def )		if( fabsf( b.nm - def.nm ) > 0.001f )	fprintf( fh, "%s=%f\n", #nm, b.nm );
#define OUTPUT_INT( b, nm, def )		if( b.nm != def.nm )					fprintf( fh, "%s=%d\n", #nm, b.nm );

		OUTPUT_BOOL( gGlobalPreferences, DisplayFramerate, defaults );
		OUTPUT_BOOL( gGlobalPreferences, SoftwareClipping, defaults );
		OUTPUT_BOOL( gGlobalPreferences, UseVFPUTnL, defaults );
		OUTPUT_BOOL( gGlobalPreferences, HighlightInexactBlendModes, defaults );
		OUTPUT_FLOAT( gGlobalPreferences, StickMinDeadzone, defaults );
		OUTPUT_FLOAT( gGlobalPreferences, StickMaxDeadzone, defaults );
		OUTPUT_INT( gGlobalPreferences, ViewportType, defaults );

		fprintf( fh, "\n\n" );

		for ( PreferencesMap::const_iterator it = mPreferences.begin(); it != mPreferences.end(); ++it )
		{
			OutputSectionDetails( it->first, it->second, fh );
		}

		fclose( fh );
		mDirty = false;
	}
}

//*****************************************************************************
// Retreive the preferences for the specified rom. Returns false if the rom is 
// not in the database
//*****************************************************************************
bool	IPreferences::GetRomPreferences( const RomID & id, SRomPreferences * preferences ) const
{
	PreferencesMap::const_iterator	it( mPreferences.find( id ) );
	if ( it != mPreferences.end() )
	{
		*preferences = it->second;
		return true;
	}
	else
	{
		return false;
	}
}

//*****************************************************************************
// Update the preferences for the specified rom - creates a new entry if necessary
//*****************************************************************************
void	IPreferences::SetRomPreferences( const RomID & id, const SRomPreferences & preferences )
{
	PreferencesMap::iterator	it( mPreferences.find( id ) );
	if ( it != mPreferences.end() )
	{
		it->second = preferences;
	}
	else
	{
		mPreferences[ id ] = preferences;
	}
	
	mDirty = true;
}

//*****************************************************************************
//
//*****************************************************************************
SGlobalPreferences::SGlobalPreferences()
:	DisplayFramerate( false )
,	SoftwareClipping( true )
,	UseVFPUTnL( true )
,	HighlightInexactBlendModes( false )
,	StickMinDeadzone( 0.23f )
,	StickMaxDeadzone( 1.0f )
,	ViewportType( VT_FULLSCREEN )
{
}

//*****************************************************************************
//
//*****************************************************************************
void SGlobalPreferences::Apply() const
{

}

//*****************************************************************************
//
//*****************************************************************************
SRomPreferences::SRomPreferences()
	:	PatchesEnabled( false )
	,	SpeedSyncEnabled( false )
	,	DynarecEnabled( true )
	,	DynarecStackOptimisation( true )
#ifdef DAEDALUS_PSP
	,	CheckTextureHashFrequency( THF_DISABLED )
	,	Frameskip( FV_DISABLED )
	,	AudioEnabled( false )
	,	AudioAdaptFrequency( false )
	,	ControllerIndex( 0 )
#endif
{
}

//*****************************************************************************
//
//*****************************************************************************
void SRomPreferences::Reset()
{
	PatchesEnabled = false;
	SpeedSyncEnabled = false;
	DynarecEnabled = true;
	DynarecStackOptimisation = true;
#ifdef DAEDALUS_PSP
	CheckTextureHashFrequency = THF_DISABLED;
	Frameskip = FV_DISABLED;
	AudioEnabled = false;
	AudioAdaptFrequency = false;
	ControllerIndex = 0;
#endif
}

//*****************************************************************************
//
//*****************************************************************************
void	SRomPreferences::Apply() const
{
	gOSHooksEnabled		= PatchesEnabled;
	FramerateLimiter_SetLimit( SpeedSyncEnabled );
	gDynarecEnabled		= g_ROM.settings.DynarecSupported && DynarecEnabled;
	gDynarecStackOptimisation	= g_ROM.settings.DynarecStackOptimisation && DynarecStackOptimisation;
#ifdef DAEDALUS_PSP
	gCheckTextureHashFrequency = ROM_GetTexureHashFrequencyAsFrames( CheckTextureHashFrequency );
	gFrameskipValue = Frameskip;

	gAudioPluginEnabled = AudioEnabled;
	gAdaptFrequency = AudioAdaptFrequency;

	CInputManager::Get()->SetConfiguration( ControllerIndex );
#endif

}


//*****************************************************************************
//
//*****************************************************************************
static const u32 gTextureHashFreqeuncies[] =
{
	0,	//THF_DISABLED = 0,
	1,	//THF_EVERY_FRAME,
	2,	//THF_EVERY_2,
	5,	//THF_EVERY_5,
	10,	//THF_EVERY_10,
	30,	//THF_EVERY_30,
};

static const char * const gTextureHashFreqeuncyDescriptions[] =
{
	"Disabled",			//THF_DISABLED = 0,
	"Every Frame",		//THF_EVERY_FRAME,
	"Every 2 Frames",	//THF_EVERY_2,
	"Every 5 Frames",	//THF_EVERY_5,
	"Every 10 Frames",	//THF_EVERY_10,
	"Every 30 Frames",	//THF_EVERY_30,
};

//*****************************************************************************
//
//*****************************************************************************
u32	ROM_GetTexureHashFrequencyAsFrames( ETextureHashFrequency thf )
{
	if(thf >= 0 && thf < NUM_THF)
	{
		return gTextureHashFreqeuncies[ thf ];
	}

	return 0;
}

//*****************************************************************************
//
//*****************************************************************************
ETextureHashFrequency	ROM_GetTextureHashFrequencyFromFrames( u32 frames )
{
	for( u32 i = 0; i < NUM_THF; ++i )
	{
		if( frames <= gTextureHashFreqeuncies[ i ] )
		{
			return ETextureHashFrequency( i );
		}
	}

	return THF_EVERY_30;	// Return the maximum
}


//*****************************************************************************
//
//*****************************************************************************
const char * ROM_GetTextureHashFrequencyDescription( ETextureHashFrequency thf )
{
	if(thf >= 0 && thf < NUM_THF)
	{
		return gTextureHashFreqeuncyDescriptions[ thf ];
	}

	return "?";
}

//*****************************************************************************
//
//*****************************************************************************
u32	ROM_GetFrameskipValueAsInt( EFrameskipValue value )
{
	return value;
}

//*****************************************************************************
//
//*****************************************************************************
EFrameskipValue	ROM_GetFrameskipValueFromInt( u32 value )
{
	if( value < FV_DISABLED )
		return FV_DISABLED;

	if( value > FV_10 )
		return FV_10;

	return EFrameskipValue( value );
}

//*****************************************************************************
//
//*****************************************************************************
const char *			ROM_GetFrameskipDescription( EFrameskipValue value )
{
	switch( value )
	{
	case FV_DISABLED:		return "Disabled";
	case FV_1:				return "1";
	case FV_2:				return "2";
	case FV_3:				return "3";
	case FV_4:				return "4";
	case FV_5:				return "5";
	case FV_6:				return "6";
	case FV_7:				return "7";
	case FV_8:				return "8";
	case FV_9:				return "9";
	case FV_10:				return "10";

	case NUM_FRAMESKIP_VALUES:
		break;
	}
	DAEDALUS_ERROR( "Unhandled frameskip value" );
	return "?";
}


