/*
Copyright (C) 2007 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "SavestateSelectorComponent.h"

#include "UIContext.h"
#include "UIScreen.h"
#include "UIElement.h"
#include "UICommand.h"

#include "Core/ROM.h"
#include "Core/SaveState.h"

#include "DaedStream.h"
#include "DaedIO.h"

#include "PSPGraphics/Vector2.h"

#include "PSPGraphics/DrawText.h"

#include <pspctrl.h>
#include <pspgu.h>

namespace
{
	const char *			INSTRUCTIONS_TEXT = "Select a savestate slot";

	const char * const		SAVING_STATUS_TEXT  = "Saving...";
	const char * const		LOADING_STATUS_TEXT = "Loading...";

	const u32				TEXT_AREA_TOP = 32;
	const u32				TEXT_AREA_LEFT = 20;
	const u32				TEXT_AREA_RIGHT = 460;

	const s32				DESCRIPTION_AREA_TOP = 272-20;		// We render text aligned from the bottom, so this is largely irrelevant
	const s32				DESCRIPTION_AREA_BOTTOM = 272-10;
	const s32				DESCRIPTION_AREA_LEFT = 16;
	const s32				DESCRIPTION_AREA_RIGHT = 480-16;

	const u32				NUM_SAVESTATE_SLOTS = 16;

	const u32				INVALID_SLOT = u32( -1 );
}

//*************************************************************************************
//
//*************************************************************************************
class ISavestateSelectorComponent : public CSavestateSelectorComponent
{
	public:

		ISavestateSelectorComponent( CUIContext * p_context, EAccessType access_type, CFunctor1< const char * > * on_slot_selected );
		~ISavestateSelectorComponent();

		// CUIScreen
		virtual void				Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons );
		virtual void				Render();
		virtual bool				IsFinished() const									{ return mIsFinished; }

	private:
				void				OnSlotSelected( u32 slot_idx );

	private:
		EAccessType					mAccessType;
		CFunctor1< const char * > *	mOnSlotSelected;

		u32							mSelectedSlot;
		bool						mIsFinished;

		CUIElementBag				mElements;

		bool						mSlotEmpty[ NUM_SAVESTATE_SLOTS ];
};

//*************************************************************************************
//
//*************************************************************************************
CSavestateSelectorComponent::~CSavestateSelectorComponent()
{
}

//*************************************************************************************
//
//*************************************************************************************
CSavestateSelectorComponent::CSavestateSelectorComponent( CUIContext * p_context )
:	CUIComponent( p_context )
{
}

//*************************************************************************************
//
//*************************************************************************************
CSavestateSelectorComponent *	CSavestateSelectorComponent::Create( CUIContext * p_context, EAccessType access_type, CFunctor1< const char * > * on_slot_selected )
{
	return new ISavestateSelectorComponent( p_context, access_type, on_slot_selected );
}

//*************************************************************************************
//
//*************************************************************************************
namespace
{
	void MakeSaveSlotPath( char * path, u32 slot_idx )
	{
		char	filename[ MAX_PATH ];

		sprintf( filename, "saveslot%u.ss", slot_idx );

		IO::Path::Combine( path, gDaedalusExePath, "SaveStates" );
		IO::Directory::EnsureExists( path );		// Ensure this dir exists

		IO::Path::Append( path, filename );
	}
}

//*************************************************************************************
//
//*************************************************************************************
ISavestateSelectorComponent::ISavestateSelectorComponent( CUIContext * p_context, EAccessType access_type, CFunctor1< const char * > * on_slot_selected )
:	CSavestateSelectorComponent( p_context )
,	mAccessType( access_type )
,	mOnSlotSelected( on_slot_selected )
,	mSelectedSlot( INVALID_SLOT )
,	mIsFinished( false )
{
	const char * description_text( access_type == AT_SAVING ? "Select the slot in which to save" : "Select the slot from which to load" );

	for( u32 i = 0; i < NUM_SAVESTATE_SLOTS; ++i )
	{
		COutputStringStream		str;
		str << "Slot " << (i+1) << ": ";

		char filename[ MAX_PATH ];
		MakeSaveSlotPath( filename, i );

		RomID			rom_id( SaveState_GetRomID( filename ) );
		RomSettings		settings;

		CUIElement *	element;
		if( !rom_id.Empty() && CRomSettingsDB::Get()->GetSettings( rom_id, &settings ) )
		{
			str << settings.GameName.c_str();
			mSlotEmpty[ i ] = false;
		}
		else
		{
			str << "<empty>";
			mSlotEmpty[ i ] = true;
		}

		// 
		//	Don't allow empty slots to be loaded
		//
		if( access_type == AT_LOADING && mSlotEmpty[ i ] )
		{
			element = new CUICommandDummy( str.c_str(), description_text );
		}
		else
		{
			CFunctor1< u32 > *		functor_1( new CMemberFunctor1< ISavestateSelectorComponent, u32 >( this, &ISavestateSelectorComponent::OnSlotSelected ) );
			CFunctor *				curried( new CCurriedFunctor< u32 >( functor_1, i ) );

			element = new CUICommandImpl( curried, str.c_str(), description_text );
		}

		mElements.Add( element );
	}
}

//*************************************************************************************
//
//*************************************************************************************
ISavestateSelectorComponent::~ISavestateSelectorComponent()
{
	delete mOnSlotSelected;
}

//*************************************************************************************
//
//*************************************************************************************
void	ISavestateSelectorComponent::Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons )
{
	//
	//	Trigger the save on the first update AFTER mSelectedSlot was set.
	//	This ensures we get at least one frame where we can display "Saving..." etc.
	//
	if( mSelectedSlot != INVALID_SLOT && !mIsFinished )
	{
		mIsFinished = true;

		char filename[ MAX_PATH ];
		MakeSaveSlotPath( filename, mSelectedSlot );

		(*mOnSlotSelected)( filename );
	}

	if(old_buttons != new_buttons)
	{
		if( new_buttons & PSP_CTRL_UP )
		{
			mElements.SelectPrevious();
		}
		if( new_buttons & PSP_CTRL_DOWN )
		{
			mElements.SelectNext();
		}

		CUIElement *	element( mElements.GetSelectedElement() );
		if( element != NULL )
		{
			if( new_buttons & PSP_CTRL_LEFT )
			{
				element->OnPrevious();
			}
			if( new_buttons & PSP_CTRL_RIGHT )
			{
				element->OnNext();
			}
			if( new_buttons & (PSP_CTRL_CROSS|PSP_CTRL_START) )
			{
				// Commit settings
				element->OnSelected();
			}
			if( new_buttons & (PSP_CTRL_CIRCLE|PSP_CTRL_SELECT) )
			{
				// Discard settings
				mIsFinished = true;
			}
		}
	}
}

//*************************************************************************************
//
//*************************************************************************************
void	ISavestateSelectorComponent::Render()
{
	const u32	font_height( CDrawText::GetFontHeight() );

	if( mSelectedSlot == INVALID_SLOT )
	{
		mElements.Draw( mpContext, TEXT_AREA_LEFT, TEXT_AREA_RIGHT, AT_LEFT, TEXT_AREA_TOP );

		CUIElement *	element( mElements.GetSelectedElement() );
		if( element != NULL )
		{
			const char *		p_description( element->GetDescription() );

			mpContext->DrawTextArea( DESCRIPTION_AREA_LEFT,
									 DESCRIPTION_AREA_TOP,
									 DESCRIPTION_AREA_RIGHT - DESCRIPTION_AREA_LEFT,
									 DESCRIPTION_AREA_BOTTOM - DESCRIPTION_AREA_TOP,
									 p_description,
									 DrawTextUtilities::TextWhite,
									 VA_BOTTOM );
		}
	}
	else
	{
		const char * title_text( mAccessType == AT_SAVING ? SAVING_STATUS_TEXT : LOADING_STATUS_TEXT );

		s32 x( DrawTextUtilities::AlignText( 0, mpContext->GetScreenWidth(), title_text, AT_CENTRE ) );
		s32 y( ( mpContext->GetScreenHeight() - font_height ) / 2 );
		mpContext->DrawText( x, y, title_text, mpContext->GetDefaultTextColour() ); y += font_height;
	}
}

//*************************************************************************************
//
//*************************************************************************************
void	ISavestateSelectorComponent::OnSlotSelected( u32 slot_idx )
{
	if( slot_idx >= NUM_SAVESTATE_SLOTS )
		return;

	// Don't allow empty slots to be loaded
	if( mAccessType == AT_LOADING && mSlotEmpty[ slot_idx ] )
		return;

	mSelectedSlot = slot_idx;
}
