/*
Copyright (C) 2006 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "RomSelectorComponent.h"
#include "UIContext.h"
#include "UIScreen.h"

#include <psptypes.h>
#include <pspkernel.h>
#include <pspctrl.h>
#include <pspdisplay.h>
#include <pspgu.h>

#include "PSPGraphics/Vector2.h"
#include "PSPGraphics/DrawText.h"
#include "PSPGraphics/PSPColour.h"
#include "PSPGraphics/NativeTexture.h"

#include "Core/ROM.h"
#include "Core/RomSettings.h"

#include "DaedIO.h"
#include "DaedMathUtil.h"

#include <string>
#include <vector>
#include <map>
#include <functional>

namespace
{
	const char * const		gRomsDirectories[] = 
	{
		"Roms/",
		"ms0:/n64/",
	};

	const char		gCategoryLetters[] = "1abcdefghijklmnopqrstuvwxyz?";

	enum ECategory
	{
		C_NUMBERS = 0,
		C_A, C_B, C_C, C_D, C_E, C_F, C_G, C_H, C_I, C_J, C_K, C_L, C_M,
		C_N, C_O, C_P, C_Q, C_R, C_S, C_T, C_U, C_V, C_W, C_X, C_Y, C_Z,
		C_UNK,
		NUM_CATEGORIES,
	};

	DAEDALUS_STATIC_ASSERT( ARRAYSIZE( gCategoryLetters ) == NUM_CATEGORIES +1 );

	ECategory		GetCategory( char c )
	{
		if( isalpha( c ) )
		{
			c = tolower( c );
			return ECategory( C_A + (c - 'a') );
		}
		else if( c >= '0' && c <= '9' )
		{
			return C_NUMBERS;
		}
		else
		{
			return C_UNK;
		}
	}

	char	GetCategoryLetter( ECategory category )
	{
		DAEDALUS_ASSERT( category >= 0 && category < NUM_CATEGORIES, "Invalid category" );
		return gCategoryLetters[ category ];
	}

	const u32				ICON_AREA_TOP = 32;
	const u32				ICON_AREA_LEFT = 20;
	const u32				ICON_AREA_WIDTH = 128;
	const u32				ICON_AREA_HEIGHT = 96;

	const u32				TEXT_AREA_TOP = 32;
	const u32				TEXT_AREA_LEFT = 20 + 128 + 10;
	const u32				TEXT_AREA_WIDTH = 480 - TEXT_AREA_LEFT;
	const u32				TEXT_AREA_HEIGHT = 216;

	const char * const		gNoRomsText[] =
	{
		"Daedalus could not find any roms to load.",
		"You can add roms to both the \\N64\\ directory on your memory stick (e.g. P:\\N64\\) or the Roms directory within the Daedalus folder (e.g. P:\\PSP\\GAME\\Daedalus\\Roms\\).",
		"Daedalus recognises a number of different filetypes, including .zip, .z64, .v64, .rom, .pal, .usa and .jap.",
	};

	const u32				CATEGORY_AREA_TOP = TEXT_AREA_TOP + TEXT_AREA_HEIGHT + 5;
	const u32				CATEGORY_AREA_LEFT = 20;

	const u32				ROW_HEIGHT = 12;

	const char * const		gPreviewDirectory = "Resources/Preview/";

	const f32				PREVIEW_SCROLL_WAIT = 1.0f;		// seconds to wait for scrolling to stop before loading preview (prevent thrashing)
	const f32				PREVIEW_FADE_TIME = 1.0f;		// seconds
}

//*************************************************************************************
//
//*************************************************************************************
struct SRomInfo
{
	std::string		mFilename;

	RomID			mRomID;
	u32				mRomSize;
	ECicType		mCicType;

	RomSettings		mSettings;

	SRomInfo( const char * filename )
		:	mFilename( filename )
	{
		if ( ROM_GetRomDetailsByFilename( filename, &mRomID, &mRomSize, &mCicType ) )
		{
			mSettings.GameName = IO::Path::FindFileName( filename );

			if ( !CRomSettingsDB::Get()->GetSettings( mRomID, &mSettings ) )
			{
				// Create new entry, add
				mSettings.Reset();
				mSettings.Comment = "No Comment";

				//
				// We want to get the "internal" name for this rom from the header
				// Failing that, use the filename
				//
				if ( !ROM_GetRomName( filename, mSettings.GameName ) )
				{
					mSettings.GameName = IO::Path::FindFileName( filename );
				}
				CRomSettingsDB::Get()->SetSettings( mRomID, mSettings );
			}
		}
		else
		{
			mSettings.GameName = "Can't get rom info";
		}

	}
};

//*************************************************************************************
//
//*************************************************************************************
static ECategory Categorise( const char * name )
{
	char	c( name[ 0 ] );
	return GetCategory( c );
}

static bool SortByGameName( const SRomInfo * a, const SRomInfo * b )
{
	// Sort by the category first, then on the actual string.
	ECategory	cat_a( Categorise( a->mSettings.GameName.c_str() ) );
	ECategory	cat_b( Categorise( b->mSettings.GameName.c_str() ) );

	if( cat_a != cat_b )
	{
		return cat_a < cat_b;
	}

	return a->mSettings.GameName < b->mSettings.GameName;
}

//*************************************************************************************
//
//*************************************************************************************
class IRomSelectorComponent : public CRomSelectorComponent
{
		typedef std::vector<SRomInfo*>	RomInfoList;
		typedef std::map< ECategory, u32 >	AlphaMap;
	public:

		IRomSelectorComponent( CUIContext * p_context, CFunctor1< const char * > * on_rom_selected );
		~IRomSelectorComponent();

		// CUIComponent
		virtual void				Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons );
		virtual void				Render();

	private:
				void				RenderPreview();
				void				RenderRomList();
				void				RenderCategoryList();

				void				AddRomDirectory(const char * p_roms_dir, RomInfoList & roms);

				ECategory			GetCurrentCategory() const;

				void				DrawInfoText( CUIContext * p_context, s32 y, const char * field_txt, const char * value_txt );

	private:
		CFunctor1< const char * > *	OnRomSelected; 
		RomInfoList					mRomsList;
		AlphaMap					mRomCategoryMap;
		u32							mCurrentSelection;
		s32							mCurrentScrollOffset;
		float						mSelectionAccumulator;
		std::string					mSelectedRom;

		bool						mDisplayFilenames;

		CNativeTexture *			mpPreviewTexture;
		u32							mPreviewIdx;
		float						mPreviewLoadedTime;		// How long the preview has been loaded (so we can fade in)
		float						mTimeSinceScroll;		// 
};

// XXXX If this is added as a member var seem to get random bus errors!!
SceIoDirent					gDirEntry;

// XXXX Need to reevaluate whether this safety buffer is needed anymore..
static u8					gBytes[128];

//*************************************************************************************
//
//*************************************************************************************
CRomSelectorComponent::CRomSelectorComponent( CUIContext * p_context )
:	CUIComponent( p_context )
{
}

//*************************************************************************************
//
//*************************************************************************************
CRomSelectorComponent::~CRomSelectorComponent()
{
}

//*************************************************************************************
//
//*************************************************************************************
CRomSelectorComponent *	CRomSelectorComponent::Create( CUIContext * p_context, CFunctor1< const char * > * on_rom_selected )
{
	return new IRomSelectorComponent( p_context, on_rom_selected );
}

//*************************************************************************************
//
//*************************************************************************************
IRomSelectorComponent::IRomSelectorComponent( CUIContext * p_context, CFunctor1< const char * > * on_rom_selected )
:	CRomSelectorComponent( p_context )
,	OnRomSelected( on_rom_selected )
,	mCurrentSelection( 0 )
,	mCurrentScrollOffset( 0 )
,	mSelectionAccumulator( 0 )
,	mpPreviewTexture( NULL )
,	mPreviewIdx( u32(-1) )
,	mPreviewLoadedTime( 0.0f )
,	mTimeSinceScroll( 0.0f )
{
	use(gBytes);

	for( u32 i = 0; i < ARRAYSIZE( gRomsDirectories ); ++i )
	{
		AddRomDirectory( gRomsDirectories[ i ], mRomsList );
	}

	sort( mRomsList.begin(), mRomsList.end(), SortByGameName );

	// Build up a map of the first location for each initial letter
	for( u32 i = 0; i < mRomsList.size(); ++i )
	{
		const char *	p_gamename( mRomsList[ i ]->mSettings.GameName.c_str() );
		ECategory		category( Categorise( p_gamename ) );

		if( mRomCategoryMap.find( category ) == mRomCategoryMap.end() )
		{
			mRomCategoryMap[ category ] = i;
		}
	}
}

//*************************************************************************************
//
//*************************************************************************************
IRomSelectorComponent::~IRomSelectorComponent()
{
	for(RomInfoList::iterator it = mRomsList.begin(); it != mRomsList.end(); ++it)
	{
		SRomInfo *	p_rominfo( *it );

		delete p_rominfo;
	}
	mRomsList.clear();

	SAFE_RELEASE( mpPreviewTexture );

	delete OnRomSelected;
}

//*************************************************************************************
//
//*************************************************************************************
bool	CFileSystem_FindFirstFile( const char * path, SceUID & handle )
{
	handle = sceIoDopen( path );

	return ( handle >= 0 );
}

//*************************************************************************************
//
//*************************************************************************************
bool	CFileSystem_FindNextFile( SceIoDirent & dir_entry, SceUID handle )
{
	DAEDALUS_ASSERT( handle >= 0, "Cannot search with invalid directory handle" );

	const s32	error_code( sceIoDread( handle, &dir_entry ) );

	DAEDALUS_ASSERT( error_code >= 0, "Error reading directory entry" );

	if ( error_code > 0 )
	{
		return true;
	}

	return false;
}

//*************************************************************************************
//
//*************************************************************************************
bool	CFileSystem_FindCloseFile( SceUID handle )
{
	DAEDALUS_ASSERT( handle >= 0, "Trying to close an invalid directory handle" );

	return ( sceIoDclose( handle ) >= 0 );
}

//*************************************************************************************
//
//*************************************************************************************
void	IRomSelectorComponent::AddRomDirectory(const char * p_roms_dir, RomInfoList & roms)
{
	SceUID				dir_handle;
	if(CFileSystem_FindFirstFile( p_roms_dir, dir_handle ))
	{
		while(CFileSystem_FindNextFile( gDirEntry, dir_handle ))
		{
			const char * rom_filename( gDirEntry.d_name );
			const char * last_period( strrchr( rom_filename, '.' ) );
			if(last_period != NULL)
			{
				if(_strcmpi(last_period, ".v64") == 0 ||
				   _strcmpi(last_period, ".z64") == 0 ||
				   _strcmpi(last_period, ".n64") == 0 ||
				   _strcmpi(last_period, ".rom") == 0 ||
				   _strcmpi(last_period, ".jap") == 0 ||
				   _strcmpi(last_period, ".pal") == 0 ||
				   _strcmpi(last_period, ".usa") == 0 ||
				   _strcmpi(last_period, ".zip") == 0)
				{
					std::string		full_path;

					full_path = p_roms_dir;
					full_path += rom_filename;

					SRomInfo *	p_rom_info = new SRomInfo( full_path.c_str() );

					roms.push_back( p_rom_info );
				}
			}
		}

		CFileSystem_FindCloseFile( dir_handle );
	}
}

//*************************************************************************************
//
//*************************************************************************************
ECategory	IRomSelectorComponent::GetCurrentCategory() const
{
	if( !mRomsList.empty() )
	{
		return Categorise( mRomsList[ mCurrentSelection ]->mSettings.GameName.c_str() );
	}

	return C_NUMBERS;
}

//*************************************************************************************
//
//*************************************************************************************
void IRomSelectorComponent::DrawInfoText(  CUIContext * p_context, s32 y, const char * field_txt, const char * value_txt  )
{
	s32 x;

	c32			colour(	p_context->GetDefaultTextColour() );

	x = DrawTextUtilities::AlignText( ICON_AREA_LEFT, ICON_AREA_LEFT + ICON_AREA_WIDTH, field_txt, AT_LEFT );
	p_context->DrawText( x, y, field_txt, colour );

	x = DrawTextUtilities::AlignText( ICON_AREA_LEFT, ICON_AREA_LEFT + ICON_AREA_WIDTH, value_txt, AT_RIGHT );
	p_context->DrawText( x, y, value_txt, colour );
}

//*************************************************************************************
//
//*************************************************************************************
void IRomSelectorComponent::RenderPreview()
{
	u32		text_height( CDrawText::GetFontHeight() );

	mpContext->DrawRect( ICON_AREA_LEFT-2, ICON_AREA_TOP-2, ICON_AREA_WIDTH+4, ICON_AREA_HEIGHT+4, c32::White );
	mpContext->DrawRect( ICON_AREA_LEFT-1, ICON_AREA_TOP-1, ICON_AREA_WIDTH+2, ICON_AREA_HEIGHT+2, mpContext->GetBackgroundColour() );

	v2	tl( ICON_AREA_LEFT, ICON_AREA_TOP );
	v2	wh( ICON_AREA_WIDTH, ICON_AREA_HEIGHT );

	if( mpPreviewTexture != NULL )
	{
		c32		colour( c32::White );

		if ( mPreviewLoadedTime < PREVIEW_FADE_TIME )
		{
			colour = c32( 255, 255, 255, u8( mPreviewLoadedTime * 255.f / PREVIEW_FADE_TIME ) );
		}

		mpContext->DrawRect( ICON_AREA_LEFT, ICON_AREA_TOP, ICON_AREA_WIDTH, ICON_AREA_HEIGHT, c32::Black );
		mpContext->RenderTexture( mpPreviewTexture, tl, wh, colour );
	}
	else
	{
		mpContext->DrawRect( ICON_AREA_LEFT, ICON_AREA_TOP, ICON_AREA_WIDTH, ICON_AREA_HEIGHT, c32::Black );
	}

		s32 y = ICON_AREA_TOP + ICON_AREA_HEIGHT + 10;

	if( mCurrentSelection < mRomsList.size() )
	{
		SRomInfo *	p_rominfo( mRomsList[ mCurrentSelection ] );

		const char *	cic_name( ROM_GetCicName( p_rominfo->mCicType ) );
		const char *	country( ROM_GetCountryNameFromID( p_rominfo->mRomID.CountryID ) );
		u32				rom_size( p_rominfo->mRomSize );

		DrawInfoText( mpContext, y, "Boot:", cic_name );	y += text_height;
		DrawInfoText( mpContext, y, "Country:", country );	y += text_height;
		DrawInfoText( mpContext, y, "Size:", DSPrintf( "%d MB", rom_size / (1024*1024) ) );	y += text_height;

		DrawInfoText( mpContext, y, "Save:", ROM_GetSaveTypeName( p_rominfo->mSettings.SaveType ) ); y += text_height;
		DrawInfoText( mpContext, y, "EPak:", ROM_GetExpansionPakUsageName( p_rominfo->mSettings.ExpansionPakUsage ) ); y += text_height;
		DrawInfoText( mpContext, y, "Dynarec:", p_rominfo->mSettings.DynarecSupported ? "Supported" : "Unsupported" ); y += text_height;
	}
	else
	{
		DrawInfoText( mpContext, y, "Boot:", "" );		y += text_height;
		DrawInfoText( mpContext, y, "Country:", "" );	y += text_height;
		DrawInfoText( mpContext, y, "Size:", "" );		y += text_height;

		DrawInfoText( mpContext, y, "Save:", "" );		y += text_height;
		DrawInfoText( mpContext, y, "EPak:", "" );		y += text_height;
		DrawInfoText( mpContext, y, "Dynarec:", "" );	y += text_height;
	}
}

//*************************************************************************************
//
//*************************************************************************************
void IRomSelectorComponent::RenderRomList()
{
	u32		text_height( CDrawText::GetFontHeight() );
	s32		x,y;
	x = TEXT_AREA_LEFT;
	y = TEXT_AREA_TOP + mCurrentScrollOffset;

	sceGuEnable(GU_SCISSOR_TEST);
	sceGuScissor(TEXT_AREA_LEFT, TEXT_AREA_TOP, TEXT_AREA_LEFT+TEXT_AREA_WIDTH, TEXT_AREA_TOP+TEXT_AREA_HEIGHT);

	const char * const	ptr_text( "o " );
	u32					ptr_text_width( CDrawText::GetTextWidth( ptr_text ) );

	for(u32 i = 0; i < mRomsList.size(); ++i)
	{
		const char *	p_gamename;
		if( mDisplayFilenames )
		{
			p_gamename = mRomsList[ i ]->mFilename.c_str();
		}
		else
		{
			p_gamename = mRomsList[ i ]->mSettings.GameName.c_str();
		}

		//
		// Check if this entry would be onscreen
		//
		if((y+ROW_HEIGHT) >= s32(TEXT_AREA_TOP) && y < s32(TEXT_AREA_TOP + TEXT_AREA_HEIGHT))
		{
			c32		colour;

			if(i == mCurrentSelection)
			{
				colour = mpContext->GetSelectedTextColour();
				mpContext->DrawText( x, y, ptr_text, colour );
			}
			else
			{
				colour = mpContext->GetDefaultTextColour();
			}
			mpContext->DrawText( x + ptr_text_width, y, p_gamename, colour );
		}
		y += text_height;
	}

	// Restore scissoring
	sceGuScissor(0,0, 480,272);
}


//*************************************************************************************
//
//*************************************************************************************
void IRomSelectorComponent::RenderCategoryList()
{
	s32 x = CATEGORY_AREA_LEFT;
	s32 y = CATEGORY_AREA_TOP;

	ECategory current_category( GetCurrentCategory() );

	for( u32 i = 0; i < NUM_CATEGORIES; ++i )
	{
		ECategory	category = ECategory( i );
		c32			colour;

		AlphaMap::const_iterator it( mRomCategoryMap.find( category ) );
		if( it != mRomCategoryMap.end() )
		{
			if( current_category == category )
			{
				colour = mpContext->GetSelectedTextColour();
			}
			else
			{
				colour = mpContext->GetDefaultTextColour();
			}
		}
		else
		{
			colour = c32( 180, 180, 180 );
		}

		char str[ 16 ];
		sprintf( str, "%c ", GetCategoryLetter( category ) );
		mpContext->DrawText( x, y, str, colour );
		x += CDrawText::GetTextWidth( str );
	}
}

//*************************************************************************************
//
//*************************************************************************************
void IRomSelectorComponent::Render()
{
	RenderPreview();

	if( mRomsList.empty() )
	{
		s32 offset( 0 );
		for( u32 i = 0; i < ARRAYSIZE( gNoRomsText ); ++i )
		{
			offset += mpContext->DrawTextArea( TEXT_AREA_LEFT, TEXT_AREA_TOP + offset, TEXT_AREA_WIDTH, TEXT_AREA_HEIGHT - offset, gNoRomsText[ i ], DrawTextUtilities::TextWhite, VA_TOP );
			offset += 4;
		}
	}
	else
	{
		RenderRomList();
	}

	RenderCategoryList();
}

//*************************************************************************************
//
//*************************************************************************************
void	IRomSelectorComponent::Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons )
{
	static const float	SCROLL_RATE_PER_SECOND = 25.0f;		// 25 roms/second

	mSelectionAccumulator += stick.y * SCROLL_RATE_PER_SECOND * elapsed_time; 

	ECategory current_category( GetCurrentCategory() );

	u32				initial_selection( mCurrentSelection );

	mDisplayFilenames = (new_buttons & PSP_CTRL_TRIANGLE) != 0;

	if(old_buttons != new_buttons)
	{
		if(new_buttons & PSP_CTRL_LEFT)
		{
			// Search for the next valid predecessor
			while(current_category > 0)
			{
				current_category = ECategory( current_category - 1 );
				AlphaMap::const_iterator it( mRomCategoryMap.find( current_category ) );
				if( it != mRomCategoryMap.end() )
				{
					mCurrentSelection = it->second;
					break;
				}
			}
		}
		if(new_buttons & PSP_CTRL_RIGHT)
		{
			// Search for the next valid predecessor
			while(current_category < NUM_CATEGORIES-1)
			{
				current_category = ECategory( current_category + 1 );
				AlphaMap::const_iterator it( mRomCategoryMap.find( current_category ) );
				if( it != mRomCategoryMap.end() )
				{
					mCurrentSelection = it->second;
					break;
				}
			}
		}


		if(new_buttons & PSP_CTRL_UP)
		{
			if(mCurrentSelection > 0)
			{
				mCurrentSelection--;
			}
		}
		if(new_buttons & PSP_CTRL_DOWN)
		{
			if(mCurrentSelection < mRomsList.size() - 1)
			{
				mCurrentSelection++;
			}
		}

		if((new_buttons & PSP_CTRL_START) ||
			(new_buttons & PSP_CTRL_CROSS))
		{
			if(mCurrentSelection < mRomsList.size())
			{
				mSelectedRom = mRomsList[ mCurrentSelection ]->mFilename;

				if(OnRomSelected != NULL)
				{
					(*OnRomSelected)( mSelectedRom.c_str() );
				}
			}
		}
	}

	//
	//	Apply the selection accumulator
	//
	f32		current_vel( mSelectionAccumulator );
	while(mSelectionAccumulator >= 1.0f)
	{
		if(mCurrentSelection < mRomsList.size() - 1)
		{
			mCurrentSelection++;
		}
		mSelectionAccumulator -= 1.0f;
	}
	while(mSelectionAccumulator <= -1.0f)
	{
		if(mCurrentSelection > 0)
		{
			mCurrentSelection--;
		}
		mSelectionAccumulator += 1.0f;
	}

	//
	//	Scroll to keep things in view
	//	We add on 'current_vel * 2' to keep the selection highlight as close to the
	//	center as possible (as if we're predicting 2 frames ahead)
	//
	if( mRomsList.size() * ROW_HEIGHT > TEXT_AREA_HEIGHT )
	{
		s32		current_selection_y = s32((mCurrentSelection + current_vel * 2) * ROW_HEIGHT) + (ROW_HEIGHT/2) + mCurrentScrollOffset;

		s32		adjust_amount( (TEXT_AREA_HEIGHT/2) - current_selection_y );

		float d( 1.0f - powf(0.993f, elapsed_time * 1000.0f) );

		u32		total_height( mRomsList.size() * ROW_HEIGHT );
		s32		min_offset( TEXT_AREA_HEIGHT - total_height );

		s32	new_scroll_offset = mCurrentScrollOffset + s32(float(adjust_amount) * d);

		mCurrentScrollOffset = daedalus::Clamp( new_scroll_offset, min_offset, s32(0) );
	}
	else
	{
		mCurrentScrollOffset = 0;
	}

	//
	//	Increase a timer is the current selection is still the same (i.e. if we've not scrolled)
	//
	if( initial_selection == mCurrentSelection )
	{
		mTimeSinceScroll += elapsed_time;
	}
	else
	{
		mTimeSinceScroll = 0;
	}

	//
	//	If the current selection is different from the preview, invalidate the picture.
	//	
	//
	if( mCurrentSelection < mRomsList.size() && mPreviewIdx != mCurrentSelection )
	{
		//mPreviewIdx = u32(-1);

		mPreviewLoadedTime -= elapsed_time;
		if(mPreviewLoadedTime < 0.0f)
			mPreviewLoadedTime = 0.0f;
	
		//
		//	If we've waited long enough since starting to scroll, try and load the preview image
		//	Note that it may fail, so we sort out the other flags regardless.
		//
		if( mTimeSinceScroll > PREVIEW_SCROLL_WAIT )
		{
			SAFE_RELEASE( mpPreviewTexture );
			mPreviewLoadedTime = 0.0f;
			mPreviewIdx = mCurrentSelection;

			if( !mRomsList[ mCurrentSelection ]->mSettings.Preview.empty() )
			{
				char		preview_filename[ MAX_PATH + 1 ];
				daedalus::IO::Path::Combine( preview_filename, gPreviewDirectory, mRomsList[ mCurrentSelection ]->mSettings.Preview.c_str() );
				
				mpPreviewTexture = CNativeTexture::CreateFromPng( preview_filename, TexFmt_8888 );
			}
		}
	}

	//
	//	Once the preview has been loaded, increase a timer to fade us in.
	//
	if( mPreviewIdx == mCurrentSelection )
	{
		mPreviewLoadedTime += elapsed_time;
		if(mPreviewLoadedTime > PREVIEW_FADE_TIME)
			mPreviewLoadedTime = PREVIEW_FADE_TIME;
	}
}
