/*
Copyright (C) 2006 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "MainMenuScreen.h"

#include "UIContext.h"
#include "UIScreen.h"
#include "AboutComponent.h"
#include "GlobalSettingsComponent.h"
#include "RomSelectorComponent.h"
#include "SelectedRomComponent.h"
#include "SavestateSelectorComponent.h"

#include "PSPGraphics/DrawText.h"
#include "PSPGraphics/PSPColour.h"
#include "PSPGraphics/GraphicsContext.h"

#include "Core/ROM.h"
#include "Core/CPU.h"
#include "Core/RomSettings.h"

#include "Utility/Preferences.h"

#include "DaedMathUtil.h"

#include <pspctrl.h>
#include <pspgu.h>

#include <string>

namespace 
{
	const u32				TEXT_AREA_TOP = 10;
	const u32				TEXT_AREA_LEFT = 20;
	const u32				TEXT_AREA_RIGHT = 480-20;

	enum EMenuOption
	{
		MO_GLOBAL_SETTINGS = 0,
		MO_ROMS,
		MO_SELECTED_ROM,
		MO_SAVESTATES,
		MO_ABOUT,
		NUM_MENU_OPTIONS,
	};

	const EMenuOption	MO_FIRST_OPTION = MO_GLOBAL_SETTINGS;
	const EMenuOption	MO_LAST_OPTION = MO_ABOUT;

	const char * const	gMenuOptionNames[ NUM_MENU_OPTIONS ] =
	{
		"Global Settings",
		"Roms",
		"Selected Rom",
		"Savestates",
		"About",
	};
}

//*************************************************************************************
//
//*************************************************************************************
class IMainMenuScreen : public CMainMenuScreen, public CUIScreen
{
	public:

		IMainMenuScreen( CUIContext * p_context );
		~IMainMenuScreen();

		// CMainMenuScreen
		virtual void				Run();

		// CUIScreen
		virtual void				Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons );
		virtual void				Render();
		virtual bool				IsFinished() const									{ return mIsFinished; }


	private:
		static	EMenuOption			GetNextOption( EMenuOption option )					{ return EMenuOption( (option + 1) % NUM_MENU_OPTIONS ); }
		static	EMenuOption			GetPreviousOption( EMenuOption option )				{ return EMenuOption( (option + NUM_MENU_OPTIONS -1) % NUM_MENU_OPTIONS ); }

				EMenuOption			GetPreviousOption() const							{ return GetPreviousOption( mCurrentOption ); }
				EMenuOption			GetCurrentOption() const							{ return mCurrentOption; }
				EMenuOption			GetNextOption() const								{ return GetNextOption( mCurrentOption ); }

				EMenuOption			GetPreviousValidOption() const;
				EMenuOption			GetNextValidOption() const;

				bool				IsOptionValid( EMenuOption option ) const;

				void				OnRomSelected( const char * rom_filename );
				void				OnSavestateSelected( const char * savestate_filename );
				void				OnStartEmulation();

	private:
		bool						mIsFinished;

		EMenuOption					mCurrentOption;

		CUIComponent *				mOptionComponents[ NUM_MENU_OPTIONS ];
		CSelectedRomComponent *		mSelectedRomComponent;

		std::string					mRomFilename;
		RomID						mRomID;
};

//*************************************************************************************
//
//*************************************************************************************
CMainMenuScreen::~CMainMenuScreen()
{
}

//*************************************************************************************
//
//*************************************************************************************
CMainMenuScreen *	CMainMenuScreen::Create( CUIContext * p_context )
{
	return new IMainMenuScreen( p_context );
}

//*************************************************************************************
//
//*************************************************************************************
IMainMenuScreen::IMainMenuScreen( CUIContext * p_context )
:	CUIScreen( p_context )
,	mIsFinished( false )
,	mCurrentOption( MO_ROMS )
{
	for( u32 i = 0; i < NUM_MENU_OPTIONS; ++i )
	{
		mOptionComponents[ i ] = NULL;
	}

	mSelectedRomComponent = CSelectedRomComponent::Create( mpContext, new daedalus::CMemberFunctor< IMainMenuScreen >( this, &IMainMenuScreen::OnStartEmulation ) );

	mOptionComponents[ MO_GLOBAL_SETTINGS ]	= CGlobalSettingsComponent::Create( mpContext );
	mOptionComponents[ MO_ROMS ]			= CRomSelectorComponent::Create( mpContext, new daedalus::CMemberFunctor1< IMainMenuScreen, const char * >( this, &IMainMenuScreen::OnRomSelected ) );
	mOptionComponents[ MO_SELECTED_ROM ]	= mSelectedRomComponent;
	mOptionComponents[ MO_SAVESTATES ]		= CSavestateSelectorComponent::Create( mpContext, CSavestateSelectorComponent::AT_LOADING, new daedalus::CMemberFunctor1< IMainMenuScreen, const char * >( this, &IMainMenuScreen::OnSavestateSelected ) );
	mOptionComponents[ MO_ABOUT ]			= CAboutComponent::Create( mpContext );

	for( u32 i = 0; i < NUM_MENU_OPTIONS; ++i )
	{
		DAEDALUS_ASSERT( mOptionComponents[ i ] != NULL, "Unhandled screen" );
	}
}

//*************************************************************************************
//
//*************************************************************************************
IMainMenuScreen::~IMainMenuScreen()
{
	for( u32 i = 0; i < NUM_MENU_OPTIONS; ++i )
	{
		delete mOptionComponents[ i ];
	}
}


//*************************************************************************************
//
//*************************************************************************************
EMenuOption		IMainMenuScreen::GetPreviousValidOption() const
{
	bool			looped( false );
	EMenuOption		current_option( mCurrentOption );

	do
	{
		current_option = GetPreviousOption( current_option );
		looped = current_option == mCurrentOption;
	}
	while( !IsOptionValid( current_option ) && !looped );

	return current_option;
}

//*************************************************************************************
//
//*************************************************************************************
EMenuOption		IMainMenuScreen::GetNextValidOption() const
{
	bool			looped( false );
	EMenuOption		current_option( mCurrentOption );

	do
	{
		current_option = GetNextOption( current_option );
		looped = current_option == mCurrentOption;
	}
	while( !IsOptionValid( current_option ) && !looped );

	return current_option;
}

//*************************************************************************************
//
//*************************************************************************************
bool	IMainMenuScreen::IsOptionValid( EMenuOption option ) const
{
	// Rom Settings is only valid if a rom has already been selected
	if( option == MO_SELECTED_ROM )
	{
		return !mRomFilename.empty();
	}

	return true;
}


//*************************************************************************************
//
//*************************************************************************************
void	IMainMenuScreen::Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons )
{
	if(old_buttons != new_buttons)
	{
		if(new_buttons & PSP_CTRL_LTRIGGER)
		{
			mCurrentOption = GetPreviousValidOption();
			new_buttons &= ~PSP_CTRL_LTRIGGER;
		}
		if(new_buttons & PSP_CTRL_RTRIGGER)
		{
			mCurrentOption = GetNextValidOption();
			new_buttons &= ~PSP_CTRL_RTRIGGER;
		}
	}

	mOptionComponents[ mCurrentOption ]->Update( elapsed_time, stick, old_buttons, new_buttons );
}

//*************************************************************************************
//
//*************************************************************************************
void	IMainMenuScreen::Render()
{
	mpContext->ClearBackground();

	s32			x,y;

	y = TEXT_AREA_TOP;

	const char * p_option_text;

	c32		valid_colour( mpContext->GetDefaultTextColour() );
	c32		invalid_colour( 200, 200, 200 );

	EMenuOption		previous( GetPreviousOption() );
	EMenuOption		current( GetCurrentOption() );
	EMenuOption		next( GetNextOption() );
		
	p_option_text = gMenuOptionNames[ previous ];
	x = DrawTextUtilities::AlignText( TEXT_AREA_LEFT, TEXT_AREA_RIGHT, p_option_text, AT_LEFT );
	mpContext->DrawText( x, y, p_option_text, IsOptionValid( previous ) ? valid_colour : invalid_colour );

	p_option_text = gMenuOptionNames[ current ];
	x = DrawTextUtilities::AlignText( TEXT_AREA_LEFT, TEXT_AREA_RIGHT, p_option_text, AT_CENTRE );
	mpContext->DrawText( x, y, p_option_text, IsOptionValid( current ) ? valid_colour : invalid_colour );

	p_option_text = gMenuOptionNames[ next ];
	x = DrawTextUtilities::AlignText( TEXT_AREA_LEFT, TEXT_AREA_RIGHT, p_option_text, AT_RIGHT );
	mpContext->DrawText( x, y, p_option_text, IsOptionValid( next ) ? valid_colour : invalid_colour );

	mOptionComponents[ mCurrentOption ]->Render();
}

//*************************************************************************************
//
//*************************************************************************************
void	IMainMenuScreen::Run()
{
	mIsFinished = false;
	CUIScreen::Run();

	// Clear everything to black - looks a bit tidier
	CGraphicsContext::Get()->ClearAllSurfaces();
}

//*************************************************************************************
//
//*************************************************************************************
void	IMainMenuScreen::OnRomSelected( const char * rom_filename )
{
	u32			rom_size;
	ECicType	boot_type;

	if(ROM_GetRomDetailsByFilename( rom_filename, &mRomID, &rom_size, &boot_type ))
	{
		mRomFilename = rom_filename;
		mSelectedRomComponent->SetRomID( mRomID );
		mCurrentOption = MO_SELECTED_ROM;
	}
	else
	{
		// XXXX Problem retrieving rom info- should report this!
		mRomFilename = "";
	}
}

//*************************************************************************************
//
//*************************************************************************************
void	IMainMenuScreen::OnSavestateSelected( const char * savestate_filename )
{
	if( CPU_LoadState( savestate_filename ) )
	{
		mIsFinished = true;
	}
	else
	{
		// Should report some kind of error
	}
}

//*************************************************************************************
//
//*************************************************************************************
void	IMainMenuScreen::OnStartEmulation()
{
	SRomPreferences	preferences;
	RomSettings		settings;

	CRomSettingsDB::Get()->GetSettings( mRomID, &settings );
	CPreferences::Get()->GetRomPreferences( mRomID, &preferences );

	ROM_LoadFile( mRomFilename.c_str(), mRomID, settings, preferences );
	mIsFinished = true;
}
