/*
Copyright (C) 2006 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "GlobalSettingsComponent.h"

#include "UIContext.h"
#include "UIScreen.h"
#include "UISetting.h"
#include "AdjustDeadzoneScreen.h"
#include "PSPGraphics/DrawText.h"
#include "PSPGraphics/PSPColour.h"
#include "PSPGraphics/DaedalusGraphics.h"

#include "Utility/FramerateLimiter.h"

#include "Input/InputManager.h"

#include <pspctrl.h>
#include <pspgu.h>

namespace
{
	const u32		TEXT_AREA_TOP = 15+16+16;
	const u32		TEXT_AREA_LEFT = 40;
	const u32		TEXT_AREA_RIGHT = 440;

	const s32		DESCRIPTION_AREA_TOP = 0;		// We render text aligned from the bottom, so this is largely irrelevant
	const s32		DESCRIPTION_AREA_BOTTOM = 272-10;
	const s32		DESCRIPTION_AREA_LEFT = 16;
	const s32		DESCRIPTION_AREA_RIGHT = 480-16;

	class CViewPortSetting : public CUISetting
	{
	public:
		CViewPortSetting( const char * name, const char * description )
			:	CUISetting( name, description )
		{
		}

		virtual	void	OnNext()				{ gGlobalPreferences.ViewportType = EViewportType( (gGlobalPreferences.ViewportType+1) % NUM_VIEWPORT_TYPES ); }
		virtual	void	OnPrevious()			{ gGlobalPreferences.ViewportType = EViewportType( (gGlobalPreferences.ViewportType + NUM_VIEWPORT_TYPES - 1) % NUM_VIEWPORT_TYPES ); }

		virtual const char *	GetSettingName() const
		{
			switch( gGlobalPreferences.ViewportType )
			{
			case VT_UNSCALED_4_3:	return "4:3 unscaled (320x240)";
			case VT_SCALED_4_3:		return "4:3 scaled (362x272)";
			case VT_FULLSCREEN:		return "Fullscreen (480x272)";
			}
			DAEDALUS_ERROR( "Unhandled viewport type" );
			return "?";
		}

	};

	class CAdjustDeadzoneSetting : public CUISetting
	{
	public:
		CAdjustDeadzoneSetting( CUIContext * p_context, const char * name, const char * description )
			:	CUISetting( name, description )
			,	mpContext( p_context )
		{
		}

		virtual	void			OnSelected()
		{
			CAdjustDeadzoneScreen *	adjust_deadzone( CAdjustDeadzoneScreen::Create( mpContext ) );
			adjust_deadzone->Run();
			delete adjust_deadzone;
		}

		virtual const char *	GetSettingName() const
		{
			f32 min_deadzone( gGlobalPreferences.StickMinDeadzone );
			f32 max_deadzone( gGlobalPreferences.StickMaxDeadzone );

			return DSPrintf( "%d/%d", s32( 100.0f * min_deadzone ), s32( 100.0f * max_deadzone ) );
		}

	private:
		CUIContext *			mpContext;
	};

}

//*************************************************************************************
//
//*************************************************************************************
class IGlobalSettingsComponent : public CGlobalSettingsComponent
{
	public:

		IGlobalSettingsComponent( CUIContext * p_context );
		~IGlobalSettingsComponent();

		// CUIComponent
		virtual void				Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons );
		virtual void				Render();

	private:
		CUIElementBag				mElements;
};

//*************************************************************************************
//
//*************************************************************************************
CGlobalSettingsComponent::CGlobalSettingsComponent( CUIContext * p_context )
:	CUIComponent( p_context )
{
}

//*************************************************************************************
//
//*************************************************************************************
CGlobalSettingsComponent::~CGlobalSettingsComponent()
{
}

//*************************************************************************************
//
//*************************************************************************************
CGlobalSettingsComponent *	CGlobalSettingsComponent::Create( CUIContext * p_context )
{
	return new IGlobalSettingsComponent( p_context );
}

//*************************************************************************************
//
//*************************************************************************************
IGlobalSettingsComponent::IGlobalSettingsComponent( CUIContext * p_context )
:	CGlobalSettingsComponent( p_context )
{
	mElements.Add( new CBoolSetting( &gGlobalPreferences.DisplayFramerate, "Display Framerate", "Whether to show the framerate while the rom is running.", "Yes", "No" ) );
	mElements.Add( new CViewPortSetting( "Viewport Size", "The size of the viewport on the PSP." ) );
	mElements.Add( new CAdjustDeadzoneSetting( mpContext, "Stick Deadzone", "Adjust the size of the deadzone applied to the PSP stick while playing. Press Start/X to edit." ) );

	mElements.Add( new CBoolSetting( &gGlobalPreferences.SoftwareClipping, "Software Clipping",	"Enable software clipping of vertices. Disable this for a small speedup, at the expense of image quality.", "Enabled", "Disabled" ) );

#ifndef DAEDALUS_PUBLIC_RELEASE
	mElements.Add( new CBoolSetting( &gGlobalPreferences.HighlightInexactBlendModes, "Highlight Inexact Blend Modes",	"Replace inexact blend modes with a placeholder texture.", "Yes", "No" ) );
	mElements.Add( new CBoolSetting( &gGlobalPreferences.UseVFPUTnL, "VFPU TnL",	"Enable VFPU acceleration of vertex transform and lighting.", "Enabled", "Disabled" ) );
#endif
}

//*************************************************************************************
//
//*************************************************************************************
IGlobalSettingsComponent::~IGlobalSettingsComponent()
{
}

//*************************************************************************************
//
//*************************************************************************************
void	IGlobalSettingsComponent::Update( float elapsed_time, const v2 & stick, u32 old_buttons, u32 new_buttons )
{
	if(old_buttons != new_buttons)
	{
		if( new_buttons & PSP_CTRL_UP )
		{
			mElements.SelectPrevious();
		}
		if( new_buttons & PSP_CTRL_DOWN )
		{
			mElements.SelectNext();
		}

		CUIElement *	element( mElements.GetSelectedElement() );
		if( element != NULL )
		{
			if( new_buttons & PSP_CTRL_LEFT )
			{
				element->OnPrevious();
			}
			if( new_buttons & PSP_CTRL_RIGHT )
			{
				element->OnNext();
			}
			if( new_buttons & (PSP_CTRL_CROSS|PSP_CTRL_START) )
			{
				element->OnSelected();
			}
		}
	}
}

//*************************************************************************************
//
//*************************************************************************************
void	IGlobalSettingsComponent::Render()
{
	mElements.Draw( mpContext, TEXT_AREA_LEFT, TEXT_AREA_RIGHT, AT_CENTRE, TEXT_AREA_TOP );

	CUIElement *	element( mElements.GetSelectedElement() );
	if( element != NULL )
	{
		const char *		p_description( element->GetDescription() );
		mpContext->DrawTextArea( DESCRIPTION_AREA_LEFT,
								 DESCRIPTION_AREA_TOP,
								 DESCRIPTION_AREA_RIGHT - DESCRIPTION_AREA_LEFT,
								 DESCRIPTION_AREA_BOTTOM - DESCRIPTION_AREA_TOP,
								 p_description,
								 DrawTextUtilities::TextWhite,
								 VA_BOTTOM );
	}
}

