/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Various stuff to map an address onto the correct memory region

#include "stdafx.h"
#include "Memory.h"

#include "CPU.h"
#include "DMA.h"
#include "RSP.h"
#include "RSP_HLE.h"
#include "Interrupt.h"
#include "ROM.h"
#include "ROMBuffer.h"
#include "Compatibility.h"

#include "DaedMathUtil.h"

#include "Debug/Dump.h"		// Dump_GetSaveDirectory()
#include "Debug/DebugLog.h"
#include "Debug/DBGConsole.h"

#include "OSHLE/ultra_r4300.h"

#include "Plugins/AudioPlugin.h"
#include "Plugins/GraphicsPlugin.h"

#ifdef DAEDALUS_PSP
// XXXX: Several roms run out of memory if this is set to 8MB. Need to implement pooled allocator for Texture stuff.
static const u32	MAXIMUM_MEM_SIZE( MEMORY_8_MEG );
#else
static const u32	MAXIMUM_MEM_SIZE( MEMORY_8_MEG );
#endif

#undef min

//
//	VirtualAlloc is only supported on Win32 architectures
//
#if defined(DAEDALUS_W32) | defined(DAEDALUS_XB)
#define DAED_USE_VIRTUAL_ALLOC
#endif

// Arbritary counter for least recently used algorithm
static s32 s_nTLBTimeNow = 0;

static void MemoryDoDP();
//static void MemoryDoAI();

static void MemoryUpdateSPStatus( u32 flags );
static void MemoryUpdateDP( u32 address, u32 value );
static void MemoryUpdateMI( u32 address, u32 value );
static void MemoryUpdateVI( u32 address, u32 value );
static void MemoryUpdatePI( u32 address, u32 value );
static void MemoryUpdatePIF();

static void Memory_InitTables();

//*****************************************************************************
//
//*****************************************************************************
u32 MemoryRegionSizes[NUM_MEM_BUFFERS] =
{
	8*1024,				// Allocate 8k bytes - a bit excessive but some of the internal functions assume it's there!
	MAXIMUM_MEM_SIZE,	// RD_RAM
	0x2000,				// SP_MEM

	0x04,				// SP_PC_REG

	0x800,				// PI_ROM/RAM

	//1*1024*1024,		// RD_REG Don't need this much really?
	0x30,				// RD_REG0
	0x30,				// RD_REG1
	0x30,				// RD_REG2

	0x20,				// SP_REG
	0x20,				// DPC_REG
	0x10,				// DPS_REG
	0x10,				// MI_REG
	0x38,				// VI_REG
	0x18,				// AI_REG
	0x34,				// PI_REG
	0x20,				// RI_REG
	0x1C,				// SI_REG

	0x40000,			// SRAM
};

//*****************************************************************************
//
//*****************************************************************************
u32			gRamSize( MAXIMUM_MEM_SIZE );	// Size of emulated RAM
u32			gTLBReadHit( 0 );
u32			gTLBWriteHit( 0 );

#ifdef DAED_USE_VIRTUAL_ALLOC
static u32	gMemBase( 0 );				// Virtual memory base
#endif

// Ram base, offset by 0x80000000 and 0xa0000000
static u8 * g_pu8RamBase_8000 = NULL;
static u8 * g_pu8RamBase_A000 = NULL;


static bool	Memory_FindTLBEntry( u32 address, u32 & last_matching_idx, u32 * p_idx );
static void Memory_TLBRefillStore( u32 virtual_addr );
static void Memory_TLBInvalidStore( u32 virtual_addr );
static void Memory_TLBRefillLoad( u32 virtual_addr );
static void Memory_TLBInvalidLoad( u32 virtual_addr );


//*****************************************************************************
//
//*****************************************************************************
#include "Memory_Read.inl"
#include "Memory_Write.inl"
#include "Memory_WriteValue.inl"
#include "Memory_ReadInternal.inl"

//*****************************************************************************
//
//*****************************************************************************
struct InternalMemMapEntry
{
	u32 mStartAddr, mEndAddr;
	InternalMemFastFunction InternalReadFastFunction;
};

// Physical ram: 0x80000000 upwards is set up when tables are initialised
InternalMemMapEntry InternalMemMapEntries[] =
{
	{ 0x0000, 0x7FFF, InternalReadMapped },			// Mapped Memory
	{ 0x8000, 0x807F, InternalReadInvalid },		// RDRAM - Initialised later
	{ 0x8080, 0x83FF, InternalReadInvalid },		// Cartridge Domain 2 Address 1
	{ 0x8400, 0x8400, InternalRead_8400_8400 },		// Cartridge Domain 2 Address 1
	{ 0x8404, 0x85FF, InternalReadInvalid },		// Cartridge Domain 2 Address 1
	{ 0x8600, 0x87FF, InternalReadROM },			// Cartridge Domain 1 Address 1
	{ 0x8800, 0x8FFF, InternalReadROM },			// Cartridge Domain 2 Address 2
	{ 0x9000, 0x9FBF, InternalReadROM },			// Cartridge Domain 1 Address 2
	{ 0x9FC0, 0x9FCF, InternalRead_9FC0_9FCF },		// pif RAM/ROM
	{ 0x9FD0, 0x9FFF, InternalReadROM },			// Cartridge Domain 1 Address 3

	{ 0xA000, 0xA07F, InternalReadInvalid },		// Physical RAM - Copy of above
	{ 0xA080, 0xA3FF, InternalReadInvalid },		// Unused
	{ 0xA400, 0xA400, InternalRead_8400_8400 },		// Unused
	{ 0xA404, 0xA4FF, InternalReadInvalid },		// Unused
	{ 0xA500, 0xA5FF, InternalReadROM },			// Cartridge Domain 2 Address 1
	{ 0xA600, 0xA7FF, InternalReadROM },			// Cartridge Domain 1 Address 1
	{ 0xA800, 0xAFFF, InternalReadROM },			// Cartridge Domain 2 Address 2
	{ 0xB000, 0xBFBF, InternalReadROM },			// Cartridge Domain 1 Address 2
	{ 0xBFC0, 0xBFCF, InternalRead_9FC0_9FCF },		// pif RAM/ROM
	{ 0xBFD0, 0xBFFF, InternalReadROM },			// Cartridge Domain 1 Address 3

	{ 0xC000, 0xDFFF, InternalReadMapped },			// Mapped Memory
	{ 0xE000, 0xFFFF, InternalReadMapped },			// Mapped Memory

	{ ~0,  ~0, NULL}
};


//*****************************************************************************
//
//*****************************************************************************
struct MemMapEntry
{
	u32 mStartAddr, mEndAddr;
	MemFastFunction ReadFastFunction;
	MemFastFunction WriteFastFunction;
	MemWriteValueFunction WriteValueFunction;
	unsigned int ReadRegion;
	unsigned int WriteRegion;
};

MemMapEntry MemMapEntries[] =
{
	{ 0x0000, 0x7FFF, ReadMapped,		WriteMapped,	WriteValueMapped,		0,			0 },			// Mapped Memory
	{ 0x8000, 0x807F, Read_Noise,		Write_Noise,	WriteValueNoise,		0,			0 },			// RDRAM - filled in with correct function later
	{ 0x8080, 0x83EF, Read_Noise,		Write_Noise,	WriteValueNoise,		0,			0 },			// Unused - electrical noise
	{ 0x83F0, 0x83F0, Read_83F0_83F0,	Write_83F0_83F0,WriteValue_83F0_83F0,	MEM_RD_REG0,	MEM_RD_REG0 },	// RDRAM reg
	{ 0x83F4, 0x83FF, ReadInvalid,		WriteInvalid,	WriteValueInvalid,		0,			0},				// Unused
	{ 0x8400, 0x8400, Read_8400_8400,	WriteInvalid,	WriteValue_8400_8400,	MEM_SP_MEM, MEM_SP_MEM },	// DMEM/IMEM
	{ 0x8404, 0x8404, Read_8404_8404,	Write_8400_8400,WriteValue_8404_8404,	MEM_SP_REG, 0 },			// SP Reg
	{ 0x8408, 0x8408, Read_8408_8408,	WriteInvalid,	WriteValue_8408_8408,	0,			0 },			// SP PC/IBIST
	{ 0x840C, 0x840F, ReadInvalid,		WriteInvalid,	WriteValueInvalid,		0,			0 },			// Unused
	{ 0x8410, 0x841F, Read_8410_841F,	WriteInvalid,	WriteValue_8410_841F,	0/*MEM_DPC_REG*/, 0 },		// DPC Reg
	{ 0x8420, 0x842F, Read_8420_842F,	WriteInvalid,	WriteValue_8420_842F,	0,			0 },			// DPS Reg
	{ 0x8430, 0x843F, Read_8430_843F,	WriteInvalid,	WriteValue_8430_843F,	MEM_MI_REG, 0 },			// MI Reg
	{ 0x8440, 0x844F, Read_8440_844F,	WriteInvalid,	WriteValue_8440_844F,	0,			0 },			// VI Reg
	{ 0x8450, 0x845F, Read_8450_845F,	WriteInvalid,	WriteValue_8450_845F,	0,			0 },			// AI Reg
	{ 0x8460, 0x846F, Read_8460_846F,	WriteInvalid,	WriteValue_8460_846F,	MEM_PI_REG, 0 },			// PI Reg
	{ 0x8470, 0x847F, Read_8470_847F,	WriteInvalid,	WriteValue_8470_847F,	MEM_RI_REG, MEM_RI_REG },	// RI Reg
	{ 0x8480, 0x848F, Read_8480_848F,	WriteInvalid,	WriteValue_8480_848F,	0,			0 },			// SI Reg
	{ 0x8490, 0x84FF, ReadInvalid,		WriteInvalid,	WriteValueInvalid,		0,			0 },			// Unused
	{ 0x8500, 0x85FF, ReadInvalid/*ReadROM*/,	WriteInvalid, WriteValue_Cartridge, 0,		0 },			// Cartridge Domain 2 Address 1
	{ 0x8600, 0x87FF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 1 Address 1
	{ 0x8800, 0x8FFF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 2 Address 2
	{ 0x9000, 0x9FBF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 1 Address 2
	{ 0x9FC0, 0x9FCF, Read_9FC0_9FCF,	WriteInvalid,	WriteValue_9FC0_9FCF,	0,			0 },			// pif RAM/ROM
	{ 0x9FD0, 0x9FFF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 1 Address 3

	{ 0xA000, 0xA07F, Read_Noise,		Write_Noise,	WriteValueNoise,		0,			0 },			// Physical RAM - Copy of above
	{ 0xA080, 0xA3EF, Read_Noise,		Write_Noise,	WriteValueNoise,		0,			0 },			// Unused
	{ 0xA3F0, 0xA3FF, Read_83F0_83F0,	Write_83F0_83F0,WriteValue_83F0_83F0,	MEM_RD_REG0,	MEM_RD_REG0 },	// RDRAM Reg
	//{ 0xA3F4, 0xA3FF, ReadInvalid,		WriteInvalid },		// Unused
	{ 0xA400, 0xA400, Read_8400_8400,	Write_8400_8400,	WriteValue_8400_8400, MEM_SP_MEM, MEM_SP_MEM },	// DMEM/IMEM
	{ 0xA404, 0xA404, Read_8404_8404,	WriteInvalid,	WriteValue_8404_8404,	MEM_SP_REG, 0 },			// SP Reg
	{ 0xA408, 0xA408, Read_8408_8408,	WriteInvalid,	WriteValue_8408_8408,	0,			0 },			// SP PC/OBOST
	{ 0xA40C, 0xA40F, ReadInvalid,		WriteInvalid,	WriteValueInvalid,		0,			0 },			// Unused
	{ 0xA410, 0xA41F, Read_8410_841F,	WriteInvalid,	WriteValue_8410_841F,	0/*MEM_DPC_REG*/, 0 },		// DPC Reg
	{ 0xA420, 0xA42F, Read_8420_842F,	WriteInvalid,	WriteValue_8420_842F,	0,			0 },			// DPS Reg

	{ 0xA430, 0xA43F, Read_8430_843F,	WriteInvalid,	WriteValue_8430_843F,	MEM_MI_REG, 0 },			// MI Reg
	{ 0xA440, 0xA44F, Read_8440_844F,	WriteInvalid,	WriteValue_8440_844F,	0,			0 },			// VI Reg
	{ 0xA450, 0xA45F, Read_8450_845F,	WriteInvalid,	WriteValue_8450_845F,	0,			0 },			// AI Reg
	{ 0xA460, 0xA46F, Read_8460_846F,	WriteInvalid,	WriteValue_8460_846F,	MEM_PI_REG, 0 },			// PI Reg
	{ 0xA470, 0xA47F, Read_8470_847F,	WriteInvalid,	WriteValue_8470_847F,	MEM_RI_REG, MEM_RI_REG },	// RI Reg
	{ 0xA480, 0xA48F, Read_8480_848F,	WriteInvalid,	WriteValue_8480_848F,	0,			0 },			// SI Reg
	{ 0xA490, 0xA4FF, ReadInvalid,		WriteInvalid,	WriteValueInvalid,		0,			0 },			// Unused
	{ 0xA500, 0xA5FF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 2 Address 1
	{ 0xA600, 0xA7FF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 1 Address 1
	{ 0xA800, 0xAFFF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 2 Address 2
	{ 0xB000, 0xBFBF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 1 Address 2
	{ 0xBFC0, 0xBFCF, Read_9FC0_9FCF,	WriteInvalid,	WriteValue_9FC0_9FCF,	0,			0 },			// pif RAM/ROM
	{ 0xBFD0, 0xBFFF, ReadROM,			WriteInvalid,	WriteValue_Cartridge,	0,			0 },			// Cartridge Domain 1 Address 3

	{ 0xC000, 0xDFFF, ReadMapped,		WriteMapped,	WriteValueMapped,		0,			0 },			// Mapped Memory
	{ 0xE000, 0xFFFF, ReadMapped,		WriteMapped,	WriteValueMapped,		0,			0 },			// Mapped Memory

	{ ~0,  ~0, NULL, NULL, 0,  0 }
};

//*****************************************************************************
//
//*****************************************************************************
#ifndef DAEDALUS_ALIGN_REGISTERS
MemFastFunction g_ReadAddressLookupTable[0x4000];
MemFastFunction g_WriteAddressLookupTable[0x4000];
MemWriteValueFunction g_WriteAddressValueLookupTable[0x4000];
InternalMemFastFunction InternalReadFastTable[0x4000];
void* g_ReadAddressPointerLookupTable[0x4000];
void* g_WriteAddressPointerLookupTable[0x4000];

#else // DAEDALUS_ALIGN_REGISTERS

ALIGNED_GLOBAL(memory_tables_struct_t, memory_tables_struct, PAGE_ALIGN);

#endif // DAEDALUS_ALIGN_REGISTERS


//*****************************************************************************
//
//*****************************************************************************
void * g_pMemoryBuffers[NUM_MEM_BUFFERS] =
{
	NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, NULL,
	NULL, NULL, NULL, NULL,
	NULL, NULL
};

//*****************************************************************************
//
//*****************************************************************************
bool Memory_Init()
{
	gRamSize = MAXIMUM_MEM_SIZE;

#ifdef DAED_USE_VIRTUAL_ALLOC
	void * p_base( VirtualAlloc(0, 512*1024*1024, MEM_RESERVE, PAGE_READWRITE) );

	gMemBase = (u32)p_base;
	if (gMemBase == 0)
	{
		return false;
	}

	g_pMemoryBuffers[ MEM_RD_RAM    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x00000000),	8*1024*1024,MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_SP_MEM    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04000000),	0x2000,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_RD_REG0   ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x03F00000),	0x30,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_RD_REG4   ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x03F04000),	0x30,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_RD_REG8   ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x03F80000),	0x30,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_SP_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04040000),	0x20,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_SP_PC_REG ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04080000),	0x08,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_DPC_REG   ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04100000),	0x20,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_DPS_REG   ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04200000),	0x10,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_MI_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04300000),	0x10,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_VI_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04400000),	0x38,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_AI_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04500000),	0x18,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_PI_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04600000),	0x34,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_RI_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04700000),	0x20,		MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_SI_REG    ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x04800000),	0x1C,		MEM_COMMIT, PAGE_READWRITE );
	//cartDom2= (u8*)VirtualAlloc( (void*)(gMemBase+0x05000000),	0x10000,	MEM_COMMIT, PAGE_READWRITE );
	//cartDom1= (u8*)VirtualAlloc( (void*)(gMemBase+0x06000000),	0x10000,	MEM_COMMIT, PAGE_READWRITE );
	g_pMemoryBuffers[ MEM_SRAM      ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x08000000),	0x40000,	MEM_COMMIT, PAGE_READWRITE );
	//g_pMemoryBuffers[MEM_CARTROM  ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x10000000),	cart_size,	MEM_COMMIT, PAGE_READWRITE);
	g_pMemoryBuffers[ MEM_PIF_RAM   ] = (u8*)VirtualAlloc( (void*)(gMemBase+0x1FC00000),	0x800,		MEM_COMMIT, PAGE_READWRITE );
	//cartDom4= (u8*)VirtualAlloc( (void*)(gMemBase+0x1FD00000),	0x10000,	MEM_COMMIT, PAGE_READWRITE );

	g_pMemoryBuffers[ MEM_UNUSED    ] = new u8[ MemoryRegionSizes[MEM_UNUSED] ];

#else

	for(u32 m = 0; m < NUM_MEM_BUFFERS; m++)
	{
		u32		region_size( MemoryRegionSizes[m] );
		// Skip zero sized areas. An example of this is the cart rom
		if(region_size > 0)
		{
			g_pMemoryBuffers[m] = new u8[region_size];
			//g_pMemoryBuffers[m] = Memory_AllocRegion(region_size);
			
			if(g_pMemoryBuffers[m] == NULL)
			{
				return false;
			}

			// Necessary?
			memset(g_pMemoryBuffers[m], 0, region_size);
		}
	}

#endif

	g_pu8RamBase_8000 = ((u8*)g_pMemoryBuffers[MEM_RD_RAM]) - 0x80000000;
	g_pu8RamBase_A000 = ((u8*)g_pMemoryBuffers[MEM_RD_RAM]) - 0xa0000000;

	Memory_InitTables();

	return true;
}


//*****************************************************************************
//
//*****************************************************************************
void Memory_Fini(void)
{
	DPF(DEBUG_MEMORY, "Freeing Memory");

#ifdef DAED_USE_VIRTUAL_ALLOC

	//
	//	We have to free this buffer separately
	//
	if(g_pMemoryBuffers[MEM_UNUSED])
	{
		delete [] reinterpret_cast< u8 * >( g_pMemoryBuffers[MEM_UNUSED] );
		g_pMemoryBuffers[MEM_UNUSED] = NULL;
	}

	VirtualFree( (void*)gMemBase, 0, MEM_RELEASE );
	gMemBase = 0;

#else
	for(u32 m = 0; m < NUM_MEM_BUFFERS; m++)
	{
		if(g_pMemoryBuffers[m] != NULL)
		{
			delete [] (u8*)(g_pMemoryBuffers[m]);
			g_pMemoryBuffers[m] = NULL;
		}
	}
#endif

	g_pu8RamBase_8000 = NULL;
	g_pu8RamBase_A000 = NULL;

	memset( g_pMemoryBuffers, 0, sizeof( g_pMemoryBuffers ) );
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_Reset( u32 main_mem )
{
	u32 i;
	
	DBGConsole_Msg(0, "Reseting Memory - %d MB", main_mem/(1024*1024));

	s_nNumDmaTransfers = 0;
	s_nTotalDmaTransferSize = 0;
	s_nNumSPTransfers = 0;
	s_nTotalSPTransferSize = 0;

	if(main_mem > MAXIMUM_MEM_SIZE)
	{
		DBGConsole_Msg( 0, "Memory_Reset: Can't reset with more than %dMB ram", MAXIMUM_MEM_SIZE / (1024*1024) );
		main_mem = MAXIMUM_MEM_SIZE;
	}

	// Set memory size to specified value
	// Note that we do not reallocate the memory - we always have 8Mb!
	gRamSize = main_mem;

	// Reinit the tables - this will update the RAM pointers
	Memory_InitTables();

	// Required - virtual alloc gives zeroed memory but this is also used when resetting
	// Clear memory
	for (i = 0; i < NUM_MEM_BUFFERS; i++)
	{
		if ( g_pMemoryBuffers[i] ) 
		{
			memset(g_pMemoryBuffers[i], 0, MemoryRegionSizes[i]);
		}
	}

	g_bSRAMUsed = false;
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_Cleanup()
{
	if (g_bSRAMUsed)
	{
		Memory_SaveSRAM();
	}
}


//*****************************************************************************
//
//*****************************************************************************
void Memory_LoadSRAM()
{
	char szSRAMFileName[MAX_PATH+1];
	FILE * fp;
	u8 b[2048];
	u8 * pDst = (u8*)g_pMemoryBuffers[MEM_SRAM];

	Dump_GetSaveDirectory(szSRAMFileName, g_ROM.szFileName, ".sra");

	DBGConsole_Msg(0, "Loading sram from [C%s]", szSRAMFileName);

	fp = fopen(szSRAMFileName, "rb");
	if (fp != NULL)
	{
		for ( u32 d = 0; d < MemoryRegionSizes[MEM_SRAM]; d += sizeof(b) )
		{
			fread(b, sizeof(b), 1, fp);

			for ( u32 i = 0; i < sizeof(b); i++ )
			{
				pDst[d+i] = b[i^0x3];
			}
		}
		fclose(fp);
	}
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_SaveSRAM()
{
	char szSRAMFileName[MAX_PATH+1];
	FILE * fp;
	u8 b[2048];
	u8 * p_src = (u8*)g_pMemoryBuffers[MEM_SRAM];
	

	Dump_GetSaveDirectory(szSRAMFileName, g_ROM.szFileName, ".sra");

	DBGConsole_Msg(0, "Saving sram to [C%s]", szSRAMFileName);

	fp = fopen(szSRAMFileName, "wb");
	if (fp != NULL)
	{
		for ( u32 d = 0; d < MemoryRegionSizes[MEM_SRAM]; d += sizeof(b) )
		{
			for ( u32 i = 0; i < sizeof(b); i++ )
			{
				b[i^0x3] = p_src[d+i];
			}		
			fwrite(b, 1, sizeof(b), fp);
		}
		fclose(fp);
	}
}

//*****************************************************************************
//
//*****************************************************************************
static void Memory_InitTables_ROM_PointerTable( const void * p_rom_address, u32 rom_size, u32 startPhys, u32 endPhys )
{
	u32 start_addr = startPhys >> 16;
	u32 end_addr = (daedalus::Min(endPhys, startPhys + rom_size) >> 16) - 1;
	u32 i;
	void* pointerLookupTableEntry;

	start_addr += 0x8000;
	end_addr += 0x8000;

	pointerLookupTableEntry = (void*)(reinterpret_cast< u32 >( p_rom_address ) - (start_addr << 16));
	
	for (i = (start_addr>>2); i <= (end_addr>>2); i++)
	{
		g_ReadAddressPointerLookupTable[i] = pointerLookupTableEntry;
	}

	start_addr += 0x2000;
	end_addr += 0x2000;

	pointerLookupTableEntry = (void*)(reinterpret_cast< u32 >( p_rom_address ) - (start_addr << 16));
	
	for (i = (start_addr>>2); i <= (end_addr>>2); i++)
	{
		g_ReadAddressPointerLookupTable[i] = pointerLookupTableEntry;
	}
}

//*****************************************************************************
//
//*****************************************************************************
static void * Memory_GetInvalidPointerTableEntry( int entry )
{
	return (void*)(0xf0000000 - (entry << 18));
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_InitTables()
{
	u32 start_addr;
	u32 end_addr;
	u32 i;

	// 0x00000000 - 0x7FFFFFFF Mapped Memory
	u32 entry = 0;

	memset(g_ReadAddressLookupTable, 0, sizeof(MemFastFunction) * 0x4000);
	memset(g_WriteAddressLookupTable, 0, sizeof(MemFastFunction) * 0x4000);
	memset(g_WriteAddressValueLookupTable, 0, sizeof(MemWriteValueFunction) * 0x4000);
	memset(InternalReadFastTable, 0, sizeof(InternalMemFastFunction) * 0x4000);
	memset(g_ReadAddressPointerLookupTable, 0, sizeof(void*) * 0x4000);
	memset(g_WriteAddressPointerLookupTable, 0, sizeof(void*) * 0x4000);
	
	// these values cause the result of the addition to be negative (>= 0x80000000)
	for(i = 0; i < 0x4000; i++)
	{
		g_WriteAddressPointerLookupTable[i] = g_ReadAddressPointerLookupTable[i] = Memory_GetInvalidPointerTableEntry(i);
	}
	
	while (MemMapEntries[entry].mStartAddr != u32(~0))
	{
		start_addr = MemMapEntries[entry].mStartAddr;
		end_addr = MemMapEntries[entry].mEndAddr;

#ifndef MEMORY_DO_BOUNDS_CHECKING // this kills bounds checking
		void* readPointerLookupTableEntry = MemMapEntries[entry].ReadRegion ? (void*)(reinterpret_cast< u32 >(g_pMemoryBuffers[MemMapEntries[entry].ReadRegion]) - (start_addr << 16)) : 0;
		void* writePointerLookupTableEntry = MemMapEntries[entry].WriteRegion ? (void*)(reinterpret_cast< u32 >(g_pMemoryBuffers[MemMapEntries[entry].WriteRegion]) - (start_addr << 16)) : 0;
#endif

		for (i = (start_addr>>2); i <= (end_addr>>2); i++)
		{
			g_ReadAddressLookupTable[i]  = MemMapEntries[entry].ReadFastFunction;
			g_WriteAddressLookupTable[i] = MemMapEntries[entry].WriteFastFunction;
			g_WriteAddressValueLookupTable[i] = MemMapEntries[entry].WriteValueFunction;

#ifndef MEMORY_DO_BOUNDS_CHECKING // this kills bounds checking
			if(MemMapEntries[entry].ReadRegion)
				g_ReadAddressPointerLookupTable[i] = readPointerLookupTableEntry;

			if(MemMapEntries[entry].WriteRegion)
				g_WriteAddressPointerLookupTable[i] = writePointerLookupTableEntry;
#endif
		}

		entry++;
	}

	entry = 0;
	while (InternalMemMapEntries[entry].mStartAddr != u32(~0))
	{
		start_addr = InternalMemMapEntries[entry].mStartAddr;
		end_addr = InternalMemMapEntries[entry].mEndAddr;

		for (i = (start_addr>>2); i <= (end_addr>>2); i++)
		{
			InternalReadFastTable[i]  = InternalMemMapEntries[entry].InternalReadFastFunction;
		}

		entry++;
	}

	// Check the tables
	/*for (i = 0; i < 0x4000; i++)
	{
		if (g_ReadAddressLookupTable[i] == NULL ||
			g_WriteAddressLookupTable[i] == NULL ||
			g_WriteAddressValueLookupTable[i] == NULL ||
			InternalReadFastTable[i] == NULL)
		{
			CHAR str[300];
			wsprintf(str, "Warning: 0x%08x is null", i<<14);
			MessageBox(NULL, str, g_szDaedalusName, MB_OK);
		}
	}*/


	// Set up RDRAM areas:
	u32 ram_size = gRamSize;
	MemFastFunction pReadRam;
	MemFastFunction pWriteRam;
	MemWriteValueFunction pWriteValueRam;

	InternalMemFastFunction pInternalReadRam;
	
	void* pointerLookupTableEntry;

	if (ram_size == MEMORY_4_MEG)
	{
		DBGConsole_Msg(0, "Initialising 4Mb main memory");
		pReadRam = Read_RAM_4Mb_8000_803F;
		pWriteRam = Write_RAM_4Mb_8000_803F;
		pWriteValueRam = WriteValue_RAM_4Mb_8000_803F;

		pInternalReadRam = InternalRead_4Mb_8000_803F;
	}
	else
	{
		DBGConsole_Msg(0, "Initialising 8Mb main memory");
		pReadRam = Read_RAM_8Mb_8000_807F;
		pWriteRam = Write_RAM_8Mb_8000_807F;
		pWriteValueRam = WriteValue_RAM_8Mb_8000_807F;
		
		pInternalReadRam = InternalRead_8Mb_8000_807F;
	}

	// "Real"
	start_addr = 0x8000;
	end_addr = 0x8000 + ((ram_size>>16)-1);

	pointerLookupTableEntry = (void*)(reinterpret_cast< u32 >( g_pMemoryBuffers[MEM_RD_RAM] ) - (start_addr << 16));

	for (i = (start_addr>>2); i <= (end_addr>>2); i++)
	{
		g_ReadAddressLookupTable[i]  = pReadRam;
		g_WriteAddressLookupTable[i] = pWriteRam;
		g_WriteAddressValueLookupTable[i] = pWriteValueRam;

		InternalReadFastTable[i]  = pInternalReadRam;

		g_ReadAddressPointerLookupTable[i] = pointerLookupTableEntry;
		g_WriteAddressPointerLookupTable[i] = pointerLookupTableEntry;
	}


	// Shadow
	if (ram_size == MEMORY_4_MEG)
	{
		pReadRam = Read_RAM_4Mb_A000_A03F;
		pWriteRam = Write_RAM_4Mb_A000_A03F;
		pWriteValueRam = WriteValue_RAM_4Mb_A000_A03F;

		pInternalReadRam = InternalRead_4Mb_8000_803F;
	}
	else
	{
		pReadRam = Read_RAM_8Mb_A000_A07F;
		pWriteRam = Write_RAM_8Mb_A000_A07F;
		pWriteValueRam = WriteValue_RAM_8Mb_A000_A07F;
		
		pInternalReadRam = InternalRead_8Mb_8000_807F;
	}


	start_addr = 0xA000;
	end_addr = 0xA000 + ((ram_size>>16)-1);

	pointerLookupTableEntry = (void*)(reinterpret_cast< u32 >( g_pMemoryBuffers[MEM_RD_RAM] ) - (start_addr << 16));

	for (i = (start_addr>>2); i <= (end_addr>>2); i++)
	{
		g_ReadAddressLookupTable[i]  = pReadRam;
		g_WriteAddressLookupTable[i] = pWriteRam;
		g_WriteAddressValueLookupTable[i] = pWriteValueRam;

		InternalReadFastTable[i]  = pInternalReadRam;

		g_ReadAddressPointerLookupTable[i] = pointerLookupTableEntry;
		g_WriteAddressPointerLookupTable[i] = pointerLookupTableEntry;
	}
	
	// Map ROM regions here if the address is fixed
	if(RomBuffer::IsRomLoaded() && RomBuffer::IsRomAddressFixed())
	{
		const void *	p_rom_address( RomBuffer::GetFixedRomBaseAddress() );
		u32				rom_size( RomBuffer::GetRomSize() );

		Memory_InitTables_ROM_PointerTable( p_rom_address, rom_size, PI_DOM1_ADDR1, PI_DOM2_ADDR2 );
		Memory_InitTables_ROM_PointerTable( p_rom_address, rom_size, PI_DOM1_ADDR2, 0x1FC00000 );
		Memory_InitTables_ROM_PointerTable( p_rom_address, rom_size, PI_DOM1_ADDR3, 0x20000000 );
	}
}

//*****************************************************************************
//
//*****************************************************************************

/* RDP commands: */
#define	G_NOOP				0xc0	/*   0 */
#define	G_SETCIMG			0xff	/*  -1 */
#define	G_SETZIMG			0xfe	/*  -2 */
#define	G_SETTIMG			0xfd	/*  -3 */
#define	G_SETCOMBINE		0xfc	/*  -4 */
#define	G_SETENVCOLOR		0xfb	/*  -5 */
#define	G_SETPRIMCOLOR		0xfa	/*  -6 */
#define	G_SETBLENDCOLOR		0xf9	/*  -7 */
#define	G_SETFOGCOLOR		0xf8	/*  -8 */
#define	G_SETFILLCOLOR		0xf7	/*  -9 */
#define	G_FILLRECT			0xf6	/* -10 */
#define	G_SETTILE			0xf5	/* -11 */
#define	G_LOADTILE			0xf4	/* -12 */
#define	G_LOADBLOCK			0xf3	/* -13 */
#define	G_SETTILESIZE		0xf2	/* -14 */
#define	G_LOADTLUT			0xf0	/* -16 */
#define	G_RDPSETOTHERMODE	0xef	/* -17 */
#define	G_SETPRIMDEPTH		0xee	/* -18 */
#define	G_SETSCISSOR		0xed	/* -19 */
#define	G_SETCONVERT		0xec	/* -20 */
#define	G_SETKEYR			0xeb	/* -21 */
#define	G_SETKEYGB			0xea	/* -22 */
#define	G_RDPFULLSYNC		0xe9	/* -23 */
#define	G_RDPTILESYNC		0xe8	/* -24 */
#define	G_RDPPIPESYNC		0xe7	/* -25 */
#define	G_RDPLOADSYNC		0xe6	/* -26 */
#define G_TEXRECTFLIP		0xe5	/* -27 */
#define G_TEXRECT			0xe4	/* -28 */

/* 
 * The following commands are the "generated" RDP commands; the user
 * never sees them, the RSP microcode generates them.
 *
 * The layout of the bits is magical, to save work in the ucode.
 * These id's are -56, -52, -54, -50, -55, -51, -53, -49, ...
 *                                 edge, shade, texture, zbuff bits:  estz
 */
#define G_TRI_FILL		0xc8 /* fill triangle:            11001000 */
#define G_TRI_SHADE		0xcc /* shade triangle:           11001100 */
#define G_TRI_TXTR		0xca /* texture triangle:         11001010 */
#define G_TRI_SHADE_TXTR	0xce /* shade, texture triangle:  11001110 */
#define G_TRI_FILL_ZBUFF	0xc9 /* fill, zbuff triangle:     11001001 */
#define G_TRI_SHADE_ZBUFF	0xcd /* shade, zbuff triangle:    11001101 */
#define G_TRI_TXTR_ZBUFF	0xcb /* texture, zbuff triangle:  11001011 */
#define G_TRI_SHADE_TXTR_ZBUFF	0xcf /* shade, txtr, zbuff trngl: 11001111 */

#undef DISPLAY_RDP_COMMANDS


void MemoryDoDP()
{
	u32 dpc_start	= Memory_DPC_GetRegister( DPC_START_REG );
	u32 dpc_current = Memory_DPC_GetRegister( DPC_CURRENT_REG );
	u32 dpc_end		= Memory_DPC_GetRegister( DPC_END_REG );

#ifdef DISPLAY_RDP_COMMANDS
	u32 dpc_status	= Memory_DPC_GetRegister( DPC_STATUS_REG );
	u32 dpc_clock	= Memory_DPC_GetRegister( DPC_CLOCK_REG );
	u32 dpc_bufbusy	= Memory_DPC_GetRegister( DPC_BUFBUSY_REG );
	u32 dpc_pipebusy= Memory_DPC_GetRegister( DPC_PIPEBUSY_REG );
	u32 dpc_tmem	= Memory_DPC_GetRegister( DPC_TMEM_REG );
#endif

	REG64 command;

	while ( dpc_current >= dpc_start && dpc_current < dpc_end )
	{
		command._u64 = Read64Bits( PHYS_TO_K0( dpc_current ) );

#ifdef DISPLAY_RDP_COMMANDS
		const CHAR * desc = "Unk";
		switch ( command._u8[7] )
		{
			case	0x00:						desc = "G_SPNOOP"; break;
			case	G_NOOP:						desc = "G_NOOP"; break;
			case	G_SETCIMG:					desc = "G_SETCIMG"; break;
			case	G_SETZIMG:					desc = "G_SETZIMG"; break;
			case	G_SETTIMG:					desc = "G_SETTIMG"; break;
			case	G_SETCOMBINE:				desc = "G_SETCOMBINE"; break;
			case	G_SETENVCOLOR:				desc = "G_SETENVCOLOR"; break;
			case	G_SETPRIMCOLOR:				desc = "G_SETPRIMCOLOR"; break;
			case	G_SETBLENDCOLOR:			desc = "G_SETBLENDCOLOR"; break;
			case	G_SETFOGCOLOR:				desc = "G_SETFOGCOLOR"; break;
			case	G_SETFILLCOLOR:				desc = "G_SETFILLCOLOR"; break;
			case	G_FILLRECT:					desc = "G_FILLRECT"; break;
			case	G_SETTILE:					desc = "G_SETTILE"; break;
			case	G_LOADTILE:					desc = "G_LOADTILE"; break;
			case	G_LOADBLOCK:				desc = "G_LOADBLOCK"; break;
			case	G_SETTILESIZE:				desc = "G_SETTILESIZE"; break;
			case	G_LOADTLUT:					desc = "G_LOADTLUT"; break;
			case	G_RDPSETOTHERMODE:			desc = "G_RDPSETOTHERMODE"; break;
			case	G_SETPRIMDEPTH:				desc = "G_SETPRIMDEPTH"; break;
			case	G_SETSCISSOR:				desc = "G_SETSCISSOR"; break;
			case	G_SETCONVERT:				desc = "G_SETCONVERT"; break;
			case	G_SETKEYR:					desc = "G_SETKEYR"; break;
			case	G_SETKEYGB:					desc = "G_SETKEYGB"; break;
			case	G_RDPFULLSYNC:				desc = "G_RDPFULLSYNC"; break;
			case	G_RDPTILESYNC:				desc = "G_RDPTILESYNC"; break;
			case	G_RDPPIPESYNC:				desc = "G_RDPPIPESYNC"; break;
			case	G_RDPLOADSYNC:				desc = "G_RDPLOADSYNC"; break;
			case	G_TEXRECTFLIP:				desc = "G_TEXRECTFLIP"; break;
			case	G_TEXRECT:					desc = "G_TEXRECT"; break;
			case	G_TRI_FILL:					desc = "G_TRI_FILL"; break;
			case	G_TRI_SHADE:				desc = "G_TRI_SHADE"; break;
			case	G_TRI_TXTR:					desc = "G_TRI_TXTR"; break;
			case	G_TRI_SHADE_TXTR:			desc = "G_TRI_SHADE_TXTR"; break;
			case	G_TRI_FILL_ZBUFF:			desc = "G_TRI_FILL_ZBUFF"; break;
			case	G_TRI_SHADE_ZBUFF:			desc = "G_TRI_SHADE_ZBUFF"; break;
			case	G_TRI_TXTR_ZBUFF:			desc = "G_TRI_TXTR_ZBUFF"; break;
			case	G_TRI_SHADE_TXTR_ZBUFF:		desc = "G_TRI_SHADE_TXTR_ZBUFF"; break;
			default:							CPU_Halt( "" );
		}
		
		DBGConsole_Msg(0, "DP: S: %08x C: %08x E: %08x Cmd:%08x%08x %s", dpc_start, dpc_current, dpc_end, command._u32[1], command._u32[0], desc );
#endif

		if ( command._u8[7] == G_RDPFULLSYNC )
		{
			DBGConsole_Msg( 0, "FullSync: Executing DP interrupt" );

			Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_DP);
			R4300_Interrupt_UpdateCause3();	
		}

		dpc_current += 8;
	}

/*

	DBGConsole_Msg(0, "DP: S: %08x C: %08x E: %08x", dpc_start, dpc_current, dpc_end );
	DBGConsole_Msg(0, "    Status: %08x Clock: %08x", dpc_status, dpc_clock );
	DBGConsole_Msg(0, "    BufBusy: %08x PipeBusy: %08x Tmem: %08x", dpc_bufbusy, dpc_pipebusy, dpc_tmem );

*/
	Memory_DPC_SetRegister( DPC_CURRENT_REG, dpc_current );
/*
	Memory_DPC_ClrRegisterBits(DPC_STATUS_REG, DPC_STATUS_DMA_BUSY);
	Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_DP);
	R4300_Interrupt_UpdateCause3();	
*/
}

//*****************************************************************************
//
//*****************************************************************************
/*void MemoryDoAI()
{
#ifdef DAEDALUS_LOG
	u32 ai_src_reg  = Memory_AI_GetRegister(AI_DRAM_ADDR_REG) & 0xfffff8;
	u32 ai_len_reg  = Memory_AI_GetRegister(AI_LEN_REG) & 0x3fff8;
	u32 ai_dacrate = Memory_AI_GetRegister(AI_DACRATE_REG) + 1;
	u32 ai_bitrate = Memory_AI_GetRegister(AI_BITRATE_REG) + 1;
	u32 ai_dma_enabled = Memory_AI_GetRegister(AI_CONTROL_REG);

	u32 frequency = (VI_NTSC_CLOCK/ai_dacrate);	// Might be divided by bytes/sample

	if (ai_dma_enabled == false)
	{
		return;
	}

	u16 *p_src = (u16 *)(g_pu8RamBase + ai_src_reg);

	DPF( DEBUG_MEMORY_AI, DSPrintf( "AI: Playing %d bytes of data from 0x%08x", ai_len_reg, ai_src_reg ) );
	DPF( DEBUG_MEMORY_AI, DSPrintf( "    Bitrate: %d. DACRate: 0x%08x, Freq: %d", ai_bitrate, ai_dacrate, frequency ) );
	DPF( DEBUG_MEMORY_AI, DSPrintf( "    DMA: 0x%08x", ai_dma_enabled ) );

	//TODO - Ensure ranges are OK.

	// Set length to 0
	//Memory_AI_SetRegister(AI_LEN_REG, 0);
	//g_dwNextAIInterrupt = (u32)g_qwCPR[0][C0_COUNT] + ((VID_CLOCK*30)*ai_len_reg)/(22050*4);

	//Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_AI);
	//R4300_Interrupt_UpdateCause3();
#endif
}*/

//*****************************************************************************
//
//*****************************************************************************
void MemoryUpdateSPStatus( u32 flags )
{
	u32 sp_status = Memory_SP_GetRegister(SP_STATUS_REG);

#ifdef DEBUG_SP_STATUS_REG
	DBGConsole_Msg( 0, "----------" );
	if (flags & SP_CLR_HALT)				DBGConsole_Msg( 0, "SP: Clearing Halt" );
	if (flags & SP_SET_HALT)				DBGConsole_Msg( 0, "SP: Setting Halt" );
	if (flags & SP_CLR_BROKE)				DBGConsole_Msg( 0, "SP: Clearing Broke" );
	// No SP_SET_BROKE
	if (flags & SP_CLR_INTR)				DBGConsole_Msg( 0, "SP: Clearing Interrupt" );
	if (flags & SP_SET_INTR)				DBGConsole_Msg( 0, "SP: Setting Interrupt" );
	if (flags & SP_CLR_SSTEP)				DBGConsole_Msg( 0, "SP: Clearing Single Step" );
	if (flags & SP_SET_SSTEP)				DBGConsole_Msg( 0, "SP: Setting Single Step" );
	if (flags & SP_CLR_INTR_BREAK)			DBGConsole_Msg( 0, "SP: Clearing Interrupt on break" );
	if (flags & SP_SET_INTR_BREAK)			DBGConsole_Msg( 0, "SP: Setting Interrupt on break" );
	if (flags & SP_CLR_SIG0)				DBGConsole_Msg( 0, "SP: Clearing Sig0 (Yield)" );
	if (flags & SP_SET_SIG0)				DBGConsole_Msg( 0, "SP: Setting Sig0 (Yield)" );
	if (flags & SP_CLR_SIG1)				DBGConsole_Msg( 0, "SP: Clearing Sig1 (Yielded)" );
	if (flags & SP_SET_SIG1)				DBGConsole_Msg( 0, "SP: Setting Sig1 (Yielded)" );
	if (flags & SP_CLR_SIG2)				DBGConsole_Msg( 0, "SP: Clearing Sig2 (TaskDone)" );
	if (flags & SP_SET_SIG2)				DBGConsole_Msg( 0, "SP: Setting Sig2 (TaskDone)" );
	if (flags & SP_CLR_SIG3)				DBGConsole_Msg( 0, "SP: Clearing Sig3" );
	if (flags & SP_SET_SIG3)				DBGConsole_Msg( 0, "SP: Setting Sig3" );
	if (flags & SP_CLR_SIG4)				DBGConsole_Msg( 0, "SP: Clearing Sig4" );
	if (flags & SP_SET_SIG4)				DBGConsole_Msg( 0, "SP: Setting Sig4" );
	if (flags & SP_CLR_SIG5)				DBGConsole_Msg( 0, "SP: Clearing Sig5" );
	if (flags & SP_SET_SIG5)				DBGConsole_Msg( 0, "SP: Setting Sig5" );
	if (flags & SP_CLR_SIG6)				DBGConsole_Msg( 0, "SP: Clearing Sig6" );
	if (flags & SP_SET_SIG6)				DBGConsole_Msg( 0, "SP: Setting Sig6" );
	if (flags & SP_CLR_SIG7)				DBGConsole_Msg( 0, "SP: Clearing Sig7" );
	if (flags & SP_SET_SIG7)				DBGConsole_Msg( 0, "SP: Setting Sig7" );
#endif

	// If !HALT && !BROKE

	bool start_rsp = false;
	bool stop_rsp = false;

	if (flags & SP_CLR_HALT)
	{
		sp_status &= ~SP_STATUS_HALT;
		
		start_rsp = true;
	}

	if (flags & SP_SET_HALT)
	{
		sp_status |= SP_STATUS_HALT;
		
		stop_rsp = true;
	}

	if (flags & SP_CLR_BROKE)				sp_status &= ~SP_STATUS_BROKE;
	// No SP_SET_BROKE
	if (flags & SP_CLR_INTR)				{ Memory_MI_ClrRegisterBits(MI_INTR_REG, MI_INTR_SP); R4300_Interrupt_UpdateCause3(); }
	if (flags & SP_SET_INTR)				{ Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_SP); R4300_Interrupt_UpdateCause3(); }		// Shouldn't ever set this?
	if (flags & SP_CLR_SSTEP)				sp_status &= ~SP_STATUS_SSTEP;
	if (flags & SP_SET_SSTEP)				sp_status |=  SP_STATUS_SSTEP;
	if (flags & SP_CLR_INTR_BREAK)			sp_status &= ~SP_STATUS_INTR_BREAK;
	if (flags & SP_SET_INTR_BREAK)			sp_status |=  SP_STATUS_INTR_BREAK;
	if (flags & SP_CLR_SIG0)				sp_status &= ~SP_STATUS_SIG0;
	if (flags & SP_SET_SIG0)				sp_status |=  SP_STATUS_SIG0;
	if (flags & SP_CLR_SIG1)				sp_status &= ~SP_STATUS_SIG1;
	if (flags & SP_SET_SIG1)				sp_status |=  SP_STATUS_SIG1;
	if (flags & SP_CLR_SIG2)				sp_status &= ~SP_STATUS_SIG2;
	if (flags & SP_SET_SIG2)				sp_status |=  SP_STATUS_SIG2;
	if (flags & SP_CLR_SIG3)				sp_status &= ~SP_STATUS_SIG3;
	if (flags & SP_SET_SIG3)				sp_status |=  SP_STATUS_SIG3;
	if (flags & SP_CLR_SIG4)				sp_status &= ~SP_STATUS_SIG4;
	if (flags & SP_SET_SIG4)				sp_status |=  SP_STATUS_SIG4;
	if (flags & SP_CLR_SIG5)				sp_status &= ~SP_STATUS_SIG5;
	if (flags & SP_SET_SIG5)				sp_status |=  SP_STATUS_SIG5;
	if (flags & SP_CLR_SIG6)				sp_status &= ~SP_STATUS_SIG6;
	if (flags & SP_SET_SIG6)				sp_status |=  SP_STATUS_SIG6;
	if (flags & SP_CLR_SIG7)				sp_status &= ~SP_STATUS_SIG7;
	if (flags & SP_SET_SIG7)				sp_status |=  SP_STATUS_SIG7;

	Memory_SP_SetRegister( SP_STATUS_REG, sp_status );

	//
	// We execute the task here, after we've written to the SP status register.
	//
	if( start_rsp )
	{
		// Check for tasks whenever the RSP is started
		if( !RSP_HLE_ProcessTask() )
		{
			CPU_SelectCore();
		}
	}
	else if ( stop_rsp )
	{
		CPU_SelectCore();
	}
}

//*****************************************************************************
//
//*****************************************************************************
#undef DISPLAY_DPC_WRITES

void MemoryUpdateDP( u32 address, u32 flags )
{
	// Ignore address, as this is only called with DPC_STATUS_REG write
	// DBGConsole_Msg(0, "DP Status: 0x%08x", flags);

	u32 dpc_status  =  Memory_DPC_GetRegister(DPC_STATUS_REG);

	if (flags & DPC_CLR_XBUS_DMEM_DMA)			dpc_status &= ~DPC_STATUS_XBUS_DMEM_DMA;
	if (flags & DPC_SET_XBUS_DMEM_DMA)			dpc_status |= DPC_STATUS_XBUS_DMEM_DMA;
	if (flags & DPC_CLR_FREEZE)					dpc_status &= ~DPC_STATUS_FREEZE;
	//if (flags & DPC_SET_FREEZE)				dpc_status |= DPC_STATUS_FREEZE;	// Thanks Lemmy!
	if (flags & DPC_CLR_FLUSH)					dpc_status &= ~DPC_STATUS_FLUSH;
	if (flags & DPC_SET_FLUSH)					dpc_status |= DPC_STATUS_FLUSH;
	if (flags & DPC_CLR_TMEM_CTR)				Memory_DPC_SetRegister(DPC_TMEM_REG, 0);
	if (flags & DPC_CLR_PIPE_CTR)				Memory_DPC_SetRegister(DPC_PIPEBUSY_REG, 0);
	if (flags & DPC_CLR_CMD_CTR)				Memory_DPC_SetRegister(DPC_BUFBUSY_REG, 0);
	if (flags & DPC_CLR_CLOCK_CTR)				Memory_DPC_SetRegister(DPC_CLOCK_REG, 0);

#ifdef DISPLAY_DPC_WRITES
	if ( flags & DPC_CLR_XBUS_DMEM_DMA )		DBGConsole_Msg( 0, "DPC_CLR_XBUS_DMEM_DMA" );
	if ( flags & DPC_SET_XBUS_DMEM_DMA )		DBGConsole_Msg( 0, "DPC_SET_XBUS_DMEM_DMA" );
	if ( flags & DPC_CLR_FREEZE )				DBGConsole_Msg( 0, "DPC_CLR_FREEZE" );
	if ( flags & DPC_SET_FREEZE )				DBGConsole_Msg( 0, "DPC_SET_FREEZE" );
	if ( flags & DPC_CLR_FLUSH )				DBGConsole_Msg( 0, "DPC_CLR_FLUSH" );
	if ( flags & DPC_SET_FLUSH )				DBGConsole_Msg( 0, "DPC_SET_FLUSH" );
	if ( flags & DPC_CLR_TMEM_CTR )				DBGConsole_Msg( 0, "DPC_CLR_TMEM_CTR" );
	if ( flags & DPC_CLR_PIPE_CTR )				DBGConsole_Msg( 0, "DPC_CLR_PIPE_CTR" );
	if ( flags & DPC_CLR_CMD_CTR )				DBGConsole_Msg( 0, "DPC_CLR_CMD_CTR" );
	if ( flags & DPC_CLR_CLOCK_CTR )			DBGConsole_Msg( 0, "DPC_CLR_CLOCK_CTR" );

	DBGConsole_Msg( 0, "Modified DPC_STATUS_REG - now %08x", dpc_status );
#endif

	// Write back the value
	Memory_DPC_SetRegister(DPC_STATUS_REG, dpc_status);

}

//*****************************************************************************
//
//*****************************************************************************
void MemoryUpdateMI( u32 address, u32 value )
{
	u32 offset;
	u32 mi_mode_reg      = Memory_MI_GetRegister(MI_MODE_REG);
	u32 mi_intr_mask_reg = Memory_MI_GetRegister(MI_INTR_MASK_REG);
	u32 mi_intr_reg		 = Memory_MI_GetRegister(MI_INTR_REG);

	offset = address & 0xFF;

	//bool interrupts_live_before((mi_intr_mask_reg & mi_intr_reg) != 0);

	switch (MI_BASE_REG + offset)
	{

	case MI_INIT_MODE_REG:

		if (value & MI_CLR_INIT) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing Mode Init. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_mode_reg &= ~MI_MODE_INIT;			}
		if (value & MI_SET_INIT) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting Mode Init. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_mode_reg |= MI_MODE_INIT;			}

		if (value & MI_CLR_EBUS) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing EBUS test. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_mode_reg &= ~MI_MODE_EBUS;			}
		if (value & MI_SET_EBUS) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting EBUS test. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_mode_reg |= MI_MODE_EBUS;			}

		if (value & MI_CLR_DP_INTR) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing DP Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_reg &= ~MI_INTR_DP;			}

		if (value & MI_CLR_RDRAM) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing RDRAM reg. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_mode_reg &= ~MI_MODE_RDRAM;			}
		if (value & MI_SET_RDRAM) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting RDRAM reg mode. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_mode_reg |= MI_MODE_RDRAM;			}

		break;
	case MI_INTR_MASK_REG:

		// Do IntrMaskReg writes
		if (value & MI_INTR_MASK_CLR_SP) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing SP Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg &= ~MI_INTR_MASK_SP;					}
		if (value & MI_INTR_MASK_SET_SP) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting SP Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg |= MI_INTR_MASK_SP;					}

		if (value & MI_INTR_MASK_CLR_SI) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing SI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg &= ~MI_INTR_MASK_SI;					}	
		if (value & MI_INTR_MASK_SET_SI) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting SI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg |= MI_INTR_MASK_SI;					}

		if (value & MI_INTR_MASK_CLR_AI) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing AI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg &= ~MI_INTR_MASK_AI;					}
		if (value & MI_INTR_MASK_SET_AI) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting AI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg |= MI_INTR_MASK_AI;					}

		if (value & MI_INTR_MASK_CLR_VI) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing VI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg &= ~MI_INTR_MASK_VI;					}
		if (value & MI_INTR_MASK_SET_VI) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting VI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg |= MI_INTR_MASK_VI;					}

		if (value & MI_INTR_MASK_CLR_PI) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing PI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg &= ~MI_INTR_MASK_PI;					}
		if (value & MI_INTR_MASK_SET_PI) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting PI Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg |= MI_INTR_MASK_PI;					}

		if (value & MI_INTR_MASK_CLR_DP) {		DPF( DEBUG_MI, DSPrintf( "MI: Clearing DP Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg &= ~MI_INTR_MASK_DP;					}
		if (value & MI_INTR_MASK_SET_DP) {		DPF( DEBUG_MI, DSPrintf( "MI: Setting DP Interrupt. PC: 0x%08x", gCPUState.CurrentPC ) );
			mi_intr_mask_reg |= MI_INTR_MASK_DP;					}

		break;
	}

	// Write back
	Memory_MI_SetRegister( MI_MODE_REG, mi_mode_reg );
	Memory_MI_SetRegister( MI_INTR_REG, mi_intr_reg );	
	Memory_MI_SetRegister( MI_INTR_MASK_REG, mi_intr_mask_reg );

	R4300_Interrupt_UpdateCause3();
}

//*****************************************************************************
//
//*****************************************************************************
#ifdef DAEDALUS_LOG
static void DisplayVIControlInfo( u32 control_reg )
{
	DPF( DEBUG_VI, DSPrintf( "VI Control:", (control_reg & VI_CTRL_GAMMA_DITHER_ON) ? "On" : "Off" ) );

	CHAR *szMode = "Disabled/Unknown";
	     if ((control_reg & VI_CTRL_TYPE_16) == VI_CTRL_TYPE_16) szMode = "16-bit";
	else if ((control_reg & VI_CTRL_TYPE_32) == VI_CTRL_TYPE_32) szMode = "32-bit";

	DPF( DEBUG_VI, DSPrintf( "         ColorDepth: %s", szMode ) );
	DPF( DEBUG_VI, DSPrintf( "         Gamma Dither: %s", (control_reg & VI_CTRL_GAMMA_DITHER_ON) ? "On" : "Off" ) );
	DPF( DEBUG_VI, DSPrintf( "         Gamma: %s", (control_reg & VI_CTRL_GAMMA_ON) ? "On" : "Off" ) );
	DPF( DEBUG_VI, DSPrintf( "         Divot: %s", (control_reg & VI_CTRL_DIVOT_ON) ? "On" : "Off" ) );
	DPF( DEBUG_VI, DSPrintf( "         Interlace: %s", (control_reg & VI_CTRL_SERRATE_ON) ? "On" : "Off" ) );
	DPF( DEBUG_VI, DSPrintf( "         AAMask: 0x%02x", (control_reg&VI_CTRL_ANTIALIAS_MASK)>>8 ) );
	DPF( DEBUG_VI, DSPrintf( "         DitherFilter: %s", (control_reg & VI_CTRL_DITHER_FILTER_ON) ? "On" : "Off" ) );

}
#endif


//*****************************************************************************
//
//*****************************************************************************
void MemoryUpdateVI( u32 address, u32 value )
{
	u32 offset;
	
	offset = address & 0xFF;

	switch (VI_BASE_REG + offset)
	{
	case VI_CONTROL_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_CONTROL_REG set to 0x%08x", value ) );
#ifdef DAEDALUS_LOG
		DisplayVIControlInfo(value);
#endif

		if ( gGraphicsPlugin != NULL )
		{
			gGraphicsPlugin->ViStatusChanged();
		}
		

		break;
	case VI_ORIGIN_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_ORIGIN_REG set to 0x%08x (was 0x%08x)", value ) );
		if (Memory_VI_GetRegister(VI_ORIGIN_REG) != value)
		{
			//DBGConsole_Msg(0, "Origin: 0x%08x (%d x %d)", value, g_dwViWidth, g_dwViHeight);
			g_dwNumFrames++;

			DPF( DEBUG_FRAME, "********************************************" );
		}
		break;

	case VI_WIDTH_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_WIDTH_REG set to %d pixels", value ) );
	//	DBGConsole_Msg(DEBUG_VI, "VI_WIDTH_REG set to %d pixels", value);
		//if (g_dwViWidth != value)
		//{
		//	DBGConsole_Msg(DEBUG_VI, "VI_WIDTH_REG set to %d pixels (was %d)", value, g_dwViWidth);
		//	g_dwViWidth = value;
		//	g_dwViHeight = (value * 3) / 4;
		//}

		if ( gGraphicsPlugin != NULL )
		{
			gGraphicsPlugin->ViWidthChanged();
		}


		break;

	case VI_INTR_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_INTR_REG set to 0x%08x", value ) );
		//DBGConsole_Msg(DEBUG_VI, "VI_INTR_REG set to %d", value);
		break;

	case VI_CURRENT_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_CURRENT_REG set to 0x%08x", value ) );

		// Any write clears interrupt line...
		DPF( DEBUG_VI, DSPrintf( "VI: Clearing interrupt flag. PC: 0x%08x", gCPUState.CurrentPC ) );

		Memory_MI_ClrRegisterBits(MI_INTR_REG, MI_INTR_VI);
		R4300_Interrupt_UpdateCause3();
		break;

	case VI_BURST_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_BURST_REG set to 0x%08x", value ) );
	//	DBGConsole_Msg(DEBUG_VI, "VI_BURST_REG set to 0x%08x", value);
		break;

	case VI_V_SYNC_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_V_SYNC_REG set to 0x%08x", value ) );
	//	DBGConsole_Msg(DEBUG_VI, "VI_V_SYNC_REG set to 0x%08x", value);
		break;

	case VI_H_SYNC_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_H_SYNC_REG set to 0x%08x", value ) );
	//	DBGConsole_Msg(DEBUG_VI, "VI_H_SYNC_REG set to 0x%08x", value);
		break;

	case VI_LEAP_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_LEAP_REG set to 0x%08x", value ) );
	//	DBGConsole_Msg(DEBUG_VI, "VI_LEAP_REG set to 0x%08x", value);
		break;

	case VI_H_START_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_H_START_REG set to 0x%08x", value ) );
		//DBGConsole_Msg(DEBUG_VI, "VI_H_START_REG set to 0x%08x", value);
		break;

	case VI_V_START_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_V_START_REG set to 0x%08x", value ) );
		//DBGConsole_Msg(DEBUG_VI, "VI_V_START_REG set to 0x%08x", value);
		break;

	case VI_V_BURST_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_V_BURST_REG set to 0x%08x", value ) );
		//DBGConsole_Msg(DEBUG_VI, "VI_V_BURST_REG set to 0x%08x", value);
		break;

	case VI_X_SCALE_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_X_SCALE_REG set to 0x%08x", value ) );
		{
			//u32 scale = value & 0xFFF;
			//float fScale = (float)scale / (1<<10);
			//DBGConsole_Msg(DEBUG_VI, "VI_X_SCALE_REG set to 0x%08x (%f)", value, 1/fScale);
			//g_dwViWidth = 640 * fScale;
		}
		break;

	case VI_Y_SCALE_REG:
		DPF( DEBUG_VI, DSPrintf( "VI_Y_SCALE_REG set to 0x%08x", value ) );
		{
			/*u32 scale = value & 0xFFF;
			float fScale = (float)scale / (1<<10);
		//	DBGConsole_Msg(DEBUG_VI, "VI_Y_SCALE_REG set to 0x%08x (%f)", value, 1/fScale);

			if (g_ROM.TvType == OS_TV_NTSC)
			{
				g_dwViHeight = 240 * fScale;	// NTSC
			}
			else if (g_ROM.TvType == OS_TV_PAL)
			{
				g_dwViHeight = 288 * fScale;	// PAL
			}
			else
			{
				g_dwViHeight = 240 * fScale;	// NTSC
			}*/

		}
		break;
	}

	*(u32 *)((u8 *)g_pMemoryBuffers[MEM_VI_REG] + offset) = value;
}



//*****************************************************************************
//
//*****************************************************************************
void MemoryUpdatePI( u32 address, u32 value )
{
	u32 offset = address & 0xFF;
	u32 pi_address = (PI_BASE_REG) + offset;

	if (pi_address == PI_STATUS_REG)
	{
		if (value & PI_STATUS_RESET)
		{
			DPF( DEBUG_PI, DSPrintf( "PI: Resetting Status. PC: 0x%08x", gCPUState.CurrentPC ) );
			// Reset PIC here
			Memory_PI_SetRegister(PI_STATUS_REG, 0);
		}
		if (value & PI_STATUS_CLR_INTR)
		{
			DPF( DEBUG_PI, DSPrintf( "PI: Clearing interrupt flag. PC: 0x%08x", gCPUState.CurrentPC ) );
			Memory_MI_ClrRegisterBits(MI_INTR_REG, MI_INTR_PI);
			R4300_Interrupt_UpdateCause3();
		}
	}
	else
	{
		switch (pi_address)
		{
		case PI_BSD_DOM1_LAT_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM1_LAT_REG written to (dom1 device latency) - 0x%08x", value ) );
			break;
		case PI_BSD_DOM1_PWD_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM1_PWD_REG written to (dom1 device R/W strobe pulse width) - 0x%08x", value ) );
			break;
		case PI_BSD_DOM1_PGS_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM1_PGS_REG written to (dom1 device page size) - 0x%08x",	value ) );
			break;
		case PI_BSD_DOM1_RLS_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM1_RLS_REG written to (dom1 device R/W release duration) - 0x%08x",value ) );
			break;
		case PI_BSD_DOM2_LAT_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM2_LAT_REG written to (dom2 device latency) - 0x%08x", value ) );
			break;
		case PI_BSD_DOM2_PWD_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM2_PWD_REG written to (dom2 device R/W strobe pulse width) - 0x%08x", value ) );
			break;
		case PI_BSD_DOM2_PGS_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM2_PGS_REG written to (dom2 device page size) - 0x%08x",	value ) );
			break;
		case PI_BSD_DOM2_RLS_REG:
			DPF( DEBUG_MEMORY_PI, DSPrintf( "PI_BSD_DOM2_RLS_REG written to (dome device R/W release duration) - 0x%08x", value ) );
			break;
		}

		*(u32 *)((u8 *)g_pMemoryBuffers[MEM_PI_REG] + offset) = value;
	}
}


//*****************************************************************************
//	The PIF control byte has been written to - process this command
//*****************************************************************************
void MemoryUpdatePIF()
{
	u8 * pPIFRom = (u8 *)g_pMemoryBuffers[MEM_PIF_RAM];
	u8 * pPIFRam = (u8 *)g_pMemoryBuffers[MEM_PIF_RAM] + 0x7C0;

	u8 command = pPIFRam[ 0x3F^3 ];
			
	switch ( command )
	{
	case 0x01:		// Standard block
		DBGConsole_Msg( 0, "[GStandard execute block control value: 0x%02x", command );
		break;
	case 0x08: 
		pPIFRam[ 0x3F^3 ] = 0x00; 

		Memory_SI_SetRegisterBits(SI_STATUS_REG, SI_STATUS_INTERRUPT);
		Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_SI);
		R4300_Interrupt_UpdateCause3();
		DBGConsole_Msg( 0, "[GSI Interrupt control value: 0x%02x", command );
		break;
	case 0x10:
		memset( pPIFRom, 0, 0x7C0 );
		DBGConsole_Msg( 0, "[GClear ROM control value: 0x%02x", command );
		break;
	case 0x30:
		pPIFRam[ 0x3F^3 ] = 0x80;		
		DBGConsole_Msg( 0, "[GSet 0x80 control value: 0x%02x", command );
		break;
	case 0xC0:
		memset( pPIFRam, 0, 0x40);
		DBGConsole_Msg( 0, "[GClear PIF Ram control value: 0x%02x", command );
		break;
	default:
		DBGConsole_Msg( 0, "[GUnknown control value: 0x%02x", command );
		break;
	}


}

//*****************************************************************************
//	
//*****************************************************************************
bool	Memory_FindTLBEntry( u32 address, u32 & last_matching_idx, u32 * p_idx )
{
	u32 i;

	for ( u32 count = 0; count < 32; count++ )
	{
		// Hack to check most recently reference entry first
		// This gives some speedup if the matched address is near 
		// the end of the tlb array (32 entries)
		i = (count + last_matching_idx) & 0x1F;

		struct TLBEntry & tlb = g_TLBs[i];

		// Check that the VPN numbers match
		if ((address & tlb.vpnmask) == tlb.addrcheck)
		{
			if (!tlb.g)
			{
				if ( (tlb.hi & TLBHI_PIDMASK) !=
					 (g_qwCPR[0][C0_ENTRYHI]._u32[0] & TLBHI_PIDMASK) )
				{
					// Entries ASID must match.
					continue;
				}
			}

			*p_idx = i;
			last_matching_idx = i;

			// touch entry to show it has been referenced recently
			// we don't need to do this if this was refenced previously
			g_TLBs[i].lastaccessed = ++s_nTLBTimeNow;
			return true;
		}
	}

	return false;
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_TLBRefillLoad( u32 virtual_addr )
{	
	//DBGConsole_Msg(0, "[WTLB: Refill] 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC);
	DPF( DEBUG_TLB, DSPrintf( "TLB: Refill - 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC ) );
	//CPU_Halt("TLBRefill");

	R4300_Exception_TLB( virtual_addr, EXCEPTION_TLB_REFILL_LOAD );
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_TLBInvalidLoad( u32 virtual_addr )
{
	//DBGConsole_Msg(0, "[WTLB: Invalid] 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC);
	DPF( DEBUG_TLB, DSPrintf( "TLB: Invalid - 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC ) );
	//CPU_Halt("TLBInvalid");

	R4300_Exception_TLB( virtual_addr, EXCEPTION_TLB_INVALID_LOAD );
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_TLBRefillStore( u32 virtual_addr )
{	
	DBGConsole_Msg(0, "TLB: Refill - 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC);
	DPF( DEBUG_TLB, DSPrintf( "TLB: Refill - 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC ) );
	//CPU_Halt("TLBRefill");

	R4300_Exception_TLB( virtual_addr, EXCEPTION_TLB_REFILL_STORE );
}

//*****************************************************************************
//
//*****************************************************************************
void Memory_TLBInvalidStore( u32 virtual_addr )
{
	DBGConsole_Msg(0, "TLB: Invalid - 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC);
	DPF( DEBUG_TLB, DSPrintf( "TLB: Invalid - 0x%08x at 0x%08x", virtual_addr, gCPUState.CurrentPC ) );
	//CPU_Halt("TLBStore");

	R4300_Exception_TLB( virtual_addr, EXCEPTION_TLB_INVALID_STORE );
}
