/*
Copyright (C) 206,2007 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"

#include <pspkernel.h>
#include <pspdebug.h>
#include <stdlib.h>
#include <stdio.h>

#include <pspctrl.h>
#include <psprtc.h>
#include <psppower.h>

#include "Debug/DBGConsole.h"
#include "Debug/DebugLog.h"
#include "Interface/RomDB.h"
#include "Core/RomSettings.h"
#include "Core/Memory.h"
#include "Core/CPU.h"
#include "Core/PIF.h"
#include "Core/CPU.h"
#include "Input/InputManager.h"
#include "Utility/Profiler.h"
#include "Utility/Synchroniser.h"
#include "DaedIO.h"

#include "Utility/Preferences.h"
#include "Utility/Timer.h"

#include "SysPSP/UI/UIContext.h"
#include "SysPSP/UI/MainMenuScreen.h"
#include "SysPSP/UI/PauseScreen.h"
#include "SysPSP/UI/SplashScreen.h"
#include "PSPGraphics/VideoMemoryManager.h"
#include "PSPGraphics/GraphicsContext.h"
#include "PSPGraphics/DrawText.h"
#include "PSPGraphics/DaedalusGraphics.h"
#include "PSPGraphics/TextureCache.h"

//#define DAEDALUS_KERNEL_MODE

char						gDaedalusExePath[MAX_PATH+1] = "";
static const char * const	gSettingsIniFile = "roms.ini";
static const char * const	gPreferencesIniFile = "preferences.ini";

extern "C" { void _DisableFPUExceptions(); } 

//*************************************************************************************
//
//*************************************************************************************
#define BOOTLOADER_NAME "Daedalus"

#ifdef DAEDALUS_KERNEL_MODE

PSP_MODULE_INFO( BOOTLOADER_NAME, 0x1000, 1, 1 );
PSP_MAIN_THREAD_ATTR( 0 | THREAD_ATTR_VFPU );

#else

PSP_MODULE_INFO( BOOTLOADER_NAME, 0, 1, 1 );
PSP_MAIN_THREAD_ATTR( THREAD_ATTR_USER | THREAD_ATTR_VFPU );

#endif

#ifdef DAEDALUS_KERNEL_MODE
//*************************************************************************************
//
//*************************************************************************************
static void DaedalusExceptionHandler( PspDebugRegBlock * regs )
{
	pspDebugScreenPrintf("\n\n");
	pspDebugDumpException(regs);
}
#endif

#ifdef DAEDALUS_KERNEL_MODE
//*************************************************************************************
// Function that is called from _init in kernelmode before the
// main thread is started in usermode.
//*************************************************************************************
__attribute__ ((constructor))
void loaderInit()
{
    pspKernelSetKernelPC();
	pspDebugScreenInit();
	pspDebugInstallKprintfHandler(NULL);
	pspDebugInstallErrorHandler(DaedalusExceptionHandler);
	pspSdkInstallNoDeviceCheckPatch();
	pspSdkInstallNoPlainModuleCheckPatch();
}
#endif

//*************************************************************************************
//
//*************************************************************************************
static int ExitCallback( int arg1, int arg2, void * common )
{
	sceKernelExitGame();
	return 0;
}

//*************************************************************************************
//
//*************************************************************************************
static int CallbackThread( SceSize args, void * argp )
{
	int cbid;

	cbid = sceKernelCreateCallback( "Exit Callback", ExitCallback, NULL );
	sceKernelRegisterExitCallback( cbid );

	sceKernelSleepThreadCB();

	return 0;
}

//*************************************************************************************
// Sets up the callback thread and returns its thread id
//*************************************************************************************
static int SetupCallbacks()
{
	int thid = 0;

	thid = sceKernelCreateThread( "update_thread", CallbackThread, 0x11, 0xFA0, 0, 0 );
	if(thid >= 0)
	{
		sceKernelStartThread(thid, 0, 0);
	}

	return thid;
}

//*************************************************************************************
//
//*************************************************************************************
static bool InitialiseSynchroniser( bool produce )
{
#ifdef DAEDALUS_ENABLE_SYNCHRONISATION
	CSynchroniser *	p_synch;
	const CHAR *	filename( "synch.bin" );

	if ( produce )
	{
		p_synch = CSynchroniser::CreateProducer( filename );
	}
	else
	{
		p_synch = CSynchroniser::CreateConsumer( filename );
	}

	return p_synch != NULL && p_synch->IsOpen();
#else
	return true;
#endif
}

//*************************************************************************************
//
//*************************************************************************************
static bool	Inititalise()
{
	printf( "Cpu was: %dMHz, Bus: %dMHz\n", scePowerGetCpuClockFrequency(), scePowerGetBusClockFrequency() );
	if (scePowerSetClockFrequency(333, 333, 166) != 0)
	{
		printf( "Could not set CPU to 333MHz\n" );
	}
	printf( "Cpu now: %dMHz, Bus: %dMHz\n", scePowerGetCpuClockFrequency(), scePowerGetBusClockFrequency() );


// Disable for profiling
//	srand(time(0));

	pspDebugScreenInit();

	_DisableFPUExceptions();

	SetupCallbacks();

	InitialiseSynchroniser( false );

    //setup Pad
    sceCtrlSetSamplingCycle(0);
    sceCtrlSetSamplingMode(PSP_CTRL_MODE_ANALOG);
	
	//
	// Read configuration before doing anything major
	//
//	ReadConfiguration();

	//
	// In ReleaseWithLogging mode, open the logfile for writing
	//
	#ifdef DAEDALUS_LOG
	if(!Debug_InitLogging())
		return false;
	#endif

	if (!CDebugConsole::Create())
		return false;

	//
	// Create the console if it's enabled. Don't care about failures
	//
#ifdef DAEDALUS_PUBLIC_RELEASE
	CDebugConsole::Get()->EnableConsole( false );
#else
	CDebugConsole::Get()->EnableConsole( g_DaedalusConfig.ShowDebug ? true : false );
#endif


	DBGConsole_Msg( 0, "ROMDB create" );
	
	//
	// Create and load the RomDB
	//
	if (!CRomDB::Create())
		return false;

	DBGConsole_Msg( 0, "ROMDB init" );

	char romdb_filename[ MAX_PATH + 1 ];
	IO::Path::Combine( romdb_filename, gDaedalusExePath, "rom.db" );
	if ( !CRomDB::Get()->OpenDB( romdb_filename ) )
	{
		// That's ok - it probably didn't exist
	}

	// Init the savegame directory
	strcpy( g_DaedalusConfig.szSaveDir, "SaveGames/" );

	//
	//	Needs initialising before rom preferences are parsed
	//
	DBGConsole_Msg( 0, "InputManager init" );

	if (!CInputManager::Create())
		return false;

	//
	// Create and load the CRomSettingsDB singleton
	//
	if (!CRomSettingsDB::Create())
		return false;

	if (!CRomSettingsDB::Get()->OpenSettingsFile( gSettingsIniFile ))
	{
		// Warn?
	}

	if (!CPreferences::Create())
		return false;

	if (!CPreferences::Get()->OpenPreferencesFile( gPreferencesIniFile ))
	{
		// Warn?
	}


	//
	// Init memory before Console (which tries to display
	// things that are refs to memory)
	//
	DBGConsole_Msg( 0, "Memory init" );

	if (!Memory_Init())
		return false;

//	DisplayDisclaimer();
//	DisplayConfig();

	//
	// Create the Profiler singleton
	//
#ifdef DAEDALUS_ENABLE_PROFILING
	if (!CProfiler::Create())
		return false;
#endif

	DBGConsole_Msg( 0, "Controller init" );

	if (!CController::Create())
		return false;

	if(!CVideoMemoryManager::Create())
		return false;

	if(!CGraphicsContext::Create())
		return false;

	// Reset the cpu state so that things like the RSP halt flag are correctly set up
	CPU_Reset(0xbfc00000);

	return true;
}

//*************************************************************************************
//
//*************************************************************************************
static void Finalise()
{
	//
	// Write current config out to the registry
	//
//	WriteConfiguration();

	//
	// Unload any loaded rom and cleanup
	//
	ROM_Unload();

	//
	// Turn off the debug console
	// 
	CDebugConsole::Get()->EnableConsole( false );

	Memory_Fini();

	//
	// Destroy the singletons
	//
//	CMainWindow::Destroy();
	CController::Destroy();
	CInputManager::Destroy();
#ifdef DAEDALUS_ENABLE_PROFILING
	CProfiler::Destroy();
#endif

	#ifdef DAEDALUS_LOG
	Debug_FinishLogging();
	#endif

	//
	// Destroy the inifile.
	// Will write ini file if changed
	//
	CRomSettingsDB::Destroy();
	CPreferences::Destroy();

	//
	// Destroy / Commit the rom database
	//
	CRomDB::Get()->Commit();
	CRomDB::Destroy();

	CDebugConsole::Destroy();
}

//*************************************************************************************
//
//*************************************************************************************
#ifdef DAEDALUS_KERNEL_MODE
bool	gProfileNextFrame = false;
bool	gWasProfilingFrame = false;
bool	gShowProfilerResults = false;
#endif

#ifdef DAEDALUS_KERNEL_MODE
//*************************************************************************************
//
//*************************************************************************************
static void	DisplayProfilerResults()
{
    SceCtrlData pad;

	static u32 oldButtons = 0;

	if(gShowProfilerResults)
	{
		bool menu_button_pressed( false );

		pspDebugScreenSetTextColor( 0xffffffff );

		CGraphicsContext::Get()->SetDebugScreenTarget( CGraphicsContext::TS_FRONTBUFFER );

		// Remain paused until the Select button is pressed again
		while(!menu_button_pressed)
		{
			sceDisplayWaitVblankStart();
			pspDebugScreenSetXY(0, 0);
			pspDebugScreenPrintf( "                                 Paused\n" );
			pspDebugScreenPrintf( "                                 ------\n" );
			pspDebugScreenPrintf( "                   Press Select to resume emulation\n" );
			pspDebugScreenPrintf( "\n\n" );

			pspDebugProfilerPrint();
			
			pspDebugScreenPrintf( "\n" );

			sceCtrlPeekBufferPositive(&pad, 1);
			if(oldButtons != pad.Buttons)
			{
				if(pad.Buttons & PSP_CTRL_SELECT)
				{
					menu_button_pressed = true;
				}
			}

			oldButtons = pad.Buttons;
		}
	}

	CGraphicsContext::Get()->SetDebugScreenTarget( CGraphicsContext::TS_BACKBUFFER );

}
#endif


#ifdef _DEBUG
//*************************************************************************************
//
//*************************************************************************************
static void	DumpDynarecStats( float elapsed_time )
{
	// Temp dynarec stats
	extern u64 gTotalInstructionsEmulated;
	extern u64 gTotalInstructionsExecuted;
	extern u32 gTotalRegistersCached;
	extern u32 gTotalRegistersUncached;
	extern u32 gFragmentLookupSuccess;
	extern u32 gFragmentLookupFailure;

	u32		dynarec_ratio( 0 );

	if(gTotalInstructionsExecuted + gTotalInstructionsEmulated > 0)
	{
		float fRatio = float(gTotalInstructionsExecuted * 100.0f / float(gTotalInstructionsEmulated+gTotalInstructionsExecuted));

		dynarec_ratio = u32( fRatio );

		//gTotalInstructionsExecuted = 0;
		//gTotalInstructionsEmulated = 0;
	}

	u32		cached_regs_ratio( 0 );
	if(gTotalRegistersCached + gTotalRegistersUncached > 0)
	{
		float fRatio = float(gTotalRegistersCached * 100.0f / float(gTotalRegistersCached+gTotalRegistersUncached));

		cached_regs_ratio = u32( fRatio );
	}

	const char * const TERMINAL_SAVE_CURSOR			= "\033[s";
	const char * const TERMINAL_RESTORE_CURSOR		= "\033[u";
//	const char * const TERMINAL_TOP_LEFT			= "\033[2A\033[2K";
	const char * const TERMINAL_TOP_LEFT			= "\033[H\033[2K";

	printf( TERMINAL_SAVE_CURSOR );
	printf( TERMINAL_TOP_LEFT );

	printf( "Frame: %dms, DynaRec %d%%, Regs cached %d%%, Lookup success %d/%d", u32(elapsed_time * 1000.0f), dynarec_ratio, cached_regs_ratio, gFragmentLookupSuccess, gFragmentLookupFailure );

	printf( TERMINAL_RESTORE_CURSOR );
	fflush( stdout );

	gFragmentLookupSuccess = 0;
	gFragmentLookupFailure = 0;
}
#endif

#ifdef DAEDALUS_DEBUG_DISPLAYLIST
extern bool gDebugDisplayList;
#include "PSPGraphics\DLParser.h"
#endif

//*************************************************************************************
//
//*************************************************************************************
static CTimer		gTimer;

void HandleEndOfFrame()
{
#ifdef DAEDALUS_DEBUG_DISPLAYLIST
	if(gDebugDisplayList)
		return;
#endif

	DPF( DEBUG_FRAME, "********************************************" );

	bool		activate_pause_menu( false );

#ifdef DAEDALUS_KERNEL_MODE
	//
	//	Update the profiling stats
	//
	if(gWasProfilingFrame)
	{
		// Show results
		pspDebugProfilerDisable();
		gShowProfilerResults = true;
		gWasProfilingFrame = false;

		// Drop back into the menu
		activate_pause_menu = true;
	}
#endif

	//
	//	Figure out how long the last frame took
	//
	float		elapsed_time( gTimer.GetElapsedSeconds() );
	float		framerate( 0.0f );
	if(elapsed_time > 0)
	{
		framerate = 1.0f / elapsed_time;
	}

	//DumpDynarecStats( elapsed_time );

	//
	//	Enter the debug menu as soon as select is newly pressed
	//
    SceCtrlData pad;

	static u32 oldButtons = 0;

	sceCtrlPeekBufferPositive(&pad, 1);
	if(oldButtons != pad.Buttons)
	{
		if(pad.Buttons & PSP_CTRL_SELECT)
		{
			activate_pause_menu = true;
		}
	}
	oldButtons = pad.Buttons;

	if(activate_pause_menu)
	{
		// See how much texture memory we're using
		//CTextureCache::Get()->DropTextures();
		//CVideoMemoryManager::Get()->DisplayDebugInfo();

		CController::Get()->FlushSaveGames();

		// Call this initially, to tidy up any state set by the emulator
		CGraphicsContext::Get()->ClearAllSurfaces();

		CDrawText::Initialise();

		CUIContext *	p_context( CUIContext::Create() );

		if(p_context != NULL)
		{
			p_context->SetBackgroundColour( c32( 94, 188, 94 ) );		// Nice green :)

			CPauseScreen *	pause( CPauseScreen::Create( p_context ) );
			pause->Run();
			delete pause;
			delete p_context;
		}

		CDrawText::Destroy();

		//
		// Commit the preferences database before starting to run
		//
		CPreferences::Get()->Commit();
	}

	//
	//	Reset the elapsed time to avoid glitches when we restart
	//
	gTimer.Reset();

	//
	//	Check whether to profile the next frame
	//
#ifdef DAEDALUS_KERNEL_MODE
	gShowProfilerResults = false;
	if(gProfileNextFrame)
	{
		gProfileNextFrame = false;
		gWasProfilingFrame = true;

		pspDebugProfilerClear();
		pspDebugProfilerEnable();
	}
#endif
}

//*************************************************************************************
//
//*************************************************************************************
static void DisplayRomsAndChoose()
{
	CDrawText::Initialise();

	CUIContext *	p_context( CUIContext::Create() );

	if(p_context != NULL)
	{
		//const c32		BACKGROUND_COLOUR = c32( 107, 188, 255 );		// blue
		const c32		BACKGROUND_COLOUR = c32( 92, 162, 219 );		// blue
		//const c32		BACKGROUND_COLOUR = c32( 94, 188, 94 );			// green

		p_context->SetBackgroundColour( BACKGROUND_COLOUR );

		CSplashScreen *		p_splash( CSplashScreen::Create( p_context ) );
		p_splash->Run();
		delete p_splash;

		CMainMenuScreen *	p_main_menu( CMainMenuScreen::Create( p_context ) );
		p_main_menu->Run();
		delete p_main_menu;
	}

	delete p_context;

	CDrawText::Destroy();
}

//*************************************************************************************
//
//*************************************************************************************
extern "C"
{
int main(void)
{
	if( Inititalise() )
	{
		for(;;)
		{
			DisplayRomsAndChoose();

			//
			// Commit the preferences and roms databases before starting to run
			//
			CRomDB::Get()->Commit();
			CPreferences::Get()->Commit();

			CPU_Run();
		}

		Finalise();
	}

	sceKernelExitGame();
	return 0;
}
}
