/*
Copyright (C) 2006 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "DrawText.h"

#include "NativeTexture.h"

#include "Vector2.h"
#include "Vector3.h"
#include <pspgu.h>
#include <pspgum.h>
#include <pspdebug.h>

#define GL_TRUE                           1
#define GL_FALSE                          0

static CNativeTexture *		gpTextTexture = NULL;
static const char * const	TEXT_TEXTURE_NAME = "Resources/bitlow.png";

static u32					CHARACTER_HEIGHT_I = 12;
static float				CHARACTER_WIDTH = 16;


struct SMetric
{
	s16		PixelOffset;
	s16		Advance;
	u16		Width;
	s16		PostAdvance;
};

static SMetric *		gpFontMetrics = NULL;
static const char * METRICS_NAME = "Resources/bitlow.dat";

struct TextVtx
{
    v2		t0;
    c32		color;
    v3		pos;
};

DAEDALUS_STATIC_ASSERT( sizeof(TextVtx) == 24 );

#define TEXT_VERTEX_FLAGS (GU_TEXTURE_32BITF|GU_COLOR_8888|GU_VERTEX_32BITF )


//*************************************************************************************
//
//*************************************************************************************
void	CDrawText::Initialise()
{
	gpTextTexture = CNativeTexture::CreateFromPng( TEXT_TEXTURE_NAME, TexFmt_4444 );

	gpFontMetrics = new SMetric[256];
	//FILE * fh( fopen( METRICS_NAME, "rb" ) );
	//if(fh != NULL)
	//{
	//	fread( gpFontMetrics, sizeof(SMetric) * 256, 1, fh );
	//	fclose( fh );
	//}
	//else
	{
		for(u32 i = 0; i < 256; ++i)
		{
			gpFontMetrics[ i ].PixelOffset = (i % 16) * 16    + 3;
			gpFontMetrics[ i ].Advance = 0;
			gpFontMetrics[ i ].Width = 7; //16;
			gpFontMetrics[ i ].PostAdvance = 0;
		}

		for(u32 i = 'A'; i <= 'Z'; ++i)
		{
			gpFontMetrics[ i ].PixelOffset = (i % 16) * 16    + 3;
			gpFontMetrics[ i ].Advance = 0;
			gpFontMetrics[ i ].Width = 6; //16;
			gpFontMetrics[ i ].PostAdvance = 1;
		}
		gpFontMetrics[ 'M' ].Width = 7;
		gpFontMetrics[ 'V' ].Width = 7;
		gpFontMetrics[ 'W' ].Width = 7;
		gpFontMetrics[ 'Y' ].Width = 7;

		for(u32 i = 'a'; i <= 'z'; ++i)
		{
			gpFontMetrics[ i ].PixelOffset = (i % 16) * 16    + 3;
			gpFontMetrics[ i ].Advance = 0;
			gpFontMetrics[ i ].Width = 5; //16;
			gpFontMetrics[ i ].PostAdvance = 1;
		}
		gpFontMetrics[ 'i' ].Width = 3;
		gpFontMetrics[ 'j' ].Width = 3;
		gpFontMetrics[ 'l' ].Width = 3;
	}
}

//*************************************************************************************
//
//*************************************************************************************
void	CDrawText::Destroy()
{
	SAFE_RELEASE( gpTextTexture );
	SAFE_DELETEARRAY( gpFontMetrics );
}

//*************************************************************************************
//
//*************************************************************************************
void	CDrawText::Render( s32 x, s32 y, const char * p_str, u32 length, c32 colour )
{
	u32		num_chars( length );
	u32		num_verts( num_chars * 2 );

	TextVtx*	p_verts = (TextVtx*)sceGuGetMemory(num_verts*sizeof(TextVtx));

	sceGuDisable(GU_DEPTH_TEST);
	sceGuDepthMask( GL_TRUE );	// GL_TRUE to disable z-writes
	sceGuShadeModel( GU_FLAT );

	sceGuTexFilter(GU_NEAREST,GU_NEAREST);
	sceGuTexScale(1.0f,1.0f);
	sceGuTexOffset(0.0f,0.0f);

	if(gpTextTexture != NULL)
	{
		gpTextTexture->InstallTexture();
	}

	//sceGuEnable(GU_ALPHA_TEST);
	sceGuDisable(GU_ALPHA_TEST);
	sceGuBlendFunc(GU_ADD, GU_SRC_ALPHA, GU_ONE_MINUS_SRC_ALPHA, 0, 0);
	sceGuEnable(GU_BLEND);
	//sceGuDisable(GU_BLEND);
	sceGuTexFunc(GU_TFX_MODULATE,GU_TCC_RGBA);
	//sceGuTexFunc(GU_TFX_REPLACE,GU_TCC_RGBA);



	sceGumMatrixMode(GU_PROJECTION);
	sceGumLoadIdentity();
	//sceGumOrtho(0, 480 +1, 272 +1, 0, 0.0f, -1.0f);

	s32		cur_x( x );
	s32		cur_y( y );

	for(u32 i = 0; i < num_chars; ++i)
	{
		u8		c( p_str[ i ] );

		s32		glyph_x( c % 16 );
		s32		glyph_y( c / 16 );

		use( glyph_x );

		s32		tex_left_edge = gpFontMetrics[ c ].PixelOffset;
		u32		glyph_width = gpFontMetrics[ c ].Width + 1;
		s32		tex_right_edge = tex_left_edge + glyph_width;
        
		float	tex_scale = CHARACTER_WIDTH;


		v2		tex_uv0( (float)tex_left_edge, (float)(glyph_y + 0) * tex_scale );
		v2		tex_uv1( (float)tex_right_edge, (float)(glyph_y + 1) * tex_scale );

		cur_x += gpFontMetrics[ c ].Advance;

		p_verts[i*2+0].pos = v3( (float)cur_x, (float)cur_y, 0.0f );
		p_verts[i*2+0].color = colour;
		p_verts[i*2+0].t0 = v2( tex_uv0.x, tex_uv0.y );

		p_verts[i*2+1].pos = v3( (float)(cur_x + glyph_width), (float)(cur_y + CHARACTER_WIDTH), 0.0f );
		p_verts[i*2+1].color = colour;
		p_verts[i*2+1].t0 = v2( tex_uv1.x, tex_uv1.y );

		cur_x += glyph_width + s32(gpFontMetrics[ c ].PostAdvance);
	}

	sceGumDrawArray(GU_SPRITES,TEXT_VERTEX_FLAGS|GU_TRANSFORM_2D,num_verts,NULL,p_verts);
}

//*************************************************************************************
//
//*************************************************************************************
s32		CDrawText::GetTextWidth( const char * p_str, u32 length )
{
	u32		num_chars( length );

	s32		cur_width( 0 );

	for(u32 i = 0; i < num_chars; ++i)
	{
		u8		c( p_str[ i ] );

		s32		glyph_width = gpFontMetrics[ c ].Width + 1;

		cur_width += gpFontMetrics[ c ].Advance + glyph_width + s32(gpFontMetrics[ c ].PostAdvance);
	}

	return cur_width;
}

//*************************************************************************************
//
//*************************************************************************************
s32		CDrawText::GetFontHeight()
{
	return CHARACTER_HEIGHT_I;
}

//*************************************************************************************
//
//*************************************************************************************
namespace DrawTextUtilities
{
	const c32	TextWhite			= c32( 238, 238, 238 );
	const c32	TextWhiteDisabled	= c32( 208, 208, 208 );
	const c32	TextBlue			= c32(  80,  80, 208 );
	const c32	TextBlueDisabled	= c32(  80,  80, 178 );
	const c32	TextRed				= c32( 255, 0, 0 );
	const c32	TextRedDisabled		= c32( 208, 208, 208 );

	static c32 COLOUR_SHADOW_HEAVY = c32( 0x80000000 );
	static c32 COLOUR_SHADOW_LIGHT = c32( 0x50000000 );

	void	DrawShadowText( s32 x, s32 y, const char * p_str, u32 length, c32 colour )
	{
		//if(gDoNicerShadow)
		{
			CDrawText::Render( x+1, y+0, p_str, length, COLOUR_SHADOW_LIGHT );
			CDrawText::Render( x+0, y+1, p_str, length, COLOUR_SHADOW_LIGHT );
			CDrawText::Render( x+2, y+2, p_str, length, COLOUR_SHADOW_LIGHT );
		}
		CDrawText::Render( x+1, y+1, p_str, length, COLOUR_SHADOW_HEAVY );
		CDrawText::Render( x, y, p_str, length, colour );

	}

	s32		AlignText( s32 min_x, s32 max_x, const char * p_str, u32 length, EAlignType align_type )
	{
		s32		x;

		switch( align_type )
		{
		case AT_LEFT:
			x = min_x;
			break;

		case AT_CENTRE:
			x = min_x + ((max_x - min_x) - CDrawText::GetTextWidth( p_str, length )) / 2;
			break;
	
		case AT_RIGHT:
			x = max_x - CDrawText::GetTextWidth( p_str, length );
			break;

		default:
			DAEDALUS_ERROR( "Unhandled alignment type" );
			x = min_x;
			break;
		}

		return x;
	}

	const char *	FindPreviousSpace( const char * p_str_start, const char * p_str_end )
	{
		while( p_str_end > p_str_start )
		{
			if( *p_str_end == ' ' )
			{
				return p_str_end;
			}
			p_str_end--;
		}

		// Not found
		return NULL;
	}

	void	WrapText( s32 width, const char * p_str, u32 length, std::vector<u32> & lengths )
	{
		lengths.clear();

		const char *	p_line_str( p_str );
		const char *	p_str_end( p_str + length );

		while( p_line_str < p_str_end )
		{
			u32		length_remaining( p_str_end - p_line_str );
			s32		chunk_width( CDrawText::GetTextWidth( p_line_str, length_remaining ) );

			if( chunk_width <= width )
			{
				lengths.push_back( length_remaining );
				p_line_str += length_remaining;
			}
			else
			{
				// Search backwards until we find a break
				const char *	p_chunk_end( p_str_end );
				bool			found_chunk( false );
				while( p_chunk_end > p_line_str )
				{
					const char * p_space( FindPreviousSpace( p_line_str, p_chunk_end ) );

					if( p_space != NULL )
					{
						u32		chunk_length( p_space + 1 - p_line_str );
						chunk_width = CDrawText::GetTextWidth( p_line_str, chunk_length );
						if( chunk_width <= width )
						{
							lengths.push_back( chunk_length );
							p_line_str += chunk_length;
							found_chunk = true;
							break;
						}
						else
						{
							// Need to try again with the previous space
							p_chunk_end = p_space - 1;
						}
					}
					else
					{
						// No more spaces - just render the whole chunk
						lengths.push_back( p_chunk_end - p_line_str );
						p_line_str = p_chunk_end;
						found_chunk = true;
						break;
					}
				}

				DAEDALUS_ASSERT( found_chunk, "Didn't find chunk while splitting string for rendering?" );
			}
		}
	}

}
