/*
Copyright (C) 2007 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#if _MSC_VER > 1000
#pragma once
#endif

#ifndef BLENDCONSTANT_H_
#define BLENDCONSTANT_H_

#include "PSPGraphics/PSPColour.h"

enum EBlendConstant
{
	BC_SHADE,
	BC_PRIMITIVE,
	BC_ENVIRONMENT,
	BC_PRIMITIVE_ALPHA,
	BC_ENVIRONMENT_ALPHA,
	BC_1,
	BC_0,
};

class CBlendConstantExpression
{
public:
	virtual ~CBlendConstantExpression() {}
	virtual c32					Evaluate( c32 shade, c32 primitive, c32 environment ) const = 0;
	virtual c32					EvaluateConstant( c32 primitive, c32 environment ) const = 0;
	virtual std::string			ToString() const = 0;
};

class CBlendConstantExpressionValue : public CBlendConstantExpression
{
public:
	CBlendConstantExpressionValue( EBlendConstant constant )
		:	mConstant( constant )
	{
	}

	c32			Evaluate( c32 shade, c32 primitive, c32 environment ) const
	{
		switch( mConstant )
		{
		case BC_SHADE:				return shade;
		case BC_PRIMITIVE:			return primitive;
		case BC_ENVIRONMENT:		return environment;
		case BC_PRIMITIVE_ALPHA:	return c32( primitive.GetA(), primitive.GetA(), primitive.GetA(), primitive.GetA() );
		case BC_ENVIRONMENT_ALPHA:	return c32( environment.GetA(), environment.GetA(), environment.GetA(), environment.GetA() );
		case BC_1:					return c32( 0xffffffff );
		case BC_0:					return c32( 0x00000000 );
		}

		DAEDALUS_ERROR( "Unhandled constant" );
		return c32( 0xffffffff );
	}

	c32			EvaluateConstant( c32 primitive, c32 environment ) const
	{
		switch( mConstant )
		{
		case BC_SHADE:				DAEDALUS_ERROR( "Shouldn't be here" ); return c32( 0xffffffff );
		case BC_PRIMITIVE:			return primitive;
		case BC_ENVIRONMENT:		return environment;
		case BC_PRIMITIVE_ALPHA:	return c32( primitive.GetA(), primitive.GetA(), primitive.GetA(), primitive.GetA() );
		case BC_ENVIRONMENT_ALPHA:	return c32( environment.GetA(), environment.GetA(), environment.GetA(), environment.GetA() );
		case BC_1:					return c32( 0xffffffff );
		case BC_0:					return c32( 0x00000000 );
		}

		DAEDALUS_ERROR( "Unhandled constant" );
		return c32( 0xffffffff );
	}

	virtual std::string			ToString() const
	{
		switch( mConstant )
		{
		case BC_SHADE:				return "Shade";
		case BC_PRIMITIVE:			return "Primitive";
		case BC_ENVIRONMENT:		return "Environment";
		case BC_PRIMITIVE_ALPHA:	return "PrimitiveAlpha";
		case BC_ENVIRONMENT_ALPHA:	return "EnvironmentAlpha";
		case BC_1:					return "1";
		case BC_0:					return "0";
		}	

		DAEDALUS_ERROR( "Unhandled constant" );
		return "?";
	}


private:
	EBlendConstant					mConstant;
};

class CBlendConstantExpressionAdd : public CBlendConstantExpression
{
public:
	CBlendConstantExpressionAdd( const CBlendConstantExpression * a, const CBlendConstantExpression * b )
		:	mA( a )
		,	mB( b )
	{
	}

	virtual ~CBlendConstantExpressionAdd()
	{
		delete mA;
		delete mB;
	}

	c32			Evaluate( c32 shade, c32 primitive, c32 environment ) const
	{
		c32		a( mA->Evaluate( shade, primitive, environment ) );
		c32		b( mB->Evaluate( shade, primitive, environment ) );

		return a.Add( b );
	}

	c32			EvaluateConstant( c32 primitive, c32 environment ) const
	{
		c32		a( mA->EvaluateConstant( primitive, environment ) );
		c32		b( mB->EvaluateConstant( primitive, environment ) );

		return a.Add( b );
	}

	virtual std::string			ToString() const
	{
		std::string	str;
		str += "(" + mA->ToString() + " + " + mB->ToString() + ")";
		return str;
	}

private:
	const CBlendConstantExpression *			mA;
	const CBlendConstantExpression *			mB;
};

class CBlendConstantExpressionSub : public CBlendConstantExpression
{
public:
	CBlendConstantExpressionSub( const CBlendConstantExpression * a, const CBlendConstantExpression * b )
		:	mA( a )
		,	mB( b )
	{
	}

	virtual ~CBlendConstantExpressionSub()
	{
		delete mA;
		delete mB;
	}

	c32			Evaluate( c32 shade, c32 primitive, c32 environment ) const
	{
		c32		a( mA->Evaluate( shade, primitive, environment ) );
		c32		b( mB->Evaluate( shade, primitive, environment ) );

		return a.Sub( b );
	}

	c32			EvaluateConstant( c32 primitive, c32 environment ) const
	{
		c32		a( mA->EvaluateConstant( primitive, environment ) );
		c32		b( mB->EvaluateConstant( primitive, environment ) );

		return a.Sub( b );
	}

	virtual std::string			ToString() const
	{
		std::string	str;
		str += "(" + mA->ToString() + " - " + mB->ToString() + ")";
		return str;
	}

private:
	const CBlendConstantExpression *			mA;
	const CBlendConstantExpression *			mB;
};

class CBlendConstantExpressionMul : public CBlendConstantExpression
{
public:
	CBlendConstantExpressionMul( const CBlendConstantExpression * a, const CBlendConstantExpression * b )
		:	mA( a )
		,	mB( b )
	{
	}

	virtual ~CBlendConstantExpressionMul()
	{
		delete mA;
		delete mB;
	}

	c32			Evaluate( c32 shade, c32 primitive, c32 environment ) const
	{
		c32		a( mA->Evaluate( shade, primitive, environment ) );
		c32		b( mB->Evaluate( shade, primitive, environment ) );

		return a.Modulate( b );
	}

	c32			EvaluateConstant( c32 primitive, c32 environment ) const
	{
		c32		a( mA->EvaluateConstant( primitive, environment ) );
		c32		b( mB->EvaluateConstant( primitive, environment ) );

		return a.Modulate( b );
	}


	virtual std::string			ToString() const
	{
		std::string	str;
		str += "(" + mA->ToString() + " * " + mB->ToString() + ")";
		return str;
	}

private:
	const CBlendConstantExpression *			mA;
	const CBlendConstantExpression *			mB;
};

class CBlendConstantExpressionBlend : public CBlendConstantExpression
{
public:
	CBlendConstantExpressionBlend( const CBlendConstantExpression * a, const CBlendConstantExpression * b, const CBlendConstantExpression * f )
		:	mA( a )
		,	mB( b )
		,	mF( f )
	{
	}

	virtual ~CBlendConstantExpressionBlend()
	{
		delete mA;
		delete mB;
		delete mF;
	}

	c32			Evaluate( c32 shade, c32 primitive, c32 environment ) const
	{
		c32		a( mA->Evaluate( shade, primitive, environment ) );
		c32		b( mB->Evaluate( shade, primitive, environment ) );
		c32		f( mB->Evaluate( shade, primitive, environment ) );

		return a.Interpolate( b, f );
	}

	c32			EvaluateConstant( c32 primitive, c32 environment ) const
	{
		c32		a( mA->EvaluateConstant( primitive, environment ) );
		c32		b( mB->EvaluateConstant( primitive, environment ) );
		c32		f( mB->EvaluateConstant( primitive, environment ) );

		return a.Interpolate( b, f );
	}

	virtual std::string			ToString() const
	{
		std::string	str;
		str += "blend(" + mA->ToString() + ", " + mB->ToString() + ", " + mF->ToString() + ")";
		return str;
	}

private:
	const CBlendConstantExpression *			mA;
	const CBlendConstantExpression *			mB;
	const CBlendConstantExpression *			mF;
};

#endif // BLENDCONSTANT_H_
