/*
Copyright (C) 2001 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "ROM.h"
#include "CPU.h"
#include "RSP.h"
#include "PIF.h"		// CController
#include "R4300.h"
#include "ROMBuffer.h"
#include "ROMImage.h"
#include "RomSettings.h"
#include "Compatibility.h"

#include "Daedalus_CRC.h"
#include "DaedStream.h"
#include "DaedIO.h"

#include "Interface/MainWindow.h"
#include "Interface/RomDB.h"

#include "Debug/DBGConsole.h"
#include "Debug/DebugLog.h"
#include "OSHLE/Patch.h"			// Patch_ApplyPatches
#include "OSHLE/ultra_r4300.h"
#include "OSHLE/ultra_os.h"		// System type

#include "Utility/ROMFile.h"
#include "Utility/Preferences.h"
#include "Utility/FramerateLimiter.h"

#include "Plugins/AudioPlugin.h"
#include "Plugins/GraphicsPlugin.h"


#undef min

//*****************************************************************************
// This isn't really the most appropriate place. Need to check with
// the graphics plugin really
//*****************************************************************************
u32 g_dwNumFrames = 0;

//*****************************************************************************
//
//*****************************************************************************
static void ROM_GetRomNameFromHeader( std::string & rom_name, const ROMHeader & header );

RomInfo g_ROM;

static bool ROM_LoadCartToMemory( const char * filename );
static void DumpROMInfo( const ROMHeader & header );

//*****************************************************************************
//
//*****************************************************************************
void DumpROMInfo( const ROMHeader & header )
{	
	// The "Header" is actually something to do with the PI_DOM_*_OFS values...
	DBGConsole_Msg(0, "Header:			0x%02x%02x%02x%02x", header.x1, header.x2, header.x3, header.x4);
	DBGConsole_Msg(0, "Clockrate:       0x%08x", header.ClockRate);
	DBGConsole_Msg(0, "BootAddr:		0x%08x", SwapEndian(header.BootAddress));
	DBGConsole_Msg(0, "Release:         0x%08x", header.Release);
	DBGConsole_Msg(0, "CRC1:            0x%08x", header.CRC1);
	DBGConsole_Msg(0, "CRC2:            0x%08x", header.CRC2);
	DBGConsole_Msg(0, "Unknown0:        0x%08x", header.Unknown0);
	DBGConsole_Msg(0, "Unknown1:        0x%08x", header.Unknown1);
	DBGConsole_Msg(0, "ImageName:       '%s'",   header.Name);
	DBGConsole_Msg(0, "Unknown2:        0x%08x", header.Unknown2);
	DBGConsole_Msg(0, "Unknown3:        0x%04x", header.Unknown3);
	DBGConsole_Msg(0, "Unknown4:        0x%02x", header.Unknown4);
	DBGConsole_Msg(0, "Manufacturer:    0x%02x", header.Manufacturer);
	DBGConsole_Msg(0, "CartID:          0x%04x", header.CartID);
	DBGConsole_Msg(0, "CountryID:       0x%02x - '%c'", header.CountryID, (char)header.CountryID);
	DBGConsole_Msg(0, "Unknown5:        0x%02x", header.Unknown5);
}

//*****************************************************************************
//
//*****************************************************************************
static void ROM_RunPIFBoot( ECicType cic_chip )
{
	CPU_SetPC(0xBFC00000);

	// Set up CIC value in 0xbfc007e4
	*(u32*)((u8*)g_pMemoryBuffers[MEM_PIF_RAM] + 0x7C0 + 0x24) = ROM_GetCicValue( cic_chip );
}

//*****************************************************************************
//
//*****************************************************************************
static void ROM_SimulatePIFBoot( ECicType cic_chip, u32 nSystemType )
{
	u32		cic( ROM_GetCicValue( cic_chip ) );

	// g_qwCPR[0][C0_SR]		= 0x34000000;	//*SR_FR |*/ SR_ERL | SR_CU2|SR_CU1|SR_CU0;
	R4300_SetSR(0x34000000);
	g_qwCPR[0][C0_CONFIG]._u64	= 0x0006E463;	// 0x00066463;	

	// These are the effects of the PIF Boot Rom:
	gGPR[REG_r0]._u64 = 0x0000000000000000LL;	// 00
	gGPR[REG_at]._u64 = 0x0000000000000000LL;	// 01
	gGPR[REG_v0]._u64 = 0xFFFFFFFFD1731BE9LL;	// 02 Required by CIC-6105, ignored by other boot codes
	gGPR[REG_v1]._u64 = 0xFFFFFFFFD1731BE9LL;	// 03
	gGPR[REG_a0]._u64 = 0x0000000000001BE9LL;	// 04
	gGPR[REG_a1]._u64 = 0xFFFFFFFFF45231E5LL;	// 05
	gGPR[REG_a2]._u64 = 0xFFFFFFFFA4001F0CLL;	// 06
	gGPR[REG_a3]._u64 = 0xFFFFFFFFA4001F08LL;	// 07
	gGPR[REG_t0]._u64 = 0x00000000000000C0LL;	// 08 0x70?
	gGPR[REG_t1]._u64 = 0x0000000000000000LL;	// 09
	gGPR[REG_t2]._u64 = 0x0000000000000040LL;	// 10
	gGPR[REG_t3]._u64 = 0xFFFFFFFFA4000040LL;	// 11 Required by CIC-6105, ignored by other boot codes
	gGPR[REG_t4]._u64 = 0xFFFFFFFFD1330BC3LL;	// 12
	gGPR[REG_t5]._u64 = 0xFFFFFFFFD1330BC3LL;	// 13
	gGPR[REG_t6]._u64 = 0x0000000025613A26LL;	// 14
	gGPR[REG_t7]._u64 = 0x000000002EA04317LL;	// 15
	gGPR[REG_s0]._u64 = 0x0000000000000000LL;	// 16
	gGPR[REG_s1]._u64 = 0x0000000000000000LL;	// 17
	gGPR[REG_s2]._u64 = 0x0000000000000000LL;	// 18
	gGPR[REG_s3]._u64 = 0x0000000000000000LL;	// 19
	gGPR[REG_s4]._u64 = nSystemType;			// 20
	gGPR[REG_s5]._u64 = 0x0000000000000000LL;	// 21
    gGPR[REG_s6]._u64 = ( cic & 0x0000FF00 ) >> 8;	// 22
    gGPR[REG_s7]._u64 = 0x0000000000000006LL;	// 23
	gGPR[REG_t8]._u64 = 0x0000000000000000LL;	// 24
	gGPR[REG_t9]._u64 = 0xFFFFFFFFD73F2993LL;	// 25
	gGPR[REG_k0]._u64 = 0x0000000000000000LL;	// 26
	gGPR[REG_k1]._u64 = 0x0000000000000000LL;	// 27
	gGPR[REG_gp]._u64 = 0x0000000000000000LL;	// 28
	gGPR[REG_sp]._u64 = 0xFFFFFFFFA4001FF0LL;	// 29
	gGPR[REG_s8]._u64 = 0x0000000000000000LL;	// 30
	gGPR[REG_ra]._u64 = 0xFFFFFFFFA4001554LL;	// 31  0xFFFFFFFFA4000040;????

	// Copy low 1000 bytes to MEM_SP_MEM
	RomBuffer::GetRomBytesRaw( (u8*)g_pMemoryBuffers[MEM_SP_MEM] + RAMROM_BOOTSTRAP_OFFSET, 
		   RAMROM_BOOTSTRAP_OFFSET,
		   RAMROM_GAME_OFFSET - RAMROM_BOOTSTRAP_OFFSET );

//				pal						               ntsc
		/*	gGPR[REG_a1]._u64 = 0xFFFFFFFFDECAAAD1;		gGPR[REG_a1]._u64 = 0x000000005493FB9A;
			gGPR[REG_t6]._u64 = 0x000000000CF85C13;		gGPR[REG_t6]._u64 = 0xFFFFFFFFC2C20384;
			gGPR[REG_t8]._u64 = 0x0000000000000002;		gGPR[REG_t8]._u64 = 0x0000000000000003;
			gGPR[REG_s4]._u64 = 0x0000000000000000;		gGPR[REG_s4]._u64 = 0x0000000000000001;
			gGPR[REG_s7]._u64 = 0x0000000000000006;		gGPR[REG_s7]._u64 = 0x0000000000000000;
			gGPR[REG_ra]._u64 = 0xFFFFFFFFA4001554;		gGPR[REG_ra]._u64 = 0xFFFFFFFFA4001550;*/

	// Need to copy crap to SP_IMEM for CIC-6105 boot.
	u8 * pIMemBase = (u8*)g_pMemoryBuffers[ MEM_SP_MEM ] + 0x1000;
	
	u32 data[] = 
	{
		0x3c0dbfc0,
		0x8da807fc,		// 0xbda807fc for pal?
		0x25ad07c0,
		0x31080080,
		0x5500fffc,
		0x3c0dbfc0,
		0x8da80024,
		0x3c0bb000
	};

	memcpy( pIMemBase, data, sizeof(data) );

	// Also need to set up PI_BSD_DOM1 regs etc!
	CPU_SetPC(0xA4000040);
}

//*****************************************************************************
// Get the name of the correct pit for the region
//*****************************************************************************
static const CHAR * ROM_GetPIFName( u32 tv_type )
{
	switch ( tv_type )
	{
		case OS_TV_PAL:		return "pal.pif";
		case OS_TV_NTSC:	return "ntsc.pif";
		case OS_TV_MPAL:	return "mpal.pif";
	}

	DAEDALUS_ASSERTMSG( "Unknown TV type" );
	return NULL;
}

//*****************************************************************************
// Load pif bootcode if it is available
//*****************************************************************************
static bool ROM_LoadPIF( u32 tv_type )
{
	const CHAR * pif_name;
	CHAR filename[ MAX_PATH ];
	FILE * fp;
	u32 op;

	pif_name = ROM_GetPIFName( tv_type );
	if ( !pif_name )
	{
		return false;
	}

	IO::Path::Combine( filename, gDaedalusExePath, "Boot\\" );
	IO::Path::Append( filename, pif_name );

	fp = fopen( filename, "rb" );
	if ( fp )
	{
		u32 * pPifRom = (u32*)g_pMemoryBuffers[MEM_PIF_RAM];
		u32 i = 0;
		while (fread(&op, 4, 1, fp) == 1 && i < MemoryRegionSizes[MEM_PIF_RAM])
		{
			*pPifRom = SwapEndian( op );
			pPifRom++;
			i+=4;
		}

		fclose(fp);

		return true;
	}
	else
	{
		DBGConsole_Msg( 0, "Couldn't find '[W%s]', simulating boot", filename );
		return false;
	}
}

//*****************************************************************************
//
//*****************************************************************************
void ROM_ReBoot()
{
	DAEDALUS_ASSERT( g_pAiPlugin == NULL, "How come there's an audio plugin?" );
	// Do this here - probably not necessary as plugin loaded/unloaded with audio thread
	if (g_pAiPlugin != NULL)
		g_pAiPlugin->RomClosed();

	if (g_ROM.settings.ExpansionPakUsage != PAK_UNUSED)
		Memory_Reset(MEMORY_8_MEG);
	else
		Memory_Reset(MEMORY_4_MEG);
	
	// Reset here
	CPU_Reset(0xbfc00000);
	RSP_Reset();
	
	g_dwNumFrames = 0;

#ifdef DAEDALUS_ENABLE_OS_HOOKS
	Patch_Reset();
#endif
	/*hr = */CController::Get()->OnRomOpen( g_ROM.settings.SaveType );

	//
	// Find out the CIC type and initialise various systems based on the CIC type
	//
	u8			rom_base[ RAMROM_GAME_OFFSET ];
	RomBuffer::GetRomBytesRaw( rom_base, 0, RAMROM_GAME_OFFSET );

	ECicType	cic_chip( ROM_GenerateCICType( rom_base ) );

	if (cic_chip == CIC_UNKNOWN)
	{
		//DAEDALUS_ASSERTMSG( DSPrintf( "Unknown CIC CRC: 0x%08x\nAssuming CIC-6102", crc ) );
		//DBGConsole_Msg(0, "[MUnknown CIC CRC: 0x%08x]", crc );
		DBGConsole_Msg(0, "[MUnknown CIC]" );
	}
	else
	{
		DBGConsole_Msg(0, "[MRom uses %s]", ROM_GetCicName( cic_chip ) );

		u32 crc1;
		u32 crc2;
		if( ROM_DoCicCheckSum( cic_chip, &crc1, &crc2 ) )
		{
			if (crc1 != RomBuffer::ReadValueRaw< u32 >( 0x10 ) ||
				crc2 != RomBuffer::ReadValueRaw< u32 >( 0x14 ))
			{
				DBGConsole_Msg(0, "[MWarning, CRC values don't match, fixing]");

				RomBuffer::WriteValueRaw< u32 >( 0x10, crc1 );
				RomBuffer::WriteValueRaw< u32 >( 0x14, crc2 );
			}
		}
		else
		{
			// Unable to checksum - just continue with what we have
		}
	}

	// XXXX Update this rom's boot info

	g_ROM.TvType = ROM_GetTvTypeFromID( g_ROM.rh.CountryID );

	FramerateLimiter_SetTvFrequency( g_ROM.TvType == OS_TV_NTSC ? TVF_60Hz : TVF_50Hz );

	if ( ROM_LoadPIF( g_ROM.TvType ) )
	{
		ROM_RunPIFBoot( cic_chip );
	}
	else
	{
		ROM_SimulatePIFBoot( cic_chip, g_ROM.TvType );
	}
	
	//CPU_AddBreakPoint( 0xA4000040 );
	//CPU_AddBreakPoint( 0xbfc00000 );

	DBGConsole_UpdateDisplay();
}

//*****************************************************************************
//
//*****************************************************************************
void ROM_Unload()
{
	RomBuffer::Finalise();
	Memory_Cleanup();
	CController::Get()->OnRomClose();
}



//*****************************************************************************
// Copy across text, null terminate, and strip spaces
//*****************************************************************************
void ROM_GetRomNameFromHeader( std::string & rom_name, const ROMHeader & header )
{
	char	buffer[20+1];
	memcpy( buffer, header.Name, 20 );
	buffer[20] = '\0';

	rom_name = buffer;

	const char * whitespace_chars( " \t\r\n" );
	rom_name.erase( 0, rom_name.find_first_not_of(whitespace_chars) );
	rom_name.erase( rom_name.find_last_not_of(whitespace_chars)+1 );
}

//*****************************************************************************
//
//*****************************************************************************
bool ROM_LoadFileAndRun(const char * filename)
{
	if( ROM_LoadFile( filename ) )
	{
		char szReason[300+1];
		// Ignore failure
		return CPU_StartThread(szReason, 300);
	}

	return false;
}

//*****************************************************************************
//
//*****************************************************************************
bool ROM_LoadFile(const char * filename)
{
	RomID		rom_id;
	u32			rom_size;
	ECicType	boot_type;

	if(ROM_GetRomDetailsByFilename( filename, &rom_id, &rom_size, &boot_type ))
	{
		RomSettings			settings;
		SRomPreferences		preferences;

		if ( !CRomSettingsDB::Get()->GetSettings( rom_id, &settings ) )
		{
			settings.Reset();
		}
		if( !CPreferences::Get()->GetRomPreferences( rom_id, &preferences ) )
		{
			preferences.Reset();
		}

		return ROM_LoadFile( filename, rom_id, settings, preferences );
	}

	return false;
}

//*****************************************************************************
//
//*****************************************************************************
bool ROM_LoadFile(const char * filename, const RomID & rom_id, const RomSettings & settings, const SRomPreferences & preferences )
{
	ROM_Unload();

	DBGConsole_Msg(0, "Reading rom image: [C%s]", filename);

	if ( !ROM_LoadCartToMemory( filename ) )
	{
		DBGConsole_Msg( 0, "ROM_LoadCartToMemory(%s) Failed", filename );
		return false;
	}

	strncpy(g_ROM.szFileName, filename, MAX_PATH);
	g_ROM.szFileName[MAX_PATH-1] = '\0';
	
	// Get information about the rom header
	RomBuffer::GetRomBytesRaw( &g_ROM.rh, 0, sizeof(ROMHeader) );
	ROMFile::ByteSwap_3210( &g_ROM.rh, sizeof(ROMHeader) );
	
	DAEDALUS_ASSERT( RomID( g_ROM.rh ) == rom_id, "Why is the rom id incorrect?" );

	// Copy across various bits
	g_ROM.mRomID = rom_id;
	g_ROM.settings = settings;

	DumpROMInfo( g_ROM.rh );

	//
	//
	//
	preferences.Apply();

	DBGConsole_Msg(0, "[G%s]", g_ROM.settings.GameName.c_str());
	DBGConsole_Msg(0, "This game has been certified as [G%s] (%s)", g_ROM.settings.Comment.c_str(), g_ROM.settings.Info.c_str());
	DBGConsole_Msg(0, "SaveType: [G%s]", ROM_GetSaveTypeName( g_ROM.settings.SaveType ) );
	DBGConsole_Msg(0, "ApplyPatches: [G%s]", gOSHooksEnabled ? "on" : "off");
#ifdef DAEDALUS_PSP
	DBGConsole_Msg(0, "Check Texture Hash Freq: [G%d]", gCheckTextureHashFrequency);
#endif
	DBGConsole_Msg(0, "SpeedSync: [G%s]", FramerateLimiter_GetLimit() ? "on" : "off");
	DBGConsole_Msg(0, "DynaRec: [G%s]", gDynarecEnabled ? "on" : "off");

	//Patch_ApplyPatches();

	ROM_ReBoot();
	return true;
}

//*****************************************************************************
//
//*****************************************************************************
bool ROM_GetRomName( const char * filename, std::string & game_name )
{
	ROMFile * p_rom_file( ROMFile::Create( filename ) );
	if(p_rom_file == NULL)
	{
		return false;
	}

	COutputStringStream messages;

	if( !p_rom_file->Open( messages ) )
	{
		delete p_rom_file;
		return false;
	}

	// Only read in the header
	const u32	bytes_to_read( sizeof(ROMHeader) );
	u8 *		p_bytes;
	u32			buffer_size;
	if(!p_rom_file->LoadDataChunk( bytes_to_read, &p_bytes, &buffer_size, messages ))
	{
		// Lots of files don't have any info - don't worry about it
		delete p_rom_file;
		return false;
	}

	//
	//	Swap into native format
	//
	ROMFile::ByteSwap_3210( p_bytes, buffer_size );

	//
	// Get the address of the rom header
	// Setup the rom id and size
	//
	const ROMHeader * prh( reinterpret_cast<const ROMHeader *>( p_bytes ) );
	ROM_GetRomNameFromHeader( game_name, *prh );

	delete [] p_bytes;
	delete p_rom_file;
	return true;
}

//*****************************************************************************
//
//*****************************************************************************
bool ROM_GetRomDetailsByFilename( const char * filename, RomID * id, u32 * rom_size, ECicType * boot_type )
{
	return CRomDB::Get()->QueryByFilename( filename, id, rom_size, boot_type );
}

//*****************************************************************************
//
//*****************************************************************************
bool ROM_GetRomDetailsByID( const RomID & id, u32 * rom_size, ECicType * boot_type )
{
	return CRomDB::Get()->QueryByID( id, rom_size, boot_type );
}

//*****************************************************************************
// Read a cartridge into memory. this routine byteswaps if necessary
//*****************************************************************************
bool ROM_LoadCartToMemory( const char * filename )
{
	COutputStringStream messages;
	if(!RomBuffer::Initialise( filename, messages ))
	{
#ifdef DAEDALUS_W32
		CMainWindow::Get()->MessageBox( messages.c_str(), g_szDaedalusName, MB_ICONERROR | MB_OK );
#endif
		return false;
	}

	return true;
}

//*****************************************************************************
// Association between a country id value, tv type and name
//*****************************************************************************
struct CountryIDInfo
{
	s8				CountryID;
	const char *	CountryName;
	u32				TvType;
};

static const CountryIDInfo g_CountryCodeInfo[] = 
{
	{  0,  "0",			OS_TV_NTSC },
	{ '7', "Beta",		OS_TV_NTSC },
	{ 'A', "NTSC",		OS_TV_NTSC },
	{ 'D', "Germany",	OS_TV_PAL },
	{ 'E', "USA",		OS_TV_NTSC },
	{ 'F', "France",	OS_TV_PAL },
	{ 'I', "Italy",		OS_TV_PAL },
	{ 'J', "Japan",		OS_TV_NTSC },
	{ 'P', "Europe",	OS_TV_PAL },
	{ 'S', "Spain",		OS_TV_PAL },
	{ 'U', "Australia", OS_TV_PAL },
	{ 'X', "PAL",		OS_TV_PAL },
	{ 'Y', "PAL",		OS_TV_PAL }
};



//*****************************************************************************
// Get a string representing the country name from an ID value
//*****************************************************************************
const char * ROM_GetCountryNameFromID( u8 country_id )
{
	for ( u32 i = 0; i < ARRAYSIZE( g_CountryCodeInfo ); i++)
	{
		if ( g_CountryCodeInfo[i].CountryID == country_id )
		{
			return g_CountryCodeInfo[i].CountryName;
		}
	}
	return "?";
}


//*****************************************************************************
//
//*****************************************************************************
u32 ROM_GetTvTypeFromID( u8 country_id )
{
	for ( u32 i = 0; ARRAYSIZE( g_CountryCodeInfo ); i++ )
	{
		if ( g_CountryCodeInfo[i].CountryID == country_id )
		{
			return g_CountryCodeInfo[i].TvType;
		}
	}
	return OS_TV_NTSC;
}

