/*
Copyright (C) 2001-2007 StrmnNrmn

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/


// Stuff to handle Processor
#include "stdafx.h"

#include "CPU.h"
#include "Registers.h"					// For REG_?? defines
#include "RSP_HLE.h"
#include "RSP.h"
#include "Memory.h"
#include "Interrupt.h"
#include "ROMBuffer.h"
#include "SaveState.h"

#include "Utility/PrintOpCode.h"
#include "Utility/Profiler.h"
#include "Utility/FramerateLimiter.h"
#include "Utility/ResourceString.h"
#include "Utility/Synchroniser.h"

#include "Resources/resource.h"

#include "DaedHash.h"
#include "DaedThread.h"
#include "DaedIO.h"
#include "DaedCritSect.h"

#include "Debug/DebugLog.h"
#include "Debug/DBGConsole.h"

#include "DynaRec/FragmentCache.h"
#include "DynaRec/Fragment.h"
#include "Dynarec/TraceRecorder.h"
#include "Dynarec/DynaRecProfile.h"

#include "OSHLE/ultra_R4300.h"
#include "OSHLE/Patch.h"				// GetCorrectOp

#include "Interface/MainWindow.h"		// For MWM_STARTEMU etc

#include "R4300.h"
#include "R4300_Regs.h"

#include "Plugins/GraphicsPlugin.h"
#include "Plugins/AudioPlugin.h"

#include <algorithm>
#include <string>

//#define ALLOW_TRACES_WHICH_EXCEPT		// Experimental

static const bool REALLY_ENABLE_DYNAREC = true;
#define COUNTER_INCREMENT_PER_OP			1



//
//	New dynarec engine
//
u64									gTotalInstructionsExecuted = 0;
u64									gTotalInstructionsEmulated = 0;

#ifdef DAEDALUS_ENABLE_DYNAREC



#ifndef DAEDALUS_RELEASE_BUILD
#define COUNT_DYNAREC_INSTRUCTION_RATIO
#endif
static const u32					gMaxFragmentCacheSize = 3000;
static const u32					gMaxHotTraceMapSize = 2500;
static const u32					gHotTraceThreshold = 50;

//typedef CMemoryPoolAllocator< std::pair< const u32, u32 > > MyAllocator;
//std::map< u32, u32, std::less<u32>, MyAllocator >				gHotTraceCountMap;
std::map< u32, u32 >				gHotTraceCountMap;
//std::map< u32, u32, std::less<u32>, boost::pool_allocator<std::pair< const u32, u32 > > >				gHotTraceCountMap;
static CFragmentCache				gFragmentCache;
static bool							gResetFragmentCache = false;

static CTraceRecorder				gTraceRecorder;
#ifdef DAEDALUS_DEBUG_DYNAREC
std::map< u32, u32 >				gAbortedTraceReasons;
#endif

static void							CPU_HandleDynaRecOnBranch( bool backwards, bool trace_already_enabled );
static void							CPU_UpdateTrace( u32 address, OpCode op_code, bool branch_delay_slot, bool branch_taken );
static void							CPU_CreateAndAddFragment();

#ifdef DAEDALUS_DEBUG_DYNAREC
void								CPU_DumpFragmentCache();
#endif
void								CPU_ResetFragmentCache();

#endif // DAEDALUS_ENABLE_DYNAREC

//*****************************************************************************
//
//*****************************************************************************
#if defined(_DEBUG) && defined(DAEDALUS_W32)

static u32		gTraceOutputFromCount = 0;
static u32		gTraceOutputContextLines = 0x150;

void TraceOutput( u32 count, OpCode op_code )
{
	if( gTraceOutputFromCount != 0 &&
		count > gTraceOutputFromCount - gTraceOutputContextLines &&
		count < gTraceOutputFromCount + gTraceOutputContextLines )
	{
		char buf[ 200 ];
		SprintOpCodeInfo( buf, gCPUState.CurrentPC, op_code );
		char buf2[ 200 ];
		sprintf( buf2, "Count %08x: PC: %08x: %s\n", count, gCPUState.CurrentPC, buf );
		OutputDebugString( buf2 );
	}
}

#define		TRACE_EXECUTION_PROGRESS( count, op_code )		TraceOutput( count, op_code )

#else

#define		TRACE_EXECUTION_PROGRESS( count, op_code )

#endif

//*****************************************************************************
//
//*****************************************************************************

#ifdef DAEDALUS_BREAKPOINTS_ENABLED
std::vector< DBG_BreakPoint > g_BreakPoints;
#endif

CGraphicsPlugin * gGraphicsPlugin = NULL;
CAudioPlugin * g_pAiPlugin		= NULL;

namespace
{

bool		gCPURunning				= false;			// CPU is actively running
bool		gCPUWantQuit			= false;			// Want to stop the CPU running for whatever reason
bool		gCPUStopOnSimpleState	= false;			// When stopping, try to stop in a 'simple' state (i.e. no RSP running and not in a branch delay slot)

CCritSect	gSaveStateCritialSection;

enum ESaveStateOperation
{
	SSO_NONE,
	SSO_SAVE,
	SSO_LOAD,
};

ESaveStateOperation		gSaveStateOperation = SSO_NONE;
std::string				gSaveStateFilename	= "";

}


//*****************************************************************************
//
//*****************************************************************************
static u32			gVerticalInterrupts( 0 );
static u32			VI_INTR_CYCLES( 625000 );

static s32			gCPUThreadHandle( daedalus::INVALID_THREAD_HANDLE );
static volatile bool gCPUThreadActive( false );

ALIGNED_GLOBAL(SCPUState, gCPUState, CACHE_ALIGN);

//*****************************************************************************
//
//*****************************************************************************
static u32 __stdcall CPUThreadFunc();
template< bool DynaRec, bool TraceEnabled, bool RunRSP > static void CPU_Go();

#ifdef DAEDALUS_ENABLE_DYNAREC
static void (* g_pCPUCore)() = CPU_Go< true, false, false >;
#else
static void (* g_pCPUCore)() = CPU_Go< false, false, false >;
#endif

//*****************************************************************************
//
//*****************************************************************************
inline bool CPU_ProcessEventCycles( u32 cycles )
{
	DAEDALUS_ASSERT( gCPUState.NumEvents > 0, "There are no events" );
	gCPUState.Events[ 0 ].mCount -= cycles;
	return gCPUState.Events[ 0 ].mCount <= 0;
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_SkipToNextEvent()
{
	DAEDALUS_ASSERT( gCPUState.NumEvents > 0, "There are no events" );
	g_qwCPR[0][C0_COUNT]._u32[0] += (gCPUState.Events[ 0 ].mCount - 1);
	gCPUState.Events[ 0 ].mCount = 1;
}

//*****************************************************************************
//
//*****************************************************************************
static void CPU_ResetEventList()
{
	gCPUState.Events[ 0 ].mCount = VI_INTR_CYCLES;
	gCPUState.Events[ 0 ].mEventType = CPU_EVENT_VBL;
	gCPUState.NumEvents = 1;
}

//*****************************************************************************
//
//*****************************************************************************
static void CPU_AddEvent( s32 count, ECPUEventType event_type )
{
	DAEDALUS_ASSERT( count > 0, "Count is invalid" );
	DAEDALUS_ASSERT( gCPUState.NumEvents < SCPUState::MAX_CPU_EVENTS, "Too many events" );

	u32 event_idx;
	for( event_idx = 0; event_idx < gCPUState.NumEvents; ++event_idx )
	{
		CPUEvent & event( gCPUState.Events[ event_idx ] );

		if( count <= event.mCount )
		{
			//
			//	This event belongs before the subsequent one so insert a space for it here and break out
			//	Don't forget to decrement the counter for the subsequent event
			//
			event.mCount -= count;

			u32 num_to_copy( gCPUState.NumEvents - event_idx );
			if( num_to_copy > 0 )
			{
				memmove( &gCPUState.Events[ event_idx+1 ], &gCPUState.Events[ event_idx ], num_to_copy * sizeof( CPUEvent ) );
			}
			break;
		}

		//
		//	Decrease counter by that for this event
		//
		count -= event.mCount;
	}

	DAEDALUS_ASSERT( event_idx <= gCPUState.NumEvents, "Invalid idx" );
	gCPUState.Events[ event_idx ].mCount = count;
	gCPUState.Events[ event_idx ].mEventType = event_type;
	gCPUState.NumEvents++;
}

//*****************************************************************************
//
//*****************************************************************************
static void CPU_SetCompareEvent( s32 count )
{
	DAEDALUS_ASSERT( count > 0, "Count is invalid" );

	//
	//	Remove any existing compare events. Need to adjust any subsequent timer's count.
	//
	for( u32 i = 0; i < gCPUState.NumEvents; ++i )
	{
		if( gCPUState.Events[ i ].mEventType == CPU_EVENT_COMPARE )
		{
			//
			//	Check for a following event, and remove
			//
			if( i+1 < gCPUState.NumEvents )
			{
				gCPUState.Events[ i+1 ].mCount += gCPUState.Events[ i ].mCount;
				u32 num_to_copy( gCPUState.NumEvents - (i+1) );
				memmove( &gCPUState.Events[ i ], &gCPUState.Events[ i+1 ], num_to_copy * sizeof( CPUEvent ) );
			}
			gCPUState.NumEvents--;
			break;
		}
	}

	CPU_AddEvent( count, CPU_EVENT_COMPARE );
}

//*****************************************************************************
//
//*****************************************************************************
static ECPUEventType CPU_PopEvent()
{
	DAEDALUS_ASSERT( gCPUState.NumEvents > 0, "Event queue empty" );
	DAEDALUS_ASSERT( gCPUState.Events[ 0 ].mCount <= 0, "Popping event when cycles remain" );
	DAEDALUS_ASSERT( gCPUState.Events[ 0 ].mCount == 0, "Popping event with a bit of underflow" );

	ECPUEventType	event_type( gCPUState.Events[ 0 ].mEventType );

	u32	num_to_copy( gCPUState.NumEvents - 1 );
	if( num_to_copy > 0 )
	{
		memmove( &gCPUState.Events[ 0 ], &gCPUState.Events[ 1 ], num_to_copy * sizeof( CPUEvent ) );
	}
	gCPUState.NumEvents--;

	return event_type;
}

//*****************************************************************************
// XXXX This is for savestate. Looks very suspicious to me
//*****************************************************************************
u32 CPU_GetVideoInterruptEventCount()
{
	for( u32 i = 0; i < gCPUState.NumEvents; ++i )
	{
		if(gCPUState.Events[ i ].mEventType == CPU_EVENT_VBL)
		{
			return gCPUState.Events[ i ].mCount;
		}
	}

	return 0;
}

//*****************************************************************************
// XXXX This is for savestate. Looks very suspicious to me
//*****************************************************************************
void CPU_SetVideoInterruptEventCount( u32 count )
{
	for( u32 i = 0; i < gCPUState.NumEvents; ++i )
	{
		if(gCPUState.Events[ i ].mEventType == CPU_EVENT_VBL)
		{
			gCPUState.Events[ i ].mCount = count;
			return;
		}
	}
}


//*****************************************************************************
//
//*****************************************************************************
SCPUState::SCPUState()
:	CurrentPC( 0 )
,	TargetPC( 0 )
,	Delay( NO_DELAY )
,	StuffToDo( 0 )
,	NumEvents( 0 )
{
	MultLo._u64 = 0;
	MultHi._u64 = 0;
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_Reset( u32 new_pc )
{
	u32 i;
	u32 dwReg;
	u32 dwCPU;
	DBGConsole_Msg(0, "Resetting with PC of 0x%08x", new_pc);

	FramerateLimiter_Reset();

	CPU_SetPC( new_pc );
	gCPUState.MultHi._u64 = 0;
	gCPUState.MultLo._u64 = 0;

	for (dwReg = 0; dwReg < 32; dwReg++)
	{
		gGPR[dwReg]._u64 = 0;
		for (dwCPU = 0; dwCPU < 3; dwCPU++)
			g_qwCPR[dwCPU][dwReg]._u64 = 0;
	}

	// Init TLBs:
	for (i = 0; i < 32; i++)
	{	
		g_TLBs[i].pfno = 0;	// Clear Valid bit;
		g_TLBs[i].pfne = 0;	// Clear Valid bit;

		g_TLBs[i].pagemask = 0;

		g_TLBs[i].g		   = 1;

		g_TLBs[i].mask     =  g_TLBs[i].pagemask | 0x00001FFF;
		g_TLBs[i].mask2    =  g_TLBs[i].mask>>1;
		g_TLBs[i].vpnmask  = ~g_TLBs[i].mask;
		g_TLBs[i].vpn2mask =  g_TLBs[i].vpnmask>>1;

		g_TLBs[i].pfnehi = ((g_TLBs[i].pfne<<TLBLO_PFNSHIFT) & g_TLBs[i].vpn2mask);
		g_TLBs[i].pfnohi = ((g_TLBs[i].pfno<<TLBLO_PFNSHIFT) & g_TLBs[i].vpn2mask);
		
		g_TLBs[i].addrcheck = 0;
		g_TLBs[i].lastaccessed = 0;
	}

	// From R4300 manual			
    g_qwCPR[0][C0_RAND]._u64		= 32-1;			// TLBENTRIES-1
    //g_qwCPR[0][C0_SR]._u64		= 0x70400004;	//*SR_FR |*/ SR_ERL | SR_CU2|SR_CU1|SR_CU0;
	R4300_SetSR(0x70400004);
    g_qwCPR[0][C0_PRID]._u64	= 0x00000b10;	// Was 0xb00 - test rom reports 0xb10!!
	g_qwCPR[0][C0_CONFIG]._u64	= 0x0006E463;	// 0x00066463;	
	g_qwCPR[0][C0_WIRED]._u64   = 0x0;	

	g_qwCCR[1][0]._u64 = 0x00000511;
	
	Memory_MI_SetRegister(MI_VERSION_REG, 0x02020102);
	Memory_SP_SetRegister(SP_STATUS_REG, SP_STATUS_HALT);			// SP is halted

	((u32 *)g_pMemoryBuffers[MEM_RI_REG])[3] = 1;					// RI_CONFIG_REG Skips most of init


	// Look for Game boot address jump (so we know when to patch)
	// This op re-patches itself to R4300_Special_JR when 
	// the jump is detected
	R4300SpecialInstruction[SpecOp_JR] = R4300_Special_JR_CheckBootAddress;

	gCPUState.Delay = NO_DELAY;
	gCPUState.StuffToDo = 0;
	gVerticalInterrupts = 0;

	Dynarec_ClearedCPUStuffToDo();

	// Clear event list:
	CPU_ResetEventList();

#ifdef DAEDALUS_BREAKPOINTS_ENABLED
	g_BreakPoints.clear();
#endif

#ifdef DAEDALUS_ENABLE_DYNAREC
	gHotTraceCountMap.clear();
	gFragmentCache.Clear();
	gResetFragmentCache = false;
	gTraceRecorder.AbortTrace();
#ifdef DAEDALUS_DEBUG_DYNAREC
	gAbortedTraceReasons.clear();
#endif
#endif

	//
	// We could call CPU_SelectCore here, but we know the RSP is halted, and we don't want to set any flags
	//
#ifdef DAEDALUS_ENABLE_DYNAREC
	if (gDynarecEnabled && REALLY_ENABLE_DYNAREC)
	{
		g_pCPUCore = CPU_Go< true, false, false >;
	}
	else
#endif
	{
		g_pCPUCore = CPU_Go< false, false, false >;
	}
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_Finalise()
{
#ifdef DAEDALUS_ENABLE_DYNAREC
#ifdef DAEDALUS_DEBUG_DYNAREC
	CPU_DumpFragmentCache();
#endif
#endif
}

//*****************************************************************************
//
//*****************************************************************************
bool	CPU_IsStateSimple()
{
	bool	rsp_halted( (Memory_SP_GetRegister( SP_STATUS_REG ) & SP_STATUS_HALT) != 0 );

	return rsp_halted && (gCPUState.Delay == NO_DELAY);
}

//*****************************************************************************
//
//*****************************************************************************
bool	CPU_SaveState( const char * filename )
{
	AUTO_CRIT_SECT( gSaveStateCritialSection );

	bool	success;

	//
	//	Abort if already in the process of loading/saving
	//
	if( gSaveStateOperation != SSO_NONE )
	{
		success = false;
	}
	else
	{
		if( gCPURunning )
		{
			gSaveStateOperation = SSO_SAVE;
			gSaveStateFilename = filename;
			CPU_AddJob(CPU_CHANGE_CORE);

			success = true;	// XXXX could fail
		}
		else
		{
			if( CPU_IsStateSimple() )
			{
				success = SaveState_SaveToFile( filename );
			}
			else
			{
				// Error - cpu is stopped, but not in a simple state
				success = false;
			}
		}
	}

	return success;
}

//*****************************************************************************
//
//*****************************************************************************
bool	CPU_LoadState( const char * filename )
{
	AUTO_CRIT_SECT( gSaveStateCritialSection );

	bool	success;

	//
	//	Abort if already in the process of loading/saving
	//
	if( gSaveStateOperation != SSO_NONE )
	{
		success = false;
	}
	else
	{
		if( gCPURunning )
		{
			gSaveStateOperation = SSO_LOAD;
			gSaveStateFilename = filename;
			CPU_AddJob(CPU_CHANGE_CORE);

			success = true;	// XXXX could fail
		}
		else
		{
			if( CPU_IsStateSimple() )
			{
				success = SaveState_LoadFromFile( filename );
			}
			else
			{
				// Error - cpu is stopped, but not in a simple state
				success = false;
			}
		}
	}

	return success;
}

//*****************************************************************************
// For running the CPU in the main thread. This may need revising when
// audio is introduced
//*****************************************************************************
bool CPU_Run()
{
	DAEDALUS_ASSERT( gCPUThreadHandle == daedalus::INVALID_THREAD_HANDLE, "Calling CPU_Run when a thread is already running" );

	if (RomBuffer::IsRomLoaded())
	{
		CPUThreadFunc();

		// Should wrap all of this up into a single cleanup function
		CPU_ResetEventList();
	#ifdef DAEDALUS_ENABLE_DYNAREC
		gFragmentCache.Clear();
		gHotTraceCountMap.clear();
		gResetFragmentCache = false;
	#endif
		return true;
	}

	return false;
}

//*****************************************************************************
// Thread stuff
//*****************************************************************************
bool CPU_StartThread( char * p_failure_reason, u32 length )
{
	if (!RomBuffer::IsRomLoaded())
	{
		strncpy( p_failure_reason, CResourceString(IDS_NOROMLOADED), length );
		p_failure_reason[ length - 1 ] = '\0';
		return false;
	}

	// If the thread is already running, just return
	if (gCPUThreadHandle != daedalus::INVALID_THREAD_HANDLE)
		return true;

	// Attempt to create the thread
	gCPUThreadHandle = daedalus::CreateThread( CPUThreadFunc, true );
	if (gCPUThreadHandle == daedalus::INVALID_THREAD_HANDLE)
	{
		strncpy( p_failure_reason, CResourceString(IDS_UNABLETOSTARTCPUTHREAD), length );
		p_failure_reason[ length - 1 ] = '\0';
		return false;
	}

	return true;
}

//*****************************************************************************
//
//*****************************************************************************
static void WaitForCPUThreadTermination()
{
	if(gCPUThreadHandle != daedalus::INVALID_THREAD_HANDLE)
	{
		// Wait forever for it to finish. It will clear/close gCPUThreadHandle when it exits
		while(gCPUThreadActive && !daedalus::WaitForThreadTermination(gCPUThreadHandle, 1000))
		{
			DBGConsole_Msg(0, "Waiting for CPU thread (0x%08x) to finish", gCPUThreadHandle);
		}

		DAEDALUS_ASSERT( !gCPUThreadActive, "How come the thread is still marked as active?" );

		daedalus::ReleaseThreadHandle( gCPUThreadHandle );
		gCPUThreadHandle = daedalus::INVALID_THREAD_HANDLE;

		DBGConsole_Msg(0, "CPU Thread finished");
	}
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_StopThread()
{
	// If it's not running, just return silently
	if (gCPUThreadHandle == daedalus::INVALID_THREAD_HANDLE)
		return;

	// If it is running, we need to signal for it to stop
	gCPUWantQuit = true;
	gCPUStopOnSimpleState = true;
	CPU_SelectCore();

	WaitForCPUThreadTermination();
}

//*****************************************************************************
//
//*****************************************************************************
u32 __stdcall CPUThreadFunc()
{
	gCPURunning = true;
	gCPUThreadActive = true;
	gCPUStopOnSimpleState = false;

	DAEDALUS_ASSERT( g_pAiPlugin == NULL, "Why is there already an audio plugin?" );

	CAudioPlugin * audio_plugin( CreateAudioPlugin() );
	if( audio_plugin != NULL )
	{
		if( !audio_plugin->StartEmulation() )
		{
			delete audio_plugin;
			audio_plugin = NULL;
		}
		g_pAiPlugin = audio_plugin;
	}

#if defined(DAEDALUS_W32)
	// From Lkb- avoid GFX exceptions when the RDP is configured for the ROM ucode 
	//CMainWindow::Get()->PostMessage( CMainWindow::MWM_STARTEMU, 0,0);
	CMainWindow::Get()->SendMessage( CMainWindow::MWM_STARTEMU, 0,0 );
#endif

	DAEDALUS_ASSERT( gGraphicsPlugin == NULL, "The graphics plugin should not be initialised at this point" );
	CGraphicsPlugin *	graphics_plugin( CreateGraphicsPlugin() );
	if( graphics_plugin != NULL )
	{
		if( !graphics_plugin->StartEmulation() )
		{
			delete graphics_plugin;
			graphics_plugin = NULL;
		}
		gGraphicsPlugin = graphics_plugin;
	}

#ifdef DAEDALUS_USE_EXCEPTIONS
	try
#endif
	{
		gCPUWantQuit = false;
		while( !gCPUWantQuit && gCPURunning )
		{
			while (gCPURunning)
			{
				g_pCPUCore();

				if( gSaveStateOperation != SSO_NONE )
				{
					if( CPU_IsStateSimple() )
					{
						AUTO_CRIT_SECT( gSaveStateCritialSection );

						printf( "Loading '%s'\n", gSaveStateFilename.c_str() );

						//
						// Handle the save state
						//
						switch( gSaveStateOperation )
						{
						case SSO_NONE:
							DAEDALUS_ERROR( "Unreachable" );
							break;
						case SSO_SAVE:
							SaveState_SaveToFile( gSaveStateFilename.c_str() );
							break;
						case SSO_LOAD:
							SaveState_LoadFromFile( gSaveStateFilename.c_str() );
							break;
						}
						gSaveStateOperation = SSO_NONE;
					}
					else
					{
						printf( "Can't load yet, state not simple\n" );
					}
				}
			}
		}

	}
#ifdef DAEDALUS_USE_EXCEPTIONS
	catch (...)
	{
		gCPURunning = false;
		if ( !g_DaedalusConfig.TrapExceptions)
		{
			throw;
		}

		#ifdef DAEDALUS_W32
			CMainWindow::Get()->MessageBox(CResourceString(IDS_CPUTHREAD_EXCEPTION),
										   g_szDaedalusName,
										   MB_ICONEXCLAMATION|MB_OK);
		#else
			DBGConsole_Msg( 0, "%s", CResourceString(IDS_CPUTHREAD_EXCEPTION).c_str() );
		#endif
	}
#endif


	// NB - this order is very critical. The main thread is polling on gCPUThreadActive after
	// a call to CPU_StopThread(), and so it can't process its message pump. This means that
	// any calls below that can potentially send a message to our main window will block indefinitely.
	gCPURunning = false;
	gCPUThreadActive = false;

	// Update the screen. It's probably better handled elsewhere...
	DBGConsole_UpdateDisplay();

	if ( gGraphicsPlugin != NULL )
	{
		gGraphicsPlugin->RomClosed();
		delete gGraphicsPlugin;
		gGraphicsPlugin = NULL;
	}

	// Make a copy of the plugin, and set the global pointer to NULL; 
	// This stops other threads from trying to access the plugin
	// while we're in the process of shutting it down.
	g_pAiPlugin = NULL;
	if (audio_plugin != NULL)
	{
		audio_plugin->StopEmulation();
		delete audio_plugin;
	}

#ifdef DAEDALUS_W32
	// This causes a few problems with synchronisation if we call here
	// (it passes a message to the main window, but the main window's 
	// handling thread is busy waiting for this thread to exit). I've
	// put the call at the end of CPU_StopThread()
	//Main_ActivateList();
	if(CMainWindow::IsAvailable())
	{
		CMainWindow::Get()->PostMessage(CMainWindow::MWM_ENDEMU, 0,0);
	}
#endif

	return 0;
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_DynarecEnable()
{
	gDynarecEnabled = true;
	CPU_AddJob(CPU_CHANGE_CORE);
}


//*****************************************************************************
//	Indicate that the instruction cache is invalid
//	(we have to dump the dynarec contents and start over, but this is 
//	better than crashing :) )
//*****************************************************************************
void	CPU_InvalidateICache()
{
#ifdef DAEDALUS_ENABLE_DYNAREC
	CPU_ResetFragmentCache();
#endif
}

//*****************************************************************************
//
//*****************************************************************************
void	CPU_InvalidateICacheRange( u32 address, u32 length )
{
#ifdef DAEDALUS_ENABLE_DYNAREC
	if( gFragmentCache.ShouldInvalidateOnWrite( address, length ) )
	{
		printf( "Write to %08x (%d bytes) overlaps fragment cache entries\n", address, length );
		CPU_ResetFragmentCache();
	}
#endif
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_SelectCore()
{
	bool run_rsp = (Memory_SP_GetRegister( SP_STATUS_REG ) & SP_STATUS_HALT) == 0;

#ifdef DAEDALUS_ENABLE_DYNAREC
	bool trace_enabled = gTraceRecorder.IsTraceActive();

	if (gDynarecEnabled && REALLY_ENABLE_DYNAREC)
	{
		if (trace_enabled)
		{
			if (run_rsp)
				g_pCPUCore = CPU_Go< true, true, true >;
			else
				g_pCPUCore = CPU_Go< true, true, false >;
		}
		else
		{
			if (run_rsp)
				g_pCPUCore = CPU_Go< true, false, true >;
			else
				g_pCPUCore = CPU_Go< true, false, false >;
		}
	}
	else
#endif
	{
#ifdef DAEDALUS_ENABLE_DYNAREC
		DAEDALUS_ASSERT( !trace_enabled, "No dynarec but trace active?" );
#endif

		if (run_rsp)
			g_pCPUCore = CPU_Go< false, false, true >;
		else
			g_pCPUCore = CPU_Go< false, false, false >;
	}

	if( gCPUStopOnSimpleState && CPU_IsStateSimple() )
	{
		CPU_AddJob( CPU_STOP_RUNNING );
	}
	else
	{
		CPU_AddJob( CPU_CHANGE_CORE );
	}
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_Halt( const char * reason )
{
	DBGConsole_Msg( 0, "CPU Halting: %s", reason );
	gCPUWantQuit = true;
	gCPUStopOnSimpleState = true;
}

#ifdef DAEDALUS_BREAKPOINTS_ENABLED
//*****************************************************************************
//
//*****************************************************************************
void CPU_AddBreakPoint( u32 address )
{
	OpCode * pdwOp;

	// Force 4 byte alignment
	address &= 0xFFFFFFFC;

	if (!Memory_GetInternalReadAddress(address, (void**)&pdwOp))
	{
		DBGConsole_Msg(0, "Invalid Address for BreakPoint: 0x%08x", address);
	}
	else
	{
		DBG_BreakPoint bpt;
		DBGConsole_Msg(0, "[YInserting BreakPoint at 0x%08x]", address);
		
		bpt.mOriginalOp = *pdwOp;
		bpt.bEnabled = TRUE;
		bpt.bTemporaryDisable = FALSE;
		g_BreakPoints.push_back(bpt);

		pdwOp->op = OP_DBG_BKPT;
		pdwOp->bp_index = (g_BreakPoints.size() - 1);
	}
}
#endif

#ifdef DAEDALUS_BREAKPOINTS_ENABLED
//*****************************************************************************
//
//*****************************************************************************
void CPU_EnableBreakPoint( u32 address, bool enable )
{
	OpCode * pdwOp;

	// Force 4 byte alignment
	address &= 0xFFFFFFFC;

	if (!Memory_GetInternalReadAddress(address, (void**)&pdwOp))
	{
		DBGConsole_Msg(0, "Invalid Address for BreakPoint: 0x%08x", address);
	}
	else
	{
		OpCode op_code( *pdwOp );

		if (op_code.op != OP_DBG_BKPT)
		{
			DBGConsole_Msg(0, "[YNo breakpoint is set at 0x%08x]", address);
			return;
		}

		// Entry is in lower 26 bits...
		u32 breakpoint_idx( op_code.bp_index );

		if (breakpoint_idx < g_BreakPoints.size())
		{
			g_BreakPoints[breakpoint_idx].bEnabled = enable;
			// Alwyas disable
			g_BreakPoints[breakpoint_idx].bTemporaryDisable = FALSE;
		}
	}
}
#endif

//*****************************************************************************
//
//*****************************************************************************
extern "C"
{
void CPU_HANDLE_COUNT_INTERRUPT()
{
	DAEDALUS_ASSERT( gCPUState.NumEvents > 0, "Should always have at least one event queued up" );

	switch ( CPU_PopEvent() )
	{
	case CPU_EVENT_VBL:
		{
			u32		vertical_sync_reg( *VI_REG_ADDRESS(VI_V_SYNC_REG) );
			if (vertical_sync_reg == 0)
			{
				VI_INTR_CYCLES = 625000;
			}
			else
			{
				VI_INTR_CYCLES = (vertical_sync_reg+1) * 1500;
			}

			// Add another Interrupt at the next time:
			CPU_AddEvent(VI_INTR_CYCLES, CPU_EVENT_VBL);

			gVerticalInterrupts++;

			FramerateLimiter_Limit();

			Memory_MI_SetRegisterBits(MI_INTR_REG, MI_INTR_VI);
			R4300_Interrupt_UpdateCause3();

			if (gGraphicsPlugin != NULL)
			{
				gGraphicsPlugin->UpdateScreen();
			}
		}
		break;
	case CPU_EVENT_COMPARE:
		{
			g_qwCPR[0][C0_CAUSE]._u64 |= CAUSE_IP8;
			CPU_AddJob( CPU_CHECK_INTERRUPTS );
		}
		break;
	default:
		NODEFAULT;
	}

	DAEDALUS_ASSERT( gCPUState.NumEvents > 0, "Should always have at least one event queued up" );
}
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_SetCompare(u64 qwNewValue)
{
	g_qwCPR[0][C0_CAUSE]._u64 &= ~CAUSE_IP8;

	DPF( DEBUG_REGS, DSPrintf( "COMPARE set to 0x%08x.", (u32)qwNewValue ) );
	//DBGConsole_Msg(0, "COMPARE set to 0x%08x Count is 0x%08x.", (u32)qwNewValue, (u32)g_qwCPR[0][C0_COUNT]);

	// Add an event for this compare:
	if (qwNewValue == g_qwCPR[0][C0_COMPARE]._u64)
	{
		//	DBGConsole_Msg(0, "Clear");
	}
	else
	{
		// XXXX Looks very suspicious to me...was this the Oot timer fix?
		u32		diff_32( (u32)qwNewValue - g_qwCPR[0][C0_COUNT]._u32[0] );
		if ((u32)qwNewValue > g_qwCPR[0][C0_COUNT]._u32[0])
		{
			CPU_SetCompareEvent( diff_32 );
		}
		else if (qwNewValue != 0)
		{
			//DBGConsole_Msg(0, "COMPARE set to 0x%08x%08x < Count is 0x%08x%08x.",
			//	(u32)(qwNewValue>>32), (u32)qwNewValue,
			//	(u32)(g_qwCPR[0][C0_COUNT]>>32), g_qwCPR[0][C0_COUNT]._u32[0]);

			CPU_SetCompareEvent( diff_32 );
			//DBGConsole_Msg(0, "0x%08x", diff_32);
		}
		g_qwCPR[0][C0_COMPARE]._u64 = qwNewValue;
	}


}

//*****************************************************************************
//
//*****************************************************************************
u32	CPU_ProduceRegisterHash()
{
	u32	hash( 0 );

	if ( DAED_SYNC_MASK & DAED_SYNC_REG_GPR )
	{
		hash = daedalus::hash( (u8*)&(gGPR[0]), sizeof( gGPR ), hash );
	}

	if ( DAED_SYNC_MASK & DAED_SYNC_REG_CPU0 )
	{
		hash = daedalus::hash( (u8*)&(g_qwCPR[0][0]), sizeof( g_qwCPR[0] ), hash );
	}
	if ( DAED_SYNC_MASK & DAED_SYNC_REG_CCR0 )
	{
		hash = daedalus::hash( (u8*)&(g_qwCCR[0][0]), sizeof( g_qwCCR[0] ), hash );
	}

	if ( DAED_SYNC_MASK & DAED_SYNC_REG_CPU1 )
	{
		hash = daedalus::hash( (u8*)&(g_qwCPR[1][0]), sizeof( g_qwCPR[1] ), hash );
	}
	if ( DAED_SYNC_MASK & DAED_SYNC_REG_CCR1 )
	{
		hash = daedalus::hash( (u8*)&(g_qwCCR[1][0]), sizeof( g_qwCCR[1] ), hash );
	}

	return hash;
}

//*****************************************************************************
//	Execute the specified opcode 
//*****************************************************************************
void	CPU_ExecuteOpRaw( u32 count, u32 address, OpCode op_code, CPU_Instruction p_instruction, bool * p_branch_taken )
{
	gCPUState.CurrentPC = address;

	SYNCH_POINT( DAED_SYNC_REG_PC, gCPUState.CurrentPC, "Program Counter doesn't match" );
	SYNCH_POINT( DAED_SYNC_REG_PC, count, "Count doesn't match" );
	TRACE_EXECUTION_PROGRESS( count, op_code );

	p_instruction( op_code._u32 );

	SYNCH_POINT( DAED_SYNC_REGS, CPU_ProduceRegisterHash(), "Registers don't match" );

	*p_branch_taken = gCPUState.Delay == DO_DELAY;
}


//*****************************************************************************
//
//*****************************************************************************
extern "C"
{
void CPU_UpdateCounter( u32 ops_executed )
{
	DAEDALUS_ASSERT( ops_executed > 0, "Expecting at least one op" );
	//SYNCH_POINT( DAED_SYNC_FRAGMENT_PC, ops_executed, "Number of executed ops doesn't match" );

#ifdef COUNT_DYNAREC_INSTRUCTION_RATIO
	gTotalInstructionsExecuted += ops_executed;
#endif

	const u32	cycles( ops_executed * COUNTER_INCREMENT_PER_OP );

	// Increment count register
	g_qwCPR[0][C0_COUNT]._u32[0] += cycles;

	if( CPU_ProcessEventCycles( cycles ) )
	{
		CPU_HANDLE_COUNT_INTERRUPT();
	}
}

// As above, but no interrupts are fired
void CPU_UpdateCounterNoInterrupt( u32 ops_executed )
{
	//SYNCH_POINT( DAED_SYNC_FRAGMENT_PC, ops_executed, "Number of executed ops doesn't match" );

	if( ops_executed > 0 )
	{
		const u32	cycles( ops_executed * COUNTER_INCREMENT_PER_OP );

#ifdef COUNT_DYNAREC_INSTRUCTION_RATIO
		gTotalInstructionsExecuted += ops_executed;
#endif

		// Increment count register
		g_qwCPR[0][C0_COUNT]._u32[0] += cycles;

		bool	ready( CPU_ProcessEventCycles( cycles ) );
		DAEDALUS_ASSERT(!ready, "Ignoring Count interrupt");	// Just a test - remove eventually (needs to handle this)
	}
}
}

//*****************************************************************************
//
//*****************************************************************************
struct SCachedInfo
{
	u32		PC;
	u8 *	Address;

	SCachedInfo( ) : PC( 0xffffffff ), Address( NULL ) {}
};

static SCachedInfo	gLast;

bool CPU_FetchInstruction_Refill( u32 pc, OpCode * opcode )
{
	gLast.Address = (u8 *)ReadAddress( pc );
	gLast.PC = pc;
	*opcode = *(OpCode *)gLast.Address;
	return gCPUState.StuffToDo == 0;

}

inline bool CPU_FetchInstruction( u32 pc, OpCode * opcode )
{
	const u32 PAGE_MASK_BITS = 12;		// 1<<12 == 4096

	if( (pc>>PAGE_MASK_BITS) == (gLast.PC>>PAGE_MASK_BITS) )
	{
		s32		offset( pc - gLast.PC );

		gLast.Address += offset;
		DAEDALUS_ASSERT( gLast.Address == ReadAddress(pc), "Cached Instruction Pointer base is out of sync" );
		gLast.PC = pc;
		*opcode = *(OpCode *)gLast.Address;
		return true;
	}
	
	return CPU_FetchInstruction_Refill( pc, opcode );
}

//*****************************************************************************
//	Execute a single MIPS op. The conditionals for the templated arguments
//	are completely optimised away by the compiler.
//	
//	DynaRec:		Run this function with dynarec enabled
//	TranslateOp:	Use this to translate breakpoints/patches to original op
//					before execution.
//*****************************************************************************
template< bool DynaRec, bool TraceEnabled, bool TranslateOp > __forceinline void CPU_EXECUTE_OP()
{
	OpCode op_code;

	if( !CPU_FetchInstruction( gCPUState.CurrentPC, &op_code ) )
	{
		//printf( "Exception on instruction fetch @%08x\n", gCPUState.CurrentPC );
		return;
	}

	if ( TranslateOp )
	{
	#ifdef DAEDALUS_BREAKPOINTS_ENABLED
		// Handle breakpoints correctly
		if (op_code.op == OP_DBG_BKPT)
		{
			// Turn temporary disable on to allow instr to be processed
			// Entry is in lower 26 bits...
			u32	breakpoint( op_code.bp_index );

			if ( breakpoint < g_BreakPoints.size() )
			{
				if (g_BreakPoints[ breakpoint ].bEnabled)
				{
					g_BreakPoints[ breakpoint ].bTemporaryDisable = true;
				}
			}
		}
		else
	#endif
		{
			op_code = GetCorrectOp( op_code );
		}
	}

	SYNCH_POINT( DAED_SYNC_REG_PC, gCPUState.CurrentPC, "Program Counter doesn't match" );
	SYNCH_POINT( DAED_SYNC_FRAGMENT_PC, gCPUState.CurrentPC + gCPUState.Delay, "Program Counter/Delay doesn't match while interpreting" );

	SYNCH_POINT( DAED_SYNC_REG_PC, g_qwCPR[0][C0_COUNT]._u32[0], "Count doesn't match" );
	TRACE_EXECUTION_PROGRESS( g_qwCPR[0][C0_COUNT]._u32[0], op_code );

#ifdef DAEDALUS_ENABLE_DYNAREC
	if( TraceEnabled )
	{
		DAEDALUS_ASSERT( gTraceRecorder.IsTraceActive(), "If TraceEnabled is set, trace should be active" );
		u32		pc( gCPUState.CurrentPC );
		bool	branch_delay_slot( gCPUState.Delay == EXEC_DELAY );

		R4300Instruction[ op_code.op ]( op_code._u32 );

		bool	branch_taken( gCPUState.Delay == DO_DELAY );

		CPU_UpdateTrace( pc, op_code, branch_delay_slot, branch_taken );
	}
	else
#endif
	{
#ifdef DAEDALUS_ENABLE_DYNAREC
		DAEDALUS_ASSERT( !gTraceRecorder.IsTraceActive(), "If TraceEnabled is not set, trace should be inactive" );
#endif

		R4300Instruction[ op_code.op ]( op_code._u32 );

#ifdef COUNT_DYNAREC_INSTRUCTION_RATIO
		gTotalInstructionsEmulated++;
#endif
	}

	SYNCH_POINT( DAED_SYNC_REGS, CPU_ProduceRegisterHash(), "Registers don't match" );

	// Increment count register
	g_qwCPR[0][C0_COUNT]._u32[0] = g_qwCPR[0][C0_COUNT]._u32[0] + COUNTER_INCREMENT_PER_OP;

	if (CPU_ProcessEventCycles( COUNTER_INCREMENT_PER_OP ) )
	{
		CPU_HANDLE_COUNT_INTERRUPT();
	}

	switch (gCPUState.Delay)
	{
		case DO_DELAY:
			// We've got a delayed instruction to execute. Increment
			// PC as normal, so that subsequent instruction is executed
			INCREMENT_PC();
			gCPUState.Delay = EXEC_DELAY;

			break;
		case EXEC_DELAY:
			{
				bool	backwards( gCPUState.TargetPC <= gCPUState.CurrentPC );

				// We've just executed the delayed instr. Now carry out jump as stored in gCPUState.TargetPC;
				CPU_SetPC(gCPUState.TargetPC);
				gCPUState.Delay = NO_DELAY;

			#ifdef DAEDALUS_ENABLE_DYNAREC
				if( DynaRec )
				{
					CPU_HandleDynaRecOnBranch( backwards, TraceEnabled );
				}
			#endif
			}
			break;
		case NO_DELAY:
			// Normal operation - just increment the PC
			INCREMENT_PC();
			break;
		default:	// MSVC extension - the default will never be reached
			NODEFAULT;
	}	
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_TakeBranch( u32 new_pc, ECPUBranchType branch_type )
{
	gCPUState.TargetPC = new_pc;
	gCPUState.Delay = DO_DELAY;
}

#ifdef DAEDALUS_ENABLE_DYNAREC
u32 gFragmentLookupSuccess = 0;
u32 gFragmentLookupFailure = 0;

//*****************************************************************************
//
//*****************************************************************************
void CPU_HandleDynaRecOnBranch( bool backwards, bool trace_already_enabled )
{
	DAEDALUS_PROFILE( "CPU_HandleDynaRecOnBranch" );

	bool	start_of_trace( false );

	if( backwards )
	{
		start_of_trace = true;
	}

	bool	change_core( false );

	DAED_LOG( DEBUG_DYNAREC_CACHE, "CPU_HandleDynaRecOnBranch" );

	while( gCPUState.StuffToDo == 0 && gCPUState.Delay == NO_DELAY )
	{
		DAEDALUS_ASSERT( gCPUState.Delay == NO_DELAY, "Why are we entering with a delay slot active?" );
		u32			entry_count( g_qwCPR[0][C0_COUNT]._u32[0] );
		u32			entry_address( gCPUState.CurrentPC );
		CFragment * p_fragment( gFragmentCache.LookupFragment( entry_address ) );
		if( p_fragment != NULL )
		{
			gFragmentLookupSuccess++;
			// Check if another trace is active and we're about to enter
			if( gTraceRecorder.IsTraceActive() )
			{
				gTraceRecorder.StopTrace( gCPUState.CurrentPC );
				CPU_CreateAndAddFragment();

				// We need to change the core when exiting
				change_core = true;
			}

			p_fragment->Execute();

			DYNAREC_PROFILE_ENTEREXIT( entry_address, gCPUState.CurrentPC, g_qwCPR[0][C0_COUNT]._u32[0] - entry_count );

			start_of_trace = true;
		}
		else
		{
			gFragmentLookupFailure++;
			if( start_of_trace )
			{
				start_of_trace = false;

				if( !gTraceRecorder.IsTraceActive() )
				{
					if( gFragmentCache.GetCacheSize() > gMaxFragmentCacheSize || gResetFragmentCache )
					{
						gFragmentCache.Clear();
						gHotTraceCountMap.clear();		// Makes sense to clear this now, to get accurate usage stats
						gResetFragmentCache = false;
					}

					// If there is no fragment for this target, start tracing
					gHotTraceCountMap[ gCPUState.CurrentPC ]++;
					if( gHotTraceCountMap.size() >= gMaxHotTraceMapSize )
					{
						DBGConsole_Msg( 0, "Hot trace cache hit %d, dumping", gHotTraceCountMap.size() );
						gHotTraceCountMap.clear();
						gFragmentCache.Clear();
					}
					else if( gHotTraceCountMap[ gCPUState.CurrentPC ] == gHotTraceThreshold )
					{
						//DBGConsole_Msg( 0, "Identified hot trace at [R%08x]! (size is %d)", gCPUState.CurrentPC, gHotTraceCountMap.size() );
						gTraceRecorder.StartTrace( gCPUState.CurrentPC );

						if(!trace_already_enabled)
						{
							change_core = true;
						}
						DAED_LOG( DEBUG_DYNAREC_CACHE, DSPrintf( "StartTrace( %08x )", gCPUState.CurrentPC ) );
					}
				#ifdef DAEDALUS_DEBUG_DYNAREC
					else if( gHotTraceCountMap[ gCPUState.CurrentPC ] > gHotTraceThreshold )
					{
						if(gAbortedTraceReasons.find( gCPUState.CurrentPC ) != gAbortedTraceReasons.end() )
						{
							u32 reason( gAbortedTraceReasons[ gCPUState.CurrentPC ] );
							//DBGConsole_Msg( 0, "Hot trace at [R%08x] has count of %d! (reason is %x) size %d", gCPUState.CurrentPC, gHotTraceCountMap[ gCPUState.CurrentPC ], reason, gHotTraceCountMap.size( ) );
							DAED_LOG( DEBUG_DYNAREC_CACHE, DSPrintf("Hot trace at %08x has count of %d! (reason is %x) size %d", gCPUState.CurrentPC, gHotTraceCountMap[ gCPUState.CurrentPC ], reason, gHotTraceCountMap.size( ) ) );
						}
						else
						{
							DAED_LOG( DEBUG_DYNAREC_CACHE, DSPrintf("Hot trace at %08x has count of %d! (reason is UNKNOWN!)", gCPUState.CurrentPC, gHotTraceCountMap[ gCPUState.CurrentPC ] ) );
						}
					}
				#endif
				}
			}
			else
			{
				DAED_LOG( DEBUG_DYNAREC_CACHE, "Not start of trace" );
			}
			break;
		}
	}

	if(change_core)
	{
		CPU_SelectCore();
	}
}
#endif

#ifdef DAEDALUS_ENABLE_DYNAREC
//*****************************************************************************
//
//*****************************************************************************
void CPU_UpdateTrace( u32 address, OpCode op_code, bool branch_delay_slot, bool branch_taken )
{
	DAEDALUS_PROFILE( "CPU_UpdateTrace" );

	DAEDALUS_ASSERT_Q( (gCPUState.Delay == EXEC_DELAY) == branch_delay_slot );

	CFragment * p_address_fragment( gFragmentCache.LookupFragment( address ) );

	if( gTraceRecorder.UpdateTrace( address, branch_delay_slot, branch_taken, op_code, p_address_fragment ) == CTraceRecorder::UTS_CREATE_FRAGMENT )
	{
		CPU_CreateAndAddFragment();

		DAEDALUS_ASSERT( !gTraceRecorder.IsTraceActive(), "Why is a trace still active?" );

		CPU_SelectCore();
	}
	else
	{
		DAEDALUS_ASSERT( gTraceRecorder.IsTraceActive(), "The trace should still be enabled" );
	}
}
#endif

#ifdef DAEDALUS_ENABLE_DYNAREC
//*****************************************************************************
//
//*****************************************************************************
void CPU_CreateAndAddFragment()
{
	CFragment * p_fragment( gTraceRecorder.CreateFragment( gFragmentCache.GetCodeBufferManager() ) );

	if( p_fragment != NULL )
	{
		gHotTraceCountMap.erase( p_fragment->GetEntryAddress() );
		gFragmentCache.InsertFragment( p_fragment );

		//DBGConsole_Msg( 0, "Inserted hot trace at [R%08x]! (size is %d. %dKB)", p_fragment->GetEntryAddress(), gFragmentCache.GetCacheSize(), gFragmentCache.GetMemoryUsage() / 1024 );
	}
}
#endif

#ifdef DAEDALUS_ENABLE_DYNAREC
#ifdef DAEDALUS_DEBUG_DYNAREC

struct SAddressHitCount
{
	u32		Address;
	u32		HitCount;

	SAddressHitCount( u32 address, u32 hitcount ) : Address( address ), HitCount( hitcount ) {}

	u32		GetAbortReason() const
	{
		std::map<u32, u32>::const_iterator		it( gAbortedTraceReasons.find( Address ) );
		if( it != gAbortedTraceReasons.end() )
		{
			return it->second;
		}

		return 0;
	}
};

bool SortByHitCount( const SAddressHitCount & a, const SAddressHitCount & b )
{
	return a.HitCount > b.HitCount;
}

//*****************************************************************************
//
//*****************************************************************************
void	CPU_DumpFragmentCache()
{
	IO::Directory::EnsureExists( "DynarecDump" );

	FILE  * fh( fopen( "DynarecDump/hot_trace_map.html", "w" ) );
	if( fh != NULL )
	{
		std::vector< SAddressHitCount >	hit_counts;

		hit_counts.reserve( gHotTraceCountMap.size() );

		for(std::map<u32,u32>::const_iterator it = gHotTraceCountMap.begin(); it != gHotTraceCountMap.end(); ++it )
		{
			hit_counts.push_back( SAddressHitCount( it->first, it->second ) );
		}

		std::sort( hit_counts.begin(), hit_counts.end(), SortByHitCount );

		fputs( "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">", fh );
		fputs( "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n", fh );
		fputs( "<head><title>Hot Trace Map</title>\n", fh );
		fputs( "<link rel=\"stylesheet\" href=\"default.css\" type=\"text/css\" media=\"all\" />\n", fh );
		fputs( "</head><body>\n", fh );
		fputs( "<h1>Hot Trace Map</h1>\n", fh );
		fputs( "<div align=\"center\"><table>\n", fh );
		fputs( "<tr><th>Address</th><th>Hit Count</th><th>Abort Reason</th></tr>\n", fh );

		for( u32 i = 0; i < hit_counts.size(); ++i )
		{
			const SAddressHitCount & info( hit_counts[ i ] );

			u32		abort_reason( info.GetAbortReason() );

			fprintf( fh, "<tr><td>%08x</td><td>%d</td>\n", info.Address, info.HitCount );

			fputs( "<td>", fh );
			if(abort_reason & CPU_CHECK_EXCEPTIONS)		{ fputs( " Exception", fh ); }
			if(abort_reason & CPU_CHECK_INTERRUPTS)		{ fputs( " Interrupt", fh ); }
			if(abort_reason & CPU_STOP_RUNNING)			{ fputs( " StopRunning", fh ); }
			if(abort_reason & CPU_CHANGE_CORE)			{ fputs( " ChangeCore", fh ); }
			fputs( "</td></tr>\n", fh );

			//if( info.HitCount >= gHotTraceThreshold )
		}
		fputs( "</table></div>\n", fh );
		fputs( "</body></html>\n", fh );

		fclose(fh);
	}

	gFragmentCache.DumpStats( "DynarecDump/" );
}
#endif
#endif

#ifdef DAEDALUS_ENABLE_DYNAREC
//*****************************************************************************
//
//*****************************************************************************
void	CPU_ResetFragmentCache()
{
	// Need to make sure this happens at a safe point, so we use a flag
	gResetFragmentCache	= true;
}
#endif

//*****************************************************************************
// Hacky function to use when debugging
//*****************************************************************************
void CPU_Skip()
{
	if (gCPURunning || gCPUThreadHandle != daedalus::INVALID_THREAD_HANDLE)
	{
		DBGConsole_Msg(0, "Already Running");
		return;
	}

	INCREMENT_PC();
}

//*****************************************************************************
//
//*****************************************************************************
void CPU_Step()
{
	if (gCPURunning || gCPUThreadHandle != daedalus::INVALID_THREAD_HANDLE)
	{
		DBGConsole_Msg(0, "Already Running");
		return;
	}

	// We do this in a slightly different order to ensure that
	// any interrupts are taken care of before we execute an op
	if (gCPUState.StuffToDo)
	{	
		// Process Interrupts/Exceptions on a priority basis
		// Call most likely first!
		if (gCPUState.StuffToDo & CPU_CHECK_INTERRUPTS)
		{
			R4300_Handle_Interrupt();
			CPU_ClearJob( CPU_CHECK_INTERRUPTS );
		}
		else if (gCPUState.StuffToDo & CPU_CHECK_EXCEPTIONS)
		{
			R4300_Handle_Exception();
			CPU_ClearJob( CPU_CHECK_EXCEPTIONS );
		}
		else if (gCPUState.StuffToDo & CPU_STOP_RUNNING)
		{
			CPU_ClearJob( CPU_STOP_RUNNING );
			gCPURunning = false;
		}
		else if (gCPUState.StuffToDo & CPU_CHANGE_CORE)
		{
			CPU_ClearJob( CPU_CHANGE_CORE );
		}
		// Clear gCPUState.StuffToDo?
	}

	CPU_EXECUTE_OP< false, false, true >();

	bool rsp_halted( (Memory_SP_GetRegister( SP_STATUS_REG ) & SP_STATUS_HALT) != 0 );

	if ( !rsp_halted )
	{
		RSP_Step();
	}

	DBGConsole_UpdateDisplay();
}


//*****************************************************************************
// Keep executing instructions until there are other tasks to do (i.e. gCPUState.StuffToDo is set)
// Process these tasks and loop
//*****************************************************************************
template < bool DynaRec, bool TraceEnabled, bool RunRSP > void CPU_Go()
{
	DAEDALUS_PROFILE( __FUNCTION__ );

	while (gCPURunning)
	{
		//
		// Keep executing ops as long as there's nothing to do
		//
		while (gCPUState.StuffToDo == 0)
		{	
			if ( RunRSP )
			{
				RSP_Step();						// At this point we know that the RSP is running. RSPStep() asserts that it is not halted
			}
			CPU_EXECUTE_OP< DynaRec, TraceEnabled, false >();
		}

	#ifdef DAEDALUS_ENABLE_DYNAREC
		if( TraceEnabled && (gCPUState.StuffToDo != CPU_CHANGE_CORE) )
		{
			if(gTraceRecorder.IsTraceActive())
			{
				#ifdef DAEDALUS_DEBUG_DYNAREC
					u32 start_address( gTraceRecorder.GetStartTraceAddress() );
					//DBGConsole_Msg( 0, "Aborting tracing of [R%08x] - StuffToDo is %08x", start_address, gCPUState.StuffToDo );

					gAbortedTraceReasons[ start_address ] = gCPUState.StuffToDo;
				#endif

				#ifdef ALLOW_TRACES_WHICH_EXCEPT
				if(gCPUState.StuffToDo == CPU_CHECK_INTERRUPTS )
				{
					//DBGConsole_Msg( 0, "Adding chunk at %08x after interrupt\n", gTraceRecorder.GetStartTraceAddress() );
					gTraceRecorder.StopTrace( gCPUState.CurrentPC );
					CPU_CreateAndAddFragment();
				}
				#endif

				gTraceRecorder.AbortTrace();		// Abort any traces that were terminated through an interrupt etc
			}
			CPU_SelectCore();
		}
	#endif
		
		//
		// Handle whatever it was that cause us to stop
		//
		if (gCPUState.StuffToDo & CPU_CHECK_INTERRUPTS)
		{
			R4300_Handle_Interrupt();
			CPU_ClearJob( CPU_CHECK_INTERRUPTS );
		}
		else if (gCPUState.StuffToDo & CPU_CHECK_EXCEPTIONS)
		{
			R4300_Handle_Exception();
			CPU_ClearJob( CPU_CHECK_EXCEPTIONS );
		}
		else if (gCPUState.StuffToDo & CPU_STOP_RUNNING)
		{
			CPU_ClearJob( CPU_STOP_RUNNING );
			gCPURunning = false;
		}
		else if (gCPUState.StuffToDo & CPU_CHANGE_CORE)
		{
			// Break out of this loop. The new core is selected automatically
			CPU_ClearJob( CPU_CHANGE_CORE );
			break;
		}
	}
}

//*****************************************************************************
//	Return true if the CPU is running
//*****************************************************************************
bool	CPU_IsRunning()
{
	return gCPURunning;
}

//*****************************************************************************
//	Return the total number of vi interrupts
//*****************************************************************************
u32	CPU_GetVerticalInterruptCount()
{
	return gVerticalInterrupts;
}
