/*
-------------------------------------------------------------------
Snezziboy v0.24

Copyright (C) 2006 bubble2k

This program is free software; you can redistribute it and/or 
modify it under the terms of the GNU General Public License as 
published by the Free Software Foundation; either version 2 of 
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of 
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
GNU General Public License for more details.
-------------------------------------------------------------------
*/

@*************************************************************************
@ All code here in ROM
@*************************************************************************

    .align 4
    .ascii  ".ROM"
    .align 4

@=========================================================================
@ where the config will be stored
@=========================================================================
    .equ    configSRAMBase,     0x0e00fff0
    .equ    configBG0Priority,  (configSRAMBase+0)
    .equ    configBG1Priority,  (configSRAMBase+1)
    .equ    configBG2Priority,  (configSRAMBase+2)
    .equ    configBG3Priority,  (configSRAMBase+3)
    .equ    configBackdrop,     (configSRAMBase+4)
    .equ    configBGEnable,     (configSRAMBase+5)
    .equ    configBGForceMode,  (configSRAMBase+6)
    .equ    configCTRL,         (configSRAMBase+7)
    .equ    configA,            (configSRAMBase+8)
    .equ    configB,            (configSRAMBase+9)
    .equ    configX,            (configSRAMBase+10)
    .equ    configY,            (configSRAMBase+11)
    .equ    configBGPrioritySet,(configSRAMBase+12)
    

@-------------------------------------------------------------------------
@ Set value in memory
@-------------------------------------------------------------------------
.macro SetDebugValue addr, reg
    mov     r1, \reg
    mov     r0, #\addr
	str		r1, [r2, r0]
.endm


@-------------------------------------------------------------------------
@ Set string in memory
@-------------------------------------------------------------------------
.macro SetDebugText addr, t
    ldr     r0, =debugMemoryBase
    add     r0, r0, \addr
    ldr     r1, =2f
1:
    ldrb    r2, [r1]
    cmp     r2, #0
    beq     3f
    strb    r2, [r0]
    add     r1, r1, #1
    add     r0, r0, #1
    b       1b
2:
    .asciz  "\t"
    .align  4
3:
.endm

.macro SetDebugTextP addr, t
    stmfd   sp!, {r0, r1, r2}
    ldr     r0, =debugMemoryBase
    add     r0, r0, \addr
    ldr     r1, =2f
1:
    ldrb    r2, [r1]
    cmp     r2, #0
    beq     3f
    strb    r2, [r0]
    add     r1, r1, #1
    add     r0, r0, #1
    b       1b
2:
    .asciz  "\t"
    .align  4
3:
    ldmfd   sp!, {r0, r1, r2}
.endm

@--------------------------------------------------------
@ set the processor state in memory for vbadump
@--------------------------------------------------------

.ifeq debug-1
SetDebugState_UpdateState:
    ldr     r2, =debugMemoryBase

    tsts	SnesMXDI, #SnesFlagM
    moveq	r0, SnesA, lsr #16
    movne	r0, SnesA, lsr #24
    ldrne   r1, =SnesB
    ldrne   r1, [r1]
    orrne   r0, r0, r1, lsr #16
    SetDebugValue	0x00, r0
    SetDebugValue	0x04, SnesX
    SetDebugValue 	0x08, SnesY
    mov		r0, SnesD, lsr #16
    SetDebugValue	0x0C, r0
    mov		r0, SnesSP, lsr #16
    SetDebugValue	0x10, r0
    ComposeP
    tsts	SnesMXDI, #SnesFlagE
    orrne   r1, r1, #0x100
    SetDebugValue	0x18, r1
    
    mov     r0, #0
    ldrb    r1, [SnesPC, #3]
    add     r0, r1, r0, lsl #8
    ldrb    r1, [SnesPC, #2]
    add     r0, r1, r0, lsl #8
    ldrb    r1, [SnesPC, #1]
    add     r0, r1, r0, lsl #8
    ldrb    r1, [SnesPC, #0]
    add     r0, r1, r0, lsl #8
    SetDebugValue   0x20, r0
    
    mov     r1, SnesDBR
    and		r1, r1, #0xff
    SetDebugValue	0x24, r1
    
    mov     r1, SnesPC
    ldr     r0, =ROMOffset
    ldr     r0, [r0]
    sub     r1, r1, r0
    SetDebugValue	0x30, r1
    
    
    mov     r0, SnesCYCLES, asr #CYCLE_SHIFT
    ldr     r1, =regROMAccess
    ldrb    r1, [r2]
    tsts    r1, #1
    addeq   r0, r0, #CYCLES_PER_SCANLINE
    addne   r0, r0, #CYCLES_PER_SCANLINE_FAST
    SetDebugValue	0x28, r0
    ldr     r0, =VerticalCount
    ldr     r0, [r0]
    ldr     r1, =NUM_SCANLINES
    add     r0, r0, r1
    SetDebugValue	0x2c, r0
    
    mov     r1, r3
    SetDebugValue	0x14, r1

    ldr     r2, [r2, #0x7C]             @ r2 = breakpoint
    mov     r2, r2, lsr #8
    ldr     pc, =SetDebugState_Step
.endif

    .ltorg

@=========================================================================
@ Hard reset
@=========================================================================

HardReset:
    @ disable force blank
    ldr     r2, =0x04000000
    ldrh    r0, [r2]
    bic     r0, r0, #0x80
    strh    r0, [r2]

    @ copy first 2 banks of the memory map to DPCache
    ldr     r1, =DPCache
    ldr     r2, =MemoryMap
    mov     r3, #16
DPCacheLoop:
    ldr     r0, [r2], #4
    str     r0, [r1], #4
    subs    r3, r3, #1
    bne     DPCacheLoop

    @ Set the ROM access prefetch and waitstates
    @
    ldr     r0,=0x04000204
    ldr     r2,=0x08000004
    ldr     r3,=magicNumber
    
    ldr     r1,=0x00004018          @ 2/1 cycles on 0x08000000 with prefetch (fast)
    strh    r1, [r0]
    ldr     r1, [r2]
    cmp     r1, r3
    beq     ROMWaitStateOk

    ldr     r1,=0x00004000          @ 4/2 cycles on 0x08000000 with prefetch (faster)
    strh    r1, [r0]
    ldr     r1, [r2]
    cmp     r1, r3
    beq     ROMWaitStateOk
   
    ldr     r1,=0x00000000          @ 4/2 cycles on 0x08000000 with no prefetch (slowest!)
    strh    r1, [r0]

ROMWaitStateOk:

ClearRAM:
    ldr     r0, =0x55555555
    ldr     r2, =snesWramBase
    ldr     r1, =0x00020000
ClearRAMLoop:
    str     r0, [r2], #4
    subs    r1, r1, #4
    bne     ClearRAMLoop

ClearVRAM:
    ldr     r0, =0x00000000
    ldr     r2, =snesVramBase
    ldr     r1, =0x00010000
ClearVRAMLoop:
    str     r0, [r2], #4
    subs    r1, r1, #4
    bne     ClearVRAMLoop

    @ Set the RAM access prefetch and waitstates
    @
    ldr     r0,=0x04000800
    ldr     r2,=0x02000000
    ldr     r3,=magicNumber
    str     r3, [r2]

    ldr     r1, [r0]
    bic     r1, r1, #0x0f000000     @ 4/2 cycles
    orr     r1, r1, #0x0e000000
    str     r1, [r0]
    
    ldr     r1, [r2]
    cmp     r1, r3
    beq     EWRAMWaitStateOk

    ldr     r1, [r0]
    bic     r1, r1, #0x0f000000     @ 4/2 cycles
    orr     r1, r1, #0x0d000000
    str     r1, [r0]

EWRAMWaitStateOk:

    @ set up interrupt
    @
    ldr     r0, =0x03007FFC
    ldr     r1, =gbaInterrupt
    str     r1, [r0]
    
    ldr     r0, =0x04000004
    mov     r1, #0x18           @ enable both V/HBlank
    str     r1, [r0]
    
    ldr     r0, =0x04000200
    mov     r1, #0x3            @ enable both V/HBlank
    str     r1, [r0]
    
    ldr     r0, =0x04000208
    mov     r1, #0x1            @ enable both V/HBlank
    str     r1, [r0]
    
    @ set up global sprite scaling
    @
    mov     r1, #0
    ldr     r3, =0xFF01         @ r3 = -1 (for scaling)
    ldr     r4, =0x0100         @ r3 = +1 (for scaling)
    ldr     r5, =0x0150         @ r3 = +1.4 (for scaling)
    ldr     r6, =0xFEB0         @ r3 = -1.4 (for scaling)
    
    ldr     r0, =0x07000006     @ rot/scale group 0x00:
    strh    r4, [r0], #8        @ pa = 1.0
    strh    r1, [r0], #8        @ pb = 0
    strh    r1, [r0], #8        @ pc = 0    
    strh    r5, [r0], #8        @ pd = 1/1.4

    strh    r3, [r0], #8        @ pa = -1.0 (group 0x01, h-flip)
    strh    r1, [r0], #8        @ pb = 0
    strh    r1, [r0], #8        @ pc = 0    
    strh    r5, [r0], #8        @ pd = 1/1.4
    
    strh    r4, [r0], #8        @ pa = 1.0 (group 0x02, v-flip)
    strh    r1, [r0], #8        @ pb = 0
    strh    r1, [r0], #8        @ pc = 0    
    strh    r6, [r0], #8        @ pd = -1/1.4
    
    strh    r3, [r0], #8        @ pa = -1.0 (group 0x03, v-flip, h-vlip)
    strh    r1, [r0], #8        @ pb = 0
    strh    r1, [r0], #8        @ pc = 0    
    strh    r6, [r0], #8        @ pd = -1/1.4
    
SoftReset:
    @ version 0.23 fix
    @
    @ check the save ram for config options
    @ if the values are greater than their 
    @ allowable range, then set them to default
    @
    ldr     r0, =configSRAMBase
    ldr     r2, =configOptionCount
    ldr     r3, =configDefaultOption
    mov     r8, #(optionCount-2)

checkOption_Loop:
    ldrb    r1, [r0], #1
    ldr     r4, [r2], #4
    ldr     r5, [r3], #4
    cmp     r1, r4
    strgeb  r5, [r0, #-1]
    subs    r8, r8, #1
    bne     checkOption_Loop
    

SetInterruptVectors:
    bl      configUpdateKeyMap
    bl      configUpdateMisc

    @ set the current frame number
    @
    ldr     r0, =bgOffsetCurrFrame
    mov     r1, #1
    str     r1, [r0]
    
    @ get all the interrupt vectors
    @
    ldr     r0, =VECTOR_COUNTRY
    TranslateAddress
    ldrb    r0, [r0]
    cmp     r0, #1
    movgt   r1, #1
    movle   r1, #0
    ldr     r2, =regPALNTSC
    strb    r1, [r2]

    ldr     r0, =VECTOR_NMI
    TranslateAddress
    ldr     r1, =NMIaddress
    str     r0, [r1]

    ldr     r0, =VECTOR_BRK
    TranslateAddress
    ldr     r1, =BRKaddress
    str     r0, [r1]

    ldr     r0, =VECTOR_COP
    TranslateAddress
    ldr     r1, =COPaddress
    str     r0, [r1]

    ldr     r0, =VECTOR_IRQ
    TranslateAddress
    ldr     r1, =IRQaddress
    str     r0, [r1]

    @ get the reset vector and set the PC
    @
    ldr     r0, =VECTOR_RESET
    TranslateAddress
    ReadAddr16
    TranslateAndSavePC
    
    @ get the ScanlineEnd code and save it
    ldr     r2, =ScanlineEnd
    ldr     r0, [r2]
    ldr     r1, =ScanlineEnd_Code
    str     r0, [r1]
    mov     r0, #0
    str     r0, [r2]
    
    ldr     r0, =0x00c00000
    
    ldr     r0, =SCANLINE_BLANK       @ if overscan=0, vblank = 225
    sub     r0, r0, #255
    sub     r0, r0, #7
    ldr     r1, =vBlankScan
    str     r0, [r1]
    
    ldr     SnesSP, =0x01FF0000         @ SP = ???, PBR = 0
    mov     SnesD, #0                   @ D = 0, DBR = 0
    mov     SnesX, #0
    mov     SnesY, #0
    mov     SnesA, #0
    mov     SnesCV, #0
    mov     SnesNZ, #1
    mov     SnesMXDI, #0
    mov     SnesCYCLES, #0
    
    @ update the current scanline/cycle
    @
    ldr     r0, =(CYCLES_PER_SCANLINE << CYCLE_SHIFT)
    sub     SnesCYCLES, SnesCYCLES, r0

    ldr     r0, =(-NUM_SCANLINES)
    ldr     r1, =VerticalCount
    str     r0, [r1]

    @ modify the RAM code for IRQs
    @
    ldr     r0, =IRQJumpCode
    ldr     r1, =IRQJump1
    ldr     r1, [r1]
    str     r1, [r0], #4
    ldr     r1, =IRQJump2
    ldr     r1, [r1]
    str     r1, [r0], #4
    
    mov     r0, #0
    ldr     r1, =IRQJump1
    str     r0, [r1]
    ldr     r1, =IRQJump2
    str     r0, [r1]
    
    @ modify jump for mode 7 render
    @
    ldr     r0, =codeRenderMode7Reg
    ldr     r1, =snesRenderScreen_RenderMode7Reg
    ldr     r2, [r1]
    str     r2, [r0]
    mov     r2, #0
    str     r2, [r1]

    ldr     EmuDecoder, =m1x1Decoder

    @ set emulation bit as on; m=1, x=1, d=0, i=1, cycles=0, e=1, romSpeed = 8 (slowROM)
    orr     SnesMXDI, SnesMXDI, #(SnesFlagE+SnesFlagM+SnesFlagX+SnesFlagI)      

    @ set the memory mapper
    ldr     SnesMemMap, =MemoryMap

    CacheMemoryMap

    .ifeq debug-1
        SetDebugText    #0x70, "LINE----"    @ for the vba dump to trace
        SetDebugText    #0x78, "BRK0" 	    @ for the vba dump to trace
    .endif

    subs    SnesCYCLES, SnesCYCLES, #0
    ldr     pc, =Fetch

    .ltorg


@=========================================================================
@ Screen render routines
@=========================================================================

    .equ    COLOR_256,      3
    .equ    COLOR_16,       2
    .equ    COLOR_4,        1
    .equ    COLOR_NONE,     0  

ModeRender:
    .long       RenderMode0, RenderMode1, RenderMode2, RenderMode3
    .long       RenderMode4, RenderMode5, RenderMode6, RenderMode7

@-------------------------------------------------------------------------
@ start the renderer
@-------------------------------------------------------------------------
.macro  StartRenderer
    stmfd   sp!, {r3-r9, lr}
.endm

@-------------------------------------------------------------------------
@ finalize the renderer
@-------------------------------------------------------------------------
RenderEndRenderer:
    @.ifeq   debug-1
        SetDebugTextP   #0x100, "BGMODE"
        ldr     r0, =debugMemoryBase
        add     r0, r0, #0x108
        ldr     r1, =regBGMode
        ldrb    r1, [r1]
        strb    r1, [r0]
        
        SetDebugTextP   #0x110, "BGxSC"
        ldr     r1, =screenCurr2
        ldr     r1, [r1]
        ldr     r0, =debugMemoryBase+0x118
        ldr     r2, =0x7c7c7c7c
        and     r1, r1, r2
        mov     r1, r1, lsr #2
        str     r1, [r0]

        SetDebugTextP   #0x120, "BGxNBA"
        ldr     r1, =screenCurr3
        ldr     r1, [r1]
        ldr     r0, =debugMemoryBase+0x128
        mov     r1, r1, lsl #2
        str     r1, [r0]

        SetDebugTextP    #0x130, "TYPE/BG#/OBJ"
        
        ldr     r1, =VRAMWrite
        ldr     r2, =VRAMObjWrite
        ldr     r3, =VRAMBG
        
        ldr     r4, =VRAMWriteTileMap
        ldr     r5, =VRAMWriteBGChar
        ldr     r6, =VRAMWriteNOP
        
        mov     r9, #0x140
        mov     r8, #0
6:
        SetDebugTextP    r9, "----/---/---"
        ldr     r7, [r1], #4
        cmp     r7, r4          @ is VRAMWrite[r1] == VRAMWriteTileMap?
        bne     4f
        SetDebugTextP    r9, "TILE"
4:
        cmp     r7, r5
        bne     4f
        SetDebugTextP    r9, "CHAR"
4:
        add     r9, r9, #5
        
        cmp     r7, r6
        beq     5f

        ldrb    r7, [r3]
        cmp     r7, #0
        bne     4f
        SetDebugTextP    r9, "BG0"
4:
        cmp     r7, #1
        bne     4f
        SetDebugTextP    r9, "BG1"
4:
        cmp     r7, #2
        bne     4f
        SetDebugTextP    r9, "BG2"
4:
        cmp     r7, #3
        bne     4f
        SetDebugTextP    r9, "BG3"
4:
5:
        add     r9, r9, #4
        
        ldr     r7, [r2], #4
        cmp     r7, r6
        beq     4f
        SetDebugTextP    r9, "OBJ"
4:
        add     r3, r3, #1
        add     r9, r9, #7
        add     r8, r8, #1
        cmp     r8, #32
        bne     6b
    @.endif

    ldmfd   sp!, {r3-r9, lr}
    bx      lr

.macro  EndRenderer
    b       RenderEndRenderer
.endm

RenderCopyBGCopyCharTable:
    .word   VRAMWriteNOP, CopyChar_4color, CopyChar_16color, CopyChar_256color

RenderCopyCharUnpackTable:
    .word   0, charUnpack4, charUnpack4, charUnpack8

    SetText "EnableBG"
@-------------------------------------------------------------------------
@ Enable backgrounds according to flags
@-------------------------------------------------------------------------
RenderDisableBG:
    ldr	    r0, =0x04000000			@ disable all BGs and OBJs first
    ldrh    r1, [r0]
    bic     r1, r1, #0xff00
    strh    r1, [r0]

    ldr     r4, =0x06010000
    ldr     r1, =bgCurTileOffset
    str     r4, [r1]

    ldr     r0, =0xffffffff             @ clear all VRAM Jump/bg/color tables
    ldr     r1, =VRAMBGColors
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    
    ldr     r1, =VRAMBG
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4
    str     r0, [r1], #4

    ldr     r0, =VRAMWrite
    ldr     r1, =VRAMWriteNOP
    mov     r2, #16

ClearVRAMWriteLoop:                             @ clears both VRAM/OBJ table
    str     r1, [r0], #4
    str     r1, [r0], #4
    str     r1, [r0], #4
    str     r1, [r0], #4
    subs    r2, r2, #1
    bne     ClearVRAMWriteLoop
    
    bx      lr

RenderEnableBG:
    ldr     r2, =0x04000000 
    ldrh    r0, [r2]
    
    ldr     r1, =regMainScreenCopy2
    ldrb    r1, [r1]
    ldr     r2, =regSubScreenCopy2
    ldrb    r2, [r2]
    orr     r1, r1, r2
    ldr     r2, =regBGMode
    ldrb    r2, [r2]
    cmp     r2, #7
    andeq   r2, r1, #0x01
    orreq   r1, r1, r2, lsl #2

    bic     r0, r0, #0x1f00
    and     r1, r1, #0x1f
    orr     r0, r0, r1, lsl #8
    and     r0, r0, r5

    bic     r0, r0, #0x7
    orr     r0, r0, r6

    ldr     r1, =configBG0Priority
    ldrb    r1, [r1]
    cmp     r1, #04
    biceq   r0, r0, #(1<<8)

    ldr     r1, =configBG1Priority
    ldrb    r1, [r1]
    cmp     r1, #04
    biceq   r0, r0, #(1<<9)

    ldr     r1, =configBG2Priority
    ldrb    r1, [r1]
    cmp     r1, #04
    biceq   r0, r0, #(1<<10)

    ldr     r1, =configBG3Priority
    ldrb    r1, [r1]
    cmp     r1, #04
    biceq   r0, r0, #(1<<11)

    ldr     r2, =0x04000000 
    strh    r0, [r2]

    bx      lr

.equ    EnableBG0,  (1<<8)
.equ    EnableBG1,  (1<<9)
.equ    EnableBG2,  (1<<10)
.equ    EnableBG3,  (1<<11)
.equ    EnableOBJ,  (1<<12)

.macro  EnableBG    BGs, Mode=0
    mov     r5, #\BGs
    mov     r6, #\Mode
    bl      RenderEnableBG
.endm

.macro  DisableBG
    bl      RenderDisableBG
.endm

    SetText "CopyBGCHR"
@-------------------------------------------------------------------------
@ Copy BG CHAR 
@   r7 = number of colors for bg3, bg2, bg1, bg0
@        (0 = no background, 1 = 4 color, 2 = 16 color, 3 = 256 color)
@ destroys r3-r8
@-------------------------------------------------------------------------

RenderCopyBGChar2:
    @ set up the VRAM BG and BG color table 
    mov     r8, #32
    mov     r2, #0xff
    mov     r3, #0xff
    ldr     r4, =VRAMBG
    ldr     r5, =VRAMBGColors
    
RenderCopyBGChar2_Loop:
    ldrb    r6, [r4]
    cmp     r6, #0xff
    movne   r2, r6
    ldrneb  r3, [r5]
    strb    r2, [r4], #1
    strb    r3, [r5], #1
    subs    r8, r8, #1
    bne     RenderCopyBGChar2_Loop
    
    @ SNES character address
    mov     r0, #0

RenderCopyBGChar2_Loop2:

    ldr     r1, =configBGEnable
    ldrb    r1, [r1]
    tsts    r1, r1
    bne     1f
    ldr     r1, =regMainScreenCopy2
    ldrb    r1, [r1]
    ldr     r2, =regSubScreenCopy2
    ldrb    r2, [r2]
    orr     r1, r1, r2
    ldr     r2, =regBGMode
    ldrb    r2, [r2]
    cmp     r2, #7
    andeq   r2, r1, #0x01
    orreq   r1, r1, r2, lsl #2

    mov     r2, #1
    tsts    r1, r2, lsl r0
    beq     RenderCopyBGChar2_SkipLoop      @ skip if background is not activated in main/sub screen
1:

    ldr     r1, =regBG1NBA
    ldrb    r1, [r1, r0]
    and     r1, r1, #0x07                   @ r6 = 00000aaa
    mov     r6, r1                          @ r6 = 00000aaa
    mov     r1, r1, lsl #13
    add     r1, r1, #(snesVramBase&0xff000000)
    add     r1, r1, #(snesVramBase&0x00ff0000)

    and     r5, r7, #0xff
    ldr     r2, =RenderCopyCharUnpackTable
    ldr     r2, [r2, r5, lsl #2]

    ldr     r3, =bg1VRAMOffset
    ldr     r3, [r3, r0, lsl #2]
    add     r3, r3, r6, lsl #13

    stmfd   sp!, {r0, lr}
    ldr     r4, =RenderCopyBGCopyCharTable
    mov     lr, pc
    ldr     pc, [r4, r5, lsl #2]
    ldmfd   sp!, {r0, lr}

RenderCopyBGChar2_SkipLoop:
    add     r0, r0, #1
    cmp     r0, #4
    mov     r7, r7, lsr #8
    bne     RenderCopyBGChar2_Loop2
    bx      lr


.macro CopyBGCharEx  bg1Colors, bg2Colors, bg3Colors, bg4Colors
    ldr     r7, =((\bg4Colors)*256*256*256 + (\bg3Colors)*256*256 + (\bg2Colors)*256 + (\bg1Colors))
    bl      RenderCopyBGChar2
.endm


    SetText "CopyBGTile"
@-------------------------------------------------------------------------
@ Copy BG TileMap
@   r0: bgNumber
@-------------------------------------------------------------------------
RenderCopyBGTileMap:
    ldr     r1, =ScreenMode
    strb    r0, [r1]

    ldr     r1, =configBGEnable
    ldrb    r1, [r1]
    tsts    r1, r1
    bne     1f
    ldr     r1, =regMainScreenCopy2
    ldrb    r1, [r1]
    ldr     r2, =regSubScreenCopy2
    ldrb    r2, [r2]
    orr     r1, r1, r2
    ldr     r2, =regBGMode
    ldrb    r2, [r2]
    cmp     r2, #7
    andeq   r2, r1, #0x01
    orreq   r1, r1, r2, lsl #2
    
    mov     r2, #1
    tsts    r1, r2, lsl r0
    bxeq    lr                              @ skip if background is not activated in main/sub screen
1:
  
    cmp     r0, #2                          @ if this is BG3, 
    moveq   r9, #0x8000                     @ add to the tile map palette
    movne   r9, #0x0000

    stmfd   sp!, {lr}
    ldr     r1, =regBG1SC
    ldrb    r1, [r1, r0]                    @ r1 = 00000000 ttttttyx
    and     r1, r1, #0x7f                   @ r1 = 00000000 0tttttyx
    mov     r0, r1, lsr #2                  @ r0 = 00000000 000ttttt
    and     r7, r1, #0x03                   @ r7 = 00000000 000000yx

    ldr     r5, =VRAMWrite
    @add     r5, r5, r0, lsl #2

    mov     r0, r0, lsl #11                 @ r0 = ttttt000 00000000
    mov     r6, r0

    ldr     r1, =ScreenMode
    ldrb    r1, [r1]
    ldr     r4, =VRAMWriteTileMap
    str     r4, [r5, r6, lsr #9]
    ldr     r4, =VRAMBG
    strb    r1, [r4, r6, lsr #11]
    mov     lr, pc
    ldr     pc, =CopyTileMap
    tsts    r7, #0x03
    beq     CopyBGTileMap_End

    ldr     r1, =ScreenMode
    ldrb    r1, [r1]
    add     r0, r6, #0x800
    bic     r0, r0, #0x00ff0000             @ version 0.24 fix
    ldr     r4, =VRAMWriteTileMap
    str     r4, [r5, r6, lsr #9]
    ldr     r4, =VRAMBG
    strb    r1, [r4, r6, lsr #11]
    mov     lr, pc
    ldr     pc, =CopyTileMap
    cmp     r7, #0x03
    bne     CopyBGTileMap_End

    ldr     r1, =ScreenMode
    ldrb    r1, [r1]
    add     r0, r6, #0x1000
    bic     r0, r0, #0x00ff0000             @ version 0.24 fix
    ldr     r4, =VRAMWriteTileMap
    str     r4, [r5, r6, lsr #9]
    ldr     r4, =VRAMBG
    strb    r1, [r4, r6, lsr #11]
    mov     lr, pc
    ldr     pc, =CopyTileMap

    ldr     r1, =ScreenMode
    ldrb    r1, [r1]
    add     r0, r6, #0x1800
    bic     r0, r0, #0x00ff0000             @ version 0.24 fix
    ldr     r4, =VRAMWriteTileMap
    str     r4, [r5, r6, lsr #9]
    ldr     r4, =VRAMBG
    strb    r1, [r4, r6, lsr #11]
    mov     lr, pc
    ldr     pc, =CopyTileMap

CopyBGTileMap_End:

    ldmfd   sp!, {lr}
    bx      lr

.macro  CopyBGTileMap   bgNumber
    mov     r0, #\bgNumber
    bl      RenderCopyBGTileMap
.endm

    SetText "CopyOBJCHR"
@-------------------------------------------------------------------------
@ Copy OBJ char
@   r8: nameAddress (0, 1)
@-------------------------------------------------------------------------
RenderCopyOBJChar:
    ldr     r0, =regObSel
    ldrb    r0, [r0]
    mov     r1, r0
    and     r0, r0, #0x07           @ r0 = -----aaa
    mov     r0, r0, lsl #3          @ r0 = --aaa000
    
    tsts    r8, #1
    andne   r5, r1, #0x18           @ r5 = ---nn---
    addne   r5, r5, #0x08           @ r5+= 00001000
    addne   r0, r0, r5, lsr #1      @ r0 = --aa*n00
    and     r0, r0, #0x1f           @ r0 = --0a*n00

    ldr     r1, =VRAMBGColors
    add     r1, r1, r0

    ldr     r4, =VRAMWrite
    add     r4, r4, r0, lsl #2

    ldr     r6, =VRAMWriteNOP
    mov     r2, #4
1:
    ldrb    r3, [r1, r2]
    cmp     r3, #COLOR_16
    strne   r6, [r4, r2, lsl #2]
    subs    r2, r2, #1
    bne     1b

    ldr     r1, =VRAMObjWrite
    add     r1, r1, r0, lsl #2
    
    tsts    r8, #1
    ldrne   r6, =VRAMWriteObj1Char_16color
    ldreq   r6, =VRAMWriteObj0Char_16color

    str     r6, [r1], #4
    str     r6, [r1], #4
    str     r6, [r1], #4
    str     r6, [r1], #4

    ldr     r2, =charUnpack4
    ldr     r1, =snesVramBase
    add     r1, r1, r0, lsl #11
    
    ldr     r3, =0x06010000
    tsts    r8, #1
    addne   r3, r3, #0x00004000

    sub     r4, r3, r0, lsl #11
    ldreq   r7, =VRAMObj0Offset
    ldrne   r7, =VRAMObj1Offset
    str     r4, [r7]

    stmfd   sp!, {lr}
    mov     lr, pc
    ldr     pc, =CopyObjChar_16color
    ldmfd   sp!, {lr}
    bx      lr

.macro  CopyOBJChar nameAddress
    mov     r8, #\nameAddress
    bl      RenderCopyOBJChar
.endm

.macro  CopyOBJCharEx nameAddress
    mov     r8, #\nameAddress
    bl      RenderCopyOBJChar
.endm

    .ltorg

    SetText "CopyBGCNT"

@-------------------------------------------------------------------------
@ Copy BG control bits and allocate GBA VRAM for snes background
@   r0 = bgNumber (0-3)
@   r2 = priority (0-3)
@   r6 = numColors (0=na, 1=4, 2=16, 3=256)
@ destroys r3-r8
@-------------------------------------------------------------------------
RenderCopyBGCNT:
    ldr     r1, =NumColors
    strb    r6, [r1]
    
    ldr     r1, =configBGEnable
    ldrb    r1, [r1]
    tsts    r1, r1
    bne     1f
    ldr     r1, =regMainScreenCopy2
    ldrb    r1, [r1]
    ldr     r2, =regSubScreenCopy2
    ldrb    r2, [r2]
    orr     r1, r1, r2
    ldr     r2, =regBGMode
    ldrb    r2, [r2]
    cmp     r2, #7
    andeq   r2, r1, #0x01
    orreq   r1, r1, r2, lsl #2
    
    mov     r2, #1
    tsts    r1, r2, lsl r0
    bxeq    lr                              @ skip if background is not activated in main/sub screen
1:

    ldr     r3, =configBG0Priority          @ override the default SNES priority
    ldrb    r3, [r3, r0]
    cmp     r3, #0x4
    movlt   r2, r3                          @ r2 = 00000000 000000pp

    cmp     r6, #COLOR_256                  
    orreq   r2, r2, #(1 << 7)
    orrne   r2, r2, #(0 << 7)               @ r2 = 00000000 c00000pp

    ldr     r1, =regBG1NBA
    ldrb    r1, [r1, r0]                    @ r1 = 0000aaaa
    and     r1, r1, #0x07
    mov     r7, r1                          @ r1, r7 = 00000aaa (stores the SNES VRAM 8k-block)

    @ set up the VRAM write table for the BG char
    @
    ldr     r4, =VRAMBGColors
    strb    r6, [r4, r7, lsl #2]
    ldr     r4, =VRAMBG
    strb    r0, [r4, r7, lsl #2]
    ldr     r4, =VRAMWrite
    mov     r7, r7, lsl #2
    mov     r8, #16
1:  ldr     r6, [r4, r7, lsl #2]
    ldr     r5, =VRAMWriteNOP
    cmp     r6, r5
    ldr     r5, =VRAMWriteBGChar
    streq   r5, [r4, r7, lsl #2]
    add     r7, r7, #1
    cmp     r7, #32
    bge     2f
    subs    r8, r8, #1
    bne     1b
2:  

    @ determine the 16-k blocks in GBA VRAM for the character maps.
    @ with this complex algorithm... 
    @ makes me want to move over to DS.
    @
    mov     r7, r1
    ldr     r5, =regBG1NBA
    ldrb    r5, [r5]
    ldr     r6, =regBG2NBA
    ldrb    r6, [r6]
    ldr     r8, =regBG3NBA
    ldrb    r8, [r8]

    cmp     r0, #2                          @ are we doing BG3?
    beq     AllocateBG3
    
    @-------------------------------------
    @ Allocate BG1 / BG2    
    @-------------------------------------
    cmp     r5, r6                          @ but is BG1 != BG2?
    bne     BG1_NEQ_BG2

    @ BG1 == BG2
    @
    mov     r1, #0
    b       EndAllocate
    
BG1_NEQ_BG2:
    
    @ by default set BG1 to block #0, BG2 to block #1
    mov     r1, r0
    
    @ here BG1 <> BG2 
    @
    sub     r3, r6, r5                      @ if NBA( BG2 - BG1 ) = 2
    cmp     r3, #2
    moveq   r1, r0                          @ if so, set BG1 = block #0, BG2 = block #1
    beq     EndAllocate
    
    sub     r3, r5, r6                      @ if NBA( BG1 - BG2 ) = 2
    cmp     r3, #2
    moveq   r1, r0                          @ if so, set BG1 = block #1, BG2 = block #0
    eoreq   r1, r1, #0x1
    beq     EndAllocate

    @ here BG1 <> BG2, so check for the mode >= 2, 
    @ 
    ldr     r3, =regBGMode
    ldrb    r3, [r3]
    and     r3, r3, #0x7

    ldr     r4, =regMainScreenCopy2
    ldrb    r4, [r4]
    ldr     r8, =regSubScreenCopy2
    ldrb    r8, [r8]
    orr     r4, r4, r8
    
    cmp     r3, #1
    beq     BG1_NEQ_BG2_Mode1
    cmp     r3, #2              
    bgt     BG1_NEQ_BG2_Mode2AndAbove
    b       EndAllocate
    
BG1_NEQ_BG2_Mode1:
    tsts    r4, #4                          @ check if BG2 is enabled
    beq     BG1_NEQ_BG2_Mode1_BG2Disabled
    
    b       EndAllocate
    
BG1_NEQ_BG2_Mode1_BG2Disabled:
    ldr     r3, =configBGEnable
    ldrb    r3, [r3]
    cmp     r3, #0
    movne   r1, r0
    moveq   r1, r0, lsl #1
    b       EndAllocate
    
BG1_NEQ_BG2_Mode2AndAbove:
    mov     r1, r0, lsl #1                  @ if so, allocate block #0 for BG1, block #2 for BG2
    b       EndAllocate
    
    @ otherwise... 
    @
    mov     r1, r0                          @ generic, allocate block #0 for BG1, block #1 for BG2
    beq     EndAllocate

AllocateBG3:

    @ by default set BG3 to block #2
    mov     r1, #2

    @-------------------------------------
    @ Allocate BG3
    @-------------------------------------
    @
    @ here we try to decide if BG3 should be moved on into block #1 instead (because
    @ BG1/2 may take up a small space)
    
    @ first , we check if BG3's character address - BG1's address is exactly 2
    @ if so, allocate block #1
    @
    sub     r3, r8, r5
    cmp     r3, #2
    moveq   r1, #1
    beq     EndAllocate
    
    @ check if BG1 = BG2?
    @
    cmp     r5, r6                          @ if NBA for BG1 != BG2, then we can never move BG3 to block #1
    bne     EndAllocate
    
    @ ok, BG1 == BG2
    @ so, check if the OBJ base starts at middle of BG1 
    @ 
    ldr     r8, =regObSel
    ldrb    r8, [r8]
    and     r8, r8, #0x03
    mov     r8, r8, lsl #1
    sub     r3, r8, r5
    cmp     r3, #2                          @ if it is, assume that BG3 can be moved to block #1.
    bne     AllocateBG3_CheckLoop
    mov     r1, #1                          @ (will not work if BG1 shares the tiles with the OBJs)
    
    ldr     r3, =regBG1NBA
    ldrb    r3, [r3]                        @ r1 = 0000aaaa
    and     r3, r3, #0x07
    
    ldr     r5, =VRAMWrite
    ldr     r6, =VRAMWriteNOP2
    mov     r8, #8
    add     r5, r5, r3, lsl #4
    add     r5, r5, #(8*4)
1:  str     r6, [r5], #4
    subs    r8, r8, #1
    bne     1b

    b       EndAllocate

AllocateBG3_CheckLoop:
    
    mov     r8, #0
    mov     r6, r6, lsl #2
BG3CheckLoop:
    ldr     r5, =regBG1SC
    ldrb    r5, [r5, r8]                    @ v0.24 fix (thanks to Gladius)
    mov     r5, r5, lsr #2
    and     r5, r5, #0x1f
    subs    r5, r5, r6
    bmi     BG3CheckSkipLoop
    cmp     r5, #8
    movle   r1, #1
    ble     EndAllocate
    
BG3CheckSkipLoop:
    add     r8, r8, #1
    cmp     r8, #3
    bne     BG3CheckLoop

EndAllocate:
    mov     r5, #0x06000000
    add     r5, r5, r1, lsl #14             @ r5 = 0x0600?000 (GBA VRAM address for the characters)
    sub     r5, r5, r7, lsl #13
    orr     r2, r2, r1, lsl #2              @ r2 = 00000000 c000AApp
    ldr     r3, =bg1VRAMOffset
    str     r5, [r3, r0, lsl #2]
    cmp     r0, #2                          @ are we doing BG3?
    moveq   r7, r7, lsl #13
    ldr     r1, =bg3Base
    streq   r7, [r1]

    @ mosaic
    @
    mov     r1, #1
    mov     r4, r1, lsl r0
    ldr     r1, =regMOSAIC
    ldrb    r1, [r1]
    tsts    r1, r4
    orrne   r2, r2, #(1 << 6)               @ r2 = 00000000 cm00AApp

    @ SNES tilemap address
    ldr     r3, =regBG1SC                   @ load tilemap address
    ldrb    r3, [r3, r0]                    @ r3 = 00000000 ttttttyx
    orr     r2, r2, r3, lsl #14             @ r2 = yx000000 cm00AApp

    @ allocate tile map in GBA VRAM
    @ (allocates from the bottom of the VRAM segment to minimize
    @ chance of overwriting)
    @
    and     r5, r3, #0x03                   @ r5 = 00000000 000000yx
    ldr     r4, =bgCurTileOffset
    ldr     r4, [r4]
    cmp     r5, #0
    moveq   r8, #1
    cmp     r5, #1
    moveq   r8, #2
    cmp     r5, #2
    moveq   r8, #2
    cmp     r5, #3
    moveq   r8, #4
    sub     r4, r4, r8, lsl #11             @ r4 = 0x0600??00
    ldr     r6, =bgCurTileOffset
    str     r4, [r6]
    and     r6, r4, #0xF800                 @ r6 = 0x0000??00
    orr     r2, r2, r6, lsr #3              @ r2 = yx0ttttt cm00AApp

    @ set up the VRAMWrite table for tile map
    @
    ldr     r6, =VRAMWrite
    ldr     r5, =VRAMWriteTileMap
    and     r3, r3, #0x7c                   @ r3 = 00000000 0ttttt00
    add     r6, r6, r3
1:  str     r5, [r6], #4
    subs    r8, r8, #1
    bne     1b
    
    sub     r4, r4, r3, lsl #9
    bic     r4, r4, #0xff000000             @ v0.24 sanity fix
    bic     r4, r4, #0x00ff0000
    add     r4, r4, #0x06000000
    ldr     r5, =bg1TileOffset
    str     r4, [r5, r0, lsl #2]
    ldr     r4, =VRAMBG
    strb    r0, [r4, r3, lsr #2]
    ldr     r4, =VRAMBGColors
    ldr     r5, =NumColors
    ldrb    r5, [r5]
    strb    r5, [r4, r3, lsr #2]

    ldr     r3, =(0x4000008)                @ gba BGxCNT address
    add     r3, r3, r0, lsl #1
    strh    r2, [r3]
    bx      lr    

.macro CopyBGCNT  bgNumber, numColors, priority
    mov     r0, #\bgNumber
    mov     r2, #\priority
    mov     r6, #\numColors
    bl      RenderCopyBGCNT
.endm


@-------------------------------------------------------------------------
@ mode 7 BG CNT
@-------------------------------------------------------------------------
RenderCopyMode7BGCNT:   
    ldr     r0, =0xD080                     @ v0.24 fix, to remove the mode7 wraparound
    ldr     r1, =0x0400000C
    strh    r0, [r1]
    
    @ set up the VRAMWrite table for mode 7
    @
    ldr     r6, =VRAMWrite
    ldr     r5, =VRAMWriteMode7
    mov     r8, #16
1:
    str     r5, [r6], #4
    subs    r8, r8, #1
    bne     1b
    
    ldr     r5, =VRAMWriteNOP
    mov     r8, #16
1:
    str     r5, [r6], #4
    subs    r8, r8, #1
    bne     1b
    
    bx      lr

.macro CopyMode7BGCNT
    bl      RenderCopyMode7BGCNT
.endm

@-------------------------------------------------------------------------
@ mode 7 BG CNT
@-------------------------------------------------------------------------
RenderCopyMode7TileChar:

    ldr     r1, =snesVramBase
    ldr     r2, =gbaVramBase
    ldr     r3, =gbaVramBase+0x8000
    mov     r8, #8192

1:
    @ mode 7 tilemap
    ldrb    r6, [r1], #1

    @ mode 7 character
    ldrb    r4, [r1], #1
    
    @ mode 7 tilemap
    ldrb    r7, [r1], #1
    orr     r6, r6, r7, lsl #8
    strh    r6, [r3], #2
    
    @ mode 7 character
    ldrb    r5, [r1], #1
    orr     r4, r4, r5, lsl #8
    strh    r4, [r2], #2

    subs    r8, r8, #1
    bne     1b

    bx      lr

.macro CopyMode7TileChar
    bl      RenderCopyMode7TileChar
.endm



    SetText "RMODE0"

@-------------------------------------------------------------------------
@ Mode 0
@-------------------------------------------------------------------------
RenderMode0:
    StartRenderer
    EndRenderer

    SetText "RMODE1"

@-------------------------------------------------------------------------
@ Mode 1
@-------------------------------------------------------------------------
RenderMode1:
    StartRenderer

    DisableBG
    CopyBGCNT       2, COLOR_4, 1
    CopyBGCNT       0, COLOR_16, 1
    CopyBGCNT       1, COLOR_16, 1
    CopyBGCharEx    COLOR_16, COLOR_16, COLOR_4, COLOR_NONE
    CopyOBJChar     1
    CopyOBJChar     0
    CopyBGTileMap   0
    CopyBGTileMap   1
    CopyBGTileMap   2
    EnableBG        EnableBG0 + EnableBG1 + EnableBG2 + EnableOBJ

    EndRenderer

    SetText "RMODE2"

@-------------------------------------------------------------------------
@ Mode 2
@-------------------------------------------------------------------------
RenderMode2:
    StartRenderer

    DisableBG
    CopyBGCNT       0, COLOR_16, 1
    CopyBGCNT       1, COLOR_16, 1
    CopyBGCharEx    COLOR_16, COLOR_16, COLOR_NONE, COLOR_NONE
    CopyOBJChar     1
    CopyOBJChar     0
    CopyBGTileMap   0
    CopyBGTileMap   1
    EnableBG        EnableBG0 + EnableBG1 + EnableOBJ

    EndRenderer

    SetText "RMODE3"
@-------------------------------------------------------------------------
@ Mode 3
@-------------------------------------------------------------------------
RenderMode3:
    StartRenderer

    DisableBG
    CopyBGCNT       0, COLOR_256, 1
    CopyBGCNT       1, COLOR_16, 1
    CopyBGCharEx    COLOR_256, COLOR_16, COLOR_NONE, COLOR_NONE
    CopyOBJChar     1
    CopyOBJChar     0
    CopyBGTileMap   0
    CopyBGTileMap   1
    EnableBG        EnableBG0 + EnableBG1 + EnableOBJ

    EndRenderer
    .ltorg

    SetText "RMODE4"
@-------------------------------------------------------------------------
@ Mode 4
@-------------------------------------------------------------------------
RenderMode4:
    StartRenderer

    @ added for version 0.22
    @
    DisableBG
    CopyBGCNT       0, COLOR_256, 1
    CopyBGCNT       1, COLOR_4, 1
    CopyBGCharEx    COLOR_256, COLOR_4, COLOR_NONE, COLOR_NONE
    CopyOBJChar     1
    CopyOBJChar     0
    CopyBGTileMap   0
    CopyBGTileMap   1
    EnableBG        EnableBG0 + EnableBG1 + EnableOBJ
    
    EndRenderer

    SetText "RMODE5"
@-------------------------------------------------------------------------
@ Mode 5
@-------------------------------------------------------------------------
RenderMode5:
    StartRenderer

    @ added for version 0.22
    @
    DisableBG
    CopyBGCNT       0, COLOR_16, 1
    CopyBGCNT       1, COLOR_4, 1
    CopyBGCharEx    COLOR_16, COLOR_4, COLOR_NONE, COLOR_NONE
    CopyOBJChar     1
    CopyOBJChar     0
    CopyBGTileMap   0
    CopyBGTileMap   1
    EnableBG        EnableBG0 + EnableBG1 + EnableOBJ
    
    EndRenderer

    SetText "RMODE6"
@-------------------------------------------------------------------------
@ Mode 6
@-------------------------------------------------------------------------
RenderMode6:
    StartRenderer
    
    @ added for version 0.23
    @
    DisableBG
    CopyBGCNT       0, COLOR_16, 1
    CopyBGCharEx    COLOR_16, COLOR_NONE, COLOR_NONE, COLOR_NONE
    CopyOBJChar     1
    CopyOBJChar     0
    CopyBGTileMap   0
    EnableBG        EnableBG0 + EnableOBJ
    
    EndRenderer

    SetText "RMODE7"
@-------------------------------------------------------------------------
@ Mode 7
@-------------------------------------------------------------------------
RenderMode7:
    StartRenderer
    
    DisableBG
    CopyMode7BGCNT
    CopyMode7TileChar
    CopyOBJChar     1
    CopyOBJChar     0
    EnableBG        EnableBG2 + EnableOBJ, 2

    EndRenderer

    .ltorg


@=========================================================================
@ Write text
@   r0: address of text
@   r1: x position on screen
@   r2: y position on screen
@=========================================================================
writeText:
    stmfd   sp!, {r0-r8}

    ldr     r4, =0x06004000
    add     r4, r4, r1, lsl #1
    add     r4, r4, r2, lsl #6
    sub     r4, r4, #2
    
writeTextNextChar:
    add     r4, r4, #2
    ldrb    r3, [r0], #1
    cmp     r3, #0
    beq     writeTextEnd
    cmp     r3, #'0'
    blt     writeTextCheckAlpha
    cmp     r3, #'9'
    bgt     writeTextCheckAlpha

    @ this is a numeral
    sub     r3, r3, #('0'-1)
    strh    r3, [r4]
    b       writeTextNextChar

writeTextCheckAlpha:
    cmp     r3, #'A'
    blt     writeTextSpace
    cmp     r3, #'Z'
    bgt     writeTextSpace

    @ this is a alpha character
    sub     r3, r3, #('A'-10-1)
    strh    r3, [r4]
    b       writeTextNextChar

writeTextSpace:
    mov     r3, #0
    strh    r3, [r4]
    b       writeTextNextChar

writeTextEnd:
    ldmfd   sp!, {r0-r8}
    bx      lr


.macro WriteText x, y, t
    ldr     r0, =1f
    mov     r1, #\x
    mov     r2, #\y
    stmfd   sp!, {lr}
    bl      writeText
    ldmfd   sp!, {lr}
    b       2f
1:
    .asciz  "\t"
    .align  4
2:
.endm

.macro WriteTextShort
    stmfd   sp!, {lr}
    bl      writeText
    ldmfd   sp!, {lr}
.endm

@=========================================================================
@ Configuration text
@=========================================================================

    .equ    optionCount, 14

configOptionLeft:
    .word   config_BG0
    .word   config_BG1
    .word   config_BG2
    .word   config_BG3
    .word   config_BACKDROP
    .word   config_BGENABLE
    .word   config_BGFORCEMODE
    .word   config_CTRL
    .word   config_A
    .word   config_B
    .word   config_X
    .word   config_Y
    .word   config_RETURN
    .word   config_RESET

configOptionCount:
    .word   5, 5, 5, 5, 2, 2, 9, 4, 4, 4, 4, 4, 1, 1

configOption:
    .word   configOptionSet_BG
    .word   configOptionSet_BG
    .word   configOptionSet_BG
    .word   configOptionSet_BG
    .word   configOptionSet_YesNo
    .word   configOptionSet_SlowFast
    .word   configOptionSet_Modes
    .word   configOptionSet_Ctrl
    .word   configOptionSet_Key
    .word   configOptionSet_Key
    .word   configOptionSet_Key
    .word   configOptionSet_Key
    .word   configOptionSet_Empty
    .word   configOptionSet_Empty
    
configDefaultOption:
    .word   1, 2, 0, 3 
    .word   0, 0, 0, 2 
    .word   2, 0, 3, 1

configOptionSet_BG:
    .word   configBG_Prio0, configBG_Prio1, configBG_Prio2, configBG_Prio3, configBG_Disabled

configOptionSet_Ctrl:
    .word   configCtrl_Select, configCtrl_Start, configCtrl_L, configCtrl_R

configOptionSet_Key:
    .word   configKey_A, configKey_B, configKey_CA, configKey_CB

configOptionSet_YesNo:
    .word   configYesNo_No, configYesNo_Yes

configOptionSet_SlowFast:
    .word   configBGEnable_Slow, configBGEnable_Fast
    
configOptionSet_Modes:
    .word   configMode_Auto,configMode_0,configMode_1,configMode_2,configMode_3
    .word   configMode_4,configMode_5,configMode_6,configMode_7

configOptionSet_Empty:
    .word   configEmpty

config_BG0:         .asciz "BG0     "
config_BG1:         .asciz "BG1     "
config_BG2:         .asciz "BG2     "
config_BG3:         .asciz "BG3     "
config_CTRL:        .asciz "CTRL    "
config_A:           .asciz "BUTTON A"
config_B:           .asciz "BUTTON B"
config_X:           .asciz "BUTTON X"
config_Y:           .asciz "BUTTON Y"
config_BACKDROP:    .asciz "BACKDROP"
config_BGENABLE:    .asciz "BG ENABLE"
config_BGFORCEMODE: .asciz "BG FORCED MODE"
config_RETURN:      .asciz "RETURN TO GAME"
config_RESET:       .asciz "RESET GAME"

configBG_Prio0:     .asciz "P0      "
configBG_Prio1:     .asciz "P1      "
configBG_Prio2:     .asciz "P2      "
configBG_Prio3:     .asciz "P3      "
configBG_Disabled:  .asciz "DISABLED"

configCtrl_Select:  .asciz "SELECT  "
configCtrl_Start:   .asciz "START   "
configCtrl_L:       .asciz "L       "
configCtrl_R:       .asciz "R       "

configKey_A:        .asciz "A       "
configKey_B:        .asciz "B       "
configKey_CA:       .asciz "CTRL A  "
configKey_CB:       .asciz "CTRL B  "

configYesNo_Yes:    .asciz "YES     "
configYesNo_No:         .asciz "NO      "

configBGEnable_Slow:    .asciz "SMART   "
configBGEnable_Fast:    .asciz "FAST    "

configMode_Auto:    .asciz "AUTO  "
configMode_0:       .asciz "MODE 0"
configMode_1:       .asciz "MODE 1"
configMode_2:       .asciz "MODE 2"
configMode_3:       .asciz "MODE 3"
configMode_4:       .asciz "MODE 4"
configMode_5:       .asciz "MODE 5"
configMode_6:       .asciz "MODE 6"
configMode_7:       .asciz "MODE 7"

configCursorText:   .asciz "X"
configSpaceText:    .asciz " "

configEmpty:        .asciz ""
    .align  4

@=========================================================================
@ Configuration screen
@=========================================================================
configScreen:
    stmfd	sp!, {r3-r8}
    stmfd   sp!, {lr}
    bl      configClearScreen
    ldmfd   sp!, {lr}

    @ Write text to the screen
    @
    WriteText 3, 1, "SNEZZIBOY CONFIGURATION"

configWriteConfig:
    stmfd   sp!, {lr}
    bl      configUpdateKeyMap
    bl      configUpdateMisc
    ldmfd   sp!, {lr}

    @ write the configuration to the screen
    @
    mov     r8, #0
    ldr     r3, =configOptionLeft
    ldr     r4, =configOption
    ldr     r5, =configSRAMBase         @ save ram is where we store our config...
    mov     r2, #3

configWriteConfigLoop:
    ldr     r0, =configCursor
    ldr     r0, [r0]
    cmps    r8, r0
    ldreq   r0, =configCursorText
    ldrne   r0, =configSpaceText
    mov     r1, #1
    WriteTextShort

1:
    @ write the option name
    ldr     r0, [r3, r8, lsl #2]
    mov     r1, #3      
    WriteTextShort

    @ write the selected option
    ldr     r0, [r4, r8, lsl #2]
    ldrb    r7, [r5, r8]
    ldr     r0, [r0, r7, lsl #2]
    mov     r1, #20
    WriteTextShort

    add     r8, r8, #1
    add     r2, r2, #1
    cmp     r8, #optionCount
    bne     configWriteConfigLoop

    @ wait for key press
configWaitRelease:
    ldr     r0, =0x04000130
    ldr     r1, [r0]
    ldr     r2, =0x3ff
    cmp     r1, r2
    bne     configWaitRelease

configWaitKey:
    ldr     r0, =0x04000130
    ldr     r1, [r0]
    ldr     r2, =0x3ff
    cmp     r1, r2
    beq     configWaitKey

    tsts    r1, #(1 << 6)       @ key up?
    bne     1f
    ldr     r0, =configCursor
    ldr     r1, [r0]
    subs    r1, r1, #1
    movmi   r1, #(optionCount-1)
    str     r1, [r0]
    b       configWriteConfig

1:
    tsts    r1, #(1 << 7)       @ key down?
    bne     1f
    ldr     r0, =configCursor
    ldr     r1, [r0]
    add     r1, r1, #1
    cmp     r1, #optionCount
    moveq   r1, #0 
    str     r1, [r0]
    b       configWriteConfig

1:
    tsts    r1, #(1 << 5)       @ key left?
    bne     1f
    mov     r7, #-1
    b       configUpdate

1:
    tsts    r1, #(1 << 4)       @ key right?
    bne     1f
    mov     r7, #1
    b       configUpdate

1:
    tsts    r1, #(1 << 0)       @ key A?
    bne     1f
    mov     r7, #1
    b       configUpdate

1:
    tsts    r1, #(1 << 3)       @ key start?
    bne     1f
    b       configExit

1:
    b       configWriteConfig

configUpdate:
    ldr     r0, =configCursor
    ldr     r0, [r0]
    cmp     r0, #(optionCount-2)
    beq     configExit
    cmp     r0, #(optionCount-1)
    beq     HardReset

    ldr     r4, =configOptionCount
    ldr     r4, [r4, r0, lsl #2]
    sub     r4, r4, #1
    ldr     r5, =configSRAMBase
    ldrb    r1, [r5, r0]
    adds    r1, r1, r7
    movmi   r1, r4
    cmp     r1, r4
    movgt   r1, #0
    strb    r1, [r5, r0]
    b       configWriteConfig

configExit:
    ldr	    r0, =0x04000000			@ disable all BGs and OBJs first
    ldrh    r1, [r0]
    bic     r1, r1, #0xff00
    strh    r1, [r0]

    @ then we restore and update the palettes
    @
    ldr     r0, =0x05000000
    ldr     r1, =0x0600ff00
    mov     r8, #16
configRestorePaletteLoop:
    ldrh    r2, [r1], #2
    strh    r2, [r0], #2
    subs    r8, r8, #1
    bne     configRestorePaletteLoop

    @ force refresh of screen
    @
    ldr     r0, =(screenCurr1)
    ldr     r1, [r0]
    mvn     r1, r1
    ldr     r0, =(screenPrev1)
    str     r1, [r0]
	
    @ enable master interrupt
    @
    ldr     r0, =0x04000208
    mov     r1, #1
    strb    r1, [r0]

    ldmfd   sp!, {r3-r8}
    bx	    lr

@-------------------------------------------------------------------------
@ Update the key mapping config
@-------------------------------------------------------------------------
configControlKey:
    .word  (1<<2), (1<<3), (1<<9), (1<<8)

configUpdateKeyMap:

    ldr     r0, =configCTRL
    ldrb    r0, [r0]                @ find out the control key

    ldr     r2, =configControlKey
    ldr     r2, [r2, r0, lsl #2]    @ r2 = control key mapping

    ldr     r0, =configA
    ldrb    r0, [r0]                @ find out the control key
    tsts    r0, #0x2
    bic     r0, r0, #0x2
    orrne   r0, r0, r2
    tsts    r0, #0x1
    bic     r0, r0, #0x1
    orrne   r0, r0, #(1<<1)
    orreq   r0, r0, #(1<<0)
    ldr     r1, =regJoyA
    strh    r0, [r1]    

    ldr     r0, =configB
    ldrb    r0, [r0]                @ find out the control key
    tsts    r0, #0x2
    bic     r0, r0, #0x2
    orrne   r0, r0, r2
    tsts    r0, #0x1
    bic     r0, r0, #0x1
    orrne   r0, r0, #(1<<1)
    orreq   r0, r0, #(1<<0)
    ldr     r1, =regJoyB
    strh    r0, [r1]    

    ldr     r0, =configX
    ldrb    r0, [r0]                @ find out the control key
    tsts    r0, #0x2
    bic     r0, r0, #0x2
    orrne   r0, r0, r2
    tsts    r0, #0x1
    bic     r0, r0, #0x1
    orrne   r0, r0, #(1<<1)
    orreq   r0, r0, #(1<<0)
    ldr     r1, =regJoyX
    strh    r0, [r1]    

    ldr     r0, =configY
    ldrb    r0, [r0]                @ find out the control key
    tsts    r0, #0x2
    bic     r0, r0, #0x2
    orrne   r0, r0, r2
    tsts    r0, #0x1
    bic     r0, r0, #0x1
    orrne   r0, r0, #(1<<1)
    orreq   r0, r0, #(1<<0)
    ldr     r1, =regJoyY
    strh    r0, [r1]    

    bx      lr

    .ltorg


@-------------------------------------------------------------------------
@ version 0.23
@ update miscellaneous config options
@ - forced mode
@-------------------------------------------------------------------------
codeForceMode:
    and     r0, r0, #7
    mov     r0, #0
    mov     r0, #1
    mov     r0, #2
    mov     r0, #3
    mov     r0, #4
    mov     r0, #5
    mov     r0, #6
    mov     r0, #7
    
configUpdateMisc:  
    @ forced mode
    @
    ldr     r2, =configBGForceMode
    ldrb    r2, [r2]
    ldr     r1, =codeForceMode
    ldr     r1, [r1, r2, lsl #2]
    
    ldr     r2, =vBlankRefreshScreen_ForceMode
    str     r1, [r2]
    ldr     r2, =snesRenderScreenAtVBlank_ForceMode
    str     r1, [r2]
    
    bx      lr

@-------------------------------------------------------------------------
@ Cycle through the various BG priority set
@-------------------------------------------------------------------------
.macro  cyclePriority  p1,p2,p3,p4
    ldr     r0, =configBG0Priority
    mov     r1, #\p1
    strb    r1, [r0]
    ldr     r0, =configBG1Priority
    mov     r1, #\p2
    strb    r1, [r0]
    ldr     r0, =configBG2Priority
    mov     r1, #\p3
    strb    r1, [r0]
    ldr     r0, =configBG3Priority
    mov     r1, #\p4
    strb    r1, [r0]
    b       configCycleBGPriorityEnd
1:
.endm

configCycleBGPriority:
    stmfd   sp!, {r3-r8,lr}
    
    @ clear screen 
    @
    bl      configClearScreen
    WriteText 3, 1, "SNEZZIBOY CONFIGURATION"
    WriteText 3, 5, "  CYCLE BG PRIORITIES  "
    
    ldr     r1, =configBGPrioritySet
    ldrb    r0, [r1]
    
    add     r0, r0, #1
    cmp     r0, #04
    movge   r0, #0
    strb    r0, [r1]
    
    cmp     r0, #0
    bne     9f
    WriteText 3, 6, "      P1 P2 P0 P3      "
    cyclePriority   1,2,0,3
9:    
    cmp     r0, #1
    bne     9f
    WriteText 3, 6, "      P2 P1 P0 P3      "
    cyclePriority   2,1,0,3
9:    
    cmp     r0, #2
    bne     9f
    WriteText 3, 6, "      P0 P1 P2 P3      "
    cyclePriority   0,1,2,3
9:    
    cmp     r0, #3
    bne     9f
    WriteText 3, 6, "      P1 P0 P2 P3      "
    cyclePriority   1,0,2,3
9:    

configCycleBGPriorityEnd:

    @ wait until keys are released
    ldr     r0, =0x04000130
    ldr     r2, =0x0f0
1:
    ldrh    r1, [r0]
    mvn     r1, r1
    tsts    r1, r2
    bne     1b
    
    ldmfd   sp!, {r3-r8,lr}
    stmfd   sp!, {r3-r8}
    b       configExit

    .ltorg
    
@-------------------------------------------------------------------------
@ Cycle through various screen modes
@-------------------------------------------------------------------------
configCycleBGForcedMode:
    stmfd   sp!, {r3-r8,lr}

    @ clear screen 
    @
    bl      configClearScreen
    WriteText 3, 1, "SNEZZIBOY CONFIGURATION"
    WriteText 3, 5, "    CYCLE BG MODES     "
    
    ldr     r1, =configBGForceMode
    ldrb    r0, [r1]
    add     r0, r0, #1
    cmp     r0, #9
    movge   r0, #0
    strb    r0, [r1]
    
    cmp     r0, #0
    bne     9f
    WriteText 3, 6, "         AUTO          "
9:
    
    cmp     r0, #1
    bne     9f
    WriteText 3, 6, "        MODE 0         "
9:
    
    cmp     r0, #2
    bne     9f 
    WriteText 3, 6, "        MODE 1         "
9:
    
    cmp     r0, #3
    bne     9f
    WriteText 3, 6, "        MODE 2         "
9:
    
    cmp     r0, #4
    bne     9f
    WriteText 3, 6, "        MODE 3         "
9:
    
    cmp     r0, #5
    bne     9f
    WriteText 3, 6, "        MODE 4         "
9:
    
    cmp     r0, #6
    bne     9f
    WriteText 3, 6, "        MODE 5         "
9:
    
    cmp     r0, #7
    bne     9f
    WriteText 3, 6, "        MODE 6         "
9:

    cmp     r0, #8
    bne     9f
    WriteText 3, 6, "        MODE 7         "
9:
    
    bl      configUpdateMisc
    
    @ wait until keys are released
    ldr     r0, =0x04000130
    ldr     r2, =0x0f0
1:
    ldrh    r1, [r0]
    mvn     r1, r1
    tsts    r1, r2
    bne     1b
    
    ldmfd   sp!, {r3-r8,lr}
    stmfd   sp!, {r3-r8}
    b       configExit

    .ltorg


@-------------------------------------------------------------------------
@ clear the screen and update palette/fonts 
@-------------------------------------------------------------------------
configClearScreen:
    @ disable master interrupt
    @
    ldr     r0, =0x04000208
    mov     r1, #0
    strb    r1, [r0]

    ldr	    r0, =0x04000000			@ disable all BGs and OBJs first
    ldrh    r1, [r0]
    bic     r1, r1, #0xff00
    strh    r1, [r0]
    
    mov     r0, #0
    ldr     r1, =0x04000050
    strh    r0, [r1]

    @ set BG0 CNT
    ldr     r0, =0x04000008
    mov     r1, #0x0800
    str     r1, [r0]

    @ clear tilemap for BG0
    ldr     r0, =0x06004000
    mov     r1, #0
    ldr     r8, =(32*32)
configClearTileMap:
    strh    r1, [r0], #2
    subs    r8, r8, #1
    bne     configClearTileMap

    @ set BG0 HOFS, VOFS
    ldr     r0, =0x04000010
    mov     r1, #0
    strh    r1, [r0], #2
    strh    r1, [r0], #2

    @ then we save and update the palettes
    @
    ldr     r0, =0x05000000
    ldr     r1, =0x0600ff00
    mov     r3, #0
    mov     r8, #16
configCopyPaletteLoop:
    ldrh    r2, [r0]
    strh    r2, [r1], #2
    strh    r3, [r0], #2
    eor     r3, r3, #0xff
    eor     r3, r3, #0xff00
    subs    r8, r8, #1
    bne     configCopyPaletteLoop

    @ here we copy the config screen fonts to the tile
    @
    ldr     r0, =FontNumbers
    ldr     r1, =charUnpack4
    ldr     r2, =0x06000000
    mov     r8, #(10+27)*8
configCopyFontLoop:
    ldrb    r3, [r0], #1
    ldr     r3, [r1, r3, lsl #2]
    str     r3, [r2], #4
    subs    r8, r8, #1
    bne     configCopyFontLoop
    
    ldr	    r0, =0x04000000			
    ldrh    r1, [r0]
    bic     r1, r1, #0x7
    orr	    r1, r1, #0x0100
    strh    r1, [r0]

    bx      lr

    .ltorg


@=========================================================================
@ GBA 8x8 Font
@=========================================================================
FontNumbers:
	@ space
	.byte	0b00000000
	.byte	0b00000000
	.byte	0b00000000
	.byte	0b00000000
	.byte	0b00000000
	.byte	0b00000000
	.byte	0b00000000
	.byte	0b00000000

	@ 0
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ 1
	.byte	0b01110000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b11111110
	.byte	0b00000000

	@ 2
	.byte	0b11111100
	.byte	0b00000010
	.byte	0b00000010
	.byte	0b01111110
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111110
	.byte	0b00000000

	@ 3
	.byte	0b11111100
	.byte	0b00000010
	.byte	0b00000010
	.byte	0b11111100
	.byte	0b00000010
	.byte	0b00000010
	.byte	0b11111100
	.byte	0b00000000

	@ 4
	.byte	0b00001000
	.byte	0b10001000
	.byte	0b10001000
	.byte	0b11111110
	.byte	0b00001000
	.byte	0b00001000
	.byte	0b00001000
	.byte	0b00000000

	@ 5
	.byte	0b11111100
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111100
	.byte	0b00000010
	.byte	0b00000010
	.byte	0b11111100
	.byte	0b00000000

	@ 6
	.byte	0b01111100
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ 7
	.byte	0b11111110
	.byte	0b00000010
	.byte	0b00000100
	.byte	0b00001000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00000000

	@ 8
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ 9
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111110
	.byte	0b00000010
	.byte	0b00000010
	.byte	0b01111100
	.byte	0b00000000


FontLetters: 
	@ A
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111110
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b00000000

	@ B
	.byte	0b11111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111100
	.byte	0b00000000

	@ C
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ D
	.byte	0b11111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111100
	.byte	0b00000000

	@ E
	.byte	0b11111110
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111110
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111110
	.byte	0b00000000

	@ F
	.byte	0b11111110
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111100
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b00000000

	@ G
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000000
	.byte	0b10011110
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ H
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111110
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b00000000

	@ I
	.byte	0b01111100
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b01111100
	.byte	0b00000000

	@ J
	.byte	0b01111100
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b11100000
	.byte	0b00000000

	@ K
	.byte	0b10000010
	.byte	0b10000100
	.byte	0b10001000
	.byte	0b11110000
	.byte	0b10001000
	.byte	0b10000100
	.byte	0b10000010
	.byte	0b00000000

	@ L
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b11111110
	.byte	0b00000000

	@ M
	.byte	0b10000010
	.byte	0b11000110
	.byte	0b10101010
	.byte	0b10010010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b00000000

	@ N
	.byte	0b10000010
	.byte	0b11000010
	.byte	0b10100010
	.byte	0b10010010
	.byte	0b10001010
	.byte	0b10000110
	.byte	0b10000010
	.byte	0b00000000

	@ O
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ P
	.byte	0b11111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111100
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b00000000

	@ Q
	.byte	0b01111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10010010
	.byte	0b10001010
	.byte	0b10000100
	.byte	0b01111010
	.byte	0b00000000

	@ R
	.byte	0b11111100
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b11111100
	.byte	0b10001000
	.byte	0b10000100
	.byte	0b10000010
	.byte	0b00000000

	@ S
	.byte	0b01111110
	.byte	0b10000000
	.byte	0b10000000
	.byte	0b01111100
	.byte	0b00000010
	.byte	0b00000010
	.byte	0b11111100
	.byte	0b00000000

	@ T
	.byte	0b11111110
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00000000

	@ U
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01111100
	.byte	0b00000000

	@ V
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b01000100
	.byte	0b00101000
	.byte	0b00010000
	.byte	0b00000000

	@ W
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10000010
	.byte	0b10010010
	.byte	0b10101010
	.byte	0b11000110
	.byte	0b10000010
	.byte	0b00000000

	@ X
	.byte	0b10000010
	.byte	0b01000100
	.byte	0b00101000
	.byte	0b00010000
	.byte	0b00101000
	.byte	0b01000100
	.byte	0b10000010
	.byte	0b00000000

	@ Y
	.byte	0b10000010
	.byte	0b01000100
	.byte	0b00101000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00010000
	.byte	0b00000000

	@ Z
	.byte	0b11111110
	.byte	0b00000100
	.byte	0b00001000
	.byte	0b00010000
	.byte	0b00100000
	.byte	0b01000000
	.byte	0b11111110
	.byte	0b00000000



@=========================================================================
@ IO mapping table
@=========================================================================

.macro  IO      c, m0, m1, m2, m3, m4, m5, m6, m7, m8, m9, ma, mb, mc, md, me, mf
    .rept   \c
    .long   \m0,\m1,\m2,\m3,\m4,\m5,\m6,\m7,\m8,\m9,\ma,\mb,\mc,\md,\me,\mf
    .endr
.endm

IORead:
    IO  0x210,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x0000-
    
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2100-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2110-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2120-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,R2134,R2135,R2136,R2137, R2138,R2139,R213A,IONOP,R213C,R213D,R213E,R213F     @ 0x2130-
    IO  0x001,  R2140,R2141,R2142,R2143,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2140-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2150-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2160-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2170-
    IO  0x001,  R2180,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2180-
    IO  0x1e7,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2190

    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4000
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,R4016,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4010
    IO  0x01e,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4020
    
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4200
    IO  0x001,  R4210,R4211,R4212,IONOP,R4214,R4215,R4216,R4217, R4218,R4219,R421A,R421B,R421C,R421D,R421E,R421F     @ 0x4210
    IO  0x00e,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4200

    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4300
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4310
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4320
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4330
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4340
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4350
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4360
    IO  0x001,  R43x0,R43x1,R43x2,R43x3,R43x4,R43x5,R43x6,R43x7, R43x8,R43x9,R43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4370
    IO  0x008,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4380
    IO  0x1c0,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4400
    
    IO  0x200,  R_SAV,R_SAV,R_SAV,R_SAV,R_SAV,R_SAV,R_SAV,R_SAV, R_SAV,R_SAV,R_SAV,R_SAV,R_SAV,R_SAV,R_SAV,R_SAV     @ 0x6000

IOWrite:
    IO  0x210,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x0000-
    
    IO  0x001,  W2100,W2101,W2102,W2103,W2104,W2105,W2106,W2107, W2108,W2109,W210A,W210B,W210C,W210D,W210E,W210F     @ 0x2100-
    IO  0x001,  W2110,W2111,W2112,W2113,W2114,W2115,W2116,W2117, W2118,W2119,IONOP,W211B,W211C,W211D,W211E,W211F     @ 0x2110-
    IO  0x001,  W2120,W2121,W2122,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,W212C,W212D,IONOP,IONOP     @ 0x2120-
    IO  0x001,  W2130,W2131,W2132,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2130-
    IO  0x001,  W2140,W2141,W2142,W2143,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2140-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2150-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2160-
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2170-
    IO  0x001,  W2180,W2181,W2182,W2183,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2180-
    IO  0x1e7,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x2190
    
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4000
    IO  0x001,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,W4016,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4010
    IO  0x01e,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4020
    
    IO  0x001,  W4200,IONOP,W4202,W4203,W4204,W4205,W4206,W4207, W4208,W4209,W420A,W420B,IONOP,W420D,IONOP,IONOP     @ 0x4200
    IO  0x00f,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4210
    
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4300
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4310
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4320
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4330
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4340
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4350
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4360
    IO  0x001,  W43x0,W43x1,W43x2,W43x3,W43x4,W43x5,W43x6,W43x7, W43x8,W43x9,W43xA,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4370
    IO  0x008,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4380
    IO  0x1c0,  IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP, IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP,IONOP     @ 0x4400

    IO  0x200,  W_SAV,W_SAV,W_SAV,W_SAV,W_SAV,W_SAV,W_SAV,W_SAV, W_SAV,W_SAV,W_SAV,W_SAV,W_SAV,W_SAV,W_SAV,W_SAV     @ 0x6000


@=========================================================================
@ GBA to SNES Keypad mapping
@=========================================================================
    .align  8
    .ascii  "KEYPMAP"
    .align  8
keypadMap:

    .hword 0x3f30,0x3f30,0x3f30,0x3f30,0x1f30,0x1f30,0x1f30,0x1f30,0x2f30,0x2f30,0x2f30,0x2f30,0xf30,0xf30,0xf30,0xf30
    .hword 0x3e30,0x3e30,0x3e30,0x3e30,0x1e30,0x1e30,0x1e30,0x1e30,0x2e30,0x2e30,0x2e30,0x2e30,0xe30,0xe30,0xe30,0xe30
    .hword 0x3d30,0x3d30,0x3d30,0x3d30,0x1d30,0x1d30,0x1d30,0x1d30,0x2d30,0x2d30,0x2d30,0x2d30,0xd30,0xd30,0xd30,0xd30
    .hword 0x3c30,0x3c30,0x3c30,0x3c30,0x1c30,0x1c30,0x1c30,0x1c30,0x2c30,0x2c30,0x2c30,0x2c30,0xc30,0xc30,0xc30,0xc30
    .hword 0x3730,0x3730,0x3730,0x3730,0x1730,0x1730,0x1730,0x1730,0x2730,0x2730,0x2730,0x2730,0x730,0x730,0x730,0x730
    .hword 0x3630,0x3630,0x3630,0x3630,0x1630,0x1630,0x1630,0x1630,0x2630,0x2630,0x2630,0x2630,0x630,0x630,0x630,0x630
    .hword 0x3530,0x3530,0x3530,0x3530,0x1530,0x1530,0x1530,0x1530,0x2530,0x2530,0x2530,0x2530,0x530,0x530,0x530,0x530
    .hword 0x3430,0x3430,0x3430,0x3430,0x1430,0x1430,0x1430,0x1430,0x2430,0x2430,0x2430,0x2430,0x430,0x430,0x430,0x430
    .hword 0x3b30,0x3b30,0x3b30,0x3b30,0x1b30,0x1b30,0x1b30,0x1b30,0x2b30,0x2b30,0x2b30,0x2b30,0xb30,0xb30,0xb30,0xb30
    .hword 0x3a30,0x3a30,0x3a30,0x3a30,0x1a30,0x1a30,0x1a30,0x1a30,0x2a30,0x2a30,0x2a30,0x2a30,0xa30,0xa30,0xa30,0xa30
    .hword 0x3930,0x3930,0x3930,0x3930,0x1930,0x1930,0x1930,0x1930,0x2930,0x2930,0x2930,0x2930,0x930,0x930,0x930,0x930
    .hword 0x3830,0x3830,0x3830,0x3830,0x1830,0x1830,0x1830,0x1830,0x2830,0x2830,0x2830,0x2830,0x830,0x830,0x830,0x830
    .hword 0x3330,0x3330,0x3330,0x3330,0x1330,0x1330,0x1330,0x1330,0x2330,0x2330,0x2330,0x2330,0x330,0x330,0x330,0x330
    .hword 0x3230,0x3230,0x3230,0x3230,0x1230,0x1230,0x1230,0x1230,0x2230,0x2230,0x2230,0x2230,0x230,0x230,0x230,0x230
    .hword 0x3130,0x3130,0x3130,0x3130,0x1130,0x1130,0x1130,0x1130,0x2130,0x2130,0x2130,0x2130,0x130,0x130,0x130,0x130
    .hword 0x3030,0x3030,0x3030,0x3030,0x1030,0x1030,0x1030,0x1030,0x2030,0x2030,0x2030,0x2030,0x30,0x30,0x30,0x30
    .hword 0x3f20,0x3f20,0x3f20,0x3f20,0x1f20,0x1f20,0x1f20,0x1f20,0x2f20,0x2f20,0x2f20,0x2f20,0xf20,0xf20,0xf20,0xf20
    .hword 0x3e20,0x3e20,0x3e20,0x3e20,0x1e20,0x1e20,0x1e20,0x1e20,0x2e20,0x2e20,0x2e20,0x2e20,0xe20,0xe20,0xe20,0xe20
    .hword 0x3d20,0x3d20,0x3d20,0x3d20,0x1d20,0x1d20,0x1d20,0x1d20,0x2d20,0x2d20,0x2d20,0x2d20,0xd20,0xd20,0xd20,0xd20
    .hword 0x3c20,0x3c20,0x3c20,0x3c20,0x1c20,0x1c20,0x1c20,0x1c20,0x2c20,0x2c20,0x2c20,0x2c20,0xc20,0xc20,0xc20,0xc20
    .hword 0x3720,0x3720,0x3720,0x3720,0x1720,0x1720,0x1720,0x1720,0x2720,0x2720,0x2720,0x2720,0x720,0x720,0x720,0x720
    .hword 0x3620,0x3620,0x3620,0x3620,0x1620,0x1620,0x1620,0x1620,0x2620,0x2620,0x2620,0x2620,0x620,0x620,0x620,0x620
    .hword 0x3520,0x3520,0x3520,0x3520,0x1520,0x1520,0x1520,0x1520,0x2520,0x2520,0x2520,0x2520,0x520,0x520,0x520,0x520
    .hword 0x3420,0x3420,0x3420,0x3420,0x1420,0x1420,0x1420,0x1420,0x2420,0x2420,0x2420,0x2420,0x420,0x420,0x420,0x420
    .hword 0x3b20,0x3b20,0x3b20,0x3b20,0x1b20,0x1b20,0x1b20,0x1b20,0x2b20,0x2b20,0x2b20,0x2b20,0xb20,0xb20,0xb20,0xb20
    .hword 0x3a20,0x3a20,0x3a20,0x3a20,0x1a20,0x1a20,0x1a20,0x1a20,0x2a20,0x2a20,0x2a20,0x2a20,0xa20,0xa20,0xa20,0xa20
    .hword 0x3920,0x3920,0x3920,0x3920,0x1920,0x1920,0x1920,0x1920,0x2920,0x2920,0x2920,0x2920,0x920,0x920,0x920,0x920
    .hword 0x3820,0x3820,0x3820,0x3820,0x1820,0x1820,0x1820,0x1820,0x2820,0x2820,0x2820,0x2820,0x820,0x820,0x820,0x820
    .hword 0x3320,0x3320,0x3320,0x3320,0x1320,0x1320,0x1320,0x1320,0x2320,0x2320,0x2320,0x2320,0x320,0x320,0x320,0x320
    .hword 0x3220,0x3220,0x3220,0x3220,0x1220,0x1220,0x1220,0x1220,0x2220,0x2220,0x2220,0x2220,0x220,0x220,0x220,0x220
    .hword 0x3120,0x3120,0x3120,0x3120,0x1120,0x1120,0x1120,0x1120,0x2120,0x2120,0x2120,0x2120,0x120,0x120,0x120,0x120
    .hword 0x3020,0x3020,0x3020,0x3020,0x1020,0x1020,0x1020,0x1020,0x2020,0x2020,0x2020,0x2020,0x20,0x20,0x20,0x20
    .hword 0x3f10,0x3f10,0x3f10,0x3f10,0x1f10,0x1f10,0x1f10,0x1f10,0x2f10,0x2f10,0x2f10,0x2f10,0xf10,0xf10,0xf10,0xf10
    .hword 0x3e10,0x3e10,0x3e10,0x3e10,0x1e10,0x1e10,0x1e10,0x1e10,0x2e10,0x2e10,0x2e10,0x2e10,0xe10,0xe10,0xe10,0xe10
    .hword 0x3d10,0x3d10,0x3d10,0x3d10,0x1d10,0x1d10,0x1d10,0x1d10,0x2d10,0x2d10,0x2d10,0x2d10,0xd10,0xd10,0xd10,0xd10
    .hword 0x3c10,0x3c10,0x3c10,0x3c10,0x1c10,0x1c10,0x1c10,0x1c10,0x2c10,0x2c10,0x2c10,0x2c10,0xc10,0xc10,0xc10,0xc10
    .hword 0x3710,0x3710,0x3710,0x3710,0x1710,0x1710,0x1710,0x1710,0x2710,0x2710,0x2710,0x2710,0x710,0x710,0x710,0x710
    .hword 0x3610,0x3610,0x3610,0x3610,0x1610,0x1610,0x1610,0x1610,0x2610,0x2610,0x2610,0x2610,0x610,0x610,0x610,0x610
    .hword 0x3510,0x3510,0x3510,0x3510,0x1510,0x1510,0x1510,0x1510,0x2510,0x2510,0x2510,0x2510,0x510,0x510,0x510,0x510
    .hword 0x3410,0x3410,0x3410,0x3410,0x1410,0x1410,0x1410,0x1410,0x2410,0x2410,0x2410,0x2410,0x410,0x410,0x410,0x410
    .hword 0x3b10,0x3b10,0x3b10,0x3b10,0x1b10,0x1b10,0x1b10,0x1b10,0x2b10,0x2b10,0x2b10,0x2b10,0xb10,0xb10,0xb10,0xb10
    .hword 0x3a10,0x3a10,0x3a10,0x3a10,0x1a10,0x1a10,0x1a10,0x1a10,0x2a10,0x2a10,0x2a10,0x2a10,0xa10,0xa10,0xa10,0xa10
    .hword 0x3910,0x3910,0x3910,0x3910,0x1910,0x1910,0x1910,0x1910,0x2910,0x2910,0x2910,0x2910,0x910,0x910,0x910,0x910
    .hword 0x3810,0x3810,0x3810,0x3810,0x1810,0x1810,0x1810,0x1810,0x2810,0x2810,0x2810,0x2810,0x810,0x810,0x810,0x810
    .hword 0x3310,0x3310,0x3310,0x3310,0x1310,0x1310,0x1310,0x1310,0x2310,0x2310,0x2310,0x2310,0x310,0x310,0x310,0x310
    .hword 0x3210,0x3210,0x3210,0x3210,0x1210,0x1210,0x1210,0x1210,0x2210,0x2210,0x2210,0x2210,0x210,0x210,0x210,0x210
    .hword 0x3110,0x3110,0x3110,0x3110,0x1110,0x1110,0x1110,0x1110,0x2110,0x2110,0x2110,0x2110,0x110,0x110,0x110,0x110
    .hword 0x3010,0x3010,0x3010,0x3010,0x1010,0x1010,0x1010,0x1010,0x2010,0x2010,0x2010,0x2010,0x10,0x10,0x10,0x10
    .hword 0x3f00,0x3f00,0x3f00,0x3f00,0x1f00,0x1f00,0x1f00,0x1f00,0x2f00,0x2f00,0x2f00,0x2f00,0xf00,0xf00,0xf00,0xf00
    .hword 0x3e00,0x3e00,0x3e00,0x3e00,0x1e00,0x1e00,0x1e00,0x1e00,0x2e00,0x2e00,0x2e00,0x2e00,0xe00,0xe00,0xe00,0xe00
    .hword 0x3d00,0x3d00,0x3d00,0x3d00,0x1d00,0x1d00,0x1d00,0x1d00,0x2d00,0x2d00,0x2d00,0x2d00,0xd00,0xd00,0xd00,0xd00
    .hword 0x3c00,0x3c00,0x3c00,0x3c00,0x1c00,0x1c00,0x1c00,0x1c00,0x2c00,0x2c00,0x2c00,0x2c00,0xc00,0xc00,0xc00,0xc00
    .hword 0x3700,0x3700,0x3700,0x3700,0x1700,0x1700,0x1700,0x1700,0x2700,0x2700,0x2700,0x2700,0x700,0x700,0x700,0x700
    .hword 0x3600,0x3600,0x3600,0x3600,0x1600,0x1600,0x1600,0x1600,0x2600,0x2600,0x2600,0x2600,0x600,0x600,0x600,0x600
    .hword 0x3500,0x3500,0x3500,0x3500,0x1500,0x1500,0x1500,0x1500,0x2500,0x2500,0x2500,0x2500,0x500,0x500,0x500,0x500
    .hword 0x3400,0x3400,0x3400,0x3400,0x1400,0x1400,0x1400,0x1400,0x2400,0x2400,0x2400,0x2400,0x400,0x400,0x400,0x400
    .hword 0x3b00,0x3b00,0x3b00,0x3b00,0x1b00,0x1b00,0x1b00,0x1b00,0x2b00,0x2b00,0x2b00,0x2b00,0xb00,0xb00,0xb00,0xb00
    .hword 0x3a00,0x3a00,0x3a00,0x3a00,0x1a00,0x1a00,0x1a00,0x1a00,0x2a00,0x2a00,0x2a00,0x2a00,0xa00,0xa00,0xa00,0xa00
    .hword 0x3900,0x3900,0x3900,0x3900,0x1900,0x1900,0x1900,0x1900,0x2900,0x2900,0x2900,0x2900,0x900,0x900,0x900,0x900
    .hword 0x3800,0x3800,0x3800,0x3800,0x1800,0x1800,0x1800,0x1800,0x2800,0x2800,0x2800,0x2800,0x800,0x800,0x800,0x800
    .hword 0x3300,0x3300,0x3300,0x3300,0x1300,0x1300,0x1300,0x1300,0x2300,0x2300,0x2300,0x2300,0x300,0x300,0x300,0x300
    .hword 0x3200,0x3200,0x3200,0x3200,0x1200,0x1200,0x1200,0x1200,0x2200,0x2200,0x2200,0x2200,0x200,0x200,0x200,0x200
    .hword 0x3100,0x3100,0x3100,0x3100,0x1100,0x1100,0x1100,0x1100,0x2100,0x2100,0x2100,0x2100,0x100,0x100,0x100,0x100
    .hword 0x3000,0x3000,0x3000,0x3000,0x1000,0x1000,0x1000,0x1000,0x2000,0x2000,0x2000,0x2000,0x00,0x00,0x00,0x00


@=========================================================================
@ SNES OAM X/Y to GBA OAM X/Y map
@=========================================================================
oamX:

    .hword 0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007
    .hword 0x0008,0x0009,0x000a,0x000b,0x000c,0x000d,0x000e,0x000f,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017
    .hword 0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027
    .hword 0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037
    .hword 0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x00a0,0x00a1,0x00a2,0x00a3,0x00a4,0x00a5,0x00a6,0x00a7
    .hword 0x00a8,0x00a9,0x00aa,0x00ab,0x00ac,0x00ad,0x00ae,0x00af,0x00b0,0x00b1,0x00b2,0x00b3,0x00b4,0x00b5,0x00b6,0x00b7
    .hword 0x00b8,0x00b9,0x00ba,0x00bb,0x00bc,0x00bd,0x00be,0x00bf,0x00c0,0x00c1,0x00c2,0x00c3,0x00c4,0x00c5,0x00c6,0x00c7
    .hword 0x00c8,0x00c9,0x00ca,0x00cb,0x00cc,0x00cd,0x00ce,0x00cf,0x00d0,0x00d1,0x00d2,0x00d3,0x00d4,0x00d5,0x00d6,0x00d7
    .hword 0x00d8,0x00d9,0x00da,0x00db,0x00dc,0x00dd,0x00de,0x00df,0x00e0,0x00e1,0x00e2,0x00e3,0x00e4,0x00e5,0x00e6,0x00e7
    .hword 0x00e8,0x00e9,0x00ea,0x00eb,0x00ec,0x00ed,0x00ee,0x00ef,0x00f0,0x00f1,0x00f2,0x00f3,0x00f4,0x00f5,0x00f6,0x00f7
    .hword 0x00f8,0x00f9,0x00fa,0x00fb,0x00fc,0x00fd,0x00fe,0x00ff,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x41f8,0x41f9,0x41fa,0x41fb,0x41fc,0x41fd,0x41fe,0x41ff,0x4000,0x4001,0x4002,0x4003,0x4004,0x4005,0x4006,0x4007
    .hword 0x4008,0x4009,0x400a,0x400b,0x400c,0x400d,0x400e,0x400f,0x4010,0x4011,0x4012,0x4013,0x4014,0x4015,0x4016,0x4017
    .hword 0x4018,0x4019,0x401a,0x401b,0x401c,0x401d,0x401e,0x401f,0x4020,0x4021,0x4022,0x4023,0x4024,0x4025,0x4026,0x4027
    .hword 0x4028,0x4029,0x402a,0x402b,0x402c,0x402d,0x402e,0x402f,0x4030,0x4031,0x4032,0x4033,0x4034,0x4035,0x4036,0x4037
    .hword 0x4038,0x4039,0x403a,0x403b,0x403c,0x403d,0x403e,0x403f,0x4040,0x4041,0x4042,0x4043,0x4044,0x4045,0x4046,0x4047
    .hword 0x4048,0x4049,0x404a,0x404b,0x404c,0x404d,0x404e,0x404f,0x4050,0x4051,0x4052,0x4053,0x4054,0x4055,0x4056,0x4057
    .hword 0x4058,0x4059,0x405a,0x405b,0x405c,0x405d,0x405e,0x405f,0x4060,0x4061,0x4062,0x4063,0x4064,0x4065,0x4066,0x4067
    .hword 0x4068,0x4069,0x406a,0x406b,0x406c,0x406d,0x406e,0x406f,0x4070,0x4071,0x4072,0x4073,0x4074,0x4075,0x4076,0x4077
    .hword 0x4078,0x4079,0x407a,0x407b,0x407c,0x407d,0x407e,0x407f,0x4080,0x4081,0x4082,0x4083,0x4084,0x4085,0x4086,0x4087
    .hword 0x4088,0x4089,0x408a,0x408b,0x408c,0x408d,0x408e,0x408f,0x4090,0x4091,0x4092,0x4093,0x4094,0x4095,0x4096,0x4097
    .hword 0x4098,0x4099,0x409a,0x409b,0x409c,0x409d,0x409e,0x409f,0x40a0,0x40a1,0x40a2,0x40a3,0x40a4,0x40a5,0x40a6,0x40a7
    .hword 0x40a8,0x40a9,0x40aa,0x40ab,0x40ac,0x40ad,0x40ae,0x40af,0x40b0,0x40b1,0x40b2,0x40b3,0x40b4,0x40b5,0x40b6,0x40b7
    .hword 0x40b8,0x40b9,0x40ba,0x40bb,0x40bc,0x40bd,0x40be,0x40bf,0x40c0,0x40c1,0x40c2,0x40c3,0x40c4,0x40c5,0x40c6,0x40c7
    .hword 0x40c8,0x40c9,0x40ca,0x40cb,0x40cc,0x40cd,0x40ce,0x40cf,0x40d0,0x40d1,0x40d2,0x40d3,0x40d4,0x40d5,0x40d6,0x40d7
    .hword 0x40d8,0x40d9,0x40da,0x40db,0x40dc,0x40dd,0x40de,0x40df,0x40e0,0x40e1,0x40e2,0x40e3,0x40e4,0x40e5,0x40e6,0x40e7
    .hword 0x40e8,0x40e9,0x40ea,0x40eb,0x40ec,0x40ed,0x40ee,0x40ef,0x40f0,0x40f1,0x40f2,0x40f3,0x40f4,0x40f5,0x40f6,0x40f7
    .hword 0x40f8,0x40f9,0x40fa,0x40fb,0x40fc,0x40fd,0x40fe,0x40ff,0x4100,0x4101,0x4102,0x4103,0x4104,0x4105,0x4106,0x4107
    .hword 0x4108,0x4109,0x410a,0x410b,0x410c,0x410d,0x410e,0x410f,0x4110,0x4111,0x4112,0x4113,0x4114,0x4115,0x4116,0x4117
    .hword 0x4118,0x4119,0x411a,0x411b,0x411c,0x411d,0x411e,0x411f,0x4120,0x4121,0x4122,0x4123,0x4124,0x4125,0x4126,0x4127
    .hword 0x4128,0x4129,0x412a,0x412b,0x412c,0x412d,0x412e,0x412f,0x4130,0x4131,0x4132,0x4133,0x4134,0x4135,0x4136,0x4137
    .hword 0x4138,0x4139,0x413a,0x413b,0x413c,0x413d,0x413e,0x413f,0x4140,0x4141,0x4142,0x4143,0x4144,0x4145,0x4146,0x4147
    .hword 0x4148,0x4149,0x414a,0x414b,0x414c,0x414d,0x414e,0x414f,0x4150,0x4151,0x4152,0x4153,0x4154,0x4155,0x4156,0x4157
    .hword 0x4158,0x4159,0x415a,0x415b,0x415c,0x415d,0x415e,0x415f,0x4160,0x4161,0x4162,0x4163,0x4164,0x4165,0x4166,0x4167
    .hword 0x4168,0x4169,0x416a,0x416b,0x416c,0x416d,0x416e,0x416f,0x4170,0x4171,0x4172,0x4173,0x4174,0x4175,0x4176,0x4177
    .hword 0x4178,0x4179,0x417a,0x417b,0x417c,0x417d,0x417e,0x417f,0x4180,0x4181,0x4182,0x4183,0x4184,0x4185,0x4186,0x4187
    .hword 0x4188,0x4189,0x418a,0x418b,0x418c,0x418d,0x418e,0x418f,0x4190,0x4191,0x4192,0x4193,0x4194,0x4195,0x4196,0x4197
    .hword 0x4198,0x4199,0x419a,0x419b,0x419c,0x419d,0x419e,0x419f,0x41a0,0x41a1,0x41a2,0x41a3,0x41a4,0x41a5,0x41a6,0x41a7
    .hword 0x41a8,0x41a9,0x41aa,0x41ab,0x41ac,0x41ad,0x41ae,0x41af,0x41b0,0x41b1,0x41b2,0x41b3,0x41b4,0x41b5,0x41b6,0x41b7
    .hword 0x41b8,0x41b9,0x41ba,0x41bb,0x41bc,0x41bd,0x41be,0x41bf,0x41c0,0x41c1,0x41c2,0x41c3,0x41c4,0x41c5,0x41c6,0x41c7
    .hword 0x41c8,0x41c9,0x41ca,0x41cb,0x41cc,0x41cd,0x41ce,0x41cf,0x41d0,0x41d1,0x41d2,0x41d3,0x41d4,0x41d5,0x41d6,0x41d7
    .hword 0x41d8,0x41d9,0x41da,0x41db,0x41dc,0x41dd,0x41de,0x41df,0x41e0,0x41e1,0x41e2,0x41e3,0x41e4,0x41e5,0x41e6,0x41e7
    .hword 0x41e8,0x41e9,0x41ea,0x41eb,0x41ec,0x41ed,0x41ee,0x41ef,0x41f0,0x41f1,0x41f2,0x41f3,0x41f4,0x41f5,0x41f6,0x41f7
    .hword 0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007
    .hword 0x0008,0x0009,0x000a,0x000b,0x000c,0x000d,0x000e,0x000f,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017
    .hword 0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027
    .hword 0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037
    .hword 0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x00a0,0x00a1,0x00a2,0x00a3,0x00a4,0x00a5,0x00a6,0x00a7
    .hword 0x00a8,0x00a9,0x00aa,0x00ab,0x00ac,0x00ad,0x00ae,0x00af,0x00b0,0x00b1,0x00b2,0x00b3,0x00b4,0x00b5,0x00b6,0x00b7
    .hword 0x00b8,0x00b9,0x00ba,0x00bb,0x00bc,0x00bd,0x00be,0x00bf,0x00c0,0x00c1,0x00c2,0x00c3,0x00c4,0x00c5,0x00c6,0x00c7
    .hword 0x00c8,0x00c9,0x00ca,0x00cb,0x00cc,0x00cd,0x00ce,0x00cf,0x00d0,0x00d1,0x00d2,0x00d3,0x00d4,0x00d5,0x00d6,0x00d7
    .hword 0x00d8,0x00d9,0x00da,0x00db,0x00dc,0x00dd,0x00de,0x00df,0x00e0,0x00e1,0x00e2,0x00e3,0x00e4,0x00e5,0x00e6,0x00e7
    .hword 0x00e8,0x00e9,0x00ea,0x00eb,0x00ec,0x00ed,0x00ee,0x00ef,0x00f0,0x00f1,0x00f2,0x00f3,0x00f4,0x00f5,0x00f6,0x00f7
    .hword 0x00f8,0x00f9,0x00fa,0x00fb,0x00fc,0x00fd,0x00fe,0x00ff,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x81f8,0x81f9,0x81fa,0x81fb,0x81fc,0x81fd,0x81fe,0x81ff,0x8000,0x8001,0x8002,0x8003,0x8004,0x8005,0x8006,0x8007
    .hword 0x8008,0x8009,0x800a,0x800b,0x800c,0x800d,0x800e,0x800f,0x8010,0x8011,0x8012,0x8013,0x8014,0x8015,0x8016,0x8017
    .hword 0x8018,0x8019,0x801a,0x801b,0x801c,0x801d,0x801e,0x801f,0x8020,0x8021,0x8022,0x8023,0x8024,0x8025,0x8026,0x8027
    .hword 0x8028,0x8029,0x802a,0x802b,0x802c,0x802d,0x802e,0x802f,0x8030,0x8031,0x8032,0x8033,0x8034,0x8035,0x8036,0x8037
    .hword 0x8038,0x8039,0x803a,0x803b,0x803c,0x803d,0x803e,0x803f,0x8040,0x8041,0x8042,0x8043,0x8044,0x8045,0x8046,0x8047
    .hword 0x8048,0x8049,0x804a,0x804b,0x804c,0x804d,0x804e,0x804f,0x8050,0x8051,0x8052,0x8053,0x8054,0x8055,0x8056,0x8057
    .hword 0x8058,0x8059,0x805a,0x805b,0x805c,0x805d,0x805e,0x805f,0x8060,0x8061,0x8062,0x8063,0x8064,0x8065,0x8066,0x8067
    .hword 0x8068,0x8069,0x806a,0x806b,0x806c,0x806d,0x806e,0x806f,0x8070,0x8071,0x8072,0x8073,0x8074,0x8075,0x8076,0x8077
    .hword 0x8078,0x8079,0x807a,0x807b,0x807c,0x807d,0x807e,0x807f,0x8080,0x8081,0x8082,0x8083,0x8084,0x8085,0x8086,0x8087
    .hword 0x8088,0x8089,0x808a,0x808b,0x808c,0x808d,0x808e,0x808f,0x8090,0x8091,0x8092,0x8093,0x8094,0x8095,0x8096,0x8097
    .hword 0x8098,0x8099,0x809a,0x809b,0x809c,0x809d,0x809e,0x809f,0x80a0,0x80a1,0x80a2,0x80a3,0x80a4,0x80a5,0x80a6,0x80a7
    .hword 0x80a8,0x80a9,0x80aa,0x80ab,0x80ac,0x80ad,0x80ae,0x80af,0x80b0,0x80b1,0x80b2,0x80b3,0x80b4,0x80b5,0x80b6,0x80b7
    .hword 0x80b8,0x80b9,0x80ba,0x80bb,0x80bc,0x80bd,0x80be,0x80bf,0x80c0,0x80c1,0x80c2,0x80c3,0x80c4,0x80c5,0x80c6,0x80c7
    .hword 0x80c8,0x80c9,0x80ca,0x80cb,0x80cc,0x80cd,0x80ce,0x80cf,0x80d0,0x80d1,0x80d2,0x80d3,0x80d4,0x80d5,0x80d6,0x80d7
    .hword 0x80d8,0x80d9,0x80da,0x80db,0x80dc,0x80dd,0x80de,0x80df,0x80e0,0x80e1,0x80e2,0x80e3,0x80e4,0x80e5,0x80e6,0x80e7
    .hword 0x80e8,0x80e9,0x80ea,0x80eb,0x80ec,0x80ed,0x80ee,0x80ef,0x80f0,0x80f1,0x80f2,0x80f3,0x80f4,0x80f5,0x80f6,0x80f7
    .hword 0x80f8,0x80f9,0x80fa,0x80fb,0x80fc,0x80fd,0x80fe,0x80ff,0x8100,0x8101,0x8102,0x8103,0x8104,0x8105,0x8106,0x8107
    .hword 0x8108,0x8109,0x810a,0x810b,0x810c,0x810d,0x810e,0x810f,0x8110,0x8111,0x8112,0x8113,0x8114,0x8115,0x8116,0x8117
    .hword 0x8118,0x8119,0x811a,0x811b,0x811c,0x811d,0x811e,0x811f,0x8120,0x8121,0x8122,0x8123,0x8124,0x8125,0x8126,0x8127
    .hword 0x8128,0x8129,0x812a,0x812b,0x812c,0x812d,0x812e,0x812f,0x8130,0x8131,0x8132,0x8133,0x8134,0x8135,0x8136,0x8137
    .hword 0x8138,0x8139,0x813a,0x813b,0x813c,0x813d,0x813e,0x813f,0x8140,0x8141,0x8142,0x8143,0x8144,0x8145,0x8146,0x8147
    .hword 0x8148,0x8149,0x814a,0x814b,0x814c,0x814d,0x814e,0x814f,0x8150,0x8151,0x8152,0x8153,0x8154,0x8155,0x8156,0x8157
    .hword 0x8158,0x8159,0x815a,0x815b,0x815c,0x815d,0x815e,0x815f,0x8160,0x8161,0x8162,0x8163,0x8164,0x8165,0x8166,0x8167
    .hword 0x8168,0x8169,0x816a,0x816b,0x816c,0x816d,0x816e,0x816f,0x8170,0x8171,0x8172,0x8173,0x8174,0x8175,0x8176,0x8177
    .hword 0x8178,0x8179,0x817a,0x817b,0x817c,0x817d,0x817e,0x817f,0x8180,0x8181,0x8182,0x8183,0x8184,0x8185,0x8186,0x8187
    .hword 0x8188,0x8189,0x818a,0x818b,0x818c,0x818d,0x818e,0x818f,0x8190,0x8191,0x8192,0x8193,0x8194,0x8195,0x8196,0x8197
    .hword 0x8198,0x8199,0x819a,0x819b,0x819c,0x819d,0x819e,0x819f,0x81a0,0x81a1,0x81a2,0x81a3,0x81a4,0x81a5,0x81a6,0x81a7
    .hword 0x81a8,0x81a9,0x81aa,0x81ab,0x81ac,0x81ad,0x81ae,0x81af,0x81b0,0x81b1,0x81b2,0x81b3,0x81b4,0x81b5,0x81b6,0x81b7
    .hword 0x81b8,0x81b9,0x81ba,0x81bb,0x81bc,0x81bd,0x81be,0x81bf,0x81c0,0x81c1,0x81c2,0x81c3,0x81c4,0x81c5,0x81c6,0x81c7
    .hword 0x81c8,0x81c9,0x81ca,0x81cb,0x81cc,0x81cd,0x81ce,0x81cf,0x81d0,0x81d1,0x81d2,0x81d3,0x81d4,0x81d5,0x81d6,0x81d7
    .hword 0x81d8,0x81d9,0x81da,0x81db,0x81dc,0x81dd,0x81de,0x81df,0x81e0,0x81e1,0x81e2,0x81e3,0x81e4,0x81e5,0x81e6,0x81e7
    .hword 0x81e8,0x81e9,0x81ea,0x81eb,0x81ec,0x81ed,0x81ee,0x81ef,0x81f0,0x81f1,0x81f2,0x81f3,0x81f4,0x81f5,0x81f6,0x81f7
    .hword 0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007
    .hword 0x0008,0x0009,0x000a,0x000b,0x000c,0x000d,0x000e,0x000f,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017
    .hword 0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027
    .hword 0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037
    .hword 0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x00a0,0x00a1,0x00a2,0x00a3,0x00a4,0x00a5,0x00a6,0x00a7
    .hword 0x00a8,0x00a9,0x00aa,0x00ab,0x00ac,0x00ad,0x00ae,0x00af,0x00b0,0x00b1,0x00b2,0x00b3,0x00b4,0x00b5,0x00b6,0x00b7
    .hword 0x00b8,0x00b9,0x00ba,0x00bb,0x00bc,0x00bd,0x00be,0x00bf,0x00c0,0x00c1,0x00c2,0x00c3,0x00c4,0x00c5,0x00c6,0x00c7
    .hword 0x00c8,0x00c9,0x00ca,0x00cb,0x00cc,0x00cd,0x00ce,0x00cf,0x00d0,0x00d1,0x00d2,0x00d3,0x00d4,0x00d5,0x00d6,0x00d7
    .hword 0x00d8,0x00d9,0x00da,0x00db,0x00dc,0x00dd,0x00de,0x00df,0x00e0,0x00e1,0x00e2,0x00e3,0x00e4,0x00e5,0x00e6,0x00e7
    .hword 0x00e8,0x00e9,0x00ea,0x00eb,0x00ec,0x00ed,0x00ee,0x00ef,0x00f0,0x00f1,0x00f2,0x00f3,0x00f4,0x00f5,0x00f6,0x00f7
    .hword 0x00f8,0x00f9,0x00fa,0x00fb,0x00fc,0x00fd,0x00fe,0x00ff,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0xc1f8,0xc1f9,0xc1fa,0xc1fb,0xc1fc,0xc1fd,0xc1fe,0xc1ff,0xc000,0xc001,0xc002,0xc003,0xc004,0xc005,0xc006,0xc007
    .hword 0xc008,0xc009,0xc00a,0xc00b,0xc00c,0xc00d,0xc00e,0xc00f,0xc010,0xc011,0xc012,0xc013,0xc014,0xc015,0xc016,0xc017
    .hword 0xc018,0xc019,0xc01a,0xc01b,0xc01c,0xc01d,0xc01e,0xc01f,0xc020,0xc021,0xc022,0xc023,0xc024,0xc025,0xc026,0xc027
    .hword 0xc028,0xc029,0xc02a,0xc02b,0xc02c,0xc02d,0xc02e,0xc02f,0xc030,0xc031,0xc032,0xc033,0xc034,0xc035,0xc036,0xc037
    .hword 0xc038,0xc039,0xc03a,0xc03b,0xc03c,0xc03d,0xc03e,0xc03f,0xc040,0xc041,0xc042,0xc043,0xc044,0xc045,0xc046,0xc047
    .hword 0xc048,0xc049,0xc04a,0xc04b,0xc04c,0xc04d,0xc04e,0xc04f,0xc050,0xc051,0xc052,0xc053,0xc054,0xc055,0xc056,0xc057
    .hword 0xc058,0xc059,0xc05a,0xc05b,0xc05c,0xc05d,0xc05e,0xc05f,0xc060,0xc061,0xc062,0xc063,0xc064,0xc065,0xc066,0xc067
    .hword 0xc068,0xc069,0xc06a,0xc06b,0xc06c,0xc06d,0xc06e,0xc06f,0xc070,0xc071,0xc072,0xc073,0xc074,0xc075,0xc076,0xc077
    .hword 0xc078,0xc079,0xc07a,0xc07b,0xc07c,0xc07d,0xc07e,0xc07f,0xc080,0xc081,0xc082,0xc083,0xc084,0xc085,0xc086,0xc087
    .hword 0xc088,0xc089,0xc08a,0xc08b,0xc08c,0xc08d,0xc08e,0xc08f,0xc090,0xc091,0xc092,0xc093,0xc094,0xc095,0xc096,0xc097
    .hword 0xc098,0xc099,0xc09a,0xc09b,0xc09c,0xc09d,0xc09e,0xc09f,0xc0a0,0xc0a1,0xc0a2,0xc0a3,0xc0a4,0xc0a5,0xc0a6,0xc0a7
    .hword 0xc0a8,0xc0a9,0xc0aa,0xc0ab,0xc0ac,0xc0ad,0xc0ae,0xc0af,0xc0b0,0xc0b1,0xc0b2,0xc0b3,0xc0b4,0xc0b5,0xc0b6,0xc0b7
    .hword 0xc0b8,0xc0b9,0xc0ba,0xc0bb,0xc0bc,0xc0bd,0xc0be,0xc0bf,0xc0c0,0xc0c1,0xc0c2,0xc0c3,0xc0c4,0xc0c5,0xc0c6,0xc0c7
    .hword 0xc0c8,0xc0c9,0xc0ca,0xc0cb,0xc0cc,0xc0cd,0xc0ce,0xc0cf,0xc0d0,0xc0d1,0xc0d2,0xc0d3,0xc0d4,0xc0d5,0xc0d6,0xc0d7
    .hword 0xc0d8,0xc0d9,0xc0da,0xc0db,0xc0dc,0xc0dd,0xc0de,0xc0df,0xc0e0,0xc0e1,0xc0e2,0xc0e3,0xc0e4,0xc0e5,0xc0e6,0xc0e7
    .hword 0xc0e8,0xc0e9,0xc0ea,0xc0eb,0xc0ec,0xc0ed,0xc0ee,0xc0ef,0xc0f0,0xc0f1,0xc0f2,0xc0f3,0xc0f4,0xc0f5,0xc0f6,0xc0f7
    .hword 0xc0f8,0xc0f9,0xc0fa,0xc0fb,0xc0fc,0xc0fd,0xc0fe,0xc0ff,0xc100,0xc101,0xc102,0xc103,0xc104,0xc105,0xc106,0xc107
    .hword 0xc108,0xc109,0xc10a,0xc10b,0xc10c,0xc10d,0xc10e,0xc10f,0xc110,0xc111,0xc112,0xc113,0xc114,0xc115,0xc116,0xc117
    .hword 0xc118,0xc119,0xc11a,0xc11b,0xc11c,0xc11d,0xc11e,0xc11f,0xc120,0xc121,0xc122,0xc123,0xc124,0xc125,0xc126,0xc127
    .hword 0xc128,0xc129,0xc12a,0xc12b,0xc12c,0xc12d,0xc12e,0xc12f,0xc130,0xc131,0xc132,0xc133,0xc134,0xc135,0xc136,0xc137
    .hword 0xc138,0xc139,0xc13a,0xc13b,0xc13c,0xc13d,0xc13e,0xc13f,0xc140,0xc141,0xc142,0xc143,0xc144,0xc145,0xc146,0xc147
    .hword 0xc148,0xc149,0xc14a,0xc14b,0xc14c,0xc14d,0xc14e,0xc14f,0xc150,0xc151,0xc152,0xc153,0xc154,0xc155,0xc156,0xc157
    .hword 0xc158,0xc159,0xc15a,0xc15b,0xc15c,0xc15d,0xc15e,0xc15f,0xc160,0xc161,0xc162,0xc163,0xc164,0xc165,0xc166,0xc167
    .hword 0xc168,0xc169,0xc16a,0xc16b,0xc16c,0xc16d,0xc16e,0xc16f,0xc170,0xc171,0xc172,0xc173,0xc174,0xc175,0xc176,0xc177
    .hword 0xc178,0xc179,0xc17a,0xc17b,0xc17c,0xc17d,0xc17e,0xc17f,0xc180,0xc181,0xc182,0xc183,0xc184,0xc185,0xc186,0xc187
    .hword 0xc188,0xc189,0xc18a,0xc18b,0xc18c,0xc18d,0xc18e,0xc18f,0xc190,0xc191,0xc192,0xc193,0xc194,0xc195,0xc196,0xc197
    .hword 0xc198,0xc199,0xc19a,0xc19b,0xc19c,0xc19d,0xc19e,0xc19f,0xc1a0,0xc1a1,0xc1a2,0xc1a3,0xc1a4,0xc1a5,0xc1a6,0xc1a7
    .hword 0xc1a8,0xc1a9,0xc1aa,0xc1ab,0xc1ac,0xc1ad,0xc1ae,0xc1af,0xc1b0,0xc1b1,0xc1b2,0xc1b3,0xc1b4,0xc1b5,0xc1b6,0xc1b7
    .hword 0xc1b8,0xc1b9,0xc1ba,0xc1bb,0xc1bc,0xc1bd,0xc1be,0xc1bf,0xc1c0,0xc1c1,0xc1c2,0xc1c3,0xc1c4,0xc1c5,0xc1c6,0xc1c7
    .hword 0xc1c8,0xc1c9,0xc1ca,0xc1cb,0xc1cc,0xc1cd,0xc1ce,0xc1cf,0xc1d0,0xc1d1,0xc1d2,0xc1d3,0xc1d4,0xc1d5,0xc1d6,0xc1d7
    .hword 0xc1d8,0xc1d9,0xc1da,0xc1db,0xc1dc,0xc1dd,0xc1de,0xc1df,0xc1e0,0xc1e1,0xc1e2,0xc1e3,0xc1e4,0xc1e5,0xc1e6,0xc1e7
    .hword 0xc1e8,0xc1e9,0xc1ea,0xc1eb,0xc1ec,0xc1ed,0xc1ee,0xc1ef,0xc1f0,0xc1f1,0xc1f2,0xc1f3,0xc1f4,0xc1f5,0xc1f6,0xc1f7
    .hword 0x41f8,0x41f9,0x41fa,0x41fb,0x41fc,0x41fd,0x41fe,0x41ff,0x4000,0x4001,0x4002,0x4003,0x4004,0x4005,0x4006,0x4007
    .hword 0x4008,0x4009,0x400a,0x400b,0x400c,0x400d,0x400e,0x400f,0x4010,0x4011,0x4012,0x4013,0x4014,0x4015,0x4016,0x4017
    .hword 0x4018,0x4019,0x401a,0x401b,0x401c,0x401d,0x401e,0x401f,0x4020,0x4021,0x4022,0x4023,0x4024,0x4025,0x4026,0x4027
    .hword 0x4028,0x4029,0x402a,0x402b,0x402c,0x402d,0x402e,0x402f,0x4030,0x4031,0x4032,0x4033,0x4034,0x4035,0x4036,0x4037
    .hword 0x4038,0x4039,0x403a,0x403b,0x403c,0x403d,0x403e,0x403f,0x4040,0x4041,0x4042,0x4043,0x4044,0x4045,0x4046,0x4047
    .hword 0x4048,0x4049,0x404a,0x404b,0x404c,0x404d,0x404e,0x404f,0x4050,0x4051,0x4052,0x4053,0x4054,0x4055,0x4056,0x4057
    .hword 0x4058,0x4059,0x405a,0x405b,0x405c,0x405d,0x405e,0x405f,0x4060,0x4061,0x4062,0x4063,0x4064,0x4065,0x4066,0x4067
    .hword 0x4068,0x4069,0x406a,0x406b,0x406c,0x406d,0x406e,0x406f,0x4070,0x4071,0x4072,0x4073,0x4074,0x4075,0x4076,0x4077
    .hword 0x4078,0x4079,0x407a,0x407b,0x407c,0x407d,0x407e,0x407f,0x4080,0x4081,0x4082,0x4083,0x4084,0x4085,0x4086,0x4087
    .hword 0x4088,0x4089,0x408a,0x408b,0x408c,0x408d,0x408e,0x408f,0x4090,0x4091,0x4092,0x4093,0x4094,0x4095,0x4096,0x4097
    .hword 0x4098,0x4099,0x409a,0x409b,0x409c,0x409d,0x409e,0x409f,0x40a0,0x40a1,0x40a2,0x40a3,0x40a4,0x40a5,0x40a6,0x40a7
    .hword 0x40a8,0x40a9,0x40aa,0x40ab,0x40ac,0x40ad,0x40ae,0x40af,0x40b0,0x40b1,0x40b2,0x40b3,0x40b4,0x40b5,0x40b6,0x40b7
    .hword 0x40b8,0x40b9,0x40ba,0x40bb,0x40bc,0x40bd,0x40be,0x40bf,0x40c0,0x40c1,0x40c2,0x40c3,0x40c4,0x40c5,0x40c6,0x40c7
    .hword 0x40c8,0x40c9,0x40ca,0x40cb,0x40cc,0x40cd,0x40ce,0x40cf,0x40d0,0x40d1,0x40d2,0x40d3,0x40d4,0x40d5,0x40d6,0x40d7
    .hword 0x40d8,0x40d9,0x40da,0x40db,0x40dc,0x40dd,0x40de,0x40df,0x40e0,0x40e1,0x40e2,0x40e3,0x40e4,0x40e5,0x40e6,0x40e7
    .hword 0x40e8,0x40e9,0x40ea,0x40eb,0x40ec,0x40ed,0x40ee,0x40ef,0x40f0,0x40f1,0x40f2,0x40f3,0x40f4,0x40f5,0x40f6,0x40f7
    .hword 0x40f8,0x40f9,0x40fa,0x40fb,0x40fc,0x40fd,0x40fe,0x40ff,0x4100,0x4101,0x4102,0x4103,0x4104,0x4105,0x4106,0x4107
    .hword 0x4108,0x4109,0x410a,0x410b,0x410c,0x410d,0x410e,0x410f,0x4110,0x4111,0x4112,0x4113,0x4114,0x4115,0x4116,0x4117
    .hword 0x4118,0x4119,0x411a,0x411b,0x411c,0x411d,0x411e,0x411f,0x4120,0x4121,0x4122,0x4123,0x4124,0x4125,0x4126,0x4127
    .hword 0x4128,0x4129,0x412a,0x412b,0x412c,0x412d,0x412e,0x412f,0x4130,0x4131,0x4132,0x4133,0x4134,0x4135,0x4136,0x4137
    .hword 0x4138,0x4139,0x413a,0x413b,0x413c,0x413d,0x413e,0x413f,0x4140,0x4141,0x4142,0x4143,0x4144,0x4145,0x4146,0x4147
    .hword 0x4148,0x4149,0x414a,0x414b,0x414c,0x414d,0x414e,0x414f,0x4150,0x4151,0x4152,0x4153,0x4154,0x4155,0x4156,0x4157
    .hword 0x4158,0x4159,0x415a,0x415b,0x415c,0x415d,0x415e,0x415f,0x4160,0x4161,0x4162,0x4163,0x4164,0x4165,0x4166,0x4167
    .hword 0x4168,0x4169,0x416a,0x416b,0x416c,0x416d,0x416e,0x416f,0x4170,0x4171,0x4172,0x4173,0x4174,0x4175,0x4176,0x4177
    .hword 0x4178,0x4179,0x417a,0x417b,0x417c,0x417d,0x417e,0x417f,0x4180,0x4181,0x4182,0x4183,0x4184,0x4185,0x4186,0x4187
    .hword 0x4188,0x4189,0x418a,0x418b,0x418c,0x418d,0x418e,0x418f,0x4190,0x4191,0x4192,0x4193,0x4194,0x4195,0x4196,0x4197
    .hword 0x4198,0x4199,0x419a,0x419b,0x419c,0x419d,0x419e,0x419f,0x41a0,0x41a1,0x41a2,0x41a3,0x41a4,0x41a5,0x41a6,0x41a7
    .hword 0x41a8,0x41a9,0x41aa,0x41ab,0x41ac,0x41ad,0x41ae,0x41af,0x41b0,0x41b1,0x41b2,0x41b3,0x41b4,0x41b5,0x41b6,0x41b7
    .hword 0x41b8,0x41b9,0x41ba,0x41bb,0x41bc,0x41bd,0x41be,0x41bf,0x41c0,0x41c1,0x41c2,0x41c3,0x41c4,0x41c5,0x41c6,0x41c7
    .hword 0x41c8,0x41c9,0x41ca,0x41cb,0x41cc,0x41cd,0x41ce,0x41cf,0x41d0,0x41d1,0x41d2,0x41d3,0x41d4,0x41d5,0x41d6,0x41d7
    .hword 0x41d8,0x41d9,0x41da,0x41db,0x41dc,0x41dd,0x41de,0x41df,0x41e0,0x41e1,0x41e2,0x41e3,0x41e4,0x41e5,0x41e6,0x41e7
    .hword 0x41e8,0x41e9,0x41ea,0x41eb,0x41ec,0x41ed,0x41ee,0x41ef,0x41f0,0x41f1,0x41f2,0x41f3,0x41f4,0x41f5,0x41f6,0x41f7
    .hword 0x81f8,0x81f9,0x81fa,0x81fb,0x81fc,0x81fd,0x81fe,0x81ff,0x8000,0x8001,0x8002,0x8003,0x8004,0x8005,0x8006,0x8007
    .hword 0x8008,0x8009,0x800a,0x800b,0x800c,0x800d,0x800e,0x800f,0x8010,0x8011,0x8012,0x8013,0x8014,0x8015,0x8016,0x8017
    .hword 0x8018,0x8019,0x801a,0x801b,0x801c,0x801d,0x801e,0x801f,0x8020,0x8021,0x8022,0x8023,0x8024,0x8025,0x8026,0x8027
    .hword 0x8028,0x8029,0x802a,0x802b,0x802c,0x802d,0x802e,0x802f,0x8030,0x8031,0x8032,0x8033,0x8034,0x8035,0x8036,0x8037
    .hword 0x8038,0x8039,0x803a,0x803b,0x803c,0x803d,0x803e,0x803f,0x8040,0x8041,0x8042,0x8043,0x8044,0x8045,0x8046,0x8047
    .hword 0x8048,0x8049,0x804a,0x804b,0x804c,0x804d,0x804e,0x804f,0x8050,0x8051,0x8052,0x8053,0x8054,0x8055,0x8056,0x8057
    .hword 0x8058,0x8059,0x805a,0x805b,0x805c,0x805d,0x805e,0x805f,0x8060,0x8061,0x8062,0x8063,0x8064,0x8065,0x8066,0x8067
    .hword 0x8068,0x8069,0x806a,0x806b,0x806c,0x806d,0x806e,0x806f,0x8070,0x8071,0x8072,0x8073,0x8074,0x8075,0x8076,0x8077
    .hword 0x8078,0x8079,0x807a,0x807b,0x807c,0x807d,0x807e,0x807f,0x8080,0x8081,0x8082,0x8083,0x8084,0x8085,0x8086,0x8087
    .hword 0x8088,0x8089,0x808a,0x808b,0x808c,0x808d,0x808e,0x808f,0x8090,0x8091,0x8092,0x8093,0x8094,0x8095,0x8096,0x8097
    .hword 0x8098,0x8099,0x809a,0x809b,0x809c,0x809d,0x809e,0x809f,0x80a0,0x80a1,0x80a2,0x80a3,0x80a4,0x80a5,0x80a6,0x80a7
    .hword 0x80a8,0x80a9,0x80aa,0x80ab,0x80ac,0x80ad,0x80ae,0x80af,0x80b0,0x80b1,0x80b2,0x80b3,0x80b4,0x80b5,0x80b6,0x80b7
    .hword 0x80b8,0x80b9,0x80ba,0x80bb,0x80bc,0x80bd,0x80be,0x80bf,0x80c0,0x80c1,0x80c2,0x80c3,0x80c4,0x80c5,0x80c6,0x80c7
    .hword 0x80c8,0x80c9,0x80ca,0x80cb,0x80cc,0x80cd,0x80ce,0x80cf,0x80d0,0x80d1,0x80d2,0x80d3,0x80d4,0x80d5,0x80d6,0x80d7
    .hword 0x80d8,0x80d9,0x80da,0x80db,0x80dc,0x80dd,0x80de,0x80df,0x80e0,0x80e1,0x80e2,0x80e3,0x80e4,0x80e5,0x80e6,0x80e7
    .hword 0x80e8,0x80e9,0x80ea,0x80eb,0x80ec,0x80ed,0x80ee,0x80ef,0x80f0,0x80f1,0x80f2,0x80f3,0x80f4,0x80f5,0x80f6,0x80f7
    .hword 0x80f8,0x80f9,0x80fa,0x80fb,0x80fc,0x80fd,0x80fe,0x80ff,0x8100,0x8101,0x8102,0x8103,0x8104,0x8105,0x8106,0x8107
    .hword 0x8108,0x8109,0x810a,0x810b,0x810c,0x810d,0x810e,0x810f,0x8110,0x8111,0x8112,0x8113,0x8114,0x8115,0x8116,0x8117
    .hword 0x8118,0x8119,0x811a,0x811b,0x811c,0x811d,0x811e,0x811f,0x8120,0x8121,0x8122,0x8123,0x8124,0x8125,0x8126,0x8127
    .hword 0x8128,0x8129,0x812a,0x812b,0x812c,0x812d,0x812e,0x812f,0x8130,0x8131,0x8132,0x8133,0x8134,0x8135,0x8136,0x8137
    .hword 0x8138,0x8139,0x813a,0x813b,0x813c,0x813d,0x813e,0x813f,0x8140,0x8141,0x8142,0x8143,0x8144,0x8145,0x8146,0x8147
    .hword 0x8148,0x8149,0x814a,0x814b,0x814c,0x814d,0x814e,0x814f,0x8150,0x8151,0x8152,0x8153,0x8154,0x8155,0x8156,0x8157
    .hword 0x8158,0x8159,0x815a,0x815b,0x815c,0x815d,0x815e,0x815f,0x8160,0x8161,0x8162,0x8163,0x8164,0x8165,0x8166,0x8167
    .hword 0x8168,0x8169,0x816a,0x816b,0x816c,0x816d,0x816e,0x816f,0x8170,0x8171,0x8172,0x8173,0x8174,0x8175,0x8176,0x8177
    .hword 0x8178,0x8179,0x817a,0x817b,0x817c,0x817d,0x817e,0x817f,0x8180,0x8181,0x8182,0x8183,0x8184,0x8185,0x8186,0x8187
    .hword 0x8188,0x8189,0x818a,0x818b,0x818c,0x818d,0x818e,0x818f,0x8190,0x8191,0x8192,0x8193,0x8194,0x8195,0x8196,0x8197
    .hword 0x8198,0x8199,0x819a,0x819b,0x819c,0x819d,0x819e,0x819f,0x81a0,0x81a1,0x81a2,0x81a3,0x81a4,0x81a5,0x81a6,0x81a7
    .hword 0x81a8,0x81a9,0x81aa,0x81ab,0x81ac,0x81ad,0x81ae,0x81af,0x81b0,0x81b1,0x81b2,0x81b3,0x81b4,0x81b5,0x81b6,0x81b7
    .hword 0x81b8,0x81b9,0x81ba,0x81bb,0x81bc,0x81bd,0x81be,0x81bf,0x81c0,0x81c1,0x81c2,0x81c3,0x81c4,0x81c5,0x81c6,0x81c7
    .hword 0x81c8,0x81c9,0x81ca,0x81cb,0x81cc,0x81cd,0x81ce,0x81cf,0x81d0,0x81d1,0x81d2,0x81d3,0x81d4,0x81d5,0x81d6,0x81d7
    .hword 0x81d8,0x81d9,0x81da,0x81db,0x81dc,0x81dd,0x81de,0x81df,0x81e0,0x81e1,0x81e2,0x81e3,0x81e4,0x81e5,0x81e6,0x81e7
    .hword 0x81e8,0x81e9,0x81ea,0x81eb,0x81ec,0x81ed,0x81ee,0x81ef,0x81f0,0x81f1,0x81f2,0x81f3,0x81f4,0x81f5,0x81f6,0x81f7
    .hword 0x41f8,0x41f9,0x41fa,0x41fb,0x41fc,0x41fd,0x41fe,0x41ff,0x4000,0x4001,0x4002,0x4003,0x4004,0x4005,0x4006,0x4007
    .hword 0x4008,0x4009,0x400a,0x400b,0x400c,0x400d,0x400e,0x400f,0x4010,0x4011,0x4012,0x4013,0x4014,0x4015,0x4016,0x4017
    .hword 0x4018,0x4019,0x401a,0x401b,0x401c,0x401d,0x401e,0x401f,0x4020,0x4021,0x4022,0x4023,0x4024,0x4025,0x4026,0x4027
    .hword 0x4028,0x4029,0x402a,0x402b,0x402c,0x402d,0x402e,0x402f,0x4030,0x4031,0x4032,0x4033,0x4034,0x4035,0x4036,0x4037
    .hword 0x4038,0x4039,0x403a,0x403b,0x403c,0x403d,0x403e,0x403f,0x4040,0x4041,0x4042,0x4043,0x4044,0x4045,0x4046,0x4047
    .hword 0x4048,0x4049,0x404a,0x404b,0x404c,0x404d,0x404e,0x404f,0x4050,0x4051,0x4052,0x4053,0x4054,0x4055,0x4056,0x4057
    .hword 0x4058,0x4059,0x405a,0x405b,0x405c,0x405d,0x405e,0x405f,0x4060,0x4061,0x4062,0x4063,0x4064,0x4065,0x4066,0x4067
    .hword 0x4068,0x4069,0x406a,0x406b,0x406c,0x406d,0x406e,0x406f,0x4070,0x4071,0x4072,0x4073,0x4074,0x4075,0x4076,0x4077
    .hword 0x4078,0x4079,0x407a,0x407b,0x407c,0x407d,0x407e,0x407f,0x4080,0x4081,0x4082,0x4083,0x4084,0x4085,0x4086,0x4087
    .hword 0x4088,0x4089,0x408a,0x408b,0x408c,0x408d,0x408e,0x408f,0x4090,0x4091,0x4092,0x4093,0x4094,0x4095,0x4096,0x4097
    .hword 0x4098,0x4099,0x409a,0x409b,0x409c,0x409d,0x409e,0x409f,0x40a0,0x40a1,0x40a2,0x40a3,0x40a4,0x40a5,0x40a6,0x40a7
    .hword 0x40a8,0x40a9,0x40aa,0x40ab,0x40ac,0x40ad,0x40ae,0x40af,0x40b0,0x40b1,0x40b2,0x40b3,0x40b4,0x40b5,0x40b6,0x40b7
    .hword 0x40b8,0x40b9,0x40ba,0x40bb,0x40bc,0x40bd,0x40be,0x40bf,0x40c0,0x40c1,0x40c2,0x40c3,0x40c4,0x40c5,0x40c6,0x40c7
    .hword 0x40c8,0x40c9,0x40ca,0x40cb,0x40cc,0x40cd,0x40ce,0x40cf,0x40d0,0x40d1,0x40d2,0x40d3,0x40d4,0x40d5,0x40d6,0x40d7
    .hword 0x40d8,0x40d9,0x40da,0x40db,0x40dc,0x40dd,0x40de,0x40df,0x40e0,0x40e1,0x40e2,0x40e3,0x40e4,0x40e5,0x40e6,0x40e7
    .hword 0x40e8,0x40e9,0x40ea,0x40eb,0x40ec,0x40ed,0x40ee,0x40ef,0x40f0,0x40f1,0x40f2,0x40f3,0x40f4,0x40f5,0x40f6,0x40f7
    .hword 0x40f8,0x40f9,0x40fa,0x40fb,0x40fc,0x40fd,0x40fe,0x40ff,0x4100,0x4101,0x4102,0x4103,0x4104,0x4105,0x4106,0x4107
    .hword 0x4108,0x4109,0x410a,0x410b,0x410c,0x410d,0x410e,0x410f,0x4110,0x4111,0x4112,0x4113,0x4114,0x4115,0x4116,0x4117
    .hword 0x4118,0x4119,0x411a,0x411b,0x411c,0x411d,0x411e,0x411f,0x4120,0x4121,0x4122,0x4123,0x4124,0x4125,0x4126,0x4127
    .hword 0x4128,0x4129,0x412a,0x412b,0x412c,0x412d,0x412e,0x412f,0x4130,0x4131,0x4132,0x4133,0x4134,0x4135,0x4136,0x4137
    .hword 0x4138,0x4139,0x413a,0x413b,0x413c,0x413d,0x413e,0x413f,0x4140,0x4141,0x4142,0x4143,0x4144,0x4145,0x4146,0x4147
    .hword 0x4148,0x4149,0x414a,0x414b,0x414c,0x414d,0x414e,0x414f,0x4150,0x4151,0x4152,0x4153,0x4154,0x4155,0x4156,0x4157
    .hword 0x4158,0x4159,0x415a,0x415b,0x415c,0x415d,0x415e,0x415f,0x4160,0x4161,0x4162,0x4163,0x4164,0x4165,0x4166,0x4167
    .hword 0x4168,0x4169,0x416a,0x416b,0x416c,0x416d,0x416e,0x416f,0x4170,0x4171,0x4172,0x4173,0x4174,0x4175,0x4176,0x4177
    .hword 0x4178,0x4179,0x417a,0x417b,0x417c,0x417d,0x417e,0x417f,0x4180,0x4181,0x4182,0x4183,0x4184,0x4185,0x4186,0x4187
    .hword 0x4188,0x4189,0x418a,0x418b,0x418c,0x418d,0x418e,0x418f,0x4190,0x4191,0x4192,0x4193,0x4194,0x4195,0x4196,0x4197
    .hword 0x4198,0x4199,0x419a,0x419b,0x419c,0x419d,0x419e,0x419f,0x41a0,0x41a1,0x41a2,0x41a3,0x41a4,0x41a5,0x41a6,0x41a7
    .hword 0x41a8,0x41a9,0x41aa,0x41ab,0x41ac,0x41ad,0x41ae,0x41af,0x41b0,0x41b1,0x41b2,0x41b3,0x41b4,0x41b5,0x41b6,0x41b7
    .hword 0x41b8,0x41b9,0x41ba,0x41bb,0x41bc,0x41bd,0x41be,0x41bf,0x41c0,0x41c1,0x41c2,0x41c3,0x41c4,0x41c5,0x41c6,0x41c7
    .hword 0x41c8,0x41c9,0x41ca,0x41cb,0x41cc,0x41cd,0x41ce,0x41cf,0x41d0,0x41d1,0x41d2,0x41d3,0x41d4,0x41d5,0x41d6,0x41d7
    .hword 0x41d8,0x41d9,0x41da,0x41db,0x41dc,0x41dd,0x41de,0x41df,0x41e0,0x41e1,0x41e2,0x41e3,0x41e4,0x41e5,0x41e6,0x41e7
    .hword 0x41e8,0x41e9,0x41ea,0x41eb,0x41ec,0x41ed,0x41ee,0x41ef,0x41f0,0x41f1,0x41f2,0x41f3,0x41f4,0x41f5,0x41f6,0x41f7
    .hword 0xc1f8,0xc1f9,0xc1fa,0xc1fb,0xc1fc,0xc1fd,0xc1fe,0xc1ff,0xc000,0xc001,0xc002,0xc003,0xc004,0xc005,0xc006,0xc007
    .hword 0xc008,0xc009,0xc00a,0xc00b,0xc00c,0xc00d,0xc00e,0xc00f,0xc010,0xc011,0xc012,0xc013,0xc014,0xc015,0xc016,0xc017
    .hword 0xc018,0xc019,0xc01a,0xc01b,0xc01c,0xc01d,0xc01e,0xc01f,0xc020,0xc021,0xc022,0xc023,0xc024,0xc025,0xc026,0xc027
    .hword 0xc028,0xc029,0xc02a,0xc02b,0xc02c,0xc02d,0xc02e,0xc02f,0xc030,0xc031,0xc032,0xc033,0xc034,0xc035,0xc036,0xc037
    .hword 0xc038,0xc039,0xc03a,0xc03b,0xc03c,0xc03d,0xc03e,0xc03f,0xc040,0xc041,0xc042,0xc043,0xc044,0xc045,0xc046,0xc047
    .hword 0xc048,0xc049,0xc04a,0xc04b,0xc04c,0xc04d,0xc04e,0xc04f,0xc050,0xc051,0xc052,0xc053,0xc054,0xc055,0xc056,0xc057
    .hword 0xc058,0xc059,0xc05a,0xc05b,0xc05c,0xc05d,0xc05e,0xc05f,0xc060,0xc061,0xc062,0xc063,0xc064,0xc065,0xc066,0xc067
    .hword 0xc068,0xc069,0xc06a,0xc06b,0xc06c,0xc06d,0xc06e,0xc06f,0xc070,0xc071,0xc072,0xc073,0xc074,0xc075,0xc076,0xc077
    .hword 0xc078,0xc079,0xc07a,0xc07b,0xc07c,0xc07d,0xc07e,0xc07f,0xc080,0xc081,0xc082,0xc083,0xc084,0xc085,0xc086,0xc087
    .hword 0xc088,0xc089,0xc08a,0xc08b,0xc08c,0xc08d,0xc08e,0xc08f,0xc090,0xc091,0xc092,0xc093,0xc094,0xc095,0xc096,0xc097
    .hword 0xc098,0xc099,0xc09a,0xc09b,0xc09c,0xc09d,0xc09e,0xc09f,0xc0a0,0xc0a1,0xc0a2,0xc0a3,0xc0a4,0xc0a5,0xc0a6,0xc0a7
    .hword 0xc0a8,0xc0a9,0xc0aa,0xc0ab,0xc0ac,0xc0ad,0xc0ae,0xc0af,0xc0b0,0xc0b1,0xc0b2,0xc0b3,0xc0b4,0xc0b5,0xc0b6,0xc0b7
    .hword 0xc0b8,0xc0b9,0xc0ba,0xc0bb,0xc0bc,0xc0bd,0xc0be,0xc0bf,0xc0c0,0xc0c1,0xc0c2,0xc0c3,0xc0c4,0xc0c5,0xc0c6,0xc0c7
    .hword 0xc0c8,0xc0c9,0xc0ca,0xc0cb,0xc0cc,0xc0cd,0xc0ce,0xc0cf,0xc0d0,0xc0d1,0xc0d2,0xc0d3,0xc0d4,0xc0d5,0xc0d6,0xc0d7
    .hword 0xc0d8,0xc0d9,0xc0da,0xc0db,0xc0dc,0xc0dd,0xc0de,0xc0df,0xc0e0,0xc0e1,0xc0e2,0xc0e3,0xc0e4,0xc0e5,0xc0e6,0xc0e7
    .hword 0xc0e8,0xc0e9,0xc0ea,0xc0eb,0xc0ec,0xc0ed,0xc0ee,0xc0ef,0xc0f0,0xc0f1,0xc0f2,0xc0f3,0xc0f4,0xc0f5,0xc0f6,0xc0f7
    .hword 0xc0f8,0xc0f9,0xc0fa,0xc0fb,0xc0fc,0xc0fd,0xc0fe,0xc0ff,0xc100,0xc101,0xc102,0xc103,0xc104,0xc105,0xc106,0xc107
    .hword 0xc108,0xc109,0xc10a,0xc10b,0xc10c,0xc10d,0xc10e,0xc10f,0xc110,0xc111,0xc112,0xc113,0xc114,0xc115,0xc116,0xc117
    .hword 0xc118,0xc119,0xc11a,0xc11b,0xc11c,0xc11d,0xc11e,0xc11f,0xc120,0xc121,0xc122,0xc123,0xc124,0xc125,0xc126,0xc127
    .hword 0xc128,0xc129,0xc12a,0xc12b,0xc12c,0xc12d,0xc12e,0xc12f,0xc130,0xc131,0xc132,0xc133,0xc134,0xc135,0xc136,0xc137
    .hword 0xc138,0xc139,0xc13a,0xc13b,0xc13c,0xc13d,0xc13e,0xc13f,0xc140,0xc141,0xc142,0xc143,0xc144,0xc145,0xc146,0xc147
    .hword 0xc148,0xc149,0xc14a,0xc14b,0xc14c,0xc14d,0xc14e,0xc14f,0xc150,0xc151,0xc152,0xc153,0xc154,0xc155,0xc156,0xc157
    .hword 0xc158,0xc159,0xc15a,0xc15b,0xc15c,0xc15d,0xc15e,0xc15f,0xc160,0xc161,0xc162,0xc163,0xc164,0xc165,0xc166,0xc167
    .hword 0xc168,0xc169,0xc16a,0xc16b,0xc16c,0xc16d,0xc16e,0xc16f,0xc170,0xc171,0xc172,0xc173,0xc174,0xc175,0xc176,0xc177
    .hword 0xc178,0xc179,0xc17a,0xc17b,0xc17c,0xc17d,0xc17e,0xc17f,0xc180,0xc181,0xc182,0xc183,0xc184,0xc185,0xc186,0xc187
    .hword 0xc188,0xc189,0xc18a,0xc18b,0xc18c,0xc18d,0xc18e,0xc18f,0xc190,0xc191,0xc192,0xc193,0xc194,0xc195,0xc196,0xc197
    .hword 0xc198,0xc199,0xc19a,0xc19b,0xc19c,0xc19d,0xc19e,0xc19f,0xc1a0,0xc1a1,0xc1a2,0xc1a3,0xc1a4,0xc1a5,0xc1a6,0xc1a7
    .hword 0xc1a8,0xc1a9,0xc1aa,0xc1ab,0xc1ac,0xc1ad,0xc1ae,0xc1af,0xc1b0,0xc1b1,0xc1b2,0xc1b3,0xc1b4,0xc1b5,0xc1b6,0xc1b7
    .hword 0xc1b8,0xc1b9,0xc1ba,0xc1bb,0xc1bc,0xc1bd,0xc1be,0xc1bf,0xc1c0,0xc1c1,0xc1c2,0xc1c3,0xc1c4,0xc1c5,0xc1c6,0xc1c7
    .hword 0xc1c8,0xc1c9,0xc1ca,0xc1cb,0xc1cc,0xc1cd,0xc1ce,0xc1cf,0xc1d0,0xc1d1,0xc1d2,0xc1d3,0xc1d4,0xc1d5,0xc1d6,0xc1d7
    .hword 0xc1d8,0xc1d9,0xc1da,0xc1db,0xc1dc,0xc1dd,0xc1de,0xc1df,0xc1e0,0xc1e1,0xc1e2,0xc1e3,0xc1e4,0xc1e5,0xc1e6,0xc1e7
    .hword 0xc1e8,0xc1e9,0xc1ea,0xc1eb,0xc1ec,0xc1ed,0xc1ee,0xc1ef,0xc1f0,0xc1f1,0xc1f2,0xc1f3,0xc1f4,0xc1f5,0xc1f6,0xc1f7
    .hword 0x81f8,0x81f9,0x81fa,0x81fb,0x81fc,0x81fd,0x81fe,0x81ff,0x8000,0x8001,0x8002,0x8003,0x8004,0x8005,0x8006,0x8007
    .hword 0x8008,0x8009,0x800a,0x800b,0x800c,0x800d,0x800e,0x800f,0x8010,0x8011,0x8012,0x8013,0x8014,0x8015,0x8016,0x8017
    .hword 0x8018,0x8019,0x801a,0x801b,0x801c,0x801d,0x801e,0x801f,0x8020,0x8021,0x8022,0x8023,0x8024,0x8025,0x8026,0x8027
    .hword 0x8028,0x8029,0x802a,0x802b,0x802c,0x802d,0x802e,0x802f,0x8030,0x8031,0x8032,0x8033,0x8034,0x8035,0x8036,0x8037
    .hword 0x8038,0x8039,0x803a,0x803b,0x803c,0x803d,0x803e,0x803f,0x8040,0x8041,0x8042,0x8043,0x8044,0x8045,0x8046,0x8047
    .hword 0x8048,0x8049,0x804a,0x804b,0x804c,0x804d,0x804e,0x804f,0x8050,0x8051,0x8052,0x8053,0x8054,0x8055,0x8056,0x8057
    .hword 0x8058,0x8059,0x805a,0x805b,0x805c,0x805d,0x805e,0x805f,0x8060,0x8061,0x8062,0x8063,0x8064,0x8065,0x8066,0x8067
    .hword 0x8068,0x8069,0x806a,0x806b,0x806c,0x806d,0x806e,0x806f,0x8070,0x8071,0x8072,0x8073,0x8074,0x8075,0x8076,0x8077
    .hword 0x8078,0x8079,0x807a,0x807b,0x807c,0x807d,0x807e,0x807f,0x8080,0x8081,0x8082,0x8083,0x8084,0x8085,0x8086,0x8087
    .hword 0x8088,0x8089,0x808a,0x808b,0x808c,0x808d,0x808e,0x808f,0x8090,0x8091,0x8092,0x8093,0x8094,0x8095,0x8096,0x8097
    .hword 0x8098,0x8099,0x809a,0x809b,0x809c,0x809d,0x809e,0x809f,0x80a0,0x80a1,0x80a2,0x80a3,0x80a4,0x80a5,0x80a6,0x80a7
    .hword 0x80a8,0x80a9,0x80aa,0x80ab,0x80ac,0x80ad,0x80ae,0x80af,0x80b0,0x80b1,0x80b2,0x80b3,0x80b4,0x80b5,0x80b6,0x80b7
    .hword 0x80b8,0x80b9,0x80ba,0x80bb,0x80bc,0x80bd,0x80be,0x80bf,0x80c0,0x80c1,0x80c2,0x80c3,0x80c4,0x80c5,0x80c6,0x80c7
    .hword 0x80c8,0x80c9,0x80ca,0x80cb,0x80cc,0x80cd,0x80ce,0x80cf,0x80d0,0x80d1,0x80d2,0x80d3,0x80d4,0x80d5,0x80d6,0x80d7
    .hword 0x80d8,0x80d9,0x80da,0x80db,0x80dc,0x80dd,0x80de,0x80df,0x80e0,0x80e1,0x80e2,0x80e3,0x80e4,0x80e5,0x80e6,0x80e7
    .hword 0x80e8,0x80e9,0x80ea,0x80eb,0x80ec,0x80ed,0x80ee,0x80ef,0x80f0,0x80f1,0x80f2,0x80f3,0x80f4,0x80f5,0x80f6,0x80f7
    .hword 0x80f8,0x80f9,0x80fa,0x80fb,0x80fc,0x80fd,0x80fe,0x80ff,0x8100,0x8101,0x8102,0x8103,0x8104,0x8105,0x8106,0x8107
    .hword 0x8108,0x8109,0x810a,0x810b,0x810c,0x810d,0x810e,0x810f,0x8110,0x8111,0x8112,0x8113,0x8114,0x8115,0x8116,0x8117
    .hword 0x8118,0x8119,0x811a,0x811b,0x811c,0x811d,0x811e,0x811f,0x8120,0x8121,0x8122,0x8123,0x8124,0x8125,0x8126,0x8127
    .hword 0x8128,0x8129,0x812a,0x812b,0x812c,0x812d,0x812e,0x812f,0x8130,0x8131,0x8132,0x8133,0x8134,0x8135,0x8136,0x8137
    .hword 0x8138,0x8139,0x813a,0x813b,0x813c,0x813d,0x813e,0x813f,0x8140,0x8141,0x8142,0x8143,0x8144,0x8145,0x8146,0x8147
    .hword 0x8148,0x8149,0x814a,0x814b,0x814c,0x814d,0x814e,0x814f,0x8150,0x8151,0x8152,0x8153,0x8154,0x8155,0x8156,0x8157
    .hword 0x8158,0x8159,0x815a,0x815b,0x815c,0x815d,0x815e,0x815f,0x8160,0x8161,0x8162,0x8163,0x8164,0x8165,0x8166,0x8167
    .hword 0x8168,0x8169,0x816a,0x816b,0x816c,0x816d,0x816e,0x816f,0x8170,0x8171,0x8172,0x8173,0x8174,0x8175,0x8176,0x8177
    .hword 0x8178,0x8179,0x817a,0x817b,0x817c,0x817d,0x817e,0x817f,0x8180,0x8181,0x8182,0x8183,0x8184,0x8185,0x8186,0x8187
    .hword 0x8188,0x8189,0x818a,0x818b,0x818c,0x818d,0x818e,0x818f,0x8190,0x8191,0x8192,0x8193,0x8194,0x8195,0x8196,0x8197
    .hword 0x8198,0x8199,0x819a,0x819b,0x819c,0x819d,0x819e,0x819f,0x81a0,0x81a1,0x81a2,0x81a3,0x81a4,0x81a5,0x81a6,0x81a7
    .hword 0x81a8,0x81a9,0x81aa,0x81ab,0x81ac,0x81ad,0x81ae,0x81af,0x81b0,0x81b1,0x81b2,0x81b3,0x81b4,0x81b5,0x81b6,0x81b7
    .hword 0x81b8,0x81b9,0x81ba,0x81bb,0x81bc,0x81bd,0x81be,0x81bf,0x81c0,0x81c1,0x81c2,0x81c3,0x81c4,0x81c5,0x81c6,0x81c7
    .hword 0x81c8,0x81c9,0x81ca,0x81cb,0x81cc,0x81cd,0x81ce,0x81cf,0x81d0,0x81d1,0x81d2,0x81d3,0x81d4,0x81d5,0x81d6,0x81d7
    .hword 0x81d8,0x81d9,0x81da,0x81db,0x81dc,0x81dd,0x81de,0x81df,0x81e0,0x81e1,0x81e2,0x81e3,0x81e4,0x81e5,0x81e6,0x81e7
    .hword 0x81e8,0x81e9,0x81ea,0x81eb,0x81ec,0x81ed,0x81ee,0x81ef,0x81f0,0x81f1,0x81f2,0x81f3,0x81f4,0x81f5,0x81f6,0x81f7
    .hword 0xc1f8,0xc1f9,0xc1fa,0xc1fb,0xc1fc,0xc1fd,0xc1fe,0xc1ff,0xc000,0xc001,0xc002,0xc003,0xc004,0xc005,0xc006,0xc007
    .hword 0xc008,0xc009,0xc00a,0xc00b,0xc00c,0xc00d,0xc00e,0xc00f,0xc010,0xc011,0xc012,0xc013,0xc014,0xc015,0xc016,0xc017
    .hword 0xc018,0xc019,0xc01a,0xc01b,0xc01c,0xc01d,0xc01e,0xc01f,0xc020,0xc021,0xc022,0xc023,0xc024,0xc025,0xc026,0xc027
    .hword 0xc028,0xc029,0xc02a,0xc02b,0xc02c,0xc02d,0xc02e,0xc02f,0xc030,0xc031,0xc032,0xc033,0xc034,0xc035,0xc036,0xc037
    .hword 0xc038,0xc039,0xc03a,0xc03b,0xc03c,0xc03d,0xc03e,0xc03f,0xc040,0xc041,0xc042,0xc043,0xc044,0xc045,0xc046,0xc047
    .hword 0xc048,0xc049,0xc04a,0xc04b,0xc04c,0xc04d,0xc04e,0xc04f,0xc050,0xc051,0xc052,0xc053,0xc054,0xc055,0xc056,0xc057
    .hword 0xc058,0xc059,0xc05a,0xc05b,0xc05c,0xc05d,0xc05e,0xc05f,0xc060,0xc061,0xc062,0xc063,0xc064,0xc065,0xc066,0xc067
    .hword 0xc068,0xc069,0xc06a,0xc06b,0xc06c,0xc06d,0xc06e,0xc06f,0xc070,0xc071,0xc072,0xc073,0xc074,0xc075,0xc076,0xc077
    .hword 0xc078,0xc079,0xc07a,0xc07b,0xc07c,0xc07d,0xc07e,0xc07f,0xc080,0xc081,0xc082,0xc083,0xc084,0xc085,0xc086,0xc087
    .hword 0xc088,0xc089,0xc08a,0xc08b,0xc08c,0xc08d,0xc08e,0xc08f,0xc090,0xc091,0xc092,0xc093,0xc094,0xc095,0xc096,0xc097
    .hword 0xc098,0xc099,0xc09a,0xc09b,0xc09c,0xc09d,0xc09e,0xc09f,0xc0a0,0xc0a1,0xc0a2,0xc0a3,0xc0a4,0xc0a5,0xc0a6,0xc0a7
    .hword 0xc0a8,0xc0a9,0xc0aa,0xc0ab,0xc0ac,0xc0ad,0xc0ae,0xc0af,0xc0b0,0xc0b1,0xc0b2,0xc0b3,0xc0b4,0xc0b5,0xc0b6,0xc0b7
    .hword 0xc0b8,0xc0b9,0xc0ba,0xc0bb,0xc0bc,0xc0bd,0xc0be,0xc0bf,0xc0c0,0xc0c1,0xc0c2,0xc0c3,0xc0c4,0xc0c5,0xc0c6,0xc0c7
    .hword 0xc0c8,0xc0c9,0xc0ca,0xc0cb,0xc0cc,0xc0cd,0xc0ce,0xc0cf,0xc0d0,0xc0d1,0xc0d2,0xc0d3,0xc0d4,0xc0d5,0xc0d6,0xc0d7
    .hword 0xc0d8,0xc0d9,0xc0da,0xc0db,0xc0dc,0xc0dd,0xc0de,0xc0df,0xc0e0,0xc0e1,0xc0e2,0xc0e3,0xc0e4,0xc0e5,0xc0e6,0xc0e7
    .hword 0xc0e8,0xc0e9,0xc0ea,0xc0eb,0xc0ec,0xc0ed,0xc0ee,0xc0ef,0xc0f0,0xc0f1,0xc0f2,0xc0f3,0xc0f4,0xc0f5,0xc0f6,0xc0f7
    .hword 0xc0f8,0xc0f9,0xc0fa,0xc0fb,0xc0fc,0xc0fd,0xc0fe,0xc0ff,0xc100,0xc101,0xc102,0xc103,0xc104,0xc105,0xc106,0xc107
    .hword 0xc108,0xc109,0xc10a,0xc10b,0xc10c,0xc10d,0xc10e,0xc10f,0xc110,0xc111,0xc112,0xc113,0xc114,0xc115,0xc116,0xc117
    .hword 0xc118,0xc119,0xc11a,0xc11b,0xc11c,0xc11d,0xc11e,0xc11f,0xc120,0xc121,0xc122,0xc123,0xc124,0xc125,0xc126,0xc127
    .hword 0xc128,0xc129,0xc12a,0xc12b,0xc12c,0xc12d,0xc12e,0xc12f,0xc130,0xc131,0xc132,0xc133,0xc134,0xc135,0xc136,0xc137
    .hword 0xc138,0xc139,0xc13a,0xc13b,0xc13c,0xc13d,0xc13e,0xc13f,0xc140,0xc141,0xc142,0xc143,0xc144,0xc145,0xc146,0xc147
    .hword 0xc148,0xc149,0xc14a,0xc14b,0xc14c,0xc14d,0xc14e,0xc14f,0xc150,0xc151,0xc152,0xc153,0xc154,0xc155,0xc156,0xc157
    .hword 0xc158,0xc159,0xc15a,0xc15b,0xc15c,0xc15d,0xc15e,0xc15f,0xc160,0xc161,0xc162,0xc163,0xc164,0xc165,0xc166,0xc167
    .hword 0xc168,0xc169,0xc16a,0xc16b,0xc16c,0xc16d,0xc16e,0xc16f,0xc170,0xc171,0xc172,0xc173,0xc174,0xc175,0xc176,0xc177
    .hword 0xc178,0xc179,0xc17a,0xc17b,0xc17c,0xc17d,0xc17e,0xc17f,0xc180,0xc181,0xc182,0xc183,0xc184,0xc185,0xc186,0xc187
    .hword 0xc188,0xc189,0xc18a,0xc18b,0xc18c,0xc18d,0xc18e,0xc18f,0xc190,0xc191,0xc192,0xc193,0xc194,0xc195,0xc196,0xc197
    .hword 0xc198,0xc199,0xc19a,0xc19b,0xc19c,0xc19d,0xc19e,0xc19f,0xc1a0,0xc1a1,0xc1a2,0xc1a3,0xc1a4,0xc1a5,0xc1a6,0xc1a7
    .hword 0xc1a8,0xc1a9,0xc1aa,0xc1ab,0xc1ac,0xc1ad,0xc1ae,0xc1af,0xc1b0,0xc1b1,0xc1b2,0xc1b3,0xc1b4,0xc1b5,0xc1b6,0xc1b7
    .hword 0xc1b8,0xc1b9,0xc1ba,0xc1bb,0xc1bc,0xc1bd,0xc1be,0xc1bf,0xc1c0,0xc1c1,0xc1c2,0xc1c3,0xc1c4,0xc1c5,0xc1c6,0xc1c7
    .hword 0xc1c8,0xc1c9,0xc1ca,0xc1cb,0xc1cc,0xc1cd,0xc1ce,0xc1cf,0xc1d0,0xc1d1,0xc1d2,0xc1d3,0xc1d4,0xc1d5,0xc1d6,0xc1d7
    .hword 0xc1d8,0xc1d9,0xc1da,0xc1db,0xc1dc,0xc1dd,0xc1de,0xc1df,0xc1e0,0xc1e1,0xc1e2,0xc1e3,0xc1e4,0xc1e5,0xc1e6,0xc1e7
    .hword 0xc1e8,0xc1e9,0xc1ea,0xc1eb,0xc1ec,0xc1ed,0xc1ee,0xc1ef,0xc1f0,0xc1f1,0xc1f2,0xc1f3,0xc1f4,0xc1f5,0xc1f6,0xc1f7
    .hword 0x41f8,0x41f9,0x41fa,0x41fb,0x41fc,0x41fd,0x41fe,0x41ff,0x4000,0x4001,0x4002,0x4003,0x4004,0x4005,0x4006,0x4007
    .hword 0x4008,0x4009,0x400a,0x400b,0x400c,0x400d,0x400e,0x400f,0x4010,0x4011,0x4012,0x4013,0x4014,0x4015,0x4016,0x4017
    .hword 0x4018,0x4019,0x401a,0x401b,0x401c,0x401d,0x401e,0x401f,0x4020,0x4021,0x4022,0x4023,0x4024,0x4025,0x4026,0x4027
    .hword 0x4028,0x4029,0x402a,0x402b,0x402c,0x402d,0x402e,0x402f,0x4030,0x4031,0x4032,0x4033,0x4034,0x4035,0x4036,0x4037
    .hword 0x4038,0x4039,0x403a,0x403b,0x403c,0x403d,0x403e,0x403f,0x4040,0x4041,0x4042,0x4043,0x4044,0x4045,0x4046,0x4047
    .hword 0x4048,0x4049,0x404a,0x404b,0x404c,0x404d,0x404e,0x404f,0x4050,0x4051,0x4052,0x4053,0x4054,0x4055,0x4056,0x4057
    .hword 0x4058,0x4059,0x405a,0x405b,0x405c,0x405d,0x405e,0x405f,0x4060,0x4061,0x4062,0x4063,0x4064,0x4065,0x4066,0x4067
    .hword 0x4068,0x4069,0x406a,0x406b,0x406c,0x406d,0x406e,0x406f,0x4070,0x4071,0x4072,0x4073,0x4074,0x4075,0x4076,0x4077
    .hword 0x4078,0x4079,0x407a,0x407b,0x407c,0x407d,0x407e,0x407f,0x4080,0x4081,0x4082,0x4083,0x4084,0x4085,0x4086,0x4087
    .hword 0x4088,0x4089,0x408a,0x408b,0x408c,0x408d,0x408e,0x408f,0x4090,0x4091,0x4092,0x4093,0x4094,0x4095,0x4096,0x4097
    .hword 0x4098,0x4099,0x409a,0x409b,0x409c,0x409d,0x409e,0x409f,0x40a0,0x40a1,0x40a2,0x40a3,0x40a4,0x40a5,0x40a6,0x40a7
    .hword 0x40a8,0x40a9,0x40aa,0x40ab,0x40ac,0x40ad,0x40ae,0x40af,0x40b0,0x40b1,0x40b2,0x40b3,0x40b4,0x40b5,0x40b6,0x40b7
    .hword 0x40b8,0x40b9,0x40ba,0x40bb,0x40bc,0x40bd,0x40be,0x40bf,0x40c0,0x40c1,0x40c2,0x40c3,0x40c4,0x40c5,0x40c6,0x40c7
    .hword 0x40c8,0x40c9,0x40ca,0x40cb,0x40cc,0x40cd,0x40ce,0x40cf,0x40d0,0x40d1,0x40d2,0x40d3,0x40d4,0x40d5,0x40d6,0x40d7
    .hword 0x40d8,0x40d9,0x40da,0x40db,0x40dc,0x40dd,0x40de,0x40df,0x40e0,0x40e1,0x40e2,0x40e3,0x40e4,0x40e5,0x40e6,0x40e7
    .hword 0x40e8,0x40e9,0x40ea,0x40eb,0x40ec,0x40ed,0x40ee,0x40ef,0x40f0,0x40f1,0x40f2,0x40f3,0x40f4,0x40f5,0x40f6,0x40f7
    .hword 0x40f8,0x40f9,0x40fa,0x40fb,0x40fc,0x40fd,0x40fe,0x40ff,0x4100,0x4101,0x4102,0x4103,0x4104,0x4105,0x4106,0x4107
    .hword 0x4108,0x4109,0x410a,0x410b,0x410c,0x410d,0x410e,0x410f,0x4110,0x4111,0x4112,0x4113,0x4114,0x4115,0x4116,0x4117
    .hword 0x4118,0x4119,0x411a,0x411b,0x411c,0x411d,0x411e,0x411f,0x4120,0x4121,0x4122,0x4123,0x4124,0x4125,0x4126,0x4127
    .hword 0x4128,0x4129,0x412a,0x412b,0x412c,0x412d,0x412e,0x412f,0x4130,0x4131,0x4132,0x4133,0x4134,0x4135,0x4136,0x4137
    .hword 0x4138,0x4139,0x413a,0x413b,0x413c,0x413d,0x413e,0x413f,0x4140,0x4141,0x4142,0x4143,0x4144,0x4145,0x4146,0x4147
    .hword 0x4148,0x4149,0x414a,0x414b,0x414c,0x414d,0x414e,0x414f,0x4150,0x4151,0x4152,0x4153,0x4154,0x4155,0x4156,0x4157
    .hword 0x4158,0x4159,0x415a,0x415b,0x415c,0x415d,0x415e,0x415f,0x4160,0x4161,0x4162,0x4163,0x4164,0x4165,0x4166,0x4167
    .hword 0x4168,0x4169,0x416a,0x416b,0x416c,0x416d,0x416e,0x416f,0x4170,0x4171,0x4172,0x4173,0x4174,0x4175,0x4176,0x4177
    .hword 0x4178,0x4179,0x417a,0x417b,0x417c,0x417d,0x417e,0x417f,0x4180,0x4181,0x4182,0x4183,0x4184,0x4185,0x4186,0x4187
    .hword 0x4188,0x4189,0x418a,0x418b,0x418c,0x418d,0x418e,0x418f,0x4190,0x4191,0x4192,0x4193,0x4194,0x4195,0x4196,0x4197
    .hword 0x4198,0x4199,0x419a,0x419b,0x419c,0x419d,0x419e,0x419f,0x41a0,0x41a1,0x41a2,0x41a3,0x41a4,0x41a5,0x41a6,0x41a7
    .hword 0x41a8,0x41a9,0x41aa,0x41ab,0x41ac,0x41ad,0x41ae,0x41af,0x41b0,0x41b1,0x41b2,0x41b3,0x41b4,0x41b5,0x41b6,0x41b7
    .hword 0x41b8,0x41b9,0x41ba,0x41bb,0x41bc,0x41bd,0x41be,0x41bf,0x41c0,0x41c1,0x41c2,0x41c3,0x41c4,0x41c5,0x41c6,0x41c7
    .hword 0x41c8,0x41c9,0x41ca,0x41cb,0x41cc,0x41cd,0x41ce,0x41cf,0x41d0,0x41d1,0x41d2,0x41d3,0x41d4,0x41d5,0x41d6,0x41d7
    .hword 0x41d8,0x41d9,0x41da,0x41db,0x41dc,0x41dd,0x41de,0x41df,0x41e0,0x41e1,0x41e2,0x41e3,0x41e4,0x41e5,0x41e6,0x41e7
    .hword 0x41e8,0x41e9,0x41ea,0x41eb,0x41ec,0x41ed,0x41ee,0x41ef,0x41f0,0x41f1,0x41f2,0x41f3,0x41f4,0x41f5,0x41f6,0x41f7
    .hword 0x81f8,0x81f9,0x81fa,0x81fb,0x81fc,0x81fd,0x81fe,0x81ff,0x8000,0x8001,0x8002,0x8003,0x8004,0x8005,0x8006,0x8007
    .hword 0x8008,0x8009,0x800a,0x800b,0x800c,0x800d,0x800e,0x800f,0x8010,0x8011,0x8012,0x8013,0x8014,0x8015,0x8016,0x8017
    .hword 0x8018,0x8019,0x801a,0x801b,0x801c,0x801d,0x801e,0x801f,0x8020,0x8021,0x8022,0x8023,0x8024,0x8025,0x8026,0x8027
    .hword 0x8028,0x8029,0x802a,0x802b,0x802c,0x802d,0x802e,0x802f,0x8030,0x8031,0x8032,0x8033,0x8034,0x8035,0x8036,0x8037
    .hword 0x8038,0x8039,0x803a,0x803b,0x803c,0x803d,0x803e,0x803f,0x8040,0x8041,0x8042,0x8043,0x8044,0x8045,0x8046,0x8047
    .hword 0x8048,0x8049,0x804a,0x804b,0x804c,0x804d,0x804e,0x804f,0x8050,0x8051,0x8052,0x8053,0x8054,0x8055,0x8056,0x8057
    .hword 0x8058,0x8059,0x805a,0x805b,0x805c,0x805d,0x805e,0x805f,0x8060,0x8061,0x8062,0x8063,0x8064,0x8065,0x8066,0x8067
    .hword 0x8068,0x8069,0x806a,0x806b,0x806c,0x806d,0x806e,0x806f,0x8070,0x8071,0x8072,0x8073,0x8074,0x8075,0x8076,0x8077
    .hword 0x8078,0x8079,0x807a,0x807b,0x807c,0x807d,0x807e,0x807f,0x8080,0x8081,0x8082,0x8083,0x8084,0x8085,0x8086,0x8087
    .hword 0x8088,0x8089,0x808a,0x808b,0x808c,0x808d,0x808e,0x808f,0x8090,0x8091,0x8092,0x8093,0x8094,0x8095,0x8096,0x8097
    .hword 0x8098,0x8099,0x809a,0x809b,0x809c,0x809d,0x809e,0x809f,0x80a0,0x80a1,0x80a2,0x80a3,0x80a4,0x80a5,0x80a6,0x80a7
    .hword 0x80a8,0x80a9,0x80aa,0x80ab,0x80ac,0x80ad,0x80ae,0x80af,0x80b0,0x80b1,0x80b2,0x80b3,0x80b4,0x80b5,0x80b6,0x80b7
    .hword 0x80b8,0x80b9,0x80ba,0x80bb,0x80bc,0x80bd,0x80be,0x80bf,0x80c0,0x80c1,0x80c2,0x80c3,0x80c4,0x80c5,0x80c6,0x80c7
    .hword 0x80c8,0x80c9,0x80ca,0x80cb,0x80cc,0x80cd,0x80ce,0x80cf,0x80d0,0x80d1,0x80d2,0x80d3,0x80d4,0x80d5,0x80d6,0x80d7
    .hword 0x80d8,0x80d9,0x80da,0x80db,0x80dc,0x80dd,0x80de,0x80df,0x80e0,0x80e1,0x80e2,0x80e3,0x80e4,0x80e5,0x80e6,0x80e7
    .hword 0x80e8,0x80e9,0x80ea,0x80eb,0x80ec,0x80ed,0x80ee,0x80ef,0x80f0,0x80f1,0x80f2,0x80f3,0x80f4,0x80f5,0x80f6,0x80f7
    .hword 0x80f8,0x80f9,0x80fa,0x80fb,0x80fc,0x80fd,0x80fe,0x80ff,0x8100,0x8101,0x8102,0x8103,0x8104,0x8105,0x8106,0x8107
    .hword 0x8108,0x8109,0x810a,0x810b,0x810c,0x810d,0x810e,0x810f,0x8110,0x8111,0x8112,0x8113,0x8114,0x8115,0x8116,0x8117
    .hword 0x8118,0x8119,0x811a,0x811b,0x811c,0x811d,0x811e,0x811f,0x8120,0x8121,0x8122,0x8123,0x8124,0x8125,0x8126,0x8127
    .hword 0x8128,0x8129,0x812a,0x812b,0x812c,0x812d,0x812e,0x812f,0x8130,0x8131,0x8132,0x8133,0x8134,0x8135,0x8136,0x8137
    .hword 0x8138,0x8139,0x813a,0x813b,0x813c,0x813d,0x813e,0x813f,0x8140,0x8141,0x8142,0x8143,0x8144,0x8145,0x8146,0x8147
    .hword 0x8148,0x8149,0x814a,0x814b,0x814c,0x814d,0x814e,0x814f,0x8150,0x8151,0x8152,0x8153,0x8154,0x8155,0x8156,0x8157
    .hword 0x8158,0x8159,0x815a,0x815b,0x815c,0x815d,0x815e,0x815f,0x8160,0x8161,0x8162,0x8163,0x8164,0x8165,0x8166,0x8167
    .hword 0x8168,0x8169,0x816a,0x816b,0x816c,0x816d,0x816e,0x816f,0x8170,0x8171,0x8172,0x8173,0x8174,0x8175,0x8176,0x8177
    .hword 0x8178,0x8179,0x817a,0x817b,0x817c,0x817d,0x817e,0x817f,0x8180,0x8181,0x8182,0x8183,0x8184,0x8185,0x8186,0x8187
    .hword 0x8188,0x8189,0x818a,0x818b,0x818c,0x818d,0x818e,0x818f,0x8190,0x8191,0x8192,0x8193,0x8194,0x8195,0x8196,0x8197
    .hword 0x8198,0x8199,0x819a,0x819b,0x819c,0x819d,0x819e,0x819f,0x81a0,0x81a1,0x81a2,0x81a3,0x81a4,0x81a5,0x81a6,0x81a7
    .hword 0x81a8,0x81a9,0x81aa,0x81ab,0x81ac,0x81ad,0x81ae,0x81af,0x81b0,0x81b1,0x81b2,0x81b3,0x81b4,0x81b5,0x81b6,0x81b7
    .hword 0x81b8,0x81b9,0x81ba,0x81bb,0x81bc,0x81bd,0x81be,0x81bf,0x81c0,0x81c1,0x81c2,0x81c3,0x81c4,0x81c5,0x81c6,0x81c7
    .hword 0x81c8,0x81c9,0x81ca,0x81cb,0x81cc,0x81cd,0x81ce,0x81cf,0x81d0,0x81d1,0x81d2,0x81d3,0x81d4,0x81d5,0x81d6,0x81d7
    .hword 0x81d8,0x81d9,0x81da,0x81db,0x81dc,0x81dd,0x81de,0x81df,0x81e0,0x81e1,0x81e2,0x81e3,0x81e4,0x81e5,0x81e6,0x81e7
    .hword 0x81e8,0x81e9,0x81ea,0x81eb,0x81ec,0x81ed,0x81ee,0x81ef,0x81f0,0x81f1,0x81f2,0x81f3,0x81f4,0x81f5,0x81f6,0x81f7
    .hword 0x41f8,0x41f9,0x41fa,0x41fb,0x41fc,0x41fd,0x41fe,0x41ff,0x4000,0x4001,0x4002,0x4003,0x4004,0x4005,0x4006,0x4007
    .hword 0x4008,0x4009,0x400a,0x400b,0x400c,0x400d,0x400e,0x400f,0x4010,0x4011,0x4012,0x4013,0x4014,0x4015,0x4016,0x4017
    .hword 0x4018,0x4019,0x401a,0x401b,0x401c,0x401d,0x401e,0x401f,0x4020,0x4021,0x4022,0x4023,0x4024,0x4025,0x4026,0x4027
    .hword 0x4028,0x4029,0x402a,0x402b,0x402c,0x402d,0x402e,0x402f,0x4030,0x4031,0x4032,0x4033,0x4034,0x4035,0x4036,0x4037
    .hword 0x4038,0x4039,0x403a,0x403b,0x403c,0x403d,0x403e,0x403f,0x4040,0x4041,0x4042,0x4043,0x4044,0x4045,0x4046,0x4047
    .hword 0x4048,0x4049,0x404a,0x404b,0x404c,0x404d,0x404e,0x404f,0x4050,0x4051,0x4052,0x4053,0x4054,0x4055,0x4056,0x4057
    .hword 0x4058,0x4059,0x405a,0x405b,0x405c,0x405d,0x405e,0x405f,0x4060,0x4061,0x4062,0x4063,0x4064,0x4065,0x4066,0x4067
    .hword 0x4068,0x4069,0x406a,0x406b,0x406c,0x406d,0x406e,0x406f,0x4070,0x4071,0x4072,0x4073,0x4074,0x4075,0x4076,0x4077
    .hword 0x4078,0x4079,0x407a,0x407b,0x407c,0x407d,0x407e,0x407f,0x4080,0x4081,0x4082,0x4083,0x4084,0x4085,0x4086,0x4087
    .hword 0x4088,0x4089,0x408a,0x408b,0x408c,0x408d,0x408e,0x408f,0x4090,0x4091,0x4092,0x4093,0x4094,0x4095,0x4096,0x4097
    .hword 0x4098,0x4099,0x409a,0x409b,0x409c,0x409d,0x409e,0x409f,0x40a0,0x40a1,0x40a2,0x40a3,0x40a4,0x40a5,0x40a6,0x40a7
    .hword 0x40a8,0x40a9,0x40aa,0x40ab,0x40ac,0x40ad,0x40ae,0x40af,0x40b0,0x40b1,0x40b2,0x40b3,0x40b4,0x40b5,0x40b6,0x40b7
    .hword 0x40b8,0x40b9,0x40ba,0x40bb,0x40bc,0x40bd,0x40be,0x40bf,0x40c0,0x40c1,0x40c2,0x40c3,0x40c4,0x40c5,0x40c6,0x40c7
    .hword 0x40c8,0x40c9,0x40ca,0x40cb,0x40cc,0x40cd,0x40ce,0x40cf,0x40d0,0x40d1,0x40d2,0x40d3,0x40d4,0x40d5,0x40d6,0x40d7
    .hword 0x40d8,0x40d9,0x40da,0x40db,0x40dc,0x40dd,0x40de,0x40df,0x40e0,0x40e1,0x40e2,0x40e3,0x40e4,0x40e5,0x40e6,0x40e7
    .hword 0x40e8,0x40e9,0x40ea,0x40eb,0x40ec,0x40ed,0x40ee,0x40ef,0x40f0,0x40f1,0x40f2,0x40f3,0x40f4,0x40f5,0x40f6,0x40f7
    .hword 0x40f8,0x40f9,0x40fa,0x40fb,0x40fc,0x40fd,0x40fe,0x40ff,0x4100,0x4101,0x4102,0x4103,0x4104,0x4105,0x4106,0x4107
    .hword 0x4108,0x4109,0x410a,0x410b,0x410c,0x410d,0x410e,0x410f,0x4110,0x4111,0x4112,0x4113,0x4114,0x4115,0x4116,0x4117
    .hword 0x4118,0x4119,0x411a,0x411b,0x411c,0x411d,0x411e,0x411f,0x4120,0x4121,0x4122,0x4123,0x4124,0x4125,0x4126,0x4127
    .hword 0x4128,0x4129,0x412a,0x412b,0x412c,0x412d,0x412e,0x412f,0x4130,0x4131,0x4132,0x4133,0x4134,0x4135,0x4136,0x4137
    .hword 0x4138,0x4139,0x413a,0x413b,0x413c,0x413d,0x413e,0x413f,0x4140,0x4141,0x4142,0x4143,0x4144,0x4145,0x4146,0x4147
    .hword 0x4148,0x4149,0x414a,0x414b,0x414c,0x414d,0x414e,0x414f,0x4150,0x4151,0x4152,0x4153,0x4154,0x4155,0x4156,0x4157
    .hword 0x4158,0x4159,0x415a,0x415b,0x415c,0x415d,0x415e,0x415f,0x4160,0x4161,0x4162,0x4163,0x4164,0x4165,0x4166,0x4167
    .hword 0x4168,0x4169,0x416a,0x416b,0x416c,0x416d,0x416e,0x416f,0x4170,0x4171,0x4172,0x4173,0x4174,0x4175,0x4176,0x4177
    .hword 0x4178,0x4179,0x417a,0x417b,0x417c,0x417d,0x417e,0x417f,0x4180,0x4181,0x4182,0x4183,0x4184,0x4185,0x4186,0x4187
    .hword 0x4188,0x4189,0x418a,0x418b,0x418c,0x418d,0x418e,0x418f,0x4190,0x4191,0x4192,0x4193,0x4194,0x4195,0x4196,0x4197
    .hword 0x4198,0x4199,0x419a,0x419b,0x419c,0x419d,0x419e,0x419f,0x41a0,0x41a1,0x41a2,0x41a3,0x41a4,0x41a5,0x41a6,0x41a7
    .hword 0x41a8,0x41a9,0x41aa,0x41ab,0x41ac,0x41ad,0x41ae,0x41af,0x41b0,0x41b1,0x41b2,0x41b3,0x41b4,0x41b5,0x41b6,0x41b7
    .hword 0x41b8,0x41b9,0x41ba,0x41bb,0x41bc,0x41bd,0x41be,0x41bf,0x41c0,0x41c1,0x41c2,0x41c3,0x41c4,0x41c5,0x41c6,0x41c7
    .hword 0x41c8,0x41c9,0x41ca,0x41cb,0x41cc,0x41cd,0x41ce,0x41cf,0x41d0,0x41d1,0x41d2,0x41d3,0x41d4,0x41d5,0x41d6,0x41d7
    .hword 0x41d8,0x41d9,0x41da,0x41db,0x41dc,0x41dd,0x41de,0x41df,0x41e0,0x41e1,0x41e2,0x41e3,0x41e4,0x41e5,0x41e6,0x41e7
    .hword 0x41e8,0x41e9,0x41ea,0x41eb,0x41ec,0x41ed,0x41ee,0x41ef,0x41f0,0x41f1,0x41f2,0x41f3,0x41f4,0x41f5,0x41f6,0x41f7
    .hword 0x81f8,0x81f9,0x81fa,0x81fb,0x81fc,0x81fd,0x81fe,0x81ff,0x8000,0x8001,0x8002,0x8003,0x8004,0x8005,0x8006,0x8007
    .hword 0x8008,0x8009,0x800a,0x800b,0x800c,0x800d,0x800e,0x800f,0x8010,0x8011,0x8012,0x8013,0x8014,0x8015,0x8016,0x8017
    .hword 0x8018,0x8019,0x801a,0x801b,0x801c,0x801d,0x801e,0x801f,0x8020,0x8021,0x8022,0x8023,0x8024,0x8025,0x8026,0x8027
    .hword 0x8028,0x8029,0x802a,0x802b,0x802c,0x802d,0x802e,0x802f,0x8030,0x8031,0x8032,0x8033,0x8034,0x8035,0x8036,0x8037
    .hword 0x8038,0x8039,0x803a,0x803b,0x803c,0x803d,0x803e,0x803f,0x8040,0x8041,0x8042,0x8043,0x8044,0x8045,0x8046,0x8047
    .hword 0x8048,0x8049,0x804a,0x804b,0x804c,0x804d,0x804e,0x804f,0x8050,0x8051,0x8052,0x8053,0x8054,0x8055,0x8056,0x8057
    .hword 0x8058,0x8059,0x805a,0x805b,0x805c,0x805d,0x805e,0x805f,0x8060,0x8061,0x8062,0x8063,0x8064,0x8065,0x8066,0x8067
    .hword 0x8068,0x8069,0x806a,0x806b,0x806c,0x806d,0x806e,0x806f,0x8070,0x8071,0x8072,0x8073,0x8074,0x8075,0x8076,0x8077
    .hword 0x8078,0x8079,0x807a,0x807b,0x807c,0x807d,0x807e,0x807f,0x8080,0x8081,0x8082,0x8083,0x8084,0x8085,0x8086,0x8087
    .hword 0x8088,0x8089,0x808a,0x808b,0x808c,0x808d,0x808e,0x808f,0x8090,0x8091,0x8092,0x8093,0x8094,0x8095,0x8096,0x8097
    .hword 0x8098,0x8099,0x809a,0x809b,0x809c,0x809d,0x809e,0x809f,0x80a0,0x80a1,0x80a2,0x80a3,0x80a4,0x80a5,0x80a6,0x80a7
    .hword 0x80a8,0x80a9,0x80aa,0x80ab,0x80ac,0x80ad,0x80ae,0x80af,0x80b0,0x80b1,0x80b2,0x80b3,0x80b4,0x80b5,0x80b6,0x80b7
    .hword 0x80b8,0x80b9,0x80ba,0x80bb,0x80bc,0x80bd,0x80be,0x80bf,0x80c0,0x80c1,0x80c2,0x80c3,0x80c4,0x80c5,0x80c6,0x80c7
    .hword 0x80c8,0x80c9,0x80ca,0x80cb,0x80cc,0x80cd,0x80ce,0x80cf,0x80d0,0x80d1,0x80d2,0x80d3,0x80d4,0x80d5,0x80d6,0x80d7
    .hword 0x80d8,0x80d9,0x80da,0x80db,0x80dc,0x80dd,0x80de,0x80df,0x80e0,0x80e1,0x80e2,0x80e3,0x80e4,0x80e5,0x80e6,0x80e7
    .hword 0x80e8,0x80e9,0x80ea,0x80eb,0x80ec,0x80ed,0x80ee,0x80ef,0x80f0,0x80f1,0x80f2,0x80f3,0x80f4,0x80f5,0x80f6,0x80f7
    .hword 0x80f8,0x80f9,0x80fa,0x80fb,0x80fc,0x80fd,0x80fe,0x80ff,0x8100,0x8101,0x8102,0x8103,0x8104,0x8105,0x8106,0x8107
    .hword 0x8108,0x8109,0x810a,0x810b,0x810c,0x810d,0x810e,0x810f,0x8110,0x8111,0x8112,0x8113,0x8114,0x8115,0x8116,0x8117
    .hword 0x8118,0x8119,0x811a,0x811b,0x811c,0x811d,0x811e,0x811f,0x8120,0x8121,0x8122,0x8123,0x8124,0x8125,0x8126,0x8127
    .hword 0x8128,0x8129,0x812a,0x812b,0x812c,0x812d,0x812e,0x812f,0x8130,0x8131,0x8132,0x8133,0x8134,0x8135,0x8136,0x8137
    .hword 0x8138,0x8139,0x813a,0x813b,0x813c,0x813d,0x813e,0x813f,0x8140,0x8141,0x8142,0x8143,0x8144,0x8145,0x8146,0x8147
    .hword 0x8148,0x8149,0x814a,0x814b,0x814c,0x814d,0x814e,0x814f,0x8150,0x8151,0x8152,0x8153,0x8154,0x8155,0x8156,0x8157
    .hword 0x8158,0x8159,0x815a,0x815b,0x815c,0x815d,0x815e,0x815f,0x8160,0x8161,0x8162,0x8163,0x8164,0x8165,0x8166,0x8167
    .hword 0x8168,0x8169,0x816a,0x816b,0x816c,0x816d,0x816e,0x816f,0x8170,0x8171,0x8172,0x8173,0x8174,0x8175,0x8176,0x8177
    .hword 0x8178,0x8179,0x817a,0x817b,0x817c,0x817d,0x817e,0x817f,0x8180,0x8181,0x8182,0x8183,0x8184,0x8185,0x8186,0x8187
    .hword 0x8188,0x8189,0x818a,0x818b,0x818c,0x818d,0x818e,0x818f,0x8190,0x8191,0x8192,0x8193,0x8194,0x8195,0x8196,0x8197
    .hword 0x8198,0x8199,0x819a,0x819b,0x819c,0x819d,0x819e,0x819f,0x81a0,0x81a1,0x81a2,0x81a3,0x81a4,0x81a5,0x81a6,0x81a7
    .hword 0x81a8,0x81a9,0x81aa,0x81ab,0x81ac,0x81ad,0x81ae,0x81af,0x81b0,0x81b1,0x81b2,0x81b3,0x81b4,0x81b5,0x81b6,0x81b7
    .hword 0x81b8,0x81b9,0x81ba,0x81bb,0x81bc,0x81bd,0x81be,0x81bf,0x81c0,0x81c1,0x81c2,0x81c3,0x81c4,0x81c5,0x81c6,0x81c7
    .hword 0x81c8,0x81c9,0x81ca,0x81cb,0x81cc,0x81cd,0x81ce,0x81cf,0x81d0,0x81d1,0x81d2,0x81d3,0x81d4,0x81d5,0x81d6,0x81d7
    .hword 0x81d8,0x81d9,0x81da,0x81db,0x81dc,0x81dd,0x81de,0x81df,0x81e0,0x81e1,0x81e2,0x81e3,0x81e4,0x81e5,0x81e6,0x81e7
    .hword 0x81e8,0x81e9,0x81ea,0x81eb,0x81ec,0x81ed,0x81ee,0x81ef,0x81f0,0x81f1,0x81f2,0x81f3,0x81f4,0x81f5,0x81f6,0x81f7
oamY:

    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
    .hword 0x0100,0x0100,0x0101,0x0102,0x0102,0x0103,0x0104,0x0105,0x0105,0x0106,0x0107,0x0107,0x0108,0x0109,0x010a,0x010a
    .hword 0x010b,0x010c,0x010c,0x010d,0x010e,0x010f,0x010f,0x0110,0x0111,0x0111,0x0112,0x0113,0x0114,0x0114,0x0115,0x0116
    .hword 0x0116,0x0117,0x0118,0x0119,0x0119,0x011a,0x011b,0x011b,0x011c,0x011d,0x011e,0x011e,0x011f,0x0120,0x0120,0x0121
    .hword 0x0122,0x0123,0x0123,0x0124,0x0125,0x0125,0x0126,0x0127,0x0128,0x0128,0x0129,0x012a,0x012a,0x012b,0x012c,0x012d
    .hword 0x012d,0x012e,0x012f,0x012f,0x0130,0x0131,0x0132,0x0132,0x0133,0x0134,0x0134,0x0135,0x0136,0x0137,0x0137,0x0138
    .hword 0x0139,0x0139,0x013a,0x013b,0x013c,0x013c,0x013d,0x013e,0x013e,0x013f,0x0140,0x0141,0x0141,0x0142,0x0143,0x0143
    .hword 0x0144,0x0145,0x0146,0x0146,0x0147,0x0148,0x0148,0x0149,0x014a,0x014b,0x014b,0x014c,0x014d,0x014d,0x014e,0x014f
    .hword 0x0150,0x0150,0x0151,0x0152,0x0152,0x0153,0x0154,0x0155,0x0155,0x0156,0x0157,0x0157,0x0158,0x0159,0x015a,0x015a
    .hword 0x015b,0x015c,0x015c,0x015d,0x015e,0x015f,0x015f,0x0160,0x0161,0x0161,0x0162,0x0163,0x0164,0x0164,0x0165,0x0166
    .hword 0x0166,0x0167,0x0168,0x0169,0x0169,0x016a,0x016b,0x016b,0x016c,0x016d,0x016e,0x016e,0x016f,0x0170,0x0170,0x0171
    .hword 0x0172,0x0173,0x0173,0x0174,0x0175,0x0175,0x0176,0x0177,0x0178,0x0178,0x0179,0x017a,0x017a,0x017b,0x017c,0x017d
    .hword 0x017d,0x017e,0x017f,0x017f,0x0180,0x0181,0x0182,0x0182,0x0183,0x0184,0x0184,0x0185,0x0186,0x0187,0x0187,0x0188
    .hword 0x0189,0x0189,0x018a,0x018b,0x018c,0x018c,0x018d,0x018e,0x018e,0x018f,0x0190,0x0191,0x0191,0x0192,0x0193,0x0193
    .hword 0x0194,0x0195,0x0196,0x0196,0x0197,0x0198,0x0198,0x0199,0x019a,0x019b,0x019b,0x019c,0x01e7,0x01e7,0x01e8,0x01e9
    .hword 0x01ea,0x01ea,0x01eb,0x01ec,0x01ec,0x01ed,0x01ee,0x01ef,0x01ef,0x01f0,0x01f1,0x01f1,0x01f2,0x01f3,0x01f4,0x01f4
    .hword 0x01f5,0x01f6,0x01f6,0x01f7,0x01f8,0x01f9,0x01f9,0x01fa,0x01fb,0x01fb,0x01fc,0x01fd,0x01fe,0x01fe,0x01ff,0x0100
oamControl:

    .hword 0x8c00,0x8c01,0x8c02,0x8c03,0x8c04,0x8c05,0x8c06,0x8c07,0x8c08,0x8c09,0x8c0a,0x8c0b,0x8c0c,0x8c0d,0x8c0e,0x8c0f
    .hword 0x8c20,0x8c21,0x8c22,0x8c23,0x8c24,0x8c25,0x8c26,0x8c27,0x8c28,0x8c29,0x8c2a,0x8c2b,0x8c2c,0x8c2d,0x8c2e,0x8c2f
    .hword 0x8c40,0x8c41,0x8c42,0x8c43,0x8c44,0x8c45,0x8c46,0x8c47,0x8c48,0x8c49,0x8c4a,0x8c4b,0x8c4c,0x8c4d,0x8c4e,0x8c4f
    .hword 0x8c60,0x8c61,0x8c62,0x8c63,0x8c64,0x8c65,0x8c66,0x8c67,0x8c68,0x8c69,0x8c6a,0x8c6b,0x8c6c,0x8c6d,0x8c6e,0x8c6f
    .hword 0x8c80,0x8c81,0x8c82,0x8c83,0x8c84,0x8c85,0x8c86,0x8c87,0x8c88,0x8c89,0x8c8a,0x8c8b,0x8c8c,0x8c8d,0x8c8e,0x8c8f
    .hword 0x8ca0,0x8ca1,0x8ca2,0x8ca3,0x8ca4,0x8ca5,0x8ca6,0x8ca7,0x8ca8,0x8ca9,0x8caa,0x8cab,0x8cac,0x8cad,0x8cae,0x8caf
    .hword 0x8cc0,0x8cc1,0x8cc2,0x8cc3,0x8cc4,0x8cc5,0x8cc6,0x8cc7,0x8cc8,0x8cc9,0x8cca,0x8ccb,0x8ccc,0x8ccd,0x8cce,0x8ccf
    .hword 0x8ce0,0x8ce1,0x8ce2,0x8ce3,0x8ce4,0x8ce5,0x8ce6,0x8ce7,0x8ce8,0x8ce9,0x8cea,0x8ceb,0x8cec,0x8ced,0x8cee,0x8cef
    .hword 0x8d00,0x8d01,0x8d02,0x8d03,0x8d04,0x8d05,0x8d06,0x8d07,0x8d08,0x8d09,0x8d0a,0x8d0b,0x8d0c,0x8d0d,0x8d0e,0x8d0f
    .hword 0x8d20,0x8d21,0x8d22,0x8d23,0x8d24,0x8d25,0x8d26,0x8d27,0x8d28,0x8d29,0x8d2a,0x8d2b,0x8d2c,0x8d2d,0x8d2e,0x8d2f
    .hword 0x8d40,0x8d41,0x8d42,0x8d43,0x8d44,0x8d45,0x8d46,0x8d47,0x8d48,0x8d49,0x8d4a,0x8d4b,0x8d4c,0x8d4d,0x8d4e,0x8d4f
    .hword 0x8d60,0x8d61,0x8d62,0x8d63,0x8d64,0x8d65,0x8d66,0x8d67,0x8d68,0x8d69,0x8d6a,0x8d6b,0x8d6c,0x8d6d,0x8d6e,0x8d6f
    .hword 0x8d80,0x8d81,0x8d82,0x8d83,0x8d84,0x8d85,0x8d86,0x8d87,0x8d88,0x8d89,0x8d8a,0x8d8b,0x8d8c,0x8d8d,0x8d8e,0x8d8f
    .hword 0x8da0,0x8da1,0x8da2,0x8da3,0x8da4,0x8da5,0x8da6,0x8da7,0x8da8,0x8da9,0x8daa,0x8dab,0x8dac,0x8dad,0x8dae,0x8daf
    .hword 0x8dc0,0x8dc1,0x8dc2,0x8dc3,0x8dc4,0x8dc5,0x8dc6,0x8dc7,0x8dc8,0x8dc9,0x8dca,0x8dcb,0x8dcc,0x8dcd,0x8dce,0x8dcf
    .hword 0x8de0,0x8de1,0x8de2,0x8de3,0x8de4,0x8de5,0x8de6,0x8de7,0x8de8,0x8de9,0x8dea,0x8deb,0x8dec,0x8ded,0x8dee,0x8def
    .hword 0x8e00,0x8e01,0x8e02,0x8e03,0x8e04,0x8e05,0x8e06,0x8e07,0x8e08,0x8e09,0x8e0a,0x8e0b,0x8e0c,0x8e0d,0x8e0e,0x8e0f
    .hword 0x8e20,0x8e21,0x8e22,0x8e23,0x8e24,0x8e25,0x8e26,0x8e27,0x8e28,0x8e29,0x8e2a,0x8e2b,0x8e2c,0x8e2d,0x8e2e,0x8e2f
    .hword 0x8e40,0x8e41,0x8e42,0x8e43,0x8e44,0x8e45,0x8e46,0x8e47,0x8e48,0x8e49,0x8e4a,0x8e4b,0x8e4c,0x8e4d,0x8e4e,0x8e4f
    .hword 0x8e60,0x8e61,0x8e62,0x8e63,0x8e64,0x8e65,0x8e66,0x8e67,0x8e68,0x8e69,0x8e6a,0x8e6b,0x8e6c,0x8e6d,0x8e6e,0x8e6f
    .hword 0x8e80,0x8e81,0x8e82,0x8e83,0x8e84,0x8e85,0x8e86,0x8e87,0x8e88,0x8e89,0x8e8a,0x8e8b,0x8e8c,0x8e8d,0x8e8e,0x8e8f
    .hword 0x8ea0,0x8ea1,0x8ea2,0x8ea3,0x8ea4,0x8ea5,0x8ea6,0x8ea7,0x8ea8,0x8ea9,0x8eaa,0x8eab,0x8eac,0x8ead,0x8eae,0x8eaf
    .hword 0x8ec0,0x8ec1,0x8ec2,0x8ec3,0x8ec4,0x8ec5,0x8ec6,0x8ec7,0x8ec8,0x8ec9,0x8eca,0x8ecb,0x8ecc,0x8ecd,0x8ece,0x8ecf
    .hword 0x8ee0,0x8ee1,0x8ee2,0x8ee3,0x8ee4,0x8ee5,0x8ee6,0x8ee7,0x8ee8,0x8ee9,0x8eea,0x8eeb,0x8eec,0x8eed,0x8eee,0x8eef
    .hword 0x8f00,0x8f01,0x8f02,0x8f03,0x8f04,0x8f05,0x8f06,0x8f07,0x8f08,0x8f09,0x8f0a,0x8f0b,0x8f0c,0x8f0d,0x8f0e,0x8f0f
    .hword 0x8f20,0x8f21,0x8f22,0x8f23,0x8f24,0x8f25,0x8f26,0x8f27,0x8f28,0x8f29,0x8f2a,0x8f2b,0x8f2c,0x8f2d,0x8f2e,0x8f2f
    .hword 0x8f40,0x8f41,0x8f42,0x8f43,0x8f44,0x8f45,0x8f46,0x8f47,0x8f48,0x8f49,0x8f4a,0x8f4b,0x8f4c,0x8f4d,0x8f4e,0x8f4f
    .hword 0x8f60,0x8f61,0x8f62,0x8f63,0x8f64,0x8f65,0x8f66,0x8f67,0x8f68,0x8f69,0x8f6a,0x8f6b,0x8f6c,0x8f6d,0x8f6e,0x8f6f
    .hword 0x8f80,0x8f81,0x8f82,0x8f83,0x8f84,0x8f85,0x8f86,0x8f87,0x8f88,0x8f89,0x8f8a,0x8f8b,0x8f8c,0x8f8d,0x8f8e,0x8f8f
    .hword 0x8fa0,0x8fa1,0x8fa2,0x8fa3,0x8fa4,0x8fa5,0x8fa6,0x8fa7,0x8fa8,0x8fa9,0x8faa,0x8fab,0x8fac,0x8fad,0x8fae,0x8faf
    .hword 0x8fc0,0x8fc1,0x8fc2,0x8fc3,0x8fc4,0x8fc5,0x8fc6,0x8fc7,0x8fc8,0x8fc9,0x8fca,0x8fcb,0x8fcc,0x8fcd,0x8fce,0x8fcf
    .hword 0x8fe0,0x8fe1,0x8fe2,0x8fe3,0x8fe4,0x8fe5,0x8fe6,0x8fe7,0x8fe8,0x8fe9,0x8fea,0x8feb,0x8fec,0x8fed,0x8fee,0x8fef
    .hword 0x9c00,0x9c01,0x9c02,0x9c03,0x9c04,0x9c05,0x9c06,0x9c07,0x9c08,0x9c09,0x9c0a,0x9c0b,0x9c0c,0x9c0d,0x9c0e,0x9c0f
    .hword 0x9c20,0x9c21,0x9c22,0x9c23,0x9c24,0x9c25,0x9c26,0x9c27,0x9c28,0x9c29,0x9c2a,0x9c2b,0x9c2c,0x9c2d,0x9c2e,0x9c2f
    .hword 0x9c40,0x9c41,0x9c42,0x9c43,0x9c44,0x9c45,0x9c46,0x9c47,0x9c48,0x9c49,0x9c4a,0x9c4b,0x9c4c,0x9c4d,0x9c4e,0x9c4f
    .hword 0x9c60,0x9c61,0x9c62,0x9c63,0x9c64,0x9c65,0x9c66,0x9c67,0x9c68,0x9c69,0x9c6a,0x9c6b,0x9c6c,0x9c6d,0x9c6e,0x9c6f
    .hword 0x9c80,0x9c81,0x9c82,0x9c83,0x9c84,0x9c85,0x9c86,0x9c87,0x9c88,0x9c89,0x9c8a,0x9c8b,0x9c8c,0x9c8d,0x9c8e,0x9c8f
    .hword 0x9ca0,0x9ca1,0x9ca2,0x9ca3,0x9ca4,0x9ca5,0x9ca6,0x9ca7,0x9ca8,0x9ca9,0x9caa,0x9cab,0x9cac,0x9cad,0x9cae,0x9caf
    .hword 0x9cc0,0x9cc1,0x9cc2,0x9cc3,0x9cc4,0x9cc5,0x9cc6,0x9cc7,0x9cc8,0x9cc9,0x9cca,0x9ccb,0x9ccc,0x9ccd,0x9cce,0x9ccf
    .hword 0x9ce0,0x9ce1,0x9ce2,0x9ce3,0x9ce4,0x9ce5,0x9ce6,0x9ce7,0x9ce8,0x9ce9,0x9cea,0x9ceb,0x9cec,0x9ced,0x9cee,0x9cef
    .hword 0x9d00,0x9d01,0x9d02,0x9d03,0x9d04,0x9d05,0x9d06,0x9d07,0x9d08,0x9d09,0x9d0a,0x9d0b,0x9d0c,0x9d0d,0x9d0e,0x9d0f
    .hword 0x9d20,0x9d21,0x9d22,0x9d23,0x9d24,0x9d25,0x9d26,0x9d27,0x9d28,0x9d29,0x9d2a,0x9d2b,0x9d2c,0x9d2d,0x9d2e,0x9d2f
    .hword 0x9d40,0x9d41,0x9d42,0x9d43,0x9d44,0x9d45,0x9d46,0x9d47,0x9d48,0x9d49,0x9d4a,0x9d4b,0x9d4c,0x9d4d,0x9d4e,0x9d4f
    .hword 0x9d60,0x9d61,0x9d62,0x9d63,0x9d64,0x9d65,0x9d66,0x9d67,0x9d68,0x9d69,0x9d6a,0x9d6b,0x9d6c,0x9d6d,0x9d6e,0x9d6f
    .hword 0x9d80,0x9d81,0x9d82,0x9d83,0x9d84,0x9d85,0x9d86,0x9d87,0x9d88,0x9d89,0x9d8a,0x9d8b,0x9d8c,0x9d8d,0x9d8e,0x9d8f
    .hword 0x9da0,0x9da1,0x9da2,0x9da3,0x9da4,0x9da5,0x9da6,0x9da7,0x9da8,0x9da9,0x9daa,0x9dab,0x9dac,0x9dad,0x9dae,0x9daf
    .hword 0x9dc0,0x9dc1,0x9dc2,0x9dc3,0x9dc4,0x9dc5,0x9dc6,0x9dc7,0x9dc8,0x9dc9,0x9dca,0x9dcb,0x9dcc,0x9dcd,0x9dce,0x9dcf
    .hword 0x9de0,0x9de1,0x9de2,0x9de3,0x9de4,0x9de5,0x9de6,0x9de7,0x9de8,0x9de9,0x9dea,0x9deb,0x9dec,0x9ded,0x9dee,0x9def
    .hword 0x9e00,0x9e01,0x9e02,0x9e03,0x9e04,0x9e05,0x9e06,0x9e07,0x9e08,0x9e09,0x9e0a,0x9e0b,0x9e0c,0x9e0d,0x9e0e,0x9e0f
    .hword 0x9e20,0x9e21,0x9e22,0x9e23,0x9e24,0x9e25,0x9e26,0x9e27,0x9e28,0x9e29,0x9e2a,0x9e2b,0x9e2c,0x9e2d,0x9e2e,0x9e2f
    .hword 0x9e40,0x9e41,0x9e42,0x9e43,0x9e44,0x9e45,0x9e46,0x9e47,0x9e48,0x9e49,0x9e4a,0x9e4b,0x9e4c,0x9e4d,0x9e4e,0x9e4f
    .hword 0x9e60,0x9e61,0x9e62,0x9e63,0x9e64,0x9e65,0x9e66,0x9e67,0x9e68,0x9e69,0x9e6a,0x9e6b,0x9e6c,0x9e6d,0x9e6e,0x9e6f
    .hword 0x9e80,0x9e81,0x9e82,0x9e83,0x9e84,0x9e85,0x9e86,0x9e87,0x9e88,0x9e89,0x9e8a,0x9e8b,0x9e8c,0x9e8d,0x9e8e,0x9e8f
    .hword 0x9ea0,0x9ea1,0x9ea2,0x9ea3,0x9ea4,0x9ea5,0x9ea6,0x9ea7,0x9ea8,0x9ea9,0x9eaa,0x9eab,0x9eac,0x9ead,0x9eae,0x9eaf
    .hword 0x9ec0,0x9ec1,0x9ec2,0x9ec3,0x9ec4,0x9ec5,0x9ec6,0x9ec7,0x9ec8,0x9ec9,0x9eca,0x9ecb,0x9ecc,0x9ecd,0x9ece,0x9ecf
    .hword 0x9ee0,0x9ee1,0x9ee2,0x9ee3,0x9ee4,0x9ee5,0x9ee6,0x9ee7,0x9ee8,0x9ee9,0x9eea,0x9eeb,0x9eec,0x9eed,0x9eee,0x9eef
    .hword 0x9f00,0x9f01,0x9f02,0x9f03,0x9f04,0x9f05,0x9f06,0x9f07,0x9f08,0x9f09,0x9f0a,0x9f0b,0x9f0c,0x9f0d,0x9f0e,0x9f0f
    .hword 0x9f20,0x9f21,0x9f22,0x9f23,0x9f24,0x9f25,0x9f26,0x9f27,0x9f28,0x9f29,0x9f2a,0x9f2b,0x9f2c,0x9f2d,0x9f2e,0x9f2f
    .hword 0x9f40,0x9f41,0x9f42,0x9f43,0x9f44,0x9f45,0x9f46,0x9f47,0x9f48,0x9f49,0x9f4a,0x9f4b,0x9f4c,0x9f4d,0x9f4e,0x9f4f
    .hword 0x9f60,0x9f61,0x9f62,0x9f63,0x9f64,0x9f65,0x9f66,0x9f67,0x9f68,0x9f69,0x9f6a,0x9f6b,0x9f6c,0x9f6d,0x9f6e,0x9f6f
    .hword 0x9f80,0x9f81,0x9f82,0x9f83,0x9f84,0x9f85,0x9f86,0x9f87,0x9f88,0x9f89,0x9f8a,0x9f8b,0x9f8c,0x9f8d,0x9f8e,0x9f8f
    .hword 0x9fa0,0x9fa1,0x9fa2,0x9fa3,0x9fa4,0x9fa5,0x9fa6,0x9fa7,0x9fa8,0x9fa9,0x9faa,0x9fab,0x9fac,0x9fad,0x9fae,0x9faf
    .hword 0x9fc0,0x9fc1,0x9fc2,0x9fc3,0x9fc4,0x9fc5,0x9fc6,0x9fc7,0x9fc8,0x9fc9,0x9fca,0x9fcb,0x9fcc,0x9fcd,0x9fce,0x9fcf
    .hword 0x9fe0,0x9fe1,0x9fe2,0x9fe3,0x9fe4,0x9fe5,0x9fe6,0x9fe7,0x9fe8,0x9fe9,0x9fea,0x9feb,0x9fec,0x9fed,0x9fee,0x9fef
    .hword 0xac00,0xac01,0xac02,0xac03,0xac04,0xac05,0xac06,0xac07,0xac08,0xac09,0xac0a,0xac0b,0xac0c,0xac0d,0xac0e,0xac0f
    .hword 0xac20,0xac21,0xac22,0xac23,0xac24,0xac25,0xac26,0xac27,0xac28,0xac29,0xac2a,0xac2b,0xac2c,0xac2d,0xac2e,0xac2f
    .hword 0xac40,0xac41,0xac42,0xac43,0xac44,0xac45,0xac46,0xac47,0xac48,0xac49,0xac4a,0xac4b,0xac4c,0xac4d,0xac4e,0xac4f
    .hword 0xac60,0xac61,0xac62,0xac63,0xac64,0xac65,0xac66,0xac67,0xac68,0xac69,0xac6a,0xac6b,0xac6c,0xac6d,0xac6e,0xac6f
    .hword 0xac80,0xac81,0xac82,0xac83,0xac84,0xac85,0xac86,0xac87,0xac88,0xac89,0xac8a,0xac8b,0xac8c,0xac8d,0xac8e,0xac8f
    .hword 0xaca0,0xaca1,0xaca2,0xaca3,0xaca4,0xaca5,0xaca6,0xaca7,0xaca8,0xaca9,0xacaa,0xacab,0xacac,0xacad,0xacae,0xacaf
    .hword 0xacc0,0xacc1,0xacc2,0xacc3,0xacc4,0xacc5,0xacc6,0xacc7,0xacc8,0xacc9,0xacca,0xaccb,0xaccc,0xaccd,0xacce,0xaccf
    .hword 0xace0,0xace1,0xace2,0xace3,0xace4,0xace5,0xace6,0xace7,0xace8,0xace9,0xacea,0xaceb,0xacec,0xaced,0xacee,0xacef
    .hword 0xad00,0xad01,0xad02,0xad03,0xad04,0xad05,0xad06,0xad07,0xad08,0xad09,0xad0a,0xad0b,0xad0c,0xad0d,0xad0e,0xad0f
    .hword 0xad20,0xad21,0xad22,0xad23,0xad24,0xad25,0xad26,0xad27,0xad28,0xad29,0xad2a,0xad2b,0xad2c,0xad2d,0xad2e,0xad2f
    .hword 0xad40,0xad41,0xad42,0xad43,0xad44,0xad45,0xad46,0xad47,0xad48,0xad49,0xad4a,0xad4b,0xad4c,0xad4d,0xad4e,0xad4f
    .hword 0xad60,0xad61,0xad62,0xad63,0xad64,0xad65,0xad66,0xad67,0xad68,0xad69,0xad6a,0xad6b,0xad6c,0xad6d,0xad6e,0xad6f
    .hword 0xad80,0xad81,0xad82,0xad83,0xad84,0xad85,0xad86,0xad87,0xad88,0xad89,0xad8a,0xad8b,0xad8c,0xad8d,0xad8e,0xad8f
    .hword 0xada0,0xada1,0xada2,0xada3,0xada4,0xada5,0xada6,0xada7,0xada8,0xada9,0xadaa,0xadab,0xadac,0xadad,0xadae,0xadaf
    .hword 0xadc0,0xadc1,0xadc2,0xadc3,0xadc4,0xadc5,0xadc6,0xadc7,0xadc8,0xadc9,0xadca,0xadcb,0xadcc,0xadcd,0xadce,0xadcf
    .hword 0xade0,0xade1,0xade2,0xade3,0xade4,0xade5,0xade6,0xade7,0xade8,0xade9,0xadea,0xadeb,0xadec,0xaded,0xadee,0xadef
    .hword 0xae00,0xae01,0xae02,0xae03,0xae04,0xae05,0xae06,0xae07,0xae08,0xae09,0xae0a,0xae0b,0xae0c,0xae0d,0xae0e,0xae0f
    .hword 0xae20,0xae21,0xae22,0xae23,0xae24,0xae25,0xae26,0xae27,0xae28,0xae29,0xae2a,0xae2b,0xae2c,0xae2d,0xae2e,0xae2f
    .hword 0xae40,0xae41,0xae42,0xae43,0xae44,0xae45,0xae46,0xae47,0xae48,0xae49,0xae4a,0xae4b,0xae4c,0xae4d,0xae4e,0xae4f
    .hword 0xae60,0xae61,0xae62,0xae63,0xae64,0xae65,0xae66,0xae67,0xae68,0xae69,0xae6a,0xae6b,0xae6c,0xae6d,0xae6e,0xae6f
    .hword 0xae80,0xae81,0xae82,0xae83,0xae84,0xae85,0xae86,0xae87,0xae88,0xae89,0xae8a,0xae8b,0xae8c,0xae8d,0xae8e,0xae8f
    .hword 0xaea0,0xaea1,0xaea2,0xaea3,0xaea4,0xaea5,0xaea6,0xaea7,0xaea8,0xaea9,0xaeaa,0xaeab,0xaeac,0xaead,0xaeae,0xaeaf
    .hword 0xaec0,0xaec1,0xaec2,0xaec3,0xaec4,0xaec5,0xaec6,0xaec7,0xaec8,0xaec9,0xaeca,0xaecb,0xaecc,0xaecd,0xaece,0xaecf
    .hword 0xaee0,0xaee1,0xaee2,0xaee3,0xaee4,0xaee5,0xaee6,0xaee7,0xaee8,0xaee9,0xaeea,0xaeeb,0xaeec,0xaeed,0xaeee,0xaeef
    .hword 0xaf00,0xaf01,0xaf02,0xaf03,0xaf04,0xaf05,0xaf06,0xaf07,0xaf08,0xaf09,0xaf0a,0xaf0b,0xaf0c,0xaf0d,0xaf0e,0xaf0f
    .hword 0xaf20,0xaf21,0xaf22,0xaf23,0xaf24,0xaf25,0xaf26,0xaf27,0xaf28,0xaf29,0xaf2a,0xaf2b,0xaf2c,0xaf2d,0xaf2e,0xaf2f
    .hword 0xaf40,0xaf41,0xaf42,0xaf43,0xaf44,0xaf45,0xaf46,0xaf47,0xaf48,0xaf49,0xaf4a,0xaf4b,0xaf4c,0xaf4d,0xaf4e,0xaf4f
    .hword 0xaf60,0xaf61,0xaf62,0xaf63,0xaf64,0xaf65,0xaf66,0xaf67,0xaf68,0xaf69,0xaf6a,0xaf6b,0xaf6c,0xaf6d,0xaf6e,0xaf6f
    .hword 0xaf80,0xaf81,0xaf82,0xaf83,0xaf84,0xaf85,0xaf86,0xaf87,0xaf88,0xaf89,0xaf8a,0xaf8b,0xaf8c,0xaf8d,0xaf8e,0xaf8f
    .hword 0xafa0,0xafa1,0xafa2,0xafa3,0xafa4,0xafa5,0xafa6,0xafa7,0xafa8,0xafa9,0xafaa,0xafab,0xafac,0xafad,0xafae,0xafaf
    .hword 0xafc0,0xafc1,0xafc2,0xafc3,0xafc4,0xafc5,0xafc6,0xafc7,0xafc8,0xafc9,0xafca,0xafcb,0xafcc,0xafcd,0xafce,0xafcf
    .hword 0xafe0,0xafe1,0xafe2,0xafe3,0xafe4,0xafe5,0xafe6,0xafe7,0xafe8,0xafe9,0xafea,0xafeb,0xafec,0xafed,0xafee,0xafef
    .hword 0xbc00,0xbc01,0xbc02,0xbc03,0xbc04,0xbc05,0xbc06,0xbc07,0xbc08,0xbc09,0xbc0a,0xbc0b,0xbc0c,0xbc0d,0xbc0e,0xbc0f
    .hword 0xbc20,0xbc21,0xbc22,0xbc23,0xbc24,0xbc25,0xbc26,0xbc27,0xbc28,0xbc29,0xbc2a,0xbc2b,0xbc2c,0xbc2d,0xbc2e,0xbc2f
    .hword 0xbc40,0xbc41,0xbc42,0xbc43,0xbc44,0xbc45,0xbc46,0xbc47,0xbc48,0xbc49,0xbc4a,0xbc4b,0xbc4c,0xbc4d,0xbc4e,0xbc4f
    .hword 0xbc60,0xbc61,0xbc62,0xbc63,0xbc64,0xbc65,0xbc66,0xbc67,0xbc68,0xbc69,0xbc6a,0xbc6b,0xbc6c,0xbc6d,0xbc6e,0xbc6f
    .hword 0xbc80,0xbc81,0xbc82,0xbc83,0xbc84,0xbc85,0xbc86,0xbc87,0xbc88,0xbc89,0xbc8a,0xbc8b,0xbc8c,0xbc8d,0xbc8e,0xbc8f
    .hword 0xbca0,0xbca1,0xbca2,0xbca3,0xbca4,0xbca5,0xbca6,0xbca7,0xbca8,0xbca9,0xbcaa,0xbcab,0xbcac,0xbcad,0xbcae,0xbcaf
    .hword 0xbcc0,0xbcc1,0xbcc2,0xbcc3,0xbcc4,0xbcc5,0xbcc6,0xbcc7,0xbcc8,0xbcc9,0xbcca,0xbccb,0xbccc,0xbccd,0xbcce,0xbccf
    .hword 0xbce0,0xbce1,0xbce2,0xbce3,0xbce4,0xbce5,0xbce6,0xbce7,0xbce8,0xbce9,0xbcea,0xbceb,0xbcec,0xbced,0xbcee,0xbcef
    .hword 0xbd00,0xbd01,0xbd02,0xbd03,0xbd04,0xbd05,0xbd06,0xbd07,0xbd08,0xbd09,0xbd0a,0xbd0b,0xbd0c,0xbd0d,0xbd0e,0xbd0f
    .hword 0xbd20,0xbd21,0xbd22,0xbd23,0xbd24,0xbd25,0xbd26,0xbd27,0xbd28,0xbd29,0xbd2a,0xbd2b,0xbd2c,0xbd2d,0xbd2e,0xbd2f
    .hword 0xbd40,0xbd41,0xbd42,0xbd43,0xbd44,0xbd45,0xbd46,0xbd47,0xbd48,0xbd49,0xbd4a,0xbd4b,0xbd4c,0xbd4d,0xbd4e,0xbd4f
    .hword 0xbd60,0xbd61,0xbd62,0xbd63,0xbd64,0xbd65,0xbd66,0xbd67,0xbd68,0xbd69,0xbd6a,0xbd6b,0xbd6c,0xbd6d,0xbd6e,0xbd6f
    .hword 0xbd80,0xbd81,0xbd82,0xbd83,0xbd84,0xbd85,0xbd86,0xbd87,0xbd88,0xbd89,0xbd8a,0xbd8b,0xbd8c,0xbd8d,0xbd8e,0xbd8f
    .hword 0xbda0,0xbda1,0xbda2,0xbda3,0xbda4,0xbda5,0xbda6,0xbda7,0xbda8,0xbda9,0xbdaa,0xbdab,0xbdac,0xbdad,0xbdae,0xbdaf
    .hword 0xbdc0,0xbdc1,0xbdc2,0xbdc3,0xbdc4,0xbdc5,0xbdc6,0xbdc7,0xbdc8,0xbdc9,0xbdca,0xbdcb,0xbdcc,0xbdcd,0xbdce,0xbdcf
    .hword 0xbde0,0xbde1,0xbde2,0xbde3,0xbde4,0xbde5,0xbde6,0xbde7,0xbde8,0xbde9,0xbdea,0xbdeb,0xbdec,0xbded,0xbdee,0xbdef
    .hword 0xbe00,0xbe01,0xbe02,0xbe03,0xbe04,0xbe05,0xbe06,0xbe07,0xbe08,0xbe09,0xbe0a,0xbe0b,0xbe0c,0xbe0d,0xbe0e,0xbe0f
    .hword 0xbe20,0xbe21,0xbe22,0xbe23,0xbe24,0xbe25,0xbe26,0xbe27,0xbe28,0xbe29,0xbe2a,0xbe2b,0xbe2c,0xbe2d,0xbe2e,0xbe2f
    .hword 0xbe40,0xbe41,0xbe42,0xbe43,0xbe44,0xbe45,0xbe46,0xbe47,0xbe48,0xbe49,0xbe4a,0xbe4b,0xbe4c,0xbe4d,0xbe4e,0xbe4f
    .hword 0xbe60,0xbe61,0xbe62,0xbe63,0xbe64,0xbe65,0xbe66,0xbe67,0xbe68,0xbe69,0xbe6a,0xbe6b,0xbe6c,0xbe6d,0xbe6e,0xbe6f
    .hword 0xbe80,0xbe81,0xbe82,0xbe83,0xbe84,0xbe85,0xbe86,0xbe87,0xbe88,0xbe89,0xbe8a,0xbe8b,0xbe8c,0xbe8d,0xbe8e,0xbe8f
    .hword 0xbea0,0xbea1,0xbea2,0xbea3,0xbea4,0xbea5,0xbea6,0xbea7,0xbea8,0xbea9,0xbeaa,0xbeab,0xbeac,0xbead,0xbeae,0xbeaf
    .hword 0xbec0,0xbec1,0xbec2,0xbec3,0xbec4,0xbec5,0xbec6,0xbec7,0xbec8,0xbec9,0xbeca,0xbecb,0xbecc,0xbecd,0xbece,0xbecf
    .hword 0xbee0,0xbee1,0xbee2,0xbee3,0xbee4,0xbee5,0xbee6,0xbee7,0xbee8,0xbee9,0xbeea,0xbeeb,0xbeec,0xbeed,0xbeee,0xbeef
    .hword 0xbf00,0xbf01,0xbf02,0xbf03,0xbf04,0xbf05,0xbf06,0xbf07,0xbf08,0xbf09,0xbf0a,0xbf0b,0xbf0c,0xbf0d,0xbf0e,0xbf0f
    .hword 0xbf20,0xbf21,0xbf22,0xbf23,0xbf24,0xbf25,0xbf26,0xbf27,0xbf28,0xbf29,0xbf2a,0xbf2b,0xbf2c,0xbf2d,0xbf2e,0xbf2f
    .hword 0xbf40,0xbf41,0xbf42,0xbf43,0xbf44,0xbf45,0xbf46,0xbf47,0xbf48,0xbf49,0xbf4a,0xbf4b,0xbf4c,0xbf4d,0xbf4e,0xbf4f
    .hword 0xbf60,0xbf61,0xbf62,0xbf63,0xbf64,0xbf65,0xbf66,0xbf67,0xbf68,0xbf69,0xbf6a,0xbf6b,0xbf6c,0xbf6d,0xbf6e,0xbf6f
    .hword 0xbf80,0xbf81,0xbf82,0xbf83,0xbf84,0xbf85,0xbf86,0xbf87,0xbf88,0xbf89,0xbf8a,0xbf8b,0xbf8c,0xbf8d,0xbf8e,0xbf8f
    .hword 0xbfa0,0xbfa1,0xbfa2,0xbfa3,0xbfa4,0xbfa5,0xbfa6,0xbfa7,0xbfa8,0xbfa9,0xbfaa,0xbfab,0xbfac,0xbfad,0xbfae,0xbfaf
    .hword 0xbfc0,0xbfc1,0xbfc2,0xbfc3,0xbfc4,0xbfc5,0xbfc6,0xbfc7,0xbfc8,0xbfc9,0xbfca,0xbfcb,0xbfcc,0xbfcd,0xbfce,0xbfcf
    .hword 0xbfe0,0xbfe1,0xbfe2,0xbfe3,0xbfe4,0xbfe5,0xbfe6,0xbfe7,0xbfe8,0xbfe9,0xbfea,0xbfeb,0xbfec,0xbfed,0xbfee,0xbfef
    .hword 0xcc00,0xcc01,0xcc02,0xcc03,0xcc04,0xcc05,0xcc06,0xcc07,0xcc08,0xcc09,0xcc0a,0xcc0b,0xcc0c,0xcc0d,0xcc0e,0xcc0f
    .hword 0xcc20,0xcc21,0xcc22,0xcc23,0xcc24,0xcc25,0xcc26,0xcc27,0xcc28,0xcc29,0xcc2a,0xcc2b,0xcc2c,0xcc2d,0xcc2e,0xcc2f
    .hword 0xcc40,0xcc41,0xcc42,0xcc43,0xcc44,0xcc45,0xcc46,0xcc47,0xcc48,0xcc49,0xcc4a,0xcc4b,0xcc4c,0xcc4d,0xcc4e,0xcc4f
    .hword 0xcc60,0xcc61,0xcc62,0xcc63,0xcc64,0xcc65,0xcc66,0xcc67,0xcc68,0xcc69,0xcc6a,0xcc6b,0xcc6c,0xcc6d,0xcc6e,0xcc6f
    .hword 0xcc80,0xcc81,0xcc82,0xcc83,0xcc84,0xcc85,0xcc86,0xcc87,0xcc88,0xcc89,0xcc8a,0xcc8b,0xcc8c,0xcc8d,0xcc8e,0xcc8f
    .hword 0xcca0,0xcca1,0xcca2,0xcca3,0xcca4,0xcca5,0xcca6,0xcca7,0xcca8,0xcca9,0xccaa,0xccab,0xccac,0xccad,0xccae,0xccaf
    .hword 0xccc0,0xccc1,0xccc2,0xccc3,0xccc4,0xccc5,0xccc6,0xccc7,0xccc8,0xccc9,0xccca,0xcccb,0xcccc,0xcccd,0xccce,0xcccf
    .hword 0xcce0,0xcce1,0xcce2,0xcce3,0xcce4,0xcce5,0xcce6,0xcce7,0xcce8,0xcce9,0xccea,0xcceb,0xccec,0xcced,0xccee,0xccef
    .hword 0xcd00,0xcd01,0xcd02,0xcd03,0xcd04,0xcd05,0xcd06,0xcd07,0xcd08,0xcd09,0xcd0a,0xcd0b,0xcd0c,0xcd0d,0xcd0e,0xcd0f
    .hword 0xcd20,0xcd21,0xcd22,0xcd23,0xcd24,0xcd25,0xcd26,0xcd27,0xcd28,0xcd29,0xcd2a,0xcd2b,0xcd2c,0xcd2d,0xcd2e,0xcd2f
    .hword 0xcd40,0xcd41,0xcd42,0xcd43,0xcd44,0xcd45,0xcd46,0xcd47,0xcd48,0xcd49,0xcd4a,0xcd4b,0xcd4c,0xcd4d,0xcd4e,0xcd4f
    .hword 0xcd60,0xcd61,0xcd62,0xcd63,0xcd64,0xcd65,0xcd66,0xcd67,0xcd68,0xcd69,0xcd6a,0xcd6b,0xcd6c,0xcd6d,0xcd6e,0xcd6f
    .hword 0xcd80,0xcd81,0xcd82,0xcd83,0xcd84,0xcd85,0xcd86,0xcd87,0xcd88,0xcd89,0xcd8a,0xcd8b,0xcd8c,0xcd8d,0xcd8e,0xcd8f
    .hword 0xcda0,0xcda1,0xcda2,0xcda3,0xcda4,0xcda5,0xcda6,0xcda7,0xcda8,0xcda9,0xcdaa,0xcdab,0xcdac,0xcdad,0xcdae,0xcdaf
    .hword 0xcdc0,0xcdc1,0xcdc2,0xcdc3,0xcdc4,0xcdc5,0xcdc6,0xcdc7,0xcdc8,0xcdc9,0xcdca,0xcdcb,0xcdcc,0xcdcd,0xcdce,0xcdcf
    .hword 0xcde0,0xcde1,0xcde2,0xcde3,0xcde4,0xcde5,0xcde6,0xcde7,0xcde8,0xcde9,0xcdea,0xcdeb,0xcdec,0xcded,0xcdee,0xcdef
    .hword 0xce00,0xce01,0xce02,0xce03,0xce04,0xce05,0xce06,0xce07,0xce08,0xce09,0xce0a,0xce0b,0xce0c,0xce0d,0xce0e,0xce0f
    .hword 0xce20,0xce21,0xce22,0xce23,0xce24,0xce25,0xce26,0xce27,0xce28,0xce29,0xce2a,0xce2b,0xce2c,0xce2d,0xce2e,0xce2f
    .hword 0xce40,0xce41,0xce42,0xce43,0xce44,0xce45,0xce46,0xce47,0xce48,0xce49,0xce4a,0xce4b,0xce4c,0xce4d,0xce4e,0xce4f
    .hword 0xce60,0xce61,0xce62,0xce63,0xce64,0xce65,0xce66,0xce67,0xce68,0xce69,0xce6a,0xce6b,0xce6c,0xce6d,0xce6e,0xce6f
    .hword 0xce80,0xce81,0xce82,0xce83,0xce84,0xce85,0xce86,0xce87,0xce88,0xce89,0xce8a,0xce8b,0xce8c,0xce8d,0xce8e,0xce8f
    .hword 0xcea0,0xcea1,0xcea2,0xcea3,0xcea4,0xcea5,0xcea6,0xcea7,0xcea8,0xcea9,0xceaa,0xceab,0xceac,0xcead,0xceae,0xceaf
    .hword 0xcec0,0xcec1,0xcec2,0xcec3,0xcec4,0xcec5,0xcec6,0xcec7,0xcec8,0xcec9,0xceca,0xcecb,0xcecc,0xcecd,0xcece,0xcecf
    .hword 0xcee0,0xcee1,0xcee2,0xcee3,0xcee4,0xcee5,0xcee6,0xcee7,0xcee8,0xcee9,0xceea,0xceeb,0xceec,0xceed,0xceee,0xceef
    .hword 0xcf00,0xcf01,0xcf02,0xcf03,0xcf04,0xcf05,0xcf06,0xcf07,0xcf08,0xcf09,0xcf0a,0xcf0b,0xcf0c,0xcf0d,0xcf0e,0xcf0f
    .hword 0xcf20,0xcf21,0xcf22,0xcf23,0xcf24,0xcf25,0xcf26,0xcf27,0xcf28,0xcf29,0xcf2a,0xcf2b,0xcf2c,0xcf2d,0xcf2e,0xcf2f
    .hword 0xcf40,0xcf41,0xcf42,0xcf43,0xcf44,0xcf45,0xcf46,0xcf47,0xcf48,0xcf49,0xcf4a,0xcf4b,0xcf4c,0xcf4d,0xcf4e,0xcf4f
    .hword 0xcf60,0xcf61,0xcf62,0xcf63,0xcf64,0xcf65,0xcf66,0xcf67,0xcf68,0xcf69,0xcf6a,0xcf6b,0xcf6c,0xcf6d,0xcf6e,0xcf6f
    .hword 0xcf80,0xcf81,0xcf82,0xcf83,0xcf84,0xcf85,0xcf86,0xcf87,0xcf88,0xcf89,0xcf8a,0xcf8b,0xcf8c,0xcf8d,0xcf8e,0xcf8f
    .hword 0xcfa0,0xcfa1,0xcfa2,0xcfa3,0xcfa4,0xcfa5,0xcfa6,0xcfa7,0xcfa8,0xcfa9,0xcfaa,0xcfab,0xcfac,0xcfad,0xcfae,0xcfaf
    .hword 0xcfc0,0xcfc1,0xcfc2,0xcfc3,0xcfc4,0xcfc5,0xcfc6,0xcfc7,0xcfc8,0xcfc9,0xcfca,0xcfcb,0xcfcc,0xcfcd,0xcfce,0xcfcf
    .hword 0xcfe0,0xcfe1,0xcfe2,0xcfe3,0xcfe4,0xcfe5,0xcfe6,0xcfe7,0xcfe8,0xcfe9,0xcfea,0xcfeb,0xcfec,0xcfed,0xcfee,0xcfef
    .hword 0xdc00,0xdc01,0xdc02,0xdc03,0xdc04,0xdc05,0xdc06,0xdc07,0xdc08,0xdc09,0xdc0a,0xdc0b,0xdc0c,0xdc0d,0xdc0e,0xdc0f
    .hword 0xdc20,0xdc21,0xdc22,0xdc23,0xdc24,0xdc25,0xdc26,0xdc27,0xdc28,0xdc29,0xdc2a,0xdc2b,0xdc2c,0xdc2d,0xdc2e,0xdc2f
    .hword 0xdc40,0xdc41,0xdc42,0xdc43,0xdc44,0xdc45,0xdc46,0xdc47,0xdc48,0xdc49,0xdc4a,0xdc4b,0xdc4c,0xdc4d,0xdc4e,0xdc4f
    .hword 0xdc60,0xdc61,0xdc62,0xdc63,0xdc64,0xdc65,0xdc66,0xdc67,0xdc68,0xdc69,0xdc6a,0xdc6b,0xdc6c,0xdc6d,0xdc6e,0xdc6f
    .hword 0xdc80,0xdc81,0xdc82,0xdc83,0xdc84,0xdc85,0xdc86,0xdc87,0xdc88,0xdc89,0xdc8a,0xdc8b,0xdc8c,0xdc8d,0xdc8e,0xdc8f
    .hword 0xdca0,0xdca1,0xdca2,0xdca3,0xdca4,0xdca5,0xdca6,0xdca7,0xdca8,0xdca9,0xdcaa,0xdcab,0xdcac,0xdcad,0xdcae,0xdcaf
    .hword 0xdcc0,0xdcc1,0xdcc2,0xdcc3,0xdcc4,0xdcc5,0xdcc6,0xdcc7,0xdcc8,0xdcc9,0xdcca,0xdccb,0xdccc,0xdccd,0xdcce,0xdccf
    .hword 0xdce0,0xdce1,0xdce2,0xdce3,0xdce4,0xdce5,0xdce6,0xdce7,0xdce8,0xdce9,0xdcea,0xdceb,0xdcec,0xdced,0xdcee,0xdcef
    .hword 0xdd00,0xdd01,0xdd02,0xdd03,0xdd04,0xdd05,0xdd06,0xdd07,0xdd08,0xdd09,0xdd0a,0xdd0b,0xdd0c,0xdd0d,0xdd0e,0xdd0f
    .hword 0xdd20,0xdd21,0xdd22,0xdd23,0xdd24,0xdd25,0xdd26,0xdd27,0xdd28,0xdd29,0xdd2a,0xdd2b,0xdd2c,0xdd2d,0xdd2e,0xdd2f
    .hword 0xdd40,0xdd41,0xdd42,0xdd43,0xdd44,0xdd45,0xdd46,0xdd47,0xdd48,0xdd49,0xdd4a,0xdd4b,0xdd4c,0xdd4d,0xdd4e,0xdd4f
    .hword 0xdd60,0xdd61,0xdd62,0xdd63,0xdd64,0xdd65,0xdd66,0xdd67,0xdd68,0xdd69,0xdd6a,0xdd6b,0xdd6c,0xdd6d,0xdd6e,0xdd6f
    .hword 0xdd80,0xdd81,0xdd82,0xdd83,0xdd84,0xdd85,0xdd86,0xdd87,0xdd88,0xdd89,0xdd8a,0xdd8b,0xdd8c,0xdd8d,0xdd8e,0xdd8f
    .hword 0xdda0,0xdda1,0xdda2,0xdda3,0xdda4,0xdda5,0xdda6,0xdda7,0xdda8,0xdda9,0xddaa,0xddab,0xddac,0xddad,0xddae,0xddaf
    .hword 0xddc0,0xddc1,0xddc2,0xddc3,0xddc4,0xddc5,0xddc6,0xddc7,0xddc8,0xddc9,0xddca,0xddcb,0xddcc,0xddcd,0xddce,0xddcf
    .hword 0xdde0,0xdde1,0xdde2,0xdde3,0xdde4,0xdde5,0xdde6,0xdde7,0xdde8,0xdde9,0xddea,0xddeb,0xddec,0xdded,0xddee,0xddef
    .hword 0xde00,0xde01,0xde02,0xde03,0xde04,0xde05,0xde06,0xde07,0xde08,0xde09,0xde0a,0xde0b,0xde0c,0xde0d,0xde0e,0xde0f
    .hword 0xde20,0xde21,0xde22,0xde23,0xde24,0xde25,0xde26,0xde27,0xde28,0xde29,0xde2a,0xde2b,0xde2c,0xde2d,0xde2e,0xde2f
    .hword 0xde40,0xde41,0xde42,0xde43,0xde44,0xde45,0xde46,0xde47,0xde48,0xde49,0xde4a,0xde4b,0xde4c,0xde4d,0xde4e,0xde4f
    .hword 0xde60,0xde61,0xde62,0xde63,0xde64,0xde65,0xde66,0xde67,0xde68,0xde69,0xde6a,0xde6b,0xde6c,0xde6d,0xde6e,0xde6f
    .hword 0xde80,0xde81,0xde82,0xde83,0xde84,0xde85,0xde86,0xde87,0xde88,0xde89,0xde8a,0xde8b,0xde8c,0xde8d,0xde8e,0xde8f
    .hword 0xdea0,0xdea1,0xdea2,0xdea3,0xdea4,0xdea5,0xdea6,0xdea7,0xdea8,0xdea9,0xdeaa,0xdeab,0xdeac,0xdead,0xdeae,0xdeaf
    .hword 0xdec0,0xdec1,0xdec2,0xdec3,0xdec4,0xdec5,0xdec6,0xdec7,0xdec8,0xdec9,0xdeca,0xdecb,0xdecc,0xdecd,0xdece,0xdecf
    .hword 0xdee0,0xdee1,0xdee2,0xdee3,0xdee4,0xdee5,0xdee6,0xdee7,0xdee8,0xdee9,0xdeea,0xdeeb,0xdeec,0xdeed,0xdeee,0xdeef
    .hword 0xdf00,0xdf01,0xdf02,0xdf03,0xdf04,0xdf05,0xdf06,0xdf07,0xdf08,0xdf09,0xdf0a,0xdf0b,0xdf0c,0xdf0d,0xdf0e,0xdf0f
    .hword 0xdf20,0xdf21,0xdf22,0xdf23,0xdf24,0xdf25,0xdf26,0xdf27,0xdf28,0xdf29,0xdf2a,0xdf2b,0xdf2c,0xdf2d,0xdf2e,0xdf2f
    .hword 0xdf40,0xdf41,0xdf42,0xdf43,0xdf44,0xdf45,0xdf46,0xdf47,0xdf48,0xdf49,0xdf4a,0xdf4b,0xdf4c,0xdf4d,0xdf4e,0xdf4f
    .hword 0xdf60,0xdf61,0xdf62,0xdf63,0xdf64,0xdf65,0xdf66,0xdf67,0xdf68,0xdf69,0xdf6a,0xdf6b,0xdf6c,0xdf6d,0xdf6e,0xdf6f
    .hword 0xdf80,0xdf81,0xdf82,0xdf83,0xdf84,0xdf85,0xdf86,0xdf87,0xdf88,0xdf89,0xdf8a,0xdf8b,0xdf8c,0xdf8d,0xdf8e,0xdf8f
    .hword 0xdfa0,0xdfa1,0xdfa2,0xdfa3,0xdfa4,0xdfa5,0xdfa6,0xdfa7,0xdfa8,0xdfa9,0xdfaa,0xdfab,0xdfac,0xdfad,0xdfae,0xdfaf
    .hword 0xdfc0,0xdfc1,0xdfc2,0xdfc3,0xdfc4,0xdfc5,0xdfc6,0xdfc7,0xdfc8,0xdfc9,0xdfca,0xdfcb,0xdfcc,0xdfcd,0xdfce,0xdfcf
    .hword 0xdfe0,0xdfe1,0xdfe2,0xdfe3,0xdfe4,0xdfe5,0xdfe6,0xdfe7,0xdfe8,0xdfe9,0xdfea,0xdfeb,0xdfec,0xdfed,0xdfee,0xdfef
    .hword 0xec00,0xec01,0xec02,0xec03,0xec04,0xec05,0xec06,0xec07,0xec08,0xec09,0xec0a,0xec0b,0xec0c,0xec0d,0xec0e,0xec0f
    .hword 0xec20,0xec21,0xec22,0xec23,0xec24,0xec25,0xec26,0xec27,0xec28,0xec29,0xec2a,0xec2b,0xec2c,0xec2d,0xec2e,0xec2f
    .hword 0xec40,0xec41,0xec42,0xec43,0xec44,0xec45,0xec46,0xec47,0xec48,0xec49,0xec4a,0xec4b,0xec4c,0xec4d,0xec4e,0xec4f
    .hword 0xec60,0xec61,0xec62,0xec63,0xec64,0xec65,0xec66,0xec67,0xec68,0xec69,0xec6a,0xec6b,0xec6c,0xec6d,0xec6e,0xec6f
    .hword 0xec80,0xec81,0xec82,0xec83,0xec84,0xec85,0xec86,0xec87,0xec88,0xec89,0xec8a,0xec8b,0xec8c,0xec8d,0xec8e,0xec8f
    .hword 0xeca0,0xeca1,0xeca2,0xeca3,0xeca4,0xeca5,0xeca6,0xeca7,0xeca8,0xeca9,0xecaa,0xecab,0xecac,0xecad,0xecae,0xecaf
    .hword 0xecc0,0xecc1,0xecc2,0xecc3,0xecc4,0xecc5,0xecc6,0xecc7,0xecc8,0xecc9,0xecca,0xeccb,0xeccc,0xeccd,0xecce,0xeccf
    .hword 0xece0,0xece1,0xece2,0xece3,0xece4,0xece5,0xece6,0xece7,0xece8,0xece9,0xecea,0xeceb,0xecec,0xeced,0xecee,0xecef
    .hword 0xed00,0xed01,0xed02,0xed03,0xed04,0xed05,0xed06,0xed07,0xed08,0xed09,0xed0a,0xed0b,0xed0c,0xed0d,0xed0e,0xed0f
    .hword 0xed20,0xed21,0xed22,0xed23,0xed24,0xed25,0xed26,0xed27,0xed28,0xed29,0xed2a,0xed2b,0xed2c,0xed2d,0xed2e,0xed2f
    .hword 0xed40,0xed41,0xed42,0xed43,0xed44,0xed45,0xed46,0xed47,0xed48,0xed49,0xed4a,0xed4b,0xed4c,0xed4d,0xed4e,0xed4f
    .hword 0xed60,0xed61,0xed62,0xed63,0xed64,0xed65,0xed66,0xed67,0xed68,0xed69,0xed6a,0xed6b,0xed6c,0xed6d,0xed6e,0xed6f
    .hword 0xed80,0xed81,0xed82,0xed83,0xed84,0xed85,0xed86,0xed87,0xed88,0xed89,0xed8a,0xed8b,0xed8c,0xed8d,0xed8e,0xed8f
    .hword 0xeda0,0xeda1,0xeda2,0xeda3,0xeda4,0xeda5,0xeda6,0xeda7,0xeda8,0xeda9,0xedaa,0xedab,0xedac,0xedad,0xedae,0xedaf
    .hword 0xedc0,0xedc1,0xedc2,0xedc3,0xedc4,0xedc5,0xedc6,0xedc7,0xedc8,0xedc9,0xedca,0xedcb,0xedcc,0xedcd,0xedce,0xedcf
    .hword 0xede0,0xede1,0xede2,0xede3,0xede4,0xede5,0xede6,0xede7,0xede8,0xede9,0xedea,0xedeb,0xedec,0xeded,0xedee,0xedef
    .hword 0xee00,0xee01,0xee02,0xee03,0xee04,0xee05,0xee06,0xee07,0xee08,0xee09,0xee0a,0xee0b,0xee0c,0xee0d,0xee0e,0xee0f
    .hword 0xee20,0xee21,0xee22,0xee23,0xee24,0xee25,0xee26,0xee27,0xee28,0xee29,0xee2a,0xee2b,0xee2c,0xee2d,0xee2e,0xee2f
    .hword 0xee40,0xee41,0xee42,0xee43,0xee44,0xee45,0xee46,0xee47,0xee48,0xee49,0xee4a,0xee4b,0xee4c,0xee4d,0xee4e,0xee4f
    .hword 0xee60,0xee61,0xee62,0xee63,0xee64,0xee65,0xee66,0xee67,0xee68,0xee69,0xee6a,0xee6b,0xee6c,0xee6d,0xee6e,0xee6f
    .hword 0xee80,0xee81,0xee82,0xee83,0xee84,0xee85,0xee86,0xee87,0xee88,0xee89,0xee8a,0xee8b,0xee8c,0xee8d,0xee8e,0xee8f
    .hword 0xeea0,0xeea1,0xeea2,0xeea3,0xeea4,0xeea5,0xeea6,0xeea7,0xeea8,0xeea9,0xeeaa,0xeeab,0xeeac,0xeead,0xeeae,0xeeaf
    .hword 0xeec0,0xeec1,0xeec2,0xeec3,0xeec4,0xeec5,0xeec6,0xeec7,0xeec8,0xeec9,0xeeca,0xeecb,0xeecc,0xeecd,0xeece,0xeecf
    .hword 0xeee0,0xeee1,0xeee2,0xeee3,0xeee4,0xeee5,0xeee6,0xeee7,0xeee8,0xeee9,0xeeea,0xeeeb,0xeeec,0xeeed,0xeeee,0xeeef
    .hword 0xef00,0xef01,0xef02,0xef03,0xef04,0xef05,0xef06,0xef07,0xef08,0xef09,0xef0a,0xef0b,0xef0c,0xef0d,0xef0e,0xef0f
    .hword 0xef20,0xef21,0xef22,0xef23,0xef24,0xef25,0xef26,0xef27,0xef28,0xef29,0xef2a,0xef2b,0xef2c,0xef2d,0xef2e,0xef2f
    .hword 0xef40,0xef41,0xef42,0xef43,0xef44,0xef45,0xef46,0xef47,0xef48,0xef49,0xef4a,0xef4b,0xef4c,0xef4d,0xef4e,0xef4f
    .hword 0xef60,0xef61,0xef62,0xef63,0xef64,0xef65,0xef66,0xef67,0xef68,0xef69,0xef6a,0xef6b,0xef6c,0xef6d,0xef6e,0xef6f
    .hword 0xef80,0xef81,0xef82,0xef83,0xef84,0xef85,0xef86,0xef87,0xef88,0xef89,0xef8a,0xef8b,0xef8c,0xef8d,0xef8e,0xef8f
    .hword 0xefa0,0xefa1,0xefa2,0xefa3,0xefa4,0xefa5,0xefa6,0xefa7,0xefa8,0xefa9,0xefaa,0xefab,0xefac,0xefad,0xefae,0xefaf
    .hword 0xefc0,0xefc1,0xefc2,0xefc3,0xefc4,0xefc5,0xefc6,0xefc7,0xefc8,0xefc9,0xefca,0xefcb,0xefcc,0xefcd,0xefce,0xefcf
    .hword 0xefe0,0xefe1,0xefe2,0xefe3,0xefe4,0xefe5,0xefe6,0xefe7,0xefe8,0xefe9,0xefea,0xefeb,0xefec,0xefed,0xefee,0xefef
    .hword 0xfc00,0xfc01,0xfc02,0xfc03,0xfc04,0xfc05,0xfc06,0xfc07,0xfc08,0xfc09,0xfc0a,0xfc0b,0xfc0c,0xfc0d,0xfc0e,0xfc0f
    .hword 0xfc20,0xfc21,0xfc22,0xfc23,0xfc24,0xfc25,0xfc26,0xfc27,0xfc28,0xfc29,0xfc2a,0xfc2b,0xfc2c,0xfc2d,0xfc2e,0xfc2f
    .hword 0xfc40,0xfc41,0xfc42,0xfc43,0xfc44,0xfc45,0xfc46,0xfc47,0xfc48,0xfc49,0xfc4a,0xfc4b,0xfc4c,0xfc4d,0xfc4e,0xfc4f
    .hword 0xfc60,0xfc61,0xfc62,0xfc63,0xfc64,0xfc65,0xfc66,0xfc67,0xfc68,0xfc69,0xfc6a,0xfc6b,0xfc6c,0xfc6d,0xfc6e,0xfc6f
    .hword 0xfc80,0xfc81,0xfc82,0xfc83,0xfc84,0xfc85,0xfc86,0xfc87,0xfc88,0xfc89,0xfc8a,0xfc8b,0xfc8c,0xfc8d,0xfc8e,0xfc8f
    .hword 0xfca0,0xfca1,0xfca2,0xfca3,0xfca4,0xfca5,0xfca6,0xfca7,0xfca8,0xfca9,0xfcaa,0xfcab,0xfcac,0xfcad,0xfcae,0xfcaf
    .hword 0xfcc0,0xfcc1,0xfcc2,0xfcc3,0xfcc4,0xfcc5,0xfcc6,0xfcc7,0xfcc8,0xfcc9,0xfcca,0xfccb,0xfccc,0xfccd,0xfcce,0xfccf
    .hword 0xfce0,0xfce1,0xfce2,0xfce3,0xfce4,0xfce5,0xfce6,0xfce7,0xfce8,0xfce9,0xfcea,0xfceb,0xfcec,0xfced,0xfcee,0xfcef
    .hword 0xfd00,0xfd01,0xfd02,0xfd03,0xfd04,0xfd05,0xfd06,0xfd07,0xfd08,0xfd09,0xfd0a,0xfd0b,0xfd0c,0xfd0d,0xfd0e,0xfd0f
    .hword 0xfd20,0xfd21,0xfd22,0xfd23,0xfd24,0xfd25,0xfd26,0xfd27,0xfd28,0xfd29,0xfd2a,0xfd2b,0xfd2c,0xfd2d,0xfd2e,0xfd2f
    .hword 0xfd40,0xfd41,0xfd42,0xfd43,0xfd44,0xfd45,0xfd46,0xfd47,0xfd48,0xfd49,0xfd4a,0xfd4b,0xfd4c,0xfd4d,0xfd4e,0xfd4f
    .hword 0xfd60,0xfd61,0xfd62,0xfd63,0xfd64,0xfd65,0xfd66,0xfd67,0xfd68,0xfd69,0xfd6a,0xfd6b,0xfd6c,0xfd6d,0xfd6e,0xfd6f
    .hword 0xfd80,0xfd81,0xfd82,0xfd83,0xfd84,0xfd85,0xfd86,0xfd87,0xfd88,0xfd89,0xfd8a,0xfd8b,0xfd8c,0xfd8d,0xfd8e,0xfd8f
    .hword 0xfda0,0xfda1,0xfda2,0xfda3,0xfda4,0xfda5,0xfda6,0xfda7,0xfda8,0xfda9,0xfdaa,0xfdab,0xfdac,0xfdad,0xfdae,0xfdaf
    .hword 0xfdc0,0xfdc1,0xfdc2,0xfdc3,0xfdc4,0xfdc5,0xfdc6,0xfdc7,0xfdc8,0xfdc9,0xfdca,0xfdcb,0xfdcc,0xfdcd,0xfdce,0xfdcf
    .hword 0xfde0,0xfde1,0xfde2,0xfde3,0xfde4,0xfde5,0xfde6,0xfde7,0xfde8,0xfde9,0xfdea,0xfdeb,0xfdec,0xfded,0xfdee,0xfdef
    .hword 0xfe00,0xfe01,0xfe02,0xfe03,0xfe04,0xfe05,0xfe06,0xfe07,0xfe08,0xfe09,0xfe0a,0xfe0b,0xfe0c,0xfe0d,0xfe0e,0xfe0f
    .hword 0xfe20,0xfe21,0xfe22,0xfe23,0xfe24,0xfe25,0xfe26,0xfe27,0xfe28,0xfe29,0xfe2a,0xfe2b,0xfe2c,0xfe2d,0xfe2e,0xfe2f
    .hword 0xfe40,0xfe41,0xfe42,0xfe43,0xfe44,0xfe45,0xfe46,0xfe47,0xfe48,0xfe49,0xfe4a,0xfe4b,0xfe4c,0xfe4d,0xfe4e,0xfe4f
    .hword 0xfe60,0xfe61,0xfe62,0xfe63,0xfe64,0xfe65,0xfe66,0xfe67,0xfe68,0xfe69,0xfe6a,0xfe6b,0xfe6c,0xfe6d,0xfe6e,0xfe6f
    .hword 0xfe80,0xfe81,0xfe82,0xfe83,0xfe84,0xfe85,0xfe86,0xfe87,0xfe88,0xfe89,0xfe8a,0xfe8b,0xfe8c,0xfe8d,0xfe8e,0xfe8f
    .hword 0xfea0,0xfea1,0xfea2,0xfea3,0xfea4,0xfea5,0xfea6,0xfea7,0xfea8,0xfea9,0xfeaa,0xfeab,0xfeac,0xfead,0xfeae,0xfeaf
    .hword 0xfec0,0xfec1,0xfec2,0xfec3,0xfec4,0xfec5,0xfec6,0xfec7,0xfec8,0xfec9,0xfeca,0xfecb,0xfecc,0xfecd,0xfece,0xfecf
    .hword 0xfee0,0xfee1,0xfee2,0xfee3,0xfee4,0xfee5,0xfee6,0xfee7,0xfee8,0xfee9,0xfeea,0xfeeb,0xfeec,0xfeed,0xfeee,0xfeef
    .hword 0xff00,0xff01,0xff02,0xff03,0xff04,0xff05,0xff06,0xff07,0xff08,0xff09,0xff0a,0xff0b,0xff0c,0xff0d,0xff0e,0xff0f
    .hword 0xff20,0xff21,0xff22,0xff23,0xff24,0xff25,0xff26,0xff27,0xff28,0xff29,0xff2a,0xff2b,0xff2c,0xff2d,0xff2e,0xff2f
    .hword 0xff40,0xff41,0xff42,0xff43,0xff44,0xff45,0xff46,0xff47,0xff48,0xff49,0xff4a,0xff4b,0xff4c,0xff4d,0xff4e,0xff4f
    .hword 0xff60,0xff61,0xff62,0xff63,0xff64,0xff65,0xff66,0xff67,0xff68,0xff69,0xff6a,0xff6b,0xff6c,0xff6d,0xff6e,0xff6f
    .hword 0xff80,0xff81,0xff82,0xff83,0xff84,0xff85,0xff86,0xff87,0xff88,0xff89,0xff8a,0xff8b,0xff8c,0xff8d,0xff8e,0xff8f
    .hword 0xffa0,0xffa1,0xffa2,0xffa3,0xffa4,0xffa5,0xffa6,0xffa7,0xffa8,0xffa9,0xffaa,0xffab,0xffac,0xffad,0xffae,0xffaf
    .hword 0xffc0,0xffc1,0xffc2,0xffc3,0xffc4,0xffc5,0xffc6,0xffc7,0xffc8,0xffc9,0xffca,0xffcb,0xffcc,0xffcd,0xffce,0xffcf
    .hword 0xffe0,0xffe1,0xffe2,0xffe3,0xffe4,0xffe5,0xffe6,0xffe7,0xffe8,0xffe9,0xffea,0xffeb,0xffec,0xffed,0xffee,0xffef
    .hword 0x8800,0x8801,0x8802,0x8803,0x8804,0x8805,0x8806,0x8807,0x8808,0x8809,0x880a,0x880b,0x880c,0x880d,0x880e,0x880f
    .hword 0x8820,0x8821,0x8822,0x8823,0x8824,0x8825,0x8826,0x8827,0x8828,0x8829,0x882a,0x882b,0x882c,0x882d,0x882e,0x882f
    .hword 0x8840,0x8841,0x8842,0x8843,0x8844,0x8845,0x8846,0x8847,0x8848,0x8849,0x884a,0x884b,0x884c,0x884d,0x884e,0x884f
    .hword 0x8860,0x8861,0x8862,0x8863,0x8864,0x8865,0x8866,0x8867,0x8868,0x8869,0x886a,0x886b,0x886c,0x886d,0x886e,0x886f
    .hword 0x8880,0x8881,0x8882,0x8883,0x8884,0x8885,0x8886,0x8887,0x8888,0x8889,0x888a,0x888b,0x888c,0x888d,0x888e,0x888f
    .hword 0x88a0,0x88a1,0x88a2,0x88a3,0x88a4,0x88a5,0x88a6,0x88a7,0x88a8,0x88a9,0x88aa,0x88ab,0x88ac,0x88ad,0x88ae,0x88af
    .hword 0x88c0,0x88c1,0x88c2,0x88c3,0x88c4,0x88c5,0x88c6,0x88c7,0x88c8,0x88c9,0x88ca,0x88cb,0x88cc,0x88cd,0x88ce,0x88cf
    .hword 0x88e0,0x88e1,0x88e2,0x88e3,0x88e4,0x88e5,0x88e6,0x88e7,0x88e8,0x88e9,0x88ea,0x88eb,0x88ec,0x88ed,0x88ee,0x88ef
    .hword 0x8900,0x8901,0x8902,0x8903,0x8904,0x8905,0x8906,0x8907,0x8908,0x8909,0x890a,0x890b,0x890c,0x890d,0x890e,0x890f
    .hword 0x8920,0x8921,0x8922,0x8923,0x8924,0x8925,0x8926,0x8927,0x8928,0x8929,0x892a,0x892b,0x892c,0x892d,0x892e,0x892f
    .hword 0x8940,0x8941,0x8942,0x8943,0x8944,0x8945,0x8946,0x8947,0x8948,0x8949,0x894a,0x894b,0x894c,0x894d,0x894e,0x894f
    .hword 0x8960,0x8961,0x8962,0x8963,0x8964,0x8965,0x8966,0x8967,0x8968,0x8969,0x896a,0x896b,0x896c,0x896d,0x896e,0x896f
    .hword 0x8980,0x8981,0x8982,0x8983,0x8984,0x8985,0x8986,0x8987,0x8988,0x8989,0x898a,0x898b,0x898c,0x898d,0x898e,0x898f
    .hword 0x89a0,0x89a1,0x89a2,0x89a3,0x89a4,0x89a5,0x89a6,0x89a7,0x89a8,0x89a9,0x89aa,0x89ab,0x89ac,0x89ad,0x89ae,0x89af
    .hword 0x89c0,0x89c1,0x89c2,0x89c3,0x89c4,0x89c5,0x89c6,0x89c7,0x89c8,0x89c9,0x89ca,0x89cb,0x89cc,0x89cd,0x89ce,0x89cf
    .hword 0x89e0,0x89e1,0x89e2,0x89e3,0x89e4,0x89e5,0x89e6,0x89e7,0x89e8,0x89e9,0x89ea,0x89eb,0x89ec,0x89ed,0x89ee,0x89ef
    .hword 0x8a00,0x8a01,0x8a02,0x8a03,0x8a04,0x8a05,0x8a06,0x8a07,0x8a08,0x8a09,0x8a0a,0x8a0b,0x8a0c,0x8a0d,0x8a0e,0x8a0f
    .hword 0x8a20,0x8a21,0x8a22,0x8a23,0x8a24,0x8a25,0x8a26,0x8a27,0x8a28,0x8a29,0x8a2a,0x8a2b,0x8a2c,0x8a2d,0x8a2e,0x8a2f
    .hword 0x8a40,0x8a41,0x8a42,0x8a43,0x8a44,0x8a45,0x8a46,0x8a47,0x8a48,0x8a49,0x8a4a,0x8a4b,0x8a4c,0x8a4d,0x8a4e,0x8a4f
    .hword 0x8a60,0x8a61,0x8a62,0x8a63,0x8a64,0x8a65,0x8a66,0x8a67,0x8a68,0x8a69,0x8a6a,0x8a6b,0x8a6c,0x8a6d,0x8a6e,0x8a6f
    .hword 0x8a80,0x8a81,0x8a82,0x8a83,0x8a84,0x8a85,0x8a86,0x8a87,0x8a88,0x8a89,0x8a8a,0x8a8b,0x8a8c,0x8a8d,0x8a8e,0x8a8f
    .hword 0x8aa0,0x8aa1,0x8aa2,0x8aa3,0x8aa4,0x8aa5,0x8aa6,0x8aa7,0x8aa8,0x8aa9,0x8aaa,0x8aab,0x8aac,0x8aad,0x8aae,0x8aaf
    .hword 0x8ac0,0x8ac1,0x8ac2,0x8ac3,0x8ac4,0x8ac5,0x8ac6,0x8ac7,0x8ac8,0x8ac9,0x8aca,0x8acb,0x8acc,0x8acd,0x8ace,0x8acf
    .hword 0x8ae0,0x8ae1,0x8ae2,0x8ae3,0x8ae4,0x8ae5,0x8ae6,0x8ae7,0x8ae8,0x8ae9,0x8aea,0x8aeb,0x8aec,0x8aed,0x8aee,0x8aef
    .hword 0x8b00,0x8b01,0x8b02,0x8b03,0x8b04,0x8b05,0x8b06,0x8b07,0x8b08,0x8b09,0x8b0a,0x8b0b,0x8b0c,0x8b0d,0x8b0e,0x8b0f
    .hword 0x8b20,0x8b21,0x8b22,0x8b23,0x8b24,0x8b25,0x8b26,0x8b27,0x8b28,0x8b29,0x8b2a,0x8b2b,0x8b2c,0x8b2d,0x8b2e,0x8b2f
    .hword 0x8b40,0x8b41,0x8b42,0x8b43,0x8b44,0x8b45,0x8b46,0x8b47,0x8b48,0x8b49,0x8b4a,0x8b4b,0x8b4c,0x8b4d,0x8b4e,0x8b4f
    .hword 0x8b60,0x8b61,0x8b62,0x8b63,0x8b64,0x8b65,0x8b66,0x8b67,0x8b68,0x8b69,0x8b6a,0x8b6b,0x8b6c,0x8b6d,0x8b6e,0x8b6f
    .hword 0x8b80,0x8b81,0x8b82,0x8b83,0x8b84,0x8b85,0x8b86,0x8b87,0x8b88,0x8b89,0x8b8a,0x8b8b,0x8b8c,0x8b8d,0x8b8e,0x8b8f
    .hword 0x8ba0,0x8ba1,0x8ba2,0x8ba3,0x8ba4,0x8ba5,0x8ba6,0x8ba7,0x8ba8,0x8ba9,0x8baa,0x8bab,0x8bac,0x8bad,0x8bae,0x8baf
    .hword 0x8bc0,0x8bc1,0x8bc2,0x8bc3,0x8bc4,0x8bc5,0x8bc6,0x8bc7,0x8bc8,0x8bc9,0x8bca,0x8bcb,0x8bcc,0x8bcd,0x8bce,0x8bcf
    .hword 0x8be0,0x8be1,0x8be2,0x8be3,0x8be4,0x8be5,0x8be6,0x8be7,0x8be8,0x8be9,0x8bea,0x8beb,0x8bec,0x8bed,0x8bee,0x8bef
    .hword 0x9800,0x9801,0x9802,0x9803,0x9804,0x9805,0x9806,0x9807,0x9808,0x9809,0x980a,0x980b,0x980c,0x980d,0x980e,0x980f
    .hword 0x9820,0x9821,0x9822,0x9823,0x9824,0x9825,0x9826,0x9827,0x9828,0x9829,0x982a,0x982b,0x982c,0x982d,0x982e,0x982f
    .hword 0x9840,0x9841,0x9842,0x9843,0x9844,0x9845,0x9846,0x9847,0x9848,0x9849,0x984a,0x984b,0x984c,0x984d,0x984e,0x984f
    .hword 0x9860,0x9861,0x9862,0x9863,0x9864,0x9865,0x9866,0x9867,0x9868,0x9869,0x986a,0x986b,0x986c,0x986d,0x986e,0x986f
    .hword 0x9880,0x9881,0x9882,0x9883,0x9884,0x9885,0x9886,0x9887,0x9888,0x9889,0x988a,0x988b,0x988c,0x988d,0x988e,0x988f
    .hword 0x98a0,0x98a1,0x98a2,0x98a3,0x98a4,0x98a5,0x98a6,0x98a7,0x98a8,0x98a9,0x98aa,0x98ab,0x98ac,0x98ad,0x98ae,0x98af
    .hword 0x98c0,0x98c1,0x98c2,0x98c3,0x98c4,0x98c5,0x98c6,0x98c7,0x98c8,0x98c9,0x98ca,0x98cb,0x98cc,0x98cd,0x98ce,0x98cf
    .hword 0x98e0,0x98e1,0x98e2,0x98e3,0x98e4,0x98e5,0x98e6,0x98e7,0x98e8,0x98e9,0x98ea,0x98eb,0x98ec,0x98ed,0x98ee,0x98ef
    .hword 0x9900,0x9901,0x9902,0x9903,0x9904,0x9905,0x9906,0x9907,0x9908,0x9909,0x990a,0x990b,0x990c,0x990d,0x990e,0x990f
    .hword 0x9920,0x9921,0x9922,0x9923,0x9924,0x9925,0x9926,0x9927,0x9928,0x9929,0x992a,0x992b,0x992c,0x992d,0x992e,0x992f
    .hword 0x9940,0x9941,0x9942,0x9943,0x9944,0x9945,0x9946,0x9947,0x9948,0x9949,0x994a,0x994b,0x994c,0x994d,0x994e,0x994f
    .hword 0x9960,0x9961,0x9962,0x9963,0x9964,0x9965,0x9966,0x9967,0x9968,0x9969,0x996a,0x996b,0x996c,0x996d,0x996e,0x996f
    .hword 0x9980,0x9981,0x9982,0x9983,0x9984,0x9985,0x9986,0x9987,0x9988,0x9989,0x998a,0x998b,0x998c,0x998d,0x998e,0x998f
    .hword 0x99a0,0x99a1,0x99a2,0x99a3,0x99a4,0x99a5,0x99a6,0x99a7,0x99a8,0x99a9,0x99aa,0x99ab,0x99ac,0x99ad,0x99ae,0x99af
    .hword 0x99c0,0x99c1,0x99c2,0x99c3,0x99c4,0x99c5,0x99c6,0x99c7,0x99c8,0x99c9,0x99ca,0x99cb,0x99cc,0x99cd,0x99ce,0x99cf
    .hword 0x99e0,0x99e1,0x99e2,0x99e3,0x99e4,0x99e5,0x99e6,0x99e7,0x99e8,0x99e9,0x99ea,0x99eb,0x99ec,0x99ed,0x99ee,0x99ef
    .hword 0x9a00,0x9a01,0x9a02,0x9a03,0x9a04,0x9a05,0x9a06,0x9a07,0x9a08,0x9a09,0x9a0a,0x9a0b,0x9a0c,0x9a0d,0x9a0e,0x9a0f
    .hword 0x9a20,0x9a21,0x9a22,0x9a23,0x9a24,0x9a25,0x9a26,0x9a27,0x9a28,0x9a29,0x9a2a,0x9a2b,0x9a2c,0x9a2d,0x9a2e,0x9a2f
    .hword 0x9a40,0x9a41,0x9a42,0x9a43,0x9a44,0x9a45,0x9a46,0x9a47,0x9a48,0x9a49,0x9a4a,0x9a4b,0x9a4c,0x9a4d,0x9a4e,0x9a4f
    .hword 0x9a60,0x9a61,0x9a62,0x9a63,0x9a64,0x9a65,0x9a66,0x9a67,0x9a68,0x9a69,0x9a6a,0x9a6b,0x9a6c,0x9a6d,0x9a6e,0x9a6f
    .hword 0x9a80,0x9a81,0x9a82,0x9a83,0x9a84,0x9a85,0x9a86,0x9a87,0x9a88,0x9a89,0x9a8a,0x9a8b,0x9a8c,0x9a8d,0x9a8e,0x9a8f
    .hword 0x9aa0,0x9aa1,0x9aa2,0x9aa3,0x9aa4,0x9aa5,0x9aa6,0x9aa7,0x9aa8,0x9aa9,0x9aaa,0x9aab,0x9aac,0x9aad,0x9aae,0x9aaf
    .hword 0x9ac0,0x9ac1,0x9ac2,0x9ac3,0x9ac4,0x9ac5,0x9ac6,0x9ac7,0x9ac8,0x9ac9,0x9aca,0x9acb,0x9acc,0x9acd,0x9ace,0x9acf
    .hword 0x9ae0,0x9ae1,0x9ae2,0x9ae3,0x9ae4,0x9ae5,0x9ae6,0x9ae7,0x9ae8,0x9ae9,0x9aea,0x9aeb,0x9aec,0x9aed,0x9aee,0x9aef
    .hword 0x9b00,0x9b01,0x9b02,0x9b03,0x9b04,0x9b05,0x9b06,0x9b07,0x9b08,0x9b09,0x9b0a,0x9b0b,0x9b0c,0x9b0d,0x9b0e,0x9b0f
    .hword 0x9b20,0x9b21,0x9b22,0x9b23,0x9b24,0x9b25,0x9b26,0x9b27,0x9b28,0x9b29,0x9b2a,0x9b2b,0x9b2c,0x9b2d,0x9b2e,0x9b2f
    .hword 0x9b40,0x9b41,0x9b42,0x9b43,0x9b44,0x9b45,0x9b46,0x9b47,0x9b48,0x9b49,0x9b4a,0x9b4b,0x9b4c,0x9b4d,0x9b4e,0x9b4f
    .hword 0x9b60,0x9b61,0x9b62,0x9b63,0x9b64,0x9b65,0x9b66,0x9b67,0x9b68,0x9b69,0x9b6a,0x9b6b,0x9b6c,0x9b6d,0x9b6e,0x9b6f
    .hword 0x9b80,0x9b81,0x9b82,0x9b83,0x9b84,0x9b85,0x9b86,0x9b87,0x9b88,0x9b89,0x9b8a,0x9b8b,0x9b8c,0x9b8d,0x9b8e,0x9b8f
    .hword 0x9ba0,0x9ba1,0x9ba2,0x9ba3,0x9ba4,0x9ba5,0x9ba6,0x9ba7,0x9ba8,0x9ba9,0x9baa,0x9bab,0x9bac,0x9bad,0x9bae,0x9baf
    .hword 0x9bc0,0x9bc1,0x9bc2,0x9bc3,0x9bc4,0x9bc5,0x9bc6,0x9bc7,0x9bc8,0x9bc9,0x9bca,0x9bcb,0x9bcc,0x9bcd,0x9bce,0x9bcf
    .hword 0x9be0,0x9be1,0x9be2,0x9be3,0x9be4,0x9be5,0x9be6,0x9be7,0x9be8,0x9be9,0x9bea,0x9beb,0x9bec,0x9bed,0x9bee,0x9bef
    .hword 0xa800,0xa801,0xa802,0xa803,0xa804,0xa805,0xa806,0xa807,0xa808,0xa809,0xa80a,0xa80b,0xa80c,0xa80d,0xa80e,0xa80f
    .hword 0xa820,0xa821,0xa822,0xa823,0xa824,0xa825,0xa826,0xa827,0xa828,0xa829,0xa82a,0xa82b,0xa82c,0xa82d,0xa82e,0xa82f
    .hword 0xa840,0xa841,0xa842,0xa843,0xa844,0xa845,0xa846,0xa847,0xa848,0xa849,0xa84a,0xa84b,0xa84c,0xa84d,0xa84e,0xa84f
    .hword 0xa860,0xa861,0xa862,0xa863,0xa864,0xa865,0xa866,0xa867,0xa868,0xa869,0xa86a,0xa86b,0xa86c,0xa86d,0xa86e,0xa86f
    .hword 0xa880,0xa881,0xa882,0xa883,0xa884,0xa885,0xa886,0xa887,0xa888,0xa889,0xa88a,0xa88b,0xa88c,0xa88d,0xa88e,0xa88f
    .hword 0xa8a0,0xa8a1,0xa8a2,0xa8a3,0xa8a4,0xa8a5,0xa8a6,0xa8a7,0xa8a8,0xa8a9,0xa8aa,0xa8ab,0xa8ac,0xa8ad,0xa8ae,0xa8af
    .hword 0xa8c0,0xa8c1,0xa8c2,0xa8c3,0xa8c4,0xa8c5,0xa8c6,0xa8c7,0xa8c8,0xa8c9,0xa8ca,0xa8cb,0xa8cc,0xa8cd,0xa8ce,0xa8cf
    .hword 0xa8e0,0xa8e1,0xa8e2,0xa8e3,0xa8e4,0xa8e5,0xa8e6,0xa8e7,0xa8e8,0xa8e9,0xa8ea,0xa8eb,0xa8ec,0xa8ed,0xa8ee,0xa8ef
    .hword 0xa900,0xa901,0xa902,0xa903,0xa904,0xa905,0xa906,0xa907,0xa908,0xa909,0xa90a,0xa90b,0xa90c,0xa90d,0xa90e,0xa90f
    .hword 0xa920,0xa921,0xa922,0xa923,0xa924,0xa925,0xa926,0xa927,0xa928,0xa929,0xa92a,0xa92b,0xa92c,0xa92d,0xa92e,0xa92f
    .hword 0xa940,0xa941,0xa942,0xa943,0xa944,0xa945,0xa946,0xa947,0xa948,0xa949,0xa94a,0xa94b,0xa94c,0xa94d,0xa94e,0xa94f
    .hword 0xa960,0xa961,0xa962,0xa963,0xa964,0xa965,0xa966,0xa967,0xa968,0xa969,0xa96a,0xa96b,0xa96c,0xa96d,0xa96e,0xa96f
    .hword 0xa980,0xa981,0xa982,0xa983,0xa984,0xa985,0xa986,0xa987,0xa988,0xa989,0xa98a,0xa98b,0xa98c,0xa98d,0xa98e,0xa98f
    .hword 0xa9a0,0xa9a1,0xa9a2,0xa9a3,0xa9a4,0xa9a5,0xa9a6,0xa9a7,0xa9a8,0xa9a9,0xa9aa,0xa9ab,0xa9ac,0xa9ad,0xa9ae,0xa9af
    .hword 0xa9c0,0xa9c1,0xa9c2,0xa9c3,0xa9c4,0xa9c5,0xa9c6,0xa9c7,0xa9c8,0xa9c9,0xa9ca,0xa9cb,0xa9cc,0xa9cd,0xa9ce,0xa9cf
    .hword 0xa9e0,0xa9e1,0xa9e2,0xa9e3,0xa9e4,0xa9e5,0xa9e6,0xa9e7,0xa9e8,0xa9e9,0xa9ea,0xa9eb,0xa9ec,0xa9ed,0xa9ee,0xa9ef
    .hword 0xaa00,0xaa01,0xaa02,0xaa03,0xaa04,0xaa05,0xaa06,0xaa07,0xaa08,0xaa09,0xaa0a,0xaa0b,0xaa0c,0xaa0d,0xaa0e,0xaa0f
    .hword 0xaa20,0xaa21,0xaa22,0xaa23,0xaa24,0xaa25,0xaa26,0xaa27,0xaa28,0xaa29,0xaa2a,0xaa2b,0xaa2c,0xaa2d,0xaa2e,0xaa2f
    .hword 0xaa40,0xaa41,0xaa42,0xaa43,0xaa44,0xaa45,0xaa46,0xaa47,0xaa48,0xaa49,0xaa4a,0xaa4b,0xaa4c,0xaa4d,0xaa4e,0xaa4f
    .hword 0xaa60,0xaa61,0xaa62,0xaa63,0xaa64,0xaa65,0xaa66,0xaa67,0xaa68,0xaa69,0xaa6a,0xaa6b,0xaa6c,0xaa6d,0xaa6e,0xaa6f
    .hword 0xaa80,0xaa81,0xaa82,0xaa83,0xaa84,0xaa85,0xaa86,0xaa87,0xaa88,0xaa89,0xaa8a,0xaa8b,0xaa8c,0xaa8d,0xaa8e,0xaa8f
    .hword 0xaaa0,0xaaa1,0xaaa2,0xaaa3,0xaaa4,0xaaa5,0xaaa6,0xaaa7,0xaaa8,0xaaa9,0xaaaa,0xaaab,0xaaac,0xaaad,0xaaae,0xaaaf
    .hword 0xaac0,0xaac1,0xaac2,0xaac3,0xaac4,0xaac5,0xaac6,0xaac7,0xaac8,0xaac9,0xaaca,0xaacb,0xaacc,0xaacd,0xaace,0xaacf
    .hword 0xaae0,0xaae1,0xaae2,0xaae3,0xaae4,0xaae5,0xaae6,0xaae7,0xaae8,0xaae9,0xaaea,0xaaeb,0xaaec,0xaaed,0xaaee,0xaaef
    .hword 0xab00,0xab01,0xab02,0xab03,0xab04,0xab05,0xab06,0xab07,0xab08,0xab09,0xab0a,0xab0b,0xab0c,0xab0d,0xab0e,0xab0f
    .hword 0xab20,0xab21,0xab22,0xab23,0xab24,0xab25,0xab26,0xab27,0xab28,0xab29,0xab2a,0xab2b,0xab2c,0xab2d,0xab2e,0xab2f
    .hword 0xab40,0xab41,0xab42,0xab43,0xab44,0xab45,0xab46,0xab47,0xab48,0xab49,0xab4a,0xab4b,0xab4c,0xab4d,0xab4e,0xab4f
    .hword 0xab60,0xab61,0xab62,0xab63,0xab64,0xab65,0xab66,0xab67,0xab68,0xab69,0xab6a,0xab6b,0xab6c,0xab6d,0xab6e,0xab6f
    .hword 0xab80,0xab81,0xab82,0xab83,0xab84,0xab85,0xab86,0xab87,0xab88,0xab89,0xab8a,0xab8b,0xab8c,0xab8d,0xab8e,0xab8f
    .hword 0xaba0,0xaba1,0xaba2,0xaba3,0xaba4,0xaba5,0xaba6,0xaba7,0xaba8,0xaba9,0xabaa,0xabab,0xabac,0xabad,0xabae,0xabaf
    .hword 0xabc0,0xabc1,0xabc2,0xabc3,0xabc4,0xabc5,0xabc6,0xabc7,0xabc8,0xabc9,0xabca,0xabcb,0xabcc,0xabcd,0xabce,0xabcf
    .hword 0xabe0,0xabe1,0xabe2,0xabe3,0xabe4,0xabe5,0xabe6,0xabe7,0xabe8,0xabe9,0xabea,0xabeb,0xabec,0xabed,0xabee,0xabef
    .hword 0xb800,0xb801,0xb802,0xb803,0xb804,0xb805,0xb806,0xb807,0xb808,0xb809,0xb80a,0xb80b,0xb80c,0xb80d,0xb80e,0xb80f
    .hword 0xb820,0xb821,0xb822,0xb823,0xb824,0xb825,0xb826,0xb827,0xb828,0xb829,0xb82a,0xb82b,0xb82c,0xb82d,0xb82e,0xb82f
    .hword 0xb840,0xb841,0xb842,0xb843,0xb844,0xb845,0xb846,0xb847,0xb848,0xb849,0xb84a,0xb84b,0xb84c,0xb84d,0xb84e,0xb84f
    .hword 0xb860,0xb861,0xb862,0xb863,0xb864,0xb865,0xb866,0xb867,0xb868,0xb869,0xb86a,0xb86b,0xb86c,0xb86d,0xb86e,0xb86f
    .hword 0xb880,0xb881,0xb882,0xb883,0xb884,0xb885,0xb886,0xb887,0xb888,0xb889,0xb88a,0xb88b,0xb88c,0xb88d,0xb88e,0xb88f
    .hword 0xb8a0,0xb8a1,0xb8a2,0xb8a3,0xb8a4,0xb8a5,0xb8a6,0xb8a7,0xb8a8,0xb8a9,0xb8aa,0xb8ab,0xb8ac,0xb8ad,0xb8ae,0xb8af
    .hword 0xb8c0,0xb8c1,0xb8c2,0xb8c3,0xb8c4,0xb8c5,0xb8c6,0xb8c7,0xb8c8,0xb8c9,0xb8ca,0xb8cb,0xb8cc,0xb8cd,0xb8ce,0xb8cf
    .hword 0xb8e0,0xb8e1,0xb8e2,0xb8e3,0xb8e4,0xb8e5,0xb8e6,0xb8e7,0xb8e8,0xb8e9,0xb8ea,0xb8eb,0xb8ec,0xb8ed,0xb8ee,0xb8ef
    .hword 0xb900,0xb901,0xb902,0xb903,0xb904,0xb905,0xb906,0xb907,0xb908,0xb909,0xb90a,0xb90b,0xb90c,0xb90d,0xb90e,0xb90f
    .hword 0xb920,0xb921,0xb922,0xb923,0xb924,0xb925,0xb926,0xb927,0xb928,0xb929,0xb92a,0xb92b,0xb92c,0xb92d,0xb92e,0xb92f
    .hword 0xb940,0xb941,0xb942,0xb943,0xb944,0xb945,0xb946,0xb947,0xb948,0xb949,0xb94a,0xb94b,0xb94c,0xb94d,0xb94e,0xb94f
    .hword 0xb960,0xb961,0xb962,0xb963,0xb964,0xb965,0xb966,0xb967,0xb968,0xb969,0xb96a,0xb96b,0xb96c,0xb96d,0xb96e,0xb96f
    .hword 0xb980,0xb981,0xb982,0xb983,0xb984,0xb985,0xb986,0xb987,0xb988,0xb989,0xb98a,0xb98b,0xb98c,0xb98d,0xb98e,0xb98f
    .hword 0xb9a0,0xb9a1,0xb9a2,0xb9a3,0xb9a4,0xb9a5,0xb9a6,0xb9a7,0xb9a8,0xb9a9,0xb9aa,0xb9ab,0xb9ac,0xb9ad,0xb9ae,0xb9af
    .hword 0xb9c0,0xb9c1,0xb9c2,0xb9c3,0xb9c4,0xb9c5,0xb9c6,0xb9c7,0xb9c8,0xb9c9,0xb9ca,0xb9cb,0xb9cc,0xb9cd,0xb9ce,0xb9cf
    .hword 0xb9e0,0xb9e1,0xb9e2,0xb9e3,0xb9e4,0xb9e5,0xb9e6,0xb9e7,0xb9e8,0xb9e9,0xb9ea,0xb9eb,0xb9ec,0xb9ed,0xb9ee,0xb9ef
    .hword 0xba00,0xba01,0xba02,0xba03,0xba04,0xba05,0xba06,0xba07,0xba08,0xba09,0xba0a,0xba0b,0xba0c,0xba0d,0xba0e,0xba0f
    .hword 0xba20,0xba21,0xba22,0xba23,0xba24,0xba25,0xba26,0xba27,0xba28,0xba29,0xba2a,0xba2b,0xba2c,0xba2d,0xba2e,0xba2f
    .hword 0xba40,0xba41,0xba42,0xba43,0xba44,0xba45,0xba46,0xba47,0xba48,0xba49,0xba4a,0xba4b,0xba4c,0xba4d,0xba4e,0xba4f
    .hword 0xba60,0xba61,0xba62,0xba63,0xba64,0xba65,0xba66,0xba67,0xba68,0xba69,0xba6a,0xba6b,0xba6c,0xba6d,0xba6e,0xba6f
    .hword 0xba80,0xba81,0xba82,0xba83,0xba84,0xba85,0xba86,0xba87,0xba88,0xba89,0xba8a,0xba8b,0xba8c,0xba8d,0xba8e,0xba8f
    .hword 0xbaa0,0xbaa1,0xbaa2,0xbaa3,0xbaa4,0xbaa5,0xbaa6,0xbaa7,0xbaa8,0xbaa9,0xbaaa,0xbaab,0xbaac,0xbaad,0xbaae,0xbaaf
    .hword 0xbac0,0xbac1,0xbac2,0xbac3,0xbac4,0xbac5,0xbac6,0xbac7,0xbac8,0xbac9,0xbaca,0xbacb,0xbacc,0xbacd,0xbace,0xbacf
    .hword 0xbae0,0xbae1,0xbae2,0xbae3,0xbae4,0xbae5,0xbae6,0xbae7,0xbae8,0xbae9,0xbaea,0xbaeb,0xbaec,0xbaed,0xbaee,0xbaef
    .hword 0xbb00,0xbb01,0xbb02,0xbb03,0xbb04,0xbb05,0xbb06,0xbb07,0xbb08,0xbb09,0xbb0a,0xbb0b,0xbb0c,0xbb0d,0xbb0e,0xbb0f
    .hword 0xbb20,0xbb21,0xbb22,0xbb23,0xbb24,0xbb25,0xbb26,0xbb27,0xbb28,0xbb29,0xbb2a,0xbb2b,0xbb2c,0xbb2d,0xbb2e,0xbb2f
    .hword 0xbb40,0xbb41,0xbb42,0xbb43,0xbb44,0xbb45,0xbb46,0xbb47,0xbb48,0xbb49,0xbb4a,0xbb4b,0xbb4c,0xbb4d,0xbb4e,0xbb4f
    .hword 0xbb60,0xbb61,0xbb62,0xbb63,0xbb64,0xbb65,0xbb66,0xbb67,0xbb68,0xbb69,0xbb6a,0xbb6b,0xbb6c,0xbb6d,0xbb6e,0xbb6f
    .hword 0xbb80,0xbb81,0xbb82,0xbb83,0xbb84,0xbb85,0xbb86,0xbb87,0xbb88,0xbb89,0xbb8a,0xbb8b,0xbb8c,0xbb8d,0xbb8e,0xbb8f
    .hword 0xbba0,0xbba1,0xbba2,0xbba3,0xbba4,0xbba5,0xbba6,0xbba7,0xbba8,0xbba9,0xbbaa,0xbbab,0xbbac,0xbbad,0xbbae,0xbbaf
    .hword 0xbbc0,0xbbc1,0xbbc2,0xbbc3,0xbbc4,0xbbc5,0xbbc6,0xbbc7,0xbbc8,0xbbc9,0xbbca,0xbbcb,0xbbcc,0xbbcd,0xbbce,0xbbcf
    .hword 0xbbe0,0xbbe1,0xbbe2,0xbbe3,0xbbe4,0xbbe5,0xbbe6,0xbbe7,0xbbe8,0xbbe9,0xbbea,0xbbeb,0xbbec,0xbbed,0xbbee,0xbbef
    .hword 0xc800,0xc801,0xc802,0xc803,0xc804,0xc805,0xc806,0xc807,0xc808,0xc809,0xc80a,0xc80b,0xc80c,0xc80d,0xc80e,0xc80f
    .hword 0xc820,0xc821,0xc822,0xc823,0xc824,0xc825,0xc826,0xc827,0xc828,0xc829,0xc82a,0xc82b,0xc82c,0xc82d,0xc82e,0xc82f
    .hword 0xc840,0xc841,0xc842,0xc843,0xc844,0xc845,0xc846,0xc847,0xc848,0xc849,0xc84a,0xc84b,0xc84c,0xc84d,0xc84e,0xc84f
    .hword 0xc860,0xc861,0xc862,0xc863,0xc864,0xc865,0xc866,0xc867,0xc868,0xc869,0xc86a,0xc86b,0xc86c,0xc86d,0xc86e,0xc86f
    .hword 0xc880,0xc881,0xc882,0xc883,0xc884,0xc885,0xc886,0xc887,0xc888,0xc889,0xc88a,0xc88b,0xc88c,0xc88d,0xc88e,0xc88f
    .hword 0xc8a0,0xc8a1,0xc8a2,0xc8a3,0xc8a4,0xc8a5,0xc8a6,0xc8a7,0xc8a8,0xc8a9,0xc8aa,0xc8ab,0xc8ac,0xc8ad,0xc8ae,0xc8af
    .hword 0xc8c0,0xc8c1,0xc8c2,0xc8c3,0xc8c4,0xc8c5,0xc8c6,0xc8c7,0xc8c8,0xc8c9,0xc8ca,0xc8cb,0xc8cc,0xc8cd,0xc8ce,0xc8cf
    .hword 0xc8e0,0xc8e1,0xc8e2,0xc8e3,0xc8e4,0xc8e5,0xc8e6,0xc8e7,0xc8e8,0xc8e9,0xc8ea,0xc8eb,0xc8ec,0xc8ed,0xc8ee,0xc8ef
    .hword 0xc900,0xc901,0xc902,0xc903,0xc904,0xc905,0xc906,0xc907,0xc908,0xc909,0xc90a,0xc90b,0xc90c,0xc90d,0xc90e,0xc90f
    .hword 0xc920,0xc921,0xc922,0xc923,0xc924,0xc925,0xc926,0xc927,0xc928,0xc929,0xc92a,0xc92b,0xc92c,0xc92d,0xc92e,0xc92f
    .hword 0xc940,0xc941,0xc942,0xc943,0xc944,0xc945,0xc946,0xc947,0xc948,0xc949,0xc94a,0xc94b,0xc94c,0xc94d,0xc94e,0xc94f
    .hword 0xc960,0xc961,0xc962,0xc963,0xc964,0xc965,0xc966,0xc967,0xc968,0xc969,0xc96a,0xc96b,0xc96c,0xc96d,0xc96e,0xc96f
    .hword 0xc980,0xc981,0xc982,0xc983,0xc984,0xc985,0xc986,0xc987,0xc988,0xc989,0xc98a,0xc98b,0xc98c,0xc98d,0xc98e,0xc98f
    .hword 0xc9a0,0xc9a1,0xc9a2,0xc9a3,0xc9a4,0xc9a5,0xc9a6,0xc9a7,0xc9a8,0xc9a9,0xc9aa,0xc9ab,0xc9ac,0xc9ad,0xc9ae,0xc9af
    .hword 0xc9c0,0xc9c1,0xc9c2,0xc9c3,0xc9c4,0xc9c5,0xc9c6,0xc9c7,0xc9c8,0xc9c9,0xc9ca,0xc9cb,0xc9cc,0xc9cd,0xc9ce,0xc9cf
    .hword 0xc9e0,0xc9e1,0xc9e2,0xc9e3,0xc9e4,0xc9e5,0xc9e6,0xc9e7,0xc9e8,0xc9e9,0xc9ea,0xc9eb,0xc9ec,0xc9ed,0xc9ee,0xc9ef
    .hword 0xca00,0xca01,0xca02,0xca03,0xca04,0xca05,0xca06,0xca07,0xca08,0xca09,0xca0a,0xca0b,0xca0c,0xca0d,0xca0e,0xca0f
    .hword 0xca20,0xca21,0xca22,0xca23,0xca24,0xca25,0xca26,0xca27,0xca28,0xca29,0xca2a,0xca2b,0xca2c,0xca2d,0xca2e,0xca2f
    .hword 0xca40,0xca41,0xca42,0xca43,0xca44,0xca45,0xca46,0xca47,0xca48,0xca49,0xca4a,0xca4b,0xca4c,0xca4d,0xca4e,0xca4f
    .hword 0xca60,0xca61,0xca62,0xca63,0xca64,0xca65,0xca66,0xca67,0xca68,0xca69,0xca6a,0xca6b,0xca6c,0xca6d,0xca6e,0xca6f
    .hword 0xca80,0xca81,0xca82,0xca83,0xca84,0xca85,0xca86,0xca87,0xca88,0xca89,0xca8a,0xca8b,0xca8c,0xca8d,0xca8e,0xca8f
    .hword 0xcaa0,0xcaa1,0xcaa2,0xcaa3,0xcaa4,0xcaa5,0xcaa6,0xcaa7,0xcaa8,0xcaa9,0xcaaa,0xcaab,0xcaac,0xcaad,0xcaae,0xcaaf
    .hword 0xcac0,0xcac1,0xcac2,0xcac3,0xcac4,0xcac5,0xcac6,0xcac7,0xcac8,0xcac9,0xcaca,0xcacb,0xcacc,0xcacd,0xcace,0xcacf
    .hword 0xcae0,0xcae1,0xcae2,0xcae3,0xcae4,0xcae5,0xcae6,0xcae7,0xcae8,0xcae9,0xcaea,0xcaeb,0xcaec,0xcaed,0xcaee,0xcaef
    .hword 0xcb00,0xcb01,0xcb02,0xcb03,0xcb04,0xcb05,0xcb06,0xcb07,0xcb08,0xcb09,0xcb0a,0xcb0b,0xcb0c,0xcb0d,0xcb0e,0xcb0f
    .hword 0xcb20,0xcb21,0xcb22,0xcb23,0xcb24,0xcb25,0xcb26,0xcb27,0xcb28,0xcb29,0xcb2a,0xcb2b,0xcb2c,0xcb2d,0xcb2e,0xcb2f
    .hword 0xcb40,0xcb41,0xcb42,0xcb43,0xcb44,0xcb45,0xcb46,0xcb47,0xcb48,0xcb49,0xcb4a,0xcb4b,0xcb4c,0xcb4d,0xcb4e,0xcb4f
    .hword 0xcb60,0xcb61,0xcb62,0xcb63,0xcb64,0xcb65,0xcb66,0xcb67,0xcb68,0xcb69,0xcb6a,0xcb6b,0xcb6c,0xcb6d,0xcb6e,0xcb6f
    .hword 0xcb80,0xcb81,0xcb82,0xcb83,0xcb84,0xcb85,0xcb86,0xcb87,0xcb88,0xcb89,0xcb8a,0xcb8b,0xcb8c,0xcb8d,0xcb8e,0xcb8f
    .hword 0xcba0,0xcba1,0xcba2,0xcba3,0xcba4,0xcba5,0xcba6,0xcba7,0xcba8,0xcba9,0xcbaa,0xcbab,0xcbac,0xcbad,0xcbae,0xcbaf
    .hword 0xcbc0,0xcbc1,0xcbc2,0xcbc3,0xcbc4,0xcbc5,0xcbc6,0xcbc7,0xcbc8,0xcbc9,0xcbca,0xcbcb,0xcbcc,0xcbcd,0xcbce,0xcbcf
    .hword 0xcbe0,0xcbe1,0xcbe2,0xcbe3,0xcbe4,0xcbe5,0xcbe6,0xcbe7,0xcbe8,0xcbe9,0xcbea,0xcbeb,0xcbec,0xcbed,0xcbee,0xcbef
    .hword 0xd800,0xd801,0xd802,0xd803,0xd804,0xd805,0xd806,0xd807,0xd808,0xd809,0xd80a,0xd80b,0xd80c,0xd80d,0xd80e,0xd80f
    .hword 0xd820,0xd821,0xd822,0xd823,0xd824,0xd825,0xd826,0xd827,0xd828,0xd829,0xd82a,0xd82b,0xd82c,0xd82d,0xd82e,0xd82f
    .hword 0xd840,0xd841,0xd842,0xd843,0xd844,0xd845,0xd846,0xd847,0xd848,0xd849,0xd84a,0xd84b,0xd84c,0xd84d,0xd84e,0xd84f
    .hword 0xd860,0xd861,0xd862,0xd863,0xd864,0xd865,0xd866,0xd867,0xd868,0xd869,0xd86a,0xd86b,0xd86c,0xd86d,0xd86e,0xd86f
    .hword 0xd880,0xd881,0xd882,0xd883,0xd884,0xd885,0xd886,0xd887,0xd888,0xd889,0xd88a,0xd88b,0xd88c,0xd88d,0xd88e,0xd88f
    .hword 0xd8a0,0xd8a1,0xd8a2,0xd8a3,0xd8a4,0xd8a5,0xd8a6,0xd8a7,0xd8a8,0xd8a9,0xd8aa,0xd8ab,0xd8ac,0xd8ad,0xd8ae,0xd8af
    .hword 0xd8c0,0xd8c1,0xd8c2,0xd8c3,0xd8c4,0xd8c5,0xd8c6,0xd8c7,0xd8c8,0xd8c9,0xd8ca,0xd8cb,0xd8cc,0xd8cd,0xd8ce,0xd8cf
    .hword 0xd8e0,0xd8e1,0xd8e2,0xd8e3,0xd8e4,0xd8e5,0xd8e6,0xd8e7,0xd8e8,0xd8e9,0xd8ea,0xd8eb,0xd8ec,0xd8ed,0xd8ee,0xd8ef
    .hword 0xd900,0xd901,0xd902,0xd903,0xd904,0xd905,0xd906,0xd907,0xd908,0xd909,0xd90a,0xd90b,0xd90c,0xd90d,0xd90e,0xd90f
    .hword 0xd920,0xd921,0xd922,0xd923,0xd924,0xd925,0xd926,0xd927,0xd928,0xd929,0xd92a,0xd92b,0xd92c,0xd92d,0xd92e,0xd92f
    .hword 0xd940,0xd941,0xd942,0xd943,0xd944,0xd945,0xd946,0xd947,0xd948,0xd949,0xd94a,0xd94b,0xd94c,0xd94d,0xd94e,0xd94f
    .hword 0xd960,0xd961,0xd962,0xd963,0xd964,0xd965,0xd966,0xd967,0xd968,0xd969,0xd96a,0xd96b,0xd96c,0xd96d,0xd96e,0xd96f
    .hword 0xd980,0xd981,0xd982,0xd983,0xd984,0xd985,0xd986,0xd987,0xd988,0xd989,0xd98a,0xd98b,0xd98c,0xd98d,0xd98e,0xd98f
    .hword 0xd9a0,0xd9a1,0xd9a2,0xd9a3,0xd9a4,0xd9a5,0xd9a6,0xd9a7,0xd9a8,0xd9a9,0xd9aa,0xd9ab,0xd9ac,0xd9ad,0xd9ae,0xd9af
    .hword 0xd9c0,0xd9c1,0xd9c2,0xd9c3,0xd9c4,0xd9c5,0xd9c6,0xd9c7,0xd9c8,0xd9c9,0xd9ca,0xd9cb,0xd9cc,0xd9cd,0xd9ce,0xd9cf
    .hword 0xd9e0,0xd9e1,0xd9e2,0xd9e3,0xd9e4,0xd9e5,0xd9e6,0xd9e7,0xd9e8,0xd9e9,0xd9ea,0xd9eb,0xd9ec,0xd9ed,0xd9ee,0xd9ef
    .hword 0xda00,0xda01,0xda02,0xda03,0xda04,0xda05,0xda06,0xda07,0xda08,0xda09,0xda0a,0xda0b,0xda0c,0xda0d,0xda0e,0xda0f
    .hword 0xda20,0xda21,0xda22,0xda23,0xda24,0xda25,0xda26,0xda27,0xda28,0xda29,0xda2a,0xda2b,0xda2c,0xda2d,0xda2e,0xda2f
    .hword 0xda40,0xda41,0xda42,0xda43,0xda44,0xda45,0xda46,0xda47,0xda48,0xda49,0xda4a,0xda4b,0xda4c,0xda4d,0xda4e,0xda4f
    .hword 0xda60,0xda61,0xda62,0xda63,0xda64,0xda65,0xda66,0xda67,0xda68,0xda69,0xda6a,0xda6b,0xda6c,0xda6d,0xda6e,0xda6f
    .hword 0xda80,0xda81,0xda82,0xda83,0xda84,0xda85,0xda86,0xda87,0xda88,0xda89,0xda8a,0xda8b,0xda8c,0xda8d,0xda8e,0xda8f
    .hword 0xdaa0,0xdaa1,0xdaa2,0xdaa3,0xdaa4,0xdaa5,0xdaa6,0xdaa7,0xdaa8,0xdaa9,0xdaaa,0xdaab,0xdaac,0xdaad,0xdaae,0xdaaf
    .hword 0xdac0,0xdac1,0xdac2,0xdac3,0xdac4,0xdac5,0xdac6,0xdac7,0xdac8,0xdac9,0xdaca,0xdacb,0xdacc,0xdacd,0xdace,0xdacf
    .hword 0xdae0,0xdae1,0xdae2,0xdae3,0xdae4,0xdae5,0xdae6,0xdae7,0xdae8,0xdae9,0xdaea,0xdaeb,0xdaec,0xdaed,0xdaee,0xdaef
    .hword 0xdb00,0xdb01,0xdb02,0xdb03,0xdb04,0xdb05,0xdb06,0xdb07,0xdb08,0xdb09,0xdb0a,0xdb0b,0xdb0c,0xdb0d,0xdb0e,0xdb0f
    .hword 0xdb20,0xdb21,0xdb22,0xdb23,0xdb24,0xdb25,0xdb26,0xdb27,0xdb28,0xdb29,0xdb2a,0xdb2b,0xdb2c,0xdb2d,0xdb2e,0xdb2f
    .hword 0xdb40,0xdb41,0xdb42,0xdb43,0xdb44,0xdb45,0xdb46,0xdb47,0xdb48,0xdb49,0xdb4a,0xdb4b,0xdb4c,0xdb4d,0xdb4e,0xdb4f
    .hword 0xdb60,0xdb61,0xdb62,0xdb63,0xdb64,0xdb65,0xdb66,0xdb67,0xdb68,0xdb69,0xdb6a,0xdb6b,0xdb6c,0xdb6d,0xdb6e,0xdb6f
    .hword 0xdb80,0xdb81,0xdb82,0xdb83,0xdb84,0xdb85,0xdb86,0xdb87,0xdb88,0xdb89,0xdb8a,0xdb8b,0xdb8c,0xdb8d,0xdb8e,0xdb8f
    .hword 0xdba0,0xdba1,0xdba2,0xdba3,0xdba4,0xdba5,0xdba6,0xdba7,0xdba8,0xdba9,0xdbaa,0xdbab,0xdbac,0xdbad,0xdbae,0xdbaf
    .hword 0xdbc0,0xdbc1,0xdbc2,0xdbc3,0xdbc4,0xdbc5,0xdbc6,0xdbc7,0xdbc8,0xdbc9,0xdbca,0xdbcb,0xdbcc,0xdbcd,0xdbce,0xdbcf
    .hword 0xdbe0,0xdbe1,0xdbe2,0xdbe3,0xdbe4,0xdbe5,0xdbe6,0xdbe7,0xdbe8,0xdbe9,0xdbea,0xdbeb,0xdbec,0xdbed,0xdbee,0xdbef
    .hword 0xe800,0xe801,0xe802,0xe803,0xe804,0xe805,0xe806,0xe807,0xe808,0xe809,0xe80a,0xe80b,0xe80c,0xe80d,0xe80e,0xe80f
    .hword 0xe820,0xe821,0xe822,0xe823,0xe824,0xe825,0xe826,0xe827,0xe828,0xe829,0xe82a,0xe82b,0xe82c,0xe82d,0xe82e,0xe82f
    .hword 0xe840,0xe841,0xe842,0xe843,0xe844,0xe845,0xe846,0xe847,0xe848,0xe849,0xe84a,0xe84b,0xe84c,0xe84d,0xe84e,0xe84f
    .hword 0xe860,0xe861,0xe862,0xe863,0xe864,0xe865,0xe866,0xe867,0xe868,0xe869,0xe86a,0xe86b,0xe86c,0xe86d,0xe86e,0xe86f
    .hword 0xe880,0xe881,0xe882,0xe883,0xe884,0xe885,0xe886,0xe887,0xe888,0xe889,0xe88a,0xe88b,0xe88c,0xe88d,0xe88e,0xe88f
    .hword 0xe8a0,0xe8a1,0xe8a2,0xe8a3,0xe8a4,0xe8a5,0xe8a6,0xe8a7,0xe8a8,0xe8a9,0xe8aa,0xe8ab,0xe8ac,0xe8ad,0xe8ae,0xe8af
    .hword 0xe8c0,0xe8c1,0xe8c2,0xe8c3,0xe8c4,0xe8c5,0xe8c6,0xe8c7,0xe8c8,0xe8c9,0xe8ca,0xe8cb,0xe8cc,0xe8cd,0xe8ce,0xe8cf
    .hword 0xe8e0,0xe8e1,0xe8e2,0xe8e3,0xe8e4,0xe8e5,0xe8e6,0xe8e7,0xe8e8,0xe8e9,0xe8ea,0xe8eb,0xe8ec,0xe8ed,0xe8ee,0xe8ef
    .hword 0xe900,0xe901,0xe902,0xe903,0xe904,0xe905,0xe906,0xe907,0xe908,0xe909,0xe90a,0xe90b,0xe90c,0xe90d,0xe90e,0xe90f
    .hword 0xe920,0xe921,0xe922,0xe923,0xe924,0xe925,0xe926,0xe927,0xe928,0xe929,0xe92a,0xe92b,0xe92c,0xe92d,0xe92e,0xe92f
    .hword 0xe940,0xe941,0xe942,0xe943,0xe944,0xe945,0xe946,0xe947,0xe948,0xe949,0xe94a,0xe94b,0xe94c,0xe94d,0xe94e,0xe94f
    .hword 0xe960,0xe961,0xe962,0xe963,0xe964,0xe965,0xe966,0xe967,0xe968,0xe969,0xe96a,0xe96b,0xe96c,0xe96d,0xe96e,0xe96f
    .hword 0xe980,0xe981,0xe982,0xe983,0xe984,0xe985,0xe986,0xe987,0xe988,0xe989,0xe98a,0xe98b,0xe98c,0xe98d,0xe98e,0xe98f
    .hword 0xe9a0,0xe9a1,0xe9a2,0xe9a3,0xe9a4,0xe9a5,0xe9a6,0xe9a7,0xe9a8,0xe9a9,0xe9aa,0xe9ab,0xe9ac,0xe9ad,0xe9ae,0xe9af
    .hword 0xe9c0,0xe9c1,0xe9c2,0xe9c3,0xe9c4,0xe9c5,0xe9c6,0xe9c7,0xe9c8,0xe9c9,0xe9ca,0xe9cb,0xe9cc,0xe9cd,0xe9ce,0xe9cf
    .hword 0xe9e0,0xe9e1,0xe9e2,0xe9e3,0xe9e4,0xe9e5,0xe9e6,0xe9e7,0xe9e8,0xe9e9,0xe9ea,0xe9eb,0xe9ec,0xe9ed,0xe9ee,0xe9ef
    .hword 0xea00,0xea01,0xea02,0xea03,0xea04,0xea05,0xea06,0xea07,0xea08,0xea09,0xea0a,0xea0b,0xea0c,0xea0d,0xea0e,0xea0f
    .hword 0xea20,0xea21,0xea22,0xea23,0xea24,0xea25,0xea26,0xea27,0xea28,0xea29,0xea2a,0xea2b,0xea2c,0xea2d,0xea2e,0xea2f
    .hword 0xea40,0xea41,0xea42,0xea43,0xea44,0xea45,0xea46,0xea47,0xea48,0xea49,0xea4a,0xea4b,0xea4c,0xea4d,0xea4e,0xea4f
    .hword 0xea60,0xea61,0xea62,0xea63,0xea64,0xea65,0xea66,0xea67,0xea68,0xea69,0xea6a,0xea6b,0xea6c,0xea6d,0xea6e,0xea6f
    .hword 0xea80,0xea81,0xea82,0xea83,0xea84,0xea85,0xea86,0xea87,0xea88,0xea89,0xea8a,0xea8b,0xea8c,0xea8d,0xea8e,0xea8f
    .hword 0xeaa0,0xeaa1,0xeaa2,0xeaa3,0xeaa4,0xeaa5,0xeaa6,0xeaa7,0xeaa8,0xeaa9,0xeaaa,0xeaab,0xeaac,0xeaad,0xeaae,0xeaaf
    .hword 0xeac0,0xeac1,0xeac2,0xeac3,0xeac4,0xeac5,0xeac6,0xeac7,0xeac8,0xeac9,0xeaca,0xeacb,0xeacc,0xeacd,0xeace,0xeacf
    .hword 0xeae0,0xeae1,0xeae2,0xeae3,0xeae4,0xeae5,0xeae6,0xeae7,0xeae8,0xeae9,0xeaea,0xeaeb,0xeaec,0xeaed,0xeaee,0xeaef
    .hword 0xeb00,0xeb01,0xeb02,0xeb03,0xeb04,0xeb05,0xeb06,0xeb07,0xeb08,0xeb09,0xeb0a,0xeb0b,0xeb0c,0xeb0d,0xeb0e,0xeb0f
    .hword 0xeb20,0xeb21,0xeb22,0xeb23,0xeb24,0xeb25,0xeb26,0xeb27,0xeb28,0xeb29,0xeb2a,0xeb2b,0xeb2c,0xeb2d,0xeb2e,0xeb2f
    .hword 0xeb40,0xeb41,0xeb42,0xeb43,0xeb44,0xeb45,0xeb46,0xeb47,0xeb48,0xeb49,0xeb4a,0xeb4b,0xeb4c,0xeb4d,0xeb4e,0xeb4f
    .hword 0xeb60,0xeb61,0xeb62,0xeb63,0xeb64,0xeb65,0xeb66,0xeb67,0xeb68,0xeb69,0xeb6a,0xeb6b,0xeb6c,0xeb6d,0xeb6e,0xeb6f
    .hword 0xeb80,0xeb81,0xeb82,0xeb83,0xeb84,0xeb85,0xeb86,0xeb87,0xeb88,0xeb89,0xeb8a,0xeb8b,0xeb8c,0xeb8d,0xeb8e,0xeb8f
    .hword 0xeba0,0xeba1,0xeba2,0xeba3,0xeba4,0xeba5,0xeba6,0xeba7,0xeba8,0xeba9,0xebaa,0xebab,0xebac,0xebad,0xebae,0xebaf
    .hword 0xebc0,0xebc1,0xebc2,0xebc3,0xebc4,0xebc5,0xebc6,0xebc7,0xebc8,0xebc9,0xebca,0xebcb,0xebcc,0xebcd,0xebce,0xebcf
    .hword 0xebe0,0xebe1,0xebe2,0xebe3,0xebe4,0xebe5,0xebe6,0xebe7,0xebe8,0xebe9,0xebea,0xebeb,0xebec,0xebed,0xebee,0xebef
    .hword 0xf800,0xf801,0xf802,0xf803,0xf804,0xf805,0xf806,0xf807,0xf808,0xf809,0xf80a,0xf80b,0xf80c,0xf80d,0xf80e,0xf80f
    .hword 0xf820,0xf821,0xf822,0xf823,0xf824,0xf825,0xf826,0xf827,0xf828,0xf829,0xf82a,0xf82b,0xf82c,0xf82d,0xf82e,0xf82f
    .hword 0xf840,0xf841,0xf842,0xf843,0xf844,0xf845,0xf846,0xf847,0xf848,0xf849,0xf84a,0xf84b,0xf84c,0xf84d,0xf84e,0xf84f
    .hword 0xf860,0xf861,0xf862,0xf863,0xf864,0xf865,0xf866,0xf867,0xf868,0xf869,0xf86a,0xf86b,0xf86c,0xf86d,0xf86e,0xf86f
    .hword 0xf880,0xf881,0xf882,0xf883,0xf884,0xf885,0xf886,0xf887,0xf888,0xf889,0xf88a,0xf88b,0xf88c,0xf88d,0xf88e,0xf88f
    .hword 0xf8a0,0xf8a1,0xf8a2,0xf8a3,0xf8a4,0xf8a5,0xf8a6,0xf8a7,0xf8a8,0xf8a9,0xf8aa,0xf8ab,0xf8ac,0xf8ad,0xf8ae,0xf8af
    .hword 0xf8c0,0xf8c1,0xf8c2,0xf8c3,0xf8c4,0xf8c5,0xf8c6,0xf8c7,0xf8c8,0xf8c9,0xf8ca,0xf8cb,0xf8cc,0xf8cd,0xf8ce,0xf8cf
    .hword 0xf8e0,0xf8e1,0xf8e2,0xf8e3,0xf8e4,0xf8e5,0xf8e6,0xf8e7,0xf8e8,0xf8e9,0xf8ea,0xf8eb,0xf8ec,0xf8ed,0xf8ee,0xf8ef
    .hword 0xf900,0xf901,0xf902,0xf903,0xf904,0xf905,0xf906,0xf907,0xf908,0xf909,0xf90a,0xf90b,0xf90c,0xf90d,0xf90e,0xf90f
    .hword 0xf920,0xf921,0xf922,0xf923,0xf924,0xf925,0xf926,0xf927,0xf928,0xf929,0xf92a,0xf92b,0xf92c,0xf92d,0xf92e,0xf92f
    .hword 0xf940,0xf941,0xf942,0xf943,0xf944,0xf945,0xf946,0xf947,0xf948,0xf949,0xf94a,0xf94b,0xf94c,0xf94d,0xf94e,0xf94f
    .hword 0xf960,0xf961,0xf962,0xf963,0xf964,0xf965,0xf966,0xf967,0xf968,0xf969,0xf96a,0xf96b,0xf96c,0xf96d,0xf96e,0xf96f
    .hword 0xf980,0xf981,0xf982,0xf983,0xf984,0xf985,0xf986,0xf987,0xf988,0xf989,0xf98a,0xf98b,0xf98c,0xf98d,0xf98e,0xf98f
    .hword 0xf9a0,0xf9a1,0xf9a2,0xf9a3,0xf9a4,0xf9a5,0xf9a6,0xf9a7,0xf9a8,0xf9a9,0xf9aa,0xf9ab,0xf9ac,0xf9ad,0xf9ae,0xf9af
    .hword 0xf9c0,0xf9c1,0xf9c2,0xf9c3,0xf9c4,0xf9c5,0xf9c6,0xf9c7,0xf9c8,0xf9c9,0xf9ca,0xf9cb,0xf9cc,0xf9cd,0xf9ce,0xf9cf
    .hword 0xf9e0,0xf9e1,0xf9e2,0xf9e3,0xf9e4,0xf9e5,0xf9e6,0xf9e7,0xf9e8,0xf9e9,0xf9ea,0xf9eb,0xf9ec,0xf9ed,0xf9ee,0xf9ef
    .hword 0xfa00,0xfa01,0xfa02,0xfa03,0xfa04,0xfa05,0xfa06,0xfa07,0xfa08,0xfa09,0xfa0a,0xfa0b,0xfa0c,0xfa0d,0xfa0e,0xfa0f
    .hword 0xfa20,0xfa21,0xfa22,0xfa23,0xfa24,0xfa25,0xfa26,0xfa27,0xfa28,0xfa29,0xfa2a,0xfa2b,0xfa2c,0xfa2d,0xfa2e,0xfa2f
    .hword 0xfa40,0xfa41,0xfa42,0xfa43,0xfa44,0xfa45,0xfa46,0xfa47,0xfa48,0xfa49,0xfa4a,0xfa4b,0xfa4c,0xfa4d,0xfa4e,0xfa4f
    .hword 0xfa60,0xfa61,0xfa62,0xfa63,0xfa64,0xfa65,0xfa66,0xfa67,0xfa68,0xfa69,0xfa6a,0xfa6b,0xfa6c,0xfa6d,0xfa6e,0xfa6f
    .hword 0xfa80,0xfa81,0xfa82,0xfa83,0xfa84,0xfa85,0xfa86,0xfa87,0xfa88,0xfa89,0xfa8a,0xfa8b,0xfa8c,0xfa8d,0xfa8e,0xfa8f
    .hword 0xfaa0,0xfaa1,0xfaa2,0xfaa3,0xfaa4,0xfaa5,0xfaa6,0xfaa7,0xfaa8,0xfaa9,0xfaaa,0xfaab,0xfaac,0xfaad,0xfaae,0xfaaf
    .hword 0xfac0,0xfac1,0xfac2,0xfac3,0xfac4,0xfac5,0xfac6,0xfac7,0xfac8,0xfac9,0xfaca,0xfacb,0xfacc,0xfacd,0xface,0xfacf
    .hword 0xfae0,0xfae1,0xfae2,0xfae3,0xfae4,0xfae5,0xfae6,0xfae7,0xfae8,0xfae9,0xfaea,0xfaeb,0xfaec,0xfaed,0xfaee,0xfaef
    .hword 0xfb00,0xfb01,0xfb02,0xfb03,0xfb04,0xfb05,0xfb06,0xfb07,0xfb08,0xfb09,0xfb0a,0xfb0b,0xfb0c,0xfb0d,0xfb0e,0xfb0f
    .hword 0xfb20,0xfb21,0xfb22,0xfb23,0xfb24,0xfb25,0xfb26,0xfb27,0xfb28,0xfb29,0xfb2a,0xfb2b,0xfb2c,0xfb2d,0xfb2e,0xfb2f
    .hword 0xfb40,0xfb41,0xfb42,0xfb43,0xfb44,0xfb45,0xfb46,0xfb47,0xfb48,0xfb49,0xfb4a,0xfb4b,0xfb4c,0xfb4d,0xfb4e,0xfb4f
    .hword 0xfb60,0xfb61,0xfb62,0xfb63,0xfb64,0xfb65,0xfb66,0xfb67,0xfb68,0xfb69,0xfb6a,0xfb6b,0xfb6c,0xfb6d,0xfb6e,0xfb6f
    .hword 0xfb80,0xfb81,0xfb82,0xfb83,0xfb84,0xfb85,0xfb86,0xfb87,0xfb88,0xfb89,0xfb8a,0xfb8b,0xfb8c,0xfb8d,0xfb8e,0xfb8f
    .hword 0xfba0,0xfba1,0xfba2,0xfba3,0xfba4,0xfba5,0xfba6,0xfba7,0xfba8,0xfba9,0xfbaa,0xfbab,0xfbac,0xfbad,0xfbae,0xfbaf
    .hword 0xfbc0,0xfbc1,0xfbc2,0xfbc3,0xfbc4,0xfbc5,0xfbc6,0xfbc7,0xfbc8,0xfbc9,0xfbca,0xfbcb,0xfbcc,0xfbcd,0xfbce,0xfbcf
    .hword 0xfbe0,0xfbe1,0xfbe2,0xfbe3,0xfbe4,0xfbe5,0xfbe6,0xfbe7,0xfbe8,0xfbe9,0xfbea,0xfbeb,0xfbec,0xfbed,0xfbee,0xfbef
    .hword 0x8400,0x8401,0x8402,0x8403,0x8404,0x8405,0x8406,0x8407,0x8408,0x8409,0x840a,0x840b,0x840c,0x840d,0x840e,0x840f
    .hword 0x8420,0x8421,0x8422,0x8423,0x8424,0x8425,0x8426,0x8427,0x8428,0x8429,0x842a,0x842b,0x842c,0x842d,0x842e,0x842f
    .hword 0x8440,0x8441,0x8442,0x8443,0x8444,0x8445,0x8446,0x8447,0x8448,0x8449,0x844a,0x844b,0x844c,0x844d,0x844e,0x844f
    .hword 0x8460,0x8461,0x8462,0x8463,0x8464,0x8465,0x8466,0x8467,0x8468,0x8469,0x846a,0x846b,0x846c,0x846d,0x846e,0x846f
    .hword 0x8480,0x8481,0x8482,0x8483,0x8484,0x8485,0x8486,0x8487,0x8488,0x8489,0x848a,0x848b,0x848c,0x848d,0x848e,0x848f
    .hword 0x84a0,0x84a1,0x84a2,0x84a3,0x84a4,0x84a5,0x84a6,0x84a7,0x84a8,0x84a9,0x84aa,0x84ab,0x84ac,0x84ad,0x84ae,0x84af
    .hword 0x84c0,0x84c1,0x84c2,0x84c3,0x84c4,0x84c5,0x84c6,0x84c7,0x84c8,0x84c9,0x84ca,0x84cb,0x84cc,0x84cd,0x84ce,0x84cf
    .hword 0x84e0,0x84e1,0x84e2,0x84e3,0x84e4,0x84e5,0x84e6,0x84e7,0x84e8,0x84e9,0x84ea,0x84eb,0x84ec,0x84ed,0x84ee,0x84ef
    .hword 0x8500,0x8501,0x8502,0x8503,0x8504,0x8505,0x8506,0x8507,0x8508,0x8509,0x850a,0x850b,0x850c,0x850d,0x850e,0x850f
    .hword 0x8520,0x8521,0x8522,0x8523,0x8524,0x8525,0x8526,0x8527,0x8528,0x8529,0x852a,0x852b,0x852c,0x852d,0x852e,0x852f
    .hword 0x8540,0x8541,0x8542,0x8543,0x8544,0x8545,0x8546,0x8547,0x8548,0x8549,0x854a,0x854b,0x854c,0x854d,0x854e,0x854f
    .hword 0x8560,0x8561,0x8562,0x8563,0x8564,0x8565,0x8566,0x8567,0x8568,0x8569,0x856a,0x856b,0x856c,0x856d,0x856e,0x856f
    .hword 0x8580,0x8581,0x8582,0x8583,0x8584,0x8585,0x8586,0x8587,0x8588,0x8589,0x858a,0x858b,0x858c,0x858d,0x858e,0x858f
    .hword 0x85a0,0x85a1,0x85a2,0x85a3,0x85a4,0x85a5,0x85a6,0x85a7,0x85a8,0x85a9,0x85aa,0x85ab,0x85ac,0x85ad,0x85ae,0x85af
    .hword 0x85c0,0x85c1,0x85c2,0x85c3,0x85c4,0x85c5,0x85c6,0x85c7,0x85c8,0x85c9,0x85ca,0x85cb,0x85cc,0x85cd,0x85ce,0x85cf
    .hword 0x85e0,0x85e1,0x85e2,0x85e3,0x85e4,0x85e5,0x85e6,0x85e7,0x85e8,0x85e9,0x85ea,0x85eb,0x85ec,0x85ed,0x85ee,0x85ef
    .hword 0x8600,0x8601,0x8602,0x8603,0x8604,0x8605,0x8606,0x8607,0x8608,0x8609,0x860a,0x860b,0x860c,0x860d,0x860e,0x860f
    .hword 0x8620,0x8621,0x8622,0x8623,0x8624,0x8625,0x8626,0x8627,0x8628,0x8629,0x862a,0x862b,0x862c,0x862d,0x862e,0x862f
    .hword 0x8640,0x8641,0x8642,0x8643,0x8644,0x8645,0x8646,0x8647,0x8648,0x8649,0x864a,0x864b,0x864c,0x864d,0x864e,0x864f
    .hword 0x8660,0x8661,0x8662,0x8663,0x8664,0x8665,0x8666,0x8667,0x8668,0x8669,0x866a,0x866b,0x866c,0x866d,0x866e,0x866f
    .hword 0x8680,0x8681,0x8682,0x8683,0x8684,0x8685,0x8686,0x8687,0x8688,0x8689,0x868a,0x868b,0x868c,0x868d,0x868e,0x868f
    .hword 0x86a0,0x86a1,0x86a2,0x86a3,0x86a4,0x86a5,0x86a6,0x86a7,0x86a8,0x86a9,0x86aa,0x86ab,0x86ac,0x86ad,0x86ae,0x86af
    .hword 0x86c0,0x86c1,0x86c2,0x86c3,0x86c4,0x86c5,0x86c6,0x86c7,0x86c8,0x86c9,0x86ca,0x86cb,0x86cc,0x86cd,0x86ce,0x86cf
    .hword 0x86e0,0x86e1,0x86e2,0x86e3,0x86e4,0x86e5,0x86e6,0x86e7,0x86e8,0x86e9,0x86ea,0x86eb,0x86ec,0x86ed,0x86ee,0x86ef
    .hword 0x8700,0x8701,0x8702,0x8703,0x8704,0x8705,0x8706,0x8707,0x8708,0x8709,0x870a,0x870b,0x870c,0x870d,0x870e,0x870f
    .hword 0x8720,0x8721,0x8722,0x8723,0x8724,0x8725,0x8726,0x8727,0x8728,0x8729,0x872a,0x872b,0x872c,0x872d,0x872e,0x872f
    .hword 0x8740,0x8741,0x8742,0x8743,0x8744,0x8745,0x8746,0x8747,0x8748,0x8749,0x874a,0x874b,0x874c,0x874d,0x874e,0x874f
    .hword 0x8760,0x8761,0x8762,0x8763,0x8764,0x8765,0x8766,0x8767,0x8768,0x8769,0x876a,0x876b,0x876c,0x876d,0x876e,0x876f
    .hword 0x8780,0x8781,0x8782,0x8783,0x8784,0x8785,0x8786,0x8787,0x8788,0x8789,0x878a,0x878b,0x878c,0x878d,0x878e,0x878f
    .hword 0x87a0,0x87a1,0x87a2,0x87a3,0x87a4,0x87a5,0x87a6,0x87a7,0x87a8,0x87a9,0x87aa,0x87ab,0x87ac,0x87ad,0x87ae,0x87af
    .hword 0x87c0,0x87c1,0x87c2,0x87c3,0x87c4,0x87c5,0x87c6,0x87c7,0x87c8,0x87c9,0x87ca,0x87cb,0x87cc,0x87cd,0x87ce,0x87cf
    .hword 0x87e0,0x87e1,0x87e2,0x87e3,0x87e4,0x87e5,0x87e6,0x87e7,0x87e8,0x87e9,0x87ea,0x87eb,0x87ec,0x87ed,0x87ee,0x87ef
    .hword 0x9400,0x9401,0x9402,0x9403,0x9404,0x9405,0x9406,0x9407,0x9408,0x9409,0x940a,0x940b,0x940c,0x940d,0x940e,0x940f
    .hword 0x9420,0x9421,0x9422,0x9423,0x9424,0x9425,0x9426,0x9427,0x9428,0x9429,0x942a,0x942b,0x942c,0x942d,0x942e,0x942f
    .hword 0x9440,0x9441,0x9442,0x9443,0x9444,0x9445,0x9446,0x9447,0x9448,0x9449,0x944a,0x944b,0x944c,0x944d,0x944e,0x944f
    .hword 0x9460,0x9461,0x9462,0x9463,0x9464,0x9465,0x9466,0x9467,0x9468,0x9469,0x946a,0x946b,0x946c,0x946d,0x946e,0x946f
    .hword 0x9480,0x9481,0x9482,0x9483,0x9484,0x9485,0x9486,0x9487,0x9488,0x9489,0x948a,0x948b,0x948c,0x948d,0x948e,0x948f
    .hword 0x94a0,0x94a1,0x94a2,0x94a3,0x94a4,0x94a5,0x94a6,0x94a7,0x94a8,0x94a9,0x94aa,0x94ab,0x94ac,0x94ad,0x94ae,0x94af
    .hword 0x94c0,0x94c1,0x94c2,0x94c3,0x94c4,0x94c5,0x94c6,0x94c7,0x94c8,0x94c9,0x94ca,0x94cb,0x94cc,0x94cd,0x94ce,0x94cf
    .hword 0x94e0,0x94e1,0x94e2,0x94e3,0x94e4,0x94e5,0x94e6,0x94e7,0x94e8,0x94e9,0x94ea,0x94eb,0x94ec,0x94ed,0x94ee,0x94ef
    .hword 0x9500,0x9501,0x9502,0x9503,0x9504,0x9505,0x9506,0x9507,0x9508,0x9509,0x950a,0x950b,0x950c,0x950d,0x950e,0x950f
    .hword 0x9520,0x9521,0x9522,0x9523,0x9524,0x9525,0x9526,0x9527,0x9528,0x9529,0x952a,0x952b,0x952c,0x952d,0x952e,0x952f
    .hword 0x9540,0x9541,0x9542,0x9543,0x9544,0x9545,0x9546,0x9547,0x9548,0x9549,0x954a,0x954b,0x954c,0x954d,0x954e,0x954f
    .hword 0x9560,0x9561,0x9562,0x9563,0x9564,0x9565,0x9566,0x9567,0x9568,0x9569,0x956a,0x956b,0x956c,0x956d,0x956e,0x956f
    .hword 0x9580,0x9581,0x9582,0x9583,0x9584,0x9585,0x9586,0x9587,0x9588,0x9589,0x958a,0x958b,0x958c,0x958d,0x958e,0x958f
    .hword 0x95a0,0x95a1,0x95a2,0x95a3,0x95a4,0x95a5,0x95a6,0x95a7,0x95a8,0x95a9,0x95aa,0x95ab,0x95ac,0x95ad,0x95ae,0x95af
    .hword 0x95c0,0x95c1,0x95c2,0x95c3,0x95c4,0x95c5,0x95c6,0x95c7,0x95c8,0x95c9,0x95ca,0x95cb,0x95cc,0x95cd,0x95ce,0x95cf
    .hword 0x95e0,0x95e1,0x95e2,0x95e3,0x95e4,0x95e5,0x95e6,0x95e7,0x95e8,0x95e9,0x95ea,0x95eb,0x95ec,0x95ed,0x95ee,0x95ef
    .hword 0x9600,0x9601,0x9602,0x9603,0x9604,0x9605,0x9606,0x9607,0x9608,0x9609,0x960a,0x960b,0x960c,0x960d,0x960e,0x960f
    .hword 0x9620,0x9621,0x9622,0x9623,0x9624,0x9625,0x9626,0x9627,0x9628,0x9629,0x962a,0x962b,0x962c,0x962d,0x962e,0x962f
    .hword 0x9640,0x9641,0x9642,0x9643,0x9644,0x9645,0x9646,0x9647,0x9648,0x9649,0x964a,0x964b,0x964c,0x964d,0x964e,0x964f
    .hword 0x9660,0x9661,0x9662,0x9663,0x9664,0x9665,0x9666,0x9667,0x9668,0x9669,0x966a,0x966b,0x966c,0x966d,0x966e,0x966f
    .hword 0x9680,0x9681,0x9682,0x9683,0x9684,0x9685,0x9686,0x9687,0x9688,0x9689,0x968a,0x968b,0x968c,0x968d,0x968e,0x968f
    .hword 0x96a0,0x96a1,0x96a2,0x96a3,0x96a4,0x96a5,0x96a6,0x96a7,0x96a8,0x96a9,0x96aa,0x96ab,0x96ac,0x96ad,0x96ae,0x96af
    .hword 0x96c0,0x96c1,0x96c2,0x96c3,0x96c4,0x96c5,0x96c6,0x96c7,0x96c8,0x96c9,0x96ca,0x96cb,0x96cc,0x96cd,0x96ce,0x96cf
    .hword 0x96e0,0x96e1,0x96e2,0x96e3,0x96e4,0x96e5,0x96e6,0x96e7,0x96e8,0x96e9,0x96ea,0x96eb,0x96ec,0x96ed,0x96ee,0x96ef
    .hword 0x9700,0x9701,0x9702,0x9703,0x9704,0x9705,0x9706,0x9707,0x9708,0x9709,0x970a,0x970b,0x970c,0x970d,0x970e,0x970f
    .hword 0x9720,0x9721,0x9722,0x9723,0x9724,0x9725,0x9726,0x9727,0x9728,0x9729,0x972a,0x972b,0x972c,0x972d,0x972e,0x972f
    .hword 0x9740,0x9741,0x9742,0x9743,0x9744,0x9745,0x9746,0x9747,0x9748,0x9749,0x974a,0x974b,0x974c,0x974d,0x974e,0x974f
    .hword 0x9760,0x9761,0x9762,0x9763,0x9764,0x9765,0x9766,0x9767,0x9768,0x9769,0x976a,0x976b,0x976c,0x976d,0x976e,0x976f
    .hword 0x9780,0x9781,0x9782,0x9783,0x9784,0x9785,0x9786,0x9787,0x9788,0x9789,0x978a,0x978b,0x978c,0x978d,0x978e,0x978f
    .hword 0x97a0,0x97a1,0x97a2,0x97a3,0x97a4,0x97a5,0x97a6,0x97a7,0x97a8,0x97a9,0x97aa,0x97ab,0x97ac,0x97ad,0x97ae,0x97af
    .hword 0x97c0,0x97c1,0x97c2,0x97c3,0x97c4,0x97c5,0x97c6,0x97c7,0x97c8,0x97c9,0x97ca,0x97cb,0x97cc,0x97cd,0x97ce,0x97cf
    .hword 0x97e0,0x97e1,0x97e2,0x97e3,0x97e4,0x97e5,0x97e6,0x97e7,0x97e8,0x97e9,0x97ea,0x97eb,0x97ec,0x97ed,0x97ee,0x97ef
    .hword 0xa400,0xa401,0xa402,0xa403,0xa404,0xa405,0xa406,0xa407,0xa408,0xa409,0xa40a,0xa40b,0xa40c,0xa40d,0xa40e,0xa40f
    .hword 0xa420,0xa421,0xa422,0xa423,0xa424,0xa425,0xa426,0xa427,0xa428,0xa429,0xa42a,0xa42b,0xa42c,0xa42d,0xa42e,0xa42f
    .hword 0xa440,0xa441,0xa442,0xa443,0xa444,0xa445,0xa446,0xa447,0xa448,0xa449,0xa44a,0xa44b,0xa44c,0xa44d,0xa44e,0xa44f
    .hword 0xa460,0xa461,0xa462,0xa463,0xa464,0xa465,0xa466,0xa467,0xa468,0xa469,0xa46a,0xa46b,0xa46c,0xa46d,0xa46e,0xa46f
    .hword 0xa480,0xa481,0xa482,0xa483,0xa484,0xa485,0xa486,0xa487,0xa488,0xa489,0xa48a,0xa48b,0xa48c,0xa48d,0xa48e,0xa48f
    .hword 0xa4a0,0xa4a1,0xa4a2,0xa4a3,0xa4a4,0xa4a5,0xa4a6,0xa4a7,0xa4a8,0xa4a9,0xa4aa,0xa4ab,0xa4ac,0xa4ad,0xa4ae,0xa4af
    .hword 0xa4c0,0xa4c1,0xa4c2,0xa4c3,0xa4c4,0xa4c5,0xa4c6,0xa4c7,0xa4c8,0xa4c9,0xa4ca,0xa4cb,0xa4cc,0xa4cd,0xa4ce,0xa4cf
    .hword 0xa4e0,0xa4e1,0xa4e2,0xa4e3,0xa4e4,0xa4e5,0xa4e6,0xa4e7,0xa4e8,0xa4e9,0xa4ea,0xa4eb,0xa4ec,0xa4ed,0xa4ee,0xa4ef
    .hword 0xa500,0xa501,0xa502,0xa503,0xa504,0xa505,0xa506,0xa507,0xa508,0xa509,0xa50a,0xa50b,0xa50c,0xa50d,0xa50e,0xa50f
    .hword 0xa520,0xa521,0xa522,0xa523,0xa524,0xa525,0xa526,0xa527,0xa528,0xa529,0xa52a,0xa52b,0xa52c,0xa52d,0xa52e,0xa52f
    .hword 0xa540,0xa541,0xa542,0xa543,0xa544,0xa545,0xa546,0xa547,0xa548,0xa549,0xa54a,0xa54b,0xa54c,0xa54d,0xa54e,0xa54f
    .hword 0xa560,0xa561,0xa562,0xa563,0xa564,0xa565,0xa566,0xa567,0xa568,0xa569,0xa56a,0xa56b,0xa56c,0xa56d,0xa56e,0xa56f
    .hword 0xa580,0xa581,0xa582,0xa583,0xa584,0xa585,0xa586,0xa587,0xa588,0xa589,0xa58a,0xa58b,0xa58c,0xa58d,0xa58e,0xa58f
    .hword 0xa5a0,0xa5a1,0xa5a2,0xa5a3,0xa5a4,0xa5a5,0xa5a6,0xa5a7,0xa5a8,0xa5a9,0xa5aa,0xa5ab,0xa5ac,0xa5ad,0xa5ae,0xa5af
    .hword 0xa5c0,0xa5c1,0xa5c2,0xa5c3,0xa5c4,0xa5c5,0xa5c6,0xa5c7,0xa5c8,0xa5c9,0xa5ca,0xa5cb,0xa5cc,0xa5cd,0xa5ce,0xa5cf
    .hword 0xa5e0,0xa5e1,0xa5e2,0xa5e3,0xa5e4,0xa5e5,0xa5e6,0xa5e7,0xa5e8,0xa5e9,0xa5ea,0xa5eb,0xa5ec,0xa5ed,0xa5ee,0xa5ef
    .hword 0xa600,0xa601,0xa602,0xa603,0xa604,0xa605,0xa606,0xa607,0xa608,0xa609,0xa60a,0xa60b,0xa60c,0xa60d,0xa60e,0xa60f
    .hword 0xa620,0xa621,0xa622,0xa623,0xa624,0xa625,0xa626,0xa627,0xa628,0xa629,0xa62a,0xa62b,0xa62c,0xa62d,0xa62e,0xa62f
    .hword 0xa640,0xa641,0xa642,0xa643,0xa644,0xa645,0xa646,0xa647,0xa648,0xa649,0xa64a,0xa64b,0xa64c,0xa64d,0xa64e,0xa64f
    .hword 0xa660,0xa661,0xa662,0xa663,0xa664,0xa665,0xa666,0xa667,0xa668,0xa669,0xa66a,0xa66b,0xa66c,0xa66d,0xa66e,0xa66f
    .hword 0xa680,0xa681,0xa682,0xa683,0xa684,0xa685,0xa686,0xa687,0xa688,0xa689,0xa68a,0xa68b,0xa68c,0xa68d,0xa68e,0xa68f
    .hword 0xa6a0,0xa6a1,0xa6a2,0xa6a3,0xa6a4,0xa6a5,0xa6a6,0xa6a7,0xa6a8,0xa6a9,0xa6aa,0xa6ab,0xa6ac,0xa6ad,0xa6ae,0xa6af
    .hword 0xa6c0,0xa6c1,0xa6c2,0xa6c3,0xa6c4,0xa6c5,0xa6c6,0xa6c7,0xa6c8,0xa6c9,0xa6ca,0xa6cb,0xa6cc,0xa6cd,0xa6ce,0xa6cf
    .hword 0xa6e0,0xa6e1,0xa6e2,0xa6e3,0xa6e4,0xa6e5,0xa6e6,0xa6e7,0xa6e8,0xa6e9,0xa6ea,0xa6eb,0xa6ec,0xa6ed,0xa6ee,0xa6ef
    .hword 0xa700,0xa701,0xa702,0xa703,0xa704,0xa705,0xa706,0xa707,0xa708,0xa709,0xa70a,0xa70b,0xa70c,0xa70d,0xa70e,0xa70f
    .hword 0xa720,0xa721,0xa722,0xa723,0xa724,0xa725,0xa726,0xa727,0xa728,0xa729,0xa72a,0xa72b,0xa72c,0xa72d,0xa72e,0xa72f
    .hword 0xa740,0xa741,0xa742,0xa743,0xa744,0xa745,0xa746,0xa747,0xa748,0xa749,0xa74a,0xa74b,0xa74c,0xa74d,0xa74e,0xa74f
    .hword 0xa760,0xa761,0xa762,0xa763,0xa764,0xa765,0xa766,0xa767,0xa768,0xa769,0xa76a,0xa76b,0xa76c,0xa76d,0xa76e,0xa76f
    .hword 0xa780,0xa781,0xa782,0xa783,0xa784,0xa785,0xa786,0xa787,0xa788,0xa789,0xa78a,0xa78b,0xa78c,0xa78d,0xa78e,0xa78f
    .hword 0xa7a0,0xa7a1,0xa7a2,0xa7a3,0xa7a4,0xa7a5,0xa7a6,0xa7a7,0xa7a8,0xa7a9,0xa7aa,0xa7ab,0xa7ac,0xa7ad,0xa7ae,0xa7af
    .hword 0xa7c0,0xa7c1,0xa7c2,0xa7c3,0xa7c4,0xa7c5,0xa7c6,0xa7c7,0xa7c8,0xa7c9,0xa7ca,0xa7cb,0xa7cc,0xa7cd,0xa7ce,0xa7cf
    .hword 0xa7e0,0xa7e1,0xa7e2,0xa7e3,0xa7e4,0xa7e5,0xa7e6,0xa7e7,0xa7e8,0xa7e9,0xa7ea,0xa7eb,0xa7ec,0xa7ed,0xa7ee,0xa7ef
    .hword 0xb400,0xb401,0xb402,0xb403,0xb404,0xb405,0xb406,0xb407,0xb408,0xb409,0xb40a,0xb40b,0xb40c,0xb40d,0xb40e,0xb40f
    .hword 0xb420,0xb421,0xb422,0xb423,0xb424,0xb425,0xb426,0xb427,0xb428,0xb429,0xb42a,0xb42b,0xb42c,0xb42d,0xb42e,0xb42f
    .hword 0xb440,0xb441,0xb442,0xb443,0xb444,0xb445,0xb446,0xb447,0xb448,0xb449,0xb44a,0xb44b,0xb44c,0xb44d,0xb44e,0xb44f
    .hword 0xb460,0xb461,0xb462,0xb463,0xb464,0xb465,0xb466,0xb467,0xb468,0xb469,0xb46a,0xb46b,0xb46c,0xb46d,0xb46e,0xb46f
    .hword 0xb480,0xb481,0xb482,0xb483,0xb484,0xb485,0xb486,0xb487,0xb488,0xb489,0xb48a,0xb48b,0xb48c,0xb48d,0xb48e,0xb48f
    .hword 0xb4a0,0xb4a1,0xb4a2,0xb4a3,0xb4a4,0xb4a5,0xb4a6,0xb4a7,0xb4a8,0xb4a9,0xb4aa,0xb4ab,0xb4ac,0xb4ad,0xb4ae,0xb4af
    .hword 0xb4c0,0xb4c1,0xb4c2,0xb4c3,0xb4c4,0xb4c5,0xb4c6,0xb4c7,0xb4c8,0xb4c9,0xb4ca,0xb4cb,0xb4cc,0xb4cd,0xb4ce,0xb4cf
    .hword 0xb4e0,0xb4e1,0xb4e2,0xb4e3,0xb4e4,0xb4e5,0xb4e6,0xb4e7,0xb4e8,0xb4e9,0xb4ea,0xb4eb,0xb4ec,0xb4ed,0xb4ee,0xb4ef
    .hword 0xb500,0xb501,0xb502,0xb503,0xb504,0xb505,0xb506,0xb507,0xb508,0xb509,0xb50a,0xb50b,0xb50c,0xb50d,0xb50e,0xb50f
    .hword 0xb520,0xb521,0xb522,0xb523,0xb524,0xb525,0xb526,0xb527,0xb528,0xb529,0xb52a,0xb52b,0xb52c,0xb52d,0xb52e,0xb52f
    .hword 0xb540,0xb541,0xb542,0xb543,0xb544,0xb545,0xb546,0xb547,0xb548,0xb549,0xb54a,0xb54b,0xb54c,0xb54d,0xb54e,0xb54f
    .hword 0xb560,0xb561,0xb562,0xb563,0xb564,0xb565,0xb566,0xb567,0xb568,0xb569,0xb56a,0xb56b,0xb56c,0xb56d,0xb56e,0xb56f
    .hword 0xb580,0xb581,0xb582,0xb583,0xb584,0xb585,0xb586,0xb587,0xb588,0xb589,0xb58a,0xb58b,0xb58c,0xb58d,0xb58e,0xb58f
    .hword 0xb5a0,0xb5a1,0xb5a2,0xb5a3,0xb5a4,0xb5a5,0xb5a6,0xb5a7,0xb5a8,0xb5a9,0xb5aa,0xb5ab,0xb5ac,0xb5ad,0xb5ae,0xb5af
    .hword 0xb5c0,0xb5c1,0xb5c2,0xb5c3,0xb5c4,0xb5c5,0xb5c6,0xb5c7,0xb5c8,0xb5c9,0xb5ca,0xb5cb,0xb5cc,0xb5cd,0xb5ce,0xb5cf
    .hword 0xb5e0,0xb5e1,0xb5e2,0xb5e3,0xb5e4,0xb5e5,0xb5e6,0xb5e7,0xb5e8,0xb5e9,0xb5ea,0xb5eb,0xb5ec,0xb5ed,0xb5ee,0xb5ef
    .hword 0xb600,0xb601,0xb602,0xb603,0xb604,0xb605,0xb606,0xb607,0xb608,0xb609,0xb60a,0xb60b,0xb60c,0xb60d,0xb60e,0xb60f
    .hword 0xb620,0xb621,0xb622,0xb623,0xb624,0xb625,0xb626,0xb627,0xb628,0xb629,0xb62a,0xb62b,0xb62c,0xb62d,0xb62e,0xb62f
    .hword 0xb640,0xb641,0xb642,0xb643,0xb644,0xb645,0xb646,0xb647,0xb648,0xb649,0xb64a,0xb64b,0xb64c,0xb64d,0xb64e,0xb64f
    .hword 0xb660,0xb661,0xb662,0xb663,0xb664,0xb665,0xb666,0xb667,0xb668,0xb669,0xb66a,0xb66b,0xb66c,0xb66d,0xb66e,0xb66f
    .hword 0xb680,0xb681,0xb682,0xb683,0xb684,0xb685,0xb686,0xb687,0xb688,0xb689,0xb68a,0xb68b,0xb68c,0xb68d,0xb68e,0xb68f
    .hword 0xb6a0,0xb6a1,0xb6a2,0xb6a3,0xb6a4,0xb6a5,0xb6a6,0xb6a7,0xb6a8,0xb6a9,0xb6aa,0xb6ab,0xb6ac,0xb6ad,0xb6ae,0xb6af
    .hword 0xb6c0,0xb6c1,0xb6c2,0xb6c3,0xb6c4,0xb6c5,0xb6c6,0xb6c7,0xb6c8,0xb6c9,0xb6ca,0xb6cb,0xb6cc,0xb6cd,0xb6ce,0xb6cf
    .hword 0xb6e0,0xb6e1,0xb6e2,0xb6e3,0xb6e4,0xb6e5,0xb6e6,0xb6e7,0xb6e8,0xb6e9,0xb6ea,0xb6eb,0xb6ec,0xb6ed,0xb6ee,0xb6ef
    .hword 0xb700,0xb701,0xb702,0xb703,0xb704,0xb705,0xb706,0xb707,0xb708,0xb709,0xb70a,0xb70b,0xb70c,0xb70d,0xb70e,0xb70f
    .hword 0xb720,0xb721,0xb722,0xb723,0xb724,0xb725,0xb726,0xb727,0xb728,0xb729,0xb72a,0xb72b,0xb72c,0xb72d,0xb72e,0xb72f
    .hword 0xb740,0xb741,0xb742,0xb743,0xb744,0xb745,0xb746,0xb747,0xb748,0xb749,0xb74a,0xb74b,0xb74c,0xb74d,0xb74e,0xb74f
    .hword 0xb760,0xb761,0xb762,0xb763,0xb764,0xb765,0xb766,0xb767,0xb768,0xb769,0xb76a,0xb76b,0xb76c,0xb76d,0xb76e,0xb76f
    .hword 0xb780,0xb781,0xb782,0xb783,0xb784,0xb785,0xb786,0xb787,0xb788,0xb789,0xb78a,0xb78b,0xb78c,0xb78d,0xb78e,0xb78f
    .hword 0xb7a0,0xb7a1,0xb7a2,0xb7a3,0xb7a4,0xb7a5,0xb7a6,0xb7a7,0xb7a8,0xb7a9,0xb7aa,0xb7ab,0xb7ac,0xb7ad,0xb7ae,0xb7af
    .hword 0xb7c0,0xb7c1,0xb7c2,0xb7c3,0xb7c4,0xb7c5,0xb7c6,0xb7c7,0xb7c8,0xb7c9,0xb7ca,0xb7cb,0xb7cc,0xb7cd,0xb7ce,0xb7cf
    .hword 0xb7e0,0xb7e1,0xb7e2,0xb7e3,0xb7e4,0xb7e5,0xb7e6,0xb7e7,0xb7e8,0xb7e9,0xb7ea,0xb7eb,0xb7ec,0xb7ed,0xb7ee,0xb7ef
    .hword 0xc400,0xc401,0xc402,0xc403,0xc404,0xc405,0xc406,0xc407,0xc408,0xc409,0xc40a,0xc40b,0xc40c,0xc40d,0xc40e,0xc40f
    .hword 0xc420,0xc421,0xc422,0xc423,0xc424,0xc425,0xc426,0xc427,0xc428,0xc429,0xc42a,0xc42b,0xc42c,0xc42d,0xc42e,0xc42f
    .hword 0xc440,0xc441,0xc442,0xc443,0xc444,0xc445,0xc446,0xc447,0xc448,0xc449,0xc44a,0xc44b,0xc44c,0xc44d,0xc44e,0xc44f
    .hword 0xc460,0xc461,0xc462,0xc463,0xc464,0xc465,0xc466,0xc467,0xc468,0xc469,0xc46a,0xc46b,0xc46c,0xc46d,0xc46e,0xc46f
    .hword 0xc480,0xc481,0xc482,0xc483,0xc484,0xc485,0xc486,0xc487,0xc488,0xc489,0xc48a,0xc48b,0xc48c,0xc48d,0xc48e,0xc48f
    .hword 0xc4a0,0xc4a1,0xc4a2,0xc4a3,0xc4a4,0xc4a5,0xc4a6,0xc4a7,0xc4a8,0xc4a9,0xc4aa,0xc4ab,0xc4ac,0xc4ad,0xc4ae,0xc4af
    .hword 0xc4c0,0xc4c1,0xc4c2,0xc4c3,0xc4c4,0xc4c5,0xc4c6,0xc4c7,0xc4c8,0xc4c9,0xc4ca,0xc4cb,0xc4cc,0xc4cd,0xc4ce,0xc4cf
    .hword 0xc4e0,0xc4e1,0xc4e2,0xc4e3,0xc4e4,0xc4e5,0xc4e6,0xc4e7,0xc4e8,0xc4e9,0xc4ea,0xc4eb,0xc4ec,0xc4ed,0xc4ee,0xc4ef
    .hword 0xc500,0xc501,0xc502,0xc503,0xc504,0xc505,0xc506,0xc507,0xc508,0xc509,0xc50a,0xc50b,0xc50c,0xc50d,0xc50e,0xc50f
    .hword 0xc520,0xc521,0xc522,0xc523,0xc524,0xc525,0xc526,0xc527,0xc528,0xc529,0xc52a,0xc52b,0xc52c,0xc52d,0xc52e,0xc52f
    .hword 0xc540,0xc541,0xc542,0xc543,0xc544,0xc545,0xc546,0xc547,0xc548,0xc549,0xc54a,0xc54b,0xc54c,0xc54d,0xc54e,0xc54f
    .hword 0xc560,0xc561,0xc562,0xc563,0xc564,0xc565,0xc566,0xc567,0xc568,0xc569,0xc56a,0xc56b,0xc56c,0xc56d,0xc56e,0xc56f
    .hword 0xc580,0xc581,0xc582,0xc583,0xc584,0xc585,0xc586,0xc587,0xc588,0xc589,0xc58a,0xc58b,0xc58c,0xc58d,0xc58e,0xc58f
    .hword 0xc5a0,0xc5a1,0xc5a2,0xc5a3,0xc5a4,0xc5a5,0xc5a6,0xc5a7,0xc5a8,0xc5a9,0xc5aa,0xc5ab,0xc5ac,0xc5ad,0xc5ae,0xc5af
    .hword 0xc5c0,0xc5c1,0xc5c2,0xc5c3,0xc5c4,0xc5c5,0xc5c6,0xc5c7,0xc5c8,0xc5c9,0xc5ca,0xc5cb,0xc5cc,0xc5cd,0xc5ce,0xc5cf
    .hword 0xc5e0,0xc5e1,0xc5e2,0xc5e3,0xc5e4,0xc5e5,0xc5e6,0xc5e7,0xc5e8,0xc5e9,0xc5ea,0xc5eb,0xc5ec,0xc5ed,0xc5ee,0xc5ef
    .hword 0xc600,0xc601,0xc602,0xc603,0xc604,0xc605,0xc606,0xc607,0xc608,0xc609,0xc60a,0xc60b,0xc60c,0xc60d,0xc60e,0xc60f
    .hword 0xc620,0xc621,0xc622,0xc623,0xc624,0xc625,0xc626,0xc627,0xc628,0xc629,0xc62a,0xc62b,0xc62c,0xc62d,0xc62e,0xc62f
    .hword 0xc640,0xc641,0xc642,0xc643,0xc644,0xc645,0xc646,0xc647,0xc648,0xc649,0xc64a,0xc64b,0xc64c,0xc64d,0xc64e,0xc64f
    .hword 0xc660,0xc661,0xc662,0xc663,0xc664,0xc665,0xc666,0xc667,0xc668,0xc669,0xc66a,0xc66b,0xc66c,0xc66d,0xc66e,0xc66f
    .hword 0xc680,0xc681,0xc682,0xc683,0xc684,0xc685,0xc686,0xc687,0xc688,0xc689,0xc68a,0xc68b,0xc68c,0xc68d,0xc68e,0xc68f
    .hword 0xc6a0,0xc6a1,0xc6a2,0xc6a3,0xc6a4,0xc6a5,0xc6a6,0xc6a7,0xc6a8,0xc6a9,0xc6aa,0xc6ab,0xc6ac,0xc6ad,0xc6ae,0xc6af
    .hword 0xc6c0,0xc6c1,0xc6c2,0xc6c3,0xc6c4,0xc6c5,0xc6c6,0xc6c7,0xc6c8,0xc6c9,0xc6ca,0xc6cb,0xc6cc,0xc6cd,0xc6ce,0xc6cf
    .hword 0xc6e0,0xc6e1,0xc6e2,0xc6e3,0xc6e4,0xc6e5,0xc6e6,0xc6e7,0xc6e8,0xc6e9,0xc6ea,0xc6eb,0xc6ec,0xc6ed,0xc6ee,0xc6ef
    .hword 0xc700,0xc701,0xc702,0xc703,0xc704,0xc705,0xc706,0xc707,0xc708,0xc709,0xc70a,0xc70b,0xc70c,0xc70d,0xc70e,0xc70f
    .hword 0xc720,0xc721,0xc722,0xc723,0xc724,0xc725,0xc726,0xc727,0xc728,0xc729,0xc72a,0xc72b,0xc72c,0xc72d,0xc72e,0xc72f
    .hword 0xc740,0xc741,0xc742,0xc743,0xc744,0xc745,0xc746,0xc747,0xc748,0xc749,0xc74a,0xc74b,0xc74c,0xc74d,0xc74e,0xc74f
    .hword 0xc760,0xc761,0xc762,0xc763,0xc764,0xc765,0xc766,0xc767,0xc768,0xc769,0xc76a,0xc76b,0xc76c,0xc76d,0xc76e,0xc76f
    .hword 0xc780,0xc781,0xc782,0xc783,0xc784,0xc785,0xc786,0xc787,0xc788,0xc789,0xc78a,0xc78b,0xc78c,0xc78d,0xc78e,0xc78f
    .hword 0xc7a0,0xc7a1,0xc7a2,0xc7a3,0xc7a4,0xc7a5,0xc7a6,0xc7a7,0xc7a8,0xc7a9,0xc7aa,0xc7ab,0xc7ac,0xc7ad,0xc7ae,0xc7af
    .hword 0xc7c0,0xc7c1,0xc7c2,0xc7c3,0xc7c4,0xc7c5,0xc7c6,0xc7c7,0xc7c8,0xc7c9,0xc7ca,0xc7cb,0xc7cc,0xc7cd,0xc7ce,0xc7cf
    .hword 0xc7e0,0xc7e1,0xc7e2,0xc7e3,0xc7e4,0xc7e5,0xc7e6,0xc7e7,0xc7e8,0xc7e9,0xc7ea,0xc7eb,0xc7ec,0xc7ed,0xc7ee,0xc7ef
    .hword 0xd400,0xd401,0xd402,0xd403,0xd404,0xd405,0xd406,0xd407,0xd408,0xd409,0xd40a,0xd40b,0xd40c,0xd40d,0xd40e,0xd40f
    .hword 0xd420,0xd421,0xd422,0xd423,0xd424,0xd425,0xd426,0xd427,0xd428,0xd429,0xd42a,0xd42b,0xd42c,0xd42d,0xd42e,0xd42f
    .hword 0xd440,0xd441,0xd442,0xd443,0xd444,0xd445,0xd446,0xd447,0xd448,0xd449,0xd44a,0xd44b,0xd44c,0xd44d,0xd44e,0xd44f
    .hword 0xd460,0xd461,0xd462,0xd463,0xd464,0xd465,0xd466,0xd467,0xd468,0xd469,0xd46a,0xd46b,0xd46c,0xd46d,0xd46e,0xd46f
    .hword 0xd480,0xd481,0xd482,0xd483,0xd484,0xd485,0xd486,0xd487,0xd488,0xd489,0xd48a,0xd48b,0xd48c,0xd48d,0xd48e,0xd48f
    .hword 0xd4a0,0xd4a1,0xd4a2,0xd4a3,0xd4a4,0xd4a5,0xd4a6,0xd4a7,0xd4a8,0xd4a9,0xd4aa,0xd4ab,0xd4ac,0xd4ad,0xd4ae,0xd4af
    .hword 0xd4c0,0xd4c1,0xd4c2,0xd4c3,0xd4c4,0xd4c5,0xd4c6,0xd4c7,0xd4c8,0xd4c9,0xd4ca,0xd4cb,0xd4cc,0xd4cd,0xd4ce,0xd4cf
    .hword 0xd4e0,0xd4e1,0xd4e2,0xd4e3,0xd4e4,0xd4e5,0xd4e6,0xd4e7,0xd4e8,0xd4e9,0xd4ea,0xd4eb,0xd4ec,0xd4ed,0xd4ee,0xd4ef
    .hword 0xd500,0xd501,0xd502,0xd503,0xd504,0xd505,0xd506,0xd507,0xd508,0xd509,0xd50a,0xd50b,0xd50c,0xd50d,0xd50e,0xd50f
    .hword 0xd520,0xd521,0xd522,0xd523,0xd524,0xd525,0xd526,0xd527,0xd528,0xd529,0xd52a,0xd52b,0xd52c,0xd52d,0xd52e,0xd52f
    .hword 0xd540,0xd541,0xd542,0xd543,0xd544,0xd545,0xd546,0xd547,0xd548,0xd549,0xd54a,0xd54b,0xd54c,0xd54d,0xd54e,0xd54f
    .hword 0xd560,0xd561,0xd562,0xd563,0xd564,0xd565,0xd566,0xd567,0xd568,0xd569,0xd56a,0xd56b,0xd56c,0xd56d,0xd56e,0xd56f
    .hword 0xd580,0xd581,0xd582,0xd583,0xd584,0xd585,0xd586,0xd587,0xd588,0xd589,0xd58a,0xd58b,0xd58c,0xd58d,0xd58e,0xd58f
    .hword 0xd5a0,0xd5a1,0xd5a2,0xd5a3,0xd5a4,0xd5a5,0xd5a6,0xd5a7,0xd5a8,0xd5a9,0xd5aa,0xd5ab,0xd5ac,0xd5ad,0xd5ae,0xd5af
    .hword 0xd5c0,0xd5c1,0xd5c2,0xd5c3,0xd5c4,0xd5c5,0xd5c6,0xd5c7,0xd5c8,0xd5c9,0xd5ca,0xd5cb,0xd5cc,0xd5cd,0xd5ce,0xd5cf
    .hword 0xd5e0,0xd5e1,0xd5e2,0xd5e3,0xd5e4,0xd5e5,0xd5e6,0xd5e7,0xd5e8,0xd5e9,0xd5ea,0xd5eb,0xd5ec,0xd5ed,0xd5ee,0xd5ef
    .hword 0xd600,0xd601,0xd602,0xd603,0xd604,0xd605,0xd606,0xd607,0xd608,0xd609,0xd60a,0xd60b,0xd60c,0xd60d,0xd60e,0xd60f
    .hword 0xd620,0xd621,0xd622,0xd623,0xd624,0xd625,0xd626,0xd627,0xd628,0xd629,0xd62a,0xd62b,0xd62c,0xd62d,0xd62e,0xd62f
    .hword 0xd640,0xd641,0xd642,0xd643,0xd644,0xd645,0xd646,0xd647,0xd648,0xd649,0xd64a,0xd64b,0xd64c,0xd64d,0xd64e,0xd64f
    .hword 0xd660,0xd661,0xd662,0xd663,0xd664,0xd665,0xd666,0xd667,0xd668,0xd669,0xd66a,0xd66b,0xd66c,0xd66d,0xd66e,0xd66f
    .hword 0xd680,0xd681,0xd682,0xd683,0xd684,0xd685,0xd686,0xd687,0xd688,0xd689,0xd68a,0xd68b,0xd68c,0xd68d,0xd68e,0xd68f
    .hword 0xd6a0,0xd6a1,0xd6a2,0xd6a3,0xd6a4,0xd6a5,0xd6a6,0xd6a7,0xd6a8,0xd6a9,0xd6aa,0xd6ab,0xd6ac,0xd6ad,0xd6ae,0xd6af
    .hword 0xd6c0,0xd6c1,0xd6c2,0xd6c3,0xd6c4,0xd6c5,0xd6c6,0xd6c7,0xd6c8,0xd6c9,0xd6ca,0xd6cb,0xd6cc,0xd6cd,0xd6ce,0xd6cf
    .hword 0xd6e0,0xd6e1,0xd6e2,0xd6e3,0xd6e4,0xd6e5,0xd6e6,0xd6e7,0xd6e8,0xd6e9,0xd6ea,0xd6eb,0xd6ec,0xd6ed,0xd6ee,0xd6ef
    .hword 0xd700,0xd701,0xd702,0xd703,0xd704,0xd705,0xd706,0xd707,0xd708,0xd709,0xd70a,0xd70b,0xd70c,0xd70d,0xd70e,0xd70f
    .hword 0xd720,0xd721,0xd722,0xd723,0xd724,0xd725,0xd726,0xd727,0xd728,0xd729,0xd72a,0xd72b,0xd72c,0xd72d,0xd72e,0xd72f
    .hword 0xd740,0xd741,0xd742,0xd743,0xd744,0xd745,0xd746,0xd747,0xd748,0xd749,0xd74a,0xd74b,0xd74c,0xd74d,0xd74e,0xd74f
    .hword 0xd760,0xd761,0xd762,0xd763,0xd764,0xd765,0xd766,0xd767,0xd768,0xd769,0xd76a,0xd76b,0xd76c,0xd76d,0xd76e,0xd76f
    .hword 0xd780,0xd781,0xd782,0xd783,0xd784,0xd785,0xd786,0xd787,0xd788,0xd789,0xd78a,0xd78b,0xd78c,0xd78d,0xd78e,0xd78f
    .hword 0xd7a0,0xd7a1,0xd7a2,0xd7a3,0xd7a4,0xd7a5,0xd7a6,0xd7a7,0xd7a8,0xd7a9,0xd7aa,0xd7ab,0xd7ac,0xd7ad,0xd7ae,0xd7af
    .hword 0xd7c0,0xd7c1,0xd7c2,0xd7c3,0xd7c4,0xd7c5,0xd7c6,0xd7c7,0xd7c8,0xd7c9,0xd7ca,0xd7cb,0xd7cc,0xd7cd,0xd7ce,0xd7cf
    .hword 0xd7e0,0xd7e1,0xd7e2,0xd7e3,0xd7e4,0xd7e5,0xd7e6,0xd7e7,0xd7e8,0xd7e9,0xd7ea,0xd7eb,0xd7ec,0xd7ed,0xd7ee,0xd7ef
    .hword 0xe400,0xe401,0xe402,0xe403,0xe404,0xe405,0xe406,0xe407,0xe408,0xe409,0xe40a,0xe40b,0xe40c,0xe40d,0xe40e,0xe40f
    .hword 0xe420,0xe421,0xe422,0xe423,0xe424,0xe425,0xe426,0xe427,0xe428,0xe429,0xe42a,0xe42b,0xe42c,0xe42d,0xe42e,0xe42f
    .hword 0xe440,0xe441,0xe442,0xe443,0xe444,0xe445,0xe446,0xe447,0xe448,0xe449,0xe44a,0xe44b,0xe44c,0xe44d,0xe44e,0xe44f
    .hword 0xe460,0xe461,0xe462,0xe463,0xe464,0xe465,0xe466,0xe467,0xe468,0xe469,0xe46a,0xe46b,0xe46c,0xe46d,0xe46e,0xe46f
    .hword 0xe480,0xe481,0xe482,0xe483,0xe484,0xe485,0xe486,0xe487,0xe488,0xe489,0xe48a,0xe48b,0xe48c,0xe48d,0xe48e,0xe48f
    .hword 0xe4a0,0xe4a1,0xe4a2,0xe4a3,0xe4a4,0xe4a5,0xe4a6,0xe4a7,0xe4a8,0xe4a9,0xe4aa,0xe4ab,0xe4ac,0xe4ad,0xe4ae,0xe4af
    .hword 0xe4c0,0xe4c1,0xe4c2,0xe4c3,0xe4c4,0xe4c5,0xe4c6,0xe4c7,0xe4c8,0xe4c9,0xe4ca,0xe4cb,0xe4cc,0xe4cd,0xe4ce,0xe4cf
    .hword 0xe4e0,0xe4e1,0xe4e2,0xe4e3,0xe4e4,0xe4e5,0xe4e6,0xe4e7,0xe4e8,0xe4e9,0xe4ea,0xe4eb,0xe4ec,0xe4ed,0xe4ee,0xe4ef
    .hword 0xe500,0xe501,0xe502,0xe503,0xe504,0xe505,0xe506,0xe507,0xe508,0xe509,0xe50a,0xe50b,0xe50c,0xe50d,0xe50e,0xe50f
    .hword 0xe520,0xe521,0xe522,0xe523,0xe524,0xe525,0xe526,0xe527,0xe528,0xe529,0xe52a,0xe52b,0xe52c,0xe52d,0xe52e,0xe52f
    .hword 0xe540,0xe541,0xe542,0xe543,0xe544,0xe545,0xe546,0xe547,0xe548,0xe549,0xe54a,0xe54b,0xe54c,0xe54d,0xe54e,0xe54f
    .hword 0xe560,0xe561,0xe562,0xe563,0xe564,0xe565,0xe566,0xe567,0xe568,0xe569,0xe56a,0xe56b,0xe56c,0xe56d,0xe56e,0xe56f
    .hword 0xe580,0xe581,0xe582,0xe583,0xe584,0xe585,0xe586,0xe587,0xe588,0xe589,0xe58a,0xe58b,0xe58c,0xe58d,0xe58e,0xe58f
    .hword 0xe5a0,0xe5a1,0xe5a2,0xe5a3,0xe5a4,0xe5a5,0xe5a6,0xe5a7,0xe5a8,0xe5a9,0xe5aa,0xe5ab,0xe5ac,0xe5ad,0xe5ae,0xe5af
    .hword 0xe5c0,0xe5c1,0xe5c2,0xe5c3,0xe5c4,0xe5c5,0xe5c6,0xe5c7,0xe5c8,0xe5c9,0xe5ca,0xe5cb,0xe5cc,0xe5cd,0xe5ce,0xe5cf
    .hword 0xe5e0,0xe5e1,0xe5e2,0xe5e3,0xe5e4,0xe5e5,0xe5e6,0xe5e7,0xe5e8,0xe5e9,0xe5ea,0xe5eb,0xe5ec,0xe5ed,0xe5ee,0xe5ef
    .hword 0xe600,0xe601,0xe602,0xe603,0xe604,0xe605,0xe606,0xe607,0xe608,0xe609,0xe60a,0xe60b,0xe60c,0xe60d,0xe60e,0xe60f
    .hword 0xe620,0xe621,0xe622,0xe623,0xe624,0xe625,0xe626,0xe627,0xe628,0xe629,0xe62a,0xe62b,0xe62c,0xe62d,0xe62e,0xe62f
    .hword 0xe640,0xe641,0xe642,0xe643,0xe644,0xe645,0xe646,0xe647,0xe648,0xe649,0xe64a,0xe64b,0xe64c,0xe64d,0xe64e,0xe64f
    .hword 0xe660,0xe661,0xe662,0xe663,0xe664,0xe665,0xe666,0xe667,0xe668,0xe669,0xe66a,0xe66b,0xe66c,0xe66d,0xe66e,0xe66f
    .hword 0xe680,0xe681,0xe682,0xe683,0xe684,0xe685,0xe686,0xe687,0xe688,0xe689,0xe68a,0xe68b,0xe68c,0xe68d,0xe68e,0xe68f
    .hword 0xe6a0,0xe6a1,0xe6a2,0xe6a3,0xe6a4,0xe6a5,0xe6a6,0xe6a7,0xe6a8,0xe6a9,0xe6aa,0xe6ab,0xe6ac,0xe6ad,0xe6ae,0xe6af
    .hword 0xe6c0,0xe6c1,0xe6c2,0xe6c3,0xe6c4,0xe6c5,0xe6c6,0xe6c7,0xe6c8,0xe6c9,0xe6ca,0xe6cb,0xe6cc,0xe6cd,0xe6ce,0xe6cf
    .hword 0xe6e0,0xe6e1,0xe6e2,0xe6e3,0xe6e4,0xe6e5,0xe6e6,0xe6e7,0xe6e8,0xe6e9,0xe6ea,0xe6eb,0xe6ec,0xe6ed,0xe6ee,0xe6ef
    .hword 0xe700,0xe701,0xe702,0xe703,0xe704,0xe705,0xe706,0xe707,0xe708,0xe709,0xe70a,0xe70b,0xe70c,0xe70d,0xe70e,0xe70f
    .hword 0xe720,0xe721,0xe722,0xe723,0xe724,0xe725,0xe726,0xe727,0xe728,0xe729,0xe72a,0xe72b,0xe72c,0xe72d,0xe72e,0xe72f
    .hword 0xe740,0xe741,0xe742,0xe743,0xe744,0xe745,0xe746,0xe747,0xe748,0xe749,0xe74a,0xe74b,0xe74c,0xe74d,0xe74e,0xe74f
    .hword 0xe760,0xe761,0xe762,0xe763,0xe764,0xe765,0xe766,0xe767,0xe768,0xe769,0xe76a,0xe76b,0xe76c,0xe76d,0xe76e,0xe76f
    .hword 0xe780,0xe781,0xe782,0xe783,0xe784,0xe785,0xe786,0xe787,0xe788,0xe789,0xe78a,0xe78b,0xe78c,0xe78d,0xe78e,0xe78f
    .hword 0xe7a0,0xe7a1,0xe7a2,0xe7a3,0xe7a4,0xe7a5,0xe7a6,0xe7a7,0xe7a8,0xe7a9,0xe7aa,0xe7ab,0xe7ac,0xe7ad,0xe7ae,0xe7af
    .hword 0xe7c0,0xe7c1,0xe7c2,0xe7c3,0xe7c4,0xe7c5,0xe7c6,0xe7c7,0xe7c8,0xe7c9,0xe7ca,0xe7cb,0xe7cc,0xe7cd,0xe7ce,0xe7cf
    .hword 0xe7e0,0xe7e1,0xe7e2,0xe7e3,0xe7e4,0xe7e5,0xe7e6,0xe7e7,0xe7e8,0xe7e9,0xe7ea,0xe7eb,0xe7ec,0xe7ed,0xe7ee,0xe7ef
    .hword 0xf400,0xf401,0xf402,0xf403,0xf404,0xf405,0xf406,0xf407,0xf408,0xf409,0xf40a,0xf40b,0xf40c,0xf40d,0xf40e,0xf40f
    .hword 0xf420,0xf421,0xf422,0xf423,0xf424,0xf425,0xf426,0xf427,0xf428,0xf429,0xf42a,0xf42b,0xf42c,0xf42d,0xf42e,0xf42f
    .hword 0xf440,0xf441,0xf442,0xf443,0xf444,0xf445,0xf446,0xf447,0xf448,0xf449,0xf44a,0xf44b,0xf44c,0xf44d,0xf44e,0xf44f
    .hword 0xf460,0xf461,0xf462,0xf463,0xf464,0xf465,0xf466,0xf467,0xf468,0xf469,0xf46a,0xf46b,0xf46c,0xf46d,0xf46e,0xf46f
    .hword 0xf480,0xf481,0xf482,0xf483,0xf484,0xf485,0xf486,0xf487,0xf488,0xf489,0xf48a,0xf48b,0xf48c,0xf48d,0xf48e,0xf48f
    .hword 0xf4a0,0xf4a1,0xf4a2,0xf4a3,0xf4a4,0xf4a5,0xf4a6,0xf4a7,0xf4a8,0xf4a9,0xf4aa,0xf4ab,0xf4ac,0xf4ad,0xf4ae,0xf4af
    .hword 0xf4c0,0xf4c1,0xf4c2,0xf4c3,0xf4c4,0xf4c5,0xf4c6,0xf4c7,0xf4c8,0xf4c9,0xf4ca,0xf4cb,0xf4cc,0xf4cd,0xf4ce,0xf4cf
    .hword 0xf4e0,0xf4e1,0xf4e2,0xf4e3,0xf4e4,0xf4e5,0xf4e6,0xf4e7,0xf4e8,0xf4e9,0xf4ea,0xf4eb,0xf4ec,0xf4ed,0xf4ee,0xf4ef
    .hword 0xf500,0xf501,0xf502,0xf503,0xf504,0xf505,0xf506,0xf507,0xf508,0xf509,0xf50a,0xf50b,0xf50c,0xf50d,0xf50e,0xf50f
    .hword 0xf520,0xf521,0xf522,0xf523,0xf524,0xf525,0xf526,0xf527,0xf528,0xf529,0xf52a,0xf52b,0xf52c,0xf52d,0xf52e,0xf52f
    .hword 0xf540,0xf541,0xf542,0xf543,0xf544,0xf545,0xf546,0xf547,0xf548,0xf549,0xf54a,0xf54b,0xf54c,0xf54d,0xf54e,0xf54f
    .hword 0xf560,0xf561,0xf562,0xf563,0xf564,0xf565,0xf566,0xf567,0xf568,0xf569,0xf56a,0xf56b,0xf56c,0xf56d,0xf56e,0xf56f
    .hword 0xf580,0xf581,0xf582,0xf583,0xf584,0xf585,0xf586,0xf587,0xf588,0xf589,0xf58a,0xf58b,0xf58c,0xf58d,0xf58e,0xf58f
    .hword 0xf5a0,0xf5a1,0xf5a2,0xf5a3,0xf5a4,0xf5a5,0xf5a6,0xf5a7,0xf5a8,0xf5a9,0xf5aa,0xf5ab,0xf5ac,0xf5ad,0xf5ae,0xf5af
    .hword 0xf5c0,0xf5c1,0xf5c2,0xf5c3,0xf5c4,0xf5c5,0xf5c6,0xf5c7,0xf5c8,0xf5c9,0xf5ca,0xf5cb,0xf5cc,0xf5cd,0xf5ce,0xf5cf
    .hword 0xf5e0,0xf5e1,0xf5e2,0xf5e3,0xf5e4,0xf5e5,0xf5e6,0xf5e7,0xf5e8,0xf5e9,0xf5ea,0xf5eb,0xf5ec,0xf5ed,0xf5ee,0xf5ef
    .hword 0xf600,0xf601,0xf602,0xf603,0xf604,0xf605,0xf606,0xf607,0xf608,0xf609,0xf60a,0xf60b,0xf60c,0xf60d,0xf60e,0xf60f
    .hword 0xf620,0xf621,0xf622,0xf623,0xf624,0xf625,0xf626,0xf627,0xf628,0xf629,0xf62a,0xf62b,0xf62c,0xf62d,0xf62e,0xf62f
    .hword 0xf640,0xf641,0xf642,0xf643,0xf644,0xf645,0xf646,0xf647,0xf648,0xf649,0xf64a,0xf64b,0xf64c,0xf64d,0xf64e,0xf64f
    .hword 0xf660,0xf661,0xf662,0xf663,0xf664,0xf665,0xf666,0xf667,0xf668,0xf669,0xf66a,0xf66b,0xf66c,0xf66d,0xf66e,0xf66f
    .hword 0xf680,0xf681,0xf682,0xf683,0xf684,0xf685,0xf686,0xf687,0xf688,0xf689,0xf68a,0xf68b,0xf68c,0xf68d,0xf68e,0xf68f
    .hword 0xf6a0,0xf6a1,0xf6a2,0xf6a3,0xf6a4,0xf6a5,0xf6a6,0xf6a7,0xf6a8,0xf6a9,0xf6aa,0xf6ab,0xf6ac,0xf6ad,0xf6ae,0xf6af
    .hword 0xf6c0,0xf6c1,0xf6c2,0xf6c3,0xf6c4,0xf6c5,0xf6c6,0xf6c7,0xf6c8,0xf6c9,0xf6ca,0xf6cb,0xf6cc,0xf6cd,0xf6ce,0xf6cf
    .hword 0xf6e0,0xf6e1,0xf6e2,0xf6e3,0xf6e4,0xf6e5,0xf6e6,0xf6e7,0xf6e8,0xf6e9,0xf6ea,0xf6eb,0xf6ec,0xf6ed,0xf6ee,0xf6ef
    .hword 0xf700,0xf701,0xf702,0xf703,0xf704,0xf705,0xf706,0xf707,0xf708,0xf709,0xf70a,0xf70b,0xf70c,0xf70d,0xf70e,0xf70f
    .hword 0xf720,0xf721,0xf722,0xf723,0xf724,0xf725,0xf726,0xf727,0xf728,0xf729,0xf72a,0xf72b,0xf72c,0xf72d,0xf72e,0xf72f
    .hword 0xf740,0xf741,0xf742,0xf743,0xf744,0xf745,0xf746,0xf747,0xf748,0xf749,0xf74a,0xf74b,0xf74c,0xf74d,0xf74e,0xf74f
    .hword 0xf760,0xf761,0xf762,0xf763,0xf764,0xf765,0xf766,0xf767,0xf768,0xf769,0xf76a,0xf76b,0xf76c,0xf76d,0xf76e,0xf76f
    .hword 0xf780,0xf781,0xf782,0xf783,0xf784,0xf785,0xf786,0xf787,0xf788,0xf789,0xf78a,0xf78b,0xf78c,0xf78d,0xf78e,0xf78f
    .hword 0xf7a0,0xf7a1,0xf7a2,0xf7a3,0xf7a4,0xf7a5,0xf7a6,0xf7a7,0xf7a8,0xf7a9,0xf7aa,0xf7ab,0xf7ac,0xf7ad,0xf7ae,0xf7af
    .hword 0xf7c0,0xf7c1,0xf7c2,0xf7c3,0xf7c4,0xf7c5,0xf7c6,0xf7c7,0xf7c8,0xf7c9,0xf7ca,0xf7cb,0xf7cc,0xf7cd,0xf7ce,0xf7cf
    .hword 0xf7e0,0xf7e1,0xf7e2,0xf7e3,0xf7e4,0xf7e5,0xf7e6,0xf7e7,0xf7e8,0xf7e9,0xf7ea,0xf7eb,0xf7ec,0xf7ed,0xf7ee,0xf7ef
    .hword 0x8000,0x8001,0x8002,0x8003,0x8004,0x8005,0x8006,0x8007,0x8008,0x8009,0x800a,0x800b,0x800c,0x800d,0x800e,0x800f
    .hword 0x8020,0x8021,0x8022,0x8023,0x8024,0x8025,0x8026,0x8027,0x8028,0x8029,0x802a,0x802b,0x802c,0x802d,0x802e,0x802f
    .hword 0x8040,0x8041,0x8042,0x8043,0x8044,0x8045,0x8046,0x8047,0x8048,0x8049,0x804a,0x804b,0x804c,0x804d,0x804e,0x804f
    .hword 0x8060,0x8061,0x8062,0x8063,0x8064,0x8065,0x8066,0x8067,0x8068,0x8069,0x806a,0x806b,0x806c,0x806d,0x806e,0x806f
    .hword 0x8080,0x8081,0x8082,0x8083,0x8084,0x8085,0x8086,0x8087,0x8088,0x8089,0x808a,0x808b,0x808c,0x808d,0x808e,0x808f
    .hword 0x80a0,0x80a1,0x80a2,0x80a3,0x80a4,0x80a5,0x80a6,0x80a7,0x80a8,0x80a9,0x80aa,0x80ab,0x80ac,0x80ad,0x80ae,0x80af
    .hword 0x80c0,0x80c1,0x80c2,0x80c3,0x80c4,0x80c5,0x80c6,0x80c7,0x80c8,0x80c9,0x80ca,0x80cb,0x80cc,0x80cd,0x80ce,0x80cf
    .hword 0x80e0,0x80e1,0x80e2,0x80e3,0x80e4,0x80e5,0x80e6,0x80e7,0x80e8,0x80e9,0x80ea,0x80eb,0x80ec,0x80ed,0x80ee,0x80ef
    .hword 0x8100,0x8101,0x8102,0x8103,0x8104,0x8105,0x8106,0x8107,0x8108,0x8109,0x810a,0x810b,0x810c,0x810d,0x810e,0x810f
    .hword 0x8120,0x8121,0x8122,0x8123,0x8124,0x8125,0x8126,0x8127,0x8128,0x8129,0x812a,0x812b,0x812c,0x812d,0x812e,0x812f
    .hword 0x8140,0x8141,0x8142,0x8143,0x8144,0x8145,0x8146,0x8147,0x8148,0x8149,0x814a,0x814b,0x814c,0x814d,0x814e,0x814f
    .hword 0x8160,0x8161,0x8162,0x8163,0x8164,0x8165,0x8166,0x8167,0x8168,0x8169,0x816a,0x816b,0x816c,0x816d,0x816e,0x816f
    .hword 0x8180,0x8181,0x8182,0x8183,0x8184,0x8185,0x8186,0x8187,0x8188,0x8189,0x818a,0x818b,0x818c,0x818d,0x818e,0x818f
    .hword 0x81a0,0x81a1,0x81a2,0x81a3,0x81a4,0x81a5,0x81a6,0x81a7,0x81a8,0x81a9,0x81aa,0x81ab,0x81ac,0x81ad,0x81ae,0x81af
    .hword 0x81c0,0x81c1,0x81c2,0x81c3,0x81c4,0x81c5,0x81c6,0x81c7,0x81c8,0x81c9,0x81ca,0x81cb,0x81cc,0x81cd,0x81ce,0x81cf
    .hword 0x81e0,0x81e1,0x81e2,0x81e3,0x81e4,0x81e5,0x81e6,0x81e7,0x81e8,0x81e9,0x81ea,0x81eb,0x81ec,0x81ed,0x81ee,0x81ef
    .hword 0x8200,0x8201,0x8202,0x8203,0x8204,0x8205,0x8206,0x8207,0x8208,0x8209,0x820a,0x820b,0x820c,0x820d,0x820e,0x820f
    .hword 0x8220,0x8221,0x8222,0x8223,0x8224,0x8225,0x8226,0x8227,0x8228,0x8229,0x822a,0x822b,0x822c,0x822d,0x822e,0x822f
    .hword 0x8240,0x8241,0x8242,0x8243,0x8244,0x8245,0x8246,0x8247,0x8248,0x8249,0x824a,0x824b,0x824c,0x824d,0x824e,0x824f
    .hword 0x8260,0x8261,0x8262,0x8263,0x8264,0x8265,0x8266,0x8267,0x8268,0x8269,0x826a,0x826b,0x826c,0x826d,0x826e,0x826f
    .hword 0x8280,0x8281,0x8282,0x8283,0x8284,0x8285,0x8286,0x8287,0x8288,0x8289,0x828a,0x828b,0x828c,0x828d,0x828e,0x828f
    .hword 0x82a0,0x82a1,0x82a2,0x82a3,0x82a4,0x82a5,0x82a6,0x82a7,0x82a8,0x82a9,0x82aa,0x82ab,0x82ac,0x82ad,0x82ae,0x82af
    .hword 0x82c0,0x82c1,0x82c2,0x82c3,0x82c4,0x82c5,0x82c6,0x82c7,0x82c8,0x82c9,0x82ca,0x82cb,0x82cc,0x82cd,0x82ce,0x82cf
    .hword 0x82e0,0x82e1,0x82e2,0x82e3,0x82e4,0x82e5,0x82e6,0x82e7,0x82e8,0x82e9,0x82ea,0x82eb,0x82ec,0x82ed,0x82ee,0x82ef
    .hword 0x8300,0x8301,0x8302,0x8303,0x8304,0x8305,0x8306,0x8307,0x8308,0x8309,0x830a,0x830b,0x830c,0x830d,0x830e,0x830f
    .hword 0x8320,0x8321,0x8322,0x8323,0x8324,0x8325,0x8326,0x8327,0x8328,0x8329,0x832a,0x832b,0x832c,0x832d,0x832e,0x832f
    .hword 0x8340,0x8341,0x8342,0x8343,0x8344,0x8345,0x8346,0x8347,0x8348,0x8349,0x834a,0x834b,0x834c,0x834d,0x834e,0x834f
    .hword 0x8360,0x8361,0x8362,0x8363,0x8364,0x8365,0x8366,0x8367,0x8368,0x8369,0x836a,0x836b,0x836c,0x836d,0x836e,0x836f
    .hword 0x8380,0x8381,0x8382,0x8383,0x8384,0x8385,0x8386,0x8387,0x8388,0x8389,0x838a,0x838b,0x838c,0x838d,0x838e,0x838f
    .hword 0x83a0,0x83a1,0x83a2,0x83a3,0x83a4,0x83a5,0x83a6,0x83a7,0x83a8,0x83a9,0x83aa,0x83ab,0x83ac,0x83ad,0x83ae,0x83af
    .hword 0x83c0,0x83c1,0x83c2,0x83c3,0x83c4,0x83c5,0x83c6,0x83c7,0x83c8,0x83c9,0x83ca,0x83cb,0x83cc,0x83cd,0x83ce,0x83cf
    .hword 0x83e0,0x83e1,0x83e2,0x83e3,0x83e4,0x83e5,0x83e6,0x83e7,0x83e8,0x83e9,0x83ea,0x83eb,0x83ec,0x83ed,0x83ee,0x83ef
    .hword 0x9000,0x9001,0x9002,0x9003,0x9004,0x9005,0x9006,0x9007,0x9008,0x9009,0x900a,0x900b,0x900c,0x900d,0x900e,0x900f
    .hword 0x9020,0x9021,0x9022,0x9023,0x9024,0x9025,0x9026,0x9027,0x9028,0x9029,0x902a,0x902b,0x902c,0x902d,0x902e,0x902f
    .hword 0x9040,0x9041,0x9042,0x9043,0x9044,0x9045,0x9046,0x9047,0x9048,0x9049,0x904a,0x904b,0x904c,0x904d,0x904e,0x904f
    .hword 0x9060,0x9061,0x9062,0x9063,0x9064,0x9065,0x9066,0x9067,0x9068,0x9069,0x906a,0x906b,0x906c,0x906d,0x906e,0x906f
    .hword 0x9080,0x9081,0x9082,0x9083,0x9084,0x9085,0x9086,0x9087,0x9088,0x9089,0x908a,0x908b,0x908c,0x908d,0x908e,0x908f
    .hword 0x90a0,0x90a1,0x90a2,0x90a3,0x90a4,0x90a5,0x90a6,0x90a7,0x90a8,0x90a9,0x90aa,0x90ab,0x90ac,0x90ad,0x90ae,0x90af
    .hword 0x90c0,0x90c1,0x90c2,0x90c3,0x90c4,0x90c5,0x90c6,0x90c7,0x90c8,0x90c9,0x90ca,0x90cb,0x90cc,0x90cd,0x90ce,0x90cf
    .hword 0x90e0,0x90e1,0x90e2,0x90e3,0x90e4,0x90e5,0x90e6,0x90e7,0x90e8,0x90e9,0x90ea,0x90eb,0x90ec,0x90ed,0x90ee,0x90ef
    .hword 0x9100,0x9101,0x9102,0x9103,0x9104,0x9105,0x9106,0x9107,0x9108,0x9109,0x910a,0x910b,0x910c,0x910d,0x910e,0x910f
    .hword 0x9120,0x9121,0x9122,0x9123,0x9124,0x9125,0x9126,0x9127,0x9128,0x9129,0x912a,0x912b,0x912c,0x912d,0x912e,0x912f
    .hword 0x9140,0x9141,0x9142,0x9143,0x9144,0x9145,0x9146,0x9147,0x9148,0x9149,0x914a,0x914b,0x914c,0x914d,0x914e,0x914f
    .hword 0x9160,0x9161,0x9162,0x9163,0x9164,0x9165,0x9166,0x9167,0x9168,0x9169,0x916a,0x916b,0x916c,0x916d,0x916e,0x916f
    .hword 0x9180,0x9181,0x9182,0x9183,0x9184,0x9185,0x9186,0x9187,0x9188,0x9189,0x918a,0x918b,0x918c,0x918d,0x918e,0x918f
    .hword 0x91a0,0x91a1,0x91a2,0x91a3,0x91a4,0x91a5,0x91a6,0x91a7,0x91a8,0x91a9,0x91aa,0x91ab,0x91ac,0x91ad,0x91ae,0x91af
    .hword 0x91c0,0x91c1,0x91c2,0x91c3,0x91c4,0x91c5,0x91c6,0x91c7,0x91c8,0x91c9,0x91ca,0x91cb,0x91cc,0x91cd,0x91ce,0x91cf
    .hword 0x91e0,0x91e1,0x91e2,0x91e3,0x91e4,0x91e5,0x91e6,0x91e7,0x91e8,0x91e9,0x91ea,0x91eb,0x91ec,0x91ed,0x91ee,0x91ef
    .hword 0x9200,0x9201,0x9202,0x9203,0x9204,0x9205,0x9206,0x9207,0x9208,0x9209,0x920a,0x920b,0x920c,0x920d,0x920e,0x920f
    .hword 0x9220,0x9221,0x9222,0x9223,0x9224,0x9225,0x9226,0x9227,0x9228,0x9229,0x922a,0x922b,0x922c,0x922d,0x922e,0x922f
    .hword 0x9240,0x9241,0x9242,0x9243,0x9244,0x9245,0x9246,0x9247,0x9248,0x9249,0x924a,0x924b,0x924c,0x924d,0x924e,0x924f
    .hword 0x9260,0x9261,0x9262,0x9263,0x9264,0x9265,0x9266,0x9267,0x9268,0x9269,0x926a,0x926b,0x926c,0x926d,0x926e,0x926f
    .hword 0x9280,0x9281,0x9282,0x9283,0x9284,0x9285,0x9286,0x9287,0x9288,0x9289,0x928a,0x928b,0x928c,0x928d,0x928e,0x928f
    .hword 0x92a0,0x92a1,0x92a2,0x92a3,0x92a4,0x92a5,0x92a6,0x92a7,0x92a8,0x92a9,0x92aa,0x92ab,0x92ac,0x92ad,0x92ae,0x92af
    .hword 0x92c0,0x92c1,0x92c2,0x92c3,0x92c4,0x92c5,0x92c6,0x92c7,0x92c8,0x92c9,0x92ca,0x92cb,0x92cc,0x92cd,0x92ce,0x92cf
    .hword 0x92e0,0x92e1,0x92e2,0x92e3,0x92e4,0x92e5,0x92e6,0x92e7,0x92e8,0x92e9,0x92ea,0x92eb,0x92ec,0x92ed,0x92ee,0x92ef
    .hword 0x9300,0x9301,0x9302,0x9303,0x9304,0x9305,0x9306,0x9307,0x9308,0x9309,0x930a,0x930b,0x930c,0x930d,0x930e,0x930f
    .hword 0x9320,0x9321,0x9322,0x9323,0x9324,0x9325,0x9326,0x9327,0x9328,0x9329,0x932a,0x932b,0x932c,0x932d,0x932e,0x932f
    .hword 0x9340,0x9341,0x9342,0x9343,0x9344,0x9345,0x9346,0x9347,0x9348,0x9349,0x934a,0x934b,0x934c,0x934d,0x934e,0x934f
    .hword 0x9360,0x9361,0x9362,0x9363,0x9364,0x9365,0x9366,0x9367,0x9368,0x9369,0x936a,0x936b,0x936c,0x936d,0x936e,0x936f
    .hword 0x9380,0x9381,0x9382,0x9383,0x9384,0x9385,0x9386,0x9387,0x9388,0x9389,0x938a,0x938b,0x938c,0x938d,0x938e,0x938f
    .hword 0x93a0,0x93a1,0x93a2,0x93a3,0x93a4,0x93a5,0x93a6,0x93a7,0x93a8,0x93a9,0x93aa,0x93ab,0x93ac,0x93ad,0x93ae,0x93af
    .hword 0x93c0,0x93c1,0x93c2,0x93c3,0x93c4,0x93c5,0x93c6,0x93c7,0x93c8,0x93c9,0x93ca,0x93cb,0x93cc,0x93cd,0x93ce,0x93cf
    .hword 0x93e0,0x93e1,0x93e2,0x93e3,0x93e4,0x93e5,0x93e6,0x93e7,0x93e8,0x93e9,0x93ea,0x93eb,0x93ec,0x93ed,0x93ee,0x93ef
    .hword 0xa000,0xa001,0xa002,0xa003,0xa004,0xa005,0xa006,0xa007,0xa008,0xa009,0xa00a,0xa00b,0xa00c,0xa00d,0xa00e,0xa00f
    .hword 0xa020,0xa021,0xa022,0xa023,0xa024,0xa025,0xa026,0xa027,0xa028,0xa029,0xa02a,0xa02b,0xa02c,0xa02d,0xa02e,0xa02f
    .hword 0xa040,0xa041,0xa042,0xa043,0xa044,0xa045,0xa046,0xa047,0xa048,0xa049,0xa04a,0xa04b,0xa04c,0xa04d,0xa04e,0xa04f
    .hword 0xa060,0xa061,0xa062,0xa063,0xa064,0xa065,0xa066,0xa067,0xa068,0xa069,0xa06a,0xa06b,0xa06c,0xa06d,0xa06e,0xa06f
    .hword 0xa080,0xa081,0xa082,0xa083,0xa084,0xa085,0xa086,0xa087,0xa088,0xa089,0xa08a,0xa08b,0xa08c,0xa08d,0xa08e,0xa08f
    .hword 0xa0a0,0xa0a1,0xa0a2,0xa0a3,0xa0a4,0xa0a5,0xa0a6,0xa0a7,0xa0a8,0xa0a9,0xa0aa,0xa0ab,0xa0ac,0xa0ad,0xa0ae,0xa0af
    .hword 0xa0c0,0xa0c1,0xa0c2,0xa0c3,0xa0c4,0xa0c5,0xa0c6,0xa0c7,0xa0c8,0xa0c9,0xa0ca,0xa0cb,0xa0cc,0xa0cd,0xa0ce,0xa0cf
    .hword 0xa0e0,0xa0e1,0xa0e2,0xa0e3,0xa0e4,0xa0e5,0xa0e6,0xa0e7,0xa0e8,0xa0e9,0xa0ea,0xa0eb,0xa0ec,0xa0ed,0xa0ee,0xa0ef
    .hword 0xa100,0xa101,0xa102,0xa103,0xa104,0xa105,0xa106,0xa107,0xa108,0xa109,0xa10a,0xa10b,0xa10c,0xa10d,0xa10e,0xa10f
    .hword 0xa120,0xa121,0xa122,0xa123,0xa124,0xa125,0xa126,0xa127,0xa128,0xa129,0xa12a,0xa12b,0xa12c,0xa12d,0xa12e,0xa12f
    .hword 0xa140,0xa141,0xa142,0xa143,0xa144,0xa145,0xa146,0xa147,0xa148,0xa149,0xa14a,0xa14b,0xa14c,0xa14d,0xa14e,0xa14f
    .hword 0xa160,0xa161,0xa162,0xa163,0xa164,0xa165,0xa166,0xa167,0xa168,0xa169,0xa16a,0xa16b,0xa16c,0xa16d,0xa16e,0xa16f
    .hword 0xa180,0xa181,0xa182,0xa183,0xa184,0xa185,0xa186,0xa187,0xa188,0xa189,0xa18a,0xa18b,0xa18c,0xa18d,0xa18e,0xa18f
    .hword 0xa1a0,0xa1a1,0xa1a2,0xa1a3,0xa1a4,0xa1a5,0xa1a6,0xa1a7,0xa1a8,0xa1a9,0xa1aa,0xa1ab,0xa1ac,0xa1ad,0xa1ae,0xa1af
    .hword 0xa1c0,0xa1c1,0xa1c2,0xa1c3,0xa1c4,0xa1c5,0xa1c6,0xa1c7,0xa1c8,0xa1c9,0xa1ca,0xa1cb,0xa1cc,0xa1cd,0xa1ce,0xa1cf
    .hword 0xa1e0,0xa1e1,0xa1e2,0xa1e3,0xa1e4,0xa1e5,0xa1e6,0xa1e7,0xa1e8,0xa1e9,0xa1ea,0xa1eb,0xa1ec,0xa1ed,0xa1ee,0xa1ef
    .hword 0xa200,0xa201,0xa202,0xa203,0xa204,0xa205,0xa206,0xa207,0xa208,0xa209,0xa20a,0xa20b,0xa20c,0xa20d,0xa20e,0xa20f
    .hword 0xa220,0xa221,0xa222,0xa223,0xa224,0xa225,0xa226,0xa227,0xa228,0xa229,0xa22a,0xa22b,0xa22c,0xa22d,0xa22e,0xa22f
    .hword 0xa240,0xa241,0xa242,0xa243,0xa244,0xa245,0xa246,0xa247,0xa248,0xa249,0xa24a,0xa24b,0xa24c,0xa24d,0xa24e,0xa24f
    .hword 0xa260,0xa261,0xa262,0xa263,0xa264,0xa265,0xa266,0xa267,0xa268,0xa269,0xa26a,0xa26b,0xa26c,0xa26d,0xa26e,0xa26f
    .hword 0xa280,0xa281,0xa282,0xa283,0xa284,0xa285,0xa286,0xa287,0xa288,0xa289,0xa28a,0xa28b,0xa28c,0xa28d,0xa28e,0xa28f
    .hword 0xa2a0,0xa2a1,0xa2a2,0xa2a3,0xa2a4,0xa2a5,0xa2a6,0xa2a7,0xa2a8,0xa2a9,0xa2aa,0xa2ab,0xa2ac,0xa2ad,0xa2ae,0xa2af
    .hword 0xa2c0,0xa2c1,0xa2c2,0xa2c3,0xa2c4,0xa2c5,0xa2c6,0xa2c7,0xa2c8,0xa2c9,0xa2ca,0xa2cb,0xa2cc,0xa2cd,0xa2ce,0xa2cf
    .hword 0xa2e0,0xa2e1,0xa2e2,0xa2e3,0xa2e4,0xa2e5,0xa2e6,0xa2e7,0xa2e8,0xa2e9,0xa2ea,0xa2eb,0xa2ec,0xa2ed,0xa2ee,0xa2ef
    .hword 0xa300,0xa301,0xa302,0xa303,0xa304,0xa305,0xa306,0xa307,0xa308,0xa309,0xa30a,0xa30b,0xa30c,0xa30d,0xa30e,0xa30f
    .hword 0xa320,0xa321,0xa322,0xa323,0xa324,0xa325,0xa326,0xa327,0xa328,0xa329,0xa32a,0xa32b,0xa32c,0xa32d,0xa32e,0xa32f
    .hword 0xa340,0xa341,0xa342,0xa343,0xa344,0xa345,0xa346,0xa347,0xa348,0xa349,0xa34a,0xa34b,0xa34c,0xa34d,0xa34e,0xa34f
    .hword 0xa360,0xa361,0xa362,0xa363,0xa364,0xa365,0xa366,0xa367,0xa368,0xa369,0xa36a,0xa36b,0xa36c,0xa36d,0xa36e,0xa36f
    .hword 0xa380,0xa381,0xa382,0xa383,0xa384,0xa385,0xa386,0xa387,0xa388,0xa389,0xa38a,0xa38b,0xa38c,0xa38d,0xa38e,0xa38f
    .hword 0xa3a0,0xa3a1,0xa3a2,0xa3a3,0xa3a4,0xa3a5,0xa3a6,0xa3a7,0xa3a8,0xa3a9,0xa3aa,0xa3ab,0xa3ac,0xa3ad,0xa3ae,0xa3af
    .hword 0xa3c0,0xa3c1,0xa3c2,0xa3c3,0xa3c4,0xa3c5,0xa3c6,0xa3c7,0xa3c8,0xa3c9,0xa3ca,0xa3cb,0xa3cc,0xa3cd,0xa3ce,0xa3cf
    .hword 0xa3e0,0xa3e1,0xa3e2,0xa3e3,0xa3e4,0xa3e5,0xa3e6,0xa3e7,0xa3e8,0xa3e9,0xa3ea,0xa3eb,0xa3ec,0xa3ed,0xa3ee,0xa3ef
    .hword 0xb000,0xb001,0xb002,0xb003,0xb004,0xb005,0xb006,0xb007,0xb008,0xb009,0xb00a,0xb00b,0xb00c,0xb00d,0xb00e,0xb00f
    .hword 0xb020,0xb021,0xb022,0xb023,0xb024,0xb025,0xb026,0xb027,0xb028,0xb029,0xb02a,0xb02b,0xb02c,0xb02d,0xb02e,0xb02f
    .hword 0xb040,0xb041,0xb042,0xb043,0xb044,0xb045,0xb046,0xb047,0xb048,0xb049,0xb04a,0xb04b,0xb04c,0xb04d,0xb04e,0xb04f
    .hword 0xb060,0xb061,0xb062,0xb063,0xb064,0xb065,0xb066,0xb067,0xb068,0xb069,0xb06a,0xb06b,0xb06c,0xb06d,0xb06e,0xb06f
    .hword 0xb080,0xb081,0xb082,0xb083,0xb084,0xb085,0xb086,0xb087,0xb088,0xb089,0xb08a,0xb08b,0xb08c,0xb08d,0xb08e,0xb08f
    .hword 0xb0a0,0xb0a1,0xb0a2,0xb0a3,0xb0a4,0xb0a5,0xb0a6,0xb0a7,0xb0a8,0xb0a9,0xb0aa,0xb0ab,0xb0ac,0xb0ad,0xb0ae,0xb0af
    .hword 0xb0c0,0xb0c1,0xb0c2,0xb0c3,0xb0c4,0xb0c5,0xb0c6,0xb0c7,0xb0c8,0xb0c9,0xb0ca,0xb0cb,0xb0cc,0xb0cd,0xb0ce,0xb0cf
    .hword 0xb0e0,0xb0e1,0xb0e2,0xb0e3,0xb0e4,0xb0e5,0xb0e6,0xb0e7,0xb0e8,0xb0e9,0xb0ea,0xb0eb,0xb0ec,0xb0ed,0xb0ee,0xb0ef
    .hword 0xb100,0xb101,0xb102,0xb103,0xb104,0xb105,0xb106,0xb107,0xb108,0xb109,0xb10a,0xb10b,0xb10c,0xb10d,0xb10e,0xb10f
    .hword 0xb120,0xb121,0xb122,0xb123,0xb124,0xb125,0xb126,0xb127,0xb128,0xb129,0xb12a,0xb12b,0xb12c,0xb12d,0xb12e,0xb12f
    .hword 0xb140,0xb141,0xb142,0xb143,0xb144,0xb145,0xb146,0xb147,0xb148,0xb149,0xb14a,0xb14b,0xb14c,0xb14d,0xb14e,0xb14f
    .hword 0xb160,0xb161,0xb162,0xb163,0xb164,0xb165,0xb166,0xb167,0xb168,0xb169,0xb16a,0xb16b,0xb16c,0xb16d,0xb16e,0xb16f
    .hword 0xb180,0xb181,0xb182,0xb183,0xb184,0xb185,0xb186,0xb187,0xb188,0xb189,0xb18a,0xb18b,0xb18c,0xb18d,0xb18e,0xb18f
    .hword 0xb1a0,0xb1a1,0xb1a2,0xb1a3,0xb1a4,0xb1a5,0xb1a6,0xb1a7,0xb1a8,0xb1a9,0xb1aa,0xb1ab,0xb1ac,0xb1ad,0xb1ae,0xb1af
    .hword 0xb1c0,0xb1c1,0xb1c2,0xb1c3,0xb1c4,0xb1c5,0xb1c6,0xb1c7,0xb1c8,0xb1c9,0xb1ca,0xb1cb,0xb1cc,0xb1cd,0xb1ce,0xb1cf
    .hword 0xb1e0,0xb1e1,0xb1e2,0xb1e3,0xb1e4,0xb1e5,0xb1e6,0xb1e7,0xb1e8,0xb1e9,0xb1ea,0xb1eb,0xb1ec,0xb1ed,0xb1ee,0xb1ef
    .hword 0xb200,0xb201,0xb202,0xb203,0xb204,0xb205,0xb206,0xb207,0xb208,0xb209,0xb20a,0xb20b,0xb20c,0xb20d,0xb20e,0xb20f
    .hword 0xb220,0xb221,0xb222,0xb223,0xb224,0xb225,0xb226,0xb227,0xb228,0xb229,0xb22a,0xb22b,0xb22c,0xb22d,0xb22e,0xb22f
    .hword 0xb240,0xb241,0xb242,0xb243,0xb244,0xb245,0xb246,0xb247,0xb248,0xb249,0xb24a,0xb24b,0xb24c,0xb24d,0xb24e,0xb24f
    .hword 0xb260,0xb261,0xb262,0xb263,0xb264,0xb265,0xb266,0xb267,0xb268,0xb269,0xb26a,0xb26b,0xb26c,0xb26d,0xb26e,0xb26f
    .hword 0xb280,0xb281,0xb282,0xb283,0xb284,0xb285,0xb286,0xb287,0xb288,0xb289,0xb28a,0xb28b,0xb28c,0xb28d,0xb28e,0xb28f
    .hword 0xb2a0,0xb2a1,0xb2a2,0xb2a3,0xb2a4,0xb2a5,0xb2a6,0xb2a7,0xb2a8,0xb2a9,0xb2aa,0xb2ab,0xb2ac,0xb2ad,0xb2ae,0xb2af
    .hword 0xb2c0,0xb2c1,0xb2c2,0xb2c3,0xb2c4,0xb2c5,0xb2c6,0xb2c7,0xb2c8,0xb2c9,0xb2ca,0xb2cb,0xb2cc,0xb2cd,0xb2ce,0xb2cf
    .hword 0xb2e0,0xb2e1,0xb2e2,0xb2e3,0xb2e4,0xb2e5,0xb2e6,0xb2e7,0xb2e8,0xb2e9,0xb2ea,0xb2eb,0xb2ec,0xb2ed,0xb2ee,0xb2ef
    .hword 0xb300,0xb301,0xb302,0xb303,0xb304,0xb305,0xb306,0xb307,0xb308,0xb309,0xb30a,0xb30b,0xb30c,0xb30d,0xb30e,0xb30f
    .hword 0xb320,0xb321,0xb322,0xb323,0xb324,0xb325,0xb326,0xb327,0xb328,0xb329,0xb32a,0xb32b,0xb32c,0xb32d,0xb32e,0xb32f
    .hword 0xb340,0xb341,0xb342,0xb343,0xb344,0xb345,0xb346,0xb347,0xb348,0xb349,0xb34a,0xb34b,0xb34c,0xb34d,0xb34e,0xb34f
    .hword 0xb360,0xb361,0xb362,0xb363,0xb364,0xb365,0xb366,0xb367,0xb368,0xb369,0xb36a,0xb36b,0xb36c,0xb36d,0xb36e,0xb36f
    .hword 0xb380,0xb381,0xb382,0xb383,0xb384,0xb385,0xb386,0xb387,0xb388,0xb389,0xb38a,0xb38b,0xb38c,0xb38d,0xb38e,0xb38f
    .hword 0xb3a0,0xb3a1,0xb3a2,0xb3a3,0xb3a4,0xb3a5,0xb3a6,0xb3a7,0xb3a8,0xb3a9,0xb3aa,0xb3ab,0xb3ac,0xb3ad,0xb3ae,0xb3af
    .hword 0xb3c0,0xb3c1,0xb3c2,0xb3c3,0xb3c4,0xb3c5,0xb3c6,0xb3c7,0xb3c8,0xb3c9,0xb3ca,0xb3cb,0xb3cc,0xb3cd,0xb3ce,0xb3cf
    .hword 0xb3e0,0xb3e1,0xb3e2,0xb3e3,0xb3e4,0xb3e5,0xb3e6,0xb3e7,0xb3e8,0xb3e9,0xb3ea,0xb3eb,0xb3ec,0xb3ed,0xb3ee,0xb3ef
    .hword 0xc000,0xc001,0xc002,0xc003,0xc004,0xc005,0xc006,0xc007,0xc008,0xc009,0xc00a,0xc00b,0xc00c,0xc00d,0xc00e,0xc00f
    .hword 0xc020,0xc021,0xc022,0xc023,0xc024,0xc025,0xc026,0xc027,0xc028,0xc029,0xc02a,0xc02b,0xc02c,0xc02d,0xc02e,0xc02f
    .hword 0xc040,0xc041,0xc042,0xc043,0xc044,0xc045,0xc046,0xc047,0xc048,0xc049,0xc04a,0xc04b,0xc04c,0xc04d,0xc04e,0xc04f
    .hword 0xc060,0xc061,0xc062,0xc063,0xc064,0xc065,0xc066,0xc067,0xc068,0xc069,0xc06a,0xc06b,0xc06c,0xc06d,0xc06e,0xc06f
    .hword 0xc080,0xc081,0xc082,0xc083,0xc084,0xc085,0xc086,0xc087,0xc088,0xc089,0xc08a,0xc08b,0xc08c,0xc08d,0xc08e,0xc08f
    .hword 0xc0a0,0xc0a1,0xc0a2,0xc0a3,0xc0a4,0xc0a5,0xc0a6,0xc0a7,0xc0a8,0xc0a9,0xc0aa,0xc0ab,0xc0ac,0xc0ad,0xc0ae,0xc0af
    .hword 0xc0c0,0xc0c1,0xc0c2,0xc0c3,0xc0c4,0xc0c5,0xc0c6,0xc0c7,0xc0c8,0xc0c9,0xc0ca,0xc0cb,0xc0cc,0xc0cd,0xc0ce,0xc0cf
    .hword 0xc0e0,0xc0e1,0xc0e2,0xc0e3,0xc0e4,0xc0e5,0xc0e6,0xc0e7,0xc0e8,0xc0e9,0xc0ea,0xc0eb,0xc0ec,0xc0ed,0xc0ee,0xc0ef
    .hword 0xc100,0xc101,0xc102,0xc103,0xc104,0xc105,0xc106,0xc107,0xc108,0xc109,0xc10a,0xc10b,0xc10c,0xc10d,0xc10e,0xc10f
    .hword 0xc120,0xc121,0xc122,0xc123,0xc124,0xc125,0xc126,0xc127,0xc128,0xc129,0xc12a,0xc12b,0xc12c,0xc12d,0xc12e,0xc12f
    .hword 0xc140,0xc141,0xc142,0xc143,0xc144,0xc145,0xc146,0xc147,0xc148,0xc149,0xc14a,0xc14b,0xc14c,0xc14d,0xc14e,0xc14f
    .hword 0xc160,0xc161,0xc162,0xc163,0xc164,0xc165,0xc166,0xc167,0xc168,0xc169,0xc16a,0xc16b,0xc16c,0xc16d,0xc16e,0xc16f
    .hword 0xc180,0xc181,0xc182,0xc183,0xc184,0xc185,0xc186,0xc187,0xc188,0xc189,0xc18a,0xc18b,0xc18c,0xc18d,0xc18e,0xc18f
    .hword 0xc1a0,0xc1a1,0xc1a2,0xc1a3,0xc1a4,0xc1a5,0xc1a6,0xc1a7,0xc1a8,0xc1a9,0xc1aa,0xc1ab,0xc1ac,0xc1ad,0xc1ae,0xc1af
    .hword 0xc1c0,0xc1c1,0xc1c2,0xc1c3,0xc1c4,0xc1c5,0xc1c6,0xc1c7,0xc1c8,0xc1c9,0xc1ca,0xc1cb,0xc1cc,0xc1cd,0xc1ce,0xc1cf
    .hword 0xc1e0,0xc1e1,0xc1e2,0xc1e3,0xc1e4,0xc1e5,0xc1e6,0xc1e7,0xc1e8,0xc1e9,0xc1ea,0xc1eb,0xc1ec,0xc1ed,0xc1ee,0xc1ef
    .hword 0xc200,0xc201,0xc202,0xc203,0xc204,0xc205,0xc206,0xc207,0xc208,0xc209,0xc20a,0xc20b,0xc20c,0xc20d,0xc20e,0xc20f
    .hword 0xc220,0xc221,0xc222,0xc223,0xc224,0xc225,0xc226,0xc227,0xc228,0xc229,0xc22a,0xc22b,0xc22c,0xc22d,0xc22e,0xc22f
    .hword 0xc240,0xc241,0xc242,0xc243,0xc244,0xc245,0xc246,0xc247,0xc248,0xc249,0xc24a,0xc24b,0xc24c,0xc24d,0xc24e,0xc24f
    .hword 0xc260,0xc261,0xc262,0xc263,0xc264,0xc265,0xc266,0xc267,0xc268,0xc269,0xc26a,0xc26b,0xc26c,0xc26d,0xc26e,0xc26f
    .hword 0xc280,0xc281,0xc282,0xc283,0xc284,0xc285,0xc286,0xc287,0xc288,0xc289,0xc28a,0xc28b,0xc28c,0xc28d,0xc28e,0xc28f
    .hword 0xc2a0,0xc2a1,0xc2a2,0xc2a3,0xc2a4,0xc2a5,0xc2a6,0xc2a7,0xc2a8,0xc2a9,0xc2aa,0xc2ab,0xc2ac,0xc2ad,0xc2ae,0xc2af
    .hword 0xc2c0,0xc2c1,0xc2c2,0xc2c3,0xc2c4,0xc2c5,0xc2c6,0xc2c7,0xc2c8,0xc2c9,0xc2ca,0xc2cb,0xc2cc,0xc2cd,0xc2ce,0xc2cf
    .hword 0xc2e0,0xc2e1,0xc2e2,0xc2e3,0xc2e4,0xc2e5,0xc2e6,0xc2e7,0xc2e8,0xc2e9,0xc2ea,0xc2eb,0xc2ec,0xc2ed,0xc2ee,0xc2ef
    .hword 0xc300,0xc301,0xc302,0xc303,0xc304,0xc305,0xc306,0xc307,0xc308,0xc309,0xc30a,0xc30b,0xc30c,0xc30d,0xc30e,0xc30f
    .hword 0xc320,0xc321,0xc322,0xc323,0xc324,0xc325,0xc326,0xc327,0xc328,0xc329,0xc32a,0xc32b,0xc32c,0xc32d,0xc32e,0xc32f
    .hword 0xc340,0xc341,0xc342,0xc343,0xc344,0xc345,0xc346,0xc347,0xc348,0xc349,0xc34a,0xc34b,0xc34c,0xc34d,0xc34e,0xc34f
    .hword 0xc360,0xc361,0xc362,0xc363,0xc364,0xc365,0xc366,0xc367,0xc368,0xc369,0xc36a,0xc36b,0xc36c,0xc36d,0xc36e,0xc36f
    .hword 0xc380,0xc381,0xc382,0xc383,0xc384,0xc385,0xc386,0xc387,0xc388,0xc389,0xc38a,0xc38b,0xc38c,0xc38d,0xc38e,0xc38f
    .hword 0xc3a0,0xc3a1,0xc3a2,0xc3a3,0xc3a4,0xc3a5,0xc3a6,0xc3a7,0xc3a8,0xc3a9,0xc3aa,0xc3ab,0xc3ac,0xc3ad,0xc3ae,0xc3af
    .hword 0xc3c0,0xc3c1,0xc3c2,0xc3c3,0xc3c4,0xc3c5,0xc3c6,0xc3c7,0xc3c8,0xc3c9,0xc3ca,0xc3cb,0xc3cc,0xc3cd,0xc3ce,0xc3cf
    .hword 0xc3e0,0xc3e1,0xc3e2,0xc3e3,0xc3e4,0xc3e5,0xc3e6,0xc3e7,0xc3e8,0xc3e9,0xc3ea,0xc3eb,0xc3ec,0xc3ed,0xc3ee,0xc3ef
    .hword 0xd000,0xd001,0xd002,0xd003,0xd004,0xd005,0xd006,0xd007,0xd008,0xd009,0xd00a,0xd00b,0xd00c,0xd00d,0xd00e,0xd00f
    .hword 0xd020,0xd021,0xd022,0xd023,0xd024,0xd025,0xd026,0xd027,0xd028,0xd029,0xd02a,0xd02b,0xd02c,0xd02d,0xd02e,0xd02f
    .hword 0xd040,0xd041,0xd042,0xd043,0xd044,0xd045,0xd046,0xd047,0xd048,0xd049,0xd04a,0xd04b,0xd04c,0xd04d,0xd04e,0xd04f
    .hword 0xd060,0xd061,0xd062,0xd063,0xd064,0xd065,0xd066,0xd067,0xd068,0xd069,0xd06a,0xd06b,0xd06c,0xd06d,0xd06e,0xd06f
    .hword 0xd080,0xd081,0xd082,0xd083,0xd084,0xd085,0xd086,0xd087,0xd088,0xd089,0xd08a,0xd08b,0xd08c,0xd08d,0xd08e,0xd08f
    .hword 0xd0a0,0xd0a1,0xd0a2,0xd0a3,0xd0a4,0xd0a5,0xd0a6,0xd0a7,0xd0a8,0xd0a9,0xd0aa,0xd0ab,0xd0ac,0xd0ad,0xd0ae,0xd0af
    .hword 0xd0c0,0xd0c1,0xd0c2,0xd0c3,0xd0c4,0xd0c5,0xd0c6,0xd0c7,0xd0c8,0xd0c9,0xd0ca,0xd0cb,0xd0cc,0xd0cd,0xd0ce,0xd0cf
    .hword 0xd0e0,0xd0e1,0xd0e2,0xd0e3,0xd0e4,0xd0e5,0xd0e6,0xd0e7,0xd0e8,0xd0e9,0xd0ea,0xd0eb,0xd0ec,0xd0ed,0xd0ee,0xd0ef
    .hword 0xd100,0xd101,0xd102,0xd103,0xd104,0xd105,0xd106,0xd107,0xd108,0xd109,0xd10a,0xd10b,0xd10c,0xd10d,0xd10e,0xd10f
    .hword 0xd120,0xd121,0xd122,0xd123,0xd124,0xd125,0xd126,0xd127,0xd128,0xd129,0xd12a,0xd12b,0xd12c,0xd12d,0xd12e,0xd12f
    .hword 0xd140,0xd141,0xd142,0xd143,0xd144,0xd145,0xd146,0xd147,0xd148,0xd149,0xd14a,0xd14b,0xd14c,0xd14d,0xd14e,0xd14f
    .hword 0xd160,0xd161,0xd162,0xd163,0xd164,0xd165,0xd166,0xd167,0xd168,0xd169,0xd16a,0xd16b,0xd16c,0xd16d,0xd16e,0xd16f
    .hword 0xd180,0xd181,0xd182,0xd183,0xd184,0xd185,0xd186,0xd187,0xd188,0xd189,0xd18a,0xd18b,0xd18c,0xd18d,0xd18e,0xd18f
    .hword 0xd1a0,0xd1a1,0xd1a2,0xd1a3,0xd1a4,0xd1a5,0xd1a6,0xd1a7,0xd1a8,0xd1a9,0xd1aa,0xd1ab,0xd1ac,0xd1ad,0xd1ae,0xd1af
    .hword 0xd1c0,0xd1c1,0xd1c2,0xd1c3,0xd1c4,0xd1c5,0xd1c6,0xd1c7,0xd1c8,0xd1c9,0xd1ca,0xd1cb,0xd1cc,0xd1cd,0xd1ce,0xd1cf
    .hword 0xd1e0,0xd1e1,0xd1e2,0xd1e3,0xd1e4,0xd1e5,0xd1e6,0xd1e7,0xd1e8,0xd1e9,0xd1ea,0xd1eb,0xd1ec,0xd1ed,0xd1ee,0xd1ef
    .hword 0xd200,0xd201,0xd202,0xd203,0xd204,0xd205,0xd206,0xd207,0xd208,0xd209,0xd20a,0xd20b,0xd20c,0xd20d,0xd20e,0xd20f
    .hword 0xd220,0xd221,0xd222,0xd223,0xd224,0xd225,0xd226,0xd227,0xd228,0xd229,0xd22a,0xd22b,0xd22c,0xd22d,0xd22e,0xd22f
    .hword 0xd240,0xd241,0xd242,0xd243,0xd244,0xd245,0xd246,0xd247,0xd248,0xd249,0xd24a,0xd24b,0xd24c,0xd24d,0xd24e,0xd24f
    .hword 0xd260,0xd261,0xd262,0xd263,0xd264,0xd265,0xd266,0xd267,0xd268,0xd269,0xd26a,0xd26b,0xd26c,0xd26d,0xd26e,0xd26f
    .hword 0xd280,0xd281,0xd282,0xd283,0xd284,0xd285,0xd286,0xd287,0xd288,0xd289,0xd28a,0xd28b,0xd28c,0xd28d,0xd28e,0xd28f
    .hword 0xd2a0,0xd2a1,0xd2a2,0xd2a3,0xd2a4,0xd2a5,0xd2a6,0xd2a7,0xd2a8,0xd2a9,0xd2aa,0xd2ab,0xd2ac,0xd2ad,0xd2ae,0xd2af
    .hword 0xd2c0,0xd2c1,0xd2c2,0xd2c3,0xd2c4,0xd2c5,0xd2c6,0xd2c7,0xd2c8,0xd2c9,0xd2ca,0xd2cb,0xd2cc,0xd2cd,0xd2ce,0xd2cf
    .hword 0xd2e0,0xd2e1,0xd2e2,0xd2e3,0xd2e4,0xd2e5,0xd2e6,0xd2e7,0xd2e8,0xd2e9,0xd2ea,0xd2eb,0xd2ec,0xd2ed,0xd2ee,0xd2ef
    .hword 0xd300,0xd301,0xd302,0xd303,0xd304,0xd305,0xd306,0xd307,0xd308,0xd309,0xd30a,0xd30b,0xd30c,0xd30d,0xd30e,0xd30f
    .hword 0xd320,0xd321,0xd322,0xd323,0xd324,0xd325,0xd326,0xd327,0xd328,0xd329,0xd32a,0xd32b,0xd32c,0xd32d,0xd32e,0xd32f
    .hword 0xd340,0xd341,0xd342,0xd343,0xd344,0xd345,0xd346,0xd347,0xd348,0xd349,0xd34a,0xd34b,0xd34c,0xd34d,0xd34e,0xd34f
    .hword 0xd360,0xd361,0xd362,0xd363,0xd364,0xd365,0xd366,0xd367,0xd368,0xd369,0xd36a,0xd36b,0xd36c,0xd36d,0xd36e,0xd36f
    .hword 0xd380,0xd381,0xd382,0xd383,0xd384,0xd385,0xd386,0xd387,0xd388,0xd389,0xd38a,0xd38b,0xd38c,0xd38d,0xd38e,0xd38f
    .hword 0xd3a0,0xd3a1,0xd3a2,0xd3a3,0xd3a4,0xd3a5,0xd3a6,0xd3a7,0xd3a8,0xd3a9,0xd3aa,0xd3ab,0xd3ac,0xd3ad,0xd3ae,0xd3af
    .hword 0xd3c0,0xd3c1,0xd3c2,0xd3c3,0xd3c4,0xd3c5,0xd3c6,0xd3c7,0xd3c8,0xd3c9,0xd3ca,0xd3cb,0xd3cc,0xd3cd,0xd3ce,0xd3cf
    .hword 0xd3e0,0xd3e1,0xd3e2,0xd3e3,0xd3e4,0xd3e5,0xd3e6,0xd3e7,0xd3e8,0xd3e9,0xd3ea,0xd3eb,0xd3ec,0xd3ed,0xd3ee,0xd3ef
    .hword 0xe000,0xe001,0xe002,0xe003,0xe004,0xe005,0xe006,0xe007,0xe008,0xe009,0xe00a,0xe00b,0xe00c,0xe00d,0xe00e,0xe00f
    .hword 0xe020,0xe021,0xe022,0xe023,0xe024,0xe025,0xe026,0xe027,0xe028,0xe029,0xe02a,0xe02b,0xe02c,0xe02d,0xe02e,0xe02f
    .hword 0xe040,0xe041,0xe042,0xe043,0xe044,0xe045,0xe046,0xe047,0xe048,0xe049,0xe04a,0xe04b,0xe04c,0xe04d,0xe04e,0xe04f
    .hword 0xe060,0xe061,0xe062,0xe063,0xe064,0xe065,0xe066,0xe067,0xe068,0xe069,0xe06a,0xe06b,0xe06c,0xe06d,0xe06e,0xe06f
    .hword 0xe080,0xe081,0xe082,0xe083,0xe084,0xe085,0xe086,0xe087,0xe088,0xe089,0xe08a,0xe08b,0xe08c,0xe08d,0xe08e,0xe08f
    .hword 0xe0a0,0xe0a1,0xe0a2,0xe0a3,0xe0a4,0xe0a5,0xe0a6,0xe0a7,0xe0a8,0xe0a9,0xe0aa,0xe0ab,0xe0ac,0xe0ad,0xe0ae,0xe0af
    .hword 0xe0c0,0xe0c1,0xe0c2,0xe0c3,0xe0c4,0xe0c5,0xe0c6,0xe0c7,0xe0c8,0xe0c9,0xe0ca,0xe0cb,0xe0cc,0xe0cd,0xe0ce,0xe0cf
    .hword 0xe0e0,0xe0e1,0xe0e2,0xe0e3,0xe0e4,0xe0e5,0xe0e6,0xe0e7,0xe0e8,0xe0e9,0xe0ea,0xe0eb,0xe0ec,0xe0ed,0xe0ee,0xe0ef
    .hword 0xe100,0xe101,0xe102,0xe103,0xe104,0xe105,0xe106,0xe107,0xe108,0xe109,0xe10a,0xe10b,0xe10c,0xe10d,0xe10e,0xe10f
    .hword 0xe120,0xe121,0xe122,0xe123,0xe124,0xe125,0xe126,0xe127,0xe128,0xe129,0xe12a,0xe12b,0xe12c,0xe12d,0xe12e,0xe12f
    .hword 0xe140,0xe141,0xe142,0xe143,0xe144,0xe145,0xe146,0xe147,0xe148,0xe149,0xe14a,0xe14b,0xe14c,0xe14d,0xe14e,0xe14f
    .hword 0xe160,0xe161,0xe162,0xe163,0xe164,0xe165,0xe166,0xe167,0xe168,0xe169,0xe16a,0xe16b,0xe16c,0xe16d,0xe16e,0xe16f
    .hword 0xe180,0xe181,0xe182,0xe183,0xe184,0xe185,0xe186,0xe187,0xe188,0xe189,0xe18a,0xe18b,0xe18c,0xe18d,0xe18e,0xe18f
    .hword 0xe1a0,0xe1a1,0xe1a2,0xe1a3,0xe1a4,0xe1a5,0xe1a6,0xe1a7,0xe1a8,0xe1a9,0xe1aa,0xe1ab,0xe1ac,0xe1ad,0xe1ae,0xe1af
    .hword 0xe1c0,0xe1c1,0xe1c2,0xe1c3,0xe1c4,0xe1c5,0xe1c6,0xe1c7,0xe1c8,0xe1c9,0xe1ca,0xe1cb,0xe1cc,0xe1cd,0xe1ce,0xe1cf
    .hword 0xe1e0,0xe1e1,0xe1e2,0xe1e3,0xe1e4,0xe1e5,0xe1e6,0xe1e7,0xe1e8,0xe1e9,0xe1ea,0xe1eb,0xe1ec,0xe1ed,0xe1ee,0xe1ef
    .hword 0xe200,0xe201,0xe202,0xe203,0xe204,0xe205,0xe206,0xe207,0xe208,0xe209,0xe20a,0xe20b,0xe20c,0xe20d,0xe20e,0xe20f
    .hword 0xe220,0xe221,0xe222,0xe223,0xe224,0xe225,0xe226,0xe227,0xe228,0xe229,0xe22a,0xe22b,0xe22c,0xe22d,0xe22e,0xe22f
    .hword 0xe240,0xe241,0xe242,0xe243,0xe244,0xe245,0xe246,0xe247,0xe248,0xe249,0xe24a,0xe24b,0xe24c,0xe24d,0xe24e,0xe24f
    .hword 0xe260,0xe261,0xe262,0xe263,0xe264,0xe265,0xe266,0xe267,0xe268,0xe269,0xe26a,0xe26b,0xe26c,0xe26d,0xe26e,0xe26f
    .hword 0xe280,0xe281,0xe282,0xe283,0xe284,0xe285,0xe286,0xe287,0xe288,0xe289,0xe28a,0xe28b,0xe28c,0xe28d,0xe28e,0xe28f
    .hword 0xe2a0,0xe2a1,0xe2a2,0xe2a3,0xe2a4,0xe2a5,0xe2a6,0xe2a7,0xe2a8,0xe2a9,0xe2aa,0xe2ab,0xe2ac,0xe2ad,0xe2ae,0xe2af
    .hword 0xe2c0,0xe2c1,0xe2c2,0xe2c3,0xe2c4,0xe2c5,0xe2c6,0xe2c7,0xe2c8,0xe2c9,0xe2ca,0xe2cb,0xe2cc,0xe2cd,0xe2ce,0xe2cf
    .hword 0xe2e0,0xe2e1,0xe2e2,0xe2e3,0xe2e4,0xe2e5,0xe2e6,0xe2e7,0xe2e8,0xe2e9,0xe2ea,0xe2eb,0xe2ec,0xe2ed,0xe2ee,0xe2ef
    .hword 0xe300,0xe301,0xe302,0xe303,0xe304,0xe305,0xe306,0xe307,0xe308,0xe309,0xe30a,0xe30b,0xe30c,0xe30d,0xe30e,0xe30f
    .hword 0xe320,0xe321,0xe322,0xe323,0xe324,0xe325,0xe326,0xe327,0xe328,0xe329,0xe32a,0xe32b,0xe32c,0xe32d,0xe32e,0xe32f
    .hword 0xe340,0xe341,0xe342,0xe343,0xe344,0xe345,0xe346,0xe347,0xe348,0xe349,0xe34a,0xe34b,0xe34c,0xe34d,0xe34e,0xe34f
    .hword 0xe360,0xe361,0xe362,0xe363,0xe364,0xe365,0xe366,0xe367,0xe368,0xe369,0xe36a,0xe36b,0xe36c,0xe36d,0xe36e,0xe36f
    .hword 0xe380,0xe381,0xe382,0xe383,0xe384,0xe385,0xe386,0xe387,0xe388,0xe389,0xe38a,0xe38b,0xe38c,0xe38d,0xe38e,0xe38f
    .hword 0xe3a0,0xe3a1,0xe3a2,0xe3a3,0xe3a4,0xe3a5,0xe3a6,0xe3a7,0xe3a8,0xe3a9,0xe3aa,0xe3ab,0xe3ac,0xe3ad,0xe3ae,0xe3af
    .hword 0xe3c0,0xe3c1,0xe3c2,0xe3c3,0xe3c4,0xe3c5,0xe3c6,0xe3c7,0xe3c8,0xe3c9,0xe3ca,0xe3cb,0xe3cc,0xe3cd,0xe3ce,0xe3cf
    .hword 0xe3e0,0xe3e1,0xe3e2,0xe3e3,0xe3e4,0xe3e5,0xe3e6,0xe3e7,0xe3e8,0xe3e9,0xe3ea,0xe3eb,0xe3ec,0xe3ed,0xe3ee,0xe3ef
    .hword 0xf000,0xf001,0xf002,0xf003,0xf004,0xf005,0xf006,0xf007,0xf008,0xf009,0xf00a,0xf00b,0xf00c,0xf00d,0xf00e,0xf00f
    .hword 0xf020,0xf021,0xf022,0xf023,0xf024,0xf025,0xf026,0xf027,0xf028,0xf029,0xf02a,0xf02b,0xf02c,0xf02d,0xf02e,0xf02f
    .hword 0xf040,0xf041,0xf042,0xf043,0xf044,0xf045,0xf046,0xf047,0xf048,0xf049,0xf04a,0xf04b,0xf04c,0xf04d,0xf04e,0xf04f
    .hword 0xf060,0xf061,0xf062,0xf063,0xf064,0xf065,0xf066,0xf067,0xf068,0xf069,0xf06a,0xf06b,0xf06c,0xf06d,0xf06e,0xf06f
    .hword 0xf080,0xf081,0xf082,0xf083,0xf084,0xf085,0xf086,0xf087,0xf088,0xf089,0xf08a,0xf08b,0xf08c,0xf08d,0xf08e,0xf08f
    .hword 0xf0a0,0xf0a1,0xf0a2,0xf0a3,0xf0a4,0xf0a5,0xf0a6,0xf0a7,0xf0a8,0xf0a9,0xf0aa,0xf0ab,0xf0ac,0xf0ad,0xf0ae,0xf0af
    .hword 0xf0c0,0xf0c1,0xf0c2,0xf0c3,0xf0c4,0xf0c5,0xf0c6,0xf0c7,0xf0c8,0xf0c9,0xf0ca,0xf0cb,0xf0cc,0xf0cd,0xf0ce,0xf0cf
    .hword 0xf0e0,0xf0e1,0xf0e2,0xf0e3,0xf0e4,0xf0e5,0xf0e6,0xf0e7,0xf0e8,0xf0e9,0xf0ea,0xf0eb,0xf0ec,0xf0ed,0xf0ee,0xf0ef
    .hword 0xf100,0xf101,0xf102,0xf103,0xf104,0xf105,0xf106,0xf107,0xf108,0xf109,0xf10a,0xf10b,0xf10c,0xf10d,0xf10e,0xf10f
    .hword 0xf120,0xf121,0xf122,0xf123,0xf124,0xf125,0xf126,0xf127,0xf128,0xf129,0xf12a,0xf12b,0xf12c,0xf12d,0xf12e,0xf12f
    .hword 0xf140,0xf141,0xf142,0xf143,0xf144,0xf145,0xf146,0xf147,0xf148,0xf149,0xf14a,0xf14b,0xf14c,0xf14d,0xf14e,0xf14f
    .hword 0xf160,0xf161,0xf162,0xf163,0xf164,0xf165,0xf166,0xf167,0xf168,0xf169,0xf16a,0xf16b,0xf16c,0xf16d,0xf16e,0xf16f
    .hword 0xf180,0xf181,0xf182,0xf183,0xf184,0xf185,0xf186,0xf187,0xf188,0xf189,0xf18a,0xf18b,0xf18c,0xf18d,0xf18e,0xf18f
    .hword 0xf1a0,0xf1a1,0xf1a2,0xf1a3,0xf1a4,0xf1a5,0xf1a6,0xf1a7,0xf1a8,0xf1a9,0xf1aa,0xf1ab,0xf1ac,0xf1ad,0xf1ae,0xf1af
    .hword 0xf1c0,0xf1c1,0xf1c2,0xf1c3,0xf1c4,0xf1c5,0xf1c6,0xf1c7,0xf1c8,0xf1c9,0xf1ca,0xf1cb,0xf1cc,0xf1cd,0xf1ce,0xf1cf
    .hword 0xf1e0,0xf1e1,0xf1e2,0xf1e3,0xf1e4,0xf1e5,0xf1e6,0xf1e7,0xf1e8,0xf1e9,0xf1ea,0xf1eb,0xf1ec,0xf1ed,0xf1ee,0xf1ef
    .hword 0xf200,0xf201,0xf202,0xf203,0xf204,0xf205,0xf206,0xf207,0xf208,0xf209,0xf20a,0xf20b,0xf20c,0xf20d,0xf20e,0xf20f
    .hword 0xf220,0xf221,0xf222,0xf223,0xf224,0xf225,0xf226,0xf227,0xf228,0xf229,0xf22a,0xf22b,0xf22c,0xf22d,0xf22e,0xf22f
    .hword 0xf240,0xf241,0xf242,0xf243,0xf244,0xf245,0xf246,0xf247,0xf248,0xf249,0xf24a,0xf24b,0xf24c,0xf24d,0xf24e,0xf24f
    .hword 0xf260,0xf261,0xf262,0xf263,0xf264,0xf265,0xf266,0xf267,0xf268,0xf269,0xf26a,0xf26b,0xf26c,0xf26d,0xf26e,0xf26f
    .hword 0xf280,0xf281,0xf282,0xf283,0xf284,0xf285,0xf286,0xf287,0xf288,0xf289,0xf28a,0xf28b,0xf28c,0xf28d,0xf28e,0xf28f
    .hword 0xf2a0,0xf2a1,0xf2a2,0xf2a3,0xf2a4,0xf2a5,0xf2a6,0xf2a7,0xf2a8,0xf2a9,0xf2aa,0xf2ab,0xf2ac,0xf2ad,0xf2ae,0xf2af
    .hword 0xf2c0,0xf2c1,0xf2c2,0xf2c3,0xf2c4,0xf2c5,0xf2c6,0xf2c7,0xf2c8,0xf2c9,0xf2ca,0xf2cb,0xf2cc,0xf2cd,0xf2ce,0xf2cf
    .hword 0xf2e0,0xf2e1,0xf2e2,0xf2e3,0xf2e4,0xf2e5,0xf2e6,0xf2e7,0xf2e8,0xf2e9,0xf2ea,0xf2eb,0xf2ec,0xf2ed,0xf2ee,0xf2ef
    .hword 0xf300,0xf301,0xf302,0xf303,0xf304,0xf305,0xf306,0xf307,0xf308,0xf309,0xf30a,0xf30b,0xf30c,0xf30d,0xf30e,0xf30f
    .hword 0xf320,0xf321,0xf322,0xf323,0xf324,0xf325,0xf326,0xf327,0xf328,0xf329,0xf32a,0xf32b,0xf32c,0xf32d,0xf32e,0xf32f
    .hword 0xf340,0xf341,0xf342,0xf343,0xf344,0xf345,0xf346,0xf347,0xf348,0xf349,0xf34a,0xf34b,0xf34c,0xf34d,0xf34e,0xf34f
    .hword 0xf360,0xf361,0xf362,0xf363,0xf364,0xf365,0xf366,0xf367,0xf368,0xf369,0xf36a,0xf36b,0xf36c,0xf36d,0xf36e,0xf36f
    .hword 0xf380,0xf381,0xf382,0xf383,0xf384,0xf385,0xf386,0xf387,0xf388,0xf389,0xf38a,0xf38b,0xf38c,0xf38d,0xf38e,0xf38f
    .hword 0xf3a0,0xf3a1,0xf3a2,0xf3a3,0xf3a4,0xf3a5,0xf3a6,0xf3a7,0xf3a8,0xf3a9,0xf3aa,0xf3ab,0xf3ac,0xf3ad,0xf3ae,0xf3af
    .hword 0xf3c0,0xf3c1,0xf3c2,0xf3c3,0xf3c4,0xf3c5,0xf3c6,0xf3c7,0xf3c8,0xf3c9,0xf3ca,0xf3cb,0xf3cc,0xf3cd,0xf3ce,0xf3cf
    .hword 0xf3e0,0xf3e1,0xf3e2,0xf3e3,0xf3e4,0xf3e5,0xf3e6,0xf3e7,0xf3e8,0xf3e9,0xf3ea,0xf3eb,0xf3ec,0xf3ed,0xf3ee,0xf3ef

@=========================================================================
@ VRAM Translation Mapping Table
@=========================================================================
    .include    "vramxlate.s"


@=========================================================================
@ SNES to GBA Tile Map translation table
@  SNES                 GBA
@  ----                 ---
@  vhopppcc        =>   0pppvhcc 
@=========================================================================
tileMap:
    .byte 0x00,0x01,0x02,0x03,0x10,0x11,0x12,0x13,0x20,0x21,0x22,0x23,0x30,0x31,0x32,0x33
    .byte 0x40,0x41,0x42,0x43,0x50,0x51,0x52,0x53,0x60,0x61,0x62,0x63,0x70,0x71,0x72,0x73
    .byte 0x00,0x01,0x02,0x03,0x10,0x11,0x12,0x13,0x20,0x21,0x22,0x23,0x30,0x31,0x32,0x33
    .byte 0x40,0x41,0x42,0x43,0x50,0x51,0x52,0x53,0x60,0x61,0x62,0x63,0x70,0x71,0x72,0x73
    .byte 0x04,0x05,0x06,0x07,0x14,0x15,0x16,0x17,0x24,0x25,0x26,0x27,0x34,0x35,0x36,0x37
    .byte 0x44,0x45,0x46,0x47,0x54,0x55,0x56,0x57,0x64,0x65,0x66,0x67,0x74,0x75,0x76,0x77
    .byte 0x04,0x05,0x06,0x07,0x14,0x15,0x16,0x17,0x24,0x25,0x26,0x27,0x34,0x35,0x36,0x37
    .byte 0x44,0x45,0x46,0x47,0x54,0x55,0x56,0x57,0x64,0x65,0x66,0x67,0x74,0x75,0x76,0x77
    .byte 0x08,0x09,0x0a,0x0b,0x18,0x19,0x1a,0x1b,0x28,0x29,0x2a,0x2b,0x38,0x39,0x3a,0x3b
    .byte 0x48,0x49,0x4a,0x4b,0x58,0x59,0x5a,0x5b,0x68,0x69,0x6a,0x6b,0x78,0x79,0x7a,0x7b
    .byte 0x08,0x09,0x0a,0x0b,0x18,0x19,0x1a,0x1b,0x28,0x29,0x2a,0x2b,0x38,0x39,0x3a,0x3b
    .byte 0x48,0x49,0x4a,0x4b,0x58,0x59,0x5a,0x5b,0x68,0x69,0x6a,0x6b,0x78,0x79,0x7a,0x7b
    .byte 0x0c,0x0d,0x0e,0x0f,0x1c,0x1d,0x1e,0x1f,0x2c,0x2d,0x2e,0x2f,0x3c,0x3d,0x3e,0x3f
    .byte 0x4c,0x4d,0x4e,0x4f,0x5c,0x5d,0x5e,0x5f,0x6c,0x6d,0x6e,0x6f,0x7c,0x7d,0x7e,0x7f
    .byte 0x0c,0x0d,0x0e,0x0f,0x1c,0x1d,0x1e,0x1f,0x2c,0x2d,0x2e,0x2f,0x3c,0x3d,0x3e,0x3f
    .byte 0x4c,0x4d,0x4e,0x4f,0x5c,0x5d,0x5e,0x5f,0x6c,0x6d,0x6e,0x6f,0x7c,0x7d,0x7e,0x7f


@=========================================================================
@ SNES to GBA 4-bit CHR unpacker
@    SNES           GBA
@    abcdefgh   =>  000a000b 000c000d 000e000f 000g000h
@=========================================================================
charUnpack4:
    .long 0x00000000,0x10000000,0x01000000,0x11000000,0x00100000,0x10100000,0x01100000,0x11100000,0x00010000,0x10010000,0x01010000,0x11010000,0x00110000,0x10110000,0x01110000,0x11110000
    .long 0x00001000,0x10001000,0x01001000,0x11001000,0x00101000,0x10101000,0x01101000,0x11101000,0x00011000,0x10011000,0x01011000,0x11011000,0x00111000,0x10111000,0x01111000,0x11111000
    .long 0x00000100,0x10000100,0x01000100,0x11000100,0x00100100,0x10100100,0x01100100,0x11100100,0x00010100,0x10010100,0x01010100,0x11010100,0x00110100,0x10110100,0x01110100,0x11110100
    .long 0x00001100,0x10001100,0x01001100,0x11001100,0x00101100,0x10101100,0x01101100,0x11101100,0x00011100,0x10011100,0x01011100,0x11011100,0x00111100,0x10111100,0x01111100,0x11111100
    .long 0x00000010,0x10000010,0x01000010,0x11000010,0x00100010,0x10100010,0x01100010,0x11100010,0x00010010,0x10010010,0x01010010,0x11010010,0x00110010,0x10110010,0x01110010,0x11110010
    .long 0x00001010,0x10001010,0x01001010,0x11001010,0x00101010,0x10101010,0x01101010,0x11101010,0x00011010,0x10011010,0x01011010,0x11011010,0x00111010,0x10111010,0x01111010,0x11111010
    .long 0x00000110,0x10000110,0x01000110,0x11000110,0x00100110,0x10100110,0x01100110,0x11100110,0x00010110,0x10010110,0x01010110,0x11010110,0x00110110,0x10110110,0x01110110,0x11110110
    .long 0x00001110,0x10001110,0x01001110,0x11001110,0x00101110,0x10101110,0x01101110,0x11101110,0x00011110,0x10011110,0x01011110,0x11011110,0x00111110,0x10111110,0x01111110,0x11111110
    .long 0x00000001,0x10000001,0x01000001,0x11000001,0x00100001,0x10100001,0x01100001,0x11100001,0x00010001,0x10010001,0x01010001,0x11010001,0x00110001,0x10110001,0x01110001,0x11110001
    .long 0x00001001,0x10001001,0x01001001,0x11001001,0x00101001,0x10101001,0x01101001,0x11101001,0x00011001,0x10011001,0x01011001,0x11011001,0x00111001,0x10111001,0x01111001,0x11111001
    .long 0x00000101,0x10000101,0x01000101,0x11000101,0x00100101,0x10100101,0x01100101,0x11100101,0x00010101,0x10010101,0x01010101,0x11010101,0x00110101,0x10110101,0x01110101,0x11110101
    .long 0x00001101,0x10001101,0x01001101,0x11001101,0x00101101,0x10101101,0x01101101,0x11101101,0x00011101,0x10011101,0x01011101,0x11011101,0x00111101,0x10111101,0x01111101,0x11111101
    .long 0x00000011,0x10000011,0x01000011,0x11000011,0x00100011,0x10100011,0x01100011,0x11100011,0x00010011,0x10010011,0x01010011,0x11010011,0x00110011,0x10110011,0x01110011,0x11110011
    .long 0x00001011,0x10001011,0x01001011,0x11001011,0x00101011,0x10101011,0x01101011,0x11101011,0x00011011,0x10011011,0x01011011,0x11011011,0x00111011,0x10111011,0x01111011,0x11111011
    .long 0x00000111,0x10000111,0x01000111,0x11000111,0x00100111,0x10100111,0x01100111,0x11100111,0x00010111,0x10010111,0x01010111,0x11010111,0x00110111,0x10110111,0x01110111,0x11110111
    .long 0x00001111,0x10001111,0x01001111,0x11001111,0x00101111,0x10101111,0x01101111,0x11101111,0x00011111,0x10011111,0x01011111,0x11011111,0x00111111,0x10111111,0x01111111,0x11111111

@=========================================================================
@ SNES to GBA 8-bit CHR unpacker
@    SNES           GBA
@    0000abcd   =>  0000000a 0000000b 0000000c 0000000d
@=========================================================================
charUnpack8:
    .word 0x00000000,0x01000000,0x00010000,0x01010000,0x00000100,0x01000100,0x00010100,0x01010100,0x00000001,0x01000001,0x00010001,0x01010001,0x00000101,0x01000101,0x00010101,0x01010101

@=========================================================================
@ lookup table for decimal mode addition
@=========================================================================
decimalAdd:

    .hword 0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015
    .hword 0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025
    .hword 0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016
    .hword 0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026
    .hword 0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017
    .hword 0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027
    .hword 0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0003,0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018
    .hword 0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028
    .hword 0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0004,0x0005,0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019
    .hword 0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029
    .hword 0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0005,0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a
    .hword 0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a
    .hword 0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0006,0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b
    .hword 0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b
    .hword 0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b
    .hword 0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b
    .hword 0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b
    .hword 0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0007,0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c
    .hword 0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c
    .hword 0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c
    .hword 0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c
    .hword 0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c
    .hword 0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0008,0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d
    .hword 0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d
    .hword 0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d
    .hword 0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d
    .hword 0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d
    .hword 0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0009,0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e
    .hword 0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e
    .hword 0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e
    .hword 0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e
    .hword 0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e
    .hword 0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f
    .hword 0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020
    .hword 0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021
    .hword 0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021,0x0022
    .hword 0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021,0x0022,0x0023
    .hword 0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0015,0x0016,0x0017,0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,0x0020,0x0021,0x0022,0x0023,0x0024
    .hword 0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033,0x0034
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043,0x0044
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025
    .hword 0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026
    .hword 0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027
    .hword 0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0013,0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028
    .hword 0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0014,0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029
    .hword 0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0015,0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a
    .hword 0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0016,0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b
    .hword 0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b
    .hword 0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b
    .hword 0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b
    .hword 0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0017,0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c
    .hword 0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c
    .hword 0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c
    .hword 0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c
    .hword 0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0018,0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d
    .hword 0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d
    .hword 0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d
    .hword 0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d
    .hword 0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0019,0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e
    .hword 0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e
    .hword 0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e
    .hword 0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e
    .hword 0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0025,0x0026,0x0027,0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,0x0030,0x0031,0x0032,0x0033,0x0034
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043,0x0044
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0023,0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0024,0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0025,0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0026,0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b
    .hword 0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b
    .hword 0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b
    .hword 0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0027,0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c
    .hword 0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c
    .hword 0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c
    .hword 0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0028,0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d
    .hword 0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d
    .hword 0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d
    .hword 0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0029,0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e
    .hword 0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e
    .hword 0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e
    .hword 0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,0x0040,0x0041,0x0042,0x0043,0x0044
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0033,0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0034,0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0035,0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0036,0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b
    .hword 0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b
    .hword 0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0037,0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c
    .hword 0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c
    .hword 0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0038,0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d
    .hword 0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d
    .hword 0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0039,0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e
    .hword 0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e
    .hword 0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,0x0050,0x0051,0x0052,0x0053,0x0054
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x0043,0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x0044,0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x0045,0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x0046,0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b
    .hword 0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x0047,0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c
    .hword 0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x0048,0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d
    .hword 0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x0049,0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e
    .hword 0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,0x0060,0x0061,0x0062,0x0063,0x0064
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x0053,0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x0054,0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x0055,0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x0056,0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x0057,0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x0058,0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x0059,0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,0x0070,0x0071,0x0072,0x0073,0x0074
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x0063,0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x0064,0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x0065,0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x0066,0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x0067,0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x0068,0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x0069,0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,0x0080,0x0081,0x0082,0x0083,0x0084
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x0073,0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x0074,0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x0075,0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x0076,0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x0077,0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x0078,0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x0079,0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,0x0090,0x0091,0x0092,0x0093,0x0094
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x0083,0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x0084,0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x0085,0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x0086,0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x0087,0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x0088,0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x0089,0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,0x0100,0x0101,0x0102,0x0103,0x0104
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x0093,0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x0094,0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x0095,0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x0096,0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x0097,0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x0098,0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x0099,0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x010a,0x010b,0x010c,0x010d,0x010e,0x010f,0x0110,0x0111,0x0112,0x0113,0x0114
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0100,0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205
    .hword 0x0101,0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206
    .hword 0x0102,0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207
    .hword 0x0103,0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208
    .hword 0x0104,0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209
    .hword 0x0105,0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a
    .hword 0x0106,0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b
    .hword 0x0107,0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c
    .hword 0x0108,0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d
    .hword 0x0109,0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x011a,0x011b,0x011c,0x011d,0x011e,0x011f,0x0120,0x0121,0x0122,0x0123,0x0124
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213,0x0214
    .hword 0x0110,0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215
    .hword 0x0111,0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216
    .hword 0x0112,0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217
    .hword 0x0113,0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218
    .hword 0x0114,0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219
    .hword 0x0115,0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a
    .hword 0x0116,0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b
    .hword 0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b
    .hword 0x0117,0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c
    .hword 0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c
    .hword 0x0118,0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d
    .hword 0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d
    .hword 0x0119,0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e
    .hword 0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x012a,0x012b,0x012c,0x012d,0x012e,0x012f,0x0130,0x0131,0x0132,0x0133,0x0134
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213,0x0214
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223,0x0224
    .hword 0x0120,0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225
    .hword 0x0121,0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226
    .hword 0x0122,0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227
    .hword 0x0123,0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228
    .hword 0x0124,0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229
    .hword 0x0125,0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a
    .hword 0x0126,0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b
    .hword 0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b
    .hword 0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b
    .hword 0x0127,0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c
    .hword 0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c
    .hword 0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c
    .hword 0x0128,0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d
    .hword 0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d
    .hword 0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d
    .hword 0x0129,0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e
    .hword 0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e
    .hword 0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x013a,0x013b,0x013c,0x013d,0x013e,0x013f,0x0140,0x0141,0x0142,0x0143,0x0144
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213,0x0214
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223,0x0224
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233,0x0234
    .hword 0x0130,0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235
    .hword 0x0131,0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236
    .hword 0x0132,0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237
    .hword 0x0133,0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238
    .hword 0x0134,0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239
    .hword 0x0135,0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a
    .hword 0x0136,0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b
    .hword 0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b
    .hword 0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b
    .hword 0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b
    .hword 0x0137,0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c
    .hword 0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c
    .hword 0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c
    .hword 0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c
    .hword 0x0138,0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d
    .hword 0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d
    .hword 0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d
    .hword 0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d
    .hword 0x0139,0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e
    .hword 0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e
    .hword 0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e
    .hword 0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f
    .hword 0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230
    .hword 0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231
    .hword 0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232
    .hword 0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233
    .hword 0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242,0x0243
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x014a,0x014b,0x014c,0x014d,0x014e,0x014f,0x0150,0x0151,0x0152,0x0153,0x0154
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213,0x0214
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223,0x0224
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233,0x0234
    .hword 0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242,0x0243,0x0244
    .hword 0x0140,0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235
    .hword 0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245
    .hword 0x0141,0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236
    .hword 0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246
    .hword 0x0142,0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237
    .hword 0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247
    .hword 0x0143,0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238
    .hword 0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248
    .hword 0x0144,0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239
    .hword 0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249
    .hword 0x0145,0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a
    .hword 0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a
    .hword 0x0146,0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b
    .hword 0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b
    .hword 0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b
    .hword 0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b
    .hword 0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b
    .hword 0x0147,0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c
    .hword 0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c
    .hword 0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c
    .hword 0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c
    .hword 0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c
    .hword 0x0148,0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d
    .hword 0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d
    .hword 0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d
    .hword 0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d
    .hword 0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d
    .hword 0x0149,0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e
    .hword 0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e
    .hword 0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e
    .hword 0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e
    .hword 0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f
    .hword 0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f
    .hword 0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230
    .hword 0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240
    .hword 0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231
    .hword 0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241
    .hword 0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232
    .hword 0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242
    .hword 0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251,0x0252
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233
    .hword 0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242,0x0243
    .hword 0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251,0x0252,0x0253
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x015a,0x015b,0x015c,0x015d,0x015e,0x015f,0x0160,0x0161,0x0162,0x0163,0x0164
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213,0x0214
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223,0x0224
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233,0x0234
    .hword 0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242,0x0243,0x0244
    .hword 0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251,0x0252,0x0253,0x0254
    .hword 0x0150,0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235
    .hword 0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245
    .hword 0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255
    .hword 0x0151,0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236
    .hword 0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246
    .hword 0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256
    .hword 0x0152,0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237
    .hword 0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247
    .hword 0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257
    .hword 0x0153,0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238
    .hword 0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248
    .hword 0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258
    .hword 0x0154,0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239
    .hword 0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249
    .hword 0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259
    .hword 0x0155,0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a
    .hword 0x0235,0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a
    .hword 0x0245,0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a
    .hword 0x0156,0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b
    .hword 0x0166,0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b
    .hword 0x0176,0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b
    .hword 0x0186,0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b
    .hword 0x0196,0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab
    .hword 0x01a6,0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb
    .hword 0x01b6,0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb
    .hword 0x01c6,0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db
    .hword 0x01d6,0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb
    .hword 0x01e6,0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb
    .hword 0x01f6,0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b
    .hword 0x0206,0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b
    .hword 0x0216,0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b
    .hword 0x0226,0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b
    .hword 0x0236,0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b
    .hword 0x0246,0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b
    .hword 0x0157,0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c
    .hword 0x0167,0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c
    .hword 0x0177,0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c
    .hword 0x0187,0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c
    .hword 0x0197,0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac
    .hword 0x01a7,0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc
    .hword 0x01b7,0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc
    .hword 0x01c7,0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc
    .hword 0x01d7,0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec
    .hword 0x01e7,0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc
    .hword 0x01f7,0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c
    .hword 0x0207,0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c
    .hword 0x0217,0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c
    .hword 0x0227,0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c
    .hword 0x0237,0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c
    .hword 0x0247,0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c
    .hword 0x0158,0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d
    .hword 0x0168,0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d
    .hword 0x0178,0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d
    .hword 0x0188,0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d
    .hword 0x0198,0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad
    .hword 0x01a8,0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd
    .hword 0x01b8,0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd
    .hword 0x01c8,0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd
    .hword 0x01d8,0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed
    .hword 0x01e8,0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd
    .hword 0x01f8,0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d
    .hword 0x0208,0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d
    .hword 0x0218,0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d
    .hword 0x0228,0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d
    .hword 0x0238,0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d
    .hword 0x0248,0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d
    .hword 0x0159,0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e
    .hword 0x0169,0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e
    .hword 0x0179,0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e
    .hword 0x0189,0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e
    .hword 0x0199,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae
    .hword 0x01a9,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be
    .hword 0x01b9,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce
    .hword 0x01c9,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de
    .hword 0x01d9,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee
    .hword 0x01e9,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe
    .hword 0x01f9,0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e
    .hword 0x0209,0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e
    .hword 0x0219,0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e
    .hword 0x0229,0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e
    .hword 0x0239,0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e
    .hword 0x0249,0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e
    .hword 0x0160,0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f
    .hword 0x0170,0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f
    .hword 0x0180,0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f
    .hword 0x0190,0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f
    .hword 0x01a0,0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af
    .hword 0x01b0,0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf
    .hword 0x01c0,0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf
    .hword 0x01d0,0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df
    .hword 0x01e0,0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef
    .hword 0x01f0,0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff
    .hword 0x0200,0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f
    .hword 0x0210,0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f
    .hword 0x0220,0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f
    .hword 0x0230,0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f
    .hword 0x0240,0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f
    .hword 0x0250,0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e,0x025f
    .hword 0x0161,0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170
    .hword 0x0171,0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180
    .hword 0x0181,0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190
    .hword 0x0191,0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0
    .hword 0x01a1,0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0
    .hword 0x01b1,0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0
    .hword 0x01c1,0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0
    .hword 0x01d1,0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0
    .hword 0x01e1,0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0
    .hword 0x01f1,0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200
    .hword 0x0201,0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210
    .hword 0x0211,0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220
    .hword 0x0221,0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230
    .hword 0x0231,0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240
    .hword 0x0241,0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250
    .hword 0x0251,0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e,0x025f,0x0260
    .hword 0x0162,0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171
    .hword 0x0172,0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181
    .hword 0x0182,0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191
    .hword 0x0192,0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1
    .hword 0x01a2,0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1
    .hword 0x01b2,0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1
    .hword 0x01c2,0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1
    .hword 0x01d2,0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1
    .hword 0x01e2,0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1
    .hword 0x01f2,0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201
    .hword 0x0202,0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211
    .hword 0x0212,0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221
    .hword 0x0222,0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231
    .hword 0x0232,0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241
    .hword 0x0242,0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251
    .hword 0x0252,0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e,0x025f,0x0260,0x0261
    .hword 0x0163,0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172
    .hword 0x0173,0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182
    .hword 0x0183,0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192
    .hword 0x0193,0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2
    .hword 0x01a3,0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2
    .hword 0x01b3,0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2
    .hword 0x01c3,0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2
    .hword 0x01d3,0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2
    .hword 0x01e3,0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2
    .hword 0x01f3,0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202
    .hword 0x0203,0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212
    .hword 0x0213,0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222
    .hword 0x0223,0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232
    .hword 0x0233,0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242
    .hword 0x0243,0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251,0x0252
    .hword 0x0253,0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e,0x025f,0x0260,0x0261,0x0262
    .hword 0x0164,0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173
    .hword 0x0174,0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183
    .hword 0x0184,0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193
    .hword 0x0194,0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3
    .hword 0x01a4,0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3
    .hword 0x01b4,0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3
    .hword 0x01c4,0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3
    .hword 0x01d4,0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3
    .hword 0x01e4,0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3
    .hword 0x01f4,0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203
    .hword 0x0204,0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213
    .hword 0x0214,0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223
    .hword 0x0224,0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233
    .hword 0x0234,0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242,0x0243
    .hword 0x0244,0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251,0x0252,0x0253
    .hword 0x0254,0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e,0x025f,0x0260,0x0261,0x0262,0x0263
    .hword 0x0165,0x0166,0x0167,0x0168,0x0169,0x016a,0x016b,0x016c,0x016d,0x016e,0x016f,0x0170,0x0171,0x0172,0x0173,0x0174
    .hword 0x0175,0x0176,0x0177,0x0178,0x0179,0x017a,0x017b,0x017c,0x017d,0x017e,0x017f,0x0180,0x0181,0x0182,0x0183,0x0184
    .hword 0x0185,0x0186,0x0187,0x0188,0x0189,0x018a,0x018b,0x018c,0x018d,0x018e,0x018f,0x0190,0x0191,0x0192,0x0193,0x0194
    .hword 0x0195,0x0196,0x0197,0x0198,0x0199,0x019a,0x019b,0x019c,0x019d,0x019e,0x019f,0x01a0,0x01a1,0x01a2,0x01a3,0x01a4
    .hword 0x01a5,0x01a6,0x01a7,0x01a8,0x01a9,0x01aa,0x01ab,0x01ac,0x01ad,0x01ae,0x01af,0x01b0,0x01b1,0x01b2,0x01b3,0x01b4
    .hword 0x01b5,0x01b6,0x01b7,0x01b8,0x01b9,0x01ba,0x01bb,0x01bc,0x01bd,0x01be,0x01bf,0x01c0,0x01c1,0x01c2,0x01c3,0x01c4
    .hword 0x01c5,0x01c6,0x01c7,0x01c8,0x01c9,0x01ca,0x01cb,0x01cc,0x01cd,0x01ce,0x01cf,0x01d0,0x01d1,0x01d2,0x01d3,0x01d4
    .hword 0x01d5,0x01d6,0x01d7,0x01d8,0x01d9,0x01da,0x01db,0x01dc,0x01dd,0x01de,0x01df,0x01e0,0x01e1,0x01e2,0x01e3,0x01e4
    .hword 0x01e5,0x01e6,0x01e7,0x01e8,0x01e9,0x01ea,0x01eb,0x01ec,0x01ed,0x01ee,0x01ef,0x01f0,0x01f1,0x01f2,0x01f3,0x01f4
    .hword 0x01f5,0x01f6,0x01f7,0x01f8,0x01f9,0x01fa,0x01fb,0x01fc,0x01fd,0x01fe,0x01ff,0x0200,0x0201,0x0202,0x0203,0x0204
    .hword 0x0205,0x0206,0x0207,0x0208,0x0209,0x020a,0x020b,0x020c,0x020d,0x020e,0x020f,0x0210,0x0211,0x0212,0x0213,0x0214
    .hword 0x0215,0x0216,0x0217,0x0218,0x0219,0x021a,0x021b,0x021c,0x021d,0x021e,0x021f,0x0220,0x0221,0x0222,0x0223,0x0224
    .hword 0x0225,0x0226,0x0227,0x0228,0x0229,0x022a,0x022b,0x022c,0x022d,0x022e,0x022f,0x0230,0x0231,0x0232,0x0233,0x0234
    .hword 0x0235,0x0236,0x0237,0x0238,0x0239,0x023a,0x023b,0x023c,0x023d,0x023e,0x023f,0x0240,0x0241,0x0242,0x0243,0x0244
    .hword 0x0245,0x0246,0x0247,0x0248,0x0249,0x024a,0x024b,0x024c,0x024d,0x024e,0x024f,0x0250,0x0251,0x0252,0x0253,0x0254
    .hword 0x0255,0x0256,0x0257,0x0258,0x0259,0x025a,0x025b,0x025c,0x025d,0x025e,0x025f,0x0260,0x0261,0x0262,0x0263,0x0264


@=========================================================================
@ y-offset for backgrounds 
@ SNES Y maps to GBA Y
@=========================================================================
bgYOffset:

    .byte 0x00,0x01,0x01,0x02,0x03,0x04,0x04,0x05,0x06,0x06,0x07,0x08,0x09,0x09,0x0a,0x0b
    .byte 0x0b,0x0c,0x0d,0x0e,0x0e,0x0f,0x10,0x10,0x11,0x12,0x13,0x13,0x14,0x15,0x15,0x16
    .byte 0x17,0x18,0x18,0x19,0x1a,0x1a,0x1b,0x1c,0x1d,0x1d,0x1e,0x1f,0x1f,0x20,0x21,0x22
    .byte 0x22,0x23,0x24,0x24,0x25,0x26,0x27,0x27,0x28,0x29,0x29,0x2a,0x2b,0x2c,0x2c,0x2d
    .byte 0x2e,0x2e,0x2f,0x30,0x31,0x31,0x32,0x33,0x33,0x34,0x35,0x36,0x36,0x37,0x38,0x38
    .byte 0x39,0x3a,0x3b,0x3b,0x3c,0x3d,0x3d,0x3e,0x3f,0x40,0x40,0x41,0x42,0x42,0x43,0x44
    .byte 0x45,0x45,0x46,0x47,0x47,0x48,0x49,0x4a,0x4a,0x4b,0x4c,0x4c,0x4d,0x4e,0x4f,0x4f
    .byte 0x50,0x51,0x51,0x52,0x53,0x54,0x54,0x55,0x56,0x56,0x57,0x58,0x59,0x59,0x5a,0x5b
    .byte 0x5b,0x5c,0x5d,0x5e,0x5e,0x5f,0x60,0x60,0x61,0x62,0x63,0x63,0x64,0x65,0x65,0x66
    .byte 0x67,0x68,0x68,0x69,0x6a,0x6a,0x6b,0x6c,0x6d,0x6d,0x6e,0x6f,0x6f,0x70,0x71,0x72
    .byte 0x72,0x73,0x74,0x74,0x75,0x76,0x77,0x77,0x78,0x79,0x79,0x7a,0x7b,0x7c,0x7c,0x7d
    .byte 0x7e,0x7e,0x7f,0x80,0x81,0x81,0x82,0x83,0x83,0x84,0x85,0x86,0x86,0x87,0x88,0x88
    .byte 0x89,0x8a,0x8b,0x8b,0x8c,0x8d,0x8d,0x8e,0x8f,0x90,0x90,0x91,0x92,0x92,0x93,0x94
    .byte 0x95,0x95,0x96,0x97,0x97,0x98,0x99,0x9a,0x9a,0x9b,0x9c,0x9c,0x9d,0x9e,0x9f,0x9f
    .byte 0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f
    .byte 0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f,0x9f
    .byte 0x9f,0x9f,0x9f,0x9f,0x9f,0x9f

@=========================================================================
@ memory mapping table
@=========================================================================

    .align  8    
    .ascii  "SMEMMAP0"
MemoryMap:
    .rept   (8*256)
    .long   0
    .endr

SaveRAMMask:
    .long   0

ROMOffset:
    .long   0

