/*** m6809: Portable 6809 emulator ******************************************

	Copyright (C) John Butler 1997

	References:

		6809 Simulator V09, By L.C. Benschop, Eidnhoven The Netherlands.

		m6809: Portable 6809 emulator, DS (6809 code in MAME, derived from
			the 6809 Simulator V09)

		6809 Microcomputer Programming & Interfacing with Experiments"
			by Andrew C. Staugaard, Jr.; Howard W. Sams & Co., Inc.

	System dependencies:	word must be 16 bit unsigned int
							byte must be 8 bit unsigned int
							long must be more than 16 bits
							arrays up to 65536 bytes must be supported
							machine must be twos complement

*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "m6809.h"
#include "driver.h"

/* 6809 registers */
m6809_Regs M6809;
#define cc 	M6809.cc
#define dpreg 	M6809.dp
#define dbreg	M6809.db
#define areg	M6809.db.b.h
#define breg	M6809.db.b.l
#define xreg	M6809.x
#define yreg	M6809.y
#define ureg	M6809.u
#define sreg	M6809.s
#define pcreg	M6809.pc
#define eaddr	M6809.ea
#define pending_interrupts M6809.pending_interrupts

/* public globals */
int m6809_ICount=50000;
int m6809_Flags;	/* flags for speed optimization */
int m6809_slapstic;

/* flag, handlers for speed optimization */
static int (*rd_u_handler)(int);
static int (*rd_u_handler_wd)(int);
static int (*rd_s_handler)(int);
static int (*rd_s_handler_wd)(int);
static void (*wr_u_handler)(int,int);
static void (*wr_u_handler_wd)(int,int);
static void (*wr_s_handler)(int,int);
static void (*wr_s_handler_wd)(int,int);

/* these are re-defined in m6809.h TO RAM, ROM or functions in cpuintrf.c */
#define M_RDMEM(A)      M6809_RDMEM(A)
#define M_WRMEM(A,V)    M6809_WRMEM(A,V)
#define M_RDOP(A)       M6809_RDOP(A)
#define M_RDOP_ARG(A)   M6809_RDOP_ARG(A)

/* macros to access memory */
#define IMMBYTE(b)	{b = M_RDOP_ARG(pcreg++);}
#define IMMWORD(w)	{w = (M_RDOP_ARG(pcreg)<<8) + M_RDOP_ARG(pcreg+1); pcreg+=2;}

#define PUSHBYTE(b) {--sreg;(*wr_s_handler)(sreg,(b));}
#define PUSHWORD(w) {sreg-=2;(*wr_s_handler_wd)(sreg,(w));}
#define PULLBYTE(b) {b=(*rd_s_handler)(sreg);sreg++;}
#define PULLWORD(w) {w=(*rd_s_handler_wd)(sreg);sreg+=2;}
#define PSHUBYTE(b) {--ureg;(*wr_u_handler)(ureg,(b));}
#define PSHUWORD(w) {ureg-=2;(*wr_u_handler_wd)(ureg,(w));}
#define PULUBYTE(b) {b=(*rd_u_handler)(ureg);ureg++;}
#define PULUWORD(w) {w=(*rd_u_handler_wd)(ureg);ureg+=2;}


/* CC masks */
#define CLR_HNZVC	cc&=0xd0
#define CLR_NZV		cc&=0xf1
#define CLR_HNZC	cc&=0xd2
#define CLR_NZVC	cc&=0xf0
#define CLR_Z		cc&=0xfb
#define CLR_NZC		cc&=0xf2
#define CLR_ZC		cc&=0xfa
/* macros for CC -- CC bits affected should be reset before calling */
#define SET_Z(a)	if(!a)SEZ
#define SET_Z8(a)	SET_Z((byte)a)
#define SET_Z16(a)	SET_Z((word)a)
#define SET_N8(a)	cc|=((a&0x80)>>4)
#define SET_N16(a)	cc|=((a&0x8000)>>12)
#define SET_H(a,b,r)	cc|=(((a^b^r)&0x10)<<1)
#define SET_C8(a)	cc|=((a&0x100)>>8)
#define SET_C16(a)	cc|=((a&0x10000)>>16)
#define SET_V8(a,b,r)	cc|=(((a^b^r^(r>>1))&0x80)>>6)
#define SET_V16(a,b,r)	cc|=(((a^b^r^(r>>1))&0x8000)>>14)

static byte flags8i[256]=	/* increment */
{
0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x0a,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08
};
static byte flags8d[256]= /* decrement */
{
0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x02,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,
0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08,0x08
};
#define SET_FLAGS8I(a)		{cc|=flags8i[(a)&0xff];}
#define SET_FLAGS8D(a)		{cc|=flags8d[(a)&0xff];}

/* combos */
#define SET_NZ8(a)			{SET_N8(a);SET_Z(a);}
#define SET_NZ16(a)			{SET_N16(a);SET_Z(a);}
#define SET_FLAGS8(a,b,r)	{SET_N8(r);SET_Z8(r);SET_V8(a,b,r);SET_C8(r);}
#define SET_FLAGS16(a,b,r)	{SET_N16(r);SET_Z16(r);SET_V16(a,b,r);SET_C16(r);}

/* for treating an unsigned byte as a signed word */
#define SIGNED(b) ((word)(b&0x80?b|0xff00:b))

/* macros to access dreg */
#define GETDREG (dbreg.w.w1)
#define SETDREG(n) {dbreg.w.w1=n;}

/* macros for addressing modes (postbytes have their own code) */
#define DIRECT {IMMBYTE(eaddr);eaddr|=(dpreg<<8);}
#define IMM8 eaddr=pcreg++
#define IMM16 {eaddr=pcreg;pcreg+=2;}
#define EXTENDED IMMWORD(eaddr)

/* macros to set status flags */
#define SEC cc|=0x01
#define CLC cc&=0xfe
#define SEZ cc|=0x04
#define CLZ cc&=0xfb
#define SEN cc|=0x08
#define CLN cc&=0xf7
#define SEV cc|=0x02
#define CLV cc&=0xfd
#define SEH cc|=0x20
#define CLH cc&=0xdf

/* macros for convenience */
#define DIRBYTE(b) {DIRECT;b=M_RDMEM(eaddr);}
#define DIRWORD(w) {DIRECT;w=M_RDMEM_WORD(eaddr);}
#define EXTBYTE(b) {EXTENDED;b=M_RDMEM(eaddr);}
#define EXTWORD(w) {EXTENDED;w=M_RDMEM_WORD(eaddr);}

/* macros for branch instructions */
#define BRANCH(f) {IMMBYTE(t);if(f){pcreg+=SIGNED(t);change_pc(pcreg);}}	/* TS 971002 */
#define LBRANCH(f) {IMMWORD(t);if(f){pcreg+=t;change_pc(pcreg);}}	/* TS 971002 */
#define NXORV  ((cc&0x08)^((cc&0x02)<<2))

/* macros for setting/getting registers in TFR/EXG instructions */
#define GETREG(val,reg) switch(reg) {\
                         case 0: val=GETDREG;break;\
                         case 1: val=xreg;break;\
                         case 2: val=yreg;break;\
                    	 case 3: val=ureg;break;\
                    	 case 4: val=sreg;break;\
                    	 case 5: val=pcreg;break;\
                    	 case 8: val=areg;break;\
                    	 case 9: val=breg;break;\
                    	 case 10: val=cc;break;\
                    	 case 11: val=dpreg;break;}

#define SETREG(val,reg) switch(reg) {\
			 case 0: SETDREG(val); break;\
			 case 1: xreg=val;break;\
			 case 2: yreg=val;break;\
			 case 3: ureg=val;break;\
			 case 4: sreg=val;break;\
			 case 5: pcreg=val;change_pc(pcreg);break;	/* TS 971002 */ \
			 case 8: areg=val;break;\
			 case 9: breg=val;break;\
			 case 10: cc=val;break;\
			 case 11: dpreg=val;break;}

#define E   0x80			/* 0x80 = fetch effective address */
/* timings for 1-byte opcodes */
static unsigned char cycles1[] =
{
    /*    0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F */
  /*0*/	  6,  0,  0,  6,  6,  0,  6,  6,  6,  6,  6,  0,  6,  6,  3,  6,
  /*1*/	255,255,  2,  2,  0,  0,  5,  9,  0,  2,  3,  0,  3,  2,  8,  6,
  /*2*/	  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
  /*3*/	E+4,E+4,E+4,E+4,  5,  5,  5,  5,  0,  5,  3,  6,  0, 11,  0, 19,
  /*4*/	  2,  0,  0,  2,  2,  0,  2,  2,  2,  2,  2,  0,  2,  2,  0,  2,
  /*5*/	  2,  0,  0,  2,  2,  0,  2,  2,  2,  2,  2,  0,  2,  2,  0,  2,
  /*6*/	E+6,E+0,E+0,E+6,E+6,E+0,E+6,E+6,E+6,E+6,E+6,E+0,E+6,E+6,E+3,E+6,
  /*7*/	  7,  0,  0,  7,  7,  0,  7,  7,  7,  7,  7,  0,  7,  7,  4,  7,
  /*8*/	  2,  2,  2,  4,  2,  2,  2,  2,  2,  2,  2,  2,  4,  7,  3,  0,
  /*9*/	  4,  4,  4,  6,  4,  4,  4,  4,  4,  4,  4,  4,  6,  7,  5,  5,
  /*A*/	E+4,E+4,E+4,E+6,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+6,E+7,E+5,E+5,
  /*B*/	  5,  5,  5,  7,  5,  5,  5,  5,  5,  5,  5,  5,  7,  8,  6,  6,
  /*C*/	  2,  2,  2,  4,  2,  2,  2,  2,  2,  2,  2,  2,  3,  0,  3,  3,
  /*D*/	  4,  4,  4,  6,  4,  4,  4,  4,  4,  4,  4,  4,  5,  5,  5,  5,
  /*E*/	E+4,E+4,E+4,E+6,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+4,E+5,E+5,E+5,E+5,
  /*F*/	  5,  5,  5,  7,  5,  5,  5,  5,  5,  5,  5,  5,  6,  6,  6,  6
};

/* timings for 2-byte opcodes */
static unsigned char cycles2[] =
{
    /*    0   1   2   3   4   5   6   7   8   9   A   B   C   D   E   F */
  /*0*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*1*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*2*/	  0,  5,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
  /*3*/	E+0,E+0,E+0,E+0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, 20,
  /*4*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*5*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*6*/	E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,
  /*7*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
  /*8*/	  0,  0,  0,  5,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  4,  0,
  /*9*/	  0,  0,  0,  7,  0,  0,  0,  0,  0,  0,  0,  0,  7,  0,  6,  6,
  /*A*/	E+0,E+0,E+0,E+7,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+7,E+0,E+6,E+6,
  /*B*/	  0,  0,  0,  8,  0,  0,  0,  0,  0,  0,  0,  0,  8,  0,  7,  7,
  /*C*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,
  /*D*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  6,  6,
  /*E*/	E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+0,E+6,E+6,
  /*F*/	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  7,  7
};
#undef E


static int rd_slow( int addr )
{
    return M_RDMEM(addr);
}

static int rd_slow_wd( int addr )
{
    return( (M_RDMEM(addr)<<8) | (M_RDMEM((addr+1)&0xffff)) );
}

static int rd_fast( int addr )
{
	extern unsigned char *RAM;

	return RAM[addr];
}

static int rd_fast_wd( int addr )
{
	extern unsigned char *RAM;

	return( (RAM[addr]<<8) | (RAM[(addr+1)&0xffff]) );
}

static void wr_slow( int addr, int v )
{
	M_WRMEM(addr,v);
}

static void wr_slow_wd( int addr, int v )
{
	M_WRMEM(addr,v>>8);
	M_WRMEM(((addr)+1)&0xFFFF,v&255);
}

static void wr_fast( int addr, int v )
{
	extern unsigned char *RAM;

	RAM[addr] = v;
}

static void wr_fast_wd( int addr, int v )
{
	extern unsigned char *RAM;

	RAM[addr] = v>>8;
	RAM[(addr+1)&0xffff] = v&255;
}

#define M_RDMEM_WORD(A) (unsigned)((M_RDMEM(A)<<8)|(M_RDMEM(((A)+1)&0xFFFF)))
#define M_WRMEM_WORD(A,V) M_WRMEM(A,V>>8);M_WRMEM(((A)+1)&0xFFFF,V&255);

/****************************************************************************/
/* Set all registers to given values                                        */
/****************************************************************************/
void m6809_SetRegs(m6809_Regs *Regs)
{
	asm_memcpy(&M6809,Regs,sizeof(M6809));
	change_pc(pcreg);
}

/****************************************************************************/
/* Get all registers in given buffer                                        */
/****************************************************************************/
void m6809_GetRegs(m6809_Regs *Regs)
{
	asm_memcpy(Regs,&M6809,sizeof(M6809));
}

/****************************************************************************/
/* Return program counter                                                   */
/****************************************************************************/
unsigned m6809_GetPC(void)
{
	return pcreg;
}


/* Generate interrupts */
INLINE void Interrupt(void)	/* NS 970909 */
{
	if ((pending_interrupts & M6809_INT_NMI) != 0)
	{
		pending_interrupts &= ~M6809_INT_NMI;

		/* NMI */
		cc|=0x80;	/* ASG 971016 */
		PUSHWORD(pcreg);
		PUSHWORD(ureg);
		PUSHWORD(yreg);
		PUSHWORD(xreg);
		PUSHBYTE(dpreg);
		PUSHBYTE(breg);
		PUSHBYTE(areg);
		PUSHBYTE(cc);
		cc|=0xd0;
		pcreg=M_RDMEM_WORD(0xfffc);change_pc(pcreg);	/* TS 971002 */
		m6809_ICount -= 19;
	}
	else if ((pending_interrupts & M6809_INT_IRQ) != 0 && (cc & 0x10) == 0)
	{
		pending_interrupts &= ~M6809_INT_IRQ;

		/* standard IRQ */
		cc|=0x80;	/* ASG 971016 */
		PUSHWORD(pcreg);
		PUSHWORD(ureg);
		PUSHWORD(yreg);
		PUSHWORD(xreg);
		PUSHBYTE(dpreg);
		PUSHBYTE(breg);
		PUSHBYTE(areg);
		PUSHBYTE(cc);
		cc|=0x90;
		pcreg=M_RDMEM_WORD(0xfff8);change_pc(pcreg);	/* TS 971002 */
		m6809_ICount -= 19;
	}
	else if ((pending_interrupts & M6809_INT_FIRQ) != 0 && (cc & 0x40) == 0)
	{
		pending_interrupts &= ~M6809_INT_FIRQ;

		/* fast IRQ */
		PUSHWORD(pcreg);
		cc&=0x7f;	/* ASG 971016 */
		PUSHBYTE(cc);
		cc|=0x50;
		pcreg=M_RDMEM_WORD(0xfff6);change_pc(pcreg);	/* TS 971002 */
		m6809_ICount -= 10;
	}
}



void m6809_reset(void)
{
	pcreg = M_RDMEM_WORD(0xfffe);change_pc(pcreg);	/* TS 971002 */

	dpreg = 0x00;		/* Direct page register = 0x00 */
	cc = 0x00;		/* Clear all flags */
	cc |= 0x10;		/* IRQ disabled */
	cc |= 0x40;		/* FIRQ disabled */
	areg = 0x00;		/* clear accumulator a */
	breg = 0x00;		/* clear accumulator b */
	m6809_Clear_Pending_Interrupts();	/* NS 970908 */

	/* default to unoptimized memory access */
	rd_u_handler = rd_slow;
	rd_u_handler_wd = rd_slow_wd;
	rd_s_handler = rd_slow;
	rd_s_handler_wd = rd_slow_wd;
	wr_u_handler = wr_slow;
	wr_u_handler_wd = wr_slow_wd;
	wr_s_handler = wr_slow;
	wr_s_handler_wd = wr_slow_wd;

	/* optimize memory access according to flags */
	if( m6809_Flags & M6809_FAST_U )
	{
		rd_u_handler=rd_fast; rd_u_handler_wd=rd_fast_wd;
		wr_u_handler=wr_fast; wr_u_handler_wd=wr_fast_wd;
	}
	if( m6809_Flags & M6809_FAST_S )
	{
		rd_s_handler=rd_fast; rd_s_handler_wd=rd_fast_wd;
		wr_s_handler=wr_fast; wr_s_handler_wd=wr_fast_wd;
	}
}


void m6809_Cause_Interrupt(int type)	/* NS 970908 */
{
	pending_interrupts |= type;
	if (type & (M6809_INT_NMI | M6809_INT_IRQ | M6809_INT_FIRQ))
	{
		pending_interrupts &= ~M6809_SYNC;
		if (pending_interrupts & M6809_CWAI)
		{
			if ((pending_interrupts & M6809_INT_NMI) != 0)
				pending_interrupts &= ~M6809_CWAI;
			else if ((pending_interrupts & M6809_INT_IRQ) != 0 && (cc & 0x10) == 0)
				pending_interrupts &= ~M6809_CWAI;
			else if ((pending_interrupts & M6809_INT_FIRQ) != 0 && (cc & 0x40) == 0)
				pending_interrupts &= ~M6809_CWAI;
		}
	}
}
void m6809_Clear_Pending_Interrupts(void)	/* NS 970908 */
{
	pending_interrupts &= ~(M6809_INT_IRQ | M6809_INT_FIRQ | M6809_INT_NMI);
}


/* execute instructions on this CPU until icount expires */
int m6809_execute(int cycles)	/* NS 970908 */
{
	byte op_count;	/* op code clock count */
	byte ireg;
	m6809_ICount = cycles;	/* NS 970908 */
	dword t,r,d,b;

	if (pending_interrupts & (M6809_CWAI | M6809_SYNC))
	{
		m6809_ICount = 0;
		return cycles - m6809_ICount;
	}

	do
	{
		extern int previouspc;

		previouspc = pcreg;

		if (pending_interrupts != 0)
			Interrupt();	/* NS 970908 */

		ireg=M_RDOP(pcreg++);

		if( (op_count = cycles1[ireg])!=0xff ){
			if( op_count &0x80 ) {
				switch((byte)M_RDOP_ARG(pcreg++))
				{
	    				case 0x00: eaddr=xreg;op_count+=1;break;
	    				case 0x01: eaddr=xreg+1;op_count+=1;break;
	    				case 0x02: eaddr=xreg+2;op_count+=1;break;
	    				case 0x03: eaddr=xreg+3;op_count+=1;break;
	    				case 0x04: eaddr=xreg+4;op_count+=1;break;
	    				case 0x05: eaddr=xreg+5;op_count+=1;break;
	    				case 0x06: eaddr=xreg+6;op_count+=1;break;
	    				case 0x07: eaddr=xreg+7;op_count+=1;break;
	    				case 0x08: eaddr=xreg+8;op_count+=1;break;
	    				case 0x09: eaddr=xreg+9;op_count+=1;break;
	    				case 0x0A: eaddr=xreg+10;op_count+=1;break;
	    				case 0x0B: eaddr=xreg+11;op_count+=1;break;
	    				case 0x0C: eaddr=xreg+12;op_count+=1;break;
	    				case 0x0D: eaddr=xreg+13;op_count+=1;break;
	    				case 0x0E: eaddr=xreg+14;op_count+=1;break;
	    				case 0x0F: eaddr=xreg+15;op_count+=1;break;
	    				case 0x10: eaddr=xreg-16;op_count+=1;break;
	    				case 0x11: eaddr=xreg-15;op_count+=1;break;
	    				case 0x12: eaddr=xreg-14;op_count+=1;break;
	    				case 0x13: eaddr=xreg-13;op_count+=1;break;
	    				case 0x14: eaddr=xreg-12;op_count+=1;break;
	    				case 0x15: eaddr=xreg-11;op_count+=1;break;
	    				case 0x16: eaddr=xreg-10;op_count+=1;break;
	    				case 0x17: eaddr=xreg-9;op_count+=1;break;
	    				case 0x18: eaddr=xreg-8;op_count+=1;break;
	    				case 0x19: eaddr=xreg-7;op_count+=1;break;
	    				case 0x1A: eaddr=xreg-6;op_count+=1;break;
	    				case 0x1B: eaddr=xreg-5;op_count+=1;break;
	    				case 0x1C: eaddr=xreg-4;op_count+=1;break;
	    				case 0x1D: eaddr=xreg-3;op_count+=1;break;
	    				case 0x1E: eaddr=xreg-2;op_count+=1;break;
	    				case 0x1F: eaddr=xreg-1;op_count+=1;break;
	    				case 0x20: eaddr=yreg;op_count+=1;break;
	    				case 0x21: eaddr=yreg+1;op_count+=1;break;
	    				case 0x22: eaddr=yreg+2;op_count+=1;break;
	    				case 0x23: eaddr=yreg+3;op_count+=1;break;
	    				case 0x24: eaddr=yreg+4;op_count+=1;break;
	    				case 0x25: eaddr=yreg+5;op_count+=1;break;
	    				case 0x26: eaddr=yreg+6;op_count+=1;break;
	    				case 0x27: eaddr=yreg+7;op_count+=1;break;
	    				case 0x28: eaddr=yreg+8;op_count+=1;break;
	    				case 0x29: eaddr=yreg+9;op_count+=1;break;
	    				case 0x2A: eaddr=yreg+10;op_count+=1;break;
	    				case 0x2B: eaddr=yreg+11;op_count+=1;break;
	    				case 0x2C: eaddr=yreg+12;op_count+=1;break;
	    				case 0x2D: eaddr=yreg+13;op_count+=1;break;
	    				case 0x2E: eaddr=yreg+14;op_count+=1;break;
	    				case 0x2F: eaddr=yreg+15;op_count+=1;break;
	    				case 0x30: eaddr=yreg-16;op_count+=1;break;
	    				case 0x31: eaddr=yreg-15;op_count+=1;break;
	    				case 0x32: eaddr=yreg-14;op_count+=1;break;
	    				case 0x33: eaddr=yreg-13;op_count+=1;break;
	    				case 0x34: eaddr=yreg-12;op_count+=1;break;
	    				case 0x35: eaddr=yreg-11;op_count+=1;break;
	    				case 0x36: eaddr=yreg-10;op_count+=1;break;
	    				case 0x37: eaddr=yreg-9;op_count+=1;break;
	    				case 0x38: eaddr=yreg-8;op_count+=1;break;
	    				case 0x39: eaddr=yreg-7;op_count+=1;break;
	    				case 0x3A: eaddr=yreg-6;op_count+=1;break;
	    				case 0x3B: eaddr=yreg-5;op_count+=1;break;
	    				case 0x3C: eaddr=yreg-4;op_count+=1;break;
	    				case 0x3D: eaddr=yreg-3;op_count+=1;break;
	    				case 0x3E: eaddr=yreg-2;op_count+=1;break;
	    				case 0x3F: eaddr=yreg-1;op_count+=1;break;
	    				case 0x40: eaddr=ureg;op_count+=1;break;
	    				case 0x41: eaddr=ureg+1;op_count+=1;break;
	    				case 0x42: eaddr=ureg+2;op_count+=1;break;
	    				case 0x43: eaddr=ureg+3;op_count+=1;break;
	    				case 0x44: eaddr=ureg+4;op_count+=1;break;
	    				case 0x45: eaddr=ureg+5;op_count+=1;break;
	    				case 0x46: eaddr=ureg+6;op_count+=1;break;
	    				case 0x47: eaddr=ureg+7;op_count+=1;break;
	    				case 0x48: eaddr=ureg+8;op_count+=1;break;
	    				case 0x49: eaddr=ureg+9;op_count+=1;break;
	    				case 0x4A: eaddr=ureg+10;op_count+=1;break;
	    				case 0x4B: eaddr=ureg+11;op_count+=1;break;
	    				case 0x4C: eaddr=ureg+12;op_count+=1;break;
	    				case 0x4D: eaddr=ureg+13;op_count+=1;break;
	    				case 0x4E: eaddr=ureg+14;op_count+=1;break;
	    				case 0x4F: eaddr=ureg+15;op_count+=1;break;
	    				case 0x50: eaddr=ureg-16;op_count+=1;break;
	    				case 0x51: eaddr=ureg-15;op_count+=1;break;
	    				case 0x52: eaddr=ureg-14;op_count+=1;break;
	    				case 0x53: eaddr=ureg-13;op_count+=1;break;
	    				case 0x54: eaddr=ureg-12;op_count+=1;break;
	    				case 0x55: eaddr=ureg-11;op_count+=1;break;
	    				case 0x56: eaddr=ureg-10;op_count+=1;break;
	    				case 0x57: eaddr=ureg-9;op_count+=1;break;
	    				case 0x58: eaddr=ureg-8;op_count+=1;break;
	    				case 0x59: eaddr=ureg-7;op_count+=1;break;
	    				case 0x5A: eaddr=ureg-6;op_count+=1;break;
	    				case 0x5B: eaddr=ureg-5;op_count+=1;break;
	    				case 0x5C: eaddr=ureg-4;op_count+=1;break;
	    				case 0x5D: eaddr=ureg-3;op_count+=1;break;
	    				case 0x5E: eaddr=ureg-2;op_count+=1;break;
	    				case 0x5F: eaddr=ureg-1;op_count+=1;break;
	    				case 0x60: eaddr=sreg;op_count+=1;break;
	    				case 0x61: eaddr=sreg+1;op_count+=1;break;
	    				case 0x62: eaddr=sreg+2;op_count+=1;break;
	    				case 0x63: eaddr=sreg+3;op_count+=1;break;
	    				case 0x64: eaddr=sreg+4;op_count+=1;break;
	    				case 0x65: eaddr=sreg+5;op_count+=1;break;
	    				case 0x66: eaddr=sreg+6;op_count+=1;break;
	    				case 0x67: eaddr=sreg+7;op_count+=1;break;
	    				case 0x68: eaddr=sreg+8;op_count+=1;break;
	    				case 0x69: eaddr=sreg+9;op_count+=1;break;
	    				case 0x6A: eaddr=sreg+10;op_count+=1;break;
	    				case 0x6B: eaddr=sreg+11;op_count+=1;break;
	    				case 0x6C: eaddr=sreg+12;op_count+=1;break;
	    				case 0x6D: eaddr=sreg+13;op_count+=1;break;
	    				case 0x6E: eaddr=sreg+14;op_count+=1;break;
	    				case 0x6F: eaddr=sreg+15;op_count+=1;break;
	    				case 0x70: eaddr=sreg-16;op_count+=1;break;
	    				case 0x71: eaddr=sreg-15;op_count+=1;break;
	    				case 0x72: eaddr=sreg-14;op_count+=1;break;
	    				case 0x73: eaddr=sreg-13;op_count+=1;break;
	    				case 0x74: eaddr=sreg-12;op_count+=1;break;
	    				case 0x75: eaddr=sreg-11;op_count+=1;break;
	    				case 0x76: eaddr=sreg-10;op_count+=1;break;
	    				case 0x77: eaddr=sreg-9;op_count+=1;break;
	    				case 0x78: eaddr=sreg-8;op_count+=1;break;
	    				case 0x79: eaddr=sreg-7;op_count+=1;break;
	    				case 0x7A: eaddr=sreg-6;op_count+=1;break;
	    				case 0x7B: eaddr=sreg-5;op_count+=1;break;
	    				case 0x7C: eaddr=sreg-4;op_count+=1;break;
	    				case 0x7D: eaddr=sreg-3;op_count+=1;break;
	    				case 0x7E: eaddr=sreg-2;op_count+=1;break;
	    				case 0x7F: eaddr=sreg-1;op_count+=1;break;
	    				case 0x80: eaddr=xreg;xreg++;op_count+=2;break;
	    				case 0x81: eaddr=xreg;xreg+=2;op_count+=3;break;
	    				case 0x82: xreg--;eaddr=xreg;op_count+=2;break;
	    				case 0x83: xreg-=2;eaddr=xreg;op_count+=3;break;
	    				case 0x84: eaddr=xreg;break;
	    				case 0x85: eaddr=xreg+SIGNED(breg);op_count+=1;break;
	    				case 0x86: eaddr=xreg+SIGNED(areg);op_count+=1;break;
	    				case 0x88: IMMBYTE(eaddr);eaddr=xreg+SIGNED(eaddr);op_count+=1;break; /* this is a hack to make Vectrex work. It should be op_count+=1. Dunno where the cycle was lost :( */
	    				case 0x89: IMMWORD(eaddr);eaddr+=xreg;op_count+=4;break;
	    				case 0x8B: eaddr=xreg+GETDREG;op_count+=4;break;
	    				case 0x8C: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);op_count+=1;break;
	    				case 0x8D: IMMWORD(eaddr);eaddr+=pcreg;op_count+=5;break;
	    				case 0x8F: IMMWORD(eaddr);op_count+=5;break;
	    				case 0x90: eaddr=xreg;xreg++;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break; /* Indirect ,R+ not in my specs */
	    				case 0x91: eaddr=xreg;xreg+=2;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0x92: xreg--;eaddr=xreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0x93: xreg-=2;eaddr=xreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0x94: eaddr=xreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=3;break;
	    				case 0x95: eaddr=xreg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0x96: eaddr=xreg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0x98: IMMBYTE(eaddr);eaddr=xreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0x99: IMMWORD(eaddr);eaddr+=xreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0x9B: eaddr=xreg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0x9C: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0x9D: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0x9F: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0xA0: eaddr=yreg;yreg++;op_count+=2;break;
	    				case 0xA1: eaddr=yreg;yreg+=2;op_count+=3;break;
	    				case 0xA2: yreg--;eaddr=yreg;op_count+=2;break;
	    				case 0xA3: yreg-=2;eaddr=yreg;op_count+=3;break;
	    				case 0xA4: eaddr=yreg;break;
	    				case 0xA5: eaddr=yreg+SIGNED(breg);op_count+=1;break;
	    				case 0xA6: eaddr=yreg+SIGNED(areg);op_count+=1;break;
	    				case 0xA8: IMMBYTE(eaddr);eaddr=yreg+SIGNED(eaddr);op_count+=1;break;
	    				case 0xA9: IMMWORD(eaddr);eaddr+=yreg;op_count+=4;break;
	    				case 0xAB: eaddr=yreg+GETDREG;op_count+=4;break;
	    				case 0xAC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);op_count+=1;break;
	    				case 0xAD: IMMWORD(eaddr);eaddr+=pcreg;op_count+=5;break;
	    				case 0xAF: IMMWORD(eaddr);op_count+=5;break;
	    				case 0xB0: eaddr=yreg;yreg++;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0xB1: eaddr=yreg;yreg+=2;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0xB2: yreg--;eaddr=yreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0xB3: yreg-=2;eaddr=yreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0xB4: eaddr=yreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=3;break;
	    				case 0xB5: eaddr=yreg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xB6: eaddr=yreg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xB8: IMMBYTE(eaddr);eaddr=yreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xB9: IMMWORD(eaddr);eaddr+=yreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0xBB: eaddr=yreg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0xBC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xBD: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0xBF: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0xC0: eaddr=ureg;ureg++;op_count+=2;break;
	    				case 0xC1: eaddr=ureg;ureg+=2;op_count+=3;break;
	    				case 0xC2: ureg--;eaddr=ureg;op_count+=2;break;
	    				case 0xC3: ureg-=2;eaddr=ureg;op_count+=3;break;
	    				case 0xC4: eaddr=ureg;break;
	    				case 0xC5: eaddr=ureg+SIGNED(breg);op_count+=1;break;
	    				case 0xC6: eaddr=ureg+SIGNED(areg);op_count+=1;break;
	    				case 0xC8: IMMBYTE(eaddr);eaddr=ureg+SIGNED(eaddr);op_count+=1;break;
	    				case 0xC9: IMMWORD(eaddr);eaddr+=ureg;op_count+=4;break;
	    				case 0xCB: eaddr=ureg+GETDREG;op_count+=4;break;
	    				case 0xCC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);op_count+=1;break;
	    				case 0xCD: IMMWORD(eaddr);eaddr+=pcreg;op_count+=5;break;
	    				case 0xCF: IMMWORD(eaddr);op_count+=5;break;
	    				case 0xD0: eaddr=ureg;ureg++;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0xD1: eaddr=ureg;ureg+=2;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0xD2: ureg--;eaddr=ureg;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0xD3: ureg-=2;eaddr=ureg;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0xD4: eaddr=ureg;eaddr=M_RDMEM_WORD(eaddr);op_count+=3;break;
	    				case 0xD5: eaddr=ureg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xD6: eaddr=ureg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xD8: IMMBYTE(eaddr);eaddr=ureg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xD9: IMMWORD(eaddr);eaddr+=ureg;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0xDB: eaddr=ureg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0xDC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xDD: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0xDF: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0xE0: eaddr=sreg;sreg++;op_count+=2;break;
	    				case 0xE1: eaddr=sreg;sreg+=2;op_count+=3;break;
	    				case 0xE2: sreg--;eaddr=sreg;op_count+=2;break;
	    				case 0xE3: sreg-=2;eaddr=sreg;op_count+=3;break;
	    				case 0xE4: eaddr=sreg;break;
	    				case 0xE5: eaddr=sreg+SIGNED(breg);op_count+=1;break;
	    				case 0xE6: eaddr=sreg+SIGNED(areg);op_count+=1;break;
	    				case 0xE8: IMMBYTE(eaddr);eaddr=sreg+SIGNED(eaddr);op_count+=1;break;
	    				case 0xE9: IMMWORD(eaddr);eaddr+=sreg;op_count+=4;break;
	    				case 0xEB: eaddr=sreg+GETDREG;op_count+=4;break;
	    				case 0xEC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);op_count+=1;break;
	    				case 0xED: IMMWORD(eaddr);eaddr+=pcreg;op_count+=5;break;
	    				case 0xEF: IMMWORD(eaddr);op_count+=5;break;
	    				case 0xF0: eaddr=sreg;sreg++;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0xF1: eaddr=sreg;sreg+=2;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0xF2: sreg--;eaddr=sreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=5;break;
	    				case 0xF3: sreg-=2;eaddr=sreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=6;break;
	    				case 0xF4: eaddr=sreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=3;break;
	    				case 0xF5: eaddr=sreg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xF6: eaddr=sreg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xF8: IMMBYTE(eaddr);eaddr=sreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xF9: IMMWORD(eaddr);eaddr+=sreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0xFB: eaddr=sreg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);op_count+=7;break;
	    				case 0xFC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);op_count+=4;break;
	    				case 0xFD: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				case 0xFF: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);op_count+=8;break;
	    				default: eaddr=0;break; /*ILLEGAL*/
				}
			}

			switch( ireg )
			{
				case 0x00: { DIRBYTE(t); r=-t; CLR_NZVC; SET_FLAGS8(0,t,r); M_WRMEM(eaddr,r); } break;
				case 0x03: { DIRBYTE(t); t = ~t; CLR_NZV; SET_NZ8(t); SEC; M_WRMEM(eaddr,t); } break;
				case 0x04: { DIRBYTE(t); CLR_NZC; cc|=(t&0x01); t>>=1; SET_Z8(t); M_WRMEM(eaddr,t); } break;
				case 0x06: { DIRBYTE(t); r=(cc&0x01)<<7; CLR_NZC; cc|=(t&0x01); r |= t>>1; SET_NZ8(r); M_WRMEM(eaddr,r); } break;
				case 0x07: { DIRBYTE(t); CLR_NZC; cc|=(t&0x01); t>>=1; t|=((t&0x40)<<1); SET_NZ8(t); M_WRMEM(eaddr,t); } break;
				case 0x08: { DIRBYTE(t); r=t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); M_WRMEM(eaddr,r); } break;
				case 0x09: { DIRBYTE(t); r = cc&0x01; r |= t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); M_WRMEM(eaddr,r); } break;
				case 0x0a: { DIRBYTE(t); --t; CLR_NZV; SET_FLAGS8D(t); M_WRMEM(eaddr,t); } break;
				case 0x0c: { DIRBYTE(t); ++t; CLR_NZV; SET_FLAGS8I(t); M_WRMEM(eaddr,t); } break;
				case 0x0d: { DIRBYTE(t); CLR_NZV; SET_NZ8(t); } break;
				case 0x0e: DIRECT; pcreg=eaddr;change_pc(pcreg); if (m6809_slapstic) cpu_setOPbase16(pcreg); break;
				case 0x0f: DIRECT; M_WRMEM(eaddr,0); CLR_NZVC; SEZ; break;
				case 0x12: break;
				case 0x13: m6809_ICount = 0; pending_interrupts |= M6809_SYNC; break;
				case 0x16: IMMWORD(eaddr); pcreg+=eaddr;change_pc(pcreg); if ( eaddr == 0xfffd ) m6809_ICount = 0; break;
				case 0x17: IMMWORD(eaddr); PUSHWORD(pcreg); pcreg+=eaddr; change_pc(pcreg); break;
				case 0x19: { byte msn, lsn; word t, cf = 0; msn=areg & 0xf0; lsn=areg & 0x0f; if( lsn>0x09 || cc&0x20 ) cf |= 0x06; if( msn>0x80 && lsn>0x09 ) cf |= 0x60;  if( msn>0x90 || cc&0x01 ) cf |= 0x60; t = cf + areg; CLR_NZV; SET_NZ8((byte)t); SET_C8(t); areg = t; } break;
				case 0x1a: { IMMBYTE(t); cc|=t; } break;
				case 0x1c: { IMMBYTE(t); cc&=t; } break;
				case 0x1d: { t = SIGNED(breg); SETDREG(t); CLR_NZV; SET_NZ16(t); } break;
				case 0x1e: { word t1=0,t2=0; byte tb; IMMBYTE(tb); GETREG(t1,tb>>4); GETREG(t2,tb&15); SETREG(t2,tb>>4); SETREG(t1,tb&15); } break;
				case 0x1f: { byte tb; word t=0; IMMBYTE(tb); GETREG(t,tb>>4); SETREG(t,tb&15); } break;
				case 0x20: { IMMBYTE(t);pcreg+=SIGNED(t);change_pc(pcreg); if (t==0xfe) m6809_ICount = 0; } break;
				case 0x21: { IMMBYTE(t); } break;
				case 0x22: { BRANCH(!(cc&0x05)); } break;
				case 0x23: { BRANCH(cc&0x05); } break;
				case 0x24: { BRANCH(!(cc&0x01)); } break;
				case 0x25: { BRANCH(cc&0x01); } break;
				case 0x26: { BRANCH(!(cc&0x04)); } break;
				case 0x27: { BRANCH(cc&0x04); } break;
				case 0x28: { BRANCH(!(cc&0x02)); } break;
				case 0x29: { BRANCH(cc&0x02); } break;
				case 0x2a: { BRANCH(!(cc&0x08)); } break;
				case 0x2b: { BRANCH(cc&0x08); } break;
				case 0x2c: { BRANCH(!NXORV);} break;
				case 0x2d: { BRANCH(NXORV); } break;
				case 0x2e: { BRANCH(!(NXORV||cc&0x04)); } break;
				case 0x2f: { BRANCH(NXORV||cc&0x04); } break;
				case 0x30: xreg=eaddr; CLR_Z; SET_Z(xreg); break;
				case 0x31: yreg=eaddr; CLR_Z; SET_Z(yreg); break;
				case 0x32: sreg=eaddr; break;
				case 0x33: ureg=eaddr; break;
				case 0x34: { IMMBYTE(t); if(t&0x80) PUSHWORD(pcreg); if(t&0x40) PUSHWORD(ureg); if(t&0x20) PUSHWORD(yreg); if(t&0x10) PUSHWORD(xreg); if(t&0x08) PUSHBYTE(dpreg); if(t&0x04) PUSHBYTE(breg); if(t&0x02) PUSHBYTE(areg); if(t&0x01) PUSHBYTE(cc); } break;
				case 0x35: { IMMBYTE(t); if(t&0x01) PULLBYTE(cc); if(t&0x02) PULLBYTE(areg); if(t&0x04) PULLBYTE(breg); if(t&0x08) PULLBYTE(dpreg); if(t&0x10) PULLWORD(xreg); if(t&0x20) PULLWORD(yreg); if(t&0x40) PULLWORD(ureg); if(t&0x80){ PULLWORD(pcreg);change_pc(pcreg); } } break;
				case 0x36: { IMMBYTE(t); if(t&0x80) PSHUWORD(pcreg); if(t&0x40) PSHUWORD(sreg); if(t&0x20) PSHUWORD(yreg); if(t&0x10) PSHUWORD(xreg); if(t&0x08) PSHUBYTE(dpreg); if(t&0x04) PSHUBYTE(breg); if(t&0x02) PSHUBYTE(areg); if(t&0x01) PSHUBYTE(cc); } break;
				case 0x37: { IMMBYTE(t); if(t&0x01) PULUBYTE(cc); if(t&0x02) PULUBYTE(areg); if(t&0x04) PULUBYTE(breg); if(t&0x08) PULUBYTE(dpreg); if(t&0x10) PULUWORD(xreg); if(t&0x20) PULUWORD(yreg); if(t&0x40) PULUWORD(sreg); if(t&0x80) { PULUWORD(pcreg);change_pc(pcreg); } } break;
				case 0x39: { PULLWORD(pcreg);change_pc(pcreg); } break;
				case 0x3a: xreg += breg; break;
				case 0x3b: { PULLBYTE(cc); t=cc&0x80; if(t) { m6809_ICount -= 9; PULLBYTE(areg); PULLBYTE(breg); PULLBYTE(dpreg); PULLWORD(xreg); PULLWORD(yreg); PULLWORD(ureg); } PULLWORD(pcreg);change_pc(pcreg); } break;
				case 0x3c: { IMMBYTE(t); cc&=t; if ((pending_interrupts & M6809_INT_IRQ) != 0 && (cc & 0x10) == 0) ; else if ((pending_interrupts & M6809_INT_FIRQ) != 0 && (cc & 0x40) == 0) ;	else { m6809_ICount = 0; pending_interrupts |= M6809_CWAI; } } break;
				case 0x3d: { t=areg*breg; CLR_ZC; SET_Z16(t); if(t&0x80) SEC; SETDREG(t); } break;
				case 0x3f: { cc|=0x80; PUSHWORD(pcreg); PUSHWORD(ureg); PUSHWORD(yreg); PUSHWORD(xreg); PUSHBYTE(dpreg); PUSHBYTE(breg); PUSHBYTE(areg); PUSHBYTE(cc); cc|=0x50; pcreg = M_RDMEM_WORD(0xfffa);change_pc(pcreg); } break;
				case 0x40: { r=-areg; CLR_NZVC; SET_FLAGS8(0,areg,r); areg=r; } break;
				case 0x43: { areg = ~areg; CLR_NZV; SET_NZ8(areg); SEC; } break;
				case 0x44: { CLR_NZC; cc|=(areg&0x01); areg>>=1; SET_Z8(areg); } break;
				case 0x46: { byte r; r=(cc&0x01)<<7; CLR_NZC; cc|=(areg&0x01); r |= areg>>1; SET_NZ8(r); areg=r; } break;
				case 0x47: { CLR_NZC; cc|=(areg&0x01); areg>>=1; areg|=((areg&0x40)<<1); SET_NZ8(areg); } break;
				case 0x48: { r=areg<<1; CLR_NZVC; SET_FLAGS8(areg,areg,r); areg=r; } break;
				case 0x49: { t = areg; r = cc&0x01; r |= t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); areg=r; } break;
				case 0x4a: --areg; CLR_NZV; SET_FLAGS8D(areg); break;
				case 0x4c: ++areg; CLR_NZV; SET_FLAGS8I(areg); break;
				case 0x4d: CLR_NZV; SET_NZ8(areg); break;
				case 0x4f: areg=0; CLR_NZVC; SEZ; break;
				case 0x50: { r=-breg; CLR_NZVC; SET_FLAGS8(0,breg,r); breg=r; } break;
				case 0x53: breg = ~breg; CLR_NZV; SET_NZ8(breg); SEC; break;
				case 0x54: CLR_NZC; cc|=(breg&0x01); breg>>=1; SET_Z8(breg); break;
				case 0x56: { byte r; r=(cc&0x01)<<7; CLR_NZC; cc|=(breg&0x01); r |= breg>>1; SET_NZ8(r); breg=r; } break;
				case 0x57: CLR_NZC; cc|=(breg&0x01); breg>>=1; breg|=((breg&0x40)<<1); SET_NZ8(breg); break;
				case 0x58: { r=breg<<1; CLR_NZVC; SET_FLAGS8(breg,breg,r); breg=r; } break;
				case 0x59: { t = breg; r = cc&0x01; r |= t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); breg=r; } break;
				case 0x5a: --breg; CLR_NZV; SET_FLAGS8D(breg); break;
				case 0x5c: ++breg; CLR_NZV; SET_FLAGS8I(breg); break;
				case 0x5d: CLR_NZV; SET_NZ8(breg); break;
				case 0x5f: breg=0; CLR_NZVC; SEZ; break;
				case 0x60: { t=M_RDMEM(eaddr); r=-t; CLR_NZVC; SET_FLAGS8(0,t,r); M_WRMEM(eaddr,r); } break;
				case 0x63: { t = ~M_RDMEM(eaddr); CLR_NZV; SET_NZ8(t); SEC; M_WRMEM(eaddr,t); } break;
				case 0x64: { t=M_RDMEM(eaddr); CLR_NZC; cc|=(t&0x01); t>>=1; SET_Z8(t); M_WRMEM(eaddr,t); } break;
				case 0x66: { t=M_RDMEM(eaddr); r=(cc&0x01)<<7; CLR_NZC; cc|=(t&0x01); r |= t>>1; SET_NZ8(r); M_WRMEM(eaddr,r); } break;
				case 0x67: { t=M_RDMEM(eaddr); CLR_NZC; cc|=(t&0x01); t>>=1; t|=((t&0x40)<<1); SET_NZ8(t); M_WRMEM(eaddr,t); } break;
				case 0x68: { t=M_RDMEM(eaddr); r=t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); M_WRMEM(eaddr,r); } break;
				case 0x69: { t=M_RDMEM(eaddr); r = cc&0x01; r |= t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); M_WRMEM(eaddr,r); } break;
				case 0x6a: { t=M_RDMEM(eaddr)-1; CLR_NZV; SET_FLAGS8D(t); M_WRMEM(eaddr,t); } break;
				case 0x6c: { t=M_RDMEM(eaddr)+1; CLR_NZV; SET_FLAGS8I(t); M_WRMEM(eaddr,t); } break;
				case 0x6d: { t=M_RDMEM(eaddr); CLR_NZV; SET_NZ8(t); } break;
				case 0x6e: pcreg=eaddr; change_pc(pcreg); if (m6809_slapstic) cpu_setOPbase16(pcreg); break;
				case 0x6f: M_WRMEM(eaddr,0); CLR_NZVC; SEZ; break;
				case 0x70: { EXTBYTE(t); r=-t; CLR_NZVC; SET_FLAGS8(0,t,r); M_WRMEM(eaddr,r); } break;
				case 0x73: { EXTBYTE(t); t = ~t; CLR_NZV; SET_NZ8(t); SEC; M_WRMEM(eaddr,t); } break;
				case 0x74: { EXTBYTE(t); CLR_NZC; cc|=(t&0x01); t>>=1; SET_Z8(t); M_WRMEM(eaddr,t); } break;
				case 0x76: { EXTBYTE(t); r=(cc&0x01)<<7; CLR_NZC; cc|=(t&0x01); r |= t>>1; SET_NZ8(r); M_WRMEM(eaddr,r); } break;
				case 0x77: { EXTBYTE(t); CLR_NZC; cc|=(t&0x01); t>>=1; t|=((t&0x40)<<1); SET_NZ8(t); M_WRMEM(eaddr,t); } break;
				case 0x78: { EXTBYTE(t); r=t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); M_WRMEM(eaddr,r); } break;
				case 0x79: { EXTBYTE(t); r = cc&0x01; r |= t<<1; CLR_NZVC; SET_FLAGS8(t,t,r); M_WRMEM(eaddr,r); } break;
				case 0x7a: { EXTBYTE(t); --t; CLR_NZV; SET_FLAGS8D(t); M_WRMEM(eaddr,t); } break;
				case 0x7c: { EXTBYTE(t); ++t; CLR_NZV; SET_FLAGS8I(t); M_WRMEM(eaddr,t); } break;
				case 0x7d: { EXTBYTE(t); CLR_NZV; SET_NZ8(t); } break;
				case 0x7e: EXTENDED; pcreg=eaddr;change_pc(pcreg); if (m6809_slapstic) cpu_setOPbase16(pcreg); break;
				case 0x7f: EXTENDED; M_WRMEM(eaddr,0); CLR_NZVC; SEZ; break;
				case 0x80: { IMMBYTE(t); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0x81: { IMMBYTE(t); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); } break;
				case 0x82: { IMMBYTE(t); r = areg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0x83: { IMMWORD(b); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0x84: { IMMBYTE(t); areg &= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0x85: { IMMBYTE(t); r = areg&t; CLR_NZV; SET_NZ8(r); } break;
				case 0x86: IMMBYTE(areg); CLR_NZV; SET_NZ8(areg); break;
				case 0x87: CLR_NZV; SET_NZ8(areg); IMM8; M_WRMEM(eaddr,areg); break; /* ILLEGAL? */
				case 0x88: { IMMBYTE(t); areg ^= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0x89: { IMMBYTE(t); r = areg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0x8a: { IMMBYTE(t); areg |= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0x8b: { IMMBYTE(t); r = areg+t; CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0x8c: { IMMWORD(b); d = xreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); } break;
				case 0x8d: { IMMBYTE(t); PUSHWORD(pcreg); pcreg += SIGNED(t);change_pc(pcreg); } break;
				case 0x8e: IMMWORD(xreg); CLR_NZV; SET_NZ16(xreg); break;
				case 0x8f: CLR_NZV; SET_NZ16(xreg); IMM16; M_WRMEM_WORD(eaddr,xreg); break; /* ILLEGAL? */
				case 0x90: { DIRBYTE(t); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0x91: { DIRBYTE(t); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); } break;
				case 0x92: { DIRBYTE(t); r = areg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0x93: { DIRWORD(b); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0x94: { DIRBYTE(t); areg &= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0x95: { DIRBYTE(t); r = areg&t; CLR_NZV; SET_NZ8(r); } break;
				case 0x96: DIRBYTE(areg); CLR_NZV; SET_NZ8(areg); break;
				case 0x97: CLR_NZV; SET_NZ8(areg); DIRECT; M_WRMEM(eaddr,areg); break;
				case 0x98: { DIRBYTE(t); areg ^= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0x99: { DIRBYTE(t); r = areg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0x9a: { DIRBYTE(t); areg |= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0x9b: { DIRBYTE(t); r = areg+t; CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0x9c: { DIRWORD(b); d = xreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); } break;
				case 0x9d: DIRECT; PUSHWORD(pcreg); pcreg = eaddr;change_pc(pcreg); if (m6809_slapstic) cpu_setOPbase16(pcreg); break;
				case 0x9e: DIRWORD(xreg); CLR_NZV; SET_NZ16(xreg); break;
				case 0x9f: CLR_NZV; SET_NZ16(xreg); DIRECT; M_WRMEM_WORD(eaddr,xreg); break;
				case 0xa0: { t = M_RDMEM(eaddr); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0xa1: { t = M_RDMEM(eaddr); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); } break;
				case 0xa2: { t = M_RDMEM(eaddr); r = areg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0xa3: { b = M_RDMEM_WORD(eaddr); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0xa4: areg &= M_RDMEM(eaddr); CLR_NZV; SET_NZ8(areg); break;
				case 0xa5: { byte r; r = areg&M_RDMEM(eaddr); CLR_NZV; SET_NZ8(r); } break;
				case 0xa6: areg = M_RDMEM(eaddr); CLR_NZV; SET_NZ8(areg); break;
				case 0xa7: CLR_NZV; SET_NZ8(areg); M_WRMEM(eaddr,areg); break;
				case 0xa8: areg ^= M_RDMEM(eaddr); CLR_NZV; SET_NZ8(areg); break;
				case 0xa9: { t = M_RDMEM(eaddr); r = areg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0xaa: areg |= M_RDMEM(eaddr); CLR_NZV; SET_NZ8(areg); break;
				case 0xab: { t = M_RDMEM(eaddr); r = areg+t; CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0xac: { b = M_RDMEM_WORD(eaddr); d = xreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); } break;
				case 0xad: PUSHWORD(pcreg); pcreg = eaddr;change_pc(pcreg); if (m6809_slapstic) cpu_setOPbase16(pcreg); break;
				case 0xae: xreg = M_RDMEM_WORD(eaddr); CLR_NZV; SET_NZ16(xreg); break;
				case 0xaf: CLR_NZV; SET_NZ16(xreg); M_WRMEM_WORD(eaddr,xreg); break;
				case 0xb0: { EXTBYTE(t); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0xb1: { EXTBYTE(t); r = areg-t; CLR_NZVC; SET_FLAGS8(areg,t,r); } break;
				case 0xb2: { EXTBYTE(t); r = areg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(areg,t,r); areg = r; } break;
				case 0xb3: { EXTWORD(b); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0xb4: { EXTBYTE(t); areg &= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0xb5: { EXTBYTE(t); r = areg&t; CLR_NZV; SET_NZ8(r); } break;
				case 0xb6: EXTBYTE(areg); CLR_NZV; SET_NZ8(areg); break;
				case 0xb7: CLR_NZV; SET_NZ8(areg); EXTENDED; M_WRMEM(eaddr,areg); break;
				case 0xb8: { EXTBYTE(t); areg ^= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0xb9: { EXTBYTE(t); r = areg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0xba: { EXTBYTE(t); areg |= t; CLR_NZV; SET_NZ8(areg); } break;
				case 0xbb: { EXTBYTE(t); r = areg+t; CLR_HNZVC; SET_FLAGS8(areg,t,r); SET_H(areg,t,r); areg = r; } break;
				case 0xbc: { EXTWORD(b); d = xreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); } break;
				case 0xbd: EXTENDED; PUSHWORD(pcreg); pcreg = eaddr;change_pc(pcreg); if (m6809_slapstic) cpu_setOPbase16(pcreg); break;
				case 0xbe: EXTWORD(xreg); CLR_NZV; SET_NZ16(xreg); break;
				case 0xbf: CLR_NZV; SET_NZ16(xreg); EXTENDED; M_WRMEM_WORD(eaddr,xreg); break;
				case 0xc0: { IMMBYTE(t); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xc1: { IMMBYTE(t); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); } break;
				case 0xc2: { IMMBYTE(t); r = breg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xc3: { IMMWORD(b); d = GETDREG; r = d+b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0xc4: { IMMBYTE(t); breg &= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xc5: { IMMBYTE(t); r = breg&t; CLR_NZV; SET_NZ8(r); } break;
				case 0xc6: IMMBYTE(breg); CLR_NZV; SET_NZ8(breg); break;
				case 0xc7: CLR_NZV; SET_NZ8(breg); IMM8; M_WRMEM(eaddr,breg); break; /* ILLEGAL? */
				case 0xc8: { IMMBYTE(t); breg ^= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xc9: { IMMBYTE(t); r = breg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xca: { IMMBYTE(t); breg |= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xcb: { IMMBYTE(t); r = breg+t; CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xcc: { IMMWORD(t); SETDREG(t); CLR_NZV; SET_NZ16(t); } break;
				case 0xcd: { IMM16; t=GETDREG; CLR_NZV; SET_NZ16(t); M_WRMEM_WORD(eaddr,t); } break; /* ILLEGAL? */
				case 0xce: IMMWORD(ureg); CLR_NZV; SET_NZ16(ureg); break;
				case 0xcf: CLR_NZV; SET_NZ16(ureg); IMM16; M_WRMEM_WORD(eaddr,ureg); break; /* ILLEGAL? */
				case 0xd0: { DIRBYTE(t); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xd1: { DIRBYTE(t); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); } break;
				case 0xd2: { DIRBYTE(t); r = breg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xd3: { DIRWORD(b); d = GETDREG; r = d+b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0xd4: { DIRBYTE(t); breg &= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xd5: { DIRBYTE(t); r = breg&t; CLR_NZV; SET_NZ8(r); } break;
				case 0xd6: DIRBYTE(breg); CLR_NZV; SET_NZ8(breg); break;
				case 0xd7: CLR_NZV; SET_NZ8(breg); DIRECT; M_WRMEM(eaddr,breg); break;
				case 0xd8: { DIRBYTE(t); breg ^= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xd9: { DIRBYTE(t); r = breg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xda: { DIRBYTE(t); breg |= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xdb: { DIRBYTE(t); r = breg+t; CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xdc: { DIRWORD(t); SETDREG(t); CLR_NZV; SET_NZ16(t); } break;
				case 0xdd: { DIRECT; t=GETDREG; CLR_NZV; SET_NZ16(t); M_WRMEM_WORD(eaddr,t); } break;
				case 0xde: DIRWORD(ureg); CLR_NZV; SET_NZ16(ureg); break;
				case 0xdf: CLR_NZV; SET_NZ16(ureg); DIRECT; M_WRMEM_WORD(eaddr,ureg); break;
				case 0xe0: { t = M_RDMEM(eaddr); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xe1: { t = M_RDMEM(eaddr); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); } break;
				case 0xe2: { t = M_RDMEM(eaddr); r = breg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xe3: { b = M_RDMEM_WORD(eaddr); d = GETDREG; r = d+b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0xe4: breg &= M_RDMEM(eaddr); CLR_NZV; SET_NZ8(breg); break;
				case 0xe5: { byte r; r = breg&M_RDMEM(eaddr); CLR_NZV; SET_NZ8(r); } break;
				case 0xe6: breg = M_RDMEM(eaddr); CLR_NZV; SET_NZ8(breg); break;
				case 0xe7: CLR_NZV; SET_NZ8(breg); M_WRMEM(eaddr,breg); break;
				case 0xe8: breg ^= M_RDMEM(eaddr); CLR_NZV; SET_NZ8(breg); break;
				case 0xe9: { t = M_RDMEM(eaddr); r = breg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xea: breg |= M_RDMEM(eaddr); CLR_NZV; SET_NZ8(breg); break;
				case 0xeb: { t = M_RDMEM(eaddr); r = breg+t; CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xec: { t = M_RDMEM_WORD(eaddr); SETDREG(t); CLR_NZV; SET_NZ16(t); } break;
				case 0xed: { t=GETDREG; CLR_NZV; SET_NZ16(t); M_WRMEM_WORD(eaddr,t); } break;
				case 0xee: ureg = M_RDMEM_WORD(eaddr); CLR_NZV; SET_NZ16(ureg); break;
				case 0xef: CLR_NZV; SET_NZ16(ureg); M_WRMEM_WORD(eaddr,ureg); break;
				case 0xf0: { EXTBYTE(t); r = breg-t;  CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xf1: { EXTBYTE(t); r = breg-t; CLR_NZVC; SET_FLAGS8(breg,t,r); } break;
				case 0xf2: { EXTBYTE(t); r = breg-t-(cc&0x01); CLR_NZVC; SET_FLAGS8(breg,t,r); breg = r; } break;
				case 0xf3: { EXTWORD(b); d = GETDREG; r = d+b; CLR_NZVC; SET_FLAGS16(d,b,r); SETDREG(r); } break;
				case 0xf4: { EXTBYTE(t); breg &= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xf5: { EXTBYTE(t); r = breg&t; CLR_NZV; SET_NZ8(r); } break;
				case 0xf6: EXTBYTE(breg); CLR_NZV; SET_NZ8(breg); break;
				case 0xf7: CLR_NZV; SET_NZ8(breg); EXTENDED; M_WRMEM(eaddr,breg); break;
				case 0xf8: { EXTBYTE(t); breg ^= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xf9: { EXTBYTE(t); r = breg+t+(cc&0x01); CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xfa: { EXTBYTE(t); breg |= t; CLR_NZV; SET_NZ8(breg); } break;
				case 0xfb: { EXTBYTE(t); r = breg+t; CLR_HNZVC; SET_FLAGS8(breg,t,r); SET_H(breg,t,r); breg = r; } break;
				case 0xfc: { EXTWORD(t); SETDREG(t); CLR_NZV; SET_NZ16(t); } break;
				case 0xfd: { EXTENDED; t=GETDREG; CLR_NZV; SET_NZ16(t); M_WRMEM_WORD(eaddr,t); } break;
				case 0xfe: EXTWORD(ureg); CLR_NZV; SET_NZ16(ureg); break;
				case 0xff: CLR_NZV; SET_NZ16(ureg); EXTENDED; M_WRMEM_WORD(eaddr,ureg); break;
				default: break;
			}
		}
		else
		{
			byte ireg2;
			ireg2=M_RDOP(pcreg++);

			if( (op_count=cycles2[ireg2]) &0x80 ) {
				switch((byte)M_RDOP_ARG(pcreg++))
				{
	    				case 0x00: eaddr=xreg;break;
	    				case 0x01: eaddr=xreg+1;break;
	    				case 0x02: eaddr=xreg+2;break;
	    				case 0x03: eaddr=xreg+3;break;
	    				case 0x04: eaddr=xreg+4;break;
	    				case 0x05: eaddr=xreg+5;break;
	    				case 0x06: eaddr=xreg+6;break;
	    				case 0x07: eaddr=xreg+7;break;
	    				case 0x08: eaddr=xreg+8;break;
	    				case 0x09: eaddr=xreg+9;break;
	    				case 0x0A: eaddr=xreg+10;break;
	    				case 0x0B: eaddr=xreg+11;break;
	    				case 0x0C: eaddr=xreg+12;break;
	    				case 0x0D: eaddr=xreg+13;break;
	    				case 0x0E: eaddr=xreg+14;break;
	    				case 0x0F: eaddr=xreg+15;break;
	    				case 0x10: eaddr=xreg-16;break;
	    				case 0x11: eaddr=xreg-15;break;
	    				case 0x12: eaddr=xreg-14;break;
	    				case 0x13: eaddr=xreg-13;break;
	    				case 0x14: eaddr=xreg-12;break;
	    				case 0x15: eaddr=xreg-11;break;
	    				case 0x16: eaddr=xreg-10;break;
	    				case 0x17: eaddr=xreg-9;break;
	    				case 0x18: eaddr=xreg-8;break;
	    				case 0x19: eaddr=xreg-7;break;
	    				case 0x1A: eaddr=xreg-6;break;
	    				case 0x1B: eaddr=xreg-5;break;
	    				case 0x1C: eaddr=xreg-4;break;
	    				case 0x1D: eaddr=xreg-3;break;
	    				case 0x1E: eaddr=xreg-2;break;
	    				case 0x1F: eaddr=xreg-1;break;
	    				case 0x20: eaddr=yreg;break;
	    				case 0x21: eaddr=yreg+1;break;
	    				case 0x22: eaddr=yreg+2;break;
	    				case 0x23: eaddr=yreg+3;break;
	    				case 0x24: eaddr=yreg+4;break;
	    				case 0x25: eaddr=yreg+5;break;
	    				case 0x26: eaddr=yreg+6;break;
	    				case 0x27: eaddr=yreg+7;break;
	    				case 0x28: eaddr=yreg+8;break;
	    				case 0x29: eaddr=yreg+9;break;
	    				case 0x2A: eaddr=yreg+10;break;
	    				case 0x2B: eaddr=yreg+11;break;
	    				case 0x2C: eaddr=yreg+12;break;
	    				case 0x2D: eaddr=yreg+13;break;
	    				case 0x2E: eaddr=yreg+14;break;
	    				case 0x2F: eaddr=yreg+15;break;
	    				case 0x30: eaddr=yreg-16;break;
	    				case 0x31: eaddr=yreg-15;break;
	    				case 0x32: eaddr=yreg-14;break;
	    				case 0x33: eaddr=yreg-13;break;
	    				case 0x34: eaddr=yreg-12;break;
	    				case 0x35: eaddr=yreg-11;break;
	    				case 0x36: eaddr=yreg-10;break;
	    				case 0x37: eaddr=yreg-9;break;
	    				case 0x38: eaddr=yreg-8;break;
	    				case 0x39: eaddr=yreg-7;break;
	    				case 0x3A: eaddr=yreg-6;break;
	    				case 0x3B: eaddr=yreg-5;break;
	    				case 0x3C: eaddr=yreg-4;break;
	    				case 0x3D: eaddr=yreg-3;break;
	    				case 0x3E: eaddr=yreg-2;break;
	    				case 0x3F: eaddr=yreg-1;break;
	    				case 0x40: eaddr=ureg;break;
	    				case 0x41: eaddr=ureg+1;break;
	    				case 0x42: eaddr=ureg+2;break;
	    				case 0x43: eaddr=ureg+3;break;
	    				case 0x44: eaddr=ureg+4;break;
	    				case 0x45: eaddr=ureg+5;break;
	    				case 0x46: eaddr=ureg+6;break;
	    				case 0x47: eaddr=ureg+7;break;
	    				case 0x48: eaddr=ureg+8;break;
	    				case 0x49: eaddr=ureg+9;break;
	    				case 0x4A: eaddr=ureg+10;break;
	    				case 0x4B: eaddr=ureg+11;break;
	    				case 0x4C: eaddr=ureg+12;break;
	    				case 0x4D: eaddr=ureg+13;break;
	    				case 0x4E: eaddr=ureg+14;break;
	    				case 0x4F: eaddr=ureg+15;break;
	    				case 0x50: eaddr=ureg-16;break;
	    				case 0x51: eaddr=ureg-15;break;
	    				case 0x52: eaddr=ureg-14;break;
	    				case 0x53: eaddr=ureg-13;break;
	    				case 0x54: eaddr=ureg-12;break;
	    				case 0x55: eaddr=ureg-11;break;
	    				case 0x56: eaddr=ureg-10;break;
	    				case 0x57: eaddr=ureg-9;break;
	    				case 0x58: eaddr=ureg-8;break;
	    				case 0x59: eaddr=ureg-7;break;
	    				case 0x5A: eaddr=ureg-6;break;
	    				case 0x5B: eaddr=ureg-5;break;
	    				case 0x5C: eaddr=ureg-4;break;
	    				case 0x5D: eaddr=ureg-3;break;
	    				case 0x5E: eaddr=ureg-2;break;
	    				case 0x5F: eaddr=ureg-1;break;
	    				case 0x60: eaddr=sreg;break;
	    				case 0x61: eaddr=sreg+1;break;
	    				case 0x62: eaddr=sreg+2;break;
	    				case 0x63: eaddr=sreg+3;break;
	    				case 0x64: eaddr=sreg+4;break;
	    				case 0x65: eaddr=sreg+5;break;
	    				case 0x66: eaddr=sreg+6;break;
	    				case 0x67: eaddr=sreg+7;break;
	    				case 0x68: eaddr=sreg+8;break;
	    				case 0x69: eaddr=sreg+9;break;
	    				case 0x6A: eaddr=sreg+10;break;
	    				case 0x6B: eaddr=sreg+11;break;
	    				case 0x6C: eaddr=sreg+12;break;
	    				case 0x6D: eaddr=sreg+13;break;
	    				case 0x6E: eaddr=sreg+14;break;
	    				case 0x6F: eaddr=sreg+15;break;
	    				case 0x70: eaddr=sreg-16;break;
	    				case 0x71: eaddr=sreg-15;break;
	    				case 0x72: eaddr=sreg-14;break;
	    				case 0x73: eaddr=sreg-13;break;
	    				case 0x74: eaddr=sreg-12;break;
	    				case 0x75: eaddr=sreg-11;break;
	    				case 0x76: eaddr=sreg-10;break;
	    				case 0x77: eaddr=sreg-9;break;
	    				case 0x78: eaddr=sreg-8;break;
	    				case 0x79: eaddr=sreg-7;break;
	    				case 0x7A: eaddr=sreg-6;break;
	    				case 0x7B: eaddr=sreg-5;break;
	    				case 0x7C: eaddr=sreg-4;break;
	    				case 0x7D: eaddr=sreg-3;break;
	    				case 0x7E: eaddr=sreg-2;break;
	    				case 0x7F: eaddr=sreg-1;break;
	    				case 0x80: eaddr=xreg;xreg++;break;
	    				case 0x81: eaddr=xreg;xreg+=2;break;
	    				case 0x82: xreg--;eaddr=xreg;break;
	    				case 0x83: xreg-=2;eaddr=xreg;break;
	    				case 0x84: eaddr=xreg;break;
	    				case 0x85: eaddr=xreg+SIGNED(breg);break;
	    				case 0x86: eaddr=xreg+SIGNED(areg);break;
	    				case 0x88: IMMBYTE(eaddr);eaddr=xreg+SIGNED(eaddr);break; /* this is a hack to make Vectrex work. It should be ec=1. Dunno where the cycle was lost :( */
	    				case 0x89: IMMWORD(eaddr);eaddr+=xreg;break;
	    				case 0x8B: eaddr=xreg+GETDREG;break;
	    				case 0x8C: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);break;
	    				case 0x8D: IMMWORD(eaddr);eaddr+=pcreg;break;
	    				case 0x8F: IMMWORD(eaddr);break;
	    				case 0x90: eaddr=xreg;xreg++;eaddr=M_RDMEM_WORD(eaddr);break; /* Indirect ,R+ not in my specs */
	    				case 0x91: eaddr=xreg;xreg+=2;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x92: xreg--;eaddr=xreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x93: xreg-=2;eaddr=xreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x94: eaddr=xreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x95: eaddr=xreg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x96: eaddr=xreg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x98: IMMBYTE(eaddr);eaddr=xreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x99: IMMWORD(eaddr);eaddr+=xreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x9B: eaddr=xreg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x9C: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x9D: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0x9F: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xA0: eaddr=yreg;yreg++;break;
	    				case 0xA1: eaddr=yreg;yreg+=2;break;
	    				case 0xA2: yreg--;eaddr=yreg;break;
	    				case 0xA3: yreg-=2;eaddr=yreg;break;
	    				case 0xA4: eaddr=yreg;break;
	    				case 0xA5: eaddr=yreg+SIGNED(breg);break;
	    				case 0xA6: eaddr=yreg+SIGNED(areg);break;
	    				case 0xA8: IMMBYTE(eaddr);eaddr=yreg+SIGNED(eaddr);break;
	    				case 0xA9: IMMWORD(eaddr);eaddr+=yreg;break;
	    				case 0xAB: eaddr=yreg+GETDREG;break;
	    				case 0xAC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);break;
	    				case 0xAD: IMMWORD(eaddr);eaddr+=pcreg;break;
	    				case 0xAF: IMMWORD(eaddr);break;
	    				case 0xB0: eaddr=yreg;yreg++;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB1: eaddr=yreg;yreg+=2;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB2: yreg--;eaddr=yreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB3: yreg-=2;eaddr=yreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB4: eaddr=yreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB5: eaddr=yreg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB6: eaddr=yreg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB8: IMMBYTE(eaddr);eaddr=yreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xB9: IMMWORD(eaddr);eaddr+=yreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xBB: eaddr=yreg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xBC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xBD: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xBF: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xC0: eaddr=ureg;ureg++;break;
	    				case 0xC1: eaddr=ureg;ureg+=2;break;
	    				case 0xC2: ureg--;eaddr=ureg;break;
	    				case 0xC3: ureg-=2;eaddr=ureg;break;
	    				case 0xC4: eaddr=ureg;break;
	    				case 0xC5: eaddr=ureg+SIGNED(breg);break;
	    				case 0xC6: eaddr=ureg+SIGNED(areg);break;
	    				case 0xC8: IMMBYTE(eaddr);eaddr=ureg+SIGNED(eaddr);break;
	    				case 0xC9: IMMWORD(eaddr);eaddr+=ureg;break;
	    				case 0xCB: eaddr=ureg+GETDREG;break;
	    				case 0xCC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);break;
	    				case 0xCD: IMMWORD(eaddr);eaddr+=pcreg;break;
	    				case 0xCF: IMMWORD(eaddr);break;
	    				case 0xD0: eaddr=ureg;ureg++;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD1: eaddr=ureg;ureg+=2;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD2: ureg--;eaddr=ureg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD3: ureg-=2;eaddr=ureg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD4: eaddr=ureg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD5: eaddr=ureg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD6: eaddr=ureg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD8: IMMBYTE(eaddr);eaddr=ureg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xD9: IMMWORD(eaddr);eaddr+=ureg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xDB: eaddr=ureg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xDC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xDD: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xDF: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xE0: eaddr=sreg;sreg++;break;
	    				case 0xE1: eaddr=sreg;sreg+=2;break;
	    				case 0xE2: sreg--;eaddr=sreg;break;
	    				case 0xE3: sreg-=2;eaddr=sreg;break;
	    				case 0xE4: eaddr=sreg;break;
	    				case 0xE5: eaddr=sreg+SIGNED(breg);break;
	    				case 0xE6: eaddr=sreg+SIGNED(areg);break;
	    				case 0xE8: IMMBYTE(eaddr);eaddr=sreg+SIGNED(eaddr);break;
	    				case 0xE9: IMMWORD(eaddr);eaddr+=sreg;break;
	    				case 0xEB: eaddr=sreg+GETDREG;break;
	    				case 0xEC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);break;
	    				case 0xED: IMMWORD(eaddr);eaddr+=pcreg;break;
	    				case 0xEF: IMMWORD(eaddr);break;
	    				case 0xF0: eaddr=sreg;sreg++;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF1: eaddr=sreg;sreg+=2;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF2: sreg--;eaddr=sreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF3: sreg-=2;eaddr=sreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF4: eaddr=sreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF5: eaddr=sreg+SIGNED(breg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF6: eaddr=sreg+SIGNED(areg);eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF8: IMMBYTE(eaddr);eaddr=sreg+SIGNED(eaddr);
	    			           	eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xF9: IMMWORD(eaddr);eaddr+=sreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xFB: eaddr=sreg+GETDREG;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xFC: IMMBYTE(eaddr);eaddr=pcreg+SIGNED(eaddr);
	    				           eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xFD: IMMWORD(eaddr);eaddr+=pcreg;eaddr=M_RDMEM_WORD(eaddr);break;
	    				case 0xFF: IMMWORD(eaddr);eaddr=M_RDMEM_WORD(eaddr);break;
	    				default: eaddr=0;break; /*ILLEGAL*/
				}
			}

			if( ireg == 0x10 ){
				switch( ireg2 )	/* 10xx */
				{
				case 0x21: IMMWORD(t); break;
				case 0x22: LBRANCH(!(cc&0x05)); break;
				case 0x23: LBRANCH(cc&0x05); break;
				case 0x24: LBRANCH(!(cc&0x01)); break;
				case 0x25: LBRANCH(cc&0x01); break;
				case 0x26: LBRANCH(!(cc&0x04)); break;
				case 0x27: LBRANCH(cc&0x04); break;
				case 0x28: LBRANCH(!(cc&0x02)); break;
				case 0x29: LBRANCH(cc&0x02); break;
				case 0x2a: LBRANCH(!(cc&0x08)); break;
				case 0x2b: LBRANCH(cc&0x08); break;
				case 0x2c: LBRANCH(!NXORV); break;
				case 0x2d: LBRANCH(NXORV); break;
				case 0x2e: LBRANCH(!(NXORV||cc&0x04)); break;
				case 0x2f: LBRANCH(NXORV||cc&0x04); break;
				case 0x3f: cc|=0x80; PUSHWORD(pcreg); PUSHWORD(ureg); PUSHWORD(yreg); PUSHWORD(xreg); PUSHBYTE(dpreg); PUSHBYTE(breg); PUSHBYTE(areg); PUSHBYTE(cc); pcreg = M_RDMEM_WORD(0xfff4);change_pc(pcreg); break;
				case 0x83: IMMWORD(b); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0x8c: IMMWORD(b); d = yreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0x8e: IMMWORD(yreg); CLR_NZV; SET_NZ16(yreg); break;
				case 0x8f: CLR_NZV; SET_NZ16(yreg); IMM16; M_WRMEM_WORD(eaddr,yreg); break; /* ILLEGAL? */
				case 0x93: DIRWORD(b); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0x9c: DIRWORD(b); d = yreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0x9e: DIRWORD(yreg); CLR_NZV; SET_NZ16(yreg); break;
				case 0x9f: CLR_NZV; SET_NZ16(yreg); DIRECT; M_WRMEM_WORD(eaddr,yreg); break;
				case 0xa3: b = M_RDMEM_WORD(eaddr); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0xac: b = M_RDMEM_WORD(eaddr); d = yreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0xae: yreg = M_RDMEM_WORD(eaddr); CLR_NZV; SET_NZ16(yreg); break;
				case 0xaf: CLR_NZV; SET_NZ16(yreg); M_WRMEM_WORD(eaddr,yreg); break;
				case 0xb3: EXTWORD(b); d = GETDREG; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0xbc: EXTWORD(b); d = yreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0xbe: EXTWORD(yreg); CLR_NZV; SET_NZ16(yreg); break;
				case 0xbf: CLR_NZV; SET_NZ16(yreg); EXTENDED; M_WRMEM_WORD(eaddr,yreg); break;
				case 0xce: IMMWORD(sreg); CLR_NZV; SET_NZ16(sreg); break;
				case 0xcf: CLR_NZV; SET_NZ16(sreg); IMM16; M_WRMEM_WORD(eaddr,sreg); break; /* ILLEGAL? */
				case 0xde: DIRWORD(sreg); CLR_NZV; SET_NZ16(sreg); break;
				case 0xdf: CLR_NZV; SET_NZ16(sreg); DIRECT; M_WRMEM_WORD(eaddr,sreg); break;
				case 0xee: sreg = M_RDMEM_WORD(eaddr); CLR_NZV; SET_NZ16(sreg); break;
				case 0xef: CLR_NZV; SET_NZ16(sreg); M_WRMEM_WORD(eaddr,sreg); break;
				case 0xfe: EXTWORD(sreg); CLR_NZV; SET_NZ16(sreg); break;
				case 0xff: CLR_NZV; SET_NZ16(sreg); EXTENDED; M_WRMEM_WORD(eaddr,sreg); break;
				default: break;
				}
			}else{
				switch( ireg2 )	/* 11xx */
				{
				case 0x3f: cc|=0x80; PUSHWORD(pcreg); PUSHWORD(ureg); PUSHWORD(yreg); PUSHWORD(xreg); PUSHBYTE(dpreg); PUSHBYTE(breg); PUSHBYTE(areg); PUSHBYTE(cc); pcreg = M_RDMEM_WORD(0xfff2);change_pc(pcreg); break;
				case 0x83: IMMWORD(b); r = ureg-b; CLR_NZVC; SET_FLAGS16(ureg,b,r); break;
				case 0x8c: IMMWORD(b); d = sreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0x93: DIRWORD(b); r = ureg-b; CLR_NZVC; SET_FLAGS16(ureg,b,r); break;
				case 0x9c: DIRWORD(b); d = sreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0xa3: b = M_RDMEM_WORD(eaddr); r = ureg-b; CLR_NZVC; SET_FLAGS16(ureg,b,r); break;
				case 0xac: b = M_RDMEM_WORD(eaddr); d = sreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				case 0xb3: EXTWORD(b); r = ureg-b; CLR_NZVC; SET_FLAGS16(ureg,b,r); break;
				case 0xbc: EXTWORD(b); d = sreg; r = d-b; CLR_NZVC; SET_FLAGS16(d,b,r); break;
				default: break;
				}
			}
		}
		m6809_ICount -= op_count & 0x7f;

	} while( m6809_ICount>0 );

	return cycles - m6809_ICount;	/* NS 970908 */
}
