/*

PROJECTION MATRIX
=================

Matrices in GC are mirrored. Here are matrices described that are GL compatible.
Matrices in GL convert to linear by columns. 

XF_1020	PROJ_MTX_0		P0
XF_1021	PROJ_MTX_1		P1
XF_1022	PROJ_MTX_2		P2
XF_1023	PROJ_MTX_3		P3
XF_1024	PROJ_MTX_4		P4
XF_1025	PROJ_MTX_5		P5
XF_1026	PROJ_MTX_MODE	0-PERSPECTVE 1-ORTHOGRAPHIC

PERSPECTIVE PROJECTION (0)

P0  0 P1  0
 0 P2 P3  0
 0  0 P4 P5
 0  0 -1  0

ORTHOGRPHIC PROJECTION (1)

P0  0  0 P1
 0 P2  0 P3
 0  0 P4 P5
 0  0  0  1

*** IMPORTANT ***

Matrix Frustum on GC works different way than on PC. Calculation returns different values for P4 and P5.
It needs to be verified if conversion is required to PC values.

------------------------------------------------------------------------------------------------------------

MODELVIEW MATRIX
================

Matrices are stored at the beginning of XF Regs
Matrix is in row order 4x3 without last row which should be 0,0,0,1 for OpenGL


*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>


#include "gp_defs.h"
#include "xf_regs.h"
#include "graphics_processor/vertex_processor.h"
#include "debug/tracers.h"

#include "d3d_fun.h"

#include "byteswap.h"

gp_reg	gp_xf_regs[0x3000];

float	xf_modelview_matrix[16];
float	xf_normal_matrix[16];
uint32	xf_modelview_matrix_addr = 0;
uint32	xf_normal_matrix_addr = 0x400;
float	xf_projection_matrix[16];
float	xf_texture_matrix[8][16];
uint32	xf_texture_matrix_addr[8];

float xf_one[4] = 
{ 1.0f, 1.0f, 1.0f, 1.0f};
float xf_zero[4] = 
{ 0.0f, 0.0f, 0.0f, 0.0f};

uint32 xf_mat_col0;

#include <d3dx9math.h>
#pragma comment(lib, "d3dx9.lib")


void xf_set_projection_matrix(uint32 data)
{
	if (data == 0)
	{
/*		float     left    = 24.0F;
		float     top     = 32.0F;
		float     znear   = 50.0F;
		float     zfar    = 2000.0F;
		// PERSPECTIVE MATRIX
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glFrustum(-top, top, -left, left, znear, zfar);
		printf("GP: Loaded projection matrix\n");
*/		
		int i;

		for(i = 0 ; i < 16 ; i++)
			xf_projection_matrix[i] = 0.0f;

		xf_projection_matrix[0] = gp_xf_regs[0x1020].f;
		xf_projection_matrix[8] = gp_xf_regs[0x1021].f;
		xf_projection_matrix[5] = gp_xf_regs[0x1022].f;
		xf_projection_matrix[9] = gp_xf_regs[0x1023].f;
		xf_projection_matrix[10] = gp_xf_regs[0x1024].f - 1.0f;
		xf_projection_matrix[14] = gp_xf_regs[0x1025].f;
		xf_projection_matrix[11] = -1.0f;

/*
		float w1 = xf_projection_matrix[10];
		float w2 = xf_projection_matrix[14];
		float org_far = w2/w1;
		float org_near = w1 * org_far / (w1 - 1.0f);

		w1 = xf_projection_matrix[0];
		w2 = xf_projection_matrix[8];
		float org_right = ((2 * org_near / w1)*(2 * org_near / w1) - w2) / ( 2 * 2 * org_near / w1);
		float org_left  = org_right - 2*org_near/w1;
		
		w1 = xf_projection_matrix[5];
		w2 = xf_projection_matrix[9];
		float org_top = ((2 * org_near / w1)*(2 * org_near / w1) - w2) / ( 2 * 2 * org_near / w1);
		float org_bottom  = org_top - 2*org_near/w1;

		printf("FAR: %f %f %f\n", w2, w1, w2/w1);
		printf("NEAR: %f\n", org_near);
		printf("RIGHT: %f\n", org_right);
		printf("LEFT: %f\n", org_left);
		printf("TOP: %f\n", org_top);
		printf("BOTTOM: %f\n", org_bottom);

		printf("GP: Projection Matrix\n");

		// TODO: Implement D3D loading of Frustum
		D3DXMATRIX pOut;
		//D3DXMatrixPerspectiveOffCenterLH(&pOut, org_left, org_right, org_bottom, org_top, org_near, org_far);
		D3DXMatrixPerspectiveFovRH(&pOut, 60, 1.33F, 10.0F, 300.0F);

		
		{
			int i;
			syslog(DXX, "PROJ\n");
			for (i = 0 ; i < 16 ; i++)
				syslog(DXX, "%d: %f %f\n", i, pOut[i], xf_projection_matrix[i]);

			for (i = 0 ; i < 16 ; i++)	
			{
				syslog(DXX, "[%d,%d]: %f\n", i/4, i%4, pOut(i/4, i%4));
			}
			for (i = 0 ; i < 16 ; i++)	
			{
				pOut[i] = xf_projection_matrix[i];
			}
		}
		g_pd3dDevice->SetTransform(D3DTS_PROJECTION, &pOut);
*/
		g_pd3dDevice->SetTransform(D3DTS_PROJECTION, (D3DMATRIX *)xf_projection_matrix);
		
	}
	else
	{
		// 1 - ORTHOGRAPHIC MATRIX
		int i;

		for(i = 0 ; i < 16 ; i++)
			xf_projection_matrix[i] = 0.0f;
		
		xf_projection_matrix[0] = gp_xf_regs[0x1020].f;
		xf_projection_matrix[12] = gp_xf_regs[0x1021].f;
		xf_projection_matrix[5] = gp_xf_regs[0x1022].f;
		xf_projection_matrix[13] = gp_xf_regs[0x1023].f;
		xf_projection_matrix[10] = gp_xf_regs[0x1024].f;
		xf_projection_matrix[14] = gp_xf_regs[0x1025].f;
		xf_projection_matrix[15] = -1.0f;
		// conversion should be here for [14]

		//g_pd3dDevice->SetTransform(D3DTS_PROJECTION, (D3DMATRIX *)xf_projection_matrix);

		float w1 = xf_projection_matrix[10];
		float w2 = xf_projection_matrix[14];

		float org_far = (w2 - 1.0f)/(2*w1);
		float org_near = org_far + 1.0f/w1;

		w1 = xf_projection_matrix[0];
		w2 = xf_projection_matrix[12];

		float org_right = (-w2 + 1.0f)/w1;
		float org_left = org_right - 2.0f/w1;

		w1 = xf_projection_matrix[5];
		w2 = xf_projection_matrix[13];

		float org_top = (-w2 + 1.0f) / w1;
		float org_bottom = org_top - 2.0f/w1;
/*
		printf("FAR: %f %f %f\n", w2, w1, w2/w1);
		printf("NEAR: %f\n", org_near);
		printf("RIGHT: %f\n", org_right);
		printf("LEFT: %f\n", org_left);
		printf("TOP: %f\n", org_top);
		printf("BOTTOM: %f\n", org_bottom);
*/
		// TODO: Implement D3D loading of Ortho
		/*
		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		glOrtho(org_left, org_right, org_bottom, org_top, org_near, org_far);
		*/
		D3DXMATRIX pOut;
		D3DXMatrixOrthoOffCenterRH(&pOut, org_left, org_right, org_bottom, org_top, org_near, org_far);
		g_pd3dDevice->SetTransform(D3DTS_PROJECTION, &pOut);
	}
}

void xf_set_modelview_matrix(uint32 addr)
{
	int i;
	uint32 crc = 0;
	static uint32 oldcrc = 0;

	for(i = 0 ; i < 12 ; i++)
	{
		crc += gp_xf_regs[addr + i].i;
	}

	if (crc == oldcrc)
		return;

	oldcrc = crc;
	
	xf_modelview_matrix[0] = gp_xf_regs[addr + 0].f;
	xf_modelview_matrix[1] = gp_xf_regs[addr + 4].f;
	xf_modelview_matrix[2] = gp_xf_regs[addr + 8].f;
	xf_modelview_matrix[3] = 0.0f;
	xf_modelview_matrix[4] = gp_xf_regs[addr + 1].f;
	xf_modelview_matrix[5] = gp_xf_regs[addr + 5].f;
	xf_modelview_matrix[6] = gp_xf_regs[addr + 9].f;
	xf_modelview_matrix[7] = 0.0f;
	xf_modelview_matrix[8] = gp_xf_regs[addr + 2].f;
	xf_modelview_matrix[9] = gp_xf_regs[addr + 6].f;
	xf_modelview_matrix[10] = gp_xf_regs[addr + 10].f;
	xf_modelview_matrix[11] = 0.0f;
	xf_modelview_matrix[12] = gp_xf_regs[addr + 3].f;
	xf_modelview_matrix[13] = gp_xf_regs[addr + 7].f;
	xf_modelview_matrix[14] = gp_xf_regs[addr + 11].f;
	xf_modelview_matrix[15] = 1.0f;

	// TODO: Loading of modelview matrix
	g_pd3dDevice->SetTransform(D3DTS_WORLD, (D3DMATRIX *)xf_modelview_matrix);
	
	/*syslog(DXX, "MVX\n");
	{
		int i;
		for (i = 0 ; i < 16 ; i++)
			syslog(DXX, "%d:%f\n", i, xf_modelview_matrix[i]);
	}
	*/
}

void xf_set_normal_matrix(uint32 addr)
{
	// OpenGL does not let setting Normal Matrix
	// it is calculated as Transpose Inverse of Modelview matrix internally in OpenGL
	
	// TODO: Loading of Normal Matrix
	// cgGLSetMatrixParameterfr(vs_normal_mtx, &gp_xf_regs[addr].f);
}

void xf_set_texture_matrix(uint32 id)
{
#if WITH_VERTEX_SHADER
#if WITH_CG
	uint32 addr;
	addr = xf_texture_matrix_addr[id];
	cgGLSetMatrixParameterfr(vs_texture_mtx[id], &gp_xf_regs[addr].f);
//	cgGLSetMatrixParameterfc(vs_texture_mtx[id], xf_texture_matrix[id]);
#else	// WITH_CG
#error Currently Vertex Shader allowed only through CG
#endif	// WITH CG
#else
	int i;
	uint32 addr;
	addr = xf_texture_matrix_addr[id];

	for(i = 0 ; i < 16 ; i++)
		xf_texture_matrix[id][i] = 0.0f;

	xf_texture_matrix[id][0] = gp_xf_regs[addr + 0].f;
	xf_texture_matrix[id][1] = gp_xf_regs[addr + 4].f;
	xf_texture_matrix[id][2] = gp_xf_regs[addr + 8].f;
	xf_texture_matrix[id][3] = 0.0f;

	xf_texture_matrix[id][4] = gp_xf_regs[addr + 1].f;
	xf_texture_matrix[id][5] = gp_xf_regs[addr + 5].f;
	xf_texture_matrix[id][6] = gp_xf_regs[addr + 9].f;
	xf_texture_matrix[id][7] = 0.0f;

	xf_texture_matrix[id][8] = gp_xf_regs[addr + 2].f;
	xf_texture_matrix[id][9] = gp_xf_regs[addr + 6].f;
	xf_texture_matrix[id][10] = gp_xf_regs[addr + 10].f;
	xf_texture_matrix[id][11] = 0.0f;

	xf_texture_matrix[id][12] = gp_xf_regs[addr + 3].f;
	xf_texture_matrix[id][13] = gp_xf_regs[addr + 7].f;
	xf_texture_matrix[id][14] = gp_xf_regs[addr + 11].f;
	xf_texture_matrix[id][15] = 1.0f;

//	syslog(XF,"Load texture matrix %x\n", addr);
	// TODO: Loading of Texture Matrix
	uint32 tex_id = id + D3DTS_TEXTURE0;
	g_pd3dDevice->SetTransform((D3DTRANSFORMSTATETYPE)tex_id, (D3DMATRIX *)xf_texture_matrix[id]);

	/*
	glMatrixMode(GL_TEXTURE);
	glLoadMatrixf(xf_texture_matrix[id]);
	glMatrixMode(GL_MODELVIEW);
	*/
#endif // WITH_VERTEX_SHADER
}


typedef struct
{
	vx_vertex_t pos[4];
	vx_vertex_t dir[4];
	float		col[4];
} xf_light_t;

float	mat_col[4] = { 1.5, 0.5, 0.5, 1.0 };
float	col0_amb[4];

xf_light_t	xf_light[8];

D3DMATERIAL9 mat;

D3DLIGHT9 g_light[8];

void xf_update_chanctrl(uint32 chan)
{
	uint32 data = gp_xf_regs[0x100e].i;

	g_pd3dDevice->SetMaterial(&mat);

	switch((data & 0x1) | ((data >> 5) & 0x2))
	{
	case 0x3:	// AMB == 1 && DIF == 1
		syslog(XF,"COL0CTRL Enable Material Color for Ambient and Diffuse\n");
		// TODO: Material Colour
		/*
		glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
		glEnable(GL_COLOR_MATERIAL);
		*/
		break;
	case 0x2:	// AMB == 1 && DIF == 0
		syslog(XF,"COL0CTRL Enable Material Color for Ambient\n");
		// diffuse comes from COL0MAT register XF_100c
		// TODO: Material Colour
		/*
		glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_col);
		glColorMaterial(GL_FRONT, GL_AMBIENT);
		glEnable(GL_COLOR_MATERIAL);
		glColor4fv(col0_amb);
		*/
		break;
	case 0x1:	// AMB == 0 && DIF == 1
		syslog(XF,"COL0CTRL Enable Material Color for Diffuse\n");
		// ambient comes from COL0AMB register XF_100a
		// TODO: Material Colour
		/*
		glMaterialfv(GL_FRONT, GL_AMBIENT, col0_amb);
		glColorMaterial(GL_FRONT, GL_DIFFUSE);
		glEnable(GL_COLOR_MATERIAL);
		glColor4fv(mat_col);
		*/
		break;
	case 0x0:	// AMB == 0 && DIF == 0
		syslog(XF,"COL0CTRL Disable Material Color\n");
		// both colours come from their registers
		// TODO: Material Colour
		g_pd3dDevice->SetRenderState(D3DRS_DIFFUSEMATERIALSOURCE, D3DMCS_MATERIAL);
		g_pd3dDevice->SetRenderState(D3DRS_AMBIENTMATERIALSOURCE, D3DMCS_MATERIAL);
		break;
	}


	//D3DVECTOR  Vdir={-1,-1,-1};
	//memset(&light,0,sizeof(D3DLIGHT9));
	//g_light[0].Type = D3DLIGHT_POINT;
	//g_light[0].Direction = Vdir;

	//g_pd3dDevice->SetLight(0, &g_light[0]);


	// TODO: Enable Lighting
	syslog(XF,"COL0CTRL Lighting %s\n", (data & 0x2)?"enable":"disable");
	g_pd3dDevice->SetRenderState(D3DRS_LIGHTING, (data & 0x2)?TRUE:FALSE);

	// TODO: Light Enable
	int i;
	for(i = 0 ; i < 8 ; i++)
	{
		uint8 bit;
		bit = (i < 4)?(i+2):(i+7);
		syslog(XF,"COL0CTRL Light 0 %s\n", (data & (1 << bit))?"enable":"disable");
		g_pd3dDevice->LightEnable(i, (data & (1 << bit))?TRUE:FALSE);
	}
}





void gp_xf_write_reg(uint32 reg, uint32 data)
{
	gp_xf_regs[reg].i = data;

	syslog(XF,"GP: XF_%04x %08x %f\n", reg, data, gp_xf_regs[reg].f);
	// for now we want switch statement but later we need special callbacks next to registers

	if (reg >= xf_texture_matrix_addr[0] && reg <= (xf_texture_matrix_addr[0] + 8))
	{
		xf_set_texture_matrix(0);
	}
	if (reg >= xf_texture_matrix_addr[1] && reg <= (xf_texture_matrix_addr[1] + 8))
	{
		xf_set_texture_matrix(1);
	}
	if (reg >= xf_texture_matrix_addr[2] && reg <= (xf_texture_matrix_addr[2] + 8))
	{
		xf_set_texture_matrix(2);
	}
	if (reg >= xf_texture_matrix_addr[3] && reg <= (xf_texture_matrix_addr[3] + 8))
	{
		xf_set_texture_matrix(3);
	}
	if (reg >= xf_texture_matrix_addr[4] && reg <= (xf_texture_matrix_addr[4] + 8))
	{
		xf_set_texture_matrix(4);
	}
	if (reg >= xf_texture_matrix_addr[5] && reg <= (xf_texture_matrix_addr[5] + 8))
	{
		xf_set_texture_matrix(5);
	}
	if (reg >= xf_texture_matrix_addr[6] && reg <= (xf_texture_matrix_addr[6] + 8))
	{
		xf_set_texture_matrix(6);
	}
	if (reg >= xf_texture_matrix_addr[7] && reg <= (xf_texture_matrix_addr[7] + 8))
	{
		xf_set_texture_matrix(7);
	}

	if (reg >= xf_modelview_matrix_addr && reg <= (xf_modelview_matrix_addr + 11))
	{
		xf_set_modelview_matrix(xf_modelview_matrix_addr);
	}

	if (reg >= xf_normal_matrix_addr && reg <= (xf_normal_matrix_addr + 11))
	{
		xf_set_normal_matrix(xf_normal_matrix_addr);
	}

	if(reg >= 0x600 && reg < 0x680)
	{
		// light number
		uint32 id = (reg >> 4) & 0xf;
		reg &= 0xf;
		switch(reg)
		{
		case 0x3:
			// $LIGHT$ COLOUR
			g_light[id].Diffuse.b = ((data >> 24) & 0xff)/256.0f;
			g_light[id].Diffuse.g = ((data >> 16) & 0xff)/256.0f;
			g_light[id].Diffuse.r = ((data >> 8) & 0xff)/256.0f;
			g_light[id].Diffuse.a = ((data >> 0) & 0xff)/256.0f;
			// TODO: Light Colour
			g_pd3dDevice->SetLight(id, &g_light[id]);
			break;
		case 0xa:
		case 0xb:
		case 0xc:
			// $LIGHT$ POSITION
			xf_light[id].pos[reg - 0xa].i = data;
			xf_light[id].pos[3].f = 1.0f;
		
			// TODO: Light Position
			g_light[id].Position.x = xf_light[id].pos[0].f;
			g_light[id].Position.y = xf_light[id].pos[1].f;
			g_light[id].Position.z = xf_light[id].pos[2].f;
			g_light[id].Type = D3DLIGHT_POINT;
			g_light[id].Range = 1000.0f;
			g_pd3dDevice->SetLight(id, &g_light[id]);
			syslog(XF,"glLightfv(GL_LIGHT%d, GL_POSITION, %f, %f, %f, %f)\n", id, xf_light[id].pos[0].f, xf_light[id].pos[1].f, xf_light[id].pos[2].f, xf_light[id].pos[3].f);

			break;
		case 0xd:
			
		case 0xe:
		case 0xf:
			// $LIGHT$ DIRECTION
			xf_light[id].dir[reg - 0xd].i = data;
			xf_light[id].dir[3].f = 1.0f;

			g_light[id].Direction.x = xf_light[id].dir[0].f;
			g_light[id].Direction.y = xf_light[id].dir[1].f;
			g_light[id].Direction.z = xf_light[id].dir[2].f;
			//g_light[id].Type = D3DLIGHT_DIRECTIONAL;
			g_pd3dDevice->SetLight(id, &g_light[id]);

			//g_light[id].Type = D3DLIGHT_DIRECTIONAL;
			//g_light[id].Range = 1000.0f;
			//g_pd3dDevice->SetLight(id, &g_light[id]);
			// TODO: Light 
			/*
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			//glLightfv(GL_LIGHT0 + id, GL_SPOT_DIRECTION, (GLfloat *)xf_light[id].dir);
			//glLightf(GL_LIGHT0 + id, GL_SPOT_CUTOFF, 100.0f);
			glPopMatrix();
			*/
			break;
		default:
			syslog(XF,"LIGHT: Unknown register\n");
			break;
		}
		return;
	}

	switch(reg)
	{
	case 0x100a:
		// ambient colour 1
		mat.Ambient.r = ((data >> 24) & 0xff)/256.0f;
		mat.Ambient.g = ((data >> 16) & 0xff)/256.0f;
		mat.Ambient.b = ((data >> 8) & 0xff)/256.0f;
		mat.Ambient.a = ((data >> 0) & 0xff)/256.0f;
		xf_update_chanctrl(0);
		break;
//	case 0x100b:
		// ambient colour 2
//		break;
	case 0x100c:
		// material colour 1
		xf_mat_col0 = (data << 24) | (data >> 8);
//		printf("MatClr %08x\n", data);
		mat.Diffuse.r = ((data >> 24) & 0xff)/256.0f;
		mat.Diffuse.g = ((data >> 16) & 0xff)/256.0f;
		mat.Diffuse.b = ((data >> 8) & 0xff)/256.0f;
		mat.Diffuse.a = ((data >> 0) & 0xff)/256.0f;
		xf_update_chanctrl(0);
		break;
//	case 0x100d:
		// material colour 2
//		break;
	case 0x100e:
		// colour 1 control
		xf_update_chanctrl(0);
		break;
	case 0x1018:
		syslog(XF,"SetMatrix: MV %x T0: %x T1: %x T2: %x T3: %x\n", data & 0x3f, (data >> 6) & 0x3f, (data >> 12) & 0x3f, (data >> 18) & 0x3f, (data >> 24) & 0x3f);
		xf_modelview_matrix_addr = (data & 0x3f) << 2;
		xf_set_modelview_matrix(xf_modelview_matrix_addr);

		xf_texture_matrix_addr[0] = (((data >> 6) & 0x3f) << 2);
		xf_texture_matrix_addr[1] = (((data >> 12) & 0x3f) << 2);
		xf_texture_matrix_addr[2] = (((data >> 18) & 0x3f) << 2);
		xf_texture_matrix_addr[3] = (((data >> 24) & 0x3f) << 2);
		xf_set_texture_matrix(0);
		xf_set_texture_matrix(1);
		xf_set_texture_matrix(2);
		xf_set_texture_matrix(3);
		break;

	case 0x1019:
		xf_texture_matrix_addr[4] = (((data >> 0) & 0x3f) << 2);
		xf_texture_matrix_addr[5] = (((data >> 6) & 0x3f) << 2);
		xf_texture_matrix_addr[6] = (((data >> 12) & 0x3f) << 2);
		xf_texture_matrix_addr[7] = (((data >> 18) & 0x3f) << 2);
		xf_set_texture_matrix(4);
		xf_set_texture_matrix(5);
		xf_set_texture_matrix(6);
		xf_set_texture_matrix(7);
		break;

	case 0x1026:
		// write to PROJ_MTX_MODE
		xf_set_projection_matrix(data);
		break;

	case 0x101f:
		{
		float x, y, w, h;
		float zfar, znear;
		w = gp_xf_regs[0x101a].f;
		h = -gp_xf_regs[0x101b].f;

		x = gp_xf_regs[0x101d].f;
		y = gp_xf_regs[0x101e].f;

		x = x - 342 - w;
		y = y - 342 - h;
		w *= 2.0f;
		h = h * 2.0f;

//		y = 480 - y;
//		h = -h;
		//y = 480 - (y + h);
		//syslog(XF, "Viewport org: %f, %f, %f, %f\n", gp_xf_regs[0x101a].f, gp_xf_regs[0x101b].f, gp_xf_regs[0x101d].f, gp_xf_regs[0x101e].f);
		// TODO: Viewport

		// glViewport((GLint)x, (GLint)y, (GLsizei)w, (GLsizei)h);
		syslog(XF,"glViewport(%d, %d, %d, %d)\n", (int)x, (int)y, (int)w, (int)h);
		
		zfar = gp_xf_regs[0x101f].f;
		znear = gp_xf_regs[0x101c].f - zfar;
		zfar /= 16777215.0f;
		znear /= 16777215.0f;
		// TODO: Depth
		//glDepthRange(znear, zfar);
		syslog(XF,"Depth Range %f %f\n", znear, zfar);
		D3DVIEWPORT9 vp;
		vp.X = x;
		vp.Y = y;
		vp.Width = w;
		vp.Height = h;
		vp.MaxZ = zfar;
		vp.MinZ = znear;
		g_pd3dDevice->SetViewport(&vp);
		}
		break;

	case 0x1008:
		if (data & 0xf0)
		{
//			glEnable(GL_TEXTURE_2D);
		}
		else
		{
//			glDisable(GL_TEXTURE_2D);
		}
		break;

	case 0x103f:
		// texgen enabled
		// bit 0 enables texgen matrix
/*		if (data)
		{
			glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
			glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
			glEnable(GL_TEXTURE_GEN_S);
			glEnable(GL_TEXTURE_GEN_T);
		}
		else
		{
			glDisable(GL_TEXTURE_GEN_S);
			glDisable(GL_TEXTURE_GEN_T);
		}
*/		break;

	default:
		break;
	}
}


