/*====================================================================

filename:     video_interface_defines.h
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/

/*
	VTR - Vertical Timing Register (W)
	This register setups the vertical timing. The value ACV is double buffered 
    3:0		EQU		Equalization pulse in half lines
    13:4	ACV		Active video in full lines
*/

#define	VI_REG_VTR		0x00

/*
	DCR - Display Configuration Register (R/W) 
	This register set ups and enables VI. Generally, VI should be reset before enabling it. This resets the states into some known values

	0		ENB		W: This bit enables the video timing generation and data request
					R: 0
    1		RST		W: This bit clears all data request and puts VI into its idle state
					R: 0
    2		NIN		W: To select interlace or non-interlace mode. NIN = 0: interlace, 
					NIN = 1: non-interlace. In non-interlace mode, the top field is 
					drawn at field rate while the bottom field is not displayed.
	3		DLR		This bit selects the 3D display mode.
    5:4		LE0		Gun trigger mode. It enables the Display Latch Register 0. When
     the   0
                mode is 1 or 2, it will clear itself(off) automatically when a
     gun trigger
                is detected or at time out. This field is double buffered.
                0 off
                1 on for 1 field
                2 on for 2 fields
                3 always on
    LE1   7:6   To enable Display Latch Register 1. See the description of LE0.
       0
    FMT   9:8   Indicates current video format:                            0
                0 NTSC
                1 PAL
                2 MPAL
                3 Debug (CCIR656)
*/
#define VI_REG_DCR	0x02

/*
	DIR0 - Display Interrupt Register 0 (R/W) 
	DIR1 - Display Interrupt Register 1 (R/W) 
	DIR2 - Display Interrupt Register 2 (R/W) 
	DIR3 - Display Interrupt Register 3 (R/W) 
	There are a total of four display interrupt registers (0-3). 
	They are used to generate interrupts to the main processor at different 
	positions within a field. Each register has a separate enable bit. 
	The interrupt is cleared by writing a zero to the status flag (INT). 

    HCT     10:0  Horizontal count to generate interrupt.   --
    VCT     26:16 Vertical count to generate interrupt.     --
    ENB      28   Interrupt is enabled if this bit is set.   0
    INT      31   Interrupt status. A "1" indicates that an interrupt is  0
                  active.

*/
#define VI_REG_DIR0	0x30
#define VI_REG_DIR1	0x34
#define VI_REG_DIR2	0x38
#define VI_REG_DIR3	0x3c

#define VI_REG_DIR0_VCT	0x30
#define VI_REG_DIR1_VCT	0x34
#define VI_REG_DIR2_VCT	0x38
#define VI_REG_DIR3_VCT	0x3c

/*
	DPR - Display Position Register (R) 

This register contains the current raster position. 

The Horizontal Count is in pixels and runs from 1 to # pixels per line. It is reset to 1 at the beginning of every line. 

The Vertical Count is in lines (on a frame basis) and runs from 1 to # lines per frame. It is 1 at the beginning of pre-equalization. This is a frame line count. So for example: for NTSC vcount=264 is the first (full) line in the second field and vcount=525 is the last line in the frame (fields being numbered 1-4). For non-interlaced modes vcount is on a field-by-field basis (for NTSC vcount ranges from 1-263). 

This counting scheme applies the Display Position, Display Interrupt, and Display Latch registers. 
          HCT         10:0        Horizontal count.       --
          VCT         26:16       Vertical count.       --
*/

#define	VI_REG_DPR	0x2c
#define	VI_REG_VCT	0x2c
#define	VI_REG_HCT	0x2e


#define VI_REG_UINT32(A)	*((uint32 *)(vi_regs + (A)))
#define VI_REG_UINT16(A)	*((uint16 *)(vi_regs + (A)))
#define VI_REG_UINT8(A)		*((uint8 *)(vi_regs + (A)))

#define VI_WRITE_UINT32(A, D)	VI_REG_UINT32(A) = byteswap32((D))
#define VI_WRITE_UINT16(A, D)	VI_REG_UINT16(A) = byteswap16((D))
#define VI_WRITE_UINT8(A, D)	VI_REG_UINT8(A) = (uint8)(D)

#define VI_READ_UINT32(A)		(byteswap32(VI_REG_UINT32(A)))
#define VI_READ_UINT16(A)		(byteswap16(VI_REG_UINT16(A)))
#define VI_READ_UINT8(A)		(VI_REG_UINT8(A))

//----------------------------------------------------------------------//

void vi_init(void);
void vi_close(void);
void vi_write_register8(uint32 addr, uint8 data);
void vi_write_register16(uint32 addr, uint16 data);
void vi_write_register32(uint32 addr, uint32 data);
uint8 vi_read_register8(uint32 addr);
uint16 vi_read_register16(uint32 addr);
uint32 vi_read_register32(uint32 addr);
bool vi_check_interrupt(void);
void vi_next_scanline(void);
