/*====================================================================

filename:     serial_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
/*
	US Patent 6,609,977

	Serial Interface

	- inputs and outputs are double buffered (not implemented, not sure if it is necessary)
	- communication RAM is 128 bytes
	- RDST (bit indicating that the input data been copied from first input buffer to second) remains
	  on until reading INBUFL register by CPU

*/
#include <stdio.h>
#include "system/types.h"
#include "serial_interface.h"
#include "hardware/hw_io.h"
#include "hardware/memory_interface.h"
#include "debug/tracers.h"
#include "processor_interface.h"
#include "serial_interface_defines.h"
#include "pad.h"

struct pad_data pad[4];

char si_regs[SI_REGS_SIZE];

void si_callback(void);

void si_init(void)
{
	int i;
	for (i = 0 ; i < SI_REGS_SIZE ; i++)
	{
		si_regs[i] = 0;
	}
}

void si_close(void)
{
}

void si_update_rdstint(void)
{
	uint32 data, sr;
	sr = SI_READ_UINT32(SI_REG_SR);
	data = SI_READ_UINT32(SI_REG_COMCSR);

	if (sr & (SI_RDST0 | SI_RDST1 | SI_RDST2 | SI_RDST3))
	{
		data |= SI_RDSTINT;
	}
	else
	{
		data &= ~SI_RDSTINT;
	}
	SI_WRITE_UINT32(SI_REG_COMCSR, data);
}

void si_clear_rdst(uint32 chan)
{
	uint32 sr;
	sr = SI_READ_UINT32(SI_REG_SR);
	switch(chan)
	{
	case 0:
		sr &= ~SI_RDST0;
		break;
	case 1:
		sr &= ~SI_RDST1;
		break;
	case 2:
		sr &= ~SI_RDST2;
		break;
	case 3:
		sr &= ~SI_RDST3;
		break;
	default:
		break;
	}
	SI_WRITE_UINT32(SI_REG_SR, sr);
	si_update_rdstint();
}

void si_command(uint32 ch)
{
	uint8 cmd = SI_READ_UINT8(SI_REG_BUF);
	uint32 comcsr;

	syslog(SI,"ch: %d\n", ch);
	switch(cmd)
	{
	// command reset
	case 0x00:
		SI_WRITE_UINT32(SI_REG_BUF, 0x09000000);
		SI_WRITE_UINT32(SI_REG_SR, 0x20200000);
		break;
	// command origin
	case 0x41:
//		SI_WRITE_UINT8(SI_REG_BUF, 0x41);
//		SI_WRITE_UINT8(SI_REG_BUF+1, 0x80);
//		SI_WRITE_UINT8(SI_REG_BUF+2, 0x80);
//		SI_WRITE_UINT8(SI_REG_BUF+3, 0x80);
//		SI_WRITE_UINT8(SI_REG_BUF+4, 0x80);
//		SI_WRITE_UINT8(SI_REG_BUF+5, 0x1f);
//		SI_WRITE_UINT8(SI_REG_BUF+6, 0x1f);
		SI_WRITE_UINT32(SI_REG_BUF, 0x41008080);
		SI_WRITE_UINT32(SI_REG_BUF + 4, 0x80801f1f);
		SI_WRITE_UINT32(SI_REG_BUF + 8, 0x1f1f0000);
		SI_WRITE_UINT32(SI_REG_SR, 0x20200000);
		break;
	}

	comcsr = SI_READ_UINT32(SI_REG_COMCSR);
	comcsr |= SI_TCINT | SI_RDSTINT;
	comcsr &= ~SI_TSTART;
	SI_WRITE_UINT32(SI_REG_COMCSR, comcsr);

	SI_WRITE_UINT32(SI_REG_SR, SI_RDST0);// | SI_RDST1 | SI_RDST2 | SI_RDST3);
	SI_WRITE_UINT32(SI_REG_SR, SI_RDST0 | SI_RDST1 | SI_RDST2 | SI_RDST3);
	si_update_rdstint();
}

void si_write_register32(uint32 addr, uint32 data)
{
	uint32 csr;

	switch(addr & 0xff)
	{
	case SI_REG_COMCSR:
		csr = SI_READ_UINT32(SI_REG_COMCSR);
		if (data & SI_TCINT)
		{
			csr &= ~SI_TCINT;
		}
		if (data & SI_RDSTINT)
		{
			csr &= ~SI_RDSTINT;
		}
		data = (data & (~(SI_TCINT | SI_RDSTINT))) | (csr & (SI_TCINT | SI_RDSTINT));
		break;
	case SI_REG_SR:
		return;
	default:
		break;
	}

	SI_WRITE_UINT32(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case SI_REG_POLL:
		syslog(SI,"Polling Setup: X: %d Y: %d EN: %x CP: %x\n", (data >> 16) & 0x3ff, (data >> 8) & 0xff, (data >> 4) & 0xf, data & 0xf);
		//pi_schedule_irq(PI_IRQ_SI, (data >> 16) & 0x3ff, si_callback);
		pi_schedule_irq(PI_IRQ_SI, 256, si_callback);
		break;
	case SI_REG_COMCSR:
		if (data & SI_TSTART)
		{
			syslog(SI,"CH: %d size_w: %d size_r: %d\n", (data >> 1) & 0x3, ((data >> 16) & 0x7f), ((data >> 8) & 0x7f));
			si_command((data >> SI_CHANNEL_BIT) & 0x3);
		}
		break;
	case SI_REG_SR:
		data &= SI_RDST0;
		SI_WRITE_UINT32(SI_REG_SR, data);
		break;
	default:
		break;
	}
}

uint32 si_read_register32(uint32 addr)
{
	uint32 data;

	data = SI_READ_UINT32(addr & 0xff);

	switch(addr & 0xff)
	{
	case SI_REG_C0_INBUFH:
		data = pad[0].pad_datah;
		data |= 0x00800000; // force this bit on to give analog pad meaning in many games
		break;
	case SI_REG_C0_INBUFL:
		data = pad[0].pad_datal;
		si_clear_rdst(0);
		break;
	case SI_REG_C1_INBUFH:
		data = pad[1].pad_datah;
		break;
	case SI_REG_C1_INBUFL:
		data = pad[1].pad_datal;
		si_clear_rdst(1);
		break;
	case SI_REG_C2_INBUFH:
		data = pad[2].pad_datah;
		break;
	case SI_REG_C2_INBUFL:
		data = pad[2].pad_datal;
		si_clear_rdst(2);
		break;
	case SI_REG_C3_INBUFH:
		data = pad[3].pad_datah;
		break;
	case SI_REG_C3_INBUFL:
		data = pad[3].pad_datal;
		si_clear_rdst(3);
		break;
	default:
		break;
	}
	return data;
}


bool si_check_interrupt(void)
{
	uint32 data;
	data = SI_READ_UINT32(SI_REG_COMCSR);

	//syslog(SI,"check irq %08x\n", data);

	if(data & SI_TCINTMASK)
	{
		if(data & SI_TCINT)
		{
			return true;
		}
	}

	if(data & SI_RDSTINTMASK)
	{
		if(data & SI_RDSTINT)
		{
			return true;
		}
	}
	return false;
}

void si_callback(void)
{
	//syslog(SI,"callback\n");
	pi_schedule_irq(PI_IRQ_SI, 256, si_callback);

	SI_WRITE_UINT32(SI_REG_SR, (SI_RDST0 ));//| SI_RDST1 | SI_RDST2 | SI_RDST3));
	SI_WRITE_UINT32(SI_REG_SR, (SI_RDST0 | SI_RDST1 | SI_RDST2 | SI_RDST3));
	si_update_rdstint();
}
