/*====================================================================

filename:     memory_card.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "system/types.h"
#include "hardware/memory_interface.h"
#include "memory_card.h"
#include "debug/tracers.h"
#include <stdio.h>

uint8	mc[0xa000];
bool	mc_locked;

uint32	mc_lock_count = 0;
uint32	mc_read_count;
uint32	mc_init_code;
uint32	mc_init_size_1;
uint32	mc_init_size_2;

void generate_code(void);
void mc_calc_checksum(char *mem, uint32 len, uint16 *crc1, uint16 *crc2);

void mc_unlock(void)
{
	if (mc_locked)
	{
		if (mc_lock_count == 2)
		{
			fprintf(stderr, "MC INIT SIZE 2: %08x\n", mc_read_count);
			mc_init_size_2 = mc_read_count;
			generate_code();
		}
	}
}


uint32 exnor_1st(uint32 r3, uint32 r4)
{
	uint32	i;
	uint32	r31;
	uint32	r6,r5,r29,r0;
	r31 = r3;
	for(i = 0 ; i < r4 ; i++)
	{
		r6 = r31 >> 23;
		r5 = r31 >> 15;
		r0 = r31 >> 7;
		r0 = r0 ^ r31;
		r0 = r0 ^ r5;
		r29 = ~(r6 ^ r0);
		r5 = r31 >> 1;
		r0 = (r29 & 0x1) << 30;
		r31 = r5 | r0;
	}
	return r31;
}


uint32 exnor(uint32 r3, uint32 r4)
{
	uint32	i;
	uint32	r31;
	uint32	r6,r5,r29,r0;
	r31 = r3;
	for(i = 0 ; i < r4 ; i++)
	{
		r6 = r31 << 23;
		r5 = r31 << 15;
		r0 = r31 << 7;
		r0 = r31 ^ r0;
		r0 = r5 ^ r0;
		r29 = ~(r6 ^ r0);
		r5 = r31 << 1;
		r0 = (r29 >> 30) & 0x2;
		r31 = r5 | r0;
	}
	return r31;
}

uint32 bitrev(uint32 data)
{
	uint32 res;
	sint32 i;
	res = 0;
	for(i = 31 ; i >= 0 ; i--)
	{
		res |= (data & 0x1) << i;
		data >>= 1;
	}
	return res;
}

void generate_code(void)
{
	uint32	init_val;
	uint32	dummy_len;
	uint32	r3, r4, r0, r29, r28, r31;
	uint32	unlock_code_buf[10];

	uint32	temp_crc;

	uint32	r20, r21, r22, r17, r19;
//	init_val = GetInitVal();
//	dummy_len = DummyLen();
//	if (ReadArrayUnlock(var98, init_val, unlock_code_buf, dummy_len, 0) <= 0)
//		return -3;

	init_val = mc_init_code;
	dummy_len = mc_init_size_1;

	r3 = dummy_len << 3;
	r29 = r3 + 1;
	r31 = exnor_1st(init_val, r29);

	r4 = r31 >> 23;
	r3 = r31 >> 15;
	r0 = r31 >> 7;
	r0 = r31 ^ r0;
	r0 = r3 ^ r0;
	r28 = ~(r4 ^ r0);
	
	r0 = r28 << 31;
	r0 = r31 | r0;
	temp_crc = r0;
	temp_crc = bitrev(temp_crc);

//	dummy_len = DummyLen();
	dummy_len = mc_init_size_2;

//	if (ReadArrayUnlock(var98, 0, unlock_code_buf, dummy_len + 0x14, 1) <= 0)
//		return -3;

	for (int i = 0 ; i < 5 ; i++)
		unlock_code_buf[i] = 0;
	

	r22 = unlock_code_buf[0]; // on longs
	r21 = unlock_code_buf[1]; // on longs
	r17 = unlock_code_buf[2]; // on longs
	r20 = unlock_code_buf[3]; // on longs
	r19 = unlock_code_buf[4]; // on longs

	r0 = temp_crc;
	r22 ^= r0;
	r29 = 0x20;
	r31 = exnor(temp_crc, r29);
	r4 = r31 << 23;
	r3 = r31 << 15;
	r0 = r31 << 7;
	r0 = r31 ^ r0;
	r0 = r3 ^ r0;
	r28 = ~(r4 ^ r0);
	r0 = r28 >> 31;
	r0 = r31 | r0;
	temp_crc = r0;
	r21 = r21 ^ temp_crc;
	r31 = exnor(temp_crc, 0x20);
	r4 = r31 << 23;
	r3 = r31 << 15;
	r0 = r31 << 7;
	r0 = r31 ^ r0;
	r0 = r3 ^ r0;
	r28 = ~(r4 ^ r0);
	r0 = r28 >> 31;
	r0 = r31 | r0;
	temp_crc = r0;

	r17 = r17 ^ temp_crc;
	r31 = exnor(temp_crc, 0x20);
	r4 = r31 << 23;
	r3 = r31 << 15;
	r0 = r31 << 7;
	r0 = r31 ^ r0;
	r0 = r3 ^ r0;
	r28 = ~(r4 ^ r0);
	r0 = r28 >> 31;
	r0 = r31 | r0;
	temp_crc = r0;

	r20 = r20 ^ temp_crc;
	r31 = exnor(temp_crc, 0x20);
	r4 = r31 << 23;
	r3 = r31 << 15;
	r0 = r31 << 7;
	r0 = r31 ^ r0;
	r0 = r3 ^ r0;
	r28 = ~(r4 ^ r0);
	r0 = r28 >> 31;
	r0 = r31 | r0;
	temp_crc = r0;

	r19 = r19 ^ temp_crc;
	r31 = exnor(temp_crc, 0x20);
	r4 = r31 << 23;
	r3 = r31 << 15;
	r0 = r31 << 7;
	r0 = r31 ^ r0;
	r0 = r3 ^ r0;
	r28 = ~(r4 ^ r0);
	r0 = r28 >> 31;
	r0 = r31 | r0;
	temp_crc = r0;

//	printf("%08x %08x %08x %08x %08x\n", r22, r21, r17, r20, r19);

	*((uint32 *)(&mc[0x00])) = byteswap32(r22);//0x076cdd8c;
	*((uint32 *)(&mc[0x04])) = byteswap32(r21);//0xd10f7ed0;
	*((uint32 *)(&mc[0x08])) = byteswap32(r17);//0x6ded227f;
	mc_calc_checksum((char *)mc, 0x1fc, (uint16 *)(&mc[0x1fc]), (uint16 *)(&mc[0x1fe]));
	mc_locked = false;

}
















void mc_calc_checksum(char *mem, uint32 len, uint16 *crc1, uint16 *crc2)
{
	uint32	i;
	uint32	val;
	uint16	check1, check2;
	check1 = check2 = 0;
	for(i = 0 ; i < len ; i += 2)
	{
		val	=	*((uint16 *)(&mem[i]));
		val = byteswap16(val);
		check1 += val;
		check2 += ~val;
	}
	*crc1 = byteswap16(check1);
	*crc2 = byteswap16(check2);
}

uint32 mc_get_status(void)
{
	if (mc_locked)
	{
		return 0x01000000;
	}
	else
	{
		return 0x41000000;
	}
}

void mc_init(void)
{
	uint32	i;

	mc_locked = true;

	for (i = 0 ; i < 0xa000 ; i++)
		mc[i] = 0;
	
	*((uint32 *)(&mc[0x20])) = 0x80000000;

	mc_calc_checksum((char *)mc, 0x1fc, (uint16 *)(&mc[0x1fc]), (uint16 *)(&mc[0x1fe]));
	
	
	for(i = 0 ; i < 127 ; i++)
	{
		*((uint16 *)(&mc[0x2000 + i * 0x40])) = 0xff;//byteswap16(0x00ff);
		*((uint16 *)(&mc[0x4000 + i * 0x40])) = 0xff;//byteswap16(0x00ff);
	}

	mc_calc_checksum((char *)(mc + 0x2000), 0x1ffc, (uint16 *)&mc[0x3ffc], (uint16 *)&mc[0x3ffe]);
	mc_calc_checksum((char *)(mc + 0x4000), 0x1ffc, (uint16 *)&mc[0x5ffc], (uint16 *)&mc[0x5ffe]);

	
	*((uint16 *)(&mc[0x6006])) = byteswap16(0x07fb);
	*((uint16 *)(&mc[0x8006])) = byteswap16(0x07fb);

	mc_calc_checksum((char *)(mc + 0x6004), 0x1ffc, (uint16 *)&mc[0x6000], (uint16 *)&mc[0x6002]);
	mc_calc_checksum((char *)(mc + 0x8004), 0x1ffc, (uint16 *)&mc[0x8000], (uint16 *)&mc[0x8002]);
}

extern uint32	exi_ch1_cmd_ptr;
extern uint32	exi_ch1_cmd_count;
extern uint8	exi_ch1_cmd[10];


void mc_execute_cmd(void)
{
	uint32 i;
	for(i = 0 ; i < 10 ; i++)
	{
		syslog(MC,"%02x\n", exi_ch1_cmd[i]);
	}
	if(exi_ch1_cmd[0] == 0x52)
	{
		if (mc_locked)
		{
			if (mc_lock_count == 0)
			{
				mc_init_code = (exi_ch1_cmd[1] << 29) | (exi_ch1_cmd[2] << 21) | (exi_ch1_cmd[3] << 19) | (exi_ch1_cmd[4] << 12);
				fprintf(stderr, "MC INIT CODE %08x\n", mc_init_code);
			}
			if (mc_lock_count == 1)
			{
				fprintf(stderr, "MC INIT SIZE 1: %08x\n", mc_read_count);
				mc_init_size_1 = mc_read_count;
			}

			mc_lock_count++;
		}
		mc_read_count = 0;
	}
}

uint32 mc_read_data(uint32 offset, uint32 size)
{
	mc_read_count += size + 1;
	if (offset >= 0xa000)
	{
		return 0;
	}
	else
	{
		syslog(EXI,"READ WORD MEMCARD %08x\n", byteswap32(*((uint32 *)(&mc[offset]))));
		return byteswap32(*((uint32 *)(&mc[offset])));
	}
}