/*====================================================================

filename:     dvd_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdio.h>
#include <stdlib.h>

#include "system/types.h"
#include "dvd_interface.h"
#include "hardware/hw_io.h"
#include "hardware/memory_interface.h"
#include "gcm_wrapper.h"
#include "hardware/processor_interface.h"
#include "cpu/trx_ppc_rec.h"
#include "plugins/gx_plugin.h"

void di_process_command();


#define DI_REGS_SIZE	256
char di_regs[DI_REGS_SIZE];

void di_init(void)
{
	int i;
	for (i = 0 ; i < DI_REGS_SIZE ; i++)
	{
		di_regs[i] = 0;
	}
}

void di_close(void)
{
}

#define DI_REG_SR		0x00
#define DI_REG_CVR		0x04
#define DI_REG_CMDBUF0	0x08
#define DI_REG_CMDBUF1	0x0c
#define DI_REG_CMDBUF2	0x10
#define DI_REG_DMAMAR	0x14
#define DI_REG_DMALEN	0x18
#define DI_REG_CR		0x1c


#define DI_REG_UINT32(A)	*((uint32 *)(di_regs + (A)))
#define DI_REG_UINT16(A)	*((uint16 *)(di_regs + (A)))
#define DI_REG_UINT8(A)		*((uint8 *)(di_regs + (A)))

#define DI_WRITE_UINT32(A, D)	DI_REG_UINT32(A) = byteswap32((D))
#define DI_WRITE_UINT16(A, D)	DI_REG_UINT16(A) = byteswap16((D))
#define DI_WRITE_UINT8(A, D)	DI_REG_UINT8(A) = (uint8)(D)

#define DI_READ_UINT32(A)		(byteswap32(DI_REG_UINT32(A)))
#define DI_READ_UINT16(A)		(byteswap16(DI_REG_UINT16(A)))
#define DI_READ_UINT8(A)		(DI_REG_UINT8(A))


void di_write_register8(uint32 addr, uint8 data)
{
	DI_WRITE_UINT8(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case 0x00:
		if(data & (1 << 4))
		{
			uint32 sr;
			sr = DI_READ_UINT32(DI_REG_SR);
			sr &= ~(1 << 4);
			DI_WRITE_UINT32(DI_REG_SR, sr);
		}
		break;
	case 0x1c:
		di_process_command();
		break;
	default:
		break;
	}
}

void di_write_register16(uint32 addr, uint16 data)
{
	DI_WRITE_UINT16(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case 0x00:
		if(data & (1 << 4))
		{
			uint32 sr;
			sr = DI_READ_UINT32(DI_REG_SR);
			sr &= ~(1 << 4);
			DI_WRITE_UINT32(DI_REG_SR, sr);
		}
		break;
	case 0x1c:
		di_process_command();
		break;
	default:
		break;
	}
}

void di_write_register32(uint32 addr, uint32 data)
{
	DI_WRITE_UINT32(addr & 0xff, data);

	switch(addr & 0xff)
	{
	case 0x00:
		if(data & (1 << 4))
		{
			uint32 sr;
			sr = DI_READ_UINT32(DI_REG_SR);
			sr &= ~(1 << 4);
			DI_WRITE_UINT32(DI_REG_SR, sr);
		}
		break;
	case 0x1c:
		di_process_command();
		break;
	default:
		break;
	}
}

uint8 di_read_register8(uint32 addr)
{
	uint8 data;

	data = DI_READ_UINT8(addr & 0xff);

	switch(addr & 0xff)
	{
	case 0x24:
		data = 0x00;
		break;
	default:
		break;
	}
	return data;
}

uint16 di_read_register16(uint32 addr)
{
	uint16 data;

	data = DI_READ_UINT16(addr & 0xff);

	switch(addr & 0xff)
	{
	case 0x24:
		data = 0x00;
		break;
	default:
		break;
	}
	return data;
}

uint32 di_read_register32(uint32 addr)
{
	uint32 data;

	data = DI_READ_UINT32(addr & 0xff);

	switch(addr & 0xff)
	{
	case 0x24:
		data = 0x00;
		break;
	default:
		break;
	}
	return data;
}


bool di_check_interrupt(void)
{
	uint32 sr, cvr;
	uint32 mask;

	sr = DI_READ_UINT32(DI_REG_SR);
	mask = (sr << 1) & ((1 << 2) | (1 << 4) | (1 << 6));
	if (sr & mask)		// TCINT
	{
		return true;
	}

	cvr = DI_READ_UINT32(DI_REG_CVR);
	mask = (cvr << 1) & (1 << 2);
	if (cvr & mask)		// CVRINT
	{
		return true;
	}
	return false;
}

void di_open_cover(void)
{
	syslog(DI,"Open Cover\n");
	uint32 temp;
	temp = DI_READ_UINT32(DI_REG_CVR);
	temp |= ((1 << 2) | (1 << 0)); // Cover has been opened and assert irq
	DI_WRITE_UINT32(DI_REG_CVR, temp);
}

void di_close_cover(void)
{
	syslog(DI,"Open Cover\n");
	uint32 temp;
	temp = DI_READ_UINT32(DI_REG_CVR);
	temp &= ~(1<<0); // Cover has been closed
	temp |= ((1 << 2));  // and assert irq
	DI_WRITE_UINT32(DI_REG_CVR, temp);
}

void di_raise_interrupt(void)
{
	uint32 sr;
	syslog(DI,"TCINT\n");
	sr = DI_READ_UINT32(DI_REG_SR);
	sr |= (1 << 4);
	DI_WRITE_UINT32(DI_REG_SR, sr);

	// should be pi_raise_interrupt() here
}

void di_transfer_complete_callback(void)
{
	di_raise_interrupt();
}

void di_process_command(void)
{
	uint32 len;
	uint32 addr;
	uint32 ofs;
	uint32 cmd, cmd1, cmd2;
	uint32	data32;

	cmd = DI_READ_UINT32(DI_REG_CMDBUF0);

	syslog(DI,"CMD: %08x\n", cmd);

	switch (cmd)
	{
	case 0xa8000040:
		cmd1 = DI_READ_UINT32(DI_REG_CMDBUF1);
		cmd2 = DI_READ_UINT32(DI_REG_CMDBUF2);
		len = DI_READ_UINT32(DI_REG_DMALEN);
		if(len != cmd2 || cmd1 != 0)
		{
			syslog_error(DI,"SOMETHING IS WRONG %08x %08x %08x\n", cmd, cmd1, cmd2);
		}
		// otherwise continue as READ_DATA
	case 0xa8000000:
	case 0x12000000:
		len = DI_READ_UINT32(DI_REG_DMALEN);
		addr = DI_READ_UINT32(DI_REG_DMAMAR);
		ofs = DI_READ_UINT32(DI_REG_CMDBUF1);
		syslog(DI,"READ_DATA: %08x (%08x) -> %08x\n", ofs << 2, len, addr);
		gcm_read((char *)gMemory + (addr & 0x01ffffff), ofs << 2, len);
		GX_CacheMarkInvalid((addr & 0x01ffffff), len);
		//fprintf(stderr, "DVD Invalidate %08x %08x\n", (addr & 0x01ffffff), len);
		rec_cache_snoop_area((addr & 0x01ffffff), len);
		DI_WRITE_UINT32(DI_REG_DMALEN, 0);
		//di_raise_interrupt();
		pi_schedule_irq(PI_IRQ_DI, 1, di_transfer_complete_callback);
		break;
	case 0xab000000:		// SEEK
		pi_schedule_irq(PI_IRQ_DI, 1, di_transfer_complete_callback);
		break;

	case 0xe1000000:		// SEEK
	case 0xe1010000:		// SEEK
		pi_schedule_irq(PI_IRQ_DI, 1, di_transfer_complete_callback);
		break;

	case 0xe2000000:		// Stream
		pi_schedule_irq(PI_IRQ_DI, 1, di_transfer_complete_callback);
		ofs = DI_READ_UINT32(DI_REG_CMDBUF1);
		gcm_read((char *)&data32, ofs << 2, 4);
		data32 = byteswap32(data32);
		DI_WRITE_UINT32(0x20, data32);
		break;
	default:
		cmd1 = DI_READ_UINT32(DI_REG_CMDBUF1);
		cmd2 = DI_READ_UINT32(DI_REG_CMDBUF2);
		syslog_error(DI,"UNKNOWN COMMAND: %08x %08x %08x\n", cmd, cmd1, cmd2);
		pi_schedule_irq(PI_IRQ_DI, 1, di_transfer_complete_callback);
		break;

	}
}