/*====================================================================

filename:     command_processor_interface.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include "config.h"
#include "system/types.h"
#include "video_interface.h"
#include "hardware/hw_io.h"
#include "hardware/memory_interface.h"
#include "plugins/gx_plugin.h"
/*

Command Processor. 

information taken from yagcd

CP memory based FIFO implementation basically directly copied from Dolphin-emu

*** FIX IMPLEMENTATION MORE ACCORDING TO YAGCD AND DOLPHIN!!! ****

*/

#define	CP_REG_SR				0x00
#define CP_REG_CR				0x02
#define CP_REG_CLEAR			0x04
#define CP_REG_TOKEN			0x0E
#define CP_REG_BBOX_LEFT		0x10
#define CP_REG_BBOX_RIGHT		0x12
#define CP_REG_BBOX_TOP			0x14
#define CP_REG_BBOX_BOTTOM		0x16
#define CP_REG_FIFO_BASE_LO		0x20
#define CP_REG_FIFO_BASE_HI		0x22
#define CP_REG_FIFO_END_LO		0x24
#define CP_REG_FIFO_END_HI		0x26
#define CP_REG_FIFO_HIGH_WM_LO  0x28
#define CP_REG_FIFO_HIGH_WM_HI  0x2a
#define CP_REG_FIFO_LOW_WM_LO   0x2c
#define CP_REG_FIFO_LOW_WM_HI   0x2e
#define CP_REG_FIFO_RW_DIST_LO	0x30
#define CP_REG_FIFO_RW_DIST_HI	0x32
#define CP_REG_FIFO_WR_PTR_LO	0x34
#define CP_REG_FIFO_WR_PTR_HI	0x36
#define CP_REG_FIFO_RD_PTR_LO	0x38
#define CP_REG_FIFO_RD_PTR_HI	0x3a
#define CP_REG_FIFO_BP_LO		0x3c
#define CP_REG_FIFO_BP_HI		0x3e

#define CP_REGS_SIZE	256
uint16 cp_regs16[CP_REGS_SIZE];

uint32 cp_token;
uint32 cp_fifo_base;
uint32 cp_fifo_end;
uint32 cp_fifo_writepointer;
uint32 cp_fifo_readpointer;
uint32 cp_fifo_bp;
uint32 cp_fifo_readwrite_distance;
uint32 cp_fifo_watermark_low;
uint32 cp_fifo_watermark_high;

uint32 cp_fifo_read_enable;
uint32 cp_fifo_cpirq_enable;
uint32 cp_fifo_overflow_enable;
uint32 cp_fifo_underflow_enable;
uint32 cp_fifo_gplink_enable;
uint32 cp_fifo_bp_enable;

uint32 cp_status_overflow;
uint32 cp_status_underflow;
uint32 cp_status_idle;
uint32 cp_status_ready;
uint32 cp_status_bpint;

void cp_processfifo(void);

void cp_init(void)
{
	int i;
	for (i = 0 ; i < CP_REGS_SIZE ; i++)
	{
		cp_regs16[i] = 0;
	}

	cp_fifo_read_enable = 0;
	cp_fifo_cpirq_enable = 0;
	cp_fifo_overflow_enable = 0;
	cp_fifo_underflow_enable = 0;
	cp_fifo_gplink_enable = 0;
	cp_fifo_bp_enable = 0;

	cp_status_overflow = 0;
	cp_status_underflow = 0;
	cp_status_idle = 1;
	cp_status_ready = 1;
	cp_status_bpint = 0;

	// for temporary debugging!
	log_trace_on[CMP]=1;
}

void cp_close(void)
{
}

void cp_write_register32(uint32 addr, uint32 data)
{
	printf("cp write32\n");
	exit(1);
}
uint32 cp_read_register32(uint32 addr)
{
	printf("cp read32\n");
	exit(1);
}

void cp_write_register8(uint32 addr, uint8 data)
{
	printf("cp write8\n");
	exit(1);
}

uint8 cp_read_register8(uint32 addr)
{
	printf("cp read32\n");
	exit(1);
}

/////////////////////////////////////////////////////////////////////
//
// only 16bit access is supported 
//


void cp_write_register16(uint32 addr, uint16 data)
{
	uint32 data32;
	syslog(CMP, "WR16: @%x %x\n",addr, data);

	data32 = data;
	switch(addr & 0xff)
	{
	case CP_REG_SR:
		// bit 2: gp is idle (1: idle)
		// bit 3: gp is ready for commands (1: ready)
		// write 1 to clear bit
		// bit 4: bp interrupt
		if(data & 16) cp_status_bpint = 0;
		break;
	case CP_REG_CR: 
		// bit 0: gp fifo read enable
		// bit 1: cp irq enable ?
		// bit 2: FIFO overflow enable
		// bit 3: FIFO underflow enable 
		// bit 4: gp link enable
		// bit 5: bp enable
		cp_fifo_read_enable = data & 1;
		cp_fifo_cpirq_enable = (data>>1) & 1;
		cp_fifo_overflow_enable = (data>>2) & 1;
		cp_fifo_underflow_enable = (data>>3) & 1;
		cp_fifo_gplink_enable = (data>>4) & 1;
		cp_fifo_bp_enable = (data>>5) & 1;
		break;
	case CP_REG_CLEAR:
		// bit 0: fifo underflow clear
		// bit 1: fifo overflow clear
		if(data & 1)cp_status_underflow = 0;
		if(data & 2)cp_status_overflow = 0;
		break;
	case CP_REG_TOKEN:
		cp_token = data;
		break;
	case CP_REG_FIFO_BASE_LO:
		cp_fifo_base = ((cp_fifo_base  & 0xffff0000) | data32) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_BASE_HI:
		cp_fifo_base = ((cp_fifo_base  & 0xffff) | (data32 << 16)) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_END_LO:
		cp_fifo_end = ((cp_fifo_end  & 0xffff0000) | data32) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_END_HI:
		cp_fifo_end = ((cp_fifo_end  & 0xffff) | (data32 << 16)) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_HIGH_WM_LO:
		cp_fifo_watermark_high = (cp_fifo_watermark_high  & 0xffff0000) | data32; 
		break;
	case CP_REG_FIFO_HIGH_WM_HI:
		cp_fifo_watermark_high = (cp_fifo_watermark_high  & 0xffff) | (data32 << 16); 
		break;
	case CP_REG_FIFO_LOW_WM_LO:
		cp_fifo_watermark_low = (cp_fifo_watermark_low  & 0xffff0000) | data32; 
		break;
	case CP_REG_FIFO_LOW_WM_HI:
		cp_fifo_watermark_low = (cp_fifo_watermark_low  & 0xffff) | (data32 << 16); 
		break;
	case CP_REG_FIFO_RW_DIST_LO:
		cp_fifo_readwrite_distance = (cp_fifo_readwrite_distance & 0xffff0000) | data32; 
		break;
	case CP_REG_FIFO_RW_DIST_HI:
		cp_fifo_readwrite_distance = (cp_fifo_readwrite_distance & 0xffff) | (data32 << 16); 
		break;
	case CP_REG_FIFO_WR_PTR_LO:
		cp_fifo_writepointer = ((cp_fifo_writepointer & 0xffff0000) | data32) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_WR_PTR_HI:
		cp_fifo_writepointer = ((cp_fifo_writepointer & 0xffff) | (data32 << 16)) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_RD_PTR_LO:
		cp_fifo_readpointer = ((cp_fifo_readpointer & 0xffff0000) | data32) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_RD_PTR_HI:
		cp_fifo_readpointer = ((cp_fifo_readpointer & 0xffff) | (data32 << 16)) & ALIGN_32BYTES; 
		break;
	case CP_REG_FIFO_BP_LO:
		cp_fifo_bp = (cp_fifo_bp & 0xffff0000) | data32; 
		break;
	case CP_REG_FIFO_BP_HI:
		cp_fifo_bp = (cp_fifo_readpointer & 0xffff) | (data32 << 16); 
		break;
	default:
		cp_regs16[addr & 0xff] = data;
		break;
	}
}

uint16 cp_read_register16(uint32 addr)
{
	uint16 data = 0;

	switch(addr & 0xff)
	{
	case CP_REG_SR:
		// bit 0: gx fifo overflow
		// bit 1: gx fifo underflow
		// bit 2: gp is idle (1: idle)
		// bit 3: gp is ready for commands (1: ready)
		// bit 4: bp interrupt
		data |= cp_status_overflow;
		data |= cp_status_underflow << 1;
		data |= 1 << 2;
		data |= 1 << 3;
		data |= cp_status_bpint << 4;
		break;
	case CP_REG_CR: 
		// bit 0: gp fifo read enable
		// bit 1: cp irq enable ?
		// bit 2: FIFO overflow enable
		// bit 3: FIFO underflow enable 
		// bit 4: gp link enable
		// bit 5: bp enable
		data |= cp_fifo_read_enable;
		data |= cp_fifo_cpirq_enable << 1;
		data |= cp_fifo_overflow_enable << 2;
		data |= cp_fifo_underflow_enable << 3;
		data |= cp_fifo_gplink_enable << 4;
		data |= cp_fifo_bp_enable << 5;
		break;
	case CP_REG_TOKEN:
		data = cp_token;
		break;
	case CP_REG_FIFO_BASE_LO:
		data = cp_fifo_base & 0xffff; 
		break;
	case CP_REG_FIFO_BASE_HI:
		data = (cp_fifo_base >> 16);
		break;
	case CP_REG_FIFO_END_LO:
		data = cp_fifo_end & 0xffff;
		break;
	case CP_REG_FIFO_END_HI:
		data = (cp_fifo_end >> 16);
		break;
	case CP_REG_FIFO_HIGH_WM_LO:
		data = cp_fifo_watermark_high & 0xffff;
		break;
	case CP_REG_FIFO_HIGH_WM_HI:
		data = (cp_fifo_watermark_high >> 16);
		break;
	case CP_REG_FIFO_LOW_WM_LO:
		data = cp_fifo_watermark_low & 0xffff;
		break;
	case CP_REG_FIFO_LOW_WM_HI:
		data = (cp_fifo_watermark_low >> 16);
		break;
	case CP_REG_FIFO_RW_DIST_LO:
		data = cp_fifo_readwrite_distance & 0xffff;
		break;
	case CP_REG_FIFO_RW_DIST_HI:
		data = (cp_fifo_readwrite_distance >> 16);
		break;
	case CP_REG_FIFO_WR_PTR_LO:
		data = cp_fifo_writepointer & 0xffff;
		break;
	case CP_REG_FIFO_WR_PTR_HI:
		data = (cp_fifo_writepointer >> 16);
		break;
	case CP_REG_FIFO_RD_PTR_LO:
		data = cp_fifo_readpointer & 0xffff;
		break;
	case CP_REG_FIFO_RD_PTR_HI:
		data = (cp_fifo_readpointer >> 16);
		break;
	case CP_REG_FIFO_BP_LO:
		data = cp_fifo_bp & 0xffff;
		break;
	case CP_REG_FIFO_BP_HI:
		data = (cp_fifo_bp >> 16);
		break;
	default:
		data = cp_regs16[addr & 0xff];
		break;
	}
	syslog(CMP, "RR16: @%x %x\n",addr, data);
	return data;
}


bool cp_check_interrupt(void)
{
	if(cp_status_bpint == 1)return true;
	return false;
}

//
// CPU's WPAR facility will write 32byte bursts (256bit) to the command processor
// the command processor will write this burst to memory at the PI_FIFO_WRITEPOINTER which is wrapped
// using PI_FIFO_START and PI_FIFO_END
// then if the GPU is linked, it is sent to the GPU
//
void cp_write_256(uint8 *data)
{
	// 32 bytes to fifo in memory buffer controlled by PI 
	memcpy(&gMemory[pi_fifo_writepointer & MEM_MASK32MB], data, 32); 
	pi_fifo_writepointer += 32;

	if(pi_fifo_writepointer >= pi_fifo_end)
	{
		pi_fifo_writepointer = pi_fifo_start;
	}

	// if the fifo is linked, update write and try to process
	if(cp_fifo_gplink_enable == 1)
	{
		// since fifos are supposed to be linked, all we need to do is update the pointer..
		cp_fifo_writepointer += 32;
		if(cp_fifo_writepointer >= cp_fifo_end)
		{
			cp_fifo_writepointer = cp_fifo_base;
		}
		// see if there is anything to feed to GPU
		cp_processfifo();
	}
}

void cp_processfifo(void)
{
	// distance update
	if(cp_fifo_writepointer >= cp_fifo_readpointer)
	{
		cp_fifo_readwrite_distance = cp_fifo_writepointer - cp_fifo_readpointer;
	}
	else
	{
		cp_fifo_readwrite_distance = (cp_fifo_writepointer - cp_fifo_base) + (cp_fifo_end - cp_fifo_readpointer);
	}

	// anything to transfer to GPU ?
	if(cp_fifo_read_enable == 1)
	{
		// if we're on a breakpoint, dont send anything
		if(!((cp_fifo_bp_enable == 1) && (cp_status_bpint == 1)))
		{
			while(cp_fifo_readwrite_distance > 0)
			{
				uint32 size;
				// try to send maximum amount of data
				size = cp_fifo_readwrite_distance;

				// but no more than the GPU buffer will hold ..
				if(size > GPU_FIFO_SIZE)
				{
					size = GPU_FIFO_SIZE;
				}

				// or up to end of buffer ..
				if((cp_fifo_readpointer + size) > cp_fifo_end)
				{
					// send block up to end of fifo
					size = cp_fifo_end - cp_fifo_readpointer;
				}

				// or up to breakpoint
				if(cp_fifo_bp_enable == 1)
				{
					if((cp_fifo_readpointer + size) > cp_fifo_bp)
					{
						size = (cp_fifo_bp - cp_fifo_readpointer);
						cp_status_bpint = 1;
						// signal interrupt
						trx_ppc_signal_interrupt();
						GX_WriteFifo(&gMemory[cp_fifo_readpointer & MEM_MASK32MB], size);
						cp_fifo_readpointer += size;
						cp_fifo_readwrite_distance -= size;
						return;
					}
				}

				GX_WriteFifo(&gMemory[cp_fifo_readpointer & MEM_MASK32MB], size);
				cp_fifo_readpointer += size;
				cp_fifo_readwrite_distance -= size;

				if(cp_fifo_readpointer >= cp_fifo_end)
				{
					cp_fifo_readpointer = cp_fifo_base;
				}
			}
		}
	}
}
