/*====================================================================

filename:     gdsp_opcodes.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include <stdio.h>
#include "dtypes.h"
#include "gdsp_opcodes.h"
#include "gdsp_memory.h"
#include "gdsp_interpreter.h"
#include "gdsp_registers.h"

#define DSP_REG_MASK	0x1f
#define	R_SR            0x13

#define DEBUG_REGISTER_CHECK	1

#if DEBUG_REGISTER_CHECK
#define RC_OPC_EXT(x) dsp_rc_opc_ext |= x
#else // DEBUG_REGISTER_CHECK
#define RC_OPC_EXT(x)
#endif // DEBUG_REGISTER_CHECK


bool dsp_op_unknown(uint16 opc)
{
	dsp->pc = dsp->err_pc;
	return false;
}

uint16 dsp_op_read_reg(uint8 reg)
{
	uint16	val;
	switch(reg & 0x1f)
	{
	case 0x0c:
	case 0x0d:
	case 0x0e:
	case 0x0f:
		val = dsp_reg_load_stack(reg - 0x0c);
		break;
	default:
		val = dsp->r[reg];
		break;
	}
	return val;
}

void dsp_op_write_reg(uint8 reg, uint16 val)
{
	switch(reg & 0x1f)
	{
	case 0x0c:
	case 0x0d:
	case 0x0e:
	case 0x0f:
		dsp_reg_store_stack(reg - 0x0c, val);
		break;
	default:
		dsp->r[reg] = val;
		break;
	}
}

sint64 dsp_get_long_prod(void)
{
	sint64 val;
	sint64 low_prod;
	val = (sint8)dsp->r[0x16];
	val <<= 32;
	low_prod = dsp->r[0x15];
	low_prod += dsp->r[0x17];
	low_prod <<= 16;
	low_prod |= dsp->r[0x14];
	val += low_prod;
	return val;
}

void dsp_set_long_prod(sint64 val)
{
	dsp->r[0x14] = (uint16)val;
	val >>= 16;
	dsp->r[0x15] = (uint16)val;
	val >>= 16;
	dsp->r[0x16] = (uint16)val;
	dsp->r[0x17] = 0;
}

sint64 dsp_get_long_acc(uint8 reg)
{
	sint64 val;
	sint64 low_acc;
	val = (sint8)dsp->r[0x10+reg];
	val <<= 32;
	low_acc = dsp->r[0x1e+reg];
	low_acc <<= 16;
	low_acc |= dsp->r[0x1c+reg];
	val |= low_acc;
	return val;
}

uint64 dsp_get_ulong_acc(uint8 reg)
{
	uint64 val;
	uint64 low_acc;
	val = (uint8)dsp->r[0x10+reg];
	val <<= 32;
	low_acc = dsp->r[0x1e+reg];
	low_acc <<= 16;
	low_acc |= dsp->r[0x1c+reg];
	val |= low_acc;
	return val;
}

sint64 dsp_get_long_acx(uint8 reg)
{
	sint64 val;
	sint64 low_acc;
	val = (sint16)dsp->r[0x1a+reg];
	val <<= 16;
	low_acc = dsp->r[0x18+reg];
	val |= low_acc;
	return val;
}

void dsp_set_long_acc(uint8 reg, sint64 val)
{
	dsp->r[0x1c+reg] = (uint16)val;
	val >>= 16;
	dsp->r[0x1e+reg] = (uint16)val;
	val >>= 16;
	dsp->r[0x10+reg] = (uint16)val;
}





//-------------------------------------------------------------
// EXTended opcodes

bool dsp_opc_ext_l(uint16 opc)
{
	uint8	dreg;
	uint8	sreg;
	uint16	val;

	opc &= 0xff;
	sreg = (opc & 0x3);
	dreg = ((opc >> 3) & 0x7) + 0x18;
	val = dsp_dmem_read(dsp->r[sreg]);
	dsp->r[dreg] = val;
	if (opc & 0x04)
		dsp->r[sreg] += dsp->r[sreg+4];
	else
		dsp->r[sreg]++;
	return true;
}

bool dsp_opc_ext_ld(uint16 opc)
{
	uint8	dreg1;
	uint8	dreg2;
	uint8	sreg;
	uint16	val;

	if ((opc & 0x03) == 0x03)
	{
		// special version
		return dsp_op_unknown(opc);
	}
	else
	{
		dreg1 = (((opc >> 5) & 0x1) << 1) + 0x18;
		dreg2 = (((opc >> 4) & 0x1) << 1) + 0x19;
		sreg = opc & 0x3;
		val = dsp_dmem_read(dsp->r[sreg]);
		dsp->r[dreg1] = dsp->r[dreg2] = val;
		if (opc & 0x04)
			dsp->r[sreg] += dsp->r[sreg+0x04];
		else
			dsp->r[sreg]++;
		if (opc & 0x08)
			dsp->r[0x03] += dsp->r[0x07];
		else
			dsp->r[0x03]++;

	}
	return true;
}


bool dsp_opc_ext_s(uint16 opc)
{
	uint8	areg;
	uint8	sreg;
	uint16	val;

	if ((opc & 0xe0) != 0x20)
	{
		if ((opc & 0xf0) == 0x10)
		{
			areg = ((opc >> 2) & 0x03) + 0x18;
			sreg = ((opc >> 0) & 0x03) + 0x1c;
			dsp->r[areg] = dsp->r[sreg];
			return true;
		}
		else 
		{
			sreg = ((opc >> 0) & 0x03);
			switch((opc >> 2) & 0x3)
			{
			case 0x0:
				break;
			case 0x1:
				dsp->r[sreg]--;
				break;
			case 0x2:
				dsp->r[sreg]++;
				break;
			case 0x3:
				dsp->r[sreg] += dsp->r[sreg + 0x4];
				break;
			}
			return true;
		}
	}

	areg = opc & 0x3;
	sreg = ((opc >> 3) & 0x3) + 0x1c;

	val = dsp->r[sreg];
	dsp_dmem_write(dsp->r[areg], val);

	if (opc & 0x4)
		dsp->r[areg] += dsp->r[0x04];
	else
		dsp->r[areg]++;
	return true;
}

bool dsp_opc_ext_ls(uint16 opc)
{
	uint8	dreg;
	uint8	areg;
	uint16	val;
	dreg = ((opc >> 4) & 0x3) + 0x18;
	areg = (opc & 0x1) + 0x1e;

	if(opc & 0x2)
	{
		dsp_dmem_write(dsp->r[0x00], dsp->r[areg]);
		val = dsp_dmem_read(dsp->r[0x03]);
		dsp_op_write_reg(dreg, val);
	}
	else
	{
		dsp_dmem_write(dsp->r[0x03], dsp->r[areg]);
		val = dsp_dmem_read(dsp->r[0x00]);
		dsp_op_write_reg(dreg, val);
	}
	if (opc & 0x4)
		dsp->r[0x00] += dsp->r[0x04];
	else
		dsp->r[0x00]++;
	if (opc & 0x8)
		dsp->r[0x03] += dsp->r[0x07];
	else
		dsp->r[0x03]++;
	return true;
}

bool dsp_opc_ext_prolog(uint16 opc)
{
	if ((opc & 0x00ff) == 0) return true;
	//return dsp_op_unknown(opc);
	switch((opc >> 6) & 0x3)
	{
	case 0x0:
		return dsp_opc_ext_s(opc);
	case 0x2:
		return dsp_opc_ext_ls(opc);
	case 0x3:
		return dsp_opc_ext_ld(opc);
	case 0x1:
		return true;
	default:
		return dsp_op_unknown(opc);
	}
	return dsp_op_unknown(opc);
}

bool dsp_opc_ext_epilog(uint16 opc)
{
	if ((opc & 0xc0) == 0x40)
		return dsp_opc_ext_l(opc);
	else
		return true;
}


//-------------------------------------------------------------
// opcodes changing Program Counter

bool dsp_opc_call(uint16 opc)
{
	uint16 dest;
	if ((opc & 0xf) != 0xf)
		return dsp_op_unknown(opc);
	dest = dsp_fetch_code();
	dsp_reg_store_stack(DSP_STACK_C, dsp->pc);
	dsp->pc = dest;
	return true;
}

bool dsp_opc_jxx(uint16 opc)
{
	uint16	dest;
	bool	taken;

	taken = false;
	dest = dsp_fetch_code();
	switch(opc & 0xf)
	{
	case 0x1:
		if ((!(dsp->r[R_SR] & 0x02)) && (dsp->r[R_SR] & 0x08))
			taken = true;
		break;
	case 0x2:
		if (!(dsp->r[R_SR] & 0x08))
			taken = true;
		break;
	case 0x4:
		if (!(dsp->r[R_SR] & 0x04))
			taken = true;
		break;
	case 0x5:
		if (dsp->r[R_SR] & 0x04)
			taken = true;
		break;
	case 0xc:
		if (!(dsp->r[R_SR] & 0x40))
			taken = true;
		break;
	case 0xd:
		if (dsp->r[R_SR] & 0x40)
			taken = true;
		break;
	case 0xf:
		taken = true;
		break;
	default:
		return dsp_op_unknown(opc);
	}

	if (taken) dsp->pc = dest;
	return true;
}

bool dsp_opc_jmpa(uint16 opc)
{
	uint8	reg;
	uint16	addr;

	if ((opc & 0xf) != 0xf) 
		return dsp_op_unknown(opc);

	reg = (opc >> 5) & 0x7;
	addr = dsp_op_read_reg(reg);

	if (opc & 0x0010)
	{
		// CALLA
		dsp_reg_store_stack(DSP_STACK_C, dsp->pc);
	}

	dsp->pc = addr;
	return true;
}



bool dsp_opc_ret(uint16 opc)
{
	if ((opc & 0xf) != 0xf)
		return dsp_op_unknown(opc);
	dsp->pc = dsp_reg_load_stack(DSP_STACK_C);
	return true;
}

bool dsp_opc_rti(uint16 opc)
{
	if ((opc & 0xf) != 0xf)
		return dsp_op_unknown(opc);
	//dsp->r[0x10] = dsp_reg_load_stack(DSP_STACK_D);
	dsp->r[0x13] = dsp_reg_load_stack(DSP_STACK_D);
	dsp->pc = dsp_reg_load_stack(DSP_STACK_C);
//	printf("RTI -> %04x\n", dsp->pc);
	return true;
}

bool dsp_opc_halt(uint16 opc)
{
	dsp->cr |= 0x4;
	dsp->pc = dsp->err_pc;
	return true;
}

bool dsp_opc_loop(uint16 opc)
{
	uint16	reg;
	uint16	cnt;
	uint16	loop_pc;
	reg = opc & 0x1f;
	cnt = dsp->r[reg];
	loop_pc = dsp->pc;
	while(cnt--)
	{
		if (!gdsp_step())
			return false;
		dsp->pc = loop_pc;
	}
	dsp->pc = loop_pc + 1;
	return true;
}

bool dsp_opc_loopi(uint16 opc)
{
	uint16	cnt;
	uint16	loop_pc;
	cnt = opc & 0xff;
	loop_pc = dsp->pc;
	while(cnt--)
	{
		if (!gdsp_step())
			return false;
		dsp->pc = loop_pc;
	}
	dsp->pc = loop_pc + 1;
	return true;
}

bool dsp_opc_bloop(uint16 opc)
{
	uint16 reg;
	uint16	cnt;
	uint16	loop_pc;

	reg = opc & 0x1f;
	cnt = dsp->r[reg];
	loop_pc = dsp_fetch_code();
	if (cnt)
	{
		dsp->loop = 1;
		dsp_reg_store_stack(0, dsp->pc);
		dsp_reg_store_stack(2, loop_pc);
		dsp_reg_store_stack(3, cnt);
	}
	else
		dsp->pc = loop_pc + 1;
	return true;
}

bool dsp_opc_bloopi(uint16 opc)
{
	uint16	cnt;
	uint16	loop_pc;

	cnt = opc & 0xff;
	loop_pc = dsp_fetch_code();
	if (cnt)
	{
		dsp->loop = 1;
		dsp_reg_store_stack(0, dsp->pc);
		dsp_reg_store_stack(2, loop_pc);
		dsp_reg_store_stack(3, cnt);
	}
	else
		dsp->pc = loop_pc + 1;
	return true;
}

//-------------------------------------------------------------


bool dsp_opc_mrr(uint16 opc)
{
	uint16	val;
	uint8	sreg;
	uint8	dreg;
	sreg = opc & 0x1f;
	dreg = (opc >> 5) & 0x1f;

	val = dsp_op_read_reg(sreg);
	dsp_op_write_reg(dreg, val);

	return true;
}

bool dsp_opc_lrr(uint16 opc)
{
	uint8	sreg;
	uint8	dreg;
	uint16	val;

	sreg = (opc >> 5) & 0x3;
	dreg = opc & 0x1f;

	val = dsp_dmem_read(dsp->r[sreg]);
	dsp_op_write_reg(dreg, val);
	
	// post processing of source reg
	switch((opc >> 7) & 0x3)
	{
	case 0x0:	// LRR
		break;
	case 0x1:	// LRRD
		dsp->r[sreg]--;
		break;
	case 0x2:	// LRRI
		dsp->r[sreg]++;
		break;
	case 0x3:
		dsp->r[sreg] += dsp->r[sreg + 4];
		break;
	}
	return true;
}

bool dsp_opc_srr(uint16 opc)
{
	uint8	sreg;
	uint8	dreg;
	uint16	val;

	sreg = (opc >> 5) & 0x3;
	dreg = opc & 0x1f;

	val = dsp_op_read_reg(dreg);
	dsp_dmem_write(dsp->r[sreg], val);

	// post processing of source reg
	switch((opc >> 7) & 0x3)
	{
	case 0x0:	// SRR
		break;
	case 0x1:	// SRRD
		dsp->r[sreg]--;
		break;
	case 0x2:	// SRRI
		dsp->r[sreg]++;
		break;
	case 0x3:	// SRRX
		dsp->r[sreg] += dsp->r[sreg + 4];
		break;
	}
	return true;
}

bool dsp_opc_ilr(uint16 opc)
{
	uint16	reg;
	uint16	dreg;
	reg = opc & 0x3;
	dreg = 0x1e + ((opc >> 8) & 1);
	// always to acc0 ?
	dsp->r[dreg] = dsp_imem_read(dsp->r[reg]);

	switch((opc >> 2) & 0x3)
	{
	case 0x0:	// no change
		break;
	case 0x2:	// post increment
		dsp->r[reg]++;
		break;
	default:
		return dsp_op_unknown(opc);
	}
	return true;
}

bool dsp_opc_lri(uint16 opc)
{
	uint16	imm;
	uint8	reg;

	reg = opc & DSP_REG_MASK;
	imm = dsp_fetch_code();
	dsp_op_write_reg(reg, imm);
	return true;
}

bool dsp_opc_lris(uint16 opc)
{
	uint16	imm;
	uint8	reg;

	reg = ((opc >> 8) & 0x7) + 0x18;
	imm = (sint8)opc;
	dsp_op_write_reg(reg, imm);
	return true;
}

bool dsp_opc_lr(uint16 opc)
{
	uint16	addr;
	uint8	reg;
	uint16	val;
	reg = opc & DSP_REG_MASK;
	addr = dsp_fetch_code();
	val = dsp_dmem_read(addr);
	dsp_op_write_reg(reg, val);
	return true;
}

bool dsp_opc_sr(uint16 opc)
{
	uint16	addr;
	uint8	reg;
	uint16	val;
	reg = opc & DSP_REG_MASK;
	addr = dsp_fetch_code();
	val = dsp_op_read_reg(reg);
	dsp_dmem_write(addr, val);
	return true;
}

bool dsp_opc_si(uint16 opc)
{
	uint16 imm;
	uint16 addr;
	addr = (sint8)opc;
	imm = dsp_fetch_code();
	dsp_dmem_write(addr, imm);
	return true;
}

bool dsp_opc_tstaxh(uint16 opc)
{
	sint16	val;
	uint8	reg;

	reg = ((opc >> 8) & 0x1) + 0x1a;

	val = (sint16)dsp->r[reg];

	if (val < 0)
		dsp->r[R_SR] |= 0x28;
	if (val > 0)
		dsp->r[R_SR] |= 0x21;
	if (val == 0)
		dsp->r[R_SR] |= 0x25;
	return true;
}



bool dsp_opc_clr(uint16 opc)
{
	uint8	reg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	reg = (opc >> 11) & 0x1;

	dsp->r[0x10 + reg] = dsp->r[0x1c + reg] = dsp->r[0x1e + reg] = 0;
	dsp->r[0x13] = (dsp->r[0x13] & ~0x3f) | 0x24;

	return dsp_opc_ext_epilog(opc);
	
}

bool dsp_opc_clrp(uint16 opc)
{
	if (opc & 0x00ff)
		return dsp_op_unknown(opc);
	dsp->r[0x14] = 0x0000;
	dsp->r[0x15] = 0xfff0;
	dsp->r[0x16] = 0x00ff;
	dsp->r[0x17] = 0x0010;
	return true;
}


#define FLAG_UNK5	5
#define FLAG_S		3
#define FLAG_Z		2
#define FLAG_UNK0	0

void dsp_set_flag(uint8 flag)
{
	dsp->r[R_SR] |= (1 << flag);
}

bool dsp_opc_cmpar(uint16 opc)
{
	sint64	rr;
	sint64  ar;
	uint8	rreg;
	uint8	areg;

	if (opc & 0xff)
		return dsp_op_unknown(opc);
	rreg = ((opc >> 12) & 0x1) + 0x1a;
	areg = (opc >> 11) & 0x1;

	dsp->r[R_SR] &= ~0x3f;

	// we compare 
	rr = (sint16)dsp->r[rreg];
	rr <<= 16;

	ar = dsp_get_long_acc(areg);
	if (ar < rr)
		dsp->r[R_SR] |= 0x28;
	if (ar > rr)
		dsp->r[R_SR] |= 0x21;
	if (ar == rr)
		dsp->r[R_SR] |= 0x25;
	return true;
}

bool dsp_opc_cmpaa(uint16 opc)
{
	sint64	acc0;
	sint64	acc1;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	dsp->r[R_SR] &= ~0x3f;
	dsp_set_flag(FLAG_UNK5);

	acc0 = dsp_get_long_acc(0);
	acc1 = dsp_get_long_acc(1);

	if (acc0 == acc1)
	{
		dsp_set_flag(FLAG_Z);
		dsp_set_flag(FLAG_UNK0);
	}
	if (acc0 < acc1)
		dsp_set_flag(FLAG_S);

	return dsp_opc_ext_epilog(opc);
}


bool dsp_opc_tsta(uint16 opc)
{
	uint8	reg;
	sint64	acc;

	//if ((opc & 0x00ff) != 0) return dsp_op_unknown(opc);
	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	reg = (opc >> 11) & 0x1;
	dsp->r[R_SR] = (dsp->r[R_SR] & ~0x3f) | 0x20;

	acc = dsp_get_long_acc(reg);
	if (acc < 0)
		dsp->r[R_SR] |= 8;
	if (acc == 0)
		dsp->r[R_SR] |= 4;
	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_addaxa(uint16 opc)
{
	uint8	sreg;
	uint8	dreg;
	sint64	acc;
	sint64	acx;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	sreg = (opc >> 9) & 0x1;
	dreg = (opc >> 8) & 0x1;
	acc = dsp_get_long_acc(dreg);
	acx = dsp->r[sreg+0x18];

	acc += acx;

	dsp_set_long_acc(dreg, acc);

	dsp->r[R_SR] = (dsp->r[R_SR] & ~0x3f) | 0x20;

	if (acc < 0)
		dsp->r[R_SR] |= 8;
	if (acc == 0)
		dsp->r[R_SR] |= 4;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_mvaxa(uint16 opc)
{
	uint8	sreg;
	uint8	dreg;

	//if ((opc & 0x00ff) != 0) return dsp_op_unknown(opc);
	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	sreg = (opc >> 9) & 0x1;
	dreg = (opc >> 8) & 0x1;

	dsp->r[0x1c+dreg] = dsp->r[0x18+sreg];
	dsp->r[0x1e+dreg] = dsp->r[0x1a+sreg];
	if ((sint16)dsp->r[0x1a+sreg] < 0)
		dsp->r[0x10+dreg] = 0xffff;
	else
		dsp->r[0x10+dreg] = 0;

	dsp_opc_tsta(dreg << 11);

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_andr(uint16 opc)
{
	uint8	sreg;
	uint8	dreg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	sreg = (opc >> 9) & 0x1;
	dreg = (opc >> 8) & 0x1;

	dsp->r[0x1e+dreg] &= dsp->r[0x1a+sreg];

	dsp_opc_tsta(dreg << 11);

	return dsp_opc_ext_epilog(opc);
}
//-------------------------------------------------------------

bool dsp_opc_nx(uint16 opc)
{
	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);
	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_andfc(uint16 opc)
{
	uint8	reg;
	uint16	imm;
	uint16	val;

	if (opc & 0xf) 
		return dsp_op_unknown(opc);
	reg = 0x1e + ((opc >> 8) & 0x1);
	imm = dsp_fetch_code();
	val	= dsp->r[reg];
	if ((val & imm) == imm)
		dsp->r[R_SR] |= 0x40;
	else
		dsp->r[R_SR] &= ~0x40;
	return true;
}

bool dsp_opc_andf(uint16 opc)
{
	uint8	reg;
	uint16	imm;
	uint16	val;

	if (opc & 0xf) 
		return dsp_op_unknown(opc);
	reg = 0x1e + ((opc >> 8) & 0x1);
	imm = dsp_fetch_code();
	val	= dsp->r[reg];
	if ((val & imm) == 0)
		dsp->r[R_SR] |= 0x40;
	else
		dsp->r[R_SR] &= ~0x40;
	return true;
}

bool dsp_opc_subf(uint16 opc)
{
	uint8	reg;
	sint64	imm;
	sint64	val;
	sint64	res;

	if (opc & 0xf) 
		return dsp_op_unknown(opc);
	reg = 0x1e + ((opc >> 8) & 0x1);
	imm = (sint16)dsp_fetch_code();

	dsp->r[R_SR] &= ~0x3f;
	val = (sint16)dsp->r[reg];
	res = val - imm;

	if (res == 0)
		dsp->r[R_SR] |= 0x25;

	if (res > 0)
		dsp->r[R_SR] |= 0x21;

	if (res < 0)
		dsp->r[R_SR] |= 0x28;
	return true;
}


bool dsp_opc_andi(uint16 opc)
{
	uint8	reg;
	uint16	imm;
	uint16	val;
	uint16	res;

	if (opc & 0xf) 
		return dsp_op_unknown(opc);


	reg = 0x1e + ((opc >> 8) & 0x1);
	imm = dsp_fetch_code();
	val	= dsp->r[reg];

	dsp->r[R_SR] &= ~0x3f;
	dsp_set_flag(FLAG_UNK5);

	res = val & imm;

	dsp->r[reg] = res;

	if (res == 0)
		dsp_set_flag(FLAG_Z);
	return true;
}


bool dsp_opc_ori(uint16 opc)
{
	uint8	reg;
	uint16	imm;
	uint16	val;
	uint16	res;

	if (opc & 0xf) 
		return dsp_op_unknown(opc);

	reg = 0x1e + ((opc >> 8) & 0x1);
	imm = dsp_fetch_code();
	val	= dsp->r[reg];

	dsp->r[R_SR] &= ~0x3f;
	dsp_set_flag(FLAG_UNK5);

	res = val | imm;

	dsp->r[reg] = res;

	if (res == 0)
		dsp_set_flag(FLAG_Z);

	return true;
}

//-------------------------------------------------------------

bool dsp_opc_addaa(uint16 opc)
{
	sint64	res;
	sint64	acc0;
	sint64	acc1;
	uint8	areg;
	

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;

	dsp->r[R_SR] &= ~0x3f;

	acc0 = dsp_get_long_acc(0);
	acc1 = dsp_get_long_acc(1);

	res = acc0 + acc1;

	dsp_set_long_acc(areg, res);

	if (res == 0)
		dsp->r[R_SR] |= 0x25;

	if (res > 0)
		dsp->r[R_SR] |= 0x21;
	
	if (res < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_subisf(uint16 opc)
{
	uint8	areg;
	sint64	acc;
	sint64	val;
	sint64	res;

	areg = (opc >> 8) & 0x1;
	acc = dsp_get_long_acc(areg);
	val = (sint8)opc;
	val <<= 16;

	dsp->r[R_SR] &= ~0x3f;

	res = acc - val;

	if (res == 0)
		dsp->r[R_SR] |= 0x25;

	if (res > 0)
		dsp->r[R_SR] |= 0x21;

	if (res < 0)
		dsp->r[R_SR] |= 0x28;
	return true;
}

bool dsp_opc_addpaxz(uint16 opc)
{
	sint64	prod;
	sint64	acc;
	sint64	ax;
	uint8	areg;
	uint8	sreg;

	areg = (opc >> 8) & 0x1;
	sreg = ((opc >> 9) & 0x1);

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	dsp_set_long_acc(areg, 0);
	
	if (!dsp_opc_ext_epilog(opc))
		return dsp_op_unknown(opc);

	dsp->r[R_SR] &= ~0x3f;

	prod = dsp_get_long_prod();
	ax	 = dsp_get_long_acx(sreg);

	acc = prod + ax;
	acc &= ~0x0ffff;
	dsp_set_long_acc(areg, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return true;
}

bool dsp_opc_decm(uint16 opc)
{
	sint64 acc;
	sint64 sub;

	if (opc & 0x00ff)
		return dsp_op_unknown(opc);

	dsp->r[R_SR] &= ~0x3f;

	sub = 0x10000;
	acc = dsp_get_long_acc(0);
	acc -= sub;
	dsp_set_long_acc(0, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return true;
}


bool dsp_opc_inc(uint16 opc)
{
	sint64 acc;

	if (opc & 0x00ff)
		return dsp_op_unknown(opc);

	dsp->r[R_SR] &= ~0x3f;

	acc = dsp_get_long_acc(0);
	acc++;
	dsp_set_long_acc(0, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return true;
}

bool dsp_opc_neg(uint16 opc)
{
	sint64	acc;
	uint8	areg;

//	if (opc & 0x00ff)		return dsp_op_unknown(opc);
	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;

	acc = dsp_get_long_acc(areg);
	acc = 0 - acc;
	dsp_set_long_acc(areg, acc);

	dsp->r[R_SR] &= ~0x3f;

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_addax(uint16 opc)
{
	sint64	acc;
	sint64	ax;
	uint8	areg;
	uint	sreg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;
	sreg = ((opc >> 9) & 0x1) + 0x18;
	ax = (sint16)dsp->r[sreg + 2];
	ax <<= 16;
	ax |= dsp->r[sreg];

	acc = dsp_get_long_acc(areg);
	acc += ax;
	dsp_set_long_acc(areg, acc);


	dsp->r[R_SR] &= ~0x3f;
	if (acc == 0)
		dsp->r[R_SR] |= 0x25;
	if (acc > 0)
		dsp->r[R_SR] |= 0x21;
	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_addar(uint16 opc)
{
	sint64	acc;
	sint64	ax;
	uint8	areg;
	uint	sreg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;
	sreg = ((opc >> 9) & 0x3) + 0x18;
	
	ax = (sint16)dsp->r[sreg];
	ax <<= 16;

	acc = dsp_get_long_acc(areg);
	acc += ax;
	dsp_set_long_acc(areg, acc);


	dsp->r[R_SR] &= ~0x3f;
	if (acc == 0)
		dsp->r[R_SR] |= 0x25;
	if (acc > 0)
		dsp->r[R_SR] |= 0x21;
	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_subar(uint16 opc)
{
	sint64	acc;
	sint64	ax;
	uint8	areg;
	uint	sreg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;
	sreg = ((opc >> 9) & 0x3) + 0x18;

	ax = (sint16)dsp->r[sreg];
	ax <<= 16;

	acc = dsp_get_long_acc(areg);
	acc -= ax;
	dsp_set_long_acc(areg, acc);


	dsp->r[R_SR] &= ~0x3f;
	if (acc == 0)
		dsp->r[R_SR] |= 0x25;
	if (acc > 0)
		dsp->r[R_SR] |= 0x21;
	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_addis(uint16 opc)
{
	sint64	acc;
	sint64	sub;
	uint8	areg;

	areg = (opc >> 8) & 0x1;

	dsp->r[R_SR] &= ~0x3f;

	sub = (sint8)opc;
	sub <<= 16;
	acc = dsp_get_long_acc(areg);
	acc += sub;
	dsp_set_long_acc(areg, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return true;
}

bool dsp_opc_addi(uint16 opc)
{
	sint64	acc;
	sint64	sub;
	uint8	areg;

	if (opc & 0x00ff)
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;
	dsp->r[R_SR] &= ~0x3f;


	sub = (sint16)dsp_fetch_code();
	sub <<= 16;
	acc = dsp_get_long_acc(areg);
	acc += sub;
	dsp_set_long_acc(areg, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return true;
}

bool dsp_opc_lsl16(uint16 opc)
{
	uint8	areg;
	sint64	acc;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 8) & 0x1;

	acc = dsp_get_long_acc(areg);
	acc <<= 16;
	dsp_set_long_acc(areg, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_asr16(uint16 opc)
{
	uint8	areg;
	sint64	acc;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = (opc >> 11) & 0x1;

	acc = dsp_get_long_acc(areg);
	acc >>= 16;
	dsp_set_long_acc(areg, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return dsp_opc_ext_epilog(opc);
}


bool dsp_opc_shifti(uint16 opc)
{
	sint16	shift;
	uint8	areg;
	sint64	acc;
	uint64	uacc;
	uint8	dir;

	// direction: left
	dir = 0;
	shift = (opc & 0x7f) << 9;
	shift >>= 8;
	if (shift < 0) 
	{
		dir = 1;
		shift = 0 - shift;
	}
	
	areg = (opc >> 8) & 0x1;

	if (opc & 0x80)
	{
		// arithmetic shift
		uacc = dsp_get_long_acc(areg);
		if (dir == 1)
			uacc >>= shift;
		else
			uacc <<= shift;
		acc = uacc;
	}
	else
	{
		acc = dsp_get_long_acc(areg);
		if (dir == 1)
			acc >>= shift;
		else
			acc <<= shift;
	}

	dsp_set_long_acc(areg, acc);

	if (acc == 0)
		dsp->r[R_SR] |= 0x25;

	if (acc > 0)
		dsp->r[R_SR] |= 0x21;

	if (acc < 0)
		dsp->r[R_SR] |= 0x28;

	return true;
}



//-------------------------------------------------------------

bool dsp_opc_dar(uint16 opc)
{
	uint8	reg;

	reg = opc & 0x3;
	dsp->r[reg]--;
	return true;
}
//-------------------------------------------------------------

bool dsp_opc_sbclr(uint16 opc)
{
	uint8 bit;
	bit = (opc & 0xff) + 6;
	dsp->r[R_SR] &= ~(1 << bit);
	return true;
}

bool dsp_opc_sbset(uint16 opc)
{
	uint8 bit;
	bit = (opc & 0xff) + 6;
	dsp->r[R_SR] |= (1 << bit);
	return true;
}

bool dsp_opc_srbith(uint16 opc)
{
	//if ((opc & 0x00ff) != 0) return dsp_op_unknown(opc);
	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	switch((opc>>8) & 0xf)
	{
	case 0xe:	// SET40
		dsp->r[R_SR] &= ~(1 << 14);
		break;
	case 0xf:	// SET16
		dsp->r[R_SR] |= (1 << 14);
		break;
	default:
		break;
	}

	return dsp_opc_ext_epilog(opc);
}
//-------------------------------------------------------------

bool dsp_opc_mvp(uint16 opc)
{
	uint8	rreg;
	sint64	prod;
	sint64	acc;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	rreg = (opc >> 8) & 0x1;

	prod = dsp_get_long_prod();
	acc = prod;
	dsp_set_long_acc(rreg, acc);

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_mul(uint16 opc)
{
	sint64	prod;
	sint64	val1;
	sint64	val2;
	uint8	areg;
	uint8	breg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = ((opc >> 11) & 0x1) + 0x18;
	breg = ((opc >> 11) & 0x1) + 0x1a;

	val1 = (sint16)dsp->r[areg];
	val2 = (sint16)dsp->r[breg];

	prod = val1 * val2;

	dsp_set_long_prod(prod);

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_mulmv(uint16 opc)
{
	sint64	prod;
	sint64	val1;
	sint64	val2;
	uint8	areg;
	uint8	breg;
	uint8	rreg;
	sint64	acc;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = ((opc >> 11) & 0x1) + 0x18;
	breg = ((opc >> 11) & 0x1) + 0x1a;
	rreg = (opc >> 8) & 0x1;

	val1 = (sint16)dsp->r[areg];
	val2 = (sint16)dsp->r[breg];

	prod = dsp_get_long_prod();
	acc = prod;
	dsp_set_long_acc(rreg, acc);

	prod = val1 * val2;

	dsp_set_long_prod(prod);

	return dsp_opc_ext_epilog(opc);
}


bool dsp_opc_mulx(uint16 opc)
{
	sint64	prod;
	sint64	val1;
	sint64	val2;
	uint8	areg;
	uint8	breg;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = ((opc >> 11) & 0x2) + 0x18;
	breg = ((opc >> 10) & 0x2) + 0x19;

	val1 = (sint16)dsp->r[areg];
	val2 = (sint16)dsp->r[breg];

	prod = val1 * val2;

	dsp_set_long_prod(prod);

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_mulxac(uint16 opc)
{
	sint64	prod;
	sint64	val1;
	sint64	val2;
	uint8	areg;
	uint8	breg;
	uint8	rreg;
	sint64	acc;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = ((opc >> 11) & 0x2) + 0x18;
	breg = ((opc >> 10) & 0x2) + 0x19;
	rreg = (opc >> 8) & 0x1;

	val1 = (sint16)dsp->r[areg];
	val2 = (sint16)dsp->r[breg];
	
	prod = dsp_get_long_prod();
	acc = dsp_get_long_acc(rreg);
	acc += prod;
	dsp_set_long_acc(rreg, acc);

	prod = val1 * val2;

	dsp_set_long_prod(prod);

	return dsp_opc_ext_epilog(opc);
}

bool dsp_opc_mulxmv(uint16 opc)
{
	sint64	prod;
	sint64	val1;
	sint64	val2;
	uint8	areg;
	uint8	breg;
	uint8	rreg;
	sint64	acc;

	if (!dsp_opc_ext_prolog(opc))
		return dsp_op_unknown(opc);

	areg = ((opc >> 11) & 0x2) + 0x18;
	breg = ((opc >> 10) & 0x2) + 0x19;
	rreg = (opc >> 8) & 0x1;

	val1 = (sint16)dsp->r[areg];
	val2 = (sint16)dsp->r[breg];

	prod = dsp_get_long_prod();
	acc = prod;
	dsp_set_long_acc(rreg, acc);

	prod = val1 * val2;

	dsp_set_long_prod(prod);

	return dsp_opc_ext_epilog(opc);
}

//-------------------------------------------------------------
bool dsp_op0(uint16 opc)
{
	if (opc == 0) return true;
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
		switch((opc >> 4) & 0xf)
		{
		case 0x0:
			switch(opc & 0xf)
			{
			case 0x4:
			case 0x5:
			case 0x6:
			case 0x7:
				return dsp_opc_dar(opc);
			default:
				break;
			}
			break;
		case 0x2:	// HALT
			return dsp_opc_halt(opc);
		case 0x4:	// LOOP
		case 0x5:	// LOOP
			return dsp_opc_loop(opc);
		case 0x6:	// BLOOP
		case 0x7:	// BLOOP
			return dsp_opc_bloop(opc);
		case 0x8:	// LRI
		case 0x9:	// LRI
			return dsp_opc_lri(opc);
		case 0xC:	// LR
		case 0xD:	// LR
			return dsp_opc_lr(opc);
		case 0xE:	// SR
		case 0xF:	// SR
			return dsp_opc_sr(opc);
		default:
			break;
		}
		break;
	case 0x2:
		switch((opc >> 4)& 0xf)
		{
		case 0x0:	// ADDI
			return dsp_opc_addi(opc);
		case 0x1:	// IL
			return dsp_opc_ilr(opc);
		case 0x4:	// ANDI
			return dsp_opc_andi(opc);
		case 0x6:	// ORI
			return dsp_opc_ori(opc);
		case 0x8:	// SUBF
			return dsp_opc_subf(opc);
		case 0x9:	// Jxx
			return dsp_opc_jxx(opc);
		case 0xa:	// ANDF
			return dsp_opc_andf(opc);
		case 0xb:	// CALL
			return dsp_opc_call(opc);
		case 0xc:
			return dsp_opc_andfc(opc);
		case 0xd:	// RET
			return dsp_opc_ret(opc);
		case 0xf:	// RTI
			return dsp_opc_rti(opc);
		default:
			break;
		}
		break;
	case 0x3:
		switch((opc >> 4)& 0xf)
		{
		case 0x0:	// ADDAI
			return dsp_opc_addi(opc);
		case 0x1:	// ILR
			return dsp_opc_ilr(opc);
		case 0x4:	// ANDI
			return dsp_opc_andi(opc);
		case 0x6:	// ORI
			return dsp_opc_ori(opc);
		case 0x8:	// SUBF
			return dsp_opc_subf(opc);
		case 0xa:	// ANDF
			return dsp_opc_andf(opc);
		case 0xc:	// ANDFC
			return dsp_opc_andfc(opc);
		default:
			break;
		}
		break;
	case 0x4:
	case 0x5:
		return dsp_opc_addis(opc);
	case 0x6:	// SUBISF
	case 0x7:
		return dsp_opc_subisf(opc);
	case 0x8:	// LRIS
	case 0x9:
	case 0xa:
	case 0xb:
	case 0xc:
	case 0xd:
	case 0xe:
	case 0xf:
		return dsp_opc_lris(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op1(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
		return dsp_opc_loopi(opc);
	case 0x1:	// BLOOPI
		return dsp_opc_bloopi(opc);
	case 0x2:	// SBCLR
		return dsp_opc_sbclr(opc);
	case 0x3:	// SBSET
		return dsp_opc_sbset(opc);
	case 0x4:	// shifti
	case 0x5:
		return dsp_opc_shifti(opc);
	case 0x6:	// SI
		return dsp_opc_si(opc);
	case 0x7:	// JMPA/CALLA
		return	dsp_opc_jmpa(opc);
	case 0x8:	// LRRx
	case 0x9:	// LRRx
		return dsp_opc_lrr(opc);
	case 0xa:	// SRRx
	case 0xb:	// SRRx
		return dsp_opc_srr(opc);
	case 0xc:	// MRR
	case 0xd:	// MRR
	case 0xe:	// MRR
	case 0xf:	// MRR
		return dsp_opc_mrr(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op2(uint16 opc)
{
	// lrs, srs
	uint8 reg;
	uint16 addr;

	reg = ((opc >> 8) & 0x7) + 0x18;
	addr = (sint8) opc;
	if (opc & 0x0800)
	{
		// srs
		dsp_dmem_write(addr, dsp->r[reg]);
	}
	else
	{
		// lrs
		dsp->r[reg] = dsp_dmem_read(addr);
	}
	return true;
}

bool dsp_op3(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x4:
	case 0x5:
		return dsp_opc_andr(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op4(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
	case 0x1:
	case 0x2:
	case 0x3:
	case 0x4:
	case 0x5:
	case 0x6:
	case 0x7:
		return dsp_opc_addar(opc);
	case 0x8:
	case 0x9:
	case 0xa:
	case 0xb:
		return	dsp_opc_addax(opc);
	case 0xc:
	case 0xd:
		return dsp_opc_addaa(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op5(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
	case 0x1:
	case 0x2:
	case 0x3:
	case 0x4:
	case 0x5:
	case 0x6:
	case 0x7:
		return dsp_opc_subar(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op6(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x8:	// MVAXA
	case 0x9:
	case 0xa:
	case 0xb:
		return dsp_opc_mvaxa(opc);
	case 0xe:
	case 0xf:
		return dsp_opc_mvp(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op7(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
	case 0x1:
	case 0x2:
	case 0x3:
		return dsp_opc_addaxa(opc);
	case 0x6:
	case 0x7:
		return dsp_opc_inc(opc);
	case 0x8:
	case 0x9:
		return dsp_opc_decm(opc);
	case 0xc:
	case 0xd:
		return dsp_opc_neg(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op8(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
		return dsp_opc_nx(opc);
	case 0x2:		// CMPAA
		return dsp_opc_cmpaa(opc);
	case 0x4:		// CLRP
		return dsp_opc_clrp(opc);
	case 0x1:		// CLR 0
	case 0x9:		// CLR	1
		return dsp_opc_clr(opc);
	case 0x6:
	case 0x7:
		return dsp_opc_tstaxh(opc);
	case 0xc:
	case 0xb:
	case 0xe:		// SET40
	case 0xd:
	case 0xa:
	case 0xf:
		return dsp_opc_srbith(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_op9(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x6:
	case 0x7:
	case 0xe:
	case 0xf:
		return dsp_opc_mulmv(opc);
	case 0x0:
	case 0x8:
		return dsp_opc_mul(opc);
	case 0x1:
	case 0x9:
		return dsp_opc_asr16(opc);
	default:
		break;
}
	return dsp_op_unknown(opc);
}

bool dsp_opa(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x1:
	case 0x9:
		return dsp_opc_tsta(opc);
	case 0x4:
	case 0x5:
	case 0xc:
	case 0xd:
		return dsp_opc_mulxac(opc);
	case 0x6:
	case 0x7:
		return dsp_opc_mulxmv(opc);
	case 0x0:
	case 0x8:
		return dsp_opc_mulx(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_opb(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
	case 0x8:
		return dsp_opc_mulx(opc);

	case 0x4:
	case 0x5:
	case 0xc:
	case 0xd:
		return dsp_opc_mulxac(opc);

	case 0x1:	// TSTA
	case 0x9:	// TSTA
		return dsp_opc_tsta(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_opc(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x1:	// CMPAR
		return dsp_opc_cmpar(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}

bool dsp_opd(uint16 opc)
{
	return dsp_op_unknown(opc);
}

bool dsp_ope(uint16 opc)
{
	return dsp_op_unknown(opc);
}

bool dsp_opf(uint16 opc)
{
	switch((opc >> 8) & 0xf)
	{
	case 0x0:
	case 0x1:
		return dsp_opc_lsl16(opc);
	case 0x8:	// ADDALP
	case 0x9:	// ADDALP
	case 0xa:	// ADDALP
	case 0xb:	// ADDALP
		return dsp_opc_addpaxz(opc);
	default:
		break;
	}
	return dsp_op_unknown(opc);
}
