/*====================================================================

filename:     gdsp_interface.h
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "config.h"
#include <stdio.h>
#include <windows.h>
#include "dtypes.h"
#include "gdsp_interpreter.h"
#include "gdsp_interface.h"

#define DSP_DSMAH	0xce
#define DSP_DSMAL	0xcf
#define DSP_DSCR	0xc9
#define DSP_DSPA	0xcd
#define DSP_DSBL	0xcb
#define DSP_ACSAH	0xd4
#define DSP_ACSAL	0xd5
#define DSP_ACEAH	0xd6
#define DSP_ACEAL	0xd7
#define DSP_ACCAH	0xd8
#define DSP_ACCAL	0xd9

char *reg_names[] = 
{
	// c0
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	NULL,	"DSCR",	NULL,	"DSBL",	NULL,	"DSPA",	"DSMAH","DSMAL",
	// d0
	NULL,	NULL,	NULL,	NULL,	"ACSAH","ACSAL", "ACEAH", "ACEAL",
	"ACCAH", "ACCAL",NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	// e0
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	"AMDM",
	// f0
	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,	NULL,
	NULL,	NULL,	NULL,	NULL,	"DMBH", "DMBL", "CMBH", "CMBL",
};


void gdsp_dma(void);
extern CRITICAL_SECTION xcs;

static volatile uint16	gdsp_mbox[2][2];

uint16	gdsp_ifx_regs[256];

void gdsp_ifx_init(void)
{
	int i;
	for(i = 0 ; i < 256 ; i++)
		gdsp_ifx_regs[i] = 0;
	gdsp_mbox[0][0] = 0;
	gdsp_mbox[0][1] = 0;
	gdsp_mbox[1][0] = 0;
	gdsp_mbox[1][1] = 0;
}

uint32 gdsp_mbox_peek(uint8 mbx)
{
	return (gdsp_mbox[mbx][0] << 16) | gdsp_mbox[mbx][1];
}
void gdsp_mbox_write_h(uint8 mbx, uint16 val)
{
#if WITH_DSP_ON_THREAD
	EnterCriticalSection(&xcs);
#endif
	gdsp_mbox[mbx][0] = val & 0x7fff;
#if WITH_DSP_ON_THREAD
	LeaveCriticalSection(&xcs);
	Sleep(0);
#endif
}

void gdsp_mbox_write_l(uint8 mbx, uint16 val)
{
#if WITH_DSP_ON_THREAD
	EnterCriticalSection(&xcs);
#endif
	gdsp_mbox[mbx][1] = val;
	gdsp_mbox[mbx][0] |= 0x8000;
#if WITH_DSP_ON_THREAD
	LeaveCriticalSection(&xcs);
	Sleep(0);
#endif
}

uint16 gdsp_mbox_read_h(uint8 mbx)
{
#if WITH_DSP_ON_THREAD
	Sleep(0);
#endif
	return gdsp_mbox[mbx][0];
}

uint16 gdsp_mbox_read_l(uint8 mbx)
{
	uint16	val;
#if WITH_DSP_ON_THREAD
	Sleep(0);
	EnterCriticalSection(&xcs);
#endif
	val = gdsp_mbox[mbx][1];
	gdsp_mbox[mbx][0] &= ~0x8000;
#if WITH_DSP_ON_THREAD
	LeaveCriticalSection(&xcs);
#endif
	//fprintf(stderr, "MBOX emptied\n");
	return val;
}

void gdsp_ifx_write(uint16 addr, uint16 val)
{
	//Sleep(0);
	addr &= 0xff;
	switch(addr & 0xff)
	{
	case 0xfb:	// DIRQ
		if (val & 0x1)
			dsp->irq_request();
		break;
	case 0xfc:	// DMBH
		gdsp_mbox_write_h(GDSP_MBOX_DSP, val);
		break;
	case 0xfd:	// DMBL
		gdsp_mbox_write_l(GDSP_MBOX_DSP, val);
		break;
	case 0xcb:	// DSBL
		gdsp_ifx_regs[addr] = val;
		gdsp_dma();
		gdsp_ifx_regs[DSP_DSCR] &= ~0x0004;
		break;
	case 0xcd:
	case 0xce:
	case 0xcf:
	case 0xc9:
		gdsp_ifx_regs[addr] = val;
		break;
	default:
		if ((addr & 0xff) >= 0xc0 && reg_names[addr & 0x3f])
			printf("%04x MW %s (%04x)\n", dsp->pc, reg_names[addr & 0x3f], val);
		else
			printf("%04x MW %04x (%04x)\n", dsp->pc, addr, val);
		gdsp_ifx_regs[addr] = val;
		break;
	}
}

extern uint8 *aram;
void gdsp_exception(uint8 level);

uint16 dsp_read_aram(void)
{
	uint16	*aram16;
	uint32	addr;
	uint32	eaddr;
	uint16	val;

	aram16 = (uint16 *)aram;
	addr = (gdsp_ifx_regs[DSP_ACCAH] << 16) | gdsp_ifx_regs[DSP_ACCAL];
	val = aram16[addr];
	addr++;
	eaddr = (gdsp_ifx_regs[DSP_ACEAH] << 16) | gdsp_ifx_regs[DSP_ACEAL];

	if (addr > eaddr)
	{
		addr = (gdsp_ifx_regs[DSP_ACSAH] << 16) | gdsp_ifx_regs[DSP_ACSAL];
		gdsp_exception(5);
	}

	gdsp_ifx_regs[DSP_ACCAH] = addr >> 16;
	gdsp_ifx_regs[DSP_ACCAL] = addr & 0xffff;
	return val;
}


uint16 gdsp_ifx_read(uint16 addr)
{
	uint16 val;
	//Sleep(0);
	addr &= 0xff;
	//Sleep(0);
	switch(addr & 0xff)
	{
	case 0xfc:	// DMBH
		val = gdsp_mbox_read_h(GDSP_MBOX_DSP);
		break;
	case 0xfe:	// CMBH
		val = gdsp_mbox_read_h(GDSP_MBOX_CPU);
		break;
	case 0xff:	// CMBL
		val = gdsp_mbox_read_l(GDSP_MBOX_CPU);
		break;
	case 0xc9:
		val = gdsp_ifx_regs[addr];
		break;
//	case 0xda:
//	case 0xdb:
//	case 0xdc:
	case 0xdd:
		val = dsp_read_aram();
		break;
	default:
		val = gdsp_ifx_regs[addr];
		if ((addr & 0xff) >= 0xc0 && reg_names[addr & 0x3f])
			printf("%04x MR %s (%04x)\n", dsp->pc, reg_names[addr & 0x3f], val);
		else
			printf("%04x MR %04x (%04x)\n", dsp->pc, addr, val);
		break;
	}
	return val;
}

#define DUMP_DSP_IMEM	1
uint32 dsp_dump_code(uint32 addr, uint32 len);

void gdsp_idma_in(uint16 dsp_addr, uint32 addr, uint32 size)
{
	uint32	i;
	uint8	*src;
	uint8	*dst;

#if DUMP_DSP_IMEM
	dsp_dump_code(addr, size);
#endif // DUMP_DSP_IMEM

	src = dsp->cpu_ram + (addr & 0x0fffffff);
	dst = ((uint8 *)dsp->iram) + dsp_addr;
	for(i = 0 ; i < size ; i++)
		*dst++ = *src++;
}

void gdsp_idma_out(uint16 dsp_addr, uint32 addr, uint32 size)
{
	uint32	i;
	uint8	*src;
	uint8	*dst;

	dst = dsp->cpu_ram + (addr & 0x0fffffff);
	src = ((uint8 *)dsp->iram) + dsp_addr;
	for(i = 0 ; i < size ; i++)
		*dst++ = *src++;
}

void gdsp_ddma_in(uint16 dsp_addr, uint32 addr, uint32 size)
{
	uint32	i;
	uint8	*src;
	uint8	*dst;

	src = dsp->cpu_ram + (addr & 0x0fffffff);
	dst = ((uint8 *)dsp->dram) + dsp_addr;
	for(i = 0 ; i < size ; i++)
		*dst++ = *src++;
}

void gdsp_ddma_out(uint16 dsp_addr, uint32 addr, uint32 size)
{
	uint32	i;
	uint8	*src;
	uint8	*dst;

	dst = dsp->cpu_ram + (addr & 0x0fffffff);
	src = ((uint8 *)dsp->dram) + dsp_addr;
	for(i = 0 ; i < size ; i++)
		*dst++ = *src++;
}

#define	DSP_CR_IMEM		(2)
#define	DSP_CR_DMEM		(0)
#define	DSP_CR_TO_CPU	(1)
#define	DSP_CR_FROM_CPU	(0)

#include <stdio.h>

void gdsp_dma(void)
{
	uint16	ctl;
	uint32	addr;
	uint16	dsp_addr;
	uint16	len;

	addr = (gdsp_ifx_regs[DSP_DSMAH] << 16) | gdsp_ifx_regs[DSP_DSMAL];
	ctl = gdsp_ifx_regs[DSP_DSCR];
	dsp_addr = gdsp_ifx_regs[DSP_DSPA] * 2;
	len = gdsp_ifx_regs[DSP_DSBL];

	printf("%04x DSP-DMA: A:%08x D:%04x L:%04x C:%04x\n", dsp->pc, addr, dsp_addr, len, ctl);
	fflush(stdout);
	if (ctl > 3 || len > 0x4000 )
	{
		printf("DMA ERROR pc: %04x ctl: %04x addr: %08x da: %04x size: %04x\n", dsp->pc, ctl, addr, dsp_addr, len);
		fflush(stdout);
		exit(0);
	}
/*
	if (len > 0x2000)
	{
		fprintf(stderr, "DMA WRONG SIZE\n");
//		return;

	}
*/
	//if (!(addr & 0x80000000)) fprintf(stderr, "%04x DSP-DMA: A:%08x D:%04x L:%04x C:%04x\n", dsp->pc, addr, dsp_addr, len, ctl);
	switch(ctl & 0x3)
	{
	case (DSP_CR_DMEM | DSP_CR_TO_CPU):
		gdsp_ddma_out(dsp_addr, addr, len);
		break;
	case (DSP_CR_DMEM | DSP_CR_FROM_CPU):
		gdsp_ddma_in(dsp_addr, addr, len);
		break;
	case (DSP_CR_IMEM | DSP_CR_TO_CPU):
		gdsp_idma_out(dsp_addr, addr, len);
		break;
	case (DSP_CR_IMEM | DSP_CR_FROM_CPU):
		gdsp_idma_in(dsp_addr, addr, len);
		break;
	}

}
