/*====================================================================

filename:     wnd_debugger.cpp
project:      GCemu
created:      2004-6-18
mail:		  duddie@walla.com

Copyright (c) 2005 Duddie & Tratax

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

====================================================================*/
#include "config.h"

#include <windows.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "system/types.h"
#include "gdsp_tool.h"
#include "dsp/gdsp_interpreter.h"
#include "dsp/gdsp_interface.h"

#include "w32_display.h"
#include "ppc_disasm.h"
#include "cpu/trx_ppc_cpu.h"
#include "hardware/memory_interface.h"
#include "tracers.h"

#define REG_WIN_OFFSETY 2
#define CODE_WIN_OFFSETY 20
#define CODE_WIN_LINES 16
#define CODE_WIN_MIDDLE 8

char *dbg_helpstrings[]= 
{
	"F1: Display this window",
	"F2: switch between register views",
	"F3: switch between code/data/log",
	"F5: run",
	"F6: stepover",
	"F7: step",
	"F9: toggle breakpoint",
	"F12: io core trace on/off",
	"SPACE: update code window with current PC",
	"UP: scroll up in code/data window",
	"DOWN: scroll down in code/data window",
	"PGUP: page up in code/data window",
	"PGDOWN: page down in code/data window",
	"HOME: 0x1000 up in code/data window",
	"END: 0x1000 down in code/data window",
	"TAB: switch between CPU and DSP",
	"x: stop CPU (when running)",
	"--------------------------",
	"Beer is proof that god wants mankind to be happy",
	0
};

uint32 current_tab = 0;
char *tab_names[]=
{
	" CPU Window ",
	" DSP Window ",
	0
};

uint32 reg_win_page = 0;
char *reg_win_page_names[]=
{
	" Integer ",
	" FPU ",
	" Paired Single ",
	0
};

uint32 codedata_win_page = 0;
char *codedata_win_page_names[]=
{
	" Disassembly ",
	" Memory view ",
	" Log toggles ",
	0
};

uint32 code_win_startaddress = 0, code_win_startoffset = 0;
uint32 data_win_startaddress = 0, data_win_startoffset = 0;
uint32 log_win_startindex = 0, log_win_startoffset = 0;

uint16 pos_dsp_x = 0;
uint16 pos_dsp_y = 1;

volatile uint32	dbg_state;

extern gd_globals_t *gdg;

enum
{
	DBG_WAIT = 0,
	DBG_RUN,
	DBG_STEP,
	DBG_STEPOVER,
	DBG_EXIT,
	DBG_DSP_STEP
};

extern uint32	dbg_breakpoint;

typedef struct dbg_t 
{
	sint32	font_height;

	// font size x
	sint32	fsx;
	// font size y
	sint32	fsy;

	uint32	ofx;
	uint32	ofy;

	// console buffer
	char	*cbuf;
	// console size x
	sint32	cx;
	// console size y
	sint32	cy;

	bool	size_updated;

	HDC		hdc;
	HFONT	hfont;
} dbg_t;

static dbg_t *dbg;

void dbg_init_font(void)
{
	LOGFONT	lf;
	memset( &lf, 0, sizeof(lf) );
	strcpy( (LPSTR)&lf.lfFaceName, "Courier New" );//MS Gothic" );
	//strcpy( (LPSTR)&lf.lfFaceName, "Arial" );//MS Gothic" );
	lf.lfCharSet = DEFAULT_CHARSET;
	//lf.lfCharSet = SHIFTJIS_CHARSET;
	lf.lfWidth = 0;
	lf.lfHeight = dbg->font_height;//-MulDiv(6, GetDeviceCaps(dbg->hdc, LOGPIXELSY), 16);
	lf.lfWeight = FW_DONTCARE;
	dbg->hfont = CreateFontIndirect( &lf );

}

void dbg_select_font(void)
{
	SelectObject(dbg->hdc, dbg->hfont);
}

void dbg_set_bk_color(COLORREF col)
{
	SetBkColor(dbg->hdc, col);
}

void dbg_set_color(COLORREF col)
{
	SetTextColor(dbg->hdc, col);
}

void dbg_set_font_size(void)
{
	char	buff[256];
	SIZE	sz;
	uint32	i;

	dbg->fsx = dbg->fsy = 0;
	for (i=32; i <= 126; ++i)
	{
		sprintf( buff, "%c", i );
		GetTextExtentPoint32(dbg->hdc, buff, 1, &sz);
		if ( sz.cy > dbg->fsy )
			dbg->fsy = sz.cy;
		if ( sz.cx > dbg->fsx )
			dbg->fsx = sz.cx;
	}
}

void dbg_print(int x, int y, char *text)
{
	int i;
	for(i = 0 ; i < (int)strlen(text) && (x + i) < dbg->cx; i++)
		if (text[i]) dbg->cbuf[y * dbg->cy + x + i] = text[i];
	TextOut(dbg->hdc, x * dbg->fsx + dbg->ofx, y * dbg->fsy + dbg->ofy, text, (int)strlen(text));
}

void dbg_printf(sint32 x, sint32 y, char *fmt, ...)
{
	char tmpbuf[255];
	va_list marker;
	va_start(marker,fmt);
	vsprintf(tmpbuf, fmt, marker );
	va_end(marker);
	dbg_print(x, y, tmpbuf);
}


void	dbg_get_extent(int &x, int &y, char *text)
{
	SIZE	sz;
	GetTextExtentPoint32(dbg->hdc, text, 1, &sz);
	x = sz.cx;
	y = sz.cy;
}

void dbg_display_help(void)
{
	uint32 i;
	static char dbg_helpstring[80*40];
	
	strcpy(dbg_helpstring,"");
	for(i = 0; dbg_helpstrings[i]!=0; i++)
	{
		strcat(dbg_helpstring, dbg_helpstrings[i]);
		strcat(dbg_helpstring, "\r\n");
	}
	MessageBox(NULL, dbg_helpstring, "Help Popup", MB_OK);
}

void dbg_print_tabs(int y, int index, char *heading, char *text[])
{
	uint32 i, j;
	static char clrline[]=" ";
	dbg_set_bk_color(0x7f7f7f);	
	dbg_printf(0, y, "%80s", clrline);

	dbg_set_bk_color(0x0000ff);
	dbg_set_color(0xffffff);
	dbg_print(0, y, heading);

	j = strlen(heading);
	for(i = 0; text[i]!=0; i++)
	{
		if(i == index)
		{
			dbg_set_bk_color(0xffffff);
			dbg_set_color(0xff0000);
		}
		else
		{
			dbg_set_bk_color(0x7f7f7f);
			dbg_set_color(0x000000);
		}
		dbg_print(j, y, text[i]);
		j += strlen(text[i]);
	}
}


static uint32 dbg_reg_cpu_shadow[32];
static uint16 dbg_reg_dsp_shadow[32];

void dbg_print_cpu_regs(void)
{
	static char clrline[]=" ";
	uint32	i, j;
	uint32	reg;

	dbg_print_tabs(REG_WIN_OFFSETY-1, reg_win_page, "F2:", reg_win_page_names);

	// clear window area
	dbg_set_bk_color(0xffffff);
	dbg_set_color(0x000000);
	for(i = 0 ; i < 16 ; i++)
	{
		dbg_printf(0, REG_WIN_OFFSETY+i, "%74s", clrline);
	}

	switch(reg_win_page)
		{
	case 0: // main integer registers
		for( j = 0 ; j < 2 ; j++)
		{
			for(i = 0 ; i < 16 ; i++)
			{
				reg = j * 16 + i;
			dbg_set_bk_color(0xffffff);
			dbg_set_color(0x000000);
				dbg_printf(j * 14, REG_WIN_OFFSETY + i, "r%02d:", reg);
			if (dbg_reg_cpu_shadow[reg] == CPUcurrmode->gpr[reg])
				dbg_set_color(0x7f0000);
			else
				dbg_set_color(0x0000ff);
				dbg_printf(j * 14 + 4, REG_WIN_OFFSETY + i, "%08x", CPUcurrmode->gpr[reg]);
			dbg_reg_cpu_shadow[reg] = CPUcurrmode->gpr[reg];
		}
	}
		// now for the extras
		dbg_printf(30, REG_WIN_OFFSETY + 0, "cr   : %08x  dec : %08x", CPUcurrmode->cr, CPUcurrmode->spr[PPC_DEC]);
		dbg_printf(30, REG_WIN_OFFSETY + 1, "xer  : %08x  tbr : %08x:%08x", CPUcurrmode->xer, CPUcurrmode->spr[PPC_TBH], CPUcurrmode->spr[PPC_TBL]);
		dbg_printf(30, REG_WIN_OFFSETY + 2, "ctr  : %08x  srr0: %08x", CPUcurrmode->ctr, CPUcurrmode->spr[PPC_SRR0]);
		dbg_printf(30, REG_WIN_OFFSETY + 3, "pc   : %08x  srr1: %08x", CPUcurrmode->pc, CPUcurrmode->spr[PPC_SRR1]);
		dbg_printf(30, REG_WIN_OFFSETY + 4, "lr   : %08x  dma : %08x:%08x", CPUcurrmode->lr, CPUcurrmode->spr[PPC_DMAU], CPUcurrmode->spr[PPC_DMAL]);
		dbg_printf(30, REG_WIN_OFFSETY + 5, "msr  : %08x  wpar: %08x", CPUcurrmode->msr, CPUcurrmode->spr[PPC_WPAR]);
		dbg_printf(30, REG_WIN_OFFSETY + 6, "fpscr: %08x  hid2: %08x", CPUcurrmode->fpscr, CPUcurrmode->spr[PPC_HID2]);
		break;
	case 1: // floating point registers
		for( j = 0 ; j < 2 ; j++)
		{
			for(i = 0 ; i < 16 ; i++)
			{
				reg = j * 16 + i;
				dbg_set_bk_color(0xffffff);
				dbg_set_color(0x000000);
				dbg_printf(j * 36, REG_WIN_OFFSETY + i, "f%02d=", reg);
				//if (dbg_reg_cpu_shadow[reg] == CPUcurrmode->gpr[reg])
				//	dbg_set_color(0x7f0000);
				//else
				//	dbg_set_color(0x0000ff);
				dbg_printf(j * 36 + 4, REG_WIN_OFFSETY + i, "%e %16.16I64x", CPUcurrmode->fpr[reg], CPUcurrmode->fpr[reg]);
				dbg_reg_cpu_shadow[reg] = CPUcurrmode->gpr[reg];
			}
}
		break;
	case 2: // paired single registers
		for( j = 0 ; j < 2 ; j++)
		{
			for(i = 0 ; i < 16 ; i++)
			{
				reg = j * 16 + i;
				dbg_set_bk_color(0xffffff);
				dbg_set_color(0x000000);
				dbg_printf(j * 27, REG_WIN_OFFSETY + i, "f%02d:", reg);
				//if (dbg_reg_cpu_shadow[reg] == CPUcurrmode->gpr[reg])
				//	dbg_set_color(0x7f0000);
				//else
				//	dbg_set_color(0x0000ff);
				dbg_printf(j * 27 + 4, REG_WIN_OFFSETY + i, "%3.3e %3.3e", CPUcurrmode->fpr[reg], CPUcurrmode->ps1[reg]);
				dbg_reg_cpu_shadow[reg] = CPUcurrmode->gpr[reg];
			}
		}
		for(i = 0 ; i < 8 ; i++)
		{
			dbg_set_bk_color(0xffffff);
			dbg_set_color(0x000000);
			dbg_printf(54, REG_WIN_OFFSETY + i, "gqr%1d:", i);
			//if (dbg_reg_cpu_shadow[reg] == CPUcurrmode->gpr[reg])
			//	dbg_set_color(0x7f0000);
			//else
			//	dbg_set_color(0x0000ff);
			dbg_printf(54 + 5, REG_WIN_OFFSETY + i, "%8.8x", CPUcurrmode->spr[PPC_GQR0+i]);
			dbg_reg_cpu_shadow[reg] = CPUcurrmode->gpr[reg];
		}
		break;
	default:
		break;
	}
}


void dbg_print_dsp_regs(void)
{
	uint32	i, j;
	uint32	reg;

	for( j = 0 ; j < 4 ; j++)
	{
		for(i = 0 ; i < 8 ; i++)
		{
			reg = j * 8 + i;
			SetBkColor(dbg->hdc, 0xffffff);
			SetTextColor(dbg->hdc, 0x000000);
			dbg_printf(pos_dsp_x + j * 9, pos_dsp_y + i, "$%02d=", reg);
			if (dbg_reg_dsp_shadow[reg] == dsp->r[reg])
				dbg_set_color(0x7f0000);
			else
				dbg_set_color(0x0000ff);
			dbg_printf(pos_dsp_x + j * 9 + 4, pos_dsp_y + i, "%04x", dsp->r[reg]);
			dbg_reg_dsp_shadow[reg] = dsp->r[reg];

		}
	}
	dbg_printf(pos_dsp_x, 9, "CMB: %08x DMB: %08x", gdsp_mbox_peek(GDSP_MBOX_CPU), gdsp_mbox_peek(GDSP_MBOX_DSP));
	for(i = 0 ; i < 6 ; i++)
		dbg_printf(pos_dsp_x + i * 5, 10, "%04x ", dsp->reg_stack[0][(dsp->reg_stack_ptr[0] - i) & DSP_STACK_MASK]);
	if (dsp->cr & 0x4) dbg_printf(pos_dsp_x, 11, "*HALTED*");
}


void dbg_print_cpu_disasm(void)
{
	sint32	i, j;
	char	opStr[16], parmStr[32];
	uint32	target, curraddress, opcode;
	uint8 data;

	dbg_print_tabs(CODE_WIN_OFFSETY-1, codedata_win_page, "F3:", codedata_win_page_names);

	switch(codedata_win_page)
	{
	case 0:
		curraddress = code_win_startaddress;

		for(i = 0 ; i <= CODE_WIN_LINES ; i++)
		{
			dbg_set_color(0xff0000);
			dbg_set_bk_color(0xffffff);
			if((i == code_win_startoffset))
			{
				dbg_set_bk_color(0x0000ff);
				dbg_set_color(0xffffff);
			}
			// highlight current PC
			if (curraddress == CPUcurrmode->pc)
			{
				if((i == code_win_startoffset))
				{
					dbg_set_color(0x00ffff);
				}
				else
				{
					dbg_set_color(0x0000ff);
				}
			}

			// put breakpoint marker ..
			if(is_breakpoint(curraddress))
			{
				dbg_print(0, CODE_WIN_OFFSETY + i, "* ");
			}
			else
			{
				dbg_print(0, CODE_WIN_OFFSETY + i, "  ");
			}

			if(mem_peek32(curraddress, &opcode))
			{
				GekkoDisassemble(opStr, parmStr, opcode, curraddress, &target);
				dbg_printf(2, CODE_WIN_OFFSETY + i, "%.8X  %.8X  %-10s %-30s", curraddress, opcode, opStr, parmStr);    
			}
			else
			{
				dbg_printf(2, CODE_WIN_OFFSETY + i, "%.8X  <invalid memory areas>", curraddress);
			}
			curraddress += 4;
		}
		break;
	case 1:
		curraddress = data_win_startaddress;

		for(i = 0 ; i <= CODE_WIN_LINES ; i++)
		{
			dbg_set_color(0xff0000);
			dbg_set_bk_color(0xffffff);
			if((i == data_win_startoffset))
			{
				dbg_set_bk_color(0x0000ff);
				dbg_set_color(0xffffff);
			}
			
			dbg_printf(0, CODE_WIN_OFFSETY + i, "%8.8X  ", curraddress);    
			for(j = 0; j < 16; j++)
			{
				if(mem_peek8(curraddress, &data))
				{
					dbg_printf(10 + j*3, CODE_WIN_OFFSETY + i, "%2.2X ",data); 
					if(data >= '!' && data <= '~')
					{
						dbg_printf(58 + j, CODE_WIN_OFFSETY + i, "%c", data);
					}
					else
					{
						dbg_printf(58 + j, CODE_WIN_OFFSETY + i, ".");
					}
				}
				else
				{
					dbg_print(10 + j*3, CODE_WIN_OFFSETY + i, "?? ");
					dbg_printf(58 + j, CODE_WIN_OFFSETY + i, ".");
				}
				curraddress++;
			}
		}
		break;
	case 2:
		curraddress = log_win_startindex;

		for(i = 0 ; i <= CODE_WIN_LINES ; i++)
		{
			dbg_set_color(0xff0000);
			dbg_set_bk_color(0xffffff);
			if((i == log_win_startoffset))
			{
				dbg_set_bk_color(0x0000ff);
				dbg_set_color(0xffffff);
			}
			if(curraddress < LOG_TYPE_MAX)
			{
				dbg_printf(0, CODE_WIN_OFFSETY + i, "  %-10.10s:%8.8s%56s", log_id[curraddress], log_trace_on[curraddress]?"Enabled":"Disabled"," ");
			}
			else
			{
				dbg_printf(0, CODE_WIN_OFFSETY + i, "%80s"," ");
			}
			curraddress++;
		}
		break;
	default:
		break;
	}
}

void dbg_print_dsp_disasm()
{
	sint32	i;
	uint32	j;
	char buffer[256];
	
	j = pos_dsp_y + 12;

	gdg->pc = dsp->pc;

	uint16	opcs[20];
	
	gdg->pc = dsp->pc - 10;
	for(i = 0 ; i < 11 ; i++)
	{
		opcs[i] = gdg->pc;
		if (dsp->pc & 0x8000)
			gdg->binbuf = dsp->irom;
		else
			gdg->binbuf = dsp->iram;
		gdg->pc += gd_dis_get_opcode_size(gdg);
		if (gdg->pc == dsp->pc)
			break;
	}
	
	gdg->pc = opcs[i-4];

	for(i = -5 ; i < 6 ; i++)
	{
		SetBkColor(dbg->hdc, 0xffffff);
		SetTextColor(dbg->hdc, 0xff0000);
		if (i == 0)
		{
			SetBkColor(dbg->hdc, 0x0000ff);
			SetTextColor(dbg->hdc, 0xffffff);
		}
		if (dsp->pc & 0x8000)
			gdg->binbuf = dsp->irom;
		else
			gdg->binbuf = dsp->iram;
		gdg->buffer = buffer;
		gdg->buffer_size = 256;
		gd_dis_opcode(gdg);
		dbg_printf(pos_dsp_x, j++, "%s", gdg->buffer);
	}
}

extern volatile uint32 dsp_running;

LRESULT WINAPI dbg_wndproc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{ 
	static PAINTSTRUCT ps;
	static int omx, omy, mx, my;
	HDC hdc;
	switch(uMsg) 
	{
	case WM_PAINT:
		//desktop.render();
		//		desktop.select();
		//		desktop.render();
		dbg->hdc = hdc = BeginPaint(hWnd, &ps);
		dbg_select_font();
		if (dbg->fsx == 0)
		{
			dbg_set_font_size();
			SetWindowPos(hWnd, NULL, 0, 0, dbg->cx * dbg->fsx, dbg->cy * dbg->fsy, NULL);
		}

		Rectangle(hdc, 0, 0, dbg->cx * dbg->fsx, dbg->cy * dbg->fsy);
		
		dbg_print_tabs(0, current_tab, "TAB:", tab_names );
		if(current_tab == 0)
		{
			dbg_print_cpu_regs();
			dbg_print_cpu_disasm();
			dbg_set_bk_color(0x7f7f7f);
			dbg_set_color(0x000000);
			dbg_printf(0, CODE_WIN_OFFSETY-2,"%80s"," ");
		}
		else
		{
			dbg_print_dsp_regs();
			dbg_print_dsp_disasm();
		}
		EndPaint(hWnd, &ps);
		return 0;

	case WM_SIZE:
		PostMessage(hWnd, WM_PAINT, 0, 0);
		return 0;
	case WM_KEYUP:
	case WM_SYSKEYUP:
		//fprintf(stderr, "%d\n", wParam);
		switch((uint32)wParam)
		{
		case VK_F1:
			dbg_display_help();
			break;
		case VK_F2:
			if(reg_win_page == 2)
			{
				reg_win_page = 0;
			}
			else
			{
				reg_win_page++;
			}
			dbg_print_cpu_regs();
			break;
		case VK_F3:
			if(codedata_win_page == 2)
			{
				codedata_win_page = 0;
			}
			else
			{
				codedata_win_page++;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_F5:
			dbg_state = DBG_RUN;
			break;
		case VK_F6:
			dbg_state = DBG_STEPOVER;
			break;
		case VK_F7:
			dbg_state = DBG_STEP;
			break;
		case VK_F9:
			switch(codedata_win_page)
			{
			case 0:
				if(is_breakpoint(code_win_startaddress + (code_win_startoffset*4)))
					remove_breakpoint(code_win_startaddress + (code_win_startoffset*4));
				else
					add_breakpoint(code_win_startaddress + (code_win_startoffset*4));
				break;
			case 2:
				log_trace_on[log_win_startindex + log_win_startoffset] ^=1;
				break;
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_SPACE:
			code_win_startaddress = CPUcurrmode->pc - (CODE_WIN_MIDDLE*4);
			code_win_startoffset = CODE_WIN_MIDDLE;
			dbg_print_cpu_disasm();
			break;
		case VK_UP:
			switch(codedata_win_page)
			{
			case 0:
				if(code_win_startoffset == 0)
					code_win_startaddress -=4;
				else
					code_win_startoffset--;
				break;
			case 1:
				if(data_win_startoffset == 0)
					data_win_startaddress -=16;
				else
					data_win_startoffset--;
				break;
			case 2:
				if(log_win_startoffset == 0)
				{
					if(log_win_startindex > 0)log_win_startindex--;
				}
				else
				{
					log_win_startoffset--;
				}
				break;
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_DOWN:
			switch(codedata_win_page)
			{
			case 0:
				if(code_win_startoffset == CODE_WIN_LINES)
					code_win_startaddress +=4;
				else
					code_win_startoffset++;
				break;
			case 1:
				if(data_win_startoffset == CODE_WIN_LINES)
					data_win_startaddress +=16;
				else
					data_win_startoffset++;
				break;
			case 2:
				if(log_win_startoffset == CODE_WIN_LINES)
				{
					if((log_win_startindex+log_win_startoffset) < (LOG_TYPE_MAX-1))log_win_startindex++;
				}
				else
				{
					if((log_win_startindex+log_win_startoffset) < (LOG_TYPE_MAX-1))log_win_startoffset++;
				}
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_PRIOR:
			switch(codedata_win_page)
			{
			case 0:
				if(code_win_startoffset != 0)
				{
					code_win_startoffset = 0;
				}
				else
				{
					code_win_startaddress -= (CODE_WIN_LINES * 4);
				}
				break;
			case 1:
				if(data_win_startoffset != 0)
				{
					data_win_startoffset = 0;
				}
				else
				{
					data_win_startaddress -= (CODE_WIN_LINES * 16);
				}
				break;
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_NEXT:
			switch(codedata_win_page)
			{
			case 0:
				if(code_win_startoffset != CODE_WIN_LINES)
				{
					code_win_startoffset = CODE_WIN_LINES;
				}
				else
				{
					code_win_startaddress += (CODE_WIN_LINES * 4);
				}
				break;
			case 1:
				if(data_win_startoffset != CODE_WIN_LINES)
				{
					data_win_startoffset = CODE_WIN_LINES;
				}
				else
				{
					data_win_startaddress += (CODE_WIN_LINES * 16);
				}
				break;
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_HOME:
			switch(codedata_win_page)
			{
			case 0:
				code_win_startaddress -= 0x1000;
				break;
			case 1:
				data_win_startaddress -= 0x1000;
				break;
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_END:
			switch(codedata_win_page)
			{
			case 0:
				code_win_startaddress += 0x1000;
				break;
			case 1:
				data_win_startaddress += 0x1000;
				break;
			default:
				break;
			}
			dbg_print_cpu_disasm();
			break;
		case VK_TAB:
			current_tab ^= 1;
			InvalidateRect(hWnd, NULL, true);
			break;
		default:
			break;
		}
		break;

	case WM_SYSCHAR:
		//		desktop.input(wParam);
		break;
	case WM_CHAR:
		//fprintf(stderr, "%c\n", wParam);
		switch (wParam) 
		{
		case 'p':
		case 'P':
			if (dsp_running == 0) gdsp_step();
			//dbg_state = DBG_DSP_STEP;
			InvalidateRect(hWnd, NULL, true);
			break;
		case 'o':
		case 'O':
			//gdsp_run();
			dsp_running = 1;
			//dbg_state = DBG_DSP_STEP;
			InvalidateRect(hWnd, NULL, true);
			break;
		case 'i':
		case 'I':
			//gdsp_run();
			dsp_running = 0;
			gdsp_stop();
			//dbg_state = DBG_DSP_STEP;
			InvalidateRect(hWnd, NULL, true);
			break;
		case 'r':
		case 'R':
			dbg_state = DBG_RUN;
			break;
		case 's':
		case 'S':
			dbg_state = DBG_STEP;
			break;
		case 'x':
			cpu_stop_running = 1;
			break;
		case 27:			
			PostQuitMessage(0);
			//DestroyWindow(hWnd);
			break;
		default:
			break;
		}
		break;
/*
		case WM_LBUTTONDOWN:
		case WM_RBUTTONDOWN:
		SetCapture(hWnd);
		mx = LOWORD(lParam);
		my = HIWORD(lParam);
		if (uMsg == WM_LBUTTONDOWN)
		{
		rbutton_down = true;
		state |= PAN;
		}
		if (uMsg == WM_RBUTTONDOWN)
		{
		state |= ROTATE;
		}
		return 0;

		case WM_LBUTTONUP:
		case WM_RBUTTONUP:
		ReleaseCapture();
		state = 0;
		return 0;

		case WM_MOUSEMOVE:
		if (state) 
		{
		omx = mx;
		omy = my;
		mx = LOWORD(lParam);
		my = HIWORD(lParam);
		if(mx & 1 << 15) mx -= (1 << 16);
		if(my & 1 << 15) my -= (1 << 16);
		update(state, omx, mx, omy, my);
		PostMessage(hWnd, WM_PAINT, 0, 0);
		}
		mouse_x = LOWORD(lParam);          
		mouse_y = HIWORD(lParam);
		return 0;
*/
	case WM_CLOSE:
		PostQuitMessage(0);
		break;;
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam); 
}  

HWND        hWnd;
bool dbg_create_window();

DWORD WINAPI dbg_debugger_thread(LPVOID lpParameter)
{
	MSG   msg;	

	dbg_create_window();

	while(GetMessage(&msg, NULL, 0, 0)) 
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	dbg_state = DBG_EXIT;
	return 0;
}

bool dbg_create_window()
{
	WNDCLASS    wc;
	HINSTANCE	hInstance;

	hInstance		 = GetModuleHandle(NULL);

	wc.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wc.lpfnWndProc   = (WNDPROC)dbg_wndproc;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = 0;
	wc.hInstance     = hInstance;
	wc.hIcon         = LoadIcon(NULL, IDI_WINLOGO);
	wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
	wc.hbrBackground = NULL;
	wc.lpszMenuName  = NULL;
	wc.lpszClassName = "VDDbg";

	if (!RegisterClass(&wc)) 
	{
		MessageBox(NULL, "RegisterClass() failed:  Cannot register window class.", "Error", MB_OK);
		return false;
	} 

	hWnd = CreateWindow("VDDbg", "Debugger", 
		WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN,
		0, 0, 640, 480, NULL, NULL, hInstance, NULL); 
	if (!hWnd)
	{
		MessageBox(NULL, "CreateWindow() failed:  Cannot create window.", "Error", MB_OK);
		return false;
	} 
	dbg->ofx = 2;
	dbg->ofy = 2;
	dbg->fsx = 0;
	ShowWindow(hWnd, SW_NORMAL);

	dbg_init_font();

	//SetTimer(hWnd, timid, 1000/75, NULL);
	return true;
}

void dbg_console_create(sint32 x, sint32 y, sint32 fontsize)
{
	dbg->cbuf = (char *)malloc(x * y);
	dbg->cx = x;
	dbg->cy = y;
	dbg->font_height = fontsize;
}

void dbg_main(void)
{
	dbg = (dbg_t *)malloc(sizeof(dbg_t));
	dbg_state = DBG_WAIT;
	dbg_console_create(80, 40, 14);

	CreateThread(NULL, 0, dbg_debugger_thread, 0, 0, NULL /* threadID */);
	while(1)
	{
		switch(dbg_state)
		{
		case DBG_RUN:
			trx_ppc_debug_run();
			if (dbg_state != DBG_EXIT)
				dbg_state = DBG_WAIT;
			InvalidateRect(hWnd, NULL, true);
			break;
		case DBG_EXIT:
//			trxCPU.done = true;
			return;
		case DBG_DSP_STEP:
			gdsp_step();
			if (dbg_state != DBG_EXIT)
				dbg_state = DBG_WAIT;
			InvalidateRect(hWnd, NULL, true);
			break;
		case DBG_STEP:
			trx_ppc_step();
			if (dbg_state != DBG_EXIT)
				dbg_state = DBG_WAIT;
			InvalidateRect(hWnd, NULL, true);
			break;
		case DBG_STEPOVER:
//			dbg_breakpoint = trxCPU.pc + 4;
			dbg_state = DBG_RUN;
			break;
		default:
			break;
		}

		if(!cpu_is_running)w32_check_events();
		Sleep(100);
	};
}

